#include <sys/time.h>
#include <unistd.h>
#include <iostream.h>
#include <iomanip.h>
#include <qpthr/qp.h>

class TimeMeasure {
    private:
	struct timeval start_tv;
	struct timeval stop_tv;
    public:
	void Start() {
		gettimeofday(&start_tv, NULL);
	}
	void Stop() {
		gettimeofday(&stop_tv, NULL);
	}
	void Print() {
		int sec, usec;
		char usec_str[16];
		
		sec = stop_tv.tv_sec - start_tv.tv_sec;
		if (start_tv.tv_usec > stop_tv.tv_usec) {
			sec--;
			usec = (stop_tv.tv_usec + 1000000) - start_tv.tv_usec;
		} else
			usec = stop_tv.tv_usec - start_tv.tv_usec;
		sprintf(usec_str, "%06d", usec);
		cout << "time: " << sec << "." << usec_str << endl;
	}
};


void test1(const char *label, QpLockBase *lock, int iter)
{
	TimeMeasure tme;
	int i;
	
	cout << "test 1 " << label << " ";
	cout.flush();
	tme.Start();
	for (i = 0; i < iter; i++) {
		lock->Lock();
		lock->Unlock();
	}
	tme.Stop();
	tme.Print();
}

class Test2Thread: public QpThread {
    private:
	QpLockBase *t_lock;
	int 	   t_iter;
    public:
	Test2Thread(QpLockBase *lock, int iter): t_lock(lock), t_iter(iter) {}
	virtual ~Test2Thread() {Join();}
	virtual void Main() {
		for (int i = 0; i < t_iter; i++) {
			QpSLock lock(t_lock);
			/* critical section with some code - few instructions 
			   maybe will be optimized (removed) by compiler */
			int j = t_iter;
			j++;
		}
	}
};

void test2(const char *label, QpLockBase *lock, int iter, int thr)
{
	TimeMeasure tme;
	vector<Test2Thread *> threads(thr);
	int i;
	
	cout << "test 2 " << label << " ";
	cout.flush();
	tme.Start();
	for (i = 0; i < thr; i++)
		threads[i] = new Test2Thread(lock, iter);
	for (i = 0; i < thr; i++)
		threads[i]->Start();
	for (i = 0; i < thr; i++)
		delete threads[i];
	tme.Stop();
	tme.Print();
}


class Test3Thread: public QpThread {
    private:
	QpLockBase *t_lock;
	QpCond     *t_cond;
	int 	   t_iter;
	int	   t_id;
	int	   t_maxid;
	int	   *t_token;
	int	   t_trylock;
    public:
	Test3Thread(const char *name, QpLockBase *lock, QpCond *cond,
		    int iter, int id, int maxid, int *token, int trylock = 0):
		QpThread(NULL, QP_PRIO_DFL, 0, name),
		t_lock(lock), t_cond(cond), t_iter(iter), t_id(id), 
		t_maxid(maxid), t_token(token), t_trylock(trylock) {}
	virtual ~Test3Thread() {Join();}
	
	virtual void Main() {
		int i, lock_held;
		
		for (i = 0; i < t_iter; i++) {
			lock_held = 0;
			
			if (t_trylock) {
				for (int j = 0; j < 1; j++) {
					if ((lock_held = t_lock->TryLock())) {
						break;
					}
				}
			}
			if (!lock_held) {
				t_lock->Lock();
			}
			while (*t_token != t_id) {
				t_cond->Wait();
			}
			*t_token = (*t_token + 1) % t_maxid;
			cout.flush();
			t_cond->Broadcast();
			t_lock->Unlock();
		}
	}
};

void test3(const char *label, QpLockBase *lock, int iter, int thr, int trylock = 0)
{
	TimeMeasure tme;
	vector<Test3Thread *> threads(thr);
	QpCond *cond;
	int i;
	int token;
	char name[2];
	
	cout << "test 3 " << label << " ";
	cout.flush();
	
	token = 0;
	cond = new QpCond(lock);
	name[0] = '0';
	name[1] = 0;
	tme.Start();
	for (i = 0; i < thr; i++) {
		threads[i] = new Test3Thread(name, lock, cond, iter, i, thr, 
					     &token, trylock);
		name[0]++;
	}
	for (i = 0; i < thr; i++)
		threads[i]->Start();
	for (i = 0; i < thr; i++)
		delete threads[i];
	tme.Stop();
	tme.Print();
	free(cond);
}

int main(int argc, char *argv[])
{
	QpInit qp_init;
	int c;
	int iter = 10000;
	int thr = 8;
	int try_lock_test = 0;
	
	QpSpinLock	spin_lock;
	QpMutex		mutex;
	QpMutexEx	mutex_ex;
	QpSem		sem;
	QpAsyncSafeSem	async_sem;
	
	while ((c = getopt(argc, argv, "i:t:y")) != EOF) {
		switch (c) {
		    case 'i':
			iter = atoi(optarg);
			break;
		    case 't':
			thr = atoi(optarg);
			break;
		    case 'y':
			try_lock_test = 1;
			break;
		    default:
			printf("usage: %s [-i iter] [-t threads]\n", argv[0]);
			exit(1);
		}
	}
	
	test1("QpSpinLock     ", &spin_lock, iter);
	test1("QpMutex        ", &mutex, iter);
	test1("QpMutexEx      ", &mutex_ex, iter);
	test1("QpSem          ", &sem, iter);
	test1("QpAsyncSafeSem ", &async_sem, iter);
	
	cout << endl;
	
	test2("QpSpinLock     ", &spin_lock, iter, thr);
	test2("QpMutex        ", &mutex, iter, thr);
	test2("QpMutexEx      ", &mutex_ex, iter, thr);
	test2("QpSem          ", &sem, iter, thr);
	test2("QpAsyncSafeSem ", &async_sem, iter, thr);

	cout << endl << "possibly lower the number of iterations (-i 1000)" << endl;

	test3("QpSpinLock     ", &spin_lock, iter, thr);
	test3("QpMutex        ", &mutex, iter, thr);
	test3("QpMutexEx      ", &mutex_ex, iter, thr);
	test3("QpSem          ", &sem, iter, thr);
	test3("QpAsyncSafeSem ", &async_sem, iter, thr);

	if (try_lock_test) {
		cout << endl << "next test can be very slow (-i 100 recommended)" << endl;
	
		test3("QpSpinLock     ", &spin_lock, iter, thr, 1);
		test3("QpMutex        ", &mutex, iter, thr, 1);
		test3("QpMutexEx      ", &mutex_ex, iter, thr, 1);
		test3("QpSem          ", &sem, iter, thr, 1);
		test3("QpAsyncSafeSem ", &async_sem, iter, thr, 1);
	}

	cout << endl << "done" << endl;
	return 0;
}

