#!/usr/bin/perl
##
#
# D. Scott Barninger <barninger@cvn.net>
# <http://www.b-wdesigngroup.com>
# License: licensed under the same terms as Perl (Artistic License).
#
# Based upon work by
#
# Carl Sewell <csewell@hiwaay.net>
# <http://home.hiwaay.net/~csewell/>
#
# Greg Kearney <gkearney@new-sharon.me.us>
# <http://www.new-sharon.me.us/upc.html>

## Barcode.pm version 1.5.3
$VERSION = "1.5.3";

#--------------------------------------------------------------------------------------
# Code128 package
#--------------------------------------------------------------------------------------

package Code128;
use GD;
use strict;

# object constructor
#
sub new {
	my $self = {};

	# parameters
	my ($me,$bar_code,$quiet_zone,$polarity,$density,$dpi,$image_type,$description) = @_;

	#other declarations
	my ($num_char,@table,$bar_position,$start_codeA,$start_codeB,$start_codeC,
			$start_code,$stop_code,$num_chars,$bars_per_char,@text,$CodeType,
			$number_of_bars,$pixels_per_bar,$length,$unit_length,$height,$min_height,
			$border,$im,$white,$black,$foreground,$background,$sum,$i,$char,$value,
			$ord_value,$fontsize,$color,@bar,$bar,$offset);

	# check parameters and/or set defaults
	if(!$bar_code || !$quiet_zone || !$polarity || !$density || !$dpi || !$image_type) {
		warn("No data passed to create bar code.\n");
		return;
	}

	$num_char=length($bar_code);

	# This table defines the bar/space width of the three versions of
	# code 128 and takes the form of:
	# CodeA CodeB CodeC Value Bar_width Space_width Bar_width .......

	@table = (
			"0 SP SP 00 2 1 2 2 2 2",
			"1 ! ! 01 2 2 2 1 2 2",
			"2 \" \" 02 2 2 2 2 2 1",
			"3 # # 03 1 2 1 2 2 3",
			"4 \$ \$ 04 1 2 1 3 2 2",
			"5 % % 05 1 3 1 2 2 2",
			"6 & & 06 1 2 2 2 1 3",
			"7 ' ' 07 1 2 2 3 1 2",
			"8 ( ( 08 1 3 2 2 1 2",
			"9 ) ) 09 2 2 1 2 1 3",
			"10 * * 10 2 2 1 3 1 2",
			"11 + + 11 2 3 1 2 1 2",
			"12 , , 12 1 1 2 2 3 2",
			"13 - - 13 1 2 2 1 3 2",
			"14 . . 14 1 2 2 2 3 1",
			"15 / / 15 1 1 3 2 2 2",
			"16 0 0 16 1 2 3 1 2 2",
			"17 1 1 17 1 2 3 2 2 1",
			"18 2 2 18 2 2 3 2 1 1",
			"19 3 3 19 2 2 1 1 3 2",
			"20 4 4 20 2 2 1 2 3 1",
			"21 5 5 21 2 1 3 2 1 2",
			"22 6 6 22 2 2 3 1 1 2",
			"23 7 7 23 3 1 2 1 3 1",
			"24 8 8 24 3 1 1 2 2 2",
			"25 9 9 25 3 2 1 1 2 2",
			"26 : : 26 3 2 1 2 2 1",
			"27 ; ; 27 3 1 2 2 1 2",
			"28 < < 28 3 2 2 1 1 2",
			"29 = = 29 3 2 2 2 1 1",
			"30 > > 30 2 1 2 1 2 3",
			"31 ? ? 31 2 1 2 3 2 1",
			"32 @ @ 32 2 3 2 1 2 1",
			"33 A A 33 1 1 1 3 2 3",
			"34 B B 34 1 3 1 1 2 3",
			"35 C C 35 1 3 1 3 2 1",
			"36 D D 36 1 1 2 3 1 3",
			"37 E E 37 1 3 2 1 1 3",
			"38 F F 38 1 3 2 3 1 1",
			"39 G G 39 2 1 1 3 1 3",
			"40 H H 40 2 3 1 1 1 3",
			"41 I I 41 2 3 1 3 1 1",
			"42 J J 42 1 1 2 1 3 3",
			"43 K K 43 1 1 2 3 3 1",
			"44 L L 44 1 3 2 1 3 1",
			"45 M M 45 1 1 3 1 2 3",
			"46 N N 46 1 1 3 3 2 1",
			"47 O O 47 1 3 3 1 2 1",
			"48 P P 48 3 1 3 1 2 1",
			"49 Q Q 49 2 1 1 3 3 1",
			"50 R R 50 2 3 1 1 3 1",
			"51 S S 51 2 1 3 1 1 3",
			"52 T T 52 2 1 3 3 1 1",
			"53 U U 53 2 1 3 1 3 1",
			"54 V V 54 3 1 1 1 2 3",
			"55 W W 55 3 1 1 3 2 1",
			"56 X X 56 3 3 1 1 2 1",
			"57 Y Y 57 3 1 2 1 1 3",
			"58 Z Z 58 3 1 2 3 1 1",
			"59 [ [ 59 3 3 2 1 1 1",
			"60 \\ \\ 60 3 1 4 1 1 1",
			"61 ] ] 61 2 2 1 4 1 1",
			"62 ^ ^ 62 4 3 1 1 1 1",
			"63 _ _ 63 1 1 1 2 2 4",
			"64 NUL ' 64 1 1 1 4 2 2",
			"65 SOH a 65 1 2 1 1 2 4",
			"66 STX b 66 1 2 1 4 2 1",
			"67 ETX c 67 1 4 1 1 2 2",
			"68 EOT d 68 1 4 1 2 2 1",
			"69 ENQ e 69 1 1 2 2 1 4",
			"70 ACK f 70 1 1 2 4 1 2",
			"71 BEL g 61 1 2 2 1 1 4",
			"72 BS h 72 1 2 2 4 1 1",
			"73 HT i 73 1 4 2 1 1 2",
			"74 LF j 74 1 4 2 2 1 1",
			"75 VT k 75 2 4 1 2 1 1",
			"76 FF l 76 2 2 1 1 1 4",
			"77 CR m 77 4 1 3 1 1 1",
			"78 SO n 78 2 4 1 1 1 2",
			"79 SI o 79 1 3 4 1 1 1",
			"80 DLE p 80 1 1 1 2 4 2",
			"81 DC1 q 81 1 2 1 1 4 2",
			"82 DC2 r 82 1 2 1 2 4 1",
			"83 DC3 s 83 1 1 4 2 1 2",
			"84 DC4 t 84 1 2 4 1 1 2",
			"85 NAK u 85 1 2 4 2 1 1",
			"86 SYN v 86 4 1 1 2 1 2",
			"87 ETB w 87 4 2 1 1 1 2",
			"88 CAN x 88 4 2 1 2 1 1",
			"89 EM y 89 2 1 2 1 4 1",
			"90 SUB z 90 2 1 4 1 2 1",
			"91 ESC { 91 4 1 2 1 2 1",
			"92 FS | 92 1 1 1 1 4 3",
			"93 GS } 93 1 1 1 3 4 1",
			"94 RS ~ 94 1 3 1 1 4 1",
			"95 (Hex7F) US_DEL 95 1 1 4 1 1 3",
			"96 (Hex80) FNC_3 96 1 1 4 3 1 1",
			"97 (Hex81) FNC_2 97 4 1 1 1 1 3",
			"98 (Hex82) SHIFT 98 4 1 1 3 1 1",
			"99 (Hex83) CODE_C 99 1 1 3 1 4 1",
			"100 (Hex84) CODE_B FNC_4 1 1 4 1 3 1",
			"101 (Hex85) FNC_4 CODE_A 3 1 1 1 4 1",
			"102 (Hex86) FNC_1 FNC_1 4 1 1 1 3 1",
			"103 (Hex_87) START (Code_A) 2 1 1 4 1 2",
			"104 (Hex_88) START (Code_B) 2 1 1 2 1 4",
			"105 (Hex_89) START (Code_C) 2 1 1 2 3 2",
			"106 STOP ALL CODES 2 3 3 1 1 1 2");

	$_=$bar_code;

	# Initialize the $bar_position counter
	$bar_position=0;
	#$start_codeA=103;  Code A is not supported
	$start_codeB=104;
	$start_codeC=105;
	$stop_code=106;
	$num_chars=length($bar_code);

	# If odd length or contains a non-digit use Code B
	# Code B has 11 bars per character, Code C has 5.5 bars per char.
	# which means two characters per bar pattern.
	# The data is split into 1 char lengths ("a","b",etc) for Code B and
	# two char lengths ("00","02", etc)
	if (/[\D]/||(length($_)%2!=0)){
		$start_code=$start_codeB;
		$bars_per_char=11;
		@text=split //;
		$CodeType = "Code128B";
	}
	else
	{
		$start_code=$start_codeC;
		$bars_per_char=5.5;
		$_ =~ s/(\d\d)/$1 /g;
		@text= split / /;
		$CodeType = "Code128C";
	}
	# Length of the bar code in pixels for defining the image size
	$number_of_bars = (35 + ($bars_per_char*$num_chars));

	#Calculate the minimum pixels per bar based on printer resolution
	#It must be an integer (ie a whole number)!!!!
	$pixels_per_bar = $density*$dpi/1000;
	unless ((int($pixels_per_bar)/$pixels_per_bar)==1){
		$pixels_per_bar=int($pixels_per_bar)+1;
	};
	$length=$number_of_bars*$pixels_per_bar;
	$unit_length = (int ($number_of_bars*$pixels_per_bar/$dpi*1000))/1000;
	if (($length/int($length))>0) {$length=int($length+1)};

	# The minimum height for a Code 128 is .25" or .15 times the length
	# whichever is greater.
	$height=int($length*0.150);
	$min_height=$dpi*250/1000;
	if ($height<$min_height) {$height=$min_height};

	$border=$quiet_zone*$pixels_per_bar;
	if ($border<15) {$border=15};

	# Define the image size for GD
	$im=new GD::Image($length+$border*2,$height+$border*2);

	# Initialize colors (white becomes the default background
	# Black-on-white or White-on-Black
	if ($polarity eq "bw")
	{
		$white = $im->colorAllocate(255,255,255);
		$black = $im->colorAllocate(0,0,0);
		$foreground=$black;
		$background=$white;
	}
	else
	{
		$black = $im->colorAllocate(0,0,0);
		$white = $im->colorAllocate(255,255,255);
		$foreground=$white;
		$background=$black;
	}

	# select transparent or interlaced image
	if($image_type eq "transparent") {
		$im->transparent($background);
	}
	else {
		$im->interlaced('true');
	}


	# Code 128 begins with a Start Character
	# print_bars is a subroutine for creating the bar/space pattern
	# for GD, the GIF format
	$_=$table[$start_code];
	print_bars();		

	# Code 128 employs a check character which is the sum-product of the
	# character value (ord - 32) and the sequence (or position). The
	# start_code has a position value of zero.
	$sum=$start_code;

	# Loop through the characters of the bar code and look-up the
	# bar/space pattern in the table and create the GD GIF pattern
	# If Code B then use ord value of char
	# If Code C then use the value of the two char (ie 00, 01)
	$i=1;
	foreach $char (@text) {
		if ($start_code == $start_codeB){
			$value=ord($char)-32;
		}
		else {
			$value=$char;
		}
		if ($value<0 || $value>138) {  #Outside of bounds of Code128
			$ord_value = ord($char);
			warn("Character $i (ASCII $ord_value) is not valid for $CodeType");
			return;
		}
		$sum+=($i*$value);
		$_=$table[$value];
		print_bars();
		++$i;
	}

	# The check character is calculated as the remainder of sum-product
	# as described above.
	$_=$table[$sum % 103];
	print_bars();

	# The stop Character depends on the version (Code A or B)
	$_=$table[$stop_code];
	print_bars();

	# Print the bar code text and optional description under the bar code
	# Limited size fonts available
	$fontsize=gdSmallFont;
	$im->string($fontsize,$border,$height+$border+1,$bar_code . " " . $description,$foreground);

	$self->{png_file} = $im->png;
	bless($self);
	return($self);

	sub print_bars {
	# Get the bar/space pattern from the table
	# and paint the bars and spaces, although there is
	# no need to paint the spaces (ie background color).
		($_)=/\S+\s+\S+\s+\S+\s+\S+\s+(.*)/;
		my $color=$foreground;
		my @bar=split(/ /);
		foreach $bar (@bar){
			$offset = $border+$bar_position*$pixels_per_bar;
			$im->filledRectangle($offset,$border,
				$offset+$bar*$pixels_per_bar-1,
				$border+$height,$color);

			SWITCH: {
				if ($color eq $foreground){$color=$background; last SWITCH;}
				$color=$foreground;
				}
			$bar_position+=$bar;
		}
	}

}



1;


#---------------------------------------------------------------------------------------
# CodeISBN package
#---------------------------------------------------------------------------------------

package CodeISBN;
use strict;

sub encode {
	my $self = {};

	# parameters & declarations
	my($me,$value) = @_;
	my(@code2,$isbn,@isbnarray,@newisbn,$x,$checksum);

	@code2 = split //, $value;
	$isbn =  join("", @code2[3..11]);
	@isbnarray =  split //, $isbn;
	@newisbn[0] = $isbnarray[0]*10;
	@newisbn[1] = $isbnarray[1]*9;
	@newisbn[2] = $isbnarray[2]*8;
	@newisbn[3] = $isbnarray[3]*7;
	@newisbn[4] = $isbnarray[4]*6;
	@newisbn[5] = $isbnarray[5]*5;
	@newisbn[6] = $isbnarray[6]*4;
	@newisbn[7] = $isbnarray[7]*3;
	@newisbn[8] = $isbnarray[8]*2;

	$a = $newisbn[0]+$newisbn[1]+$newisbn[2]+$newisbn[3]
		+$newisbn[4]+$newisbn[5]+$newisbn[6]+$newisbn[7]+$newisbn[8];

	$x = ($a % 11);
	$checksum = 11 - $x;

	if ($checksum >= 10) {
		$checksum = 0;
	}
	$isbn = $isbn.$checksum;

	$self->{isbn_value} = $isbn;

	bless($self);
	return($self);

}

1;

#---------------------------------------------------------------------------------------
# CueCat Decoder package
#---------------------------------------------------------------------------------------

package CueCat;
use Fcntl;
use IO::Socket;
use strict;

sub decode {
	my $self = {};

	# parameters
	my($me,$scan_string) = @_;

	# other declarations
	my($table,$outs,%barcode);
	$_ = $scan_string;
	$table = join("",'a'..'z','A'..'Z','0'..'9','+','-');

	# check the OS and warn if linux
	if($^O eq "linux") {
		warn("CueCat->decode is not recommended on linux - see documentation.");
	}

	# now decode it
	%barcode = cue_decode($_);
	$self->{barcode_data} = $barcode{'value'};
	$self->{barcode_type} = $barcode{'type'};
	$self->{cuecat_ser_no} = $barcode{'serno'};

	bless($self);
	return($self);

	sub cue_decode
	{
		chomp;
		my $work = $_;

		my @chunks = split /\./,$work;

		my $code = $chunks[3];
		my $type = $chunks[2];
		my $serno = $chunks[1];

		#my $i;
		#my $output;

		return (
				'value' => decode1($code),
				'type'  => decode1($type),
				'serno' => decode1($serno)
			);
	}

	sub decode1  # nick's version with padding added
	{
		chomp;

		my ($string) = @_;

		# check to see if we have a short string (not divisible by 4 characters)
		# and pad with 'a'
		my $chars_padded = 0;
		while((length($string)/4) =~ /\./) {
			$string .= "a";
			$chars_padded++;
		}

		my $work = "";
		my $outs = "";

		foreach my $i (split(//,$string))
		{
			$work .= substr(join("",unpack("B8",chr(index($table,$i)))),2,6);
		}

		while (length($work) > 0)
		{	
			$outs .= chr(ord(pack("B8",substr($work,0,8))) ^ 67);
			$work = substr($work,8,length($work));
		}

		# now chop as many characters as we padded above
		while($chars_padded > 0) {
			chop $outs;
			$chars_padded--;
		}

		return $outs;
	}
}

sub kdecode {
	# binding to the linux kernel cuecat driver

	# check OS and existance of kernel driver
	if($^O eq "linux") {
		open(DRIVER, "ls /dev/scanners/cuecat |");
		my $driver = <DRIVER>;
		close(DRIVER);
		chomp($driver);
		if($driver ne "/dev/scanners/cuecat") {
			die("CueCat->kdecode requires the kernel CueCat driver - see documentation.");
		}
	}
	else {
		die("CueCat->kdecode is for use with the linux kernel driver only.");
	}

	my $self = {};
	my $pid = getpgrp();
	my %data = &get_scan;
	$self->{cuecat_ser_no} = $data{'serno'};
	$self->{barcode_type} = $data{'type'};
	$self->{barcode_data} = $data{'value'};
	$self->{pid} = $pid;

	bless($self);
	return($self);

	sub get_scan {
		# read the device, split off the header and parse the data
		sysopen(CAT, "/dev/scanners/cuecat", O_NONBLOCK) || die "Can't open the CueCat";
		my $raw_data = <CAT>;
		if($raw_data =~ /^BARCODE:/) { # see if we caught a scan
			chomp($raw_data);
			my ($header,$data) = split(/:/,$raw_data); # split off BARCODE:
			my ($ser,$type,$barcode) = split(/,/,$data); # parse the data
			close(CAT);
			return (
					'serno' => $ser,
					'type' => $type,
					'value' => $barcode);
		}
	}	
}

sub destroy {
	# this destroys the buffer object created by kdecode
	my($pid) = @_;
	kill "HUP", $pid;
}

sub get_dc_cue {

	my $self = {};

	# parameters
	my($me,$scan_string) = @_;

	# other declarations
	my($activation_code,@dcnv_hosts,$host,$dcnv,$foo,$url);

	# Create a random activation code to preserve privacy
	$activation_code = rand()*rand()*158;
	$activation_code =~ s/\.//g;

	# process the scan string for transmission to dcnv server
	# Remove trailing newline
	chomp($scan_string);
	# Strip last character, which is a dot
	$scan_string = substr($scan_string,0,-1);
	# Strip first 7 characters
	$scan_string = substr($scan_string,7);
	# Invert case 
	$scan_string =~ tr/a-zA-Z/A-Za-z/;

	# select a dcnv host at random
	@dcnv_hosts = ("a","o","s","t","u");
	$host = @dcnv_hosts[int(rand(5))];

	# Open http connection to DCNV
	$dcnv = IO::Socket::INET->new(
			Proto=> "tcp",
			PeerAddr => "$host.dcnv.com",
			PeerPort => 80) ||
		die "Can't Connect To U.DCNV.com: $!";
	$dcnv->autoflush(1);
	# Send request
	print $dcnv "GET /CRQ/1..$activation_code.04.$scan_string.0 HTTP/1.1\r\n";
	print $dcnv "Host: $host.dcnv.com\r\n";
	# Pretend to be windows just in case they block other OS's
	print $dcnv "User-Agent: Mozilla/4.75 [en] (Win98; U)\r\n";
	print $dcnv "Accept: */*\r\n";
	print $dcnv "Accept-Charset: us-ascii, ISO-8859-1, ISO-8859-2, ISO-8859-4, ISO-8895-5, ISO-8859-13, windows-1250, windows-1251, windows-1257, cp437, cp850, cp852, cp866, x-cp866-u, x-mac-ce, x-kam-cs, x-koi8-r, x-koi8-u, utf-8\r\n";
	print $dcnv "Connection: Keep-Alive\r\n";
	print $dcnv "\r\n";
	# Wait for reply, then close connection to DCNV
	do {
			$foo = <$dcnv>;
			} until ($foo =~ /^url\=/);
	close($dcnv);
	# Get reply URL
	$url = $foo;
	$url =~ s/^url\=//;

	$self->{url} = $url;

	bless($self);
	return($self);

}

sub encode {

	my $self = {};

	# parameters
	my($me,$data) = @_;

	# other declarations
	my(@table,$chars_padded,$eight_bit,$six_bit,$work,$decimal,$char,$output,$i);

	@table = ('a'..'z','A'..'Z','0'..'9','+','-');

	# check that the number of characters is divisible by 3 & pad with 0's
	while((length($data)/3) =~ /\./) {
		$data .= "0";
		$chars_padded++;
	}

	# take each character, XOR with C and unpack into bit string
	foreach $i (split(//,$data)) {
		$eight_bit = join('', unpack("B8", ($i ^ 'C')));
		$work .= $eight_bit;
	}

	# now take the bit string constructed 6 bits at a time, pack  back into
	# decimal value and match it to the character in the table
	while(length($work) > 0) {
		$six_bit = substr($work,0,6);
		$decimal = unpack("N", pack("B32", substr("0" x 32 . $six_bit, -32)));
		$char = $table[$decimal];
		$output .= $char;
		$work = substr($work,6,length($work));
	}

	# if we padded the input string, chop the excess characters
	while($chars_padded > 0) {
		chop($output);
		$chars_padded--;
	}

	$self->{encoded} = $output;

	bless($self);
	return($self);

}

1;

__END__

#===============================================================================
#==== Documentation
#===============================================================================
=pod

=head1 NAME

 Barcode - version 1.5.3 Sun Feb 27 2005

 Creates a barcode image as a png file object. Currently only does type
 Code 128 (B or C). Also decodes scans from the CueCat scanner and
 retrieves the url from a Digital Convergence server.

=head1 SYNOPSIS

 use Barcode;

 Class Code128
 # create a Code 128 barcode
 my $barcode = Code128->new($bar_code,$quiet_zone,$polarity,$density,
	$dpi,$image_type,$description);


 Class CodeISBN
 # return the ISBN number after previously decoding the barcode
 my $isbn = CodeISBN->encode($scan->{'barcode_data'});


 Class Cuecat					
 # decode a barcode with the decode method
 my $scan = CueCat->decode($scan_string);

 # decode a barcode with the kdecode method
 my $scan = CueCat->kdecode;

 # retrieve a url
 my $scan_string = <STDIN>;
 my $scan = CueCat->get_dc_cue($scan_string);

 # encode data
 my $data = CueCat->encode($barcode_data);

=head1 DESCRIPTION

 Creates a barcode image as a png object which you can then print or
 save to a file, decodes barcode scans from the CueCat scanner
 <www.cuecat.com> and can query the Digital Convergence database for
 a url.

 Currently only creates type Code 128 barcodes.
 If the data string is an odd length or contains non-digit characters,
 Code B is used, otherwise Code C.

 For more information on barcodes see
 BarCode1 <http://www.adams1.com/pub/russadam/barcode1.html>.


 Class Code128
 # create a Code 128 barcode
 my $barcode = Code128->new($bar_code,$quiet_zone,$polarity,$density,
	$dpi,$image_type,$description);

 where:	$bar_code = string to be barcoded
		$quiet_zone = number of clear x-dimensions each end
		$polarity = bw (black on white) or wb (white on black)
		$density = width of individual bars in .001 inch
		$dpi = resolution of target printer
		$image_type = 'transparent' or 'interlaced't
		$description = optional description

 Returns the png file in $barcode->{'png_file'}.


 Class CodeISBN
 Constructs the ISBN number from the raw barcode value on an ISBN
 barcode (this is the number found above the ISBN barcode).

 # return the ISBN number after previously decoding the barcode
 my $isbn = CodeISBN->encode($scan->{'barcode_data'});

 Returns $isbn{'isbn_value'}.


 Class Cuecat
 Two decode methods are provided: decode and kdecode.

 CueCat->decode uses a perl decode routine on raw scan data from the
 CueCat. This method is NOT recommended on linux or any system using
 X-Windows due to the initial Alt-F10 sequence sent by the CueCat.
 It will work on a regular text console however, provided you do not
 have a virtual terminal set up on tty10.

 CueCat->kdecode is a binding to the linux CueCat kernel driver and
 is the recommended method on linux. The kernel driver is available at
 http://oss.lineo.com/cuecat/.

 # decode a barcode with the decode method
 my $scan = CueCat->decode($scan_string);

 # decode a barcode with the kdecode method
 my $scan = CueCat->kdecode;

 Returns $scan->{'barcode_type'} $scan->{'barcode_data'} and
 $scan->{'cuecat_ser_no'} in either method. If you call CueCat->kdecode
 inside a loop you must destroy the object created by calling
 $scan->destroy($scan->{'pid'}) before returning to the start of the
 loop.

 CueCat->get_dc_cue will query a Digital Convergence server and return
 the url from the database. This method requires the raw (undecoded)
 string sent by the scanner. It will generate a random "activation code"
 in order to preserve the privacy of the user. If the url is not in DC's
 database it returns the url of a form to submit the barcode to DC.

 # retrieve a url
 my $scan_string = <STDIN>;
 my $scan = CueCat->get_dc_cue($scan_string);

 Returns $scan->{'url'};

 CueCat->encode will take decoded data and re-encode it to the raw scan
 data format. This allows users of the linux kernel driver to get a
 decoded scan from the driver and re-encode it in order to call
 CueCat->get_dc_cue.

 # encode data
 my $data = CueCat->encode($barcode_data);

 Returns $data->{'encoded'}. Note you must do this for all 3 pieces of the
 original scan string and then reconstruct the scan format. See the example
 file.

=head1 AUTHOR

 D. Scott Barninger <barninger at fairfieldcomputers.com>
 <http://www.b-wdesigngroup.com>
 License: licensed under the same terms as Perl (Artistic License).

 Based upon work by

 Carl Sewell <csewell at hiwaay.net>
 <http://home.hiwaay.net/~csewell/>

 Greg Kearney <gkearney at new-sharon.me.us>
 <http://www.new-sharon.me.us/upc.html>

=cut
