
/*  ************************************************************************  *
 *				  lvbktest.cpp				      *
 *  ************************************************************************  */

/*  This is the source file for a program that demonstrates various features
    of List-View controls, but concentrating on the background (for colours,
    images, watermarks, etc).

    Compile to taste, but with INCLUDE files from Vista SDK.
    Don't forget the accompanying resource file.
    Link with /delayload:comctl32.dll.
    Run with /? for options.

    Copyright (C) 2008-2010. Geoff Chappell. All rights reserved.  */

/*  ************************************************************************  */

#define UNICODE
#define WIN32_LEAN_AND_MEAN
#define _WIN32_IE	0x0700	    // for IVisualProperties
#define _WIN32_WINNT	0x0600	    // for various new COMCTL32 features

#include    <windows.h>

#include    <stdio.h>

#include    <commctrl.h>
#include    <objbase.h>
#include    <shellapi.h>
#include    <shobjidl.h>

/*  The IListView interface - pending definition by Microsoft  */

#include    "listview.h"

/*  The undocumented message that gets the IListView interface - new for
    Windows Vista, name invented, and again something that Microsoft may
    someday add to COMMCTRL.H  */

#if _WIN32_WINNT >= 0x0600

#ifndef LVM_QUERYINTERFACE
#define LVM_QUERYINTERFACE	    (LVM_FIRST + 189)
#endif

#endif	// #if _WIN32_WINNT >= 0x0600

/*  We may as well tell the linker which import libraries we want.   */

#pragma comment (lib, "comctl32.lib")
#pragma comment (lib, "gdi32.lib")
#pragma comment (lib, "msimg32.lib")
#pragma comment (lib, "ole32.lib")
#pragma comment (lib, "user32.lib")
#pragma comment (lib, "shell32.lib")

/*  Resolution of imports from COMCTL32.DLL is deferred until we can choose
    which COMCTL32 to import from. We may as well tell the linker which
    library provides the delayload code. Ideally, we might also specify
    which DLL to delay-load, but /delayload is not among the linker switches
    that are allowed as directives in object files.  */

#pragma comment (lib, "delayimp.lib")

/*  A few things the author happens to like  */

#define AND &&
#define NOT !
#define OR  ||

/*  Resource IDs shared with LVBKTEST.RC  */

#include    "lvbktest.h"

/*  ************************************************************************  */
/*  Global Data and Forward References	*/

const WCHAR g_szCaption [] = L"ListView Background Test";

const WCHAR g_szDescription [] =
    L"Boolean options - action by default or if <b> is on, true, yes or 1:"
    L"\n"
    L"\n On by default:"
    L"\n"
    L"\n  /cominit:<b>  \t do COM initialisation"
    L"\n  /icclv:<b>    \t call InitCommonControlsEx for List-View control"
    L"\n  /manifest:<b> \t use application manifest for new COMCTL32"
    L"\n  /messages:<b> \t use window messages even if IListView supported"
    L"\n"
    L"\n Off by default:"
    L"\n"
    L"\n  /iccstd:<b>   \t call InitCommonControlsEx for standard controls"
    L"\n"
    L"\nNumerical options - <x> and <y> are decimal integers:"
    L"\n"
    L"\n  /pos:<x>,<y>  \t position of List-View in main window"
    L"\n  /size:<x>,<y> \t size of List-View control";

BOOL g_bComInit = TRUE;
BOOL g_bIccLv = TRUE;
BOOL g_bManifest = TRUE;
BOOL g_bMessages = TRUE;
BOOL g_bActCtx = FALSE;
BOOL g_bIccStd = FALSE;
POINT g_ListViewPosition = {0};
SIZE g_ListViewSize = {0};

HINSTANCE g_hInstance = NULL;

/*  ************************************************************************  */
/*  Things that are relatively general, such that they might be usable
    elsewhere or moved to other source files  */

/*  Some of these classes are perhaps a little unnatural, but having a
    destructor eases cleanup (and even makes it more certain).	*/

/*  ========================================================================  */
/*  Optional COM Initialisation  */

class CComInitialisation
{
    BOOL m_bInitDone;

    public:

    CComInitialisation (VOID)
    {
	m_bInitDone = FALSE;
    };

    BOOL Init (BOOL bInitWanted)
    {
	if (NOT bInitWanted) return TRUE;
	HRESULT hr = CoInitialize (NULL);
	if (FAILED (hr)) return FALSE;
	m_bInitDone = TRUE;
	return TRUE;
    };

    ~CComInitialisation (VOID)
    {
	if (m_bInitDone) CoUninitialize ();
    };
};

/*  ========================================================================  */
/*  Optional Selection of SxS COMCTL32	*/

class CComctl32Selection
{
    HANDLE m_hActCtx;
    ULONG_PTR m_ulCookie;

    public:

    CComctl32Selection (VOID)
    {
	m_hActCtx = INVALID_HANDLE_VALUE;
    };

    BOOL Init (BOOL bUseManifest)
    {
	if (NOT bUseManifest) return TRUE;

	/*  The aim is to use the application manifest that Microsoft
	    helpfully provides in the Windows directory.  */

	WCHAR buf [MAX_PATH];
	UINT cchdir = GetSystemWindowsDirectory (buf, RTL_NUMBER_OF (buf));
	if (cchdir == 0) return FALSE;

	/*  Be tidy and avoid doubling a trailing backslash when appending
	    the filename.  */

	if (buf [cchdir - 1] != L'\\') {
	    if (cchdir + 1 > RTL_NUMBER_OF (buf)) return FALSE;
	    buf [cchdir] = L'\\';
	    cchdir ++;
	}

	UINT cchfile = wcslen (L"windowsshell.manifest");
	if (cchdir + cchfile + 1 > RTL_NUMBER_OF (buf)) return FALSE;

	wcscpy_s (buf + cchdir, cchfile + 1, L"windowsshell.manifest");

	/*  Create an activation context from the manifest, and then
	    activate the context.  */

	ACTCTX actctx;
	actctx.cbSize = sizeof (actctx);
	actctx.dwFlags = 0;
	actctx.lpSource = buf;

	m_hActCtx = CreateActCtx (&actctx);
	if (m_hActCtx != INVALID_HANDLE_VALUE) {

	    if (ActivateActCtx (m_hActCtx, &m_ulCookie)) return TRUE;

	    ReleaseActCtx (m_hActCtx);
	    m_hActCtx = INVALID_HANDLE_VALUE;
	}

	return FALSE;
    };

    ~CComctl32Selection (VOID)
    {
	if (m_hActCtx != INVALID_HANDLE_VALUE) {
	    DeactivateActCtx (0, m_ulCookie);
	    ReleaseActCtx (m_hActCtx);
	}
    };
};

/*  ========================================================================  */
/*  Messages to User  */

VOID PutMessageBox (HWND hWnd, UINT Icon, PCWSTR Format, va_list Args)
{
    WCHAR buf [0x0100];
    int cch = vswprintf_s (buf, RTL_NUMBER_OF (buf), Format, Args);
    if (cch ++ > 0) {
	PWCHAR text = buf;
	if (cch >= RTL_NUMBER_OF (buf)) {
	    PWCHAR p = new WCHAR [cch];
	    if (p != NULL) {
		vswprintf_s (p, cch, Format, Args);
		text = p;
	    }
	}
	MessageBox (hWnd, text, g_szCaption, MB_OK | Icon);
	if (text != buf) delete text;
    }
}

VOID Report (HWND hWnd, PCWSTR Format, ...)
{
    va_list args;
    va_start (args, Format);
    PutMessageBox (hWnd, MB_ICONINFORMATION, Format, args);
    va_end (args);
}

VOID PutWarning (HWND hWnd, PCWSTR Format, ...)
{
    va_list args;
    va_start (args, Format);
    PutMessageBox (hWnd, MB_ICONWARNING, Format, args);
    va_end (args);
}

VOID PutError (HWND hWnd, PCWSTR Format, ...)
{
    va_list args;
    va_start (args, Format);
    PutMessageBox (hWnd, MB_ICONERROR, Format, args);
    va_end (args);
}

VOID PutMemoryError (HWND hWnd)
{
    PutError (hWnd, L"Insufficient memory");
}

VOID ReportColour (HWND hWnd, PCWSTR Name, COLORREF Color)
{
    if (Color == CLR_NONE) {
	Report (hWnd, L"No %s colour", Name);
    }
    else if (Color == CLR_DEFAULT) {
	Report (hWnd, L"Default %s colour", Name);
    }
    else {
	BYTE r = GetRValue (Color);
	BYTE g = GetGValue (Color);
	BYTE b = GetBValue (Color);
	Report (hWnd, L"%s colour is %d,%d,%d", Name, r, g, b);
    }
};

/*  ========================================================================  */
/*  Arbitrary List-View  */

class CListView
{
    protected:

    HWND m_hWnd;
    IListView *m_IListView;
    BOOL m_UseMessages;

    protected:

    CListView (VOID)
    {
	m_hWnd = NULL;
	m_IListView = NULL;
	m_UseMessages = TRUE;
    };

    ~CListView (VOID)
    {
	m_UseMessages = TRUE;
	if (m_IListView != NULL) m_IListView -> Release ();
	if (m_hWnd != NULL) DestroyWindow (m_hWnd);
    };

    BOOL
    InitWindow (
	DWORD dwExStyle,
	PCWSTR lpWindowName,
	DWORD dwStyle,
	int x,
	int y,
	int nWidth,
	int nHeight,
	HWND hWndParent,
	UINT id,
	HINSTANCE hInstance)
    {
	/*  Tell COMCTL32 that we want it for List-View controls.  */

	if (g_bIccLv) {
	    INITCOMMONCONTROLSEX icc;
	    icc.dwSize = sizeof (INITCOMMONCONTROLSEX);
	    icc.dwICC  = ICC_LISTVIEW_CLASSES;
	    if (NOT InitCommonControlsEx (&icc)) {
		PutError (NULL, L"InitCommonControlsEx failed "
				    L"for List-View control");
		return FALSE;
	    }
	}

	m_hWnd = CreateWindowEx (dwExStyle, WC_LISTVIEW, lpWindowName,
				    dwStyle, x, y, nWidth, nHeight,
				    hWndParent, (HMENU) id, hInstance, NULL);

	return m_hWnd != NULL ? TRUE : FALSE;
    };

    BOOL InitInterface (BOOL bUseMessages)
    {
	if (:: SendMessage (m_hWnd, LVM_QUERYINTERFACE,
		(WPARAM) &__uuidof (IListView), (LPARAM) &m_IListView)) {
	    m_UseMessages = bUseMessages;
	    return TRUE;
	}
	else {
	    m_UseMessages = TRUE;
	    return bUseMessages;
	}
    };

    public:

    HWND GetWindow (VOID) const
    {
	return m_hWnd;
    };

    /*  To forward an arbitrary message  */

    LRESULT SendMessage (UINT uMsg, WPARAM wParam, LPARAM lParam)
    {
	return :: SendMessage (m_hWnd, uMsg, wParam, lParam);
    };

    /*	The following functions each do one operation on the List-View
	control, either by sending a message or by calling the IListView
	interface (aiming to mimic how COMCTL32 itself handles the message
	by translating to an interface method). Add functions as the need
	for them arises.  */

    COLORREF GetBkColor (VOID)
    {
	if (NOT m_UseMessages) {
	    COLORREF result;
	    m_IListView -> GetBackgroundColor (&result);
	    return result;
	}
	else {
	    return ListView_GetBkColor (m_hWnd);
	}
    };

    BOOL GetBkImage (LVBKIMAGE *plvbki)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> GetBackgroundImage (plvbki);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_GetBkImage (m_hWnd, plvbki);
	}
    };

    DWORD GetExtendedListViewStyle (VOID)
    {
	if (NOT m_UseMessages) {
	    DWORD result;
	    m_IListView -> GetExtendedStyle (&result);
	    return result;
	}
	else {
	    return ListView_GetExtendedListViewStyle (m_hWnd);
	}
    };

    int GetItemCount (VOID)
    {
	if (NOT m_UseMessages) {
	    int result;
	    m_IListView -> GetItemCount (&result);
	    return result;
	}
	else {
	    return ListView_GetItemCount (m_hWnd);
	}
    };

    BOOL GetItemRect (int i, RECT *prc, int code)
    {
	if (NOT m_UseMessages) {
	    LVITEMINDEX lvii;
	    lvii.iItem = i;
	    lvii.iGroup = -1;	    // might not suffice if LVS_OWNERDATA
	    HRESULT hr = m_IListView -> GetItemRect (lvii, code, prc);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_GetItemRect (m_hWnd, i, prc, code);
	}
    };

    UINT GetSelectedColumn (VOID)
    {
	if (NOT m_UseMessages) {
	    int result;
	    m_IListView -> GetSelectedColumn (&result);
	    return result;
	}
	else {
	    return ListView_GetSelectedColumn (m_hWnd);
	}
    };

    int InsertColumn (int iCol, LVCOLUMN * const pcol)
    {
	if (NOT m_UseMessages) {
	    int result;
	    m_IListView -> InsertColumn (iCol, pcol, &result);
	    return result;
	}
	else {
	    return ListView_InsertColumn (m_hWnd, iCol, pcol);
	}
    };

    int InsertItem (LVITEM * const pitem)
    {
	if (NOT m_UseMessages) {
	    int result;
	    m_IListView -> InsertItem (pitem, &result);
	    return result;
	}
	else {
	    return ListView_InsertItem (m_hWnd, pitem);
	}
    };

    BOOL SetBkColor (COLORREF clrBk)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> SetBackgroundColor (clrBk);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_SetBkColor (m_hWnd, clrBk);
	}
    };

    BOOL SetBkImage (LVBKIMAGE * const plvbki)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> SetBackgroundImage (plvbki);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_SetBkImage (m_hWnd, plvbki);
	}
    };

    DWORD SetExtendedListViewStyleEx (DWORD dwExMask, DWORD dwExStyle)
    {
	if (NOT m_UseMessages) {
	    DWORD result;
	    m_IListView -> SetExtendedStyle (dwExMask, dwExStyle, &result);
	    return result;
	}
	else {
	    return ListView_SetExtendedListViewStyleEx (m_hWnd, dwExMask,
							dwExStyle);
	}
    };

    HIMAGELIST SetImageList (HIMAGELIST himl, int iImageList)
    {
	if (NOT m_UseMessages) {
	    HIMAGELIST result;
	    HRESULT hr = m_IListView -> SetImageList (iImageList, himl,
							&result);
	    return SUCCEEDED (hr) ? result : NULL;
	}
	else {
	    return ListView_SetImageList (m_hWnd, himl, iImageList);
	}
    };

    BOOL SetItem (LVITEM * const pitem)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> SetItem (pitem);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_SetItem (m_hWnd, pitem);
	}
    };

    BOOL SetSelectedColumn (int iCol)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> SetSelectedColumn (iCol);
	    return SUCCEEDED (hr) ? TRUE : FALSE;
	}
	else {
	    return ListView_SetSelectedColumn (m_hWnd, iCol);
	}
    };

    BOOL SetTextBkColor (COLORREF clrText)
    {
	if (NOT m_UseMessages) {
	    m_IListView -> SetTextBackgroundColor (clrText);
	    return TRUE;
	}
	else {
	    return ListView_SetTextBkColor (m_hWnd, clrText);
	}
    };

    int SetView (DWORD iView)
    {
	if (NOT m_UseMessages) {
	    HRESULT hr = m_IListView -> SetView (iView);
	    return SUCCEEDED (hr) ? 1 : -1;
	}
	else {
	    return ListView_SetView (m_hWnd, iView);
	}
    };

    /*	Operations done in old ways  */

    VOID SetViewByWindowStyle (DWORD dwType)
    {
	if (dwType & ~LVS_TYPEMASK) return;

	DWORD style = GetWindowLong (m_hWnd, GWL_STYLE);
	style = (style & ~LVS_TYPEMASK) | (dwType & LVS_TYPEMASK);
	SetWindowLong (m_hWnd, GWL_STYLE, style);
    };

    /*	Operations that aren't supported with window messages  */

    COLORREF GetSortColor (VOID)
    {
	if (m_IListView == NULL) return FALSE;

	IVisualProperties *ivp;
	HRESULT hr = m_IListView -> QueryInterface <IVisualProperties> (&ivp);
	if (SUCCEEDED (hr)) {
	    COLORREF clr;
	    hr = ivp -> GetColor (VPCF_SORTCOLUMN, &clr);
	    ivp -> Release ();
	    if (SUCCEEDED (hr)) return clr;
	}
	return CLR_DEFAULT;
    };

    BOOL SetSortColor (COLORREF clrSort)
    {
	if (m_IListView == NULL) return FALSE;

	IVisualProperties *ivp;
	HRESULT hr = m_IListView -> QueryInterface <IVisualProperties> (&ivp);
	if (SUCCEEDED (hr)) {
	    hr = ivp -> SetColor (VPCF_SORTCOLUMN, clrSort);
	    ivp -> Release ();
	    if (SUCCEEDED (hr)) return TRUE;
	}
	return FALSE;
    };
};

/*  ************************************************************************  */
/*  Things specific to the List-View Background Test  */

/*  The program has a main window with a List-View control as a child
    window. The List-View is controlled (tested) by selecting from the main
    window's menu, the general idea being that each menu item corresponds to
    one operation (window message, interface call) on the List-View.  */

/*  ========================================================================  */
/*  Dialog boxes  */

/*  Where an operation takes parameters, the menu item selects a dialog box
    and the operation is performed when the dialog box closes (unless
    cancelled). Different dialog boxes get different parameters, but in
    essentially the same way. In particular, they can have a common dialog
    procedure. Their differences are accommodated through two virtual
    functions. One initialises the dialog from persistent settings, if any
    yet exist. The other acts on the new input, both to update the
    persistent settings and to operate on the List-View control.  */

class DECLSPEC_NOVTABLE CDlgBase
{
    protected:

    PWSTR m_lpTemplateName;
    CListView *m_ListView;

    public:

    static INT_PTR CALLBACK DlgProc (HWND, UINT, WPARAM, LPARAM);

    INT_PTR Show (HWND hwndParent, CListView *ListView)
    {
	m_ListView = ListView;
	return DialogBoxParam (g_hInstance, m_lpTemplateName,
				hwndParent, DlgProc, (LPARAM) this);
    };

    protected:

    virtual BOOL OnInit (HWND) = 0;
    virtual BOOL OnOk (HWND) = 0;
};

INT_PTR
CALLBACK
CDlgBase :: DlgProc (
    HWND hwndDlg,
    UINT uMsg,
    WPARAM wParam,
    LPARAM lParam)
{
    CDlgBase *dlg;
    if (uMsg == WM_INITDIALOG) {
	dlg = (CDlgBase *) lParam;
	SetWindowLongPtr (hwndDlg, DWLP_USER, (LONG_PTR) dlg);
	return dlg -> OnInit (hwndDlg);
    }
    else if (uMsg == WM_COMMAND) {
	UINT id = LOWORD (wParam);
	if (id == IDOK) {
	    dlg = (CDlgBase *) GetWindowLongPtr (hwndDlg, DWLP_USER);
	    if (dlg -> OnOk (hwndDlg)) EndDialog (hwndDlg, 1);
	    return TRUE;
	}
	else if (id == IDCANCEL) {
	    EndDialog (hwndDlg, 0);
	    return TRUE;
	}
    }
    return FALSE;
}

/*  ------------------------------------------------------------------------  */
/*  A few helper functions  */

BOOL IsDlgItemChecked (HWND hwnd, UINT id)
{
    LRESULT state = SendDlgItemMessage (hwnd, id, BM_GETCHECK, 0, 0);
    return state == BST_CHECKED ? TRUE : FALSE;
}

VOID CheckDlgItem (HWND hwnd, UINT id, BOOL check)
{
    WPARAM state = check ? BST_CHECKED : BST_UNCHECKED;
    SendDlgItemMessage (hwnd, id, BM_SETCHECK, state, 0);
}

VOID SetDlgItemFocus (HWND hwnd, UINT id)
{
    SetFocus (GetDlgItem (hwnd, id));
    SendDlgItemMessage (hwnd, id, EM_SETSEL, 0, -1);
}

BOOL GetDlgItemColorComponent (HWND hwnd, UINT id, BYTE *p)
{
    BOOL ok;
    UINT c = GetDlgItemInt (hwnd, id, &ok, FALSE);
    if (NOT ok) return FALSE;
    if ((BYTE) c != c) return FALSE;
    *p = c;
    return TRUE;
}

HBITMAP LoadBitmap (PCWSTR Filename)
{
    HANDLE h = LoadImage (NULL, Filename, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE);
    return (HBITMAP) h;
}

/*  ------------------------------------------------------------------------  */
/*  Dialog Boxes for Setting a Colour  */

class DECLSPEC_NOVTABLE CColorDlg : public CDlgBase
{
    protected:

    COLORREF m_Color;

    public:

    CColorDlg (VOID)
    {
	m_lpTemplateName = MAKEINTRESOURCE (IDD_COLOR);
	m_Color = CLR_NONE;
    };

    protected:

    BOOL SetDlgItems (HWND hwndDlg)
    {
	if (m_Color == CLR_NONE) return TRUE;

	BYTE r = GetRValue (m_Color);
	BYTE g = GetGValue (m_Color);
	BYTE b = GetBValue (m_Color);

	SetDlgItemInt (hwndDlg, IDC_RED, r, FALSE);
	SetDlgItemInt (hwndDlg, IDC_GREEN, g, FALSE);
	SetDlgItemInt (hwndDlg, IDC_BLUE, b, FALSE);

	return TRUE;
    };

    BOOL GetDlgItems (HWND hwndDlg)
    {
	BYTE r, g, b;
	if (NOT GetDlgItemColorComponent (hwndDlg, IDC_RED, &r)) {
	    SetDlgItemFocus (hwndDlg, IDC_RED);
	    return FALSE;
	}
	if (NOT GetDlgItemColorComponent (hwndDlg, IDC_GREEN, &g)) {
	    SetDlgItemFocus (hwndDlg, IDC_GREEN);
	    return FALSE;
	}
	if (NOT GetDlgItemColorComponent (hwndDlg, IDC_BLUE, &b)) {
	    SetDlgItemFocus (hwndDlg, IDC_BLUE);
	    return FALSE;
	}
	m_Color = RGB (r, g, b);
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Box for Setting Background Colour  */

class CBkColorDlg : public CColorDlg
{
    protected:

    BOOL OnInit (HWND hwndDlg)
    {
	return SetDlgItems (hwndDlg);
    };

    BOOL OnOk (HWND hwndDlg)
    {
	if (NOT GetDlgItems (hwndDlg)) return FALSE;

	BOOL ok = m_ListView -> SetBkColor (m_Color);
	if (NOT ok) Report (hwndDlg, L"SetBkColor returned FALSE");
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Box for Setting Selected Column Colour  */

class CSortColorDlg : public CColorDlg
{
    protected:

    BOOL OnInit (HWND hwndDlg)
    {
	return SetDlgItems (hwndDlg);
    };

    BOOL OnOk (HWND hwndDlg)
    {
	if (NOT GetDlgItems (hwndDlg)) return FALSE;

	BOOL ok = m_ListView -> SetSortColor (m_Color);
	if (NOT ok) Report (hwndDlg, L"SetSortColor returned FALSE");
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Boxes for Setting Background Image  */

class DECLSPEC_NOVTABLE CImageDlg : public CDlgBase
{
    protected:

    PWSTR m_pszImage;
    BOOL m_Tile;
    BOOL m_TileOffset;
    int m_xOffsetPercent;
    int m_yOffsetPercent;

    public:

    CImageDlg (VOID)
    {
	m_pszImage = NULL;
    };

    ~CImageDlg (VOID)
    {
	if (m_pszImage != NULL) delete m_pszImage;
    };

    protected:

    BOOL SetDlgItems (HWND hwndDlg)
    {
	if (m_pszImage == NULL) return TRUE;

	SetDlgItemText (hwndDlg, IDC_IMAGE, m_pszImage);
	CheckDlgItem (hwndDlg, IDC_TILE, m_Tile);
	CheckDlgItem (hwndDlg, IDC_TILEOFFSET, m_TileOffset);
	SetDlgItemInt (hwndDlg, IDC_XOFFSET, m_xOffsetPercent, TRUE);
	SetDlgItemInt (hwndDlg, IDC_YOFFSET, m_yOffsetPercent, TRUE);

	return TRUE;
    };

    BOOL GetDlgItems (HWND hwndDlg)
    {
	m_Tile = IsDlgItemChecked (hwndDlg, IDC_TILE);
	m_TileOffset = IsDlgItemChecked (hwndDlg, IDC_TILEOFFSET);

	BOOL ok;
	m_xOffsetPercent = GetDlgItemInt (hwndDlg, IDC_XOFFSET, &ok, TRUE);
	if (NOT ok) {
	    SetDlgItemFocus (hwndDlg, IDC_XOFFSET);
	    return FALSE;
	}
	m_yOffsetPercent = GetDlgItemInt (hwndDlg, IDC_YOFFSET, &ok, TRUE);
	if (NOT ok) {
	    SetDlgItemFocus (hwndDlg, IDC_YOFFSET);
	    return FALSE;
	}

	WCHAR filename [MAX_PATH];
	UINT cch = GetDlgItemText (hwndDlg, IDC_IMAGE,
				    filename, RTL_NUMBER_OF (filename));
	if (cch == 0) {
	    SetDlgItemFocus (hwndDlg, IDC_IMAGE);
	    return FALSE;
	}
	if (m_pszImage == NULL OR wcscmp (filename, m_pszImage) != 0) {
	    if (m_pszImage != NULL) delete m_pszImage;
	    m_pszImage = _wcsdup (filename);
	    if (m_pszImage == NULL) {
		PutMemoryError (hwndDlg);
		return FALSE;
	    }
	}
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Box for Setting Background Image from Bitmap Handle	*/

class CBitmapDlg : public CImageDlg
{
    public:

    CBitmapDlg (VOID)
    {
	m_lpTemplateName = MAKEINTRESOURCE (IDD_BITMAP);
    };

    protected:

    BOOL OnInit (HWND hwndDlg)
    {
	return SetDlgItems (hwndDlg);
    };

    BOOL OnOk (HWND hwndDlg)
    {
	if (NOT GetDlgItems (hwndDlg)) return FALSE;

	LVBKIMAGE lvbki = {0};

	lvbki.ulFlags = LVBKIF_SOURCE_HBITMAP;
	if (m_Tile) lvbki.ulFlags |= LVBKIF_STYLE_TILE;
	if (m_TileOffset) lvbki.ulFlags |= LVBKIF_FLAG_TILEOFFSET;

	lvbki.hbm = LoadBitmap (m_pszImage);
	if (lvbki.hbm == NULL) {
	    PutError (hwndDlg, L"Error loading bitmap from %s", m_pszImage);
	    SetDlgItemFocus (hwndDlg, IDC_IMAGE);
	    return FALSE;
	}

	lvbki.xOffsetPercent = m_xOffsetPercent;
	lvbki.yOffsetPercent = m_yOffsetPercent;

	BOOL ok = m_ListView -> SetBkImage (&lvbki);
	if (NOT ok) Report (hwndDlg, L"SetBkImage returned FALSE");
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Box for Setting Background Image from URL  */

class CUrlDlg : public CImageDlg
{
    public:

    CUrlDlg (VOID)
    {
	m_lpTemplateName = MAKEINTRESOURCE (IDD_URL);
    };

    protected:

    BOOL OnInit (HWND hwndDlg)
    {
	return SetDlgItems (hwndDlg);
    };

    BOOL OnOk (HWND hwndDlg)
    {
	if (NOT GetDlgItems (hwndDlg)) return FALSE;

	LVBKIMAGE lvbki = {0};

	lvbki.ulFlags = LVBKIF_SOURCE_URL;
	if (m_Tile) lvbki.ulFlags |= LVBKIF_STYLE_TILE;
	if (m_TileOffset) lvbki.ulFlags |= LVBKIF_FLAG_TILEOFFSET;

	lvbki.pszImage = m_pszImage;

	lvbki.xOffsetPercent = m_xOffsetPercent;
	lvbki.yOffsetPercent = m_yOffsetPercent;

	BOOL ok = m_ListView -> SetBkImage (&lvbki);
	if (NOT ok) Report (hwndDlg, L"SetBkImage returned FALSE");
	return TRUE;
    };
};

/*  ------------------------------------------------------------------------  */
/*  Dialog Box for Setting Watermark  */

class CWatermarkDlg : public CDlgBase
{
    protected:

    PWSTR m_pszImage;
    BOOL m_AlphaBlend;

    public:

    CWatermarkDlg (VOID)
    {
	m_lpTemplateName = MAKEINTRESOURCE (IDD_WATERMARK);
	m_pszImage = NULL;
    };

    ~CWatermarkDlg (VOID)
    {
	if (m_pszImage != NULL) delete m_pszImage;
    };

    protected:

    BOOL SetDlgItems (HWND hwndDlg)
    {
	if (m_pszImage == NULL) return TRUE;

	SetDlgItemText (hwndDlg, IDC_IMAGE, m_pszImage);
	CheckDlgItem (hwndDlg, IDC_ALPHABLEND, m_AlphaBlend);

	return TRUE;
    };

    BOOL GetDlgItems (HWND hwndDlg)
    {
	m_AlphaBlend = IsDlgItemChecked (hwndDlg, IDC_ALPHABLEND);

	WCHAR filename [MAX_PATH];
	UINT cch = GetDlgItemText (hwndDlg, IDC_IMAGE,
				    filename, RTL_NUMBER_OF (filename));
	if (cch == 0) {
	    SetDlgItemFocus (hwndDlg, IDC_IMAGE);
	    return FALSE;
	}
	if (m_pszImage == NULL OR wcscmp (filename, m_pszImage) != 0) {
	    if (m_pszImage != NULL) delete m_pszImage;
	    m_pszImage = _wcsdup (filename);
	    if (m_pszImage == NULL) {
		PutMemoryError (hwndDlg);
		return FALSE;
	    }
	}
	return TRUE;
    };

    BOOL OnInit (HWND hwndDlg)
    {
	return SetDlgItems (hwndDlg);
    };

    BOOL OnOk (HWND hwndDlg)
    {
	if (NOT GetDlgItems (hwndDlg)) return FALSE;

	LVBKIMAGE lvbki = {0};

	lvbki.ulFlags = LVBKIF_TYPE_WATERMARK;
	if (m_AlphaBlend) lvbki.ulFlags |= LVBKIF_FLAG_ALPHABLEND;

	lvbki.hbm = LoadBitmap (m_pszImage);
	if (lvbki.hbm == NULL) {
	    PutError (hwndDlg, L"Error loading bitmap from %s", m_pszImage);
	    SetDlgItemFocus (hwndDlg, IDC_IMAGE);
	    return FALSE;
	}

	BOOL ok = m_ListView -> SetBkImage (&lvbki);
	if (NOT ok) Report (hwndDlg, L"SetBkImage returned FALSE");
	return TRUE;
    };
};

/*  ========================================================================  */
/*  The Sample List-View  */

class CLVBkTest : public CListView
{
    /*	Brushes for a custom background (visible when background colour is
	cleared)  */

    HBRUSH m_BackgroundBrush;
    HBRUSH m_ItemBrushes [2];
    BOOL m_HaveItemBrushes;

    CLVBkTest (VOID)
    {
	m_BackgroundBrush = CreateSolidBrush (RGB (0x80, 0x80, 0xC0));
	m_ItemBrushes [0] = CreateSolidBrush (RGB (0xC0, 0x80, 0x80));
	m_ItemBrushes [1] = CreateSolidBrush (RGB (0x80, 0xC0, 0x80));

	m_HaveItemBrushes = TRUE;
	for (ULONG n = 0; n < RTL_NUMBER_OF (m_ItemBrushes); n ++) {
	    if (m_ItemBrushes [n] == NULL) {
		m_HaveItemBrushes = FALSE;
		break;
	    }
	}
    };

    public:

    ~CLVBkTest (VOID)
    {
	if (m_BackgroundBrush != NULL) DeleteObject (m_BackgroundBrush);
	for (ULONG n = 0; n < RTL_NUMBER_OF (m_ItemBrushes); n ++) {
	    if (m_ItemBrushes [n] != NULL) DeleteObject (m_ItemBrushes [n]);
	}
    };

    static CLVBkTest *Create (DWORD, DWORD, int, int, int, int, HWND, UINT);

    BOOL DrawBackground (HDC);
    BOOL Populate (VOID);
};

CLVBkTest *
CLVBkTest :: Create (
    DWORD dwExStyle,
    DWORD dwStyle,
    int X,
    int Y,
    int nWidth,
    int nHeight,
    HWND hWndParent,
    UINT id)
{
    CLVBkTest *lv = new CLVBkTest;
    if (lv == NULL) return NULL;

    if (lv -> InitWindow (dwExStyle, L"", dwStyle, X, Y, nWidth, nHeight,
				    hWndParent, id, g_hInstance)) {

	if (NOT lv -> InitInterface (g_bMessages)) {

	    PutWarning (lv -> m_hWnd,
		    L"No IListView interface found - "
		    L"falling back to sending window messages. "
		    L"To avoid this warning, run without /messages:off.");
	}
	return lv;
    }
    return NULL;
}

/*  ------------------------------------------------------------------------  */
/*  A custom background for the sample List-View  */

/*  The following function expects the DC to have window coordinates aligned
    to the List-View control's client area.  */

BOOL CLVBkTest :: DrawBackground (HDC hDC)
{
    /*	The aim here isn't to do anything fancy - just something that
	couldn't easily be done with the features that COMCTL32 implements
	from within the control.

	All we do is give different items different background colours, set
	against yet another background colour for any of the List-View that
	remains.  */

    if (m_BackgroundBrush == NULL) return FALSE;

    RECT rlv;
    GetClientRect (GetWindow (), &rlv);

    FillRect (hDC, &rlv, m_BackgroundBrush);

    if (NOT m_HaveItemBrushes) return TRUE;

    int itemcount = GetItemCount ();
    int ibrush = 0;
    for (int i = 0; i < itemcount; i ++) {

	RECT ritem;
	if (NOT GetItemRect (i, &ritem, LVIR_BOUNDS)) continue;

	if (IntersectRect (&ritem, &ritem, &rlv)) {
	    FillRect (hDC, &ritem, m_ItemBrushes [ibrush]);
	}
	if (++ ibrush == RTL_NUMBER_OF (m_ItemBrushes)) ibrush = 0;
    }
    return TRUE;
}

/*  ------------------------------------------------------------------------  */
/*  A small amount of sample data for populating the List-View - pretty
    much copied from Microsoft's own SDK sample  */

struct PET
{
    PWSTR pszIcon;
    PWSTR pszKind;
    PWSTR pszBreed;
    PWSTR pszPrice;
};

PWSTR const rgColumns [] = {L"Kind", L"Breed", L"Price"};

PET const rgPets [] = {
    {MAKEINTRESOURCE (IDI_DOG), L"Dog", L"Poodle", L"$300.00"},
    {MAKEINTRESOURCE (IDI_CAT), L"Cat", L"Siamese", L"$100.00"},
    {MAKEINTRESOURCE (IDI_FISH), L"Fish", L"Angel Fish", L"$10.00"},
};

#define NUM_COLUMNS RTL_NUMBER_OF (rgColumns)
#define NUM_PETS    RTL_NUMBER_OF (rgPets)

/*  A function to set this sample data into the sample List-View  */

BOOL CLVBkTest :: Populate (VOID)
{
    /*	Prepare images for the sample data.  */

    HIMAGELIST hsmall = ImageList_Create (16, 16, FALSE, NUM_PETS, 0);
    if (hsmall == NULL) return FALSE;

    HIMAGELIST hlarge = ImageList_Create (32, 32, FALSE, NUM_PETS, 0);
    if (hlarge == NULL) return FALSE;

    for (ULONG n = 0; n < NUM_PETS; n ++) {
	HICON hicon = LoadIcon (g_hInstance, rgPets [n].pszIcon);
	if (ImageList_AddIcon (hsmall, hicon) == -1) return FALSE;
	if (ImageList_AddIcon (hlarge, hicon) == -1) return FALSE;
    }

    SetImageList (hsmall, LVSIL_SMALL);
    SetImageList (hlarge, LVSIL_NORMAL);

    /*	Define some columns.  */

    for (ULONG n = 0; n < NUM_COLUMNS; n ++) {

	LVCOLUMN lvc = {0};
	lvc.mask = LVCF_TEXT | LVCF_WIDTH;
	lvc.pszText = rgColumns [n];
	lvc.cx = 100;

	if (InsertColumn (n, &lvc) == -1) return FALSE;
    }

    /*	Add the items.	*/

    for (ULONG n = 0; n < NUM_PETS; n ++) {

	LVITEM lvi = {0};
	lvi.mask = LVIF_TEXT | LVIF_IMAGE;
	lvi.iItem = n;
	lvi.iImage = n;
	lvi.iSubItem = 0;
	lvi.pszText = rgPets [n].pszKind;

	if (InsertItem (&lvi) == -1) return FALSE;

	lvi.iItem = n;
	lvi.iImage = n;
	lvi.iSubItem = 1;
	lvi.pszText = rgPets [n].pszBreed;

	if (SetItem (&lvi) == -1) return FALSE;

	lvi.iItem = n;
	lvi.iImage = n;
	lvi.iSubItem = 2;
	lvi.pszText = rgPets [n].pszPrice;

	if (SetItem (&lvi) == -1) return FALSE;
    }

    return TRUE;
}

/*  ========================================================================  */
/*  Main Window  */

class CMainWindow
{
    ATOM m_aWndClass;

    /*	The sample List-View control  */

    CLVBkTest *m_ListView;
    POINT m_ListViewPosition;

    /*	For persistent data in dialog boxes raised by various menu items  */

    CBkColorDlg *m_BkColorDlg;
    CBitmapDlg *m_BitmapDlg;
    CUrlDlg *m_UrlDlg;
    CWatermarkDlg *m_WatermarkDlg;
    CSortColorDlg *m_SortColorDlg;

    CMainWindow (VOID);

    public:

    ~CMainWindow (VOID);

    static CMainWindow *Create (int);

    private:

    static LRESULT CALLBACK WndProc (HWND, UINT, WPARAM, LPARAM);

    BOOL OnCreate (HWND);
    VOID OnDestroy (VOID);
    BOOL OnEraseBkGnd (HWND, HDC);
    VOID OnSysColorChange (WPARAM, LPARAM);
    VOID OnSettingChange (WPARAM, LPARAM);
    BOOL OnNotify (WPARAM, LPARAM);
    BOOL OnCommand (HWND, WPARAM, LPARAM);
    BOOL OnPrintClient (HWND, HDC);

    BOOL DrawBackground (HWND, HDC);
};

CMainWindow :: CMainWindow (VOID)
{
    WNDCLASS wc;

    wc.style = CS_VREDRAW;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = sizeof (CMainWindow *);
    wc.hInstance = g_hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
    wc.lpszMenuName = MAKEINTRESOURCE (IDC_MENU);
    wc.lpszClassName = g_szCaption;

    m_aWndClass = RegisterClass (&wc);

    m_ListView = NULL;
    m_BkColorDlg = NULL;
    m_BitmapDlg = NULL;
    m_UrlDlg = NULL;
    m_WatermarkDlg = NULL;
    m_SortColorDlg = NULL;
}

CMainWindow :: ~CMainWindow (VOID)
{
    if (m_BkColorDlg != NULL) delete m_BkColorDlg;
    if (m_BitmapDlg != NULL) delete m_BitmapDlg;
    if (m_UrlDlg != NULL) delete m_UrlDlg;
    if (m_WatermarkDlg != NULL) delete m_WatermarkDlg;
    if (m_SortColorDlg != NULL) delete m_SortColorDlg;

    if (m_ListView != NULL) delete m_ListView;
    if (m_aWndClass != 0) UnregisterClass ((LPCWSTR) m_aWndClass, g_hInstance);
}

CMainWindow *CMainWindow :: Create (int nCmdShow)
{
    CMainWindow *mw = new CMainWindow;
    if (mw == NULL) {
	PutMemoryError (NULL);
    }
    else if (mw -> m_aWndClass == 0) {
	PutError (NULL, L"Error registering window class");
    }
    else {

	HWND hwnd = CreateWindowEx (0, g_szCaption, g_szCaption,
				    WS_OVERLAPPEDWINDOW,
				    CW_USEDEFAULT, CW_USEDEFAULT,
				    CW_USEDEFAULT, CW_USEDEFAULT,
				    NULL, NULL, g_hInstance, mw);
	if (hwnd == NULL) {
	    PutError (NULL, L"Error creating main window");
	}
	else {

	    ShowWindow (hwnd, nCmdShow);
	    UpdateWindow (hwnd);

	    return mw;
	}
	delete mw;
    }
    return NULL;
};

LRESULT
CALLBACK
CMainWindow :: WndProc (
    HWND hwnd,
    UINT uMsg,
    WPARAM wParam,
    LPARAM lParam)
{
    CMainWindow *mw;
    if (uMsg == WM_NCCREATE) {
	LPCREATESTRUCT cs = (LPCREATESTRUCT) lParam;
	mw = (CMainWindow *) cs -> lpCreateParams;
	if (mw != NULL) SetWindowLongPtr (hwnd, 0, (LONG_PTR) mw);
    }
    else {
	mw = (CMainWindow *) GetWindowLongPtr (hwnd, 0);
    }
    if (mw != NULL) {

	switch (uMsg) {
	    case WM_CREATE: {
		return mw -> OnCreate (hwnd) ? 0 : -1;
		break;
	    }
	    case WM_DESTROY: {
		mw -> OnDestroy ();
		PostQuitMessage (0);
		return 0;
	    }
	    case WM_ERASEBKGND: {
		if (mw -> OnEraseBkGnd (hwnd, (HDC) wParam)) return TRUE;
		break;
	    }
	    case WM_SYSCOLORCHANGE: {
		mw -> OnSysColorChange (wParam, lParam);
		break;
	    }
	    case WM_SETTINGCHANGE: {
		mw -> OnSettingChange (wParam, lParam);
		break;
	    }
	    case WM_NOTIFY: {
		if (mw -> OnNotify (wParam, lParam)) return 0;
		break;
	    }
	    case WM_COMMAND: {
		if (mw -> OnCommand (hwnd, wParam, lParam)) return 0;
		break;
	    }
	    case WM_PRINTCLIENT: {
		if (mw -> OnPrintClient (hwnd, (HDC) wParam)) return TRUE;
		break;
	    }
	}
    }
    return DefWindowProc (hwnd, uMsg, wParam, lParam);
}

BOOL CMainWindow :: OnCreate (HWND hWnd)
{
    /*	Create a List-View control with the position and size given on the
	command line, else as the whole of the parent window's client area.  */

    int x = g_ListViewPosition.x;
    int y = g_ListViewPosition.y;
    int cx = g_ListViewSize.cx;
    int cy = g_ListViewSize.cy;

    if (cx == 0 OR cy == 0) {

	RECT rwnd;
	GetClientRect (hWnd, &rwnd);

	int width = rwnd.right - rwnd.left;
	int height = rwnd.bottom - rwnd.top;

	if (cx == 0) cx = width > x ? width - x : 0;
	if (cy == 0) cy = height > y ? height - y : 0;
    }

    DWORD ws = WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;
    m_ListView = CLVBkTest :: Create (0, ws, x, y, cx, cy, hWnd, IDC_LISTVIEW);
    if (m_ListView != NULL) {

	m_ListViewPosition.x = x;
	m_ListViewPosition.y = y;

	m_ListView -> SetTextBkColor (CLR_NONE);

	/*  Initialise this List-View with the sample data.  */

	if (m_ListView -> Populate ()) return TRUE;

	delete m_ListView;
	m_ListView = NULL;
    }
    return FALSE;
}

VOID CMainWindow :: OnDestroy (VOID)
{
    if (m_ListView != NULL) {
	delete m_ListView;
	m_ListView = NULL;
    }
}

/*  All windows that have common controls must forward WM_SYSCOLORCHANGE to
    each of the controls. Microsoft's documentation even makes a point of
    spelling this out in a few places.	*/

VOID CMainWindow :: OnSysColorChange (WPARAM wParam, LPARAM lParam)
{
    CListView *lv = m_ListView;
    if (lv != NULL) lv -> SendMessage (WM_SYSCOLORCHANGE, wParam, lParam);
}

/*  The same seems just as true of WM_SETTINGCHANGE, no matter that
    Microsoft's documentation does and doesn't say. Put it this way, forward
    WM_SETTINGCHANGE to the List-View and it will respond immediately to a
    change in the "Use a background image for each folder type" setting.
    Don't forward the message and COMCTL32 will not know of the change and
    cannot update the List-View to conform to the new setting.	*/

VOID CMainWindow :: OnSettingChange (WPARAM wParam, LPARAM lParam)
{
    CListView *lv = m_ListView;
    if (lv != NULL) lv -> SendMessage (WM_SETTINGCHANGE, wParam, lParam);
}

/*  Some user-interface operations on a List-View control are not handled
    internally by the control but are delegated to the parent window. The
    case of interest here is to support selection of a column by clicking
    the column header.	*/

BOOL CMainWindow :: OnNotify (WPARAM wParam, LPARAM lParam)
{
    NMHDR *pnmh = (NMHDR *) lParam;

    /*	Check that the notification has come from a known window (though
	we presently have only one).  */

    if (pnmh -> idFrom == IDC_LISTVIEW) {

	CListView *lv = m_ListView;
	if (lv == NULL) return FALSE;

	switch (pnmh -> code) {

	    case LVN_COLUMNCLICK: {

		NMLISTVIEW *pnmv = (NMLISTVIEW *) pnmh;
		UINT clkcol = pnmv -> iSubItem;
		UINT selcol = lv -> GetSelectedColumn ();
		lv -> SetSelectedColumn (clkcol != selcol ? clkcol : -1);

		return TRUE;
	    }
	}
    }
    return FALSE;
}

/*  ------------------------------------------------------------------------  */
/*  Backgrounds  */

/*  When the List-View control has no background colour, the control's
    parent window will receive a WM_ERASEBKGND message whenever the
    control's background is redrawn. Presumably, the idea is that the
    parent redraws its own background and that of the control.

    Some detection and correction - not done here - is required if the
    COMCTL32 in use for the List-View is older than version 6.00. The DC
    in a WM_ERASEBKGND sent from COMCTL32 has window coordinates aligned
    to the List-View.  */

BOOL CMainWindow :: OnEraseBkGnd (HWND hWnd, HDC hDC)
{
    if (NOT DrawBackground (hWnd, hDC)) return FALSE;

    if (m_ListView != NULL) {
	HWND hwndlv = m_ListView -> GetWindow ();
	POINT org;
	org.x = m_ListViewPosition.x;
	org.y = m_ListViewPosition.y;
	if (OffsetWindowOrgEx (hDC, -org.x, -org.y, &org)) {
	    m_ListView -> DrawBackground (hDC);
	    SetWindowOrgEx (hDC, org.x, org.y, NULL);
	}
    }
    return TRUE;
}

/*  When the List-View control has the LVS_EX_TRANSPARENTBKGND extended
    style, the control's parent window will receive a WM_PRINTCLIENT
    message whenever the control's background is redrawn, i.e., whenever the
    control would otherwise get a WM_ERASEBKGND. Presumably, the idea is
    that the parent redraws its own background as if the control were not
    visible.  */

BOOL CMainWindow :: OnPrintClient (HWND hWnd, HDC hDC)
{
    return DrawBackground (hWnd, hDC);
}

BOOL CMainWindow :: DrawBackground (HWND hWnd, HDC hDC)
{
    /*	Since the main window exists only as a setting for the sample
	List-View, give it a background that's likely to stand out against
	any background that the menu operations produce for the List-View.  */

    RECT rwnd;
    GetClientRect (hWnd, &rwnd);

    TRIVERTEX vertices [2];

    vertices [0].x = rwnd.left;
    vertices [0].y = rwnd.top;
    vertices [0].Red = 0xC0 << 8;
    vertices [0].Green = 0xC0 << 8;
    vertices [0].Blue = 0x80 << 8;
    vertices [0].Alpha = 0;

    vertices [1].x = rwnd.right;
    vertices [1].y = rwnd.bottom;
    vertices [1].Red = 0x80 << 8;
    vertices [1].Green = 0x80 << 8;
    vertices [1].Blue = 0xC0 << 8;
    vertices [1].Alpha = 0;

    GRADIENT_RECT gr;
    gr.UpperLeft = 0;
    gr.LowerRight = 1;

    GradientFill (hDC, vertices, RTL_NUMBER_OF (vertices), &gr, 1,
		    GRADIENT_FILL_RECT_V);

    return TRUE;
}

/*  ------------------------------------------------------------------------  */
/*  Menu Operation  */

BOOL CMainWindow :: OnCommand (HWND hWnd, WPARAM wParam, LPARAM lParam)
{
    if (HIWORD (wParam) != 0) return FALSE;

    CListView *lv = m_ListView;
    if (lv == NULL) return FALSE;

    switch (LOWORD (wParam)) {

	/*  Window  */

	case IDM_REDRAW: {
	    HWND hwndlv = lv -> GetWindow ();
	    RedrawWindow (hwndlv, NULL, NULL, RDW_ERASE | RDW_INVALIDATE);
	    return TRUE;
	}

	case IDM_ENABLE: {
	    HWND hwndlv = lv -> GetWindow ();
	    EnableWindow (hwndlv, TRUE);
	    return TRUE;
	}

	case IDM_DISABLE: {
	    HWND hwndlv = lv -> GetWindow ();
	    EnableWindow (hwndlv, FALSE);
	    return TRUE;
	}

	/*  Views  */

	case IDM_VIEW_ICON: {
	    lv -> SetView (LV_VIEW_ICON);
	    return TRUE;
	}

	case IDM_VIEW_DETAILS: {
	    lv -> SetView (LV_VIEW_DETAILS);
	    return TRUE;
	}

	case IDM_VIEW_SMALLICON: {
	    lv -> SetView (LV_VIEW_SMALLICON);
	    return TRUE;
	}

	case IDM_VIEW_LIST: {
	    lv -> SetView (LV_VIEW_LIST);
	    return TRUE;
	}

	case IDM_VIEW_TILE: {
	    lv -> SetView (LV_VIEW_TILE);
	    return TRUE;
	}

	/*  Styles  */

	case IDM_STYLE_ICON: {
	    lv -> SetViewByWindowStyle (LVS_ICON);
	    return TRUE;
	}

	case IDM_STYLE_REPORT: {
	    lv -> SetViewByWindowStyle (LVS_REPORT);
	    return TRUE;
	}

	case IDM_STYLE_SMALLICON: {
	    lv -> SetViewByWindowStyle (LVS_SMALLICON);
	    return TRUE;
	}

	case IDM_STYLE_LIST: {
	    lv -> SetViewByWindowStyle (LVS_LIST);
	    return TRUE;
	}

	/*  Extended Styles  */

	case IDM_TRANSPARENTBKGND_SET: {
	    lv -> SetExtendedListViewStyleEx (LVS_EX_TRANSPARENTBKGND,
						LVS_EX_TRANSPARENTBKGND);
	    return TRUE;
	}

	case IDM_TRANSPARENTBKGND_CLEAR: {
	    lv -> SetExtendedListViewStyleEx (LVS_EX_TRANSPARENTBKGND, 0);
	    return TRUE;
	}

	/*  Background	*/

	case IDM_BKCOLOR_SET: {
	    if (m_BkColorDlg == NULL) m_BkColorDlg = new CBkColorDlg;
	    if (m_BkColorDlg == NULL) PutMemoryError (hWnd);
	    else m_BkColorDlg -> Show (hWnd, lv);
	    return TRUE;
	}

	case IDM_BKCOLOR_CLEAR: {
	    BOOL ok = lv -> SetBkColor (CLR_NONE);
	    if (NOT ok) Report (hWnd, L"SetBkColor returned FALSE");
	    return TRUE;
	}

	case IDM_BKCOLOR_GET: {
	    COLORREF clrbk = lv -> GetBkColor ();
	    ReportColour (hWnd, L"Background", clrbk);
	    return TRUE;
	}

	case IDM_BKIMAGE_SET_HBITMAP: {
	    if (m_BitmapDlg == NULL) m_BitmapDlg = new CBitmapDlg;
	    if (m_BitmapDlg == NULL) PutMemoryError (hWnd);
	    else m_BitmapDlg -> Show (hWnd, lv);
	    return TRUE;
	}

	case IDM_BKIMAGE_NULL_HBITMAP: {

	    LVBKIMAGE lvbki = {0};

	    lvbki.ulFlags = LVBKIF_SOURCE_HBITMAP;
	    lvbki.hbm = NULL;

	    BOOL ok = lv -> SetBkImage (&lvbki);
	    if (NOT ok) Report (hWnd, L"SetBkImage returned FALSE");
	    return TRUE;
	}

	case IDM_BKIMAGE_SET_URL: {
	    if (m_UrlDlg == NULL) m_UrlDlg = new CUrlDlg;
	    if (m_UrlDlg == NULL) PutMemoryError (hWnd);
	    else m_UrlDlg -> Show (hWnd, lv);
	    return TRUE;
	}

	case IDM_BKIMAGE_NULL_URL: {

	    LVBKIMAGE lvbki = {0};

	    lvbki.ulFlags = LVBKIF_SOURCE_URL;
	    lvbki.pszImage = NULL;

	    BOOL ok = lv -> SetBkImage (&lvbki);
	    if (NOT ok) Report (hWnd, L"SetBkImage returned FALSE");
	    return TRUE;
	}

	case IDM_BKIMAGE_CLEAR: {

	    LVBKIMAGE lvbki = {0};

	    lvbki.ulFlags = LVBKIF_SOURCE_NONE;

	    BOOL ok = lv -> SetBkImage (&lvbki);
	    if (NOT ok) Report (hWnd, L"SetBkImage returned FALSE");
	    return TRUE;
	}

	case IDM_BKIMAGE_GET: {

	    WCHAR url [MAX_PATH] = L"";

	    LVBKIMAGE lvbki = {0};

	    lvbki.pszImage = url;
	    lvbki.cchImageMax = RTL_NUMBER_OF (url);

	    BOOL ok = lv -> GetBkImage (&lvbki);
	    if (NOT ok) {
		Report (hWnd, L"GetBkImage returned FALSE");
		return TRUE;
	    }

	    Report (hWnd,
		    L"ulFlags = 0x%08X\n"
		    L"hbm = 0x%08X\n"
		    L"pszImage -> %s\n"
		    L"xOffsetPercent = %d\n"
		    L"yOffsetPercent = %d",
		    lvbki.ulFlags, lvbki.hbm, lvbki.pszImage,
			lvbki.xOffsetPercent, lvbki.yOffsetPercent);

	    return TRUE;
	}

	case IDM_WATERMARK_SET: {
	    if (m_WatermarkDlg == NULL) m_WatermarkDlg = new CWatermarkDlg;
	    if (m_WatermarkDlg == NULL) PutMemoryError (hWnd);
	    else m_WatermarkDlg -> Show (hWnd, lv);
	    return TRUE;
	}

	case IDM_WATERMARK_CLEAR: {

	    LVBKIMAGE lvbki = {0};

	    lvbki.ulFlags = LVBKIF_TYPE_WATERMARK;
	    lvbki.hbm = NULL;

	    BOOL ok = lv -> SetBkImage (&lvbki);
	    if (NOT ok) Report (hWnd, L"SetBkImage returned FALSE");
	    return TRUE;
	}

	case IDM_WATERMARK_GET: {

	    LVBKIMAGE lvbki = {0};

	    lvbki.ulFlags = LVBKIF_TYPE_WATERMARK;

	    BOOL ok = lv -> GetBkImage (&lvbki);
	    if (NOT ok) {
		Report (hWnd, L"GetBkImage returned FALSE");
		return TRUE;
	    }

	    Report (hWnd, L"ulFlags = 0x%08X\nhbm = 0x%08X",
		    lvbki.ulFlags, lvbki.hbm);

	    return TRUE;
	}

	/*  Columns  */

	case IDM_SELECTEDCOLUMN_GET: {
	    UINT selcol = lv -> GetSelectedColumn ();
	    if (selcol == -1) Report (hWnd, L"No column selected");
	    else Report (hWnd, L"Selected column is %d", selcol);
	    return TRUE;
	}

	case IDM_SORTCOLOR_SET: {
	    if (m_SortColorDlg == NULL) m_SortColorDlg = new CSortColorDlg;
	    if (m_SortColorDlg == NULL) PutMemoryError (hWnd);
	    else m_SortColorDlg -> Show (hWnd, lv);
	    return TRUE;
	}

	case IDM_SORTCOLOR_CLEAR: {
	    BOOL ok = lv -> SetSortColor (CLR_DEFAULT);
	    if (NOT ok) Report (hWnd, L"SetSortColor returned FALSE");
	    return TRUE;
	}

	case IDM_SORTCOLOR_GET: {
	    COLORREF clr = lv -> GetSortColor ();
	    ReportColour (hWnd, L"Selected Column", clr);
	    return TRUE;
	}
    }
    return FALSE;
}

/*  ========================================================================  */
/*  Command-line Options  */

BOOL ParseBool (PCWSTR CmdLine, PCWSTR Option, BOOL *Value)
{
    int cch = wcslen (Option);
    if (_wcsnicmp (CmdLine, Option, cch) != 0) return FALSE;

    PCWSTR p = CmdLine + cch;
    if (*p == L'\0') {
	*Value = TRUE;
	return TRUE;
    }
    if (*p != L':') return FALSE;
    p ++;

    PCWSTR on [] = {L"on", L"true", L"yes", L"1"};
    PCWSTR off [] = {L"off", L"false", L"no", L"0"};

    ULONG n;
    for (n = 0; n < RTL_NUMBER_OF (on); n ++) {
	if (_wcsicmp (p, on [n]) == 0) {
	    *Value = TRUE;
	    return TRUE;
	}
    }
    for (n = 0; n < RTL_NUMBER_OF (off); n ++) {
	if (_wcsicmp (p, off [n]) == 0) {
	    *Value = FALSE;
	    return TRUE;
	}
    }
    return FALSE;
}

BOOL
ParseNumbers (
    PCWSTR CmdLine,
    PCWSTR Option,
    ULONG *Numbers,
    ULONG *Count)
{
    int cch = wcslen (Option);
    if (_wcsnicmp (CmdLine, Option, cch) != 0) return FALSE;

    PCWSTR p = CmdLine + cch;
    if (*p != L':') return FALSE;
    p ++;

    ULONG numwanted = *Count;
    ULONG numgot = 0;

    BOOL getnum = TRUE;

    while (*p != L'\0' AND getnum) {

	PWSTR endptr;
	ULONG n = wcstoul (p, &endptr, 0);
	if (n == 0 AND endptr == p) return FALSE;

	if (numgot < numwanted) Numbers [numgot ++] = n;

	p = endptr;
	if (*p == L',') {
	    if (numgot == numwanted) return FALSE;
	    p ++;
	}
	else {
	    getnum = FALSE;
	}
    }

    *Count = numgot;

    return TRUE;
}

BOOL ParsePoint (PCWSTR CmdLine, PCWSTR Option, POINT *Point)
{
    ULONG point [2];
    ULONG count = RTL_NUMBER_OF (point);
    if (NOT ParseNumbers (CmdLine, Option, point, &count)) return FALSE;
    if (count != RTL_NUMBER_OF (point)) return FALSE;
    Point -> x = point [0];
    Point -> y = point [1];
    return TRUE;
}

BOOL ParseSize (PCWSTR CmdLine, PCWSTR Option, SIZE *Size)
{
    ULONG size [2];
    ULONG count = RTL_NUMBER_OF (size);
    if (NOT ParseNumbers (CmdLine, Option, size, &count)) return FALSE;
    if (count != RTL_NUMBER_OF (size)) return FALSE;
    Size -> cx = size [0];
    Size -> cy = size [1];
    return TRUE;
}

VOID ShowCmdLineOptions (VOID)
{
    MessageBox (NULL, g_szDescription, g_szCaption, MB_OK);
}

BOOL ParseCmdLine (PCWSTR pszCmdLine)
{
    /*	For the convenience of C-style parsing with argc and argv, and with
	it taken for granted that argv [0] names the executable, ignore the
	given command line and work instead with what's returned by the
	GetCommandLine function.  */

    int argc;
    PWSTR *argv = CommandLineToArgvW (GetCommandLine (), &argc);
    if (argv == NULL) return FALSE;
    if (argc == 0) return FALSE;

    BOOL result = TRUE;

    while (++ argv, -- argc != 0) {
	PWSTR arg = *argv;
	PWCHAR p = arg;

	if (*p == L'-' OR *p == L'/') {
	    p ++;

	    /*	Try to interpret this argument as a supported switch.  */

	    if (ParseBool (p, L"cominit", &g_bComInit)) continue;
	    if (ParseBool (p, L"icclv", &g_bIccLv)) continue;
	    if (ParseBool (p, L"manifest", &g_bManifest)) continue;
	    if (ParseBool (p, L"messages", &g_bMessages)) continue;
	    if (ParseBool (p, L"actctx", &g_bActCtx)) continue;
	    if (ParseBool (p, L"iccstd", &g_bIccStd)) continue;
	    if (ParsePoint (p, L"pos", &g_ListViewPosition)) continue;
	    if (ParseSize (p, L"size", &g_ListViewSize)) continue;

	    /*	For anything else, just decide whether to point at the
		error or show the syntax.  */

	    result = FALSE;

	    BOOL help = FALSE;
	    if (NOT ParseBool (p, L"?", &help)) ParseBool (p, L"help", &help);

	    if (help) {
		ShowCmdLineOptions ();
		break;
	    }

	    PutError (NULL, L"Invalid command-line option: %s", arg);
	    break;
	}
	else {

	    /*	At present, the only supported arguments are switches.	*/

	    result = FALSE;
	    PutError (NULL, L"Invalid command-line parameter: %s", arg);
	    break;
	}
    }
    LocalFree (argv);
    return result;
}

/*  ========================================================================  */
/*  Effective Entry Point to Program - Finally!  */

int
WINAPI
wWinMain (
    HINSTANCE hInstance,
    HINSTANCE hPrevInstance,
    PWSTR lpCmdLine,
    int nCmdShow)
{
    g_hInstance = hInstance;

    if (NOT ParseCmdLine (lpCmdLine)) return -1;

    CComInitialisation com;
    if (NOT com.Init (g_bComInit)) {
	PutError (NULL, L"COM initialisation failed");
	return -1;
    }

    CComctl32Selection comctl32;
    if (NOT comctl32.Init (g_bManifest)) {
	PutError (NULL, L"Unable to activate COMCTL32 assembly");
	return -1;
    }

    if (g_bIccStd) {
	INITCOMMONCONTROLSEX icc;
	icc.dwSize = sizeof (icc);
	icc.dwICC = ICC_STANDARD_CLASSES;
	if (NOT InitCommonControlsEx (&icc)) {
	    PutError (NULL, L"InitCommonControlsEx failed "
				L"for standard controls");
	    return -1;
	}
    }

    CMainWindow *mw = CMainWindow :: Create (nCmdShow);
    if (mw == NULL) return -1;

    int exitcode = -1;
    for (;;) {
	MSG msg;
	int result = GetMessage (&msg, NULL, 0, 0);
	if (result == -1) break;
	if (result == 0) {
	    exitcode = msg.wParam;
	    break;
	}
	TranslateMessage (&msg);
	DispatchMessage (&msg);
    }

    delete mw;
    return exitcode;
}

/*  ************************************************************************  */

