
/*  ************************************************************************  *
 *			      Task Bar Icons Test			      *
 *  ************************************************************************  */

/*  This is the source file for a program that demonstrates the effect that
    the TaskbarExceptionsIcons registry key has on the "Group similar
    taskbar buttons" feature.

    Compile and link to taste.

    Run on Windows XP or higher.

    The command-line syntax is simple. One argument is permitted. It is an
    icon path, i.e., a pathname to the file that contains the desired icon,
    then a comma, and finally a decimal index for the desired icon within
    the named file's resources.

    The specified icon is loaded in response to WM_GETITEM messages, e.g.,
    when EXPLORER wants to know what icon to use when representing the
    program on the taskbar.

    Copyright (C) 2008. Geoff Chappell. All rights reserved.  */

/*  ************************************************************************  */

#define     UNICODE
#define     WIN32_LEAN_AND_MEAN
#define     _WIN32_WINNT    0x0400
#include    <windows.h>

#include    <shellapi.h>
#include    <shlwapi.h>

#pragma     comment (lib, "shell32.lib")
#pragma     comment (lib, "shlwapi.lib")
#pragma     comment (lib, "user32.lib")

#include    <stdio.h>
#include    <string.h>

/*  A few things the author happens to like  */

#define     AND &&
#define     NOT !
#define     OR	||

/*  ************************************************************************  */
/*  Global Data and Forward References	*/

#define CLASS_NAME		    L"TaskbarExceptionsIcons Test"
#define CMDLINE_INTRO		    L" - "

WCHAR g_szCaption [] = CLASS_NAME;
HINSTANCE g_hInstance = NULL;

PWSTR g_lpCmdLine = NULL;

WCHAR g_szIconPath [MAX_PATH] = L"";
int g_iIconIndex = 0;

/*  ************************************************************************  */

VOID PutMessageBox (HWND hWnd, UINT Icon, PCWSTR Format, va_list Args)
{
    WCHAR buf [0x0100];
    int cch = vswprintf_s (buf, RTL_NUMBER_OF (buf), Format, Args);
    if (cch ++ > 0) {
	PWCHAR text = buf;
	if (cch >= RTL_NUMBER_OF (buf)) {
	    PWCHAR p = new WCHAR [cch];
	    if (p != NULL) {
		vswprintf_s (p, cch, Format, Args);
		text = p;
	    }
	}
	MessageBox (hWnd, text, g_szCaption, MB_OK | Icon);
	if (text != buf) delete text;
    }
}

VOID PutError (HWND hWnd, PCWSTR Format, ...)
{
    va_list args;
    va_start (args, Format);
    PutMessageBox (hWnd, MB_ICONERROR, Format, args);
    va_end (args);
}

VOID PutMemoryError (HWND hWnd)
{
    PutError (hWnd, L"Insufficient memory");
}

/*  ************************************************************************  */
/*  Main Window  */

class CMainWindow
{
    ATOM m_aWndClass;
    HICON m_hIcon;
    BOOL m_bDisableGetIcon;

    CMainWindow (VOID);

    public:

    ~CMainWindow (VOID);

    static CMainWindow *Create (int);

    private:

    static LRESULT CALLBACK WndProc (HWND, UINT, WPARAM, LPARAM);

    HICON OnGetIcon (HWND, WPARAM);
};

CMainWindow :: CMainWindow (VOID)
{
    m_aWndClass = 0;
    m_hIcon = NULL;
    m_bDisableGetIcon = FALSE;

    WNDCLASS wc;

    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = sizeof (CMainWindow *);
    wc.hInstance = g_hInstance;
    wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = CLASS_NAME;

    m_aWndClass = RegisterClass (&wc);
    if (m_aWndClass == 0) {
	DWORD ec = GetLastError ();
	PutError (NULL, L"Error 0x%08X registering window class", ec);
    }
}

CMainWindow :: ~CMainWindow (VOID)
{
    if (m_hIcon != NULL) DestroyIcon (m_hIcon);
    if (m_aWndClass != 0) UnregisterClass ((LPCWSTR) m_aWndClass, g_hInstance);
}

CMainWindow *CMainWindow :: Create (int nCmdShow)
{
    CMainWindow *mw = new CMainWindow;
    if (mw == NULL) {
	PutMemoryError (NULL);
    }
    else if (mw -> m_aWndClass != 0) {

	WCHAR windowname [MAX_PATH];
	wcscpy_s (windowname, RTL_NUMBER_OF (windowname), g_lpCmdLine);

	HWND hwnd = CreateWindowEx (0, CLASS_NAME, windowname,
				    WS_OVERLAPPEDWINDOW,
				    CW_USEDEFAULT, CW_USEDEFAULT,
				    CW_USEDEFAULT, CW_USEDEFAULT,
				    NULL, NULL, g_hInstance, mw);

	if (hwnd == NULL) {
	    PutError (NULL, L"Error creating main window");
	}
	else {

	    ShowWindow (hwnd, nCmdShow);
	    UpdateWindow (hwnd);

	    return mw;
	}
	delete mw;
    }
    return NULL;
};

LRESULT
CALLBACK
CMainWindow :: WndProc (
    HWND hwnd,
    UINT uMsg,
    WPARAM wParam,
    LPARAM lParam)
{
    CMainWindow *mw;
    if (uMsg == WM_NCCREATE) {
	LPCREATESTRUCT cs = (LPCREATESTRUCT) lParam;
	mw = (CMainWindow *) cs -> lpCreateParams;
	if (mw != NULL) {
	    SetWindowLongPtr (hwnd, 0, (LONG_PTR) mw);
	}
    }
    else {
	mw = (CMainWindow *) GetWindowLongPtr (hwnd, 0);
    }
    if (mw != NULL) {

	switch (uMsg) {
	    case WM_CREATE: {
		return 0;
	    }
	    case WM_DESTROY: {
		PostQuitMessage (0);
		return 0;
	    }
	    case WM_GETICON: {
		HICON hicon = mw -> OnGetIcon (hwnd, wParam);
		if (hicon != NULL) return (LRESULT) hicon;
	    }
	}
    }
    return DefWindowProc (hwnd, uMsg, wParam, lParam);
}

HICON CMainWindow :: OnGetIcon (HWND hWnd, WPARAM wParam)
{
    if (m_bDisableGetIcon) return NULL;

    switch (wParam) {
	case ICON_SMALL: {
	    if (m_hIcon == NULL AND g_szIconPath [0] != L'\0') {
		ExtractIconEx (g_szIconPath, g_iIconIndex, NULL, &m_hIcon, 1);
		if (m_hIcon == NULL) {
		    m_bDisableGetIcon = TRUE;
		    DWORD ec = GetLastError ();
		    PutError (hWnd, L"Error 0x%08X extracting icon %s,%d",
				ec, g_szIconPath, g_iIconIndex);
		}
	    }
	    return m_hIcon;
	}
    }
    return NULL;
}

/*  ************************************************************************  */
/*  Command-Line Parsing  */

BOOL ParseCmdLine (PCWSTR pszCmdLine)
{
    /*	For the convenience of C-style parsing with argc and argv, and with
	it taken for granted that argv [0] names the executable, ignore the
	given command line and work instead with what's returned by the
	GetCommandLine function.  */

    int argc;
    PWSTR *argv = CommandLineToArgvW (GetCommandLine (), &argc);
    if (argv == NULL) return FALSE;
    if (argc == 0) return FALSE;

    BOOL result = TRUE;

    while (++ argv, -- argc != 0) {
	PWSTR arg = *argv;
	PWCHAR p = arg;

	if (*p == L'-' OR *p == L'/') {
	    p ++;

	    PutError (NULL, L"Invalid command-line option: %s", arg);
	    break;
	}
	else {

	    if (g_szIconPath [0] == L'\0'
		    AND wcslen (p) + 1 <= RTL_NUMBER_OF (g_szIconPath)) {
		wcscpy_s (g_szIconPath, RTL_NUMBER_OF (g_szIconPath), p);
		g_iIconIndex = PathParseIconLocation (g_szIconPath);
		continue;
	    }

	    PutError (NULL, L"Invalid command-line parameter: %s", arg);
	    break;
	}
    }
    LocalFree (argv);
    return result;
}

/*  ************************************************************************  */
/*  Effective Entry Point to Program  */

int
WINAPI
wWinMain (
    HINSTANCE hInstance,
    HINSTANCE hPrevInstance,
    PWSTR lpCmdLine,
    int nCmdShow)
{
    g_hInstance = hInstance;
    g_lpCmdLine = GetCommandLine ();

    if (NOT ParseCmdLine (lpCmdLine)) return -1;

    CMainWindow *mw = CMainWindow :: Create (nCmdShow);
    if (mw == NULL) return -1;

    int exitcode = -1;
    for (;;) {
	MSG msg;
	int result = GetMessage (&msg, NULL, 0, 0);
	if (result == -1) break;
	if (result == 0) {
	    exitcode = msg.wParam;
	    break;
	}
	TranslateMessage (&msg);
	DispatchMessage (&msg);
    }

    delete mw;
    return exitcode;
}

/*  ************************************************************************  */

