
/*  ************************************************************************  *
 *				  dpamerge.cpp				      *
 *  ************************************************************************  */

/*  This is the source file for a program that demonstrates the DPA_Merge
    function.

    Compile to taste, but with include files from the Windows Vista SDK.
    Link with import libraries from the Windows Vista SDK.

    (The Windows Vista SDK not only declares the DPA functions but also
    supports them in the COMCTL32 import library.)

    Copyright (C) 2006-2009. Geoff Chappell. All rights reserved.  */

#define     WIN32_LEAN_AND_MEAN
#include    <windows.h>

#include    <commctrl.h>

#include    <stdio.h>
#include    <stdlib.h>

#pragma comment (lib, "comctl32.lib")

/*  A few things the author happens to like  */

#define     AND &&
#define     NOT !
#define     OR	||

/*  ************************************************************************  */

const CHAR Description [] =
    "\nDPAMERGE: demonstrates DPA_Merge function"
    "\n"
    "\n  DPAMERGE [options] set1 set2"
    "\n"
    "\nOptions:"
    "\n"
    "\n  /i    form intersection"
    "\n  /s    do NOT sort before merge"
    "\n  /u    form union"
    "\n  /v    verbose report"
    "\n"
    "\nEach set is a sequence of numbers in standard set notation."
    "\nThe set is enclosed in curly braces."
    "\nThe numbers are separated by commas."
    "\nDo not leave spaces between braces, numbers and commas."
    "\n";

BOOL bVerbose = FALSE;

/*  ************************************************************************  */
/*  Forward references	*/

HDPA LoadDpa (PCSTR);
VOID UnloadDpa (HDPA);

VOID PutError (PCSTR, ...);

/*  ************************************************************************  */

/*  The LoadDpa function creates a DPA and loads it with pointers according
    to some input given as text. The particular interpretation is that the
    text gives a set of numbers in standard set notation. Each pointer in
    the DPA is just a pointer to a ULONG.

    The DumpDpa function reverses the work of LoadDpa in the sense of
    producing a description in text, such as might have been used to load
    the DPA.  */

HDPA LoadDpa (PCSTR Str)
{
    if (*Str ++ != '{') return FALSE;

    HDPA dpa = DPA_Create (16);
    if (dpa == NULL) {
	PutError ("Failed to create DPA");
    }
    else {
	for (;;) {
	    PSTR end;
	    ULONG n = strtoul (Str, &end, 0);
	    if (errno != 0) break;

	    PULONG p = new ULONG;
	    if (p == NULL) {
		PutError ("Insufficient memory for pointer to %u", n);
		break;
	    }

	    *p = n;

	    if (DPA_InsertPtr (dpa, DPA_APPEND, p) == -1) {
		PutError ("Failed to insert pointer to %u", n);
		break;
	    }

	    while (isspace (*end)) {
		end ++;
	    }
	    if (*end == '}') return dpa;
	    else if (*end == ',') Str = end + 1;
	    else break;
	}
	UnloadDpa (dpa);
    }
    return NULL;
}

static int __stdcall DestroyCallback (PVOID p, PVOID pData)
{
    if (p != NULL) delete (PULONG) p;
    return TRUE;
}

VOID UnloadDpa (HDPA Dpa)
{
    DPA_DestroyCallback (Dpa, DestroyCallback, NULL);
}

static int __stdcall EnumCallback (PVOID p, PVOID pData)
{
    BOOL *first = (BOOL *) pData;
    if (*first) {
	printf ("%u", *((PULONG) p));
	*first = FALSE;
    }
    else {
	printf (",%u", *((PULONG) p));
    }
    return TRUE;
}

VOID DumpDpa (HDPA Dpa)
{
    BOOL first = TRUE;
    printf ("{");
    DPA_EnumCallback (Dpa, EnumCallback, &first);
    printf ("}");
}

/*  ========================================================================  */
/*  Callback functions for DPA_Merge  */

/*  Compare pointers in two DPAs simply by the order of the numbers they
    point to.  */

int __stdcall Compare (PVOID p1, PVOID p2, LPARAM lParam)
{
    ULONG n1 = p1 != NULL ? *((PULONG) p1) : 0;
    ULONG n2 = p2 != NULL ? *((PULONG) p2) : 0;

    int result = n1 < n2 ? -1 : n1 > n2 ? 1 : 0;

    if (bVerbose) printf ("Compare callback: %u with %u, returning %d\n",
				n1, n2, result);
    return result;
}

/*  Merge trivially, i.e., by always accepting the first pointer.  */

PVOID __stdcall Merge (UINT uMsg, PVOID p1, PVOID p2, LPARAM lParam)
{
    if (bVerbose) {

	ULONG n1 = p1 != NULL ? *((PULONG) p1) : 0;
	ULONG n2 = p2 != NULL ? *((PULONG) p2) : 0;

	switch (uMsg) {
	    case DPAMM_MERGE: {
		printf ("Merge   callback: DPAMM_MERGE  %u with %u\n", n1, n2);
		break;
	    }
	    case DPAMM_DELETE: {
		printf ("Merge   callback: DPAMM_DELETE %u\n", n1);
		break;
	    }
	    case DPAMM_INSERT: {
		printf ("Merge   callback: DPAMM_INSERT %u\n", n1);
		break;
	    }
	    default: {
		printf ("Merge   callback: 0x%08X, %u, %u\n", uMsg, n1, n2);
		break;
	    }
	}
    }
    return p1;
}

/*  ========================================================================  */
/*  Verbose reporting of DPA_Merge flags  */

static PCSTR GetMergeFlagName (DWORD Flag)
{
    switch (Flag) {
	case DPAM_SORTED:	return "DPAM_SORTED";
	case DPAM_UNION:	return "DPAM_UNION";
	case DPAM_INTERSECT:	return "DPAM_INTERSECT";
	default:		return NULL;
    }
}

VOID DumpMergeFlags (DWORD Flags)
{
    if (Flags == 0) {
	printf ("DPAM_NORMAL");
	return;
    }

    bool first = true;
    for (DWORD n = 1; Flags != 0; Flags >>= 1, n <<= 1) {
	if (Flags & 0x01) {
	    PCSTR name = GetMergeFlagName (n);
	    if (name != NULL) {
		printf (first ? "%s" : " | %s", name);
	    }
	    else {
		printf (first ? "0x%08X" : " | 0x%08X", n);
	    }
	    first = false;
	}
    }
}

/*  ========================================================================  */

int __cdecl main (int argc, char **argv)
{
    DWORD dpam = DPAM_NORMAL;
    HDPA dpa1 = NULL;
    HDPA dpa2 = NULL;

    /*	Parse the command line to get the flags and the two DPAs.  */

    while (++ argv, -- argc != 0) {
	PSTR p = *argv;
	CHAR ch = *p;
	if (ch == '-' OR ch == '/') {
	    p ++;
	    ch = toupper (*p);
	    p ++;
	    if (*p == '\0') {
		if (ch == '?') {
		    printf (Description);
		    return -1;
		}
		else if (ch == 'I') {
		    dpam |= DPAM_INTERSECT;
		    continue;
		}
		else if (ch == 'S') {
		    dpam |= DPAM_SORTED;
		    continue;
		}
		else if (ch == 'U') {
		    dpam |= DPAM_UNION;
		    continue;
		}
		else if (ch == 'V') {
		    bVerbose = TRUE;
		    continue;
		}
	    }
	    PutError ("invalid switch: %s", *argv);
	    return -1;
	}
	else {
	    if (dpa1 == NULL) {
		dpa1 = LoadDpa (p);
		if (dpa1 != NULL) continue;
	    }
	    else if (dpa2 == NULL) {
		dpa2 = LoadDpa (p);
		if (dpa2 != NULL) continue;
	    }
	    else {
		PutError ("too many parameters: %s", p);
		return -1;
	    }
	    PutError ("invalid parameter: %s", *argv);
	    return -1;
	}
    }

    /*	The two DPAs are required.  */

    if (dpa1 == NULL OR dpa2 == NULL) {
	PutError ("expected two lists");
	return -1;
    }

    int exitcode = -1;

    /*	In verbose mode, report what we mean to do.  */

    if (bVerbose) {

	printf ("\nInitial DPA contents: \n");

	printf ("\nTarget DPA = ");
	DumpDpa (dpa1);
	printf ("\nSource DPA = ");
	DumpDpa (dpa2);

	printf ("\n\nMerge (flags = ");
	DumpMergeFlags (dpam);
	printf (")\n");
    }

    /*	Merge the source DPA into the target DPA. The context is not
	actually used in our callbacks. In verbose mode, the callbacks write
	details of what they're doing.  */

    if (NOT DPA_Merge (dpa1, dpa2, dpam, Compare, Merge, 0x12345678)) {
	PutError ("failed to merge DPAs");
    }
    else {

	/*  Show the result.  */

	if (bVerbose) {
	    printf ("\nFinal DPA contents: \n");
	    printf ("\nTarget DPA = ");
	}
	printf ("\n");
	DumpDpa (dpa1);
	if (bVerbose) {
	    printf ("\nSource DPA = ");
	    DumpDpa (dpa2);
	}
	printf ("\n");

	exitcode = 0;
    }

    UnloadDpa (dpa1);
    UnloadDpa (dpa2);

    return exitcode;
}

VOID PutError (PCSTR Format, ...)
{
    va_list args;
    va_start (args, Format);
    printf ("DPAMERGE: ");
    vprintf (Format, args);
    printf ("\n");
    va_end (args);
}

/*  ************************************************************************  */

