
/*  ************************************************************************  *
 *		     diskhook.c - test rig for int 13h hook		      *
 *  ************************************************************************  */

/*  Adapted on 17th March 1995 from version first published in "DOS
    Internals".  Additional modifications on 16th July 1997 for public
    release.  */

#include    <dos.h>
#include    <errno.h>
#include    <process.h>
#include    <signal.h>
#include    <stdarg.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>

#include    "intpack.h"
#include    "memstrat.h"
#include    "standard.h"

#pragma intrinsic (_disable, _enable)

/*  ************************************************************************  */

char description [] =
"\nChanges the appearance of a DOS boot sector to demonstrate hooking"
"\nint 13h.  To ease the coding, the program simply assumes that this"
"\nsector is the one with coordinates cylinder 0, head 1, sector 1 on"
"\ndrive number 80h.  The specific change is to toggle the case of the"
"\nfirst two letters in the word \"disk\" which the sector is expected"
"\nto contain as part of an error message."
"\n"
"\nThe change is temporary.  The program hooks int 13h and then runs a"
"\ncommand processor from which to run other programs such as a disk"
"\nutility or debugger.  When the command processor exits, the program"
"\nunhooks itself from int 13h."
"\n";

char instructions [] =
"\nInt 13h hooked."
"\n"
"\nA command processor is being started to allow investigation of this"
"\nsituation with whatever tools you prefer.  Type EXIT when satisfied."
"\n";

/*  A prototype for our int 13h handler and a pointer to the int 13h handler
    that ours displaces.  */

void __cdecl __interrupt __far Int13h (struct INTPACK);

typedef void __cdecl __interrupt __far INTERRUPT_HANDLER ();
INTERRUPT_HANDLER __far *lpOrgInt13h = NULL;

void PlayWithSectorImage (BYTE _far *);

/*  Characteristics of drive 80h, as determined during our initialisation.  */

BOOL GetDiskCharacteristics (BYTE, WORD *, WORD *, WORD *);

WORD wSectorsPerTrack;
WORD wHeads;

DWORD dwTestSector;

/*  Memory strategy parameters - needed to ensure that the new command
    processor loads in conventional memory  */

WORD wOrgMemoryStrategy;
BOOL bOrgUMBLink;

/*  Exit handling  */

int __cdecl __far HardErrorTrap (unsigned, unsigned, unsigned _far *);
void RestoreInterrupts (void);
void RestoreMemoryUsage (void);

void PutError (char *, ...);

/*  ************************************************************************  */

int main (int argc, char **argv)
{
    register char *pCOMSPEC;
    int exitcode;

    /*	Check for /? on the command line as a request to describe the
	program.  Anything else is invalid.  */

    while (++ argv, -- argc != 0) {
	char *p = *argv;
	if (*p == '/') {
	    if (p [1] == '?') {
		printf (description);
		return (0);
	    }
	    else {
		PutError ("invalid switch - %s", p);
		return (-1);
	    }
	}
	else {
	    PutError ("invalid parameter - %s", p);
	    return (-1);
	}
    }

    /*	Locate the command processor.  */

    pCOMSPEC = getenv ("COMSPEC");
    if (pCOMSPEC == NULL) {
	PutError ("cannot find command processor for test");
	return (-1);
    }

    /*	Get some basic characteristics of the first hard disk.	*/

    if (NOT GetDiskCharacteristics (0x80, NULL, &wHeads, &wSectorsPerTrack)) {
	PutError ("cannot get characteristics of first hard disk");
	return (-1);
    }

    /*	To ease the coding, we choose as the test sector the first sector on
	the disk's second track, noting that this is typically a DOS boot
	sector.  */

    dwTestSector = wSectorsPerTrack;

    /*	Make sure that the program does not terminate without restoring the
	interrupt vector chains to their original condition.  That this
	program executes another would complicate any attempt to deal with
	Ctrl-C non-trivially (whatever the signal function's provisions),
	so just defeat it.  */

    atexit (RestoreInterrupts);     // covers normal exit
    signal (SIGINT, SIG_IGN);	    // prevents termination via int 23h
    _harderr (HardErrorTrap);	    // prevents termination via int 24h

    /*	Now hook int 13h.  */

    lpOrgInt13h = _dos_getvect (0x13);
    _dos_setvect (0x13, Int13h);

    /*	Before running the command processor for the test, remember to force
	the use of conventional memory.  If the upper memory link is enabled
	and a loadhigh strategy set when this program starts, the new
	command processor and all its children will be loaded into upper
	memory (unless the supply is exhausted).  This would limit the
	testing environment (for instance, Windows enhanced mode will not
	start if WIN386 has been loaded into upper memory, or indeed, if it
	has been loaded into conventional memory while the loadhigh strategy
	is in force).  */

    if (_osmajor >= 5) {
	_dos_getstrategy (&wOrgMemoryStrategy);
	_dos_getumblink (&bOrgUMBLink);
	if (wOrgMemoryStrategy != FIRST_FIT OR bOrgUMBLink != NOUMBS) {
	    atexit (RestoreMemoryUsage);
	    _dos_setstrategy (FIRST_FIT);
	    _dos_setumblink (NOUMBS);
	}
    }

    /*	Now run the command processor, as described in the instructions.  */

    printf (instructions);
    exitcode = spawnl (P_WAIT, pCOMSPEC, pCOMSPEC, NULL);
    if (exitcode == -1) {
	PutError ("%s", strerror (errno));
	return (-1);
    }
    else {
	printf ("\nDISKHOOK:  Int 13h hooking test completed.\n");
	return (exitcode);
    }
}

#pragma warning (disable : 4704)

void __cdecl __interrupt __far Int13h (struct INTPACK r)
{
    _enable ();

    /*	Our intest is in simple read and write operations on the first hard
	disk.  */

    if (r.dl == 0x80 AND (r.ah == 0x02 OR r.ah == 0x03)) {

	/*  To the int 13h interface, the starting sector is given in terms
	    of cylinder, head and sector.  Compute it as an offset of
	    sectors from the start of the disk, then determine whether the
	    area of disk affected by the operation includes the test
	    sector.  */

	WORD cyl = (((WORD) r.cl & 0xC0) << 2) | r.ch;
	WORD hd = r.dh;

	if ((r.cl & 0x3F) != 0) {
	    WORD sec = (r.cl & 0x3F) - 1;
	    DWORD start = ((cyl * wHeads) + hd) * wSectorsPerTrack + sec;

	    if (start <= dwTestSector AND dwTestSector < start + r.al) {

		/*  If reading the test sector, let the operation through
		    towards the BIOS, then play with the data that will be
		    seen by the int 13h caller.  */

		if (r.ah == 0x02) {
		    WORD returned_ax;
		    WORD returned_flags;

		    _asm {
			    mov     es,r.es
			    mov     bx,r.bx
			    mov     dx,r.dx
			    mov     cx,r.cx
			    mov     ax,r.ax
			    pushf
			    cli
			    call    lpOrgInt13h

			    push    ax
			    pushf
			    pop     ax
			    mov     returned_flags,ax
			    pop     ax
			    mov     returned_ax,ax
		    }
		    r.flags &= ~0x01;
		    r.flags |= (0x0200 | (returned_flags & 0x01));
		    r.ax = returned_ax;

		    if (NOT (r.flags & 0x01)) {
			PlayWithSectorImage ((BYTE _far *) _MK_FP (r.es, r.bx)
			    + (WORD) (dwTestSector - start) * 0x0200);
		    }
		    return;
		}
		else if (r.ah == 0x03) {

		    /*	Fail any write operation that may affect the test
			sector.  We don't want to risk a program misbehaving
			because of our deception over the contents of the
			sector.  */

		    r.ax = 0x0100;
		    r.flags |= 0x0201;
		    *((BYTE _far *) _MK_FP (0x0040, 0x0074)) = 0x01;
		    return;
		}
	    }
        }
    }
    _chain_intr (lpOrgInt13h);
}

BOOL GetDiskCharacteristics (BYTE Drv, WORD *pCyls, WORD *pHds, WORD *pSPT)
{
    WORD maxcyl;
    WORD maxhd;
    WORD maxspt;

    BOOL result = FALSE;

    _asm {
	    mov     ah,08h
	    mov     dl,Drv
	    int     13h
	    jc	    skip

	    xor     ah,ah
	    mov     al,cl
	    and     al,3Fh
	    mov     maxspt,ax

	    mov     al,dh
	    mov     maxhd,ax

	    mov     al,cl
	    shl     ax,1
            shl     ax,1
	    mov     al,ch
	    mov     maxcyl,ax

	    mov     result,TRUE
	skip:
    }
    if (result) {
	if (pCyls != NULL) *pCyls = maxcyl + 1;
	if (pHds != NULL) *pHds = maxhd + 1;
	if (pSPT != NULL) *pSPT = maxspt;
    }
    return (result);
}

#pragma warning (default : 4704)

void PlayWithSectorImage (BYTE __far *lp)
{
    int n;

    /*	Check that the buffer does plausibly contain the image of a boot
	sector.  */

    if (*((WORD __far *) (lp + 0x01FE)) != 0xAA55) return;

    if (NOT (lp [0] = 0xE9 OR (lp [0] == 0xEB AND lp [2] == 0x90))) return;

    /*	There is very likely to be the word "disk" in an error message
	somewhere in the sector and it should be harmless just to toggle the
	case of a couple of letters in the word.

	I am grateful to Ralph Shnelvar for pointing out to me that my first
	coding of the for loop - with n = 0x200 on entry - could access
	memory beyond the end of the buffer.  */

    for (n = 0x01FD; n != 0; n --) {
	if (lp [0] == 'D' OR lp [0] == 'd') {
	    if (lp [1] == 'I' OR lp [1] == 'i') {
		if (lp [2] == 'S' OR lp [2] == 's') {
		    if (lp [3] == 'K' OR lp [3] == 'k') {
			lp [0] ^= 0x20;
			lp [1] ^= 0x20;
			return;
		    }
		}
	    }
	}
	lp ++;
    }
}

/*  ========================================================================  */

void RestoreMemoryUsage (void)
{
    _dos_setstrategy (wOrgMemoryStrategy);
    _dos_setumblink (bOrgUMBLink);
}

void RestoreInterrupts (void)
{
    if (lpOrgInt13h != NULL) _dos_setvect (0x13, lpOrgInt13h);
}

int __cdecl __far HardErrorTrap
    (unsigned deverror, unsigned errorcode, unsigned _far *devhdr)
{
    return (_HARDERR_FAIL);
}

void PutError (char *str, ...)
{
    va_list va_start (argptr, str);
    printf ("\nDISKHOOK error:  ");
    vprintf (str, argptr);
    printf ("\n");
    va_end (argptr);
}

/*  ************************************************************************  */

