
/*  ************************************************************************  *
 *				  firmware.cpp				      *
 *  ************************************************************************  */

#include    <ntddk.h>

#include    "stddefs.h"

#include    "firmware.h"
#include    "fwmemmap.h"            // shared with Win32 application
#include    "ntddkx.h"

/*  Everything in this source file can be paged.  */

#pragma code_seg ("PAGE")

/*  ************************************************************************  */

void * __cdecl operator new (size_t cb)
{
    return ExAllocatePoolWithTag (PagedPool, cb, 'MMwF');
}

void __cdecl operator delete (void *p)
{
    ExFreePoolWithTag (p, 0);
}

/*  ************************************************************************  */
/*  Access to firmware	*/

class FIRMWARE
{
    USHORT m_Segment;
    USHORT m_Offset;

    public:

    FIRMWARE (VOID)
    {
	m_Segment = 0;
	m_Offset = 0;
    };

    NTSTATUS Init (VOID)
    {
	/*  Get "real mode" memory for an E820_DESCRIPTOR.  */

	ULONG cb = sizeof (E820_DESCRIPTOR);
	NTSTATUS status = x86BiosAllocateBuffer (&cb, &m_Segment, &m_Offset);
	if (NOT NT_SUCCESS (status)) return status;

	/*  Initialisation isn't necessary, but has the merit that we can
	    check that the buffer is usable.  */

	E820_DESCRIPTOR desc = {0};
	return x86BiosWriteMemory (m_Segment, m_Offset, &desc, sizeof (desc));
    };

    NTSTATUS GetNextE820Descriptor (ULONG *Index, E820_DESCRIPTOR *Descriptor)
    {
	X86BIOS_REGISTERS regs = {0};
	NTSTATUS status;

	regs.SegEs = m_Segment;
	regs.Edi = m_Offset;

	regs.Edx = 'SMAP';
	regs.Ecx = sizeof (E820_DESCRIPTOR);
	regs.Ebx = *Index;
	regs.Eax = 0xE820;

	if (NOT x86BiosCall (0x15, &regs)) return STATUS_UNSUCCESSFUL;
	if (regs.Eax != 'SMAP') return STATUS_UNSUCCESSFUL;

	E820_DESCRIPTOR desc = {0};
	ULONG cb = min (regs.Ecx, sizeof (desc));
	status = x86BiosReadMemory (m_Segment, m_Offset, &desc, cb);
	if (NOT NT_SUCCESS (status)) return status;

	*Descriptor = desc;
	*Index = regs.Ebx;
	return STATUS_SUCCESS;
    };

    ~FIRMWARE (VOID)
    {
	if (m_Segment != 0) x86BiosFreeBuffer (m_Segment, m_Offset);
    };
};

/*  ************************************************************************  */
/*  Access to firmware memory map  */

NTSTATUS GetMap (PE820_MAP *Map, ULONG *Size)
{
    /*	Prepare for accessing the BIOS.  */

    FIRMWARE firmware;
    NTSTATUS status = firmware.Init ();
    if (NOT NT_SUCCESS (status)) return status;

    /*	Read successive memory descriptors into a buffer that expands to
	hold them.  */

    PE820_MAP map = NULL;
    ULONG cb = 0;
    PE820_DESCRIPTOR desc = NULL;
    ULONG cbdone = 0;
    ULONG index = 0;
    for (;;) {

	/*  If we don't have a buffer yet or if the one we have is not large
	    enough for another descriptor, change to a new buffer that is
	    big enough.  */

	if (desc == NULL OR cbdone + sizeof (E820_DESCRIPTOR) > cb) {
	    cb += 0x0100;
	    PE820_MAP newmap = (PE820_MAP) new CHAR [cb];
	    if (newmap == NULL) {
		status = STATUS_NO_MEMORY;
		break;
	    }
	    if (cbdone != 0) memcpy (newmap, map, cbdone);
	    memset ((PCHAR) newmap + cbdone, 0x00, cb - cbdone);
	    if (map != NULL) delete map;
	    map = newmap;
	    if (cbdone == 0) {
		map -> Count = 0;
		cbdone = FIELD_OFFSET (E820_MAP, Descriptors);
	    }
	    desc = &map -> Descriptors [map -> Count];
	}

	/*  Read the current descriptor into its place in the buffer.  */

	status = firmware.GetNextE820Descriptor (&index, desc);
	if (NOT NT_SUCCESS (status)) break;

	cbdone += sizeof (E820_DESCRIPTOR);
	map -> Count ++;
	desc ++;

	/*  The descriptors end when the next index returns to zero.  */

	if (index == 0) {
	    *Map = map;
	    *Size = cbdone;
	    return STATUS_SUCCESS;
	}
    }
    delete map;
    return status;
}

VOID ReleaseMap (E820_MAP *Map)
{
    delete (PCHAR) Map;
}

/*  ************************************************************************  */

