
/*  ************************************************************************  *
 *				  fwmemmap.cpp				      *
 *  ************************************************************************  */

#include    <ntddk.h>

#include    "stddefs.h"

#include    "firmware.h"
#include    "fwmemmap.h"            // shared with Win32 application

/*  ************************************************************************  */
/*  Forward references	*/

NTSTATUS NTAPI OnCreate (PDEVICE_OBJECT, PIRP);
NTSTATUS NTAPI OnClose (PDEVICE_OBJECT, PIRP);
NTSTATUS NTAPI OnDeviceControl (PDEVICE_OBJECT, PIRP);
NTSTATUS NTAPI OnMisc (PDEVICE_OBJECT, PIRP);

VOID NTAPI OnUnload (PDRIVER_OBJECT);

NTSTATUS CreateApiDevice (PDRIVER_OBJECT);
VOID DestroyApiDevice (PDRIVER_OBJECT);

NTSTATUS OnApiGetVersion (PVOID, ULONG, ULONG, ULONG *);
NTSTATUS OnApiGetMap (PVOID, ULONG, ULONG, ULONG *);

/*  ************************************************************************  */
/*  Initialisation  */

#pragma code_seg ("INIT")

extern "C"
NTSTATUS
NTAPI
DriverEntry (
    PDRIVER_OBJECT DriverObject,
    PUNICODE_STRING RegistryPath)
{
    /*	This driver exists only for its Device I/O Control interface. Set up
	just the entry points that are needed for this access and for
	unloading the driver when done.  */

    for (int n = 0; n <= IRP_MJ_MAXIMUM_FUNCTION; n ++) {
	DriverObject -> MajorFunction [n] = OnMisc;
    }

    DriverObject -> MajorFunction [IRP_MJ_CREATE] = OnCreate;
    DriverObject -> MajorFunction [IRP_MJ_CLOSE] = OnClose;
    DriverObject -> MajorFunction [IRP_MJ_DEVICE_CONTROL] = OnDeviceControl;

    DriverObject -> DriverUnload = OnUnload;

    /*	Create the device that is to handle the Device I/O Control.  */

    return CreateApiDevice (DriverObject);
}

#pragma code_seg ("PAGE")

VOID NTAPI OnUnload (PDRIVER_OBJECT DriverObject)
{
    DestroyApiDevice (DriverObject);
}

/*  ************************************************************************  */
/*  IRP Entry Points  */

/*  Some shared functions to help with IRP completion  */

#pragma code_seg ()

__forceinline
NTSTATUS IrpDispatchDone (PIRP Irp, NTSTATUS Status)
{
    Irp -> IoStatus.Status = Status;
    Irp -> IoStatus.Information = 0;
    IoCompleteRequest (Irp, IO_NO_INCREMENT);
    return Status;
}

__forceinline
NTSTATUS IrpDispatchDoneEx (
    PIRP Irp,
    NTSTATUS Status,
    ULONG Information)
{
    Irp -> IoStatus.Status = Status;
    Irp -> IoStatus.Information = Information;
    IoCompleteRequest (Irp, IO_NO_INCREMENT);
    return Status;
}

/*  The IRP entry points themselves  */

#pragma code_seg ("PAGE")

NTSTATUS NTAPI OnCreate (PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    /*	As explicitly recommended in the DDK documentation (but not done in
	all of Microsoft's own drivers), check that the caller does indeed
	mean to open a device.	*/

    PIO_STACK_LOCATION sl = IoGetCurrentIrpStackLocation (Irp);
    PFILE_OBJECT fileobj = sl -> FileObject;
    PUNICODE_STRING filename = &(fileobj -> FileName);
    NTSTATUS status = filename -> Length != 0
			? STATUS_INVALID_PARAMETER
			: STATUS_SUCCESS;
    return IrpDispatchDone (Irp, status);
}

NTSTATUS NTAPI OnClose (PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    return IrpDispatchDone (Irp, STATUS_SUCCESS);
}

#pragma code_seg ()

NTSTATUS NTAPI OnDeviceControl (PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    /*	All our expected Device I/O Control codes use buffered I/O.  */

    ULONG numret = 0;
    PVOID buf = Irp -> AssociatedIrp.SystemBuffer;
    PIO_STACK_LOCATION sl = IoGetCurrentIrpStackLocation (Irp);
    ULONG inlen = sl -> Parameters.DeviceIoControl.InputBufferLength;
    ULONG outlen = sl -> Parameters.DeviceIoControl.OutputBufferLength;
    NTSTATUS status;

    /*	For each supported control code (generally referred to here as a
	function), divert to a subroutine.  */

    switch (sl -> Parameters.DeviceIoControl.IoControlCode) {
	case IOCTL_FWMEMMAP_GET_VERSION: {
	    status = OnApiGetVersion (buf, inlen, outlen, &numret);
	    break;
	}
	case IOCTL_FWMEMMAP_GET_MAP: {
	    status = OnApiGetMap (buf, inlen, outlen, &numret);
	    break;
	}
	default: {
	    status = STATUS_INVALID_PARAMETER;
	    break;
	}
    }
    return IrpDispatchDoneEx (Irp, status, numret);
}

NTSTATUS NTAPI OnMisc (PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    return IrpDispatchDone (Irp, STATUS_INVALID_DEVICE_REQUEST);
}

/*  ************************************************************************  */
/*  API Support  */

#define API_DEVICE_NAME     L"\\Device\\"   LTEXT (FWMEMMAP_DEVICE_NAME)
#define API_LINK_NAME	    L"\\??\\"       LTEXT (FWMEMMAP_DEVICE_NAME)

PDEVICE_OBJECT ApiDevice = NULL;

#pragma code_seg ("INIT")

NTSTATUS CreateApiDevice (PDRIVER_OBJECT DriverObject)
{
    /*	First, create the device.  */

    UNICODE_STRING devname;
    RtlInitUnicodeString (&devname, API_DEVICE_NAME);

    PDEVICE_OBJECT devobj;
    NTSTATUS status = IoCreateDevice (DriverObject, 0, &devname,
				FILE_DEVICE_UNKNOWN, 0, TRUE, &devobj);
    if (NT_SUCCESS (status)) {
	UNICODE_STRING linkname;

	/*  Second, create a symbolic link so that the device can be seen
	    from user mode.  */

	RtlInitUnicodeString (&linkname, API_LINK_NAME);

	status = IoCreateSymbolicLink (&linkname, &devname);
	if (NT_SUCCESS (status)) {
	    ApiDevice = devobj;
	    return STATUS_SUCCESS;
	}

	IoDeleteDevice (devobj);
    }
    return status;
}

#pragma code_seg ("PAGE")

VOID DestroyApiDevice (PDRIVER_OBJECT DriverObject)
{
    if (ApiDevice == NULL) return;

    /*	Delete the symbolic link.  */

    UNICODE_STRING linkname;
    RtlInitUnicodeString (&linkname, API_LINK_NAME);
    IoDeleteSymbolicLink (&linkname);

    /*	Delete the API device object.  */

    IoDeleteDevice (ApiDevice);
}

/*  ************************************************************************  */
/*  IOCTL Control Codes  */

#pragma code_seg ("PAGE")

NTSTATUS
OnApiGetVersion (
    PVOID Buffer,
    ULONG InLen,
    ULONG OutLen,
    ULONG *NumRet)
{
    /*	The Get Version function takes no input but produces a ULONG of
	output. Fail if the caller seems not to know this.  */

    if (InLen != 0 OR OutLen != sizeof (ULONG) OR Buffer == NULL) {
	return STATUS_INVALID_PARAMETER;
    }

    *((ULONG *) Buffer) = FWMEMMAP_API_VERSION;
    *NumRet = sizeof (ULONG);
    return STATUS_SUCCESS;
}

NTSTATUS
OnApiGetMap (
    PVOID Buffer,
    ULONG InLen,
    ULONG OutLen,
    ULONG *NumRet)
{
    /*	The Get Map function takes no input but produces at least a minimal
	map as output. Fail if the caller seems not to know this.  */

    if (InLen != 0 OR OutLen < sizeof (E820_MAP) OR Buffer == NULL) {
	return STATUS_INVALID_PARAMETER;
    }

    /*	Copy as output as much of the firmware map as will fit.  */

    PE820_MAP map;
    ULONG cb;
    NTSTATUS status = GetMap (&map, &cb);
    if (NT_SUCCESS (status)) {

	if (cb > OutLen) {
	    cb = OutLen;
	    status = STATUS_BUFFER_OVERFLOW;
	}

	memcpy (Buffer, map, cb);
	*NumRet = cb;

	ReleaseMap (map);
    }
    return status;
}

/*  ************************************************************************  */

