
/*  ************************************************************************  *
 *                                linkcpl.cpp                                 *
 *  ************************************************************************  */

#include    "stdinc.h"

#pragma     warning (push)
#pragma     warning (disable : 4917)    // for Microsoft's OCIDL.H
#pragma     warning (disable : 4946)    // for Microsoft's SHLOBJ.H
#include    <shlobj.h>
#pragma     warning (pop)

#include    "idc.h"
#include    "linkcpl.h"
#include    "puterror.h"
#include    "shortcut.h"

/*  ************************************************************************  */
/*  Forward references  */

HRESULT
CreateCplItemPidl (
    DWORD,
    PCWSTR,
    INT,
    PCWSTR,
    PCWSTR,
    PITEMID_CHILD *);

HRESULT ILAppendChild (PIDLIST_ABSOLUTE *, PCITEMID_CHILD);

HRESULT ConvertString (PCWSTR, PSTR *);
HRESULT ConvertSimpleString (PCWSTR, PSTR *);

/*  ************************************************************************  */

HRESULT
CreateCplItemShortcut (
    DWORD Flags,
    PCWSTR Module,
    INT Icon,
    PCWSTR Name,
    PCWSTR Info,
    PCWSTR Link)
{
    /*  Build a PIDL from the desktop folder out to the given item as a
        child of the Control Panel folder.  */

    PIDLIST_ABSOLUTE pidl;
    HRESULT hr = SHGetSpecialFolderLocation (NULL, CSIDL_CONTROLS, &pidl);
    if (FAILED (hr)) {
        PutError (hr, "getting PIDL for Control Panel");
    }
    else {

        /*  Create - by hand, this being the program's raison d'etre - a
            child ID for the given Control Panel item. Then append it to the
            absolute PIDL that we're building. Beware that appending may,
            and even typically does, re-allocate.  */

        PITEMID_CHILD item;
        hr = CreateCplItemPidl (Flags, Module, Icon, Name, Info, &item);
        if (SUCCEEDED (hr)) {

            hr = ILAppendChild (&pidl, item);
            if (SUCCEEDED (hr)) {

                /*  Wrap the PIDL into a shortcut file.  */

                hr = CreateShortcut (pidl, Link);

                /*  It's all clean-up from here.  */
            }
            ILFree (item);
        }
        ILFree (pidl);
    }
    return hr;
}

HRESULT
CreateCplItemPidl (
    DWORD Flags,
    PCWSTR Module,
    INT Icon,
    PCWSTR Name,
    PCWSTR Info,
    PITEMID_CHILD *Pidl)
{
    HRESULT hr;

    /*  If the caller requires ANSI strings in the PIDL, create the Control
        Panel's level of PIDL as the basic IDCONTROL only. Assume that the
        caller is good in the sense of not also specifying Unicode strings
        or WOW support (which both require an IDCONTROLW).  */

    if (Flags & CPLITEM_FLAG_ANSI) {

        /*  Since the caller requires the ANSI form, require that all
            conversions from Unicode succeed.  */

        PSTR module;
        hr = ConvertString (Module, &module);
        if (FAILED (hr)) {
            PutError (hr, "converting \"%ws\" to ANSI", Module);
        }
        else {
            PSTR name;
            hr = ConvertString (Name, &name);
            if (FAILED (hr)) {
                PutError (hr, "converting \"%ws\" to ANSI", Name);
            }
            else {
                PSTR info;
                hr = ConvertString (Info, &info);
                if (FAILED (hr)) {
                    PutError (hr, "converting \"%ws\" to ANSI", Info);
                }
                else {

                    hr = CreateIDControl (module, Icon, name, info, Pidl);

                    delete info;
                }
                delete name;
            }
            delete module;
        }
        return hr;
    }

    /*  If the caller requires Unicode strings in the PIDL, we'll create the
        PIDL as an IDCONTROLW specifically. We'll also do this if the caller
        requires WOW support, since it too needs an IDCONTROLW.  */

    bool wow = Flags & CPLITEM_FLAG_WOW ? true : false;

    if (NOT (Flags & CPLITEM_FLAG_UNICODE) AND NOT wow) {

        /*  When we have no guidance from the caller, choose a basic
            IDCONTROL if all the strings are suitably simple.  */

        bool simple = false;

        PSTR module;
        hr = ConvertSimpleString (Module, &module);
        if (SUCCEEDED (hr)) {
            PSTR name;
            hr = ConvertSimpleString (Name, &name);
            if (SUCCEEDED (hr)) {
                PSTR info;
                hr = ConvertSimpleString (Info, &info);
                if (SUCCEEDED (hr)) {
                    simple = true;

                    hr = CreateIDControl (module, Icon, name, info, Pidl);

                    delete info;
                }
                delete name;
            }
            delete module;
        }

        if (simple) return hr;
    }

    /*  Otherwise, the IDCONTROLW is the most capable form and is our last
        resort.  */

    return CreateIDControlW (Module, Icon, Name, Info, wow, Pidl);
}

/*  ************************************************************************  */
/*  ID List Helper  */

HRESULT
ILAppendChild (
    PIDLIST_ABSOLUTE *Parent,
    PCITEMID_CHILD Child)
{
    PIDLIST_RELATIVE pidl = ILAppendID (*Parent, &Child -> mkid, TRUE);
    if (pidl == NULL) {
        PutMemoryError ();
        return E_OUTOFMEMORY;
    }

    *Parent = (PIDLIST_ABSOLUTE) pidl;
    return S_OK;
}

/*  ************************************************************************  */
/*  String Helpers  */

HRESULT ConvertString (PCWSTR In, PSTR *Out)
{
    if (In == NULL) {
        *Out = NULL;
    }
    else {

        /*  Convert from Unicode the basic Windows way, i.e., by the
            ancient Windows API function WideCharToMultiByte. Call it first
            with no buffer to discover how big a buffer is needed. Then
            repeat with that size of buffer.  */

        PSTR out = NULL;
        int cch = 0;
        for (;;) {

            cch = WideCharToMultiByte (
                    CP_ACP,
                    0,
                    In,
                    -1,
                    out,
                    cch,
                    NULL,
                    NULL);
            if (cch == 0) {
                DWORD ec = GetLastError ();

                if (out != NULL) delete out;

                HRESULT hr = HRESULT_FROM_WIN32 (ec);
                PutError (hr, "converting \"%ws\" to ANSI", In);
                return hr;
            }

            if (out != NULL) break;

            out = new CHAR [(SIZE_T) cch];
            if (out == NULL) {
                PutMemoryError ();
                return E_OUTOFMEMORY;
            }
        }
    }
    return S_OK;
}

HRESULT ConvertSimpleString (PCWSTR In, PSTR *Out)
{
    if (In == NULL) {
        *Out = NULL;
    }
    else {

        PCWSTR in = In;
        SIZE_T cch = 0;
        WCHAR ch;
        do {
            ch = *in ++;
            if (ch > L'\x7F') return E_FAIL;
            cch ++;
        } while (ch != L'\0');

        PSTR outbuf = new CHAR [cch];
        if (outbuf == NULL) return E_OUTOFMEMORY;

        PCHAR out = outbuf;

        in = In;
        do {
            *out ++ = (CHAR) *in ++;
        } while (-- cch != 0);

        *Out = outbuf;
    }
    return S_OK;
}

/*  ************************************************************************  */

