
/*  ************************************************************************  *
 *                                  main.cpp                                  *
 *  ************************************************************************  */

#include    "stdinc.h"

#include    <objbase.h>         // for CoInitialize, etc

#include    <stdlib.h>          // for wcstoul

#include    "linkcpl.h"
#include    "puterror.h"

/*  ************************************************************************  */

static CHAR const Description [] =
"Creates a shortcut file to a Control Panel item\n"
"\n"
"  LINKCPL [options] lnkfile\n"
"\n"
"Options:\n"
"\n"
"  /ansi             shortcut uses ANSI characters\n"
"  /icon:number      icon index\n"
"  /info:text        description of Control Panel item\n"
"  /module:text      pathname of CPL module that hosts Control Panel item\n"
"  /name:text        display name of Control Panel item\n"
"  /unicode          shortcut uses Unicode characters\n"
"  /wow              32-bit Control Panel item on 64-bit Windows";

/*  ************************************************************************  */
/*  Forward references  */

bool ParseParameterColonValue (PWSTR, PCWSTR, PWSTR *);
bool ParseParameterColonValue (PWSTR, PCWSTR, INT *);

/*  ************************************************************************  */

int __cdecl wmain (int argc, PWSTR *argv)
{
    DWORD flags = 0;
    PWSTR module = NULL;
    INT icon = -1;
    PWSTR name = NULL;
    PWSTR info = NULL;
    PWSTR lnk = NULL;

    /*  Command-line parsing  */

    if (argc == 0) return -1;

    while (++ argv, -- argc != 0) {
        PWSTR arg = *argv;
        PWSTR p = arg;
        if (*p == L'-' OR *p == L'/') {
            p ++;

            if (_wcsicmp (p, L"?") == 0 OR _wcsicmp (p, L"help") == 0) {
                PutInfo (Description);
                return 1;
            }

            if (ParseParameterColonValue (p, L"module", &module)) continue;
            if (ParseParameterColonValue (p, L"icon", &icon)) continue;
            if (ParseParameterColonValue (p, L"name", &name)) continue;
            if (ParseParameterColonValue (p, L"info", &info)) continue;

            if (_wcsicmp (p, L"a") == 0 OR _wcsicmp (p, L"ansi") == 0) {

                if (flags & CPLITEM_FLAG_UNICODE) {
                    PutError ("%ws conflicts with /%ws", arg, L"unicode");
                    return -1;
                }

                if (flags & CPLITEM_FLAG_WOW) {
                    PutError ("%ws conflicts with /%ws", arg, L"wow");
                    return -1;
                }

                flags |= CPLITEM_FLAG_ANSI;
                continue;
            }

            if (_wcsicmp (p, L"u") == 0 OR _wcsicmp (p, L"unicode") == 0) {

                if (flags & CPLITEM_FLAG_ANSI) {
                    PutError ("%ws conflicts with /%ws", arg, L"ansi");
                    return -1;
                }

                flags |= CPLITEM_FLAG_UNICODE;
                continue;
            }

            if (_wcsicmp (p, L"w") == 0 OR _wcsicmp (p, L"wow") == 0) {

                if (flags & CPLITEM_FLAG_ANSI) {
                    PutError ("%ws conflicts with /%ws", arg, L"ansi");
                    return -1;
                }

                flags |= CPLITEM_FLAG_WOW;
                continue;
            }

            PutError ("Invalid switch \"%ws\"", arg);
            return -1;
        }

        if (lnk == NULL) {
            lnk = arg;
            continue;
        }

        PutError ("Too many parameters \"%ws\"", arg);
        return -1;
    }

    if (lnk == NULL) {
        PutError ("Missing name of output file");
        return -1;
    }

    HRESULT hr = CoInitialize (NULL);
    if (FAILED (hr)) {
        PutError (hr, "initialising COM");
    }
    else {

        hr = CreateCplItemShortcut (flags, module, icon, name, info, lnk);

        CoUninitialize ();
    }
    return hr;
}

/*  ************************************************************************  */
/*  Helpers  */

bool ParseParameterColonValue (PWSTR Argument, PCWSTR Parameter, PWSTR *Value)
{
    PWSTR p = Argument;
    size_t cchparm = wcslen (Parameter);
    if (_wcsnicmp (p, Parameter, cchparm) == 0) {
        p += cchparm;
        if (*p == L':') {
            p ++;
            if (*p != L'\0') {
                *Value = p;
                return true;
            }
        }
    }
    return false;
}

bool ParseValueNumeric (PCWSTR Value, INT *Data)
{
    PWCHAR end;
    ULONG n = wcstoul (Value, &end, 0);
    if ((n != 0 OR end != Value)
            AND *end == L'\0'
            AND errno == 0
            AND n == (INT) n) {
        *Data = (INT) n;
        return true;
    }
    return false;
}

bool
ParseParameterColonValue (
    PWSTR Argument,
    PCWSTR Parameter,
    INT *Data)
{
    PWSTR value;
    if (ParseParameterColonValue (Argument, Parameter, &value)) {
        if (ParseValueNumeric (value, Data)) return true;
    }
    return false;
}

/*  ************************************************************************  */

