#!/usr/bin/env python
# generates a PDF containing its own MD5 hash using JPEG images
import sys
try:
    from io import BytesIO
except ImportError:
    from cStringIO import StringIO as BytesIO
from struct import pack, unpack
from hashlib import sha1, md5
import subprocess, random
from ctypes import *

libcoll = CDLL("./libcoll-jpeg.so")

def _new_iv():
    return create_string_buffer(pack("IIII",0x67452301,0xefcdab89,0x98badcfe,0x10325476), 16);

def prepad(prefix, n):
    # pads prefix to a 64-byte boundary with optimised MD5 state
    assert (len(prefix)+n) % 64 == 0
    assert n <= 64
    if n == 0:
        return ""

    prefix2 = prefix[len(prefix)&~63:]
    prefix = prefix[:len(prefix)&~63]
    assert(len(prefix) % 64 == 0)
    iv = _new_iv();
    for i in range(0, len(prefix), 64):
        libcoll.MD5Transform(iv,prefix[i:i+64]);

    # 256 tries should be enough almost always, and it's OK if we fail.
    for i in range(0,256):
        pad = b"\x00"*(n-1) + bytes(bytearray((i,)))
        assert len(pad) == n
        iv2 = create_string_buffer(iv.raw, 16)
        libcoll.MD5Transform(iv2,prefix2+pad)
        iv2 = unpack("IIII", iv2.raw)
        if (iv2[2]>>25)&1 != (iv2[2]>>24)&1 and (iv2[3]>>25)&1 == (iv2[3]>>24)&1:
            return pad
    print("DEBUG: prepad failed")
    return "\x00"*n

def makecoll(prefix):
    assert len(prefix) % 64 == 0
    iv = _new_iv();
    for i in range(0, len(prefix), 64):
        libcoll.MD5Transform(iv,prefix[i:i+64]);
    
    block0 = create_string_buffer(64);
    libcoll.MD5CollideBlock0(iv, block0, None);
    libcoll.MD5Transform(iv,block0);
    block1 = create_string_buffer(64);
    libcoll.MD5CollideBlock1(iv, block1, None);
    block0 = list(unpack("16I",block0.raw)); block1 = list(unpack("16I",block1.raw));
    a = pack("<16I",*block0)+pack("<16I",*block1)
    
    block0[4] = (block0[4]+(1<<31)) & 0xffffffff
    block0[11] = (block0[11]+(1<<15)) & 0xffffffff
    block0[14] = (block0[14]+(1<<31)) & 0xffffffff
    block1[4] = (block1[4]-(1<<31)) & 0xffffffff
    block1[11] = (block1[11]-(1<<15)) & 0xffffffff
    block1[14] = (block1[14]-(1<<31)) & 0xffffffff
    b = pack("<16I",*block0)+pack("<16I",*block1)

    assert md5(prefix+a).hexdigest() == md5(prefix+b).hexdigest()
    return (a,b)

_img_prefix = b'\xff\xd8\xff\xe0\x00\x10JFIF\x00\x01\x01\x00\x00\x01\x00\x01\x00\x00\xff\xdb\x00C\x00\x03\x02\x02\x02\x02\x02\x03\x02\x02\x02\x03\x03\x03\x03\x04\x06\x04\x04\x04\x04\x04\x08\x06\x06\x05\x06\t\x08\n\n\t\x08\t\t\n\x0c\x0f\x0c\n\x0b\x0e\x0b\t\t\r\x11\r\x0e\x0f\x10\x10\x11\x10\n\x0c\x12\x13\x12\x10\x13\x0f\x10\x10\x10'

_img_rest = [b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x14\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\xff\xc4\x00\x1d\x10\x00\x01\x05\x01\x01\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x01\x02\x04\x05\x06\x07\x08\x12\x13\xff\xda\x00\x08\x01\x01\x00\x00?\x00I\xe6\xbe\xd9\xea\x1a\x02r\xce\xcbc\xb2\x89:\xb7\xab\xee"f%\xf3D\xa9\x18\x81\x9c\xaa\xb3\x9b>\x05d\xd0Y\xa0\xd0\xd2$\xb0\xd4s\x9eTz\xb9\x85C9\xa8(\xbf"{\x9a\xb2>\x1c\xaa\xccZ\xe4\xb3\x06\xe8\xb2\xe7rNs\xa8>\xcf\x1b\x8aZ\xd6\x8c\xf5\xd6\xaew\xe8\x04=\x9a\x11M*4c\x16aD5c^\xab!\xad9N\xd1\xb5\xab', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x08\xff\xc4\x00(\x10\x00\x00\x04\x04\x05\x03\x05\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x03\x06\x04\x05\x07\x13\x12\x14\x15\x18B\x00\t\x16\x17#456\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xac\xdauZ\xbc\xab\xdc\r\xc1E\xdf3\x06\xfc+\x10)\xba\x8ev\xfc\xa6S\xef\x9c@&\xa9B\xa7\x15\x17\x10\xaa$T"L\x00\xbe$S\x11D\x842e\x01P\xe52\xa6A\xb7\xaa\xa7\xba}\xcb\xfa\xb6\xd4\xfc\xff\x00\x87h\x9e\x19\x13\xf4z\x8ew\x0ecS\xf9\x9c/\xdb\xb7\xca\xc7\x1e\xbf', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x08\xff\xc4\x00\x1f\x10\x00\x01\x04\x02\x03\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x01\x03\x04\x06\x05\x07\x08\x11\x13\x14\x12\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xa7\x8e\xc7\xbf\xb1\xfb\xe6\xab\xabi\xdb\xe2^\xc4\xb3A\xcc&ge\xc1:\xe6:\x05^\xbd[\x91\xee\xadGEi\xa3\x9c\xc4\xe3\x02g\xe4d\xa6>\xe1\xf8\x93\xcf"2]\xaa\xba_\x8b\x9b\xe7KVk\xf4\xbc\x17$\xeb\xe7\x8d\x87h+-\x97 \x9a\xfb\xbc\xdd\xad^\x90N\xcbjt\xd7\xf2\x0f\n\x9b\xc2H\xdf\xd0\x8d{\x08\xb4\xcfF\xbf\x95B', b"\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x08\xff\xc4\x00 \x10\x00\x01\x05\x01\x00\x01\x05\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x01\x02\x04\x06\x07\x05\x13\x00\x08\x14\x15!\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xa2t-'m\xc8u\x0c\xca\xb2m\xbe\xc1k\xee\xd9\xb4\x88\\;\x0c\x0e\xa5\x18\x1c*\x8f\xd4M\x1c\x938|\xf9\xc4\x8c\x8a\xf9!\x17\x85\xa1\x12u$\x18\x85\x1a\xb5XW+\x84\x88\x16/l\xba\x86\x87\xd5\xa1\xc6\xd5\xf7\xa8\x96Z\xf6{x\x05\xe3\x9e\xa2\xa8\x8e\x07na\xe2\xbaC\xa1\x02T\xb1I\xf8\x8ec<\xeci\x14PD\xa4h\xbf<nr\xbb\xd7", b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x08\xff\xc4\x00!\x10\x00\x01\x04\x02\x01\x05\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x01\x02\x04\x05\x06\x07\x08\t\x11\x12\x13\x15#\xff\xda\x00\x08\x01\x01\x00\x00?\x00T\xa0\xea\x07c\xb5\xf9{\xa9\xe80<\xe7\x14\xa8\xd3\xb95\x85\xf5*\xc64\xd8\xaf\xb8\xb9\x91\x1e"\xa4iR\xc2\xf4R\xd7\x04\xd2\xdc!\xc3\n\xa8\xceua\x1c\xf6\xf6(\xc6\xdaSih<\xab`\xf2\x0bTn\xea\xcd\x85US\x13V\xfd\x1fU9\xf1\xf2J$\xff\x00\xa0$\x04\xcf)-\x964\x1f\xe0\xd6 \xbb\t\xde\x04G=\xde\xd6\xaa\r?', b"\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x14\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\xff\xc4\x00!\x10\x00\x01\x04\x02\x02\x02\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x04\x05\x07\x01\x06\x08\x11\x00\x13\x12\x14!\xff\xda\x00\x08\x01\x01\x00\x00?\x00t\xe3-\xbdbI\xeb5\xda\xbc\xa0\xbe\xae\r:\xc1\xd8f\x0e=x\t\xba\xf9\xa44#\xd7xp\xb96`/\x16\x84\x10\xf9\xb8l\x88\xf4\x00\xef\n\x91\x11\ny\x13\xe8p\xab\xb1q\x96\xd0\xb0\xe5t6\xd6\xbd\xf4\xd3e\xd7\xab\xdd\xe1\r\xe2?)j)\xb0\x9bx\xbbRpL\x90t\xed'?P\x80=\xe0*e&)eAK\xf3\xd6E\x92\xf3", b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x08\xff\xc4\x00\x1f\x10\x00\x02\x02\x03\x00\x02\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x04\x01\x03\x05\x06\x07\x08\x13\x11\x12\x14\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xa2|O\xf2\xf1n\xac\xb6\x1f\xa6\xf5\x9e\x8f\xb0`\x0f\xa6f\x1b\xc3\xea\xda\x95\xda\xa5\xaa\xebh\x18\xdau\xaa\xadY\xbb\x12\x81y\xeb\x05;\x8ag\xf5\x88\x1d\x97\x9d@\xb8\x98\x08\xc2\xba\x87\x82\x99]kY\xe6\x1c\xa9\xce\xbc\xa3\xbc\xeb\x92\xef\x11\xbc`R\xabZ*sw\xdfS\r\xb0\xb5\r?-\x9d\x16\x00\xd8\xdc\xfb&\xb4\xeb\x93\x10\xf8\x1fT\xcf\xda?', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x08\xff\xc4\x00(\x10\x00\x00\x04\x05\x03\x02\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x04\x07\x01\x05\x06\x11\x13\x12\x14\x15\x08\x16\x00\x17"%123\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xa2k\x05]H6\xd5\xfb:\xd5\xc9\xfa\xa4\xa8\x1c\xa7\x06{T$\x98U\xf2\x80\xd3\x124H\x8b\xa3J8\xcd\xf2\xf1\x94ZH\x9a\x88\x10\x86\x02\x021\xaa\xb9\xa6D\xc0\x93\x08\x8e\xc0\x03\xabv\xc5\xf6\x13\xec\xee=}\xd3\xbe\xf3S\x80\xf6\xcd\x8e.7\x8cF4\xdf\xb6Af\xc9\xafW\xd0\x1amoW\xcf\x8f', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x07\xff\xc4\x00\x1f\x10\x00\x01\x05\x01\x00\x02\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x03\x01\x02\x04\x05\x06\x07\x08\x13\x00\x11\x12\xff\xda\x00\x08\x01\x01\x00\x00?\x00}\xcc\xfc\xaa\xee\xd7\x95\xdccYe\xd2\xedf\xe97]@\x99\x9d5\x04\x9c\xfc0b\x85V\xe9V\x02{*\xee\x92#\x07(\xc3\x1cp\xa8\x10vr\x1eR5Z\x8d:\xaa\xb3\xe5\x83!\xe0\xa5\xaek3\xcc9T\xce\xbd\x12o:\xe4\xbb\x84\xdcP\xc2\x16i\xc1\xbb9\xc5"\\\x88\xc0\x95=e\xbc\x04cI-}\x8a8cW\xb5\x9fM\xf5*\xfe\x93', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x08\xff\xc4\x00 \x10\x00\x01\x05\x01\x00\x03\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x02\x01\x03\x04\x05\x06\x07\x00\x08\x12\x11\x13\x14\xff\xda\x00\x08\x01\x01\x00\x00?\x00I\xaf\xf6C\xd9|c\x9c~wQ\xd0\xedk\xf7{\x0e\xb0\xc6C]\x8d\xb4\xc3\x84l\xbc*\xe9sf0#_d0A_0hb\xb8\xc9\x05\x8c\x8f\xbf\xc2\x91~\xd0CO)\xfd\x87\x03\xd5u^\x8f\x96\xd1\xf5\x9e\x85Ui\x90\xc2h\x0bS\x9f\xcdS\xe7\xdc\xaeqm\x1aUH\x0f\xce\x9ar\xde9\x1f\xcc\x06\xea\xa02\x11\xc1\xc7\x0f\xe9\xc1P\x14k\xcf', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x14\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\xff\xc4\x00$\x10\x00\x00\x04\x06\x02\x02\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x04\x06\x01\x07\x11\x12\x13\x14\x05\x15\x08\x17\t\x16!\xff\xda\x00\x08\x01\x01\x00\x00?\x00^\x96\xdf"\n]\xef\xf6s\xcdK\x8d\x94\xa9\x894\xdd\x05\xb28f".@\xa1\xbb\x9b\x83\x89\xc6\x14\x8f\x97\\\\\x05A\x01I\xa12\x07\x93\xf8\x02\n\x8a\x11\x143\x86#`b[g\xc1\x8f\xaeH\x99%%=\xa3\xb1\xe9\xc9\x80\x91\xf5\xd9\xf4\x96v\xb8\x16+S\xab\x87b8+\xb9nK\xcc\xa6:\xd9\x1b\xa9\x0f', b"\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x07\xff\xc4\x00\x1c\x10\x00\x02\x02\x03\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x04\x01\x03\x05\x06\x07\x08\x15\xff\xda\x00\x08\x01\x01\x00\x00?\x00i\xa1\xfaw\xbf+\xd4(\xd6\xdc\xec;\x06\xc4\xc2\x9e\x87o\x98Z\x8e{\\\xc7'\x84c[\xae\xbb \xac,\x8d(\xafW\xd4\x89\x19*\xe8\xad\xa1\xb2\xc2\x10\x81Z\xc88\x12\xaf\xea\xde\x0f\xda\x94\xce\xaa\xc6\xeb\xda\xb1Y\\(v\x03\xedl\xa7\x8b\xd4l\xc7\xb5nr@\xa0\x17\x06,}\x80\x04\xe0\xe4fBj+$bF-\x19\x98(", b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x08\xff\xc4\x00\x1f\x10\x00\x01\x04\x03\x00\x03\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04\x01\x02\x03\x06\x05\x07\x08\x12\x13\x14\x15\xff\xda\x00\x08\x01\x01\x00\x00?\x00\xa7\xb9\xba\xe9\xd1}\x1f\xae\xea\xdd7\x86\xda\xa2a\x81\xb1\xda%\x9d\xf4\x02q\x02\xc9\x88\x1a\xb9\tr\x030\xc8[b\xfb\x9er2\x19\na>\xd6\xc4\xe9\xbcbX\x19\x12\xaa\xa2\xba\x07+\xdc\xb5`\xd8\x9du\xaf\xb7\x91x=I\x83\xb4%\x88\x1a\xe0\xd8T\xfdx\x86\xf6\xa9K\x88\\\xb2\xce\xbe`\xb8\xc5t\x8fG\x0e\xa4>\x17:\x07N\xe6\xb9U\x7f', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x14\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\xff\xc4\x00 \x10\x00\x02\x01\x04\x02\x03\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x04\x01\x05\x06\x07\x00\x08\x11\x12\x15\x13\xff\xda\x00\x08\x01\x01\x00\x00?\x00I\xd7}\x97\xdf\xd8\x8ee\xf6s\r\xa1/9\xc7\x8f\xb0\xf3\xf4Q\xd9n\x96\xabtC\x18\x87E\xd6\x1d\xd1ra\xc6IRJHY\xfa\x01\t\xa9\xc0\xcf\x02)!\xa1\xf1W]\xf4R\xebe\xcc\xbe\xee\xce\xdb\xd12\x9b\x10l\x89\xfbp,V\xbch\xad YC\xe8\xba%\xccyK\x90\xda\xc6\x8dAa-\x03P\xf66yi\xb4F\x81\xcf', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x14\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\xff\xc4\x00\x1f\x10\x00\x01\x05\x01\x01\x00\x03\x01\x00\x00\x00\x00\x00\x00\x00\x00\x02\x01\x03\x04\x05\x07\x06\x08\x11\x12\x13\x17\xff\xda\x00\x08\x01\x01\x00\x00?\x00T\xc1\xf5\x7fC\xea;\x07O\x99\xc6\xf4gV\xe5\xd7\x11\xa8\\\xc68\xb6\x9c\x9d@\xd3\xda\xf2\x14\xf6\x15\xd1\xe5\xc79l\xc0l\xc6\xc5\xc0\xb0\xfb"\xb2J\x88\x80\x88`\xc2<\xd3\xaa\xab\x9e\xf9+q\xcf\xe6Z7Y\xe9\x8a\xa8\x95]6\xa1\xfd7\xa4b\xaf\x89z$\xab\x07\x1c}\x97dV\x04\x82\xb2s\xf1\x86\xef\xe0\x02\xa8\x80N(\xfc\x89\x19\x81\x1be', b'\xff\xc0\x00\x0b\x08\x00\x10\x00\x08\x01\x01\x11\x00\xff\xc4\x00\x15\x00\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x08\xff\xc4\x00#\x10\x00\x00\x06\x02\x01\x04\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x03\x04\x05\x06\x07\x12\x08\x11\x13\x14\x16\x17!#\xff\xda\x00\x08\x01\x01\x00\x00?\x00T\xc0\xf9_\x90\xf9G0Y\xf1\x9bnFZ\xd4\x9a\xa4e\t\x96\xc7k)S\x88,<\xadB\x1eB9\xbb\xb6\xe7v\x8b\x04\xceY\x15\t!\xb0\n&\x10\x00 \x01\xc8\x80,\x92\xa3Jq\x9f\x01\xda\xb07\xc8~\xc7\x90\xa2\xad\x1e\xff\x00p{uS\xc2\xaf\xa9\x19\xe1\xbey\xd3\xc9L7v\xe3t\x7f4\xbbe\xfa1:\x1fc)\xb0i']

#_img_rest = _img_rest[:2] # HACK to speed up testing

MD5LEN = 32

def makepdf():
    xrefs = []; imgns = []; all_subs = []
    f = BytesIO()
    width, height = 8, 16
    f.write(b"%PDF-1.3\n%\xe2\xe3\xcf\xd3\n\n\n")
    TARGET_SIZE=9000
    want_imgcnt = MD5LEN
    coll_progress = 0
    for i in range(0, want_imgcnt):
        xrefs.append(f.tell())
        f.write(b"%i 0 obj\n" % len(xrefs)); imgns.append(len(xrefs))
        f.write(b"<</Width %i /Height %i /Type /XObject /Subtype /Image /Filter [ /DCTDecode ] " % (width, height))
        f.write(b"/ColorSpace /DeviceGray /Length %i /BitsPerComponent 8>>\n" % (TARGET_SIZE))
        f.write(b"stream\n")
        
        imgstart = f.tell()
        f.write(_img_prefix);
        cur_subs = []
        for rest in _img_rest[:-1]:
            f.write(b"\xff\xfe");
            pad = 120 - (f.tell() % 64)
            if pad < 58: pad += 64
            f.write(pack(">H", pad))
            # mangle this padding to make the MD5 collision faster
            f.write(prepad(f.getvalue(),pad-58))
            assert f.tell() % 64 == 0
            
            collstart = f.tell()
            (colla, collb) = makecoll(f.getvalue())
            assert colla[56:59] == b"\xff\xfe\x00"
            assert collb[56:59] == b"\xff\xfe\x00"
            assert ord(colla[59]) ^ 0x80 == ord(collb[59])
            if ord(colla[59]) > ord(collb[59]):
                # make sure colla is the side with shorter comment
                collb, colla = colla, collb
            assert ord(colla[59]) >= 70
            f.write(colla)
            collend = f.tell()
            coll_progress += 1
            print("Done %i of %i collisions" % (coll_progress, want_imgcnt*(len(_img_rest)-1)))
            
            f.write(b"\x00" * (ord(colla[59]) - 6 - 64))
            f.write(b'\xff\xfe' + pack(">H", (ord(collb[59]) - ord(colla[59]) - 2) + len(rest) + 4))
            f.write(b"\x00" * (ord(collb[59]) - ord(colla[59]) - 4))
            f.write(rest)
            f.write(b'\xff\xfe')
            f.write(pack(">H", TARGET_SIZE-(f.tell()-imgstart)-2))
            cur_subs.append( (collstart, collb) )
        f.write(_img_rest[-1])
        f.write(b'\xff\xfe')
        f.write(pack(">H", TARGET_SIZE-(f.tell()-imgstart)-2))
        f.write(b"\x00"*(TARGET_SIZE-(f.tell()-imgstart)-2))
        f.write(b'\xff\xd9')
        cur_subs.append( (collstart, "") ) # dummy value
        all_subs.append(cur_subs)

        imglen = f.tell() - imgstart
        assert(imglen == TARGET_SIZE)
        f.write(b"\nendstream\nendobj\n\n")

    xrefs.append(f.tell()); root_n = len(xrefs)
    f.write(b'%i 0 obj\n<<\n  /Type /Catalog\n  /Pages %i 0 R\n>>\nendobj\n\n\n' % (len(xrefs), len(xrefs)+1))
    xrefs.append(f.tell()); pages_n = len(xrefs)
    f.write(b'%i 0 obj\n<<\n  /Type /Pages\n  /Count 1\n  /Kids [%i 0 R]\n>>\nendobj\n\n' % (len(xrefs), len(xrefs)+1))
    xrefs.append(f.tell())
    f.write(b'%i 0 obj\n<<\n  /Type /Page\n  /Parent %i 0 R\n' % (len(xrefs), pages_n))
    f.write(b"  /MediaBox [0 0 %i %i]\n" % (width*MD5LEN, height))
    f.write(b"  /CropBox [0 0 %i %i]\n" % (width*MD5LEN, height))
    f.write(b'  /Contents %i 0 R\n  /Resources\n  <<\n' % (len(xrefs)+1))
    f.write(b"    /XObject <<")
    for i in range(0, len(imgns)):
        f.write(b"/Im%i %i 0 R" % (i, imgns[i]))
    f.write(b">>\n");
    f.write(b"  >>\n>>\nendobj\n\n")
    xrefs.append(f.tell())
    pdftext = b""
    for i in range(0, len(imgns)):
        pdftext += b"q\n  %i 0 0 %i %i 0 cm\n  /Im%i Do\nQ\n" % (width, height, width*i, i)
    f.write(b'%i 0 obj\n<</Length %i>>\nstream\n' % (len(xrefs), len(pdftext)))
    f.write(pdftext)
    f.write(b'\nendstream\nendobj\n\n\n\n')
    startxref = f.tell()
    f.write(b'xref\n0 %i \n0000000000 65535 f \n' % (len(xrefs)+1))
    for n in xrefs:
        f.write(b'%010i 00000 n \n' % n);
    f.write(b'\ntrailer << /Root %i 0 R /Size %i>>\n\nstartxref\n%i\n' % (root_n, len(xrefs)+1, startxref))
    f.write(b'%%EOF\n')
    #print(xrefs);
    cs = md5(f.getvalue()).hexdigest()
    fixed = f.getvalue()
    for i in range(0, len(all_subs)):
        pos, rep = all_subs[i][ int(cs[i], 16) ]
        fixed = fixed[:pos] + rep + fixed[pos+len(rep):]
    assert cs == md5(fixed).hexdigest()
    return fixed

if __name__ == '__main__':
    # testa.pdf -> all-zeroes base PDF, testb.pdf -> displays own MD5
    t = makepdf()
    print(md5(t).hexdigest())
    fname = "demo-selfmd5-img.pdf"
    f = open(fname,"wb"); f.write(t); f.close()
    print("Created %s" % fname)
