/*
 * $Id: discover_main.c,v 1.2 2002/12/22 19:59:56 skyper Exp $
 */

#include <sys/time.h>
#include <time.h>
#include <pcap.h>
#include <libnet.h>
#include "default.h"
#include "thcrut.h"
#include "state.h"
#include "range.h"
#include "thcrut_pcap.h"
#include "thcrut_sig.h"
#include "network_raw.h"
#include "nmap_compat.h"
#include "packets.h"
#include "discover_dispatch.h"
#include "network.h"

extern struct _opt opt;

struct sockaddr_in ip_tcp_sync_addr;
int rawsox;

#define DFL_HOSTS_PARALLEL	(5000)

#if 0
#define TV_add(tv, sec, usec) do{ \
	(tv)->tv_usec += usec; \
	if ((tv)->tv_usec > 1000000) \
	{ \
		(tv)->tv_usec -= 1000000; \
		(tv)->tv_sec++; \
	} \
	if (sec > 0) \
		(tv)->tv_sec += sec; \
}while(0)

/* Calculate the difference from small to large */
#define TV_diff(dst, small, large) do{ \
	if ((small)->tv_sec > (large)->tv_sec) \
	{ \
		(dst)->tv_usec = -1; \
		break; \
	} \
	(dst)->tv_sec = (large)->tv_sec - (small)->tv_sec; \
	if (((dst)->tv_sec == 0) && ((small)->tv_usec > (large)->tv_usec)) \
	{ \
		(dst)->tv_usec = -1; \
		break; \
	} \
	(dst)->tv_usec = (large)->tv_usec - (small)->tv_usec; \
	if ((dst)->tv_usec < 0) \
	{ \
		(dst)->tv_sec--; \
		(dst)->tv_usec = 1000000 + (dst)->tv_usec; \
	} \
}while(0)
#endif
	
void
cb_filter(void)
{
	if (pcap_dispatch(opt.ip_socket, 1024, (pcap_handler) scanner_filter, NULL) < 0)
	{
		pcap_perror(opt.ip_socket, "pcap_dispatch");
		exit(-1);
	}
}

void
cb_timeout(struct _state *state)
{
	DEBUGF(" CALLBACK\n");
	STATE_current(state) = 1;
	dis_timeout(state);
	return;
}

static void
launch(struct _ipranges *ipr)
{
	char buf[opt.sq.item_size];
	struct _state *state = (struct _state *)buf;
	int ret;

	memset(buf, 0, opt.sq.item_size);

	while (1)
	{
		IP_next(ipr);
		if (IP_current(ipr) && ((IP_current(ipr) == opt.net) || (IP_current(ipr) == opt.bcast)))
				continue;

		if (IP_current(ipr))
		{
			STATE_ip(state) = IP_current(ipr);
			ret = STATE_wait(&opt.sq, state);
		} else
			ret = STATE_wait(&opt.sq, NULL);

		if (ret != 0)
			break;
	}
}


/*
 * init_defaults is called before getopt()
 */
static void
init_defaults(void)
{
	opt.flags |= FL_OPT_HOSTDISCOVERY;
	opt.hosts_parallel = DFL_HOSTS_PARALLEL;
}

/*
 * Return 1 if NMAP file could not be loaded.
 * Return 2 if thcrut-os filge could not be loaded
 * Return 3 if both failed.
 * Return 0 on success.
 */
#define MAX_DIR_LEN       (1024)
static int
config_fp_load(char *buf, char *dir)
{
	char err = 0;

	snprintf(buf, MAX_DIR_LEN, "%s/nmap-os-fingerprints", dir);
	if (NMAP_load_fp(&opt.osfp, buf) != 0)
		return 1;
	snprintf(buf, MAX_DIR_LEN, "%s/thcrut-os-fingerprints", dir);
	if (FP_TS_load(&opt.fpts, buf) != 0)
		return 2;

	return err;
}

/*
 * Init vars is called after getopt.
 */
static void
init_vars(void)
{
	int i = 1* 1024 * 1024; /* Is reduced to max. anyway */
	size_t size;
	char buf[MAX_DIR_LEN];
	char *ptr;
	struct stat sbuf;

	/* This goes after getopt */
	if (opt.flags & FL_OPT_HOSTDISCOVERY)
	{
		if ((ptr = getenv("THCRUTDIR")))
		{
			if (config_fp_load(buf, ptr) != 0)
			{
				fprintf(stderr, "Failed to load \"%s\": %s\n", buf, strerror(errno));
				exit(-1);
			}
		} else if (config_fp_load(buf, THCRUT_DATADIR) != 0) {
			if (config_fp_load(buf, ".") != 0)
			{
				fprintf(stderr, "Failed to load \"%s\": %s\n", buf, strerror(errno));
				exit(-1);
			}
		} else if (stat("./thcrut-os-fingerprints", &sbuf) == 0) {
			fprintf(stderr, "WARNING: ./thcrut-os-fingerprints exist. Using config files from "THCRUT_DATADIR" for security reasons.\nset THCRUTDIR=. to overwrite.\n");
		}
	}
	//FP_TS_dump(&opt.fpts);

	//rawsox = libnet_open_raw_sock(IPPROTO_RAW);
	rawsox = net_sock_raw();
	if (rawsox < 0)
	{
		fprintf(stderr, "socket: %s\n", strerror(errno));
		exit(-1);
	}
#if 0
	rawsox = socket(PF_INET, SOCK_RAW, IPPROTO_RAW);
	setsockopt(rawsox, SOL_SOCKET, SO_SNDBUF, &i, sizeof i);

	if (rawsox < 0)
	{
		fprintf(stderr, "socket: %s\n", strerror(errno));
		exit(-1);
	}
	i = 1;
	setsockopt(rawsox, IPPROTO_IP, IP_HDRINCL, (char *)&i, sizeof i);
#endif

	/* init pcap before fork */
	snprintf(buf, sizeof buf, "dst host %s and (icmp[4:2] = %d or ((udp or tcp) and (dst port %d or dst port %d or dst port %d)))", int_ntoa(opt.src_ip), htons(getpid()), opt.src_port, opt.src_port + 1, opt.src_port + 2);
	//, int_ntoa(opt.src_ip));
	opt.ip_socket = init_pcap(opt.device, 0, buf, &opt.net, &opt.bcast, &opt.dlt_len);


	if (opt.flags & FL_OPT_FP)
	{
		size = sizeof(struct _state_fp);
		/*
		 * Reserve 2 bits for TCP or UDP state (Open, closed, Unknown)
		 * Reserve some bytes if we perform banner matching (this sucks memory).
		 */
		opt.fpts.cat[FP_CAT_NVT].size = opt.fpts.cat[FP_CAT_NVT].n_tests * FP_NTEST_SZ;
		opt.fpts.cat[FP_CAT_SNMP].size = opt.fpts.cat[FP_CAT_SNMP].n_tests * FP_STEST_SZ;
		opt.fpts.cat[FP_CAT_WWW].size = opt.fpts.cat[FP_CAT_WWW].n_tests * FP_WTEST_SZ; /* 64 bytes for every WWW banner */
		opt.fpts.cat[FP_CAT_BANNER].size = opt.fpts.cat[FP_CAT_BANNER].n_tests * FP_BTEST_SZ;
		opt.fpts.cat[FP_CAT_TCP].size = opt.fpts.cat[FP_CAT_TCP].n_tests?opt.fpts.cat[FP_CAT_TCP].n_tests / 4 + 1:0;
		opt.fpts.cat[FP_CAT_UDP].size = opt.fpts.cat[FP_CAT_UDP].n_tests?opt.fpts.cat[FP_CAT_UDP].n_tests  / 4 + 1:0;

		for (i = 0; i < sizeof opt.fpts.cat / sizeof *opt.fpts.cat; i++)
			size += opt.fpts.cat[i].size;

		opt.fpts.ofs_test_tcp = 0;
		opt.fpts.ofs_test_udp = opt.fpts.ofs_test_tcp + opt.fpts.cat[FP_CAT_TCP].size;
		opt.fpts.ofs_test_banner = opt.fpts.ofs_test_udp + opt.fpts.cat[FP_CAT_UDP].size;
		opt.fpts.ofs_test_www = opt.fpts.ofs_test_banner + opt.fpts.cat[FP_CAT_BANNER].size;
		opt.fpts.ofs_test_snmp = opt.fpts.ofs_test_www + opt.fpts.cat[FP_CAT_WWW].size;
		opt.fpts.ofs_test_nvt = opt.fpts.ofs_test_snmp + opt.fpts.cat[FP_CAT_SNMP].size;
	} else {
		size = sizeof(struct _state);
	}

	scanner_gen_packets();
	if (opt.src_ip == 0)
		opt.src_ip = getmyip();

	//DEBUGF("size %d %d\n", sizeof(struct _state_fp), size);
	if (!SQ_init(&opt.sq, opt.hosts_parallel, size, pcap_fileno(opt.ip_socket), dis_timeout, cb_filter))
	{
		fprintf(stderr, "Failed to init states: %s\n", strerror(errno));
		exit(-1); /* Out of Memory */
	}
}

static void
usage(char *str)
{
	if (str)
		fprintf(stderr, "%s\n", str);

	printf(""
"usage: discover [options] [IP range] ...\n"
"             with IP range of the form a.b.c.d-x.y.z.w\n"
" -d          Don't do host discovery (tcp-sync ping, ...)\n"
" -O          With OS Fingerprinting\n"
" -v          verbose output (fingerprint stamps)\n"
" -l <n>      Hosts in parallel (default: %d)\n"
"", DFL_HOSTS_PARALLEL);
	if (str)
		exit(-1);
	exit(0);
}

/*
 * Set source IP, rate limit, ...
 */
static void
do_getopt(int argc, char *argv[])
{
	int c;

	/*
	 * We call getopt() for a second time.
	 * Set optind to 0 to reinit getopt() variables.
	 * We must thus start at index 0 and must
	 * remove the programs name.
	 *
	 * Linux inits (?) but always starts at 1.
	 */
	optind = 1;
	if (argc == 1)
		usage("Arguement required");

	while ((c = getopt(argc, argv, "+Odhvl:")) != -1)
	{
		switch (c)
		{
		case 'l':
			opt.hosts_parallel = atoi(optarg);
			if (opt.hosts_parallel <= 0)
				opt.hosts_parallel = DFL_HOSTS_PARALLEL;
			break;
		case 'O':
			opt.flags |= FL_OPT_FP;
			break;
		case 'd':
			opt.flags &= ~FL_OPT_HOSTDISCOVERY;
			break;
		case 'v':
			opt.flags |= FL_OPT_VERBOSE;
			break;
		case 'h':
			usage(NULL);
			break;
		default:
			usage("Wrong option");
		}
	}

	if ((opt.flags & FL_OPT_FP) && (opt.hosts_parallel > 1000))
	{
		if (opt.hosts_parallel != DFL_HOSTS_PARALLEL)
			fprintf(stderr, "Operating System Fingerprinting limited to 1000 hosts in parallel (fixed).\n");
		opt.hosts_parallel = 1000;
	}

	opt.argvlist = &argv[optind];
	opt.argc = argc - optind;
}

int
scanner_main(int argc, char *argv[])
{
	struct _ipranges ipr;
	struct pcap_stat ps;

	init_defaults();
	do_getopt(argc, argv);
	init_vars();

	signal(SIGPIPE, SIG_IGN);  /* Have to ignore this */
	IP_init(&ipr, opt.argvlist);
	launch(&ipr);

	/* This information is unreliable. Drops much more! */
	if (pcap_stats(opt.ip_socket, &ps) == 0)
		fprintf(stderr, "%u packets received by filter, %u packets dropped by kernel\n", ps.ps_recv, ps.ps_drop);

	exit(0);
	return 0;
}

