#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/timeb.h>
#include <time.h>
#include <sys/time.h>

#define PROGRAM		"Wardrive"
#define VERSION		"v2.3"
#define AUTHOR		"van Hauser / THC <vh@reptile.rug.ac.be>"
#define RESSOURCE	"http://www.thehackerschoice.com"

#define DEFAULT_SERPORT		"/dev/ttyS1"
#define DEFAULT_INTERFACE	"eth0"
#define DEFAULT_FILE		"./wardrive.stat"
#define DEFAULT_RESET_SCRIPT	"reset_wvlan.sh"
#define DEFAULT_SAVE_INTERVAL	1000               // in milliseconds
#define DEFAULT_LINK_SAVE	1
#define DEFAULT_LINK_BEEP	5
#define DEFAULT_BEEP_INTERVAL	5

#define V_TIME		45
#define MAX_MESSAGES	100
#define NMEA_POS_STRING "$??GLL"

#define PROC_WAVELAN	"/proc/net/wireless"
//#define PROC_WAVELAN  "./wireless.test"

#define OUTPUT_FILE_HEADER "# WARDRIVE by van Hauser / THC\n# DATE     TIME     LATITUDE       LONGITUDE  STATUS LINK LEVEL NOISE NWID CRYPT MISC\n"

#define MAXLEN 256
#define MAXGPSLEN 1024

typedef struct {
   int    Hours, Minutes;
   double Seconds;
   char   Dir;
} COORD;

typedef struct {
   COORD Latitude, Longitude;
} POS;

char *device, *serport, *file, *init, *reset;
int link_save, link_beep, beep_interval, verbose, ignore_errors, pid, winfo;
unsigned long save_interval;

char *prg, lockfile[MAXLEN];
FILE *g;
int fd_serport, debug;
struct termios serport_opts, serport_opts_orig;

void help() {
    printf("%s %s by %s

Syntax: %s [-p serport] [-d interface] [-o file] [-I script]
        [-i interval] [-l level] [-b level] [-B interval] [-G] [-v]

Options:
  -d interface  wavelan interface. [%s]
  -p serport    seriell port the GPS device (NMEA) is connected to. [%s]
  -o file       output file to append the data to. [%s]
  -I script     script to run initially to configure the wvlan card []
  -R script     script to reset wvlan card after node was found [%s]
  -W            print access point hwaddr and SSID via \"iwconfig\" [%s]
  -i interval   interval to write GPS+wavelan data in seconds, 0 = amap. [%d]
  -l level      only save data with >= this link level, 0 = all. [%d]
  -b level      beep if >= this link level, 0 = disable. [%d]
  -B interval   wait time in seconds before beeping again. [%d]
  -G            ignore errors from GPS, dont exit. [%s]
  -v            be verbose. [%s]

%s is a tool to save GPS and Wavelan link level data to a file.
Start this tool, drive around in the city, and afterwards you can analyze the
data for interesting locations.
You can always find the newest version at %s
Use is only permitted for legal purposes. The GPL v2 applies to this code.
", PROGRAM, VERSION, AUTHOR, prg, DEFAULT_INTERFACE, DEFAULT_SERPORT,
DEFAULT_FILE, DEFAULT_RESET_SCRIPT, "false", DEFAULT_SAVE_INTERVAL / 1000,
DEFAULT_LINK_SAVE, DEFAULT_LINK_BEEP, DEFAULT_BEEP_INTERVAL, "false", "false",
PROGRAM, RESSOURCE);
    exit(-1);
}

void seriell_clean_up (int ignore) {
    tcsetattr(fd_serport, TCSANOW, &serport_opts_orig);
    close(fd_serport);
    (void) unlink(lockfile);
    if (! ignore)
        exit(-1);
    fd_serport = -1;
    return;
}

void clean_up (int sig) {
    if (pid != 0) {
        if (pid > 0)
            kill(pid, SIGTERM);
        if (g != NULL)
            fclose(g);
        seriell_clean_up(1);
    }
    if (sig == -1)
        exit(-1);
    exit(0);
}

char *concat(char *old, char *sep, char *add) {
    char *ptr;
    ptr = malloc(strlen(old) + strlen(sep) + strlen(add) + 1);
    strcpy(ptr, old);
    strcat(ptr, sep);
    strcat(ptr, add);
    return ptr;
}

int str2int(char *str, int no) {
    char buf[no + 1];
    buf[no] = 0;
    memcpy(buf, str, no);
    return atoi(buf);
}

int get_parameter(int ppos, char *value, int maxlen, char *msg) {
    int   idx = 0;
    char *cur = NULL;

    memset(value, '\0', maxlen);
    cur = msg;
    for (idx = 0; idx < ppos; idx++) {
        cur = strchr (cur, ',');
        if (cur == NULL)
            return -1;
        cur++;
    }

    idx = 0;
    while ((cur[idx] != ',')  && (cur[idx] != '*') && 
           (cur[idx] != '\0') && (idx < maxlen)) {
        value[idx] = cur[idx];
        idx++;
    }

    return 0;
}

int main(int argc, char *argv[]) {
    FILE *f, *iw;
    char buf[MAXLEN], buf2[MAXGPSLEN], pos[MAXLEN], *ptr, *ptr2, datetime[24];
    char *iwcfg = NULL, *ssid = NULL, *ap = NULL;
    POS coords, *position = &coords;
    int res, idx = 0, len, link_away = -1;
    unsigned long timer, beeptimer, tmp1, tmp2, tmp3;
    int status, link, level, noise, discard_nwid, discard_crypt, discard_misc;
    struct timeb timing;
    struct tm *the_time;
    time_t epoch;
    int socks[2];
    char location[MAXLEN] = "00:00:00.0000? 00:00:00.0000?";

    device        = DEFAULT_INTERFACE;
    serport       = DEFAULT_SERPORT;
    file          = DEFAULT_FILE;
    reset         = DEFAULT_RESET_SCRIPT;
    init          = NULL;
    save_interval = DEFAULT_SAVE_INTERVAL;
    link_save     = DEFAULT_LINK_SAVE;
    link_beep     = DEFAULT_LINK_BEEP;
    beep_interval = DEFAULT_BEEP_INTERVAL;
    verbose       = 0;
    ignore_errors = 0;
    winfo         = 0;

    prg = argv[0];
    g = NULL;
    debug = 0;
    pid = -1;
    strcpy(lockfile, "/var/lock/");
    
    setvbuf(stdout, NULL, _IONBF, 0);
    setvbuf(stderr, NULL, _IONBF, 0);

    if ((argc == 2) && (strncmp("-h", argv[1], 2) == 0 || strncmp("--h", argv[1], 3) == 0))
        help();

    while ((res = getopt(argc, argv, "B:b:d:DGWhi:I:R:l:p:o:vhH?-")) != EOF) {
        switch(res) {
            case 'B': beep_interval = atoi(optarg); continue;
            case 'b': link_beep = atoi(optarg); continue;
            case 'd': device = optarg; continue;
            case 'D': debug = 1; continue;
            case 'G': ignore_errors = 1; continue;
            case 'i': save_interval = (atoi(optarg) * 1000); continue;
            case 'I': init = optarg; continue;
            case 'R': reset = optarg; if (strlen(reset) == 0) reset = NULL; continue;
            case 'W': winfo = 1; continue;
            case 'l': link_save = atoi(optarg); continue;
            case 'p': serport = optarg; continue;
            case 'o': file = optarg; continue;
            case 'v': verbose = 1; continue;
            case 'h':
            case 'H':
            case '?':
            case '-':
            default:  help();
        }
    }

    if (optind != argc) {
        fprintf(stderr, "%s: Invalid command line options.\n", PROGRAM);
        help();
    }

    if (geteuid()) {
        fprintf(stderr, "%s: You must be root to run this.\n", PROGRAM);
        exit(-1);
    }

// prepare path exec and init/reset calls
    if (reset != NULL || init != NULL || winfo != 0) {
        ptr = getenv("PATH");
        if (ptr == NULL) {
            ptr2 = malloc(strlen("PATH=."));
            strcpy(ptr2, "PATH=.");
        } else {
            ptr2 = concat("PATH=", ptr, ":.");
        }
        putenv(ptr2);
        ptr = ptr2 = NULL;
    }
    if (init != NULL)
        init = concat(init, " ", device);
    if (reset != NULL)
        reset = concat(reset, " ", device);
    if (winfo)
        iwcfg = concat("iwconfig", " ", device);

    if ((f = fopen(PROC_WAVELAN, "r")) == NULL) {
        fprintf(stderr, "%s: Can not open %s for reading.\n", PROGRAM, PROC_WAVELAN);
        exit(-1);
    }
    do {
        if (fgets(buf, sizeof(buf), f) == NULL) {
            fprintf(stderr, "%s: Unable to read from %s or device %s not found - wavelan card not active?\n", PROGRAM, PROC_WAVELAN, device);
            exit(-1);
        }
    } while (strstr(buf, device) == NULL || feof(f));
    if (strstr(buf, device) == NULL) {
        fprintf(stderr, "%s: device %s not found in %s - wavelan card not active?\n", PROGRAM, device, PROC_WAVELAN);
        exit(-1);
    }
    fclose(f);

    if (init != NULL) {
        if (debug) printf("DEBUG: RUNNING %s\n", init);
        if (system(init) != 0)
            fprintf(stderr, "%s: Can not find init script %s\n", PROGRAM, init);
    }

    signal(SIGINT,  clean_up);
    signal(SIGHUP,  clean_up);
    signal(SIGTERM, clean_up);
    signal(SIGQUIT, clean_up);
    signal(SIGABRT, SIG_IGN);
    signal(SIGPIPE, SIG_IGN);
    signal(SIGCHLD, SIG_IGN);

    strncat(lockfile, (char *) (strrchr(serport, '/') + 1), sizeof(lockfile) - strlen(lockfile) - 7);
    strcat(lockfile, ".lock");
    if ((res = open(lockfile, O_RDWR | O_CREAT | O_EXCL, 0644)) < 0) {
        fprintf(stderr, "%s: Can not create lockfile %s - some other program is using %s\n", PROGRAM, lockfile, serport);
        exit(-1);
    }
    sprintf(buf, "%d\n", getpid());
    (void) write(res, buf, strlen(buf) + 1);
    close(res);

    /* some things from gpsutil-0.05 */
    if ((fd_serport = open(serport, O_RDWR | O_NOCTTY | O_NDELAY)) < 0) {
        unlink(lockfile);
        if (ignore_errors) {
            ignore_errors++;
        } else {
            fprintf(stderr, "%s: Can not open seriell port %s.\n", PROGRAM, serport);
            exit(-1);
        }
    } else {
        fcntl(fd_serport, F_SETFL, 0);
        tcgetattr(fd_serport, &serport_opts_orig);
        memset (&serport_opts, 0, sizeof(serport_opts));
        serport_opts.c_cflag |= CRTSCTS; // hadware flow on
        serport_opts.c_cflag &= ~PARENB; // no parity
        serport_opts.c_cflag &= ~CSTOPB; // one stopbit
        serport_opts.c_cflag &= CSIZE;
        serport_opts.c_cflag |= CS8;     // 8N1
        serport_opts.c_cflag |= (CLOCAL | CREAD);  // enable Localmode, receiver
        serport_opts.c_cc[VMIN] = 0;               // set min read characters if 0, VTIME takes over
        serport_opts.c_cc[VTIME] = V_TIME;         // wait V_TIME ms for character
        if (ignore_errors <= 1 && (res = cfsetospeed(&serport_opts, B4800)) < 0) {
            fprintf(stderr, "%s: Unable to configure seriell port %s of GPS device.\n", PROGRAM, serport);
            seriell_clean_up(ignore_errors);
            ignore_errors++;
        }
        if (ignore_errors <= 1 && tcsetattr(fd_serport,TCSANOW, &serport_opts) < 0) {
            fprintf(stderr, "%s: Unable to configure seriell port %s of GPS device.\n", PROGRAM, serport);
            seriell_clean_up(ignore_errors);
            ignore_errors++;
        }
        if (ignore_errors <= 1) {
            int errors = 0;
            buf[0] = ' ';
            while (buf[0] != '\r' && errors < 3)
                if (read(fd_serport, &buf[0], 1) < 1)
                    errors++;
	    if (errors >= 3) {
	        fprintf(stderr, "%s: No response from GPS on seriell port %s.\n", PROGRAM, serport);
	        seriell_clean_up(ignore_errors);
	        ignore_errors++;
	    }
        } else
            fd_serport = -1;
    }
    if (ignore_errors > 1)
        fprintf(stderr, "%s: GPS could not be configured, disabled support and still running ...\n", PROGRAM);
    if ((res = open(file, O_RDWR | O_APPEND)) < 0) {
        if ((res  = open(file, O_RDWR | O_APPEND | O_CREAT | O_EXCL, 0600)) < 0) {
            fprintf(stderr, "%s: Unable to create file %s\n", PROGRAM, file);
            clean_up(-1);
        }
        write(res, OUTPUT_FILE_HEADER, strlen(OUTPUT_FILE_HEADER));
    }
    close(res);
    g = fopen(file, "a"); /* why is this needed? this is weird! */
    f = fopen(file, "a");
    fclose(g);
    g = f;
    if (link_save > 0)
        setvbuf(g, NULL, _IONBF, 0);

    if (ignore_errors < 2) {
        if (socketpair(PF_UNIX, SOCK_STREAM, 0, socks) != 0) {
            fprintf(stderr, "%s: Could not create STREAM socketpair\n", PROGRAM);
            clean_up(-1);
        }

        if ((pid = fork()) == 0) {
            char reading, buf3[48], *pptr = NULL, *tptr = NULL;
            int lost = 1;

            memset(&coords, 0, sizeof(coords));
            fcntl(socks[0], F_SETFL, O_NONBLOCK);
            while(1) {
                res = 0;
                memset(buf2, 0, MAXGPSLEN);

                while(read(fd_serport, &buf2, MAXGPSLEN - 1) <= 0);

                while (res == 0) {
 		    if ((pptr = rindex(buf2, '\n')) != NULL && (pptr = strstr(buf2, "GLL")) != NULL && *(pptr -= 3) == '$') {
		        if ((tptr = index(pptr, '\r')) != NULL)
		            *tptr = 0;
		        if ((tptr = index(pptr, '\n')) != NULL)
		            *tptr = 0;
		        res = 1;
		        if (debug && verbose) printf("DEBUG: GPS MSG - %s\n", pptr);
		    }
		    if (pptr == NULL)
		        res = -1;
		}

/* This is the old function which resulted in a high backlog of GPS msgs
		len = 0;
                while (!res) {
                    if (read(fd_serport, &reading, 1) == 1) {
                        if (reading != '\r') {
                            buf2[len] = reading;
                        }
                        if (reading == '\n')
                            res = 1;
                        }
                }
                len = strlen(buf2);
                if (debug && verbose) printf("DEBUG: GPS MSG length %d - %s", len, buf2);
                len = strlen(NMEA_POS_STRING);
                for (idx = 0; idx < len; idx++) {
                   if (NMEA_POS_STRING[idx] == '?')
                       continue;
                   if (NMEA_POS_STRING[idx] != buf2[idx]) {
                       res = 0;
                   }
                }
                pptr = buf2;
*/

		if (res == 1 && strstr(pptr, ",,,") != NULL) {
		    res = 0;
		    if (debug && verbose == 0)
 		        printf("DEBUG: EMPTY GPS POS MSG FOUND: %s\n", pptr);
 		    else if (verbose) {
 		        if (lost == 0) {
 		            lost = 1;
 		            printf("%s: Warning - GPS lost position!\n", prg);
 		        }
 		    }
		}
		if (res == 1 && index(pptr + 2, '$') != NULL) {
		    res = 0;
		    if (debug) {
		        printf("DEBUG: FUCKED UP GPS POS MSG FOUND: %s\n", pptr);
		    }
		}

		if (res == 1) {
		    lost = 0;
                    get_parameter(1, pos, 25, pptr);
                    position->Latitude.Hours = str2int(pos, 2);
                    len = strlen(pos) + 1;
                    for (idx = 2; idx < len; idx++)
                         pos[idx - 2] = pos[idx];
                    position->Latitude.Minutes = str2int(pos, 2);
                    len = strlen(pos) + 1;
                    for (idx = 3; idx < len; idx++)
                        pos[idx - 3] = pos[idx];
                    position->Latitude.Seconds = ((atoi(pos) * 1.0) / 10000) * 60;

                    get_parameter(2, pos, 25, pptr);
                    position->Latitude.Dir = pos[0];

                    get_parameter(3, pos, 25, pptr);
		    position->Longitude.Hours = str2int(pos,3);
                    len = strlen (pos) + 1;
                    for (idx = 3; idx < len; idx++)
                    pos[idx - 3] = pos[idx];
                    position->Longitude.Minutes = str2int(pos, 2);
                    len = strlen (pos) + 1;
                    for (idx = 3; idx < len; idx++)
                    pos[idx - 3] = pos[idx];
                    position->Longitude.Seconds = ((atoi(pos) * 1.0) / 10000) * 60;
 
                    get_parameter(4, pos, 25, pptr);
                    position->Longitude.Dir = pos[0];

                    if (position->Latitude.Dir != 'N' && position->Latitude.Dir != 'S')
                        position->Latitude.Dir = '?';
                    if (position->Longitude.Dir != 'E' && position->Longitude.Dir != 'W')
                        position->Longitude.Dir = '?';
                    
// this is a temporary debugging function to find out why it sometimes fucks up
/*
                    if (position->Longitude.Dir == '?' || position->Latitude.Dir == '?') {
                        FILE *df;
			if ((df=fopen("./.wardrive.DEBUG", "a")) != NULL) {
			    fprintf(df, "DEBUG\nGPS: %spos: %02d:%02d:%02.04f%c %02d:%02d:%02.04f%c\n\n",pptr,
                              position->Latitude.Hours, position->Latitude.Minutes, position->Latitude.Seconds, position->Latitude.Dir,
                              position->Longitude.Hours, position->Longitude.Minutes, position->Longitude.Seconds, position->Longitude.Dir);
			}
                    }
*/
                }

		if (read(socks[1], &reading, 1) > 0) {
		    if (position->Longitude.Dir != '?' && position->Latitude.Dir != '?')
 		        snprintf(buf3, sizeof(buf3), "%02d:%02d:%02.04f%c %02d:%02d:%02.04f%c\n",
                          position->Latitude.Hours, position->Latitude.Minutes, position->Latitude.Seconds, position->Latitude.Dir,
                          position->Longitude.Hours, position->Longitude.Minutes, position->Longitude.Seconds, position->Longitude.Dir);
                    write(socks[1], &buf3, strlen(buf3));
                }
            }
        }
        if (pid < 0) {
            fprintf(stderr, "%s: Fork failed.\n", PROGRAM);
            clean_up(-1);
        }
    }

    ftime(&timing);
    tmp1 = timing.time;
    tmp2 = timing.millitm;
    tmp3 = tmp1 * 1000;
    timer = tmp3 + tmp2;
    beeptimer = 0;

    printf("Starting logging, saving to %s; press Control-C to end logging ...\n", file);
    while (1) {

/* wavelan */
        if ((f = fopen(PROC_WAVELAN, "r")) == NULL) {
            fprintf(stderr, "%s: Unable to open %s\n", PROGRAM, PROC_WAVELAN);
            clean_up(-1);
        }
        do {
            if (fgets(buf, sizeof(buf), f) == NULL) {
                fprintf(stderr, "%s: Unable to read from %s\n", PROGRAM, PROC_WAVELAN);
                clean_up(-1);
            }
        } while (strstr(buf, device) == NULL || feof(f));
        if (strstr(buf, device) == NULL) {
            fprintf(stderr, "%s: device %s not found in %s - card not active anymore?\n", PROGRAM, device, PROC_WAVELAN);
            clean_up(-1);
        }
        fclose(f);
        ptr = strstr(buf, device) + strlen(device) + 2;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        status = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        link = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        level = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        noise = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        discard_nwid = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;
        
        ptr2 = ptr;
        while (*ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        *ptr2 = 0;
        discard_crypt = atoi(ptr);
        ptr += strlen(ptr) + 1;
        while (*ptr == 0 || *ptr == ' ' || *ptr == '\t') ptr++;

        ptr2 = ptr;
        while (*ptr2 != '\r' && *ptr2 != '\n' && *ptr2 != 0 && *ptr2 != ' ' && *ptr2 != '\t' && *ptr2 != '.') ptr2++;
        if (ptr2 != NULL) *ptr2 = 0;
        discard_misc = atoi(ptr);

/* gps */
        if (pid > 0) {
            int end;
            write(socks[0], "N", 1);
	    end = read(socks[0], &location, sizeof(location));
	    if (end < 0) end = 0;
	    location[end] = '\0';
	    if (strlen(location) > 0 && location[strlen(location)-1] == '\n')
	        location[strlen(location)-1] = '\0';
	    if (strlen(location) == 0 || strstr(location, "00:00:0") != NULL || index(location, '?') != NULL)
	        strcpy(location, "00:00:00.0000? 00:00:00.0000?");
        }

/* save data */
	time(&epoch);
	the_time = localtime(&epoch);
	strftime(datetime, sizeof(datetime), "%Y-%m-%d %H:%M:%S", the_time);

        if (link >= link_save)
            fprintf(g, "%s %s %d %d %d %d %d %d %d\n",
		datetime, location,
		status, link, level, noise, discard_nwid, discard_crypt, discard_misc);
        if ((link >= link_save && verbose) || debug)
            printf("%s %s %d %d %d %d %d %d %d\n",
		datetime, location,
		status, link, level, noise, discard_nwid, discard_crypt, discard_misc);

/* beeping */
        ftime(&timing);
        if (link >= link_beep) {
            if (beep_interval < (timing.time - beeptimer)) {
                printf("\007");
                beeptimer = timing.time;
            }
        }

        if (link_away == 4 && reset != NULL) {
	    link_away = -1;
	    if (debug) printf("DEBUG: RUNNING %s\n", reset);
	    if (system(reset) != 0) {
	        fprintf(stderr, "%s: Can not find reset script %s, disabling -R option\n", PROGRAM, reset);
	        reset = NULL;
	    }
        }

/* get SSID and MAC from found wavelan */
        if (link > 0) {
            if (winfo && link_away == -1) {
                ap = ssid = NULL;
                iw = popen(iwcfg, "r");
                (void) wait3(NULL, WNOHANG, NULL);
                while (fgets(buf, sizeof(buf), iw) != NULL) {
                    if (ssid == NULL && (ptr = strstr(buf, "SSID")) != NULL) {
                        if ((ptr2 = index(ptr, '"')) != NULL && (ptr2 = index(++ptr2, '"')) != NULL)
                            *++ptr2 = 0;
                        ssid = malloc(strlen(ptr) + 1);
                        strcpy(ssid, ptr);
                    }
                    if (ap == NULL && (ptr2 = strstr(buf, "Access Point: ")) != NULL) {
                        ap = malloc(strlen(ptr2) + 1);
                        strcpy(ap, ptr2);
                    }
                }
                if (ssid != NULL || ap != NULL) {
                    fprintf(g, "%s %s WINFO - %s %s",
 		        datetime, location, ssid, ap);
                    if (verbose) printf("%s %s WINFO - %s %s",
 		        datetime, location, ssid, ap);
		} else {
		    fprintf(stderr, "%s: iwconfig did not show access point or ssid info, disabling -W option\n", PROGRAM);
		    winfo = 0;
		}
                pclose(iw);
                (void) wait3(NULL, WNOHANG, NULL);
            }
            link_away = 0;
	} else
	    if (link_away >= 0 && link_away < 255) link_away++;

/* wait for next check */
        tmp1 = timing.time;
        tmp2 = timing.millitm;
        tmp3 = tmp1 * 1000;
        timer = (tmp3 + tmp2) - timer;
        if (save_interval > timer) {
            struct timeval tv;
            timer = save_interval - timer;
//            if (debug && verbose) printf("DEBUG: WAITING FOR %lu milliseconds\n", timer);
            tv.tv_sec = timer / 1000;
            tv.tv_usec = (timer % 1000) * 1000;
            select(0, NULL, NULL, NULL, &tv);
            ftime(&timing);
            tmp1 = timing.time;
            tmp2 = timing.millitm;
            tmp3 = tmp1 * 1000;
            timer = tmp3 + tmp2;
        }
    }

    clean_up(0);
    return 0;
}
