#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
extern int errno;

#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/file.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>

#ifdef linux
  #include "linux_ip_icmp.h"
#else
  #include <netinet/ip.h>
  #include <netinet/ip_icmp.h>
#endif

#include <arpa/inet.h>
#include <netdb.h>

#include "icmp_tunnel.h"

#define back(eno) \
{ \
  eit_errno = eno;  \
  return(-1);       \
}

int                sockfd;          /* socket file descriptor */

u_int it_init = 1;
int   errno;
int   eit_errno;
int   rtt_to;


u_short in_chksum(u_short *ptr, int nbytes)
{
  register long           sum;            /* assumes long == 32 bits */
  u_short                 oddbyte;
  register u_short        answer;         /* assumes u_short == 16 bits */

  /*
   * Our algorithm is simple, using a 32-bit accumulator (sum),
   * we add sequential 16-bit words to it, and at the end, fold back
   * all the carry bits from the top 16 bits into the lower 16 bits.
   */

  sum = 0;
  while (nbytes > 1)
  {
    sum += *ptr++;
    nbytes -= 2;
  }

  /* mop up an odd byte, if necessary */
  if (nbytes == 1)
  {
    oddbyte = 0;            /* make sure top half is zero */
    *((u_char *) &oddbyte) = *(u_char *)ptr;   /* one byte only */
    sum += oddbyte;
  }

  /*
   * Add back carry outs from top 16 bits to low 16 bits.
   */

  sum  = (sum >> 16) + (sum & 0xffff);    /* add high-16 to low-16 */
  sum += (sum >> 16);                     /* add carry */
  answer = ~sum;          /* ones-complement, then truncate to 16 bits */

  return((u_short) answer);
}

int tunnel_init(void)
{
  struct protoent *proto;
  
  if(it_init)
  {
    /* Get protocol info. */
    if( (proto = getprotobyname("icmp")) == NULL )
      back(EIT_PROTO)
  
    /* Create socket. */
    if( (sockfd = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0 )
      back(EIT_SOCKET)

    it_init = 0;
    return(0);
  }
  
  /* Everything is done */
}

void tunnel_reset(void)
{
  close(sockfd);
  it_init = 1;
}

int tunnel_send(char *send_mesg, size_t mesglen, u_long srv_ip)
{
  u_char                sendpack[MAXPACKET] = { 0 };  /* the packet we send */
  int                   sended;
  register struct icmp  *icmp;
  int                   packsize;
  struct sockaddr_in    dest;
  int                   destlen;
      
  
  if(mesglen > MAXMESG)
    back(EIT_2BIG)
    
  if(it_init)        /* Tunnel was not set up */
    if(tunnel_init() < 0)
      return(-1);
  
  destlen = sizeof(dest);
  bzero((char *) &dest, destlen);
  dest.sin_family       = AF_INET;
  dest.sin_addr.s_addr  = srv_ip;
 
  packsize = mesglen + SIZE_ICMP_HDR;

  /*
  ** Set up ICMP header.
  */
  
  icmp = (struct icmp *) sendpack;    /* set pointer to ICMP header */
  icmp->icmp_type = ICMP_ECHOREPLY;
  bcopy(send_mesg, icmp->icmp_data, mesglen);
  icmp->icmp_cksum = in_chksum((u_short *) icmp, packsize);

  /*
  ** Send packet... Good Bye.
  */
  if( (sended = sendto(sockfd, sendpack, packsize, 0, (struct sockaddr *) &dest, destlen)) < 0 )
    back(EIT_SENDTO)
  else if(sended != packsize)
    back(EIT_SENDLESS)

  return(sended);
}


int tunnel_recv(char *recv_mesg, size_t mesglen)
{
  u_char              recvpack[MAXPACKET] = { 0 };
  int                 received;
  struct ip           *ip;
  int                 iphdrlen;
  struct icmp         *icmp;
  int                 clilen = sizeof(clisrc);
  
  
  if(it_init)        /* Tunnel was not set up */
    if(tunnel_init() < 0)
      return(-1);
  
  while(1)
  {
    bzero(recvpack, sizeof(recvpack));
    if( (received = recvfrom(sockfd, recvpack, MAXPACKET, 0, (struct sockaddr *) &clisrc, &clilen)) < 0 )
      continue;
    /* either recvfrom() error or system call was interrupted by a signal */
   
    
    /*
    ** We catch a ICMP packet.
    ** It includes the IP and ICMP header.
    **
    ** Check the IP header to get it's length and verify if it includes an
    ** ICMP packet.
    */
    ip = (struct ip *) recvpack;
    iphdrlen = ip->ip_hl << 2;    /* convert # 32-bit words to # bytes */
    if(received < (iphdrlen + ICMP_MINLEN))
      continue; /* packet is too short */
    received -= iphdrlen;
  
    icmp = (struct icmp *) (recvpack + iphdrlen);
 
    /*
    ** Check if the packet belongs to us.
    */
    if(!icmp->icmp_id && !icmp->icmp_code && icmp->icmp_type == ICMP_ECHOREPLY)
      break;
  }
        
  /*
  ** We received a ICMP ECHO REPLY Packet w/ our ident and our code.
  ** Calculate the datalength and store the data in the user's
  ** buffer.
  */
   
  received -= SIZE_ICMP_HDR;
  bcopy(icmp->icmp_data, recv_mesg, received);
    
  return(received); 
}







