#
#  Calculate total trace lengths...
#  (1) Parse trace segment lengths from the Mentor output
#  (2) Read signal names from a file
#  (3) Print total lengths to an output file
#
#  Created by Stephen Wint, 2013
#
use strict;
use warnings;
use Tkx;

our $VERSION = "1.20";

my $default_infile = "Length_Summary.txt";
my $default_signame_list = "SignalNameList.txt";
my $default_outfile = "trace_length_report.txt";
my $default_post_ext = "_TAP";
my $line1 = "------------------\n";
my $line2 = "==================\n";

my $config_file = ".siglen_cfg.txt";
my $infile = $default_infile;
my $outfile = $default_outfile;
my $signames = $default_signame_list;
my $post_ext = $default_post_ext;
my $inline;
my $otxt;
my ($i, $j, $x, $y, $z);
my ($sig_index, $len_index, $rep_index) = (0, 0, 0);
my ($base, $base_TAP);
my ($done, $sig_flag, $special) = (0, 0, 0);
my $signal;
my ($avg, $cnt, $tot) = (0, 0, 0);
my ($dif, $max, $min) = (0, 0, 10000);
my (@lengths, @report, @sorted);
my ($unit, $longest, $shortest) = ("", "", "");
my $temp;
my $verbose = 0;
my $wrtsig = 0;
my ($avg_pre, $tot_pre) = (0, 0);
my ($dif_pre, $max_pre, $min_pre) = (0, 0, 10000);
my ($pre_longest, $pre_shortest) = ("", "");
my ($avg_post, $tot_post) = (0, 0);
my ($dif_post, $max_post, $min_post) = (0, 0, 10000);
my ($post_longest, $post_shortest) = ("", "");

my $mw = Tkx::widget->new(".");
my ($b1, $b2, $b3, $b4, $bb, $bh, $bm, $bn, $bx);
my ($cb, $cc, $e1, $e2, $e3, $f1, $f2, $f3, $f4, $f5, $f6);
my ($f7, $f8, $l1, $more, $ok, $ow, $ox);
my ($rb1, $rb2, $rb3, $rb4);
my ($sel, $vb, $wc, $ww, $wx, $xb, $xx);
my ($ls1, $s1, $sv);
my @button_labels = ['OK', 'CANCEL'];
my $help;
my $tabulate = 0;
my ($sort_option, $show_prepost, $ovwrt) = (0, 0, 0);


if (-e $config_file) {
  #  read configuration items
  #  Valid options (one per line):
  #    sort_ascending
  #    sort_descending
  #    sort_alpha
  #    use_tabs
  #    print_prepost
  #    verbose
  #    ext=_xxx
  #      where "_xxx" is the Post Rtip signal extension
  #
  open(CFILE, '<', $config_file);
  while (not eof(CFILE)) {
    $inline = readline CFILE;
    chomp $inline;
    #  skip comments and blank lines
    if ((substr($inline, 0, 1) ne '#') and ($inline ne "\n")) {
      #  Sort options
      if ($inline =~ 'sort_ascending') {
        $sort_option = 1;
      } elsif ($inline =~ 'sort_descending') {
        $sort_option = 2;
      } elsif ($inline =~ 'sort_alpha') {
        $sort_option = 3;
      }
      #  tab use
      if ($inline =~ 'use_tabs') {
        $tabulate = 1;
      }
      #  pre/post printing
      if ($inline =~ 'print_prepost') {
        $show_prepost = 1;
      }
      #  verbosity
      if ($inline =~ 'verbose') {
        $verbose = 1;
      }
      #  post_fix extension for signal names
      if ($inline =~ 'ext') {
        ($x, $y) = split("=", $inline);
        $post_ext = $y;
      }
    }
  }
} else {
  #  set up default values
  $infile = $default_infile;
  $outfile = $default_outfile;
  $signames = $default_signame_list;
  $post_ext = $default_post_ext;
}


$mw->g_wm_title("Signal/Trace Length Parser V$VERSION");
$mw->g_wm_minsize(400,255);
#  Button & entry for Trace Length Summary input file name
$f1 = $mw->new_frame();
$f1->g_pack(-pady=>2, -side=>'top', -fill=>'x');
$b1 = $f1->new_ttk__button(
  -text=>"Read Trace Length Data:",
  -command=>sub { GET_TRACES() }
);
$b1->g_pack(-side=>'left', -padx=>2, -anchor=>'nw',
  -fill=>'none', -expand=>0);
$e1 = $f1->new_entry(
  -width=> 30, -background => 'white'
);
if (-e $infile) { $e1->insert('end', "$infile") }
$e1->g_pack(-side=>'left', -anchor=>'nw', -expand=>1, -fill=>'x');
#  Browse button to look for input file
$bb = $f1->new_ttk__button(-text=>"Browse",
  -command=> sub {
    my $input = Tkx::tk___getOpenFile();
    $input =~ s/\//\\/g;
    if (-e $input) {
      $infile = $input;
      $e1->delete(0, 'end');
      $e1->insert('end', "$infile");
    }
  }
);
$bb->g_pack(-side=>'right', -anchor=>'ne', -expand=>0, -fill=>'none');
#  Entry for Post Rtip signal extension
$f3 = $mw->new_frame();
$f3->g_pack(-pady=>1, -side=>'top', -anchor=>'nw', -fill=>'x');
$bx = $f3->new_label(
  -text => "Post Rtip signal extension: "
);
$bx->g_pack(-side=>'left', -padx=>3, -anchor=>'nw',
  -expand=>0, -fill=>'none');
$e3 = $f3->new_entry(
  -width=> 10,
  -background => 'white'
);
$e3->insert('end', "$post_ext");
$e3->g_pack(-side=>'left', -anchor=>'nw', -expand=>1, -fill=>'x');
#  Button & Entry for reading format file & generating report
$f2 = $mw->new_frame();
$f2->g_pack(-pady=>1, -side=>'top', -anchor=>'nw', -fill=>'x');
$b2 = $f2->new_ttk__button(
  -text => "Read Format File \&\n Process Trace Lengths ",
  -command => sub { GET_SIGLIST() }
);
$b2->g_pack(-side=>'left', -padx=>3, -anchor=>'nw');
$e2 = $f2->new_entry(
  -width=> 29, -background => 'white'
);
if (-e $signames) { $e2->insert('end', "$signames") }
$e2->g_pack(-side=>'left', -anchor=>'nw', -expand=>1, -fill=>'x');
#  Button for creating new Format File
$bn = $f2->new_ttk__button(
  -text => " Create New \n Format File ",
  -command => sub { WRITE_SIGLIST() }
);
$bn->g_pack(-side=>'right', -padx=>3, -anchor=>'nw');
#  Listbox for showing status & messages
$l1 = $mw->new_listbox(
  -width => 60, -height => 8
);
$l1->g_pack(-expand=>1, -fill=>'both', -anchor=>'nw');
#  Help, More, Clear & Exit buttons
$f4 = $mw->new_frame();
$f4->g_pack(-pady=>1, -side=>'bottom');
$bh = $f4->new_ttk__button(
  -text=>"Help", -width=>8,
  -command=> sub { HELP_BUTTON() }
);
$bh->g_pack(-padx=>6, -side=>'left');
$bm = $f4->new_ttk__button(
  -text=>'More', -width=>8,
  -command=> sub { MORE_BUTTON() }
);
$bm->g_pack(-padx=>6, -side=>'left');
$b3 = $f4->new_ttk__button(
  -text=>'Clear', -width=>8,
  -command=> sub {
    $l1->delete(0,'end');
    $sig_flag = 0;
    $len_index = 0;
    $wrtsig = 0;
    $rep_index = 0;
    @report = ();
  }
);
$b3->g_pack(-padx=>6, -side=>'left');
$b4 = $f4->new_ttk__button(
  -text=>'Exit', -width=>8,
  -command=> sub {
    $mw->g_destroy,
    $more->g_destroy
  }
);
$b4->g_pack(-padx=>6, -side=>'right');


Tkx::MainLoop();

exit;


#  subroutine for "MORE" button to cover advanced config options
sub MORE_BUTTON {
  $more = $mw->new_toplevel();
  $more->g_wm_minsize(300,220);
  $more->g_wm_title('Advanced Configuration Options');
  #  Checkbox for printing pre/post statistics
  $f5 = $more->new_frame(-borderwidth=>1, -relief=>'groove');
  $f5->g_pack(-pady=>2, -side=>'top');
  $cb = $f5->new_checkbutton(
    -text=>'Output Pre/Post Values',
    -variable=>\$show_prepost
  );
  $cb->g_pack(-padx=>[5,12]);
  if ($show_prepost) {
    $cb->select;
  } else {
    $cb->deselect;
  }
  #  place a separator here
  $s1 = $more->new_frame(-width=>75, -borderwidth=>3,
    -relief=>'groove', -background=>'black');
  $s1->g_pack(-ipadx=>4);
  $ls1 = $s1->new_label(-text=>"SORT OPTION");
  $ls1->g_pack();
  #  Add sorting options ($sort_option)
  #    0) No sort
  #    1) Sort Ascending length
  #    2) Sort Descending length
  #    3) Sort Alphabetical (by name)
  $f6 = $more->new_frame(-borderwidth=>1, -relief=>'groove');
  $f6->g_pack(-side=>'top');
  $rb1 = $f6->new_radiobutton(-text=>'Use Format File Order',
    -value=>0, -variable=>\$sort_option);
  $rb1->g_pack(-padx=>[0, 20]);
  $rb2 = $f6->new_radiobutton(-text=>'Sort Ascending (Length)',
    -value=>1, -variable=>\$sort_option);
  $rb2->g_pack(-padx=>[1,8]);
  $rb3 = $f6->new_radiobutton(-text=>'Sort Descending (Length)',
    -value=>2, -variable=>\$sort_option);
  $rb3->g_pack(-padx=>[2,3]);
  $rb4 = $f6->new_radiobutton(-text=>'Sort Alphabetical (by name)',
    -value=>3, -variable=>\$sort_option);
  $rb4->g_pack(-padx=>[14,3]);
  #  Align columns (signal length) using tabs
  $f7 = $more->new_frame(-borderwidth=>1, -relief=>'groove');
  $f7->g_pack(-side=>'top', -pady=>3);
  $cc = $f7->new_checkbutton(
    -text=>'Use tabs in report',
    -variable=>\$tabulate
  ); 
  $cc->g_pack(-padx=>[4,35]);
  #  Verbose mode
  $f8 = $more->new_frame(-borderwidth=>1, -relief=>'groove');
  $f8->g_pack(-side=>'top', -pady=>3);
  $vb = $f8->new_checkbutton(
    -text=>'Verbose message mode',
    -variable=>\$verbose
  ); 
  $vb->g_pack(-padx=>[2,5]);
  #  OK button
  $xb = $more->new_frame();
  $xb->g_pack(-side=>'bottom');
  $ok = $xb->new_ttk__button(
    -text=>'OK',
    -command=> sub { $more->g_destroy }
  );
  $ok->g_pack(-side=>'right');
  $sv = $xb->new_ttk__button(
    -text=>'Save Config',
    -command=> sub { SAVE_CONFIG_FILE() }
  );
  $sv->g_pack(-side=>'left', -ipadx=>2, -padx=>4);
}


#  Save Configuration File
sub SAVE_CONFIG_FILE {
  if ($verbose >= 1) {print STDOUT "saving config file...\n"}
  open(CFILE, '>', $config_file);
  print CFILE "#\n#  siglen configuration file\n#\n";
  #  Sort options
  if ($sort_option == 1) {
    print CFILE "sort_ascending\n";
  } elsif ($sort_option == 2) {
    print CFILE "sort_descending\n";
  } elsif ($sort_option = 3) {
    print CFILE "sort_alpha\n";
  }
  #  use tabs in output file
  if ($tabulate == 1) {
    print CFILE "use_tabs\n";
  }
  #  pre & post length statistics printing
  if ($show_prepost == 1) {
    print CFILE "print_prepost\n";
  }
  #  verbosity
  if ($verbose >= 1) {print CFILE "verbose\n"}
  #  post_fix extension for signal names
  print CFILE "ext=$post_ext\n";
  print CFILE "\n";
  close(CFILE);
}


#  help window
sub HELP_BUTTON {
  Tkx::tk___messageBox(-title=>"Signal/Trace Length Parser Help",
    -message=>"This program reads a Trace Length Data file created " .
    "from within Mentor(TM) and parses it to create a composite " .
    "trace length report. The schematic must use a standard signal " .
    "naming scheme (signal and signalxxx where xxx=extension, " .
    "default is \'_TAP\'). A Format File allows you to " .
    "specify the signal grouping and ordering so min/max/average " .
    "trace lengths can be reported (per group).\n\n You can " .
    "generate a Format File to see what the syntax should look " .
    "like. (\'Create New Format File\' button). "
  );

}


#  get Mentor trace length report filename
#  and read in data for signals and lengths
sub GET_TRACES {
  $infile = $e1->get();
  if ($infile eq "") {
    $infile = $default_infile;
    $e1->insert('end', "$infile");
  }
  if (not open(INFILE, '<', $infile)) {
    $l1->insert('end', "Cannot open $infile: $!");
    return
  }
  #  read in trace lengths and assign to variables
  #  Data format:
  #    signal_name <b> length <b>
  #  where <b> = blank space(s)
  #
  #  skip lines at the top of the Mentor report
  for ($i=1; $i<7; $i++) {
    $inline = readline INFILE;
  }
  #  read out to "Length (" to determin if this data is in:
  #    mils "th)" or
  #    mm   "mm)
  $done = 0;
  while (not eof(INFILE) and not $done) {
    #  read a line
    $inline = readline INFILE;
    if ($inline =~ /Length/) {
      #  Get units of Length
      ($x, $y) = split("\\(", $inline);
      if (substr($y, 0, 2) eq 'th') {
        $unit = 'mils'
      } elsif (substr($y, 0, 2) = 'mm') {
        $unit = 'mm'
      }
      $done = 1;
      if ($verbose >= 1) {print STDOUT "  LENGTH UNITS: $unit\n"};
    }
  }
  while (not eof(INFILE)) {
    #  get signals and trace lengths
    #  store in $lengths[x][y] where:
    #    x=index into array
    #    y=0 for name, 1 for length
    $inline = readline INFILE;
    if ((substr($inline, 0, 1) ne '=') and 
        (substr($inline, 0, 1) ne '%') and
        ($inline ne "\n") and
        (substr($inline, 0, 8) ne 'Net Name')) {
      ($x, $y, $z) = split(" ", $inline);
      $lengths[$len_index] = [$x, $y];
      $len_index++;
      if ($verbose >= 1) {print STDOUT "  DATA: $x->$y\n"};
    }
  }
  if ($verbose >= 1) {print STDOUT "  LENGTH: $len_index\n"};
  close(INFILE);
  $sig_flag = 1;
  $l1->insert('end', "Finished reading trace length data: $infile");
  $l1->insert('end', "Read $len_index traces");
}


#  open signal name listing text file and read signal names
#
#  FORMAT:
#  %c => clear statistics & start calculating
#  %d => dump statistics (& clear)
#  %l => print a line in the output file
#  %s => create space (blank line) in the output file
#  %t => add the following text to a comment line in the output file
#  %x => accumulate the next two signals (use first name)
sub GET_SIGLIST {
  #  open/prepare files
  if ($sig_flag == 0) {
    $l1->insert('end', "You must read Trace Length Data first");
    return
  }
  if (not open(SIGS, '<', $signames)) {
    $l1->insert('end', "Cannot open $signames: $!");
    return
  }
  $sig_index = 0;
  #  check for existing $outfile file
  #  ask if it should be over-written
  if (-e $outfile) {
    $l1->insert('end', "Replacing existing output file.");
  }
  if (not open(OUTFILE, '>', $outfile)) {
    $l1->insert('end', "Cannot open $outfile: $!");
    return
  }
  print OUTFILE "#\n";
  print OUTFILE "#  Trace Length Output Report\n";
  print OUTFILE "#  Length are pre+post Rtip\n";
  print OUTFILE "#  Lengths in $unit\n";
  print OUTFILE "#\n";
  #  get "extension" for postfix to signal name
  $post_ext = $e3->get();
  #  process data
  while (not eof(SIGS)) {
    $inline = readline SIGS;
    if ((substr($inline, 0, 1) ne '#') and ($inline ne "\n")) {
      if (substr($inline, 0, 1) eq '%') {
        #  this is a directive from the signal name file
        $special = 0;
        if (substr($inline, 1, 1) eq 'c') {
          #  clear statistics
          $tot = 0;
          $cnt = 0;
          $max = 0;
          $min = 10000;
          #  pre/post Rtip lengths
          $tot_pre = 0;
          $max_pre = 0;
          $min_pre = 10000;
          $tot_post = 0;
          $max_post = 0;
          $min_post = 10000;
        } elsif (substr($inline, 1, 1) eq 'd') {
          #  write processed signals out to the report file
          WRT_SIGNALS();
          #  dump current statistics to output file
          $avg = $tot / $cnt;
          $dif = $max - $min;
          $avg_pre = $tot_pre / $cnt;
          $dif_pre = $max_pre - $min_pre;
          $avg_post = $tot_post / $cnt;
          $dif_post = $max_post - $min_post;
          print OUTFILE $line1;
          #  if signal name has extension, trim it off
          if (rindex($longest, $post_ext) != -1) {
            $longest = substr($longest, 0, -length($post_ext));
          }
          if (rindex($shortest, $post_ext) != -1) {
            $shortest = substr($shortest, 0, -length($post_ext));
          }
          printf OUTFILE "  AVG: %6.2f\n  MAX: %6.2f ($longest)\n  MIN: %6.2f ($shortest)\n  DIF: %6.2f\n", $avg, $max, $min, $dif;
          if ($show_prepost) {
            #  if signal name has extension, trim it off
            if (rindex($pre_longest, $post_ext) != -1) {
              $pre_longest = substr($pre_longest, 0, -length($post_ext));
            }
            if (rindex($pre_shortest, $post_ext) != -1) {
              $pre_shortest = substr($pre_shortest, 0, -length($post_ext));
            }
            printf OUTFILE "  ...pre Rtip trace segment...\n";
            printf OUTFILE "    AVG: %6.2f\n    MAX: %6.2f ($pre_longest)\n    MIN: %6.2f ($pre_shortest)\n    DIF: %6.2f\n", $avg_pre, $max_pre, $min_pre, $dif_pre;
            #  if signal name has extension, trim it off
            if (rindex($post_longest, $post_ext) != -1) {
              $post_longest = substr($post_longest, 0, -length($post_ext));
            }
            if (rindex($post_shortest, $post_ext) != -1) {
              $post_shortest = substr($post_shortest, 0, -length($post_ext));
            }
            printf OUTFILE "  ...post Rtip trace segment...\n";
            printf OUTFILE "      AVG: %6.2f\n      MAX: %6.2f ($post_longest)\n      MIN: %6.2f ($post_shortest)\n      DIF: %6.2f\n", $avg_post, $max_post, $min_post, $dif_post;
          }
          print OUTFILE $line2;
          $tot = 0;
          $cnt = 0;
          $max = 0;
          $min = 10000;
          #  pre/post Rtip lengths
          $tot_pre = 0;
          $max_pre = 0;
          $min_pre = 10000;
          $tot_post = 0;
          $max_post = 0;
          $min_post = 10000;
        } elsif (substr($inline, 1, 1) eq 'l') {
          #  insert a line in the output
	  #  If there are signals queued to write, do that first
          if ($wrtsig == 1) {
            #  write processed signals out to the report file
            WRT_SIGNALS();
          }
          print OUTFILE $line1;
        } elsif (substr($inline, 1, 1) eq 's') {
          #  insert space (a blank line) in the output
	  #  If there are signals queued to write, do that first
          if ($wrtsig == 1) {
            #  write processed signals out to the report file
            WRT_SIGNALS();
          }
          print OUTFILE "\n";
        } elsif (substr($inline, 1, 1) eq 'x') {
          #  add together the lengths for the next two signals
          #  and list in the output file unser the first signal name
          $special = 1;
        } elsif (substr($inline, 1, 1) eq 't') {
          #  print text to the output
	  #  If there are signals queued to write, do that first
          if ($wrtsig == 1) {
            #  write processed signals out to the report file
            WRT_SIGNALS();
          }
          $otxt = substr($inline, 2);
          printf OUTFILE "#  $otxt\n";
        }
      } else {
        #  this is a signal from the signal name file
        chomp $inline;
  
        if ($special == 0) {
          #  nothing special, read base signal name
          $base = $inline;
          $base_TAP = join('', $inline, $post_ext);
          $sig_index++
        } else {
          #  really special, read two signal names...
          $base = $inline;
          $inline = readline SIGS;
          chomp $inline;
          $base_TAP = $inline;
          #  clear the $special flag
          $special = 0;
          $sig_index++
        }

        #  scan through the lengths to find total for the current signal
        $done = 0;
        $j = 0;
        $signal = 0;
        while ($done lt 2) {
          if ($lengths[$j][0] eq $base) {
            #  This is the pre Rtip portion (base signal name w/o extension)
            $temp = $lengths[$j][1] / 2;
            $signal += $temp;
            if ($temp > $max_pre) {
              $max_pre = $temp;
              $pre_longest = $lengths[$j][0];
            }
            if ($temp < $min_pre) {
              $min_pre = $temp;
              $pre_shortest = $lengths[$j][0];
            }
            $tot_pre += $temp;
            #  Divide trace length by 2 because we are measuring
            #  from the BGA ball to the test pad (signal path)
            $done++;
            $j++;
          } elsif ($lengths[$j][0] eq $base_TAP) {
            #  This is the post Rtip portion of the trace
            $temp = $lengths[$j][1];
            $signal += $temp;
            if ($temp > $max_post) {
              $max_post = $temp;
              $post_longest = $lengths[$j][0];
            }
            if ($temp < $min_post) {
              $min_post = $temp;
              $post_shortest = $lengths[$j][0];
            }
            $tot_post += $temp;
            $done += 1;
            $j++;
          } else {
            if ($j < $len_index-1) {
              $j++
            } else {
              #  oops, hit the end and did not find all lengths
              $l1->insert('end',"Encountered end of signals but could not find\n
 $base or $base_TAP");
              $done = 2;
            }
          }
        }
        #  got final length, add to stats
        $cnt +=1;
        $tot += $signal;
        if ($signal > $max) {
          $max = $signal;
          $longest = $lengths[$j-1][0]
        }
        if ($signal < $min) {
          $min = $signal;
          $shortest = $lengths[$j-1][0]
        }

        if ($verbose >= 1) {print STDOUT "$base => $signal\n"}
        #  Save the signals/lengths in an array
        $report[$rep_index][0] = $base;
        $report[$rep_index][1] = $signal;
        $rep_index++;
        $wrtsig = 1;
      }
    }
  }
  close(SIGS);
  close(OUTFILE);
  $l1->insert('end', "Finished processing trace lengths: $signames");
}


#  create a new signal name file (template)
sub WRITE_SIGLIST {
  #  read trace length summary file
  #  (if it has not already been read)
  if ($sig_flag == 0) {
    $l1->insert('end', "Reading Trace Length Data");
    GET_TRACES();
  }
  if (not open(SIGS, '>', $signames)) {
    $l1->insert('end', "Cannot open $signames: $!");
    return
  }
  print SIGS "#\n#  Signal names for trace length processing\n#\n";
  print SIGS "#  FORMAT:\n";
  print SIGS "#    %c => clear statistics & start calculating\n";
  print SIGS "#    %d => dump statistics (& clear counters)\n";
  print SIGS "#    %l => print a line in the output file\n";
  print SIGS "#    %s => create space (blank line) in the output file\n";
  print SIGS "#    %t => add the following text to a comment line in the output file\n";
  print SIGS "#          %tTYPE YOUR TEXT HERE\n";
  print SIGS "#    %x => accumulate the next two signals (use first name)\n";
  print SIGS "#          %x\n#          SIG1_abc\n#          SIG2_xyz\n";
  print SIGS "#\n";
  #  process data
  print SIGS "%l\n";
  for ($i=0; $i<$len_index; $i++) {
    print SIGS "$lengths[$i][0]\n";
  }
  print SIGS "%d\n%s\n%c\n";
  close(SIGS);
  $l1->insert('end', "Please edit $signames to generate the report file");
  $l1->insert('end', "  output format you would like");
}


#  write processed signals out to the report file
sub WRT_SIGNALS {
  if ($sort_option == 0) {
    #  no sort
    @sorted = @report;
  } elsif ($sort_option == 1) {
    #  sort this signal group in ascending length order
    @sorted = sort { $a->[1] cmp $b->[1] } @report;
  } elsif ($sort_option == 2) {
    #  sort descending length order
    @sorted = sort { $b->[1] cmp $a->[1] } @report;
  } elsif ($sort_option == 3) {
    #  sort alphabetical
    @sorted = sort { $a->[0] cmp $b->[0] } @report;
  }
  for ($i=0; $i<$rep_index; $i++) {
    if ($tabulate == 1) {
      printf OUTFILE "$sorted[$i][0]\t=>\t%6.2f\n", $sorted[$i][1];
    } else {
      printf OUTFILE "$sorted[$i][0] => %6.2f\n", $sorted[$i][1];
    }
  }
  @report = ();
  $rep_index = 0;
  $wrtsig = 0;
}


