#--------------------------------------------------------------------------
# Title: ApplicationLauncher Application Initialisation  (Tcl/Tk lib file)
#--------------------------------------------------------------------------
set ApplicationLauncher_AppVersion 1.01
# Created:       05 Jul 2009
# Last modified: 05 Jul 2009
#--------------------------------------------------------------------------
# Requires: Tcl/Tk 8.3 (or later)
#--------------------------------------------------------------------------
#  Copyright (C) 2003-2009 Agilent Technologies
#
#  All copies of this program, whether in whole or in part, and whether
#  modified or not, must display this and all other embedded copyright
#  and ownership notices in full.
#--------------------------------------------------------------------------
# Sample software conditions of use:
#
#  This software is provided as an example. It is distributed in the hope
#  that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#--------------------------------------------------------------------------

#==========================================================================
#   N a m e s p a c e s
#==========================================================================

namespace eval ::AppIni {
    # indicates that this file has been sourced already
}

namespace eval ::AppGui {
    # declare state variables
    variable appWidgets     ;# array - application GUI widget handles
}

namespace eval ::App {
    global tcl_platform

    # declare state variables
    variable appData    ;# array - application specific parameters

    array set appData [list                                     \
            agtQtImageDir        ""                             \
            agtRootDir           ""                             \
            agtSessionType       "RouterTester900"              \
            agtSessionVersion    "0.0.0.0"                      \
            scriptDir           [file dirname [info script]]    \
            scriptTitle         [lindex [split [file tail [info script]] .] 0] \
            wishCommand         \"[info nameofexecutable]\"     \
            ]
}

#==========================================================================
#   E x t e r n a l   P a c k a g e s
#==========================================================================

if { $::tcl_platform(platform) == "windows" } {
    package require registry
}

#==========================================================================
#   E x t e r n a l   S o u r c e
#==========================================================================

# none

#==========================================================================
#   P r o c e d u r e s
#==========================================================================

#--------------------------------------------------------------------------
# GetAgtRoot { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   rootDir - discovered path to the root N2X installation directory
#
# Purpose:
#   Discover the root path to the N2X installation directory on any
#   operating system.
#--------------------------------------------------------------------------
proc AppIni::GetAgtRoot { } {
    global tcl_platform
    variable ::App::appData

    set rootDir ""
    switch $tcl_platform(platform) {
        unix {
            set rootDir "~/Agilent/N2X"
        }

        windows {
            set key   HKEY_LOCAL_MACHINE\\SOFTWARE\\Agilent\\N2X\\Platform
            set vName RootDirectory
            if { [catch { registry get $key $vName } rootDir] } {
                error "Unable to retrieve N2X root from Windows Registry using key='$key', value='$vName'\n\nReported error: $rootDir\n"
                set rootDir "c:/Program Files/Agilent/N2X"
            }

            # Convert all backslashes to forward slashes
            set rootDir [string map {\\ /} $rootDir]

            # Strip down to the N2X root path
            set rootDir [split $rootDir /]
            if { ([set n2xIndex [lsearch -exact $rootDir N2X]] == -1) &&
                 ([set n2xIndex [lsearch -exact $rootDir Platform]] == -1) } {
                error "Invalid or unrecognised N2X platform installation.\n\nRoot path = '[join $rootDir /]'\n\nCannot continue"
                exit -1
            }
            set rootDir [join [lrange $rootDir 0 $n2xIndex] "/"]
        }
    }

    return $rootDir
}

#--------------------------------------------------------------------------
# GetQtRoot { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   rootDir - the QuickTest root directory
#
# Purpose:
#   Determine the QuickTest root directory path based upon the operating
#   system (Windows or Unix).  If the path is not found an appropriate
#   default value is returned.
#--------------------------------------------------------------------------
proc AppIni::GetQtRoot { } {
    global tcl_platform

    set rootDir ""
    switch $tcl_platform(platform) {
        unix {
            set rootDir "~/Agilent/N2X/QuickTest"
        }

        windows {
            set key   HKEY_LOCAL_MACHINE\\SOFTWARE\\Agilent\\N2X\\QuickTest
            set vName RootDirectory
            if [catch { registry get $key $vName } rootDir] {
                error "Unable to retrieve QuickTest root from Windows Registry using key='$key', value='$vName'\n\nReported error: $rootDir\n"
                set rootDir "c:/Program Files/Agilent/N2X/QuickTest"
            }

            # Convert all backslashes to forward slashes
            set rootDir [string map {\\ /} $rootDir]
        }
    }

    return $rootDir
}

#--------------------------------------------------------------------------
# InitialiseStdDefaults { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Initialise the standard state variables to their default values.
#--------------------------------------------------------------------------
proc AppIni::InitialiseStdDefaults { } {
    variable ::App:appData

    set appData(agtRootDir)     [AppIni::GetAgtRoot]
    set appData(qtRootDir)      [AppIni::GetQtRoot]
    set appData(agtQtImageDir)  $appData(qtRootDir)/images
}

#==========================================================================
#   M a i n
#==========================================================================

#--<end>--
