#!/usr/bin/tclsh8.3
#--------------------------------------------------------------------------
# Title:    SmallApplicationLauncher
#--------------------------------------------------------------------------
# Version:       (be sure to update scriptVersion variable below)
# Author:        Kenneth Green
# Created:       03 Jul 2009
# Last modified: 03 Jul 2009
#--------------------------------------------------------------------------
# Requires: Tcl/Tk 8.3 (or later)
#--------------------------------------------------------------------------
#  Copyright (C) 2009 Agilent Technologies
#
#  All copies of this program, whether in whole or in part, and whether
#  modified or not, must display this and all other embedded copyright
#  and ownership notices in full.
#--------------------------------------------------------------------------
# Synopsis:
#
#   A GUI tool for launching SmallApps from within the QuickTest environment
#
#--------------------------------------------------------------------------

# Ensure we are running under wish
set isTk [info exists ::tk_version]
if { !$isTk } {
    puts "<ERROR> This is a GUI script and must be run under wish83"
    exit
}

#==========================================================================
#   N a m e s p a c e   D e c l a r a t i o n
#==========================================================================
namespace eval App {
    #
    # Default parameter values
    #
    variable appData            ;# array - parameters
    variable appWidgets         ;# array - widgets

    array set appData [list \
            debugLevel              0                               \
                                                                    \
            appsDirectory           "[file join [file dirname [info script]] apps]" \
                                                                    \
            helpSelectN2xSession    "help/CTSTestManager.chm::/topics/Select_N2X_Session_dialog_box.htm" \
                                                                    \
            imageDirectory          "[file join [file dirname [info script]] images]" \
                                                                    \
            logFileId               ""                              \
                                                                    \
            n2xSessionInfoModel     ""                              \
            n2xSessionTag           "<-- click to select a session" \
                                                                    \
            scriptDirectory         [file dirname   [info script]]  \
            scriptSuffix            [file extension [info script]]  \
            scriptTitle             [lindex [split [file tail [info script]] .] 0] \
            scriptVersion           0.01                             \
                                                                    \
            statusText              "Hello"                         \
    ]

    array set appWidgets {}
}

#==========================================================================
#   G l o b a l   V a r i a b l e s
#==========================================================================

#==========================================================================
#   E x t e r n a l   P a c k a g e s
#==========================================================================

if { [llength [namespace children :: AgtQcl]] == 0 } {
    namespace eval :: {
        lappend auto_path "c:/program files/agilent/n2x/quicktest/lib"
        puts "Loaded AgtQcl [package require AgtQcl]"
    }

    AgtQclLoadClass -group ALL
}

#==========================================================================
#   S o u r c e   E x t e r n a l   L i b r a r y   F i l e s
#==========================================================================

# Local utility files
namespace eval :: {
    set libFileList {
        CLauncherTree
    }

    set suffix [file extension [info script]] ;# Allow for .tcl and .tbc
    foreach libFile $libFileList {
        set libFile [file join [file dirname [info script]] ${libFile}${suffix}]
        if [catch {source $libFile} errMsg] {
            AgtTsuShowMessage FATAL_ABORT "Failed to source file: '$libFile'\n\nReported error: $errMsg"
        }
        puts "Sourced: $libFile"
    }
}

#==========================================================================
#   P r o c e d u r e s
#==========================================================================

#--------------------------------------------------------------------------
# CbItemAction { theTree theNodeRecord }
#--------------------------------------------------------------------------
# Parameters:
#   theTree - widget path to the CLauncherTree
#   theNodeRecord - record of the form: {-id <nodeid> -name <nodename> -data <nodedata>}
#
# Returns:
#   nothing
#
# Purpose:
#   Respond to user action
#--------------------------------------------------------------------------
proc CbItemAction { theTree theNodeRecord } {
    variable ::App::appData
    variable ::App::appWidgets

    set procName [AgtTsuProcedureName]
    CbLaunchApp
}

#--------------------------------------------------------------------------
# CbItemSelected { theTree theNodeRecord }
#--------------------------------------------------------------------------
# Parameters:
#   theTree - widget path to the CLauncherTree
#   theNodeRecord - record of the form: {-id <nodeid> -name <nodename> -data <nodedata>} and <nodedata> = {-type <type> -directory <dir>}
#
# Returns:
#   nothing
#
# Purpose:
#   Respond to user action
#--------------------------------------------------------------------------
proc CbItemSelected { theTree theNodeRecord } {
    variable ::App::appData
    variable ::App::appWidgets

    set procName [AgtTsuProcedureName]
    array set nodeInfo $theNodeRecord
    array set nodeDataInfo $nodeInfo(-data)

    AgtTsuTraceMessage "$procName - item: $nodeDataInfo(-type)/$nodeInfo(-name)"
    if [string equal $nodeDataInfo(-type) APPLICATION] {
        ShowStatusMessage "Selected Application: $nodeInfo(-name)"
        $appWidgets(mainFrame) setmenustate Launch normal
        $appWidgets(bbxRunGroup) itemconfigure 0 -state normal
    } else {
        $appWidgets(mainFrame) setmenustate Launch disabled
        $appWidgets(bbxRunGroup) itemconfigure 0 -state disabled
        ShowStatusMessage ""
    }
}

#--------------------------------------------------------------------------
# CbLaunchApp { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Respond to user action
#--------------------------------------------------------------------------
proc CbLaunchApp { } {
    variable ::App::appData
    variable ::App::appWidgets

    set procName [AgtTsuProcedureName]
    ShowStatusMessage "Launching the Application..."

    $appWidgets(bbxRunGroup) itemconfigure 0 -state disabled

    #
    # Get selected Application directory
    #
    array set appInfo [$appWidgets(theTree) getSelectedItemApplicationInfo]
    AgtTsuTraceMessage "$procName - [AgtTsuDumpArray appInfo]"
    if { ![info exists appInfo(-directory)] } {
        AgtTsuTraceMessage "$procName - Application directory not found: '$appInfo(-directory)'"
        ShowStatusMessage "Nothing launchable is selected."
        $appWidgets(bbxRunGroup) itemconfigure 0 -state normal
        return
    }
    set appFile [file join $appInfo(-directory) $appInfo(-name)]
    if { [file exists ${appFile}.tbc] } {
        set appFile ${appFile}.tbc
    } elseif { [file exists ${appFile}.tcl] } {
        set appFile ${appFile}.tcl
    } else {
        AgtTsuShowMessage ERROR "Unable to locate executable application file $appFile.xxx"
        return 0
    }

    #
    # Run the Application:
    #
    # wish8x <application>.tcl -- [-D level\] \[-server hostname\] \[-sessionhandle handle\] \[-sessiontype type\]" \
    set cmd        [info nameofexecutable]
    set argList    [list \
                        -server        [$appData(n2xSessionInfoModel) cget -serverhostname] \
                        -sessionhandle [$appData(n2xSessionInfoModel) cget -handle] \
                        -sessiontype   [$appData(n2xSessionInfoModel) cget -type]   \
                   ]

    AgtTsuTraceMessage "$procName - cmd:        [info nameofexecutable]"
    AgtTsuTraceMessage "$procName - scriptName: $appFile"
    AgtTsuTraceMessage "$procName - argList:    $argList"
    if [catch {eval exec {[info nameofexecutable]} {$appFile} -- $argList &} errMsg] {
        AgtTsuShowMessage ERROR "$procName - Failed to run application $appInfo(-name).\nReported error: $errMsg"
    }

    $appWidgets(bbxRunGroup) itemconfigure 0 -state normal
    ShowStatusMessage ""
    return
}

#--------------------------------------------------------------------------
# CbProperties { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Respond to user action
#--------------------------------------------------------------------------
proc CbProperties { } {
    variable ::App::appData
    variable ::App::appWidgets

    set procName [AgtTsuProcedureName]

    if [winfo exists .dlgLp] {
        # Dialog already open so do nothing
        return
    }

    ShowStatusMessage "Get N2X session properties"
    AgtTsuTraceMessage "$procName - session info: [$appData(n2xSessionInfoModel) get optionrecord]"

    set serverHostname [$appData(n2xSessionInfoModel) cget -serverhostname]
    set sessionHandle  [$appData(n2xSessionInfoModel) cget -handle]

    # Convert all backslashes to forward slashes
    set helpLink [file join [string map {\\ /} $appData(scriptDirectory)] $appData(helpSelectN2xSession)]

    set dlg [CN2xSelectSessionDialog .dlgLp $serverHostname -helplink $helpLink]
    if [string equal [$dlg draw $serverHostname $sessionHandle] OK] {
        AgtTsuTraceMessage "$procName - User accepted the updates"
        if [$dlg updateSessionInfoModelFromSelection $appData(n2xSessionInfoModel)] {
            if ![$appData(n2xSessionInfoModel) isVersionPresentOnLocalHost] {
                set theHost    [$appData(n2xSessionInfoModel) cget -serverhostname]
                set theType    [$appData(n2xSessionInfoModel) cget -type]
                set theVersion [$appData(n2xSessionInfoModel) cget -version]
                set msg [join [list \
                    "Controller PC:\t$theHost" \
                    "Session type:\t$theType" \
                    "Session version:\t$theVersion" \
                    "" \
                    "Version '$theVersion' of the $theType software is not installed on this PC." \
                    "Ensure that the same software is installed both locally and on the remote controller." \
                    "" \
                    "Versions installed on this PC are:" \
                ] \n]
                foreach versionName [$appData(n2xSessionInfoModel) listLocalVersions] {
                    set msg "${msg}\n    $versionName"
                }

                AgtTsuShowMessage WARNING $msg -title "Warning - Mismatched local and remote software versions."
            } else {
                set appData(n2xSessionTag) "[$appData(n2xSessionInfoModel) cget -label] ([$appData(n2xSessionInfoModel) cget -handle])"
                $appWidgets(vluSession) configure -bg green -fg black
            }
        }
    } else {
        AgtTsuTraceMessage "$procName - User cancelled operation"
    }
    AgtTsuTraceMessage "$procName - session info: [$appData(n2xSessionInfoModel) get optionrecord]"
    wm title . "Application Launcher (on [$appData(n2xSessionInfoModel) cget -serverhostname])"
    destroy $dlg

    ShowStatusMessage "N2X session properties updated"
    return
}

#--------------------------------------------------------------------------
# CreateMainMenu { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   menuDefinition - the menu definition as a list of submenu records
#
# Purpose:
#  Create the main window menu definition
#--------------------------------------------------------------------------
proc CreateMainMenu {  } {
    variable ::App::appData
    variable ::App::appWidgets

    set appWidgets(viewFilterState) NONE
    set appWidgets(viewRbState)     ACTIVE
    # Define default menu
    #   (a list of menu definitions, each of the form: {name tags id tearoff entryList}
    set mainMenu [list   \
        [list                   \
            "&File"             \
            all                 \
            file                \
            0                   \
            [list               \
                separator \
                [list command "E&xit" {} "Exit script"           {Ctrl x} -command ExitScript ] \
            ]                   \
        ]                       \
        [list                   \
            "&Actions"          \
            all                 \
            file                \
            0                   \
            [list               \
                [list command "&Select N2X session" {} "Select N2X session" {} -command {CbProperties} ] \
                [list command "&Launch selected Application" {Launch} "Launch the selected Application"  {} -command {CbLaunchApp} -state disabled] \
            ]                   \
        ]                       \
        [list                   \
            "&Help"             \
            all                 \
            help                \
            0                   \
            [list               \
                [list command "&Contents" {} "Show help contents"   {}   -command {HelpTopic Contents}] \
                [list command "&Topics"   {} "Show main help topic" {F1} -command {HelpTopic Topics}] \
                separator       \
                [list command "&About"  {} "About this tool" {}   -command HelpAbout  ] \
            ]                   \
        ]                       \
    ]
    return $mainMenu
}

#--------------------------------------------------------------------------
# CreateMainToolbar { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Construct the main application toolbar. Note that separators between
#   button groups are created using three Separator widgets. One as a 1-bit
#   spacer and two (one dark grey and one white) to form the etched line.
#--------------------------------------------------------------------------
proc CreateMainToolbar { } {
    variable ::App::appData
    variable ::App::appWidgets

    set thisDir [file dirname [info script]]
    set imgDir  $thisDir
    set tbrMain [$appWidgets(mainFrame) addtoolbar]

    #
    # Run
    #
    set grpRun [ButtonBox $tbrMain.bbox1 -spacing 0 -padx 1 -pady 1]
    $grpRun add -image [Bitmap::get "$appData(imageDirectory)/launch_txt.gif"] \
        -highlightthickness 0 -relief link -takefocus 0 -borderwidth 1 -padx 1 -pady 1 \
        -command {CbLaunchApp} \
        -helptext "Launch the selected Application" \
        -state disabled

if {0} {
    #
    # Help
    #
    set grpHelp [ButtonBox $tbrMain.bbox2 -spacing 0 -padx 1 -pady 1]
    $grpHelp add -image [Bitmap::get "$appData(imageDirectory)/help_blue.gif"] \
        -highlightthickness 0  -relief link -takefocus 0 -borderwidth 1 -padx 1 -pady 1 \
        -command {HelpTopic Topics} \
        -helptext "Open the online help"
}
    #
    # Session info
    #
    #
    # Edit
    #
    set grpEdit [ButtonBox $tbrMain.bbox3 -spacing 0 -padx 1 -pady 1]
    $grpEdit add -text "N2X Session" \
        -highlightthickness 0  -relief link -takefocus 0 -borderwidth 1 -padx 1 -pady 1 \
        -command {CbProperties} \
        -helptext "Select N2X session"
    set vluSession  [Label $tbrMain.v1 -width 25 -textvariable ::App::appData(n2xSessionTag) -relief sunken -anchor w]
    $vluSession configure -bg red3 -fg white
    bind $vluSession <Button-1> CbProperties

    # Pack the widgets
    pack $grpRun -side left -anchor w -padx 2
    pack $vluSession $grpEdit -side right -padx 2

    # Remember some widgets for later
    set appWidgets(vluSession) $vluSession
    set appWidgets(bbxRunGroup) $grpRun
}


#--------------------------------------------------------------------------
# CreateMainWindow { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   ** Library Internal/Private function - not for client use **
#
#  Create and display the main window
#--------------------------------------------------------------------------
proc CreateMainWindow { args } {
    global   tk_patchLevel
    variable ::App::appData
    variable ::App::appWidgets

    wm withdraw .   ;# hide default window
    wm protocol . WM_DELETE_WINDOW ExitScript
    #
    # Configure window title-bar (title and icon)
    #
    set iconFilename "$appData(imageDirectory)/DarkBlueBall_Cog.ico"
    if { [expr $::tk_version > 8.2] && [file exists $iconFilename] } {
        wm iconbitmap . $iconFilename
    }
    wm title . "Application Launcher (on [$appData(n2xSessionInfoModel) cget -serverhostname])"

    #
    # Create our main window frame
    #
    set mfMain [MainFrame .mainFrame \
                  -menu         [join [CreateMainMenu] "\n"] \
                  -textvariable ::App::appData(statusText)]
    set appWidgets(mainFrame) $mfMain
    #
    # Main toolber
    #
    CreateMainToolbar
    CreateTraceDialog

    # - top level containers
    set fMain   [$mfMain getframe]
    set theTree [CLauncherTree $fMain.t -appsdirectory $appData(appsDirectory) -imagedirectory $appData(imageDirectory) -showresultsets 0 -actioncallback CbItemAction -selectioncallback CbItemSelected]
    set appWidgets(theTree) $theTree

    #
    # Pack primary objects
    #
    pack $appWidgets(mainFrame) -fill both -expand yes
    # - top level frames
    pack $theTree -side top -fill both -expand yes

    #
    # Additional key-bindings
    #
    bind . <Control-t> {ShowTrace}

    #
    # Main window icon
    #  - Need to catch errors on Unix
    #
    set iconFilename [file join $appData(imageDirectory) GreenBall_Cog.ico]
    catch { wm iconbitmap . $iconFilename }

    #
    # Refresh the world
    wm minsize  . 300 100
    wm geometry . 400x400
    wm deiconify .
    update idletasks
}

#--------------------------------------------------------------------------
# CreateTraceDialog { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#  Create, but don't display the Trace dialog
#--------------------------------------------------------------------------
proc CreateTraceDialog { } {
    variable ::App::appData
    variable ::App::appWidgets

    set parent [$appWidgets(mainFrame) getframe]
    set appWidgets(dlgTrace) [CTraceDialog $parent.dlgCtd]

    AgtTsuTraceMessage "$::App::appData(scriptTitle) $::App::appData(scriptVersion)"
    AgtTsuTraceMessage "========================================================="
}

#--------------------------------------------------------------------------
#  ExitScript { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Tidy up and close down the script.
#--------------------------------------------------------------------------
proc ExitScript { } {
    AgtTsuTraceMessage "\n---< All done >---\n";
    if { $::App::appData(logFileId) != "" } {
        close $::App::appData(logFileId)
    }
    exit
}

#--------------------------------------------------------------------------
# HelpAbout { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#  Display the HelpAbout dialog.
#
#--------------------------------------------------------------------------
proc HelpAbout { } {
    variable ::App::appData

    set msg "Application:\t Agilent N2X Mini Application Launcher"
    set msg "$msg\nVersion:\t\t $appData(scriptVersion)"
    AgtTsuShowMessage INFO $msg "About"
}

#--------------------------------------------------------------------------
# HelpTopic { topic }
#--------------------------------------------------------------------------
# Parameters:
#   topic - topic tag
#
# Returns:
#   nothing
#
# Purpose:
#  Display the specified help topic
#--------------------------------------------------------------------------
proc HelpTopic { topic } {
    variable ::App::appData

    switch $topic {
        Contents {
            set helpLink "help/AppLauncherMaster.chm::/topics/Introduction.htm"
        }

        Topics -
        default {
            set helpLink "help/AppLauncherManager.chm::/topics/Application_Launcher_window.htm" \
        }
    }

    # Convert all backslashes to forward slashes
    set helpLink [file join [string map {\\ /} $appData(scriptDirectory)] $helpLink]

    catch {exec hh.exe $helpLink &}
}

#--------------------------------------------------------------------------
# ShowBusyMessage { msg [title]}
#--------------------------------------------------------------------------
# Parameters:
#   msg        - the message string
#   [title]    - (optional) window title
#
# Returns:
#   dlg - the dialog widget handle
#
# Purpose:
#   Display a non-modal dialog with no buttons displaying the specified
#   message. The caller is responsible for destroying the dialog when
#   appropriate.
#--------------------------------------------------------------------------
proc ShowBusyMessage { msg args } {
    variable ::App::appData
    variable ::App::appWidgets

    set dlgTitle [lindex $args 0]
    if { $dlgTitle == ""  } {
        set dlgTitle "Please wait..."
    }
    set msg      [textutil::adjust $msg -justify center -length 40 -full 1]
    set ticks    [clock clicks]
    set myParent $appWidgets(mainFrame)
    set dlg      [Dialog .dlgBusy$ticks -parent $myParent -title $dlgTitle -modal none]
    set f        [$dlg getframe]
    set lblMsg   [Label $f.l1 -text $msg]
    pack $lblMsg -side top -fill both -expand yes -padx 10 -pady 10
    $dlg draw
    update
    after 1000 ;# ensure minimum on-screen duration for visibility

    return $dlg
}

#--------------------------------------------------------------------------
# ShowStatusMessage { msg [colour] }
#--------------------------------------------------------------------------
# Parameters:
#   msg     - the text to be displayed
#  [colour] - (optional) text colour (default=black)
#
# Returns:
#   nothing
#
# Purpose:
#   Display a message in the status line
#--------------------------------------------------------------------------
proc ShowStatusMessage { msg args} {
    variable ::App::appData
    variable ::App::appWidgets

    set appData(statusText) $msg
    update
}

#--------------------------------------------------------------------------
# ShowTrace { }
#--------------------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Display the Trace window
#--------------------------------------------------------------------------
proc ShowTrace {} {
    variable ::App::appData
    variable ::App::appWidgets

    $appWidgets(dlgTrace) draw
}

#==========================================================================
#   M a i n (script actually starts executing here)
#==========================================================================

set ::App::appData(n2xSessionInfoModel) [CN2xSessionInfoModel %AUTO% -handle 0]
CreateMainWindow

wm deiconify .

#==========================================================================
#--<end>--
#==========================================================================
