#--------------------------------------------------------------------------
# Title:    CLauncherTree (widget) Class
#--------------------------------------------------------------------------
# Author:        Kenneth Green
# Created:       05 Mar 2005
# Last modified: 03 Jul 2009
#--------------------------------------------------------------------------
# Requires: Tcl/Tk 8.3 (or later)
#--------------------------------------------------------------------------
#  Copyright (C) 2009 Agilent Technologies
#
#  All copies of this program, whether in whole or in part, and whether
#  modified or not, must display this and all other embedded copyright
#  and ownership notices in full.
#--------------------------------------------------------------------------
# Sample software conditions of use:
#
#  This software is provided as an example. It is distributed in the hope
#  that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#--------------------------------------------------------------------------
# Overview
# ========
#   This file forms part of the Small Application Launcher application
#   and cannot be used standalone.
#
# Dependencies
# ============
#   - Packages: AgtQcl
#   - Files: none
#
# Globals Used that clients must declare
# ======================================
#   None
#
#--------------------------------------------------------------------------
# Organisation:
#   Ordered alphabetically in the following groups
#       - namespace declaration
#       - global variables
#       - class definition (snit::type declaration)
#           - public  Type methods
#           - public  Instance methods
#           - private Type methods
#           - private Instance methods
#           - private procs
#       - auto-initialisation
#----------------------------------------------------------------------------

# Ensure we are running under wish
if { ![info exists ::tk_version] } {
    puts "<ERROR> This is a GUI script and must be run under Tk (wish)"
    exit
}

#==========================================================================
#   G l o b a l   V a r i a b l e s
#==========================================================================

#==========================================================================
#   N a m e s p a c e   D e c l a r a t i o n
#==========================================================================

#==========================================================================
#   E x t e r n a l   P a c k a g e s
#==========================================================================

namespace eval ::CLauncherTreeInit {
    variable pkg
    variable className

    namespace eval :: {
        foreach CLauncherTreeInit::pkg { } {
            if { [llength [namespace children :: $CLauncherTreeInit::pkg]] == 0 } {
                set result [package require $CLauncherTreeInit::pkg]
                puts "Loaded pkg: $CLauncherTreeInit::pkg $result"
            }
        }
    }
}

namespace delete ::CLauncherTreeInit

#==========================================================================
#   C l a s s   D e f i n i t i o n
#==========================================================================
::snit::widget CLauncherTree {
    component mTree

    # options
    option -actioncallback      -default "default"
    option -appsdirectory       -default [file join [file dirname [info script]] apps]
    option -imagedirectory      -default [file join [file dirname [info script]] images]
    option -selectioncallback   -default "default"
    option -showresultsets      -default 0

    # type variables
    typevariable tImages -array  {}

    # instance variables
    variable mAllAppsNodeId     0
    variable mNodeNum           0
    variable mNodeSelectedCb    "default"


    #================================================================
    # Constructor
    #================================================================
    constructor {args} {
        set procName [AgtTsuProcedureName]

        # Apply any options passed at creation time.
        $self configurelist $args
        AgtTsuTraceMessage "$procName - options: [AgtTsuDumpArray options]"

        # Initialise variables
        $type LoadImages $options(-imagedirectory)

        # Create and pack the tree widget
        set sw   [ScrolledWindow $win.sw -relief sunken -borderwidth 2]
        install mTree using Tree $sw.tree \
            -relief flat -borderwidth 0 -width 20 -highlightthickness 0\
            -padx   32 \
            -redraw 0 -dropenabled 0 -dragenabled 0    \
            -opencmd   [mymethod ToggleNode EXPAND]    \
            -closecmd  [mymethod ToggleNode COLLAPSE]
        $sw setwidget $mTree
        pack $sw -side top -fill both -expand yes -padx 2 -pady 2

        # key and event bindings
        bind $win <Key-Return>               [mymethod NodeSelected ClickText ]
        $mTree bindText    <Button-1>        [mymethod NodeSelected ClickText ]
        $mTree bindImage   <Button-1>        [mymethod NodeSelected ClickImage]
        $mTree bindText    <Double-Button-1> [mymethod NodeSelected DblClickImage]
        $mTree bindImage   <Double-Button-1> [mymethod NodeSelected DblClickImage]

        $self refresh
    }

    #================================================================
    # Destructor
    #================================================================
    destructor {
    }

    #================================================================
    # Type methods (by convention begin with lowerCase)
    #================================================================

    #================================================================
    # Instance methods (by convention begin with lowerCase)
    #================================================================

    #--------------------------------------------------------------------------
    #  getNodeRecord { nodeId }
    #--------------------------------------------------------------------------
    # Parameters:
    #   nodeId - the tree nodeId of interest
    #
    # Returns:
    #   nodeRcd - a list of the form: nodeId nodeName usrData
    #
    # Purpose:
    #   Compose and return the Node Record for the specified nodeId
    #--------------------------------------------------------------------------
    method getNodeRecord { nodeId } {

        set tree      $mTree
        set nodeName  [$tree itemcget $nodeId -text]
        set nodeData  [$self GetNodeUserData $nodeId]
        AgtTsuTraceMessage "[AgtTsuProcedureName] - nodeId = '$nodeId', nodeName = '$nodeName', nodeData = '$nodeData'"
        set nodeRcd   [list -id $nodeId -name $nodeName -data $nodeData]

        return $nodeRcd
    }

    #--------------------------------------------------------------------------
    # getNodeUserDataField { nodeId fieldName }
    #--------------------------------------------------------------------------
    # Parameters:
    #   nodeId    - the node to be acted upon
    #   fieldName - the name of the field to be retrieved (eg -type -state etc)
    #
    # Returns:
    #   fieldValue - the value of the specified field or an empty string if
    #                the field doesn't exist
    #
    # Purpose:
    #   Get the value of the specified user-data field for the specified node
    #--------------------------------------------------------------------------
    method getNodeUserDataField { nodeId fieldName } {

        set procName   [AgtTsuProcedureName]
        set fieldValue ""

        set usrData [$self GetNodeUserData $nodeId]
        array set nodeInfo $usrData

        if [catch {set fieldValue $nodeInfo($fieldName)} errMsg] {
            AgtTsuShowMessage ERROR "$procName - Invalid field: $fieldName\nReported error: $errMsg"
            set fieldValue ""
        }

        return $fieldValue
    }

    #--------------------------------------------------------------------------
    #  getSelectedItemApplicationInfo { }
    #--------------------------------------------------------------------------
    # Parameters:
    #   none
    #
    # Returns:
    #   ApplicationInfo - record of the form: {-name <app_name> -directory <app_directory>}
    #                     or an empty list of nothing is selected
    #
    # Purpose:
    #   Get the Application info for the selected item.
    #--------------------------------------------------------------------------
    method getSelectedItemApplicationInfo { } {

        set procName [AgtTsuProcedureName]
        set applicationInfo {}

        set nodeId [$mTree selection get]
        if ![string equal $nodeId ""] {
            array set nodeInfo [$self getNodeRecord $nodeId]
AgtTsuTraceMessage "$procName - nodeInfo [AgtTsuDumpArray nodeInfo]"
            array set userInfo [$self GetNodeUserData $nodeId]
AgtTsuTraceMessage "$procName - userInfo [AgtTsuDumpArray userInfo]"
            switch $userInfo(-type) {

                APPLICATION {
                    set applicationInfo [list -name $nodeInfo(-name) -directory $userInfo(-directory)]
                }

                default {
                    AgtTsuShowMessage WARNING "Please select an Application and try again."
                }
            }
        }
        return $applicationInfo
    }

    #--------------------------------------------------------------------------
    #  refresh { }
    #--------------------------------------------------------------------------
    # Parameters:
    #   none
    #
    # Returns:
    #   nothing
    #
    # Purpose:
    #   Delete all nodes in the tree then re-initialise the tree widget by
    #   iterating through the Applications directory.
    #--------------------------------------------------------------------------
    method refresh { args } {

        set procName [AgtTsuProcedureName]
        set tree     $mTree

        #
        # Delete all the tree contents and create the root node
        #
        $tree delete [$tree nodes root]
        set mNodeNum  0

        #
        # Work down from the root of the Application collection
        #
        set nodeData       [list -type COLLECTION -directory "$options(-appsdirectory)"]
        set mAllAppsNodeId [$self AddNode root Applications FOLDER $nodeData] ;# Add a top-level root node

        #
        # Automatically expand the top-level node
        #
        $tree opentree $mAllAppsNodeId  1
        if { $options(-showresultsets) != 0 } {
            $tree opentree $mAllResultsNodeId 0
        }
        $tree configure -redraw 1
    }

    #================================================================
    # Delegation
    #================================================================

    # Pass all other methods and options to the hull
    # that the remaining behavior is as expected.
    delegate method itemcget  to mTree
    delegate method parent    to mTree
    delegate method see       to mTree
    delegate method selection to mTree

    delegate method * to hull
    delegate option * to hull

    #================================================================
    # Private Type methods (by convention begin with UpperCase)
    #================================================================

    #--------------------------------------------------------------------------
    #  GetDirectoryListing { globPattern ?-filter filter? ?-fullpath bool? }
    #--------------------------------------------------------------------------
    # Parameters:
    #   globPattern - the pathname/pattern to use for 'globbing'
    #   [-filter]    - (optional) select one of: isDirectory, isDirectoryOrShortcut, isFile
    #   [-fullpath]  - (optional) select one of: 1, 0 (default: 1)
    #
    # Returns:
    #   resultList  - a sorted list of directory and/or filenames. Will be
    #                 an empty list of nothing matched the pattern+filter.
    #
    # Purpose:
    #   Get a directory listing using the specified pattern and filter.
    #   Always returns a (possibly empty) sorted list with an directory
    #   entries first, followed by files. If -fullpath is set to 1 then
    #   all entries will be prefixed with the full path as defined by the
    #   glob pattern
    #--------------------------------------------------------------------------
    typemethod GetDirectoryListing { globPattern args } {

        set procName    [AgtTsuProcedureName]
        set filter      ""
        set useFullPath 1
        set resultList  {}

        # Process optional parameters
        array set opts $args
        if [info exists opts(-filter)] {
            set filter $opts(-filter)
            unset opts(-filter)
            set validFilters {isDirectory isDirectoryOrShortcut isFile}
            if { ($filter != "") && ([lsearch -exact $validFilters $filter] == -1) } {
                AgtTsuShowMessage FATAL_ABORT "$procName - Invalid filter = $filter.\n\n\tMust be one of: [join $validFilters ",  "]"
            }
        }

        if [info exists opts(-fullpath)] {
            set useFullPath $opts(-fullpath)
            unset opts(-fullpath)
        }

        if [llength [array names opts]] {
            AgtTsuShowMessage FATAL "$procName - Invalid options: [array names opts]"
        }

        # search for directory contents using the specified glob pattern
        if [catch {glob $globPattern} allList] {
            AgtTsuDebugMsg 1 "No directory listings found using pattern:\n\n$globPattern"
        } else {
            AgtTsuDebugMsg 1 "allList = $allList"
            set dirList      {}     ;# list of full dir path
            set fileList     {}     ;# list of filenames (no path)
            set shortcutList {}     ;# list of the contents of shortcut files
            set resultList   {}

            # create separate, sorted lists of directories, files and shortcuts
            foreach fullPath $allList {
                if { $useFullPath } {
                    set theName $fullPath
                } else {
                    set theName [file tail $fullPath]
                }
                if { [file isdirectory $fullPath] } {
                    lappend dirList $theName
                }
                if { [file isfile $fullPath] } {
                    lappend fileList $theName
                    if { [string first ".shortcut.txt" $theName] != -1 } {
                        set shortcutValue [$self GetShortcut $fullPath]
                        lappend shortcutList ${shortcutValue}.shortcut.txt
                    }
                }
            }
            set dirList      [lsort $dirList]
            set fileList     [lsort $fileList]
            set shortcutList [lsort $shortcutList]
            AgtTsuDebugMsg 1 "dirList = $dirList\nfileList = $fileList\nshortcutList = $shortcutList;"

            # compose result based on the filter setting
            switch $filter {
                isDirectoryOrShortcut {
                    set resultList [lsort [concat $dirList $shortcutList]]
                }
                isDirectory           { set resultList $dirList  }
                isFile                { set resultList $fileList }
                default               { set resultList [concat $dirList $fileList] }
            }
        }
        return $resultList
    }

    typemethod LoadImages { imageDirectory } {

        if { [llength [array names tImages]] > 0 } {
            # Images already loaded
            return
        }

        # Load up all the images from the image directory
        array set tImages [list \
            COG                 [Bitmap::get $imageDirectory/GreenCog.gif]      \
            EXE_FILE            [Bitmap::get $imageDirectory/exe_file.gif]      \
            FOLDER              [Bitmap::get $imageDirectory/folder.gif]        \
        ]
    }

    #================================================================
    # Private Instance methods (by convention begin with UpperCase)
    #================================================================

    #--------------------------------------------------------------------------
    # AddNode { parent nodeLabel nodeImage usrData }
    #--------------------------------------------------------------------------
    # Parameters:
    #   parent      - the parent node to which the new node is to be added
    #   nodeLabel   - text label to be displayed for the node
    #   nodeImage   - image type: FILE, FOLDER, EXE_FILE, RESULT,<outcome>
    #   usrData     - user data to be stored in the node: {-type <type> -directory <dir>}
    #
    # Returns:
    #   nodeId - the tree node ID of the newly added node
    #
    # Purpose:
    #   Append a new node to the specified parent. Set the node data to
    #   a record of the form: { mode usrData }
    #--------------------------------------------------------------------------
    method AddNode { parent nodeLabel nodeImage usrData } {

        set procName [AgtTsuProcedureName]

        AgtTsuDebugMsg 1 "$procName - parent = $parent, label = $nodeLabel"
        set tree $mTree
        set crossMode auto
        set nodeData  $usrData

        # add the node
        set nodeId  n_$mNodeNum
AgtTsuTraceMessage "$procName - insert end $parent $nodeId -text '$nodeLabel' -image '$nodeImage' -data '$nodeData'"
        $tree insert end $parent $nodeId    \
            -text      $nodeLabel           \
            -image     $tImages($nodeImage) \
            -drawcross $crossMode           \
            -data      $nodeData
        incr mNodeNum

        #
        # Look for sub-directories
        #
        array set nodeInfo $nodeData
        set globPattern    [string map {\\ /} "$nodeInfo(-directory)/*"]
        set subDirList     [lsort -dictionary [$type GetDirectoryListing $globPattern -filter isDirectoryOrShortcut]]
        AgtTsuDebugMsg 1 "$procName - globPattern = '$globPattern', subDirList = '$subDirList'"
        foreach appDir $subDirList {
            if { ([string tolower [lindex [split $appDir "."] end]] != "hidden") } {

                #
                # Determine if this is a Collection node or an Application node
                #
                set nodeType  COLLECTION
                set imageType FOLDER
                set appName  [file tail $appDir]
                foreach suffix {tbc tcl} {
                    if { [file exists [file join $appDir $appName.$suffix]] } {
                        #
                        # Found an executable application file matching the directory name
                        # so treat this as an application node.
                        #
                        set nodeType  APPLICATION
                        set imageType COG
                        break
                    }
                }

                #
                # Add the node
                #
                set childData   [list -type $nodeType -directory "$appDir"]
                set childNodeId [$self AddNode $nodeId $appName $imageType $childData]
                set appName     [file tail $appDir]
            }
        }

        return $nodeId
    }

    #--------------------------------------------------------------------------
    #  GetNodeUserData { nodeId }
    #--------------------------------------------------------------------------
    # Parameters:
    #   nodeId - node ID in the tree
    #
    # Returns:
    #   userData - the user data that is recorded in the node
    #
    # Purpose:
    #   Retrieve the user data stored in the specified node.
    #--------------------------------------------------------------------------
    method GetNodeUserData { nodeId } {

        set tree    $mTree
        set usrData [$tree itemcget $nodeId -data]
        return $usrData
    }

    #--------------------------------------------------------------------------
    # NodeSelected { action nodeId }
    #--------------------------------------------------------------------------
    # Parameters:
    #   action - one of: ClickText, ClickImage, DblClickText, DblClickImage
    #   nodeId - selected node ID
    #   source - one of: TEXT, IMAGE
    #
    # Returns:
    #   nothing
    #
    # Purpose:
    #   React to the user either single-clicking or double-clicking on a node.
    #--------------------------------------------------------------------------
    method NodeSelected { action nodeId } {

        if { $nodeId != "root" } {
            set tree $mTree
            # ensure selection is visible
            $tree selection set $nodeId
            switch $action {
                ClickImage -
                ClickText   {
                    set callback $options(-selectioncallback)
                    if { ![string equal $callback "default"] } {
                        set nodeLabel [$tree itemcget $nodeId -text]
                        $callback $win [$self getNodeRecord $nodeId]
                    }
                }

                DblCickText -
                DblClickImage {
                    set callback $options(-actioncallback)
                    if { ![string equal $callback "default"] } {
                        set nodeLabel [$tree itemcget $nodeId -text]
                        $callback $win [$self getNodeRecord $nodeId]
                    }
                }
            }
        }
    }

    #--------------------------------------------------------------------------
    #  SetNodeUserData { nodeId newData }
    # Parameters:
    #   nodeId  - node ID in the tree
    #   newData - new user data record
    #
    # Returns:
    #   userData - the user data that is recorded in the node
    #
    # Purpose:
    #   Replace the user data stored in the specified node.
    #--------------------------------------------------------------------------
    method SetNodeUserData { nodeId newData } {

        set tree    $mTree
        set nodeData [$tree itemcget $nodeId -data] ;# form {<selectionstate> <userdatarcd>}
        set nodeData [lreplace $nodeData 0 0 $newData]
        $tree itemconfigure $nodeId -data $nodeData

        return [$tree itemcget $nodeId -data]
    }

    #--------------------------------------------------------------------------
    # ToggleNode { action nodeId }
    #--------------------------------------------------------------------------
    # Parameters:
    #   action - one of: EXPAND, COLLAPSE
    #   nodeId - the node to be acted upon
    #
    # Returns:
    #   nothing
    #
    # Purpose:
    #   React to the user clicking on the [+/-] image
    #--------------------------------------------------------------------------
    method ToggleNode { action nodeId } {

        set procName [AgtTsuProcedureName]
        set tree $mTree
        if { [$tree itemcget $nodeId -drawcross] != "never" } {
            switch $action {
                EXPAND {
                    # REVISIT - do we want to change the image ?
                    #$tree itemconfigure $nodeId -image [Bitmap::get openfold]
                }

                COLLAPSE {
                    # REVISIT - do we want to change the image ?
                    #$tree itemconfigure $nodeId -image [Bitmap::get folder]
                }

                default {
                    AgtTsuShowMessage ERROR "$procName - unrecognised action = $action on node $nodeId"
                }
            }
        }
    }

    #================================================================
    # Private Procedures (by convention begin with UpperCase)
    #================================================================
}

#==========================================================================
#   A u t o - I n i t i l i s a t i o n
#==========================================================================
puts [format "Loaded class definition: %s" [file rootname [file tail [info script]]]]

#==========================================================================
#--<end>--
#==========================================================================
