#--------------------------------------------------------------------------
# Title:  Profile Stepper
#--------------------------------------------------------------------------
# Created:       03 May 2009  Andy Westwood
# Last modified: 16 Jun 2009, kmg
#--------------------------------------------------------------------------
# Requires:     Tcl/Tk 8.3 (or later)
#--------------------------------------------------------------------------
#  Copyright (C) 2009 Agilent Technologies
#
#  All copies of this program, whether in whole or in part, and whether
#  modified or not, must display this and all other embedded copyright
#  and ownership notices in full.
#--------------------------------------------------------------------------
# Conditions of use:
#
#  This software is provided as an example. It is distributed in the hope
#  that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#--------------------------------------------------------------------------

#--------------------------------------------------------------------------
# This script progressively turns on Profiles in sequence, thereby altering
# Traffic Loading while the Real-Time traffic generator is running.
# Both Continuous and Bursty traffic types may be used, and the steps are
# linear per user entry in milliSeconds.
# All Profiles are cleared before the Stepping starts, so the user is
# CAUTIONED before proceeding to check the Session, Blade and Port.
#--------------------------------------------------------------------------

# Ensure we are running under wish
set isTk [info exists ::tk_version]

#----------------------------------------------------------------
# Load external files
#----------------------------------------------------------------

if { [llength [namespace children :: AgtQcl]] == 0 } {
    namespace eval :: {
        lappend auto_path "c:/program files/agilent/n2x/quicktest/lib"
        puts "Loaded AgtQcl [package require AgtQcl]"
    }

    AgtQclLoadClass -group ALL
}

#----------------------------------------------------------------
# Test Parameter Definition (TPD) XML
#----------------------------------------------------------------

set xmlString \
   {<?xml version="1.0" standalone="yes"?>
    <!-- ============================================================ -->
    <!-- Agilent Configuration Parameter Definition File              -->
    <!-- ============================================================ -->
    <!--  Application:    ProfileStepper                              -->
    <!--  Version:        1.0                                         -->
    <!--  Last modified:  12 Jun 2009                                 -->
    <!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ -->
    <!--  Copyright 2009 Agilent Technologies                         -->
    <!--  All rights reserved                                         -->
    <!-- ============================================================ -->

    <ConfigurationParameterSet application="ProfileStepper" version="1.01">

      <NotebookTab id="Setup">
        <ParameterSet id="testports" name="Test ports" titleframe="1" help="Selected test ports">
          <Parameter id="srcportlist"  type="n2xport" subtype="list" name="Source port" variable="::gData(srcPortName)" width="10"/>
        </ParameterSet>
        <ParameterSet id="profilecontrol" name="Profile control" titleframe="1">
          <Parameter id="mode"      type="combobox" name="Mode"                         variable="::gData(mode)"      width="20" valuelist="Incremental Round-robin" help="Whether to enable Profiles in a round-robin or incremental fashion"/>
          <Parameter id="steptime"  type="integer"  name="Step interval (milliseconds)" variable="::gData(delayStep)" width="20" help="The duration in milliseconds between enabling each Profile"/>
        </ParameterSet>
      </NotebookTab>

    </ConfigurationParameterSet>
}

#----------------------------------------------------------------
# Master Data array
#----------------------------------------------------------------
array set gData [list                                   \
                                                        \
    agtServerName           localhost                   \
    agtSessionType          RouterTester900             \
    agtSessionVersion       latest                      \
    agtSessionLabel         ""                          \
    agtSessionHandle        0                           \
    agtConnectionHandle     0                           \
                                                        \
    delayStep               1000                        \
                                                        \
    mode                    Incremental                 \
                                                        \
    scriptName              [file rootname [file tail [info script]]] \
    scriptTitle             "Profile Stepper Utility"   \
    scriptVersion           2.04                        \
                                                        \
    srcPortName             "101/1"                     \
]

#----------------------------------------------------------------
# Procedures
#----------------------------------------------------------------

#----------------------------------------------------------------
# StartTest { }
#----------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Handle the framework StartTest callback.
#----------------------------------------------------------------
proc StartTest { } {
    global gData


    #----------------------------------------------------------------
    # This is where the core script functionality goes
    #----------------------------------------------------------------

    #
    # Check for connection
    #
    if { ![AgtTsuSessionConnected] } {
       AgtTsuShowMessage FATAL "Unable to connect to the N2X test session, exiting."
    }

    $gData(myApp) runState RUNNING

    #------------------------------------------------------------------------------------
    # Here the script interrogates the Session and gets the Handles for the Port and
    # all open Profiles currently on the desired Port.
    #------------------------------------------------------------------------------------

    set userBlade [lindex [split $gData(srcPortName) /] 0]
    set userPort  [lindex [split $gData(srcPortName) /] 1]

    AgtTsuTraceMessage ""
    AgtTsuTraceMessage "Here are the ports I found within your session ="
    set portHandleList [AgtInvoke AgtPortSelector ListPorts]
    set portNameList   [AgtTsuConvertPortHandleToName $portHandleList]
    set p 1
    foreach hPort $portHandleList portName $portNameList {
        AgtTsuTraceMessage "  Port $p is $portName type [AgtTsuGetPortType $hPort]"
        incr p
    }
    AgtTsuTraceMessage ""

    set hSourcePort [AgtTsuConvertPortNameToHandle $gData(srcPortName)]
    set profileList [AgtInvoke AgtProfileList ListProfilesOnPort $hSourcePort]

    AgtTsuTraceMessage "Here are the Profiles I found in your CHOSEN port $gData(srcPortName) = "
    set i 0
    while { $i < [llength $profileList] } {
            AgtTsuTraceMessage  "  Profile $i is named [AgtInvoke AgtProfileList GetName [lindex $profileList $i]] \
            and is type [AgtInvoke AgtProfileList GetType [lindex $profileList $i]]"
            incr i
    }

    #------------------------------------------------------------------------------------
    # Here the script  DIS-ables all profiles so that they can be sequenced.
    # The API requires you use the Profile Type (Constant, Burst or Custom) to manipulate
    # each profile, so the loop works like this =
    # > Get the type of each Profile
    # > Operate on that Profile (Enable it, Disable it)
    # > There is no bulk command to turn them off/on
    #------------------------------------------------------------------------------------

    if { [string equal -nocase $gData(mode) incremental] } {
        set e 0
        set t 0
        set msg "Now I am DIS-abling all Profiles to provide a fresh start for the Stepping."
    } else {
        set e 1
        set t 1
        set msg "Now I am DIS-abling all Profiles EXCEPT PROFILE ZERO  to provide a fresh start for the Stepping."
    }

    AgtTsuTraceMessage ""
    AgtTsuTraceMessage $msg
    AgtTsuTraceMessage "NOTE - the script assumes that the traffic streams desired are enabled by the user in each Profile"
    AgtTsuTraceMessage ""

    while { $e < [llength $profileList] } {
        set thisProfileType [AgtInvoke AgtProfileList GetType [lindex $profileList $e]]
        if {$thisProfileType == "AGT_CONSTANT_PROFILE"} {
        set thisProfileType AgtConstantProfile
    }   elseif {$thisProfileType == "AGT_BURST_PROFILE"} {
        set thisProfileType AgtBurstProfile
    }
        AgtInvoke $thisProfileType Disable [lindex $profileList $e]
        incr e
    }

    #------------------------------------------------------------------------------------
    # Here the script ENABLES or turns-on the profiles one-by-one in sequence at the rate.
    # determined by the user-entry in milliSeconds.
    # Again, note the Profile Type is determined, then the action is taken.
    #------------------------------------------------------------------------------------

    if { ![string equal -nocase $gData(mode) incremental] } {
        AgtInvoke $thisProfileType Enable [lindex $profileList 0]
    }

    AgtTsuShowMessage INFO "When you hit Enter, I will begin EN-Abling all Profiles in sequence with a delay of $gData(delayStep) milliSeconds between each one."

    while { $t < [llength $profileList] } {
            set thisProfileType [AgtInvoke AgtProfileList GetType [lindex $profileList $t]]
            if {$thisProfileType == "AGT_CONSTANT_PROFILE"} {
            set thisProfileType AgtConstantProfile
    }   elseif {$thisProfileType == "AGT_BURST_PROFILE"} {
            set thisProfileType AgtBurstProfile
    }
            AgtInvoke $thisProfileType Enable [lindex $profileList $t]
            AgtTsuTraceMessage "Profile [AgtInvoke AgtProfileList GetName [lindex $profileList $t]] is enabled."
            AgtTsuDelay $gData(delayStep)
            if { ![string equal -nocase $gData(mode) incremental] } {
                AgtInvoke $thisProfileType Disable [lindex $profileList $t]
            }
            incr t
    }

    AgtTsuTraceMessage ""
    AgtTsuTraceMessage "Done"
    AgtTsuTraceMessage ""

    $gData(myApp) stopTest
}

#----------------------------------------------------------------
# StopTest { }
#----------------------------------------------------------------
# Parameters:
#   none
#
# Returns:
#   nothing
#
# Purpose:
#   Handle the framework StopTest callback.
#----------------------------------------------------------------
proc StopTest { } {
    global gData

    #----------------------------------------------------------------
    # Terminate any looping behaviour and tidy up if needed
    #----------------------------------------------------------------

    $gData(myApp) runState STOPPED
}

#----------------------------------------------------------------
# M A I N
#----------------------------------------------------------------

if { !$isTk } {
    puts "\n\n[CMiniTestApplication getDefaultUsageMessage $gData(scriptName)]"
    exit
} else {
    #
    # Create the application
    #
    set gData(myApp) [CMiniTestApplication %AUTO% -title        $gData(scriptTitle)   \
                                                  -version      $gData(scriptVersion) \
                                                  -parsecmdargs 1                     \
                                                  -cmdlineargs  $argv                 \
                                                  -cbstarttest  ::StartTest           \
                                                  -cbstoptest   ::StopTest            \
                                                  -tpdxml       $xmlString]

    # All done, GUI should be up and running for the user.
}