/************************************************************************************
 * Title         : StateMachine
 * Project       : GrizzlyDemo
 *-----------------------------------------------------------------------------------
 * File          : StateMachine.v
 * Author        : Matt Clark  <matt_clark@agilent.com>
 * Created       : 06.20.2013
 * Last modified : 06.20.2013
 *-----------------------------------------------------------------------------------
 *********SVN Revision History*************
 * $Date$
 * $Author$
 * $HeadURL$
 * $Revision$
 *----------------------------------------------------------------------------------
 * Description   :
 * This is a poorly contrived demo to show a state machine with a watchdog counter
 * that goes into an error state.  The logic analyzer is used to find the error and
 * then it will go back in time to see what went wrong.
 * It is a simple and dumb demo. Hopefully it is sufficient.
 * 
 * So basically, a state machine pretends to do some address, data protocol and then
 * it waits for an ACK signal before resetting.  If ACK doesn't come quick enough, it
 * will screw up.  The demo does a few good ACKs before "going bad" and hanging.  It will
 * reset itself after about 5 seconds of real time.
 * 
 *-----------------------------------------------------------------------------------
 **************************  Agilent Confidential  **********************************
 * All rights reserved.
 *                                   _
 *                         .-.      / \        _
 *                        /   \    /^./\__   _/ \
 *           _        .--'\/\_ \__/.      \ /    \      ___
 *          / \_    _/ ^      \/  __  :'   /\/\  /\  __/   \
 *         /    \  /    .'   _/  /  \   ^ /    \/  \/ .`'\_/\
 *        /\/\  /\/ :' __  ^/  ^/    `--./.'  ^  `-.\ _    _:\ _
 *       /    \/  \  _/  \-' __/.' ^ _   \_   .'\   _/ \ .  __/ \
 *     /\  .-   `. \/     \ / -.   _/ \ -. `_/   \ /    `._/  ^  \
 *    /  `-.__ ^   / .-'.--'    . /    --./ .-'  `-.  `-. `.  -   `.
 *   /        `.  / /       `-.  /  .-'  / .    .'  \    \  \  .-   \
 *----------------------------------------------------------------------------------
 * Copyright (c) 2013 by Agilent. This model is the confidential and
 * proprietary property of Agilent and the possession or use of this
 * file requires a written license from Agilent.
 *----------------------------------------------------------------------------------
 * Modification history / Change log :
 * 06.20.2013 : Matt Clark
 *   --Initial revision
 **********************************************************************************/

module StateMachine (
  input clk,
  input resetAsync_n,

  output reg [4:0] state,
  output reg ack,
  output wire timeout
  );

  parameter S_IDLE  = 5'h01;
  parameter S_START = 5'h02;
  parameter S_ADDR  = 5'h04;
  parameter S_DATA  = 5'h08;
  parameter S_ACK   = 5'h10;
  // remaining states are reserved

  parameter NUM_BUGS = 2; // 2**BUGS is number of loops before breaking
  parameter DEAD_TIME_SIZE = 10; // This is the number of bits in the stall between state machine loops
  parameter WATCHDOG_SIZE = 17; // 21 bits @ 250 MHz will make 8msec timeout.
//  parameter RESET_SIZE = 30;
  parameter RESET_SIZE = 18;
  

  reg [4:0]    next_state;
  reg [NUM_BUGS-1:0] bug;
  reg [DEAD_TIME_SIZE-1:0] deadTimer;
  reg [WATCHDOG_SIZE:0]  watchTimer;
  reg [RESET_SIZE:0]     resetTimer;

 
// Synchronize the reset signal for use inside this module:
reg reset1_n;
reg reset2_n;
reg syncReset_n;
always @(posedge clk or negedge resetAsync_n)
	if (!resetAsync_n)
		begin
		reset1_n <= 1'b0;
		reset2_n <= 1'b0;
		syncReset_n <= 1'b0;
		end
	else
		begin
		reset1_n <= resetAsync_n;
		reset2_n <= reset1_n;
		syncReset_n <= reset1_n & reset2_n;
		end

always @(posedge clk)
   begin
   if (!syncReset_n)
      watchTimer <= 'b0;
   else if (state == S_START)
      watchTimer <= 'b0;
   else if (!watchTimer[WATCHDOG_SIZE])
      watchTimer <= watchTimer + 'b1;
   end
  
always @(posedge clk)
   begin
   if (!syncReset_n)
      deadTimer <= 'b0;
   else if (state == S_DATA)
      deadTimer <= 'b1;
   else if (|deadTimer)
      deadTimer <= deadTimer + 'b1;
   end
  
always @(posedge clk)
   begin
   if (!syncReset_n)
      bug <= 'b0;
   else if (state == S_DATA)
      bug <= bug + 'b1;
   end
  
always @(posedge clk)
   begin
   if (!syncReset_n)
      ack <= 1'b0;
   else if (state == S_ACK)
      ack <= (ack) ? 1'b0 : ~bug[NUM_BUGS-1] & deadTimer[4];
   end

always @(posedge clk)
   begin
   if (!syncReset_n)
      resetTimer <= 'b0;
   else if (state == 5'h1f)
      resetTimer <= resetTimer + 'b1;
   else if (state == S_IDLE)
      resetTimer <= 'b0;
   end
  // End of automatics
  
  //////////////////////////////////////////////////////////////////////////////
  // Here is a simple state machine that will cycle through a couple times and
  // then it receives an interrupt or something that causes it to halt.
always @(posedge clk)
   if (!syncReset_n)
      state <= S_IDLE;
   else
      state <= next_state;
  
always @(/*AUTOSENSE*/ack or bug or deadTimer or resetTimer or state
           or watchTimer)
   begin
   case (state)
      S_IDLE : begin
               if (!(|deadTimer))
                  next_state = S_START;
               else
                  next_state = state;
               end  
      S_START : next_state = S_ADDR;      
      S_ADDR : next_state = S_DATA;
      S_DATA : next_state = S_ACK;
      S_ACK :  begin
               if (ack || &deadTimer)
                  next_state = (&bug) ? 5'h1f : S_IDLE;
               else
                  next_state = state;
               end
      default : begin
               if (resetTimer[RESET_SIZE]) // if timer is full
                  next_state = S_IDLE;
               else
                  next_state = state;
               end
   endcase // case (state)
   end // always @ (...

  //////////////////////////////////////////////////////////////////////////////
  // Assign an output for the demo board LED
assign timeout = watchTimer[WATCHDOG_SIZE]; //watchTimer[WATCHDOG_SIZE-1];
  
endmodule
