/*****************************************************************************

 File: xfer.c

 This C program demonstrates how to do block transfers between 
 a PC and the HP 35670A analyzer.  This program can be used to
 transfer time-capture files from the 35670A to a PC file.
 
 The compiled version of this program is on the Example Program disc
 in the file named XFER.EXE.
 
 The command line syntax for XFER.EXE is:

    XFER <file> [/O] [/S] [/C] [/A] [I] [D]

    <file>       path and file name
    /U           show this usage
    /I:<code>    Select code (default = 7 for Hewlett-Packard card,
                    0 for National Instruments card)
    /A:<add>     Analyzer address (default = 11)
    /S           Send <file> to analyzer
    /C:<cmd>     Analyzer command string (default = TCAP:FILE)
    /D:<H|N>     H = use Hewlett-Packard HPIB card (default)
                    N = use National Instruments GPIB card
    /O           Overwrite <file> if it exists.

For example, to get the time capture buffer from the HP 35670A to a file
named CAPT1.TIM on a PC using a Hewlett-Packard card, use this
command:

       xfer CAPT1.TIM

To send the CAPT1.TIM file back to the HP 35670A using a National AT-GPIB
card, execute this command:

       xfer /S /D:N CAPT1.TIM

This program supports the HP 82335A card and the National Instruments
AT-GPIB card. To compile it you need the clhpib.lib library for the
HP card and the mcib.obj library for the NI card.

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <CHPIB.H>              /* header file for HP-IB card */
#include <CFUNC.H>              /* header file for HP_IB card */
#include "\at-gpib\c\decl.h"    /* header file for NI GPIB card */

#define DEFAULT_HP_ISC   7      /* default select code for HP card */
#define DEFAULT_NI_ISC   0      /* default card address for NI card */
#define DEFAULT_ADDRESS  11     /* default address for analyzer */
#define MAX_XFER         32767  /* transfer buffer size */

#define A_STRING_LEN     256    /* length of string for GPIB input */

#define XFERCMD_LEN      80
#define DEFAULT_XFERCMD  "TCAP:FILE"

/*************************************************/
/* Typedefs                                      */
/*************************************************/

enum          BOOL     {FALSE, TRUE};
enum          CARD {HP, NI};                 /* card vendor */

/*************************************************/
/* Global variable definitions                   */
/*************************************************/

int           address = DEFAULT_ADDRESS;  /* analyzer address */
short         NIeotmode;                  /* end of xmission mode for NI card */
char          modelNum[A_STRING_LEN+1];   /* model num of analyzer */

enum CARD     card = HP;                  /* default GPIB card */
int           isc;                        /* select code of GPIB card */

char          xferCmd[XFERCMD_LEN+1];     /* GPIB mnemonic for transfer */
unsigned char buffer[MAX_XFER];           /* data buffer for transfer */
char          filename[80];               /* data file name */
FILE          *file = (FILE *) 0;         /* file pointer for data file */
enum BOOL     overwrite = FALSE;          /* if TRUE and data file exists,
                                             data file is overwritten */
char          aString[A_STRING_LEN+1];    /* for receiving GPIB input */
enum BOOL     send = FALSE;               /* TRUE = send data, 
                                             FALSE = receive data */
                                             
char          *optarg = NULL;             /* command line option argument */
int           optind = 0;
enum BOOL     testConnection = FALSE;     /* Just check PC to analyzer
                                             connection if TRUE */

/*************************************************/
/* Function Prototypes                           */
/*************************************************/

void checkErrorBuffer(void);


/* define usageLines, the string used to print the helptext for
   this program. */
char usageLines[] = {
"xfer <file> [/O] [/S] [/C] [/A] [I] [D]\n\
\n\
<file>       path and file name\n\
/U           show this usage\n\
/I:<code>    Select code (default = 7 for Hewlett-Packard card, \n\
                0 for National Instruments card)\n\
/A:<add>     Analyzer address (default = 11)\n\
/S           Send <file> to analyzer\n\
/C:<cmd>     Analyzer command string (default = TCAP:FILE)\n\
/D:<H|N>     H = use Hewlett-Packard HPIB card (default);\n\
             N = use National Instruments GPIB card\n\
/O           Overwrite <file> if it exists (not applicable when using /S)\n\
\n\
Gets definite block from analyzer and stores it in <file>.\n\
If /S is specified, sends <file> to the analyzer.\n\
The /C option specifies the type of data to send or receive.\n"
};

/* usage() prints the helptext */
void usage( void )
{
    printf("%s", usageLines);
}

/* myExit() is called when a fatal error occurs. */
void myExit(void)
{
  if ( file != NULL )
     fclose( file );

  exit(1);
}

/* gpibDeviceError prints the error number reported by the GPIB card,
   then exits with fatal error */
void gpibDeviceError( int err )
{
   char errMsg[80];
 
   if (card == NI)
      printf("ERROR %d reported by NI GPIB card\n", err );
   else
      printf("ERROR %d reported by HP-IB card", err);
 
   myExit();
}

/* setEOTmode() sets the end of transmission mode. Two parameters
   can be specified:

   sendCrlf: if TRUE, carriage return/line feed is sent after GPIB command
             or data block.
   setEOI:   if TRUE, end-or-identify is sent after GPIB command or data
             block.
*/
void setEOTmode(enum BOOL sendCrlf, enum BOOL setEOI)
{
  switch (card)
  {
     case HP:
        /* set the HP card for the EOT mode that we want */
        if (sendCrlf)
           IOEOL(isc, "\r\n", 2);
        else
           IOEOL(isc, "", 0);
        IOEOI(isc, ((setEOI==TRUE) ? 1 : 0) );
        break;
     case NI:
        /* set the global variables for the mode we want. The mode
           will be passed to the NI driver function when it is needed. */ 
        if ( sendCrlf && setEOI )
           NIeotmode = NLend;
        else if (setEOI && !sendCrlf)
           NIeotmode = DABend;
        else
           NIeotmode = NULLend;
        break;
  }   
} 
 
/* output a string to the analyzer. Also sends CR/LF and EOI if
   specified. */
void outputcmd(char *strn, enum BOOL sendCrlf, enum BOOL setEOI)
{
     int  err = 0;
     int  len;
     char s[80];

     strcpy(s,strn);
     len = strlen(s);

     setEOTmode(sendCrlf, setEOI);

     switch ( card )
     {
        case HP:
           err = IOOUTPUTS(isc*100+address,s,len);
           break;
        case NI:
           Send( isc, address, s, len, NIeotmode);
           if (ibsta & ERR)
              err = iberr;
           break;
     }
     if (err)   gpibDeviceError(err);
}

/* outputBlock() outputs len bytes of data pointed to by buffer to the
   analyzer. CR/LF and EOI are not sent after the block. */ 
void outputBlock( unsigned char *buffer, int len)
{
     int err = 0;

     setEOTmode(FALSE, FALSE);

     switch ( card )
     {
        case HP:
           err = IOOUTPUTB(isc*100+address,buffer,len,1);
           break;
        case NI:
           Send( isc, address, buffer, len, NIeotmode);
           if (ibsta & ERR)
              err = iberr;
           break;
     }

    if (err)   gpibDeviceError(err);
}

/* enterString() gets up to len characters from the analyzer. The
   actual length of the received string is returned in len. */
static void enterString( int *len )
{
   int err=0;

   switch (card)
   {
      case HP:
         err = IOENTERS(isc*100+address, aString, len);
         break;
      case NI:
         Receive( isc, address, aString, *len, STOPend);
         if (ibsta & ERR)
            err = iberr;
         *len = ibcnt;
         if (ibcnt <= A_STRING_LEN)
            aString[ibcnt] = 0;
         break;
   }
 
   if (err)
      gpibDeviceError(err);
}

/* enterBlock() enters up to len bytes of data from the HPIB and puts them
   in the block pointed to by buff. The actual number of bytes received
   is returned in len. gotEOI is set TRUE if EOI is received (EOI signals
   the end of the data block). */
static void enterBlock(unsigned char *buff, int *len, enum BOOL *gotEOI )
{
   int err = 0;
   int reason;  /* reason the HP card stopped reading data */

   switch (card)
   {
      case HP:
         err = IOENTERB(isc*100+address,buff,len,1);
         err = IOGETTERM(isc, &reason);
         *gotEOI = reason & 4;
         break;
      case NI:
         Receive( isc, address, buff, *len, STOPend);
         *len = ibcnt;
         if (ibsta & ERR)
            err = iberr;
         *gotEOI = ibsta & END;

         /* strip newline if this is end of block */
         if ( *gotEOI && (*len > 0) )
         {
            if ( buff[*len - 1] == '\n' )
               (*len) --;
         }
         break;
   }

   if (err)    gpibDeviceError(err);
}

/* clearDevice() sends the interface clear (IFC) GPIB command to
   the analyzer. */
void clearDevice(void)
{
    int err = 0;

    switch (card)
    {
       case HP:
          err = IOCLEAR(isc*100 + address);
          break;
       case NI:
          DevClear( isc, MakeAddr(address,0) );
          if (ibsta & ERR)
             err = iberr;
          break;
    }

    if (err)    gpibDeviceError(err);
}    

/* checkErrorBuffer() reads the analyzer error buffer util it is empty.
   All errors are printed on the screen. */
void checkErrorBuffer(void)
{    
     int     length;
     int     err, nErrs=0;
      
     clearDevice(); 

     do
     {
        outputcmd("syst:err?", TRUE, TRUE);
        length = A_STRING_LEN;
        enterString( &length );
        err = atoi( aString );
        if ( err != 0 )
        {
           nErrs++;
           if ( nErrs == 1 )
              printf("\n");
           printf("%s error:\n", modelNum);
           printf("   %s\n", aString); 
        }
     }  while ( err != 0 && (nErrs < 20) );
}

/* initHpib() sends IFC on the bus to clear all devices. For the HP
   GPIB card, it also sets the timeout value and clears the REN line
   so that the analyzer stays in local mode after commands are sent. */
void initHpib( void )
{
    int err = 0;
 
    switch (card)
    {
       case HP:
          err = IORESET( isc );
          if ( ! err )
          {
             err = IOTIMEOUT( isc, 1.0 ); /* 1 second timeout */
             /* clear the REN line so analyzer stays local */
             err = IOLOCAL( isc );
          }
          break;

       case NI:
          SendIFC(isc);
          if (ibsta & ERR)
            err = iberr;
          break;
    }

    if (err)   gpibDeviceError(err);
}

/* doesFileExist() returns TRUE if the file named filename exists,
   otherwise it returns FALSE. */
int doesFileExist( char *filename )
{
   FILE *file;

   file = fopen( filename, "r");
   if (file == NULL)
      return 0;
   else
   {
      fclose( file );
      return 1;
   }
}

/* openOut() opens filename for writing */
void openOut( void )
{
    if ( ! overwrite && doesFileExist (filename) )
    {
	   printf("%s exists; use /O to overwrite\n",filename);
	   exit(1);
    }
    if ( (file = fopen (filename, "wb") ) == NULL)
    {
	   printf("%s not found\n",filename);
	   exit(1);
    }
}

/* openIn() opens filename for reading */
void openIn( void )
{
    if ( (file = fopen (filename, "rb") ) == NULL)
    {
           printf("%s not found\n",filename);
           exit(1);
    }
}

/* displayProgress() displays the percentage completion of the block
   transfer. */ 
void displayProgress(long totalSize, long bytesSoFar)
{
  printf("\r"); 
  printf("Transfering %ld bytes (%d%% done)", totalSize,
        (int) ( 100. * ( (float) bytesSoFar/ (float) totalSize)));
}

/* getBlockSize() reads the response to the block query and determines
   the blocksize in bytes and if the response is an indefinite block. */
void getBlockSize(enum BOOL *indefinite, long *blocksize)
{
   int length;

   *indefinite = FALSE;

   /* get the "A" prefix */
   length = 1;  
   enterString( &length );
   
   if ( strcmp( aString, "#" ) )
   {
      printf("bad response to block data query\n");
      myExit();
   }

   /* get the number of characters in the length string (0-9)*/
   length = 1;
   enterString( &length );
   length = atoi( aString );

   if ( length == 0 )
   {
      *indefinite = TRUE;
      *blocksize = 0;
      return;
   }
      
   /* get the blocksize string */
   enterString( &length );

   if ( length )
      *blocksize = atol( aString );
   else
   {
      printf("bad response to block data query\n");
      myExit();
   }
} 

/* getDefiniteBlock() gets a definite block of length blocksize bytes
   from the analyzer and puts the block in a file. The number of bytes
   received is returned. */
static long getDefiniteBlock(long blocksize) 
{
   int   length;
   long  bytesXfered = 0; 
   int   expectToGet;    
   int   err;
   int   nWritten;
   enum  BOOL gotEOI;

   bytesXfered = blocksize;
 
   /* get the block */ 
   while ( blocksize ) 
   {
      displayProgress( bytesXfered, (bytesXfered-blocksize) );
   
      length = (blocksize > MAX_XFER) ? MAX_XFER : blocksize;
      
      expectToGet = length;

      enterBlock(buffer, &length, &gotEOI);

      if ( length != expectToGet )
      {
         printf("expected to get %d; only sent %d\n",
                 expectToGet, length);
      }

      nWritten = fwrite( buffer, sizeof(char), length, file);

      if (nWritten != length)
      {
         printf("can't write block to disc - check disc space\n");
         fclose( file );
         exit(1);
      }
      
      blocksize -= length;
   }
   displayProgress( bytesXfered, (bytesXfered-blocksize) );
   printf("\n");

   /* read the line feed character */
   length = A_STRING_LEN;
   enterString( &length );

   return bytesXfered - blocksize;
}

/* getBlockData() gets the block data requested by the user and puts
   the data into a file.  Only definite length blocks are supported. */
static long getBlockData( void )
{
   int   length;
   long  blocksize;
   long  bytesXfered=0;
   enum BOOL indefinite;

   /* open file to put capture in */
   openOut(); 
  
   strcpy(aString, xferCmd);

   /* if there is no question mark in the command string, add
      one to the end.  */
   if ( ! strchr(aString, '?') )   strcat(aString,"?");

   outputcmd( aString, TRUE, TRUE );

   getBlockSize(&indefinite, &blocksize);

   if ( indefinite )
   {
      printf("Indefinite block transfer not supported\n");
      myExit();
   }
   else
      bytesXfered = getDefiniteBlock( blocksize );
 
   fclose( file );
   return bytesXfered;
}

/* sendBlockData() sends the contents of the file to the analyzer using
   a definite block transfer. The number of bytes sent is returned. */
long sendBlockData(void)
{
   long blocksize; 
   long bytesXfered;
   char bsString[10]; /* blocksize as a string */
   int  length;
   int  nRead;
 
   /* open file for reading */ 
   openIn();

   /* find the size of the file */
   fseek( file, 0L, SEEK_END );
   blocksize = ftell( file );
   fseek( file, 0L, SEEK_SET );

   if ( !blocksize )
   {
      fclose( file );
      printf("file is empty, no data sent\n");
      myExit();
   }

   sprintf(bsString, "%ld", blocksize);
   length = strlen(bsString);
   sprintf(aString, "%s #%d%s", xferCmd, length, bsString);

   /* send the command string */
   outputcmd( aString, FALSE, FALSE );

   bytesXfered = blocksize;

   while (blocksize)
   {
      displayProgress( bytesXfered, (bytesXfered-blocksize) );
      length = (blocksize > MAX_XFER) ? MAX_XFER : blocksize;
      
      nRead = fread( buffer, sizeof(char), length, file);
      if (nRead != length)
      { 
         fclose( file );
         printf("unknown file error\n");
         myExit();
      }

      blocksize -= length;
      outputBlock( buffer, length);
   }

   displayProgress( bytesXfered, (bytesXfered-blocksize) );
   printf("\n");
   fclose( file );

   return bytesXfered - blocksize;
}

/* getopt() parses the command line string */
int getopt( int argc, char * argv[], char * optstring )
{
    int opt;

    optarg = NULL;
    if (++optind == argc) return( EOF );

    if ((argv[optind][0] != '-') && (argv[optind][0] != '/'))
    {
	optarg = argv[optind];
	return( '\0' );
    }

    opt = toupper( argv[optind][1] );

    if (opt == '$')
    {
	printf( "command error\n");
        usage();
	exit(1);
    }

    while (optstring[0] && (optstring[0] != (char) opt)) ++optstring;
    if (!optstring[0]) return( opt );

    if (optstring[1] == ':' || optstring[1] == ';')
    {
	if (argv[optind][2])
	{
	    optarg = &argv[optind][2];
	}
	else if (optstring[1] == ';')  /* optional argument */
	    return opt;
	else
	{
	    if (++optind == argc) return( EOF );
	    optarg = argv[optind];
	}
	if (*optarg == ':')
	{
	    optarg++;
	}
	else if (*optarg == '=')
	{
	    optarg++;
	}
    }
    return( opt );
}

/* parseCmdLine() determines what this program is going to do
   based on the command line entered by the user */
static void parseCmdLine (int argc, char *argv[])
{   
  int  c;
  char tmp[80];
  enum BOOL iscSet = FALSE;

  *filename = '\0';
  *xferCmd = '\0';

  while ( (c = getopt (argc, argv, "TUOSI:A:C:D:") ) != EOF)
  {
        switch (c) 
        {
           case 'I':
                strcpy(tmp, optarg);
                isc = atoi(tmp);
                iscSet = TRUE;
                break;
           case 'A':
                strcpy(tmp, optarg);
                address = atoi(tmp);
                if (address<0 || address > 31)
                {
                   printf("invalid device address\n");
                   exit(1);
                }
                break;
           case 'D':
                strcpy( tmp, optarg );
                if ( (!strcmp(tmp,"H")) || (!strcmp(tmp,"h")) )
                {
                   card = HP;
                }
                else if ( (!strcmp(tmp,"N")) || (!strcmp(tmp,"n")) )
                {  
                   card = NI;
                }
                else
                {
                   printf("invalid GPIB card specifier\n");
                   exit(1);
                }
                break;
           case 'U':
                usage();
                exit(1);
           case 'C':
                if (strlen(optarg) > XFERCMD_LEN)
                {
                   printf("GPIB command string too long (>%d characters)\n",
                          XFERCMD_LEN);
                   exit(1);
                }
                strcpy(xferCmd, optarg);
                break;
           case 'O':
                overwrite = TRUE;
                break;
           case 'S':
                send = TRUE;
                break;
           case 'T':
                testConnection = TRUE;
                break;
           case '\0':
                strcpy (filename, optarg);
                break;
        }
   }

   if ( *filename == '\0' && ! testConnection )
   {
      usage();
      exit(1);
   }

   if (*xferCmd == '\0')
     strcpy(xferCmd, DEFAULT_XFERCMD);

   /* if user did not specify card address, set the default */
   if ( !iscSet )
   {
      if ( card == HP )
        isc = DEFAULT_HP_ISC;
      else
        isc = DEFAULT_NI_ISC;
   }
}

void main( int argc, char *argv[])
{
     long    bytes;
     time_t  start, finish;
     double  elapsedTime;
     int     length;
     int     i;

     parseCmdLine (argc, argv);  

     /* start keeping time for the transfer */
     time( &start );

     initHpib();

     /* clear device (in case of previous bad transfer) */
     clearDevice();

     /* get the analyzer id */
     outputcmd("*idn?", TRUE, TRUE);
     length = A_STRING_LEN;
     enterString( &length );
     strcpy(modelNum, aString);

     if ( testConnection )
     {
        printf("device found at address %d:\n", address);
        printf("   %s\n", modelNum);
        exit(0);
     }

     /* clear the error buffer */
     outputcmd("*cls", TRUE, TRUE);

     /* now send or receive data block */
     if (send)    
     {
        bytes = sendBlockData();
     }
     else
     {
        bytes = getBlockData();
     }

     /* clear the GPIB interface and look for any errors */
     clearDevice(); 
     checkErrorBuffer();

     /* calculate and display the transfer speed */
     time( &finish );
     elapsedTime = difftime(finish, start);
     if (elapsedTime < 1.0) elapsedTime = 1.0;

     printf("%ld bytes transfered in %.0f seconds (%.0f bytes/sec)\n",
             bytes, elapsedTime, bytes/elapsedTime);
}
