/*
 *   Copyright 2011, Agilent Technologies. All rights reserved.
 *
 *   Implementation of the M918X kernel module for the 2.6 linux kernel.
 *   This driver provides a hardware interface to the following instruments:
 *
 *   Agilent M9181, M9182, M9183 Digital Multimeters
 *   Signametrics SM2060, SM2064, SMX2055, SMX2060, SMX2064 Digital Multimeters
 *   Signametrics SMX4030, SMX4032 Switch Cards
 */


#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/pci.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/mm.h>
#include <linux/kdev_t.h>
#include <asm/page.h>

#include "M918X.h"

int M918X_major = 0;
int M918X_nr_devices = 0;
struct M918X_dev * M918X_devices = NULL;

// BHK NOTE:
// At this point, the only thing the following values are used for is in setting
// M918X_dev.number values correctly, which is just used in some print statements.
int M918X_nr_dmms = 0;
int M918X_nr_switches = 0;


static int is_M918X_series(int type) {
	return (2055 == type) || (2060 == type) || (2064 == type) || (9181 == type) || (9182 == type) || (9183 == type);
}

static int is_SMX403X_series(int type) {
	return (4030 == type) || (4032 == type);
}

int M918X_file_open(struct inode *inode, struct file *filp)
{
	struct M918X_dev *sm_dev;

	sm_dev = container_of(inode->i_cdev, struct M918X_dev, cdev);
	filp->private_data = sm_dev;
	printk(KERN_INFO "M918X_file_open %d: %d_%d\n", sm_dev->minor, sm_dev->type, sm_dev->number);

	return 0;
}

int M918X_file_release(struct inode *inode, struct file *filp)
{
	struct M918X_dev *sm_dev = filp->private_data;

	printk(KERN_INFO "M918X_file_release %d: %d_%d\n", sm_dev->minor, sm_dev->type, sm_dev->number);

	return 0;
}

// handles ioctl commands specific to the SMX2055, SMX2060, SMX2064, and M9181, M9182, M9183
long M918X_DMM_compat_ioctl(struct M918X_dev *sm_dev, unsigned int cmd)
{
	struct pci_dev *pci_dev = &sm_dev->pci_dev;
	u16 short_value;
	int err;
	
	switch (cmd) {

	case M918X_DMM_IOC_VENDORID:
		return pci_dev->vendor;
	
	case M918X_DMM_IOC_DEVICETYPE:
		// for M918Xs, device type is stored in the device id
		if (pci_dev->vendor == AGILENT_VENDOR_ID) {
			return pci_dev->device;
		}
		// for Signametrics products, device type is stored as the subsystem id
		else {
			err = pci_read_config_word(pci_dev, PCI_SUBSYSTEM_ID, &short_value);
			if (err < 0) {
				return err;
			}
			return short_value;
		} 

	case M918X_DMM_IOC_SERIAL_MANDATE:
		return ioread32(sm_dev->addr_bar0 + 0x30);

	case M918X_DMM_IOC_HWVERSION_OPTCODE:
		return ioread16(sm_dev->addr_bar0 + 0x0C);

	case M918X_DMM_IOC_MANUF_COUNTRY:
		return ioread16(sm_dev->addr_bar0 + 0x36);

	case M918X_DMM_IOC_PCI_BAR2_OFFSET:
		return pci_resource_start(pci_dev, M918X_PCI_DATA_RESOURCE) & ~PAGE_MASK;

	default:
		printk(KERN_ERR "M918X_compat_ioctl failed:  unknown IOC command\n");
		return -ENOTTY;
	}
}

// handles ioctl commands specific to the SMX4030 and SMX4032
long SMX403X_compat_ioctl(struct M918X_dev* sm_dev, unsigned int cmd)
{
	struct pci_dev *pci_dev = &sm_dev->pci_dev;
	u16 short_value;
	int err;
	
	switch (cmd) {
	case SIGNAMET_IOC_SERIAL:
		return ioread16(sm_dev->addr_bar0 + 0x30);


	case SIGNAMET_IOC_SUBSYS_DEVTYPE:
		err = pci_read_config_word(pci_dev, PCI_SUBSYSTEM_ID, &short_value);
		if (err < 0) {
			return err;
		}
		return short_value << 16;

	case SIGNAMET_IOC_HW_VERSION:
		return ioread32(sm_dev->addr_bar0 + 0x0C);

	case SIGNAMET_IOC_PCI_BAR2_OFFSET:
		return pci_resource_start(pci_dev, M918X_PCI_DATA_RESOURCE) & ~PAGE_MASK;

	default:
		printk(KERN_ERR "signamet_compat_ioctl failed:  unknown IOC command\n");
		return -ENOTTY;
	}
}

long M918X_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg) {
	struct M918X_dev *sm_dev = filp->private_data;
	
	if (_IOC_TYPE(cmd) != M918X_IOC_MAGIC) {
		printk(KERN_ERR "M918X_compat_ioctl failed: IOC Magic numbers did not match\n");
		return -ENOTTY;
	}
	if (_IOC_NR(cmd) > M918X_IOC_MAXNR) {
		printk(KERN_ERR "M918X_compat_ioctl failed: Cmd number greater then maximum command\n");
		return -ENOTTY;
	}

	// All valid ioctl commands are neither reads nor writes
	if (_IOC_DIR(cmd) & _IOC_READ) {
		printk(KERN_ERR "M918X_compat_ioctl failed: no ioctl read commands accepted\n");
		return -ENOTTY;		
	}
	if (_IOC_DIR(cmd) & _IOC_WRITE){
		printk(KERN_ERR "M918X_compat_ioctl failed: no ioctl write commands accepted\n");
		return -ENOTTY;
	}

	if (is_M918X_series(sm_dev->type))
		return M918X_DMM_compat_ioctl(sm_dev, cmd);
	else if (is_SMX403X_series(sm_dev->type))
		return SMX403X_compat_ioctl(sm_dev, cmd);
	else
	{
		printk(KERN_ERR "M918X_compat_ioctl:  device is not a supported instrument\n");
		return -ENODEV; // there may be more appropriate error code
	}	
}

#ifdef HAVE_UNLOCKED_IOCTL
long M918X_unlocked_ioctl(struct file *filp,
				   unsigned int cmd, unsigned long arg) 
#else
int M918X_ioctl(struct inode *inode, struct file *filp,
				   unsigned int cmd, unsigned long arg) 
#endif
{

	struct M918X_dev *sm_dev = filp->private_data;
	struct pci_dev *pci_dev = &sm_dev->pci_dev;

    	// M918X_IOC_SLOT is implemented differently then all the other ioctls when in 64/32 bit compatibility mode
	if (M918X_IOC_SLOT == cmd) {
		if (!access_ok(VERIFY_WRITE, (void __user *) arg, _IOC_SIZE(cmd))) {
			printk(KERN_ERR "M918X_ioctl failed:  unable to access user provided memory address\n");
			return -EFAULT;
		}
		if (copy_to_user((char *) arg, pci_name(pci_dev), 64)) {
			printk(KERN_ERR "M918X_ioctl failed:  unable to copy slot info to user provided memory address\n");
			return -EFAULT;
		}
		return 0;
	}

	return M918X_compat_ioctl(filp, cmd, arg);
}

int M918X_file_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct M918X_dev *sm_dev = filp->private_data;
	struct pci_dev *pci_dev = &sm_dev->pci_dev;
	unsigned long resource_addr = pci_resource_start(pci_dev, M918X_PCI_DATA_RESOURCE);
	unsigned long resource_size = pci_resource_len(pci_dev, M918X_PCI_DATA_RESOURCE);
	unsigned long offset = resource_addr & ~PAGE_MASK;
	unsigned long required_size = resource_size + offset;
	unsigned long flags = pci_resource_flags(pci_dev, M918X_PCI_DATA_RESOURCE);		

	//ioremap_nocache(start, len);	

	printk(KERN_INFO "M918X_file_mmap %d: %d_%d\n", sm_dev->minor, sm_dev->type, sm_dev->number);

	// pci_resource_flags should be 0x200, indicating a memory resource
	//printk(KERN_INFO "pci_resource_flags %lx\n", flags);
	//printk("resource_addr, resource_size, offset :: %lx, %lx, %lx\n", resource_addr, resource_size, offset);

	if (!resource_addr) {
		printk(KERN_ERR "M918X_file_mmap failed: pci resource not available (pci_resource_flags = %lx)\n", flags);
		return -ENODEV;
	}	

	if (required_size > vma->vm_end - vma->vm_start) {
		printk(KERN_ERR "M918X_file_mmap failed: not enough memory available\n");
		return -EINVAL;
	}

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);

	if (remap_pfn_range
	    (vma, vma->vm_start, resource_addr >> PAGE_SHIFT,
	     required_size, vma->vm_page_prot)) {
		printk(KERN_ERR "M918X_file_mmap - remap_pfn_range failed\n");
		return -EAGAIN;
	}
	return 0;

}

static int get_device_name(struct M918X_dev * sm_dev, char * device_name) {
	if (is_M918X_series(sm_dev->type))
		sprintf(device_name, "M918X_%d", sm_dev->minor);
	else if (is_SMX403X_series(sm_dev->type))
		sprintf(device_name, "SMX403X_%d", sm_dev->minor);
	else
	{ 
		printk(KERN_ERR "M918X_get_device_name:  device is not a supported instrument\n");
		return -ENODEV;
	}
	return 0;
}

static int M918X_setup_chardev(struct M918X_dev *sm_dev)
{
	int result;
	dev_t dev;
	char device_name[16];

	printk(KERN_INFO "M918X_setup_chardev %d: %d_%d\n", sm_dev->minor, sm_dev->type, sm_dev->number);
	
	result = (get_device_name(sm_dev, device_name));
	if (result) return result;

	// if this is the first device being registered for this driver,allocate 
        // a new major number. Otherwise, use the major number that has already been allocated.
	if (M918X_major) {
		dev = MKDEV(M918X_major, sm_dev->minor);
		result = register_chrdev_region(dev, M918X_MINORS_PER_DEVICE, device_name);
	} else {
		result = alloc_chrdev_region(&dev, sm_dev->minor, M918X_MINORS_PER_DEVICE, device_name);
		M918X_major = MAJOR(dev);
	}
	if (result < 0) {
		printk(KERN_ERR "M918X_setup_chardev: cant obtain Major %d\n", M918X_major);
		return result;
	}

	cdev_init(&sm_dev->cdev, &M918X_fops);
	sm_dev->cdev.owner = THIS_MODULE;
	sm_dev->cdev.ops = &M918X_fops;

	result = cdev_add(&sm_dev->cdev, MKDEV(M918X_major, sm_dev->minor), 1);
	if (result)
		printk(KERN_ERR "M918X_setup_chardev:  Error %d adding M918X char device\n", result);
	return result;
}

// return the device type i.e. product model as a human readable integer by passing in the hex value of the subsystem id or device id
int M918X_get_device_type(u32 id) {
	switch (id) {
	case SMX2055_SUBSYSTEM_ID:
  		return 2055;
	case SMX2060_SUBSYSTEM_ID:
		return 2060;
	case SMX2064_SUBSYSTEM_ID:
		return 2064;
	
	case M9181_DEVICE_ID:
		return 9181;
	case M9182_DEVICE_ID:
		return 9182;
	case M9183_DEVICE_ID:
		return 9183;

	case SMX4030_SUBSYSTEM_ID:
		return 4030;
	case SMX4032_SUBSYSTEM_ID:
		return 4032;

	default:
		return -1;
	}

}


// This driver works under the assumption that all probe calls immediately follow init
// and all remove calls follow exit
// This should hold true as these are PCI devices and so the number of devices present
// does not change until system reboot.  
int M918X_device_probe(struct pci_dev *pci_dev,
				  const struct pci_device_id *id)
{
	int ret;
	int err;

	unsigned long start;
	unsigned long len;

	int product_type;

	// bool type.  use of type bool is not standardized on kernels this
	// code may be compiled on, so simply using an int. 
	int isDmm; 

	printk(KERN_INFO "M918X_device_probe\n");
	
	if (M918X_nr_devices == M918X_MAX_NR_DEVICES) {
		printk(KERN_ERR "M918X_device_probe:  maximum number of devices reached\n");
		return -ENOSR;	// if the M918X_devices structure is full.. we cant add the device
	}

	// determine the product type.  For Signametrics products, this is stored in the Subsystem_id field.
	// for Agilent M918X dmms, it is in the device id field.    
	// todo:  also verify proper vendor id and device id
	if (id->vendor == AGILENT_VENDOR_ID)
		product_type = M918X_get_device_type(id->device);
	else
		product_type = M918X_get_device_type(id->subdevice);

	if (is_M918X_series(product_type))
	{
		M918X_devices[M918X_nr_devices].number = M918X_nr_dmms;
		isDmm = 1;
	}
	else if (is_SMX403X_series(product_type))
	{	
		M918X_devices[M918X_nr_devices].number = M918X_nr_switches;
		isDmm = 0;		
	}
	else
	{
		printk(KERN_ERR "M918X_device_probe:  device is not a supported instrument\n");
		return -ENODEV; // there may be more appropriate error code
	}

	ret = pci_enable_device(pci_dev);
	if (ret < 0) {
		printk(KERN_ERR "M918X_device_probe:  pci_enable_device unable to enable device\n");
		return ret;
	}
	// BK, 2nd parameter is 'resource name', and i'm not sure what is the appropriate string is
	// for that parameter.  Driver appears to work regardless of what this string is.

	err = pci_request_regions(pci_dev, M918X_DEVICE_NAME);
	if (err) {
		printk(KERN_ERR "M918X_device_probe:  pci_request_regions failed\n");
		pci_disable_device(pci_dev);
		return err;
	}

	M918X_devices[M918X_nr_devices].pci_dev = *pci_dev;
	M918X_devices[M918X_nr_devices].minor = M918X_nr_devices;
	M918X_devices[M918X_nr_devices].type = product_type;

	ret = M918X_setup_chardev(&M918X_devices[M918X_nr_devices]);
	if (ret < 0) {
		printk(KERN_ERR "M918X_device_probe: setup_chardev failed\n");

		pci_disable_device(pci_dev);
		pci_release_regions(pci_dev);
		return ret;
	}

	start = pci_resource_start(pci_dev, M918X_PCI_CONFIG_RESOURCE);
	len = pci_resource_len(pci_dev, M918X_PCI_CONFIG_RESOURCE);

//	M918X_devices[M918X_nr_devices].addr_bar0 = ioremap(start, len);
	M918X_devices[M918X_nr_devices].addr_bar0 = ioremap_nocache(start, len);

	if (!M918X_devices[M918X_nr_devices].addr_bar0) {
		printk(KERN_ERR "M918X_device_probe:  ioremap failed\n");
		// BHK Note, 12/19/2012, added calls to pci_disable_device and pci_release_regions as is seems appropriate
		pci_disable_device(pci_dev);
		pci_release_regions(pci_dev);
		return -ENOMEM;
	}

	M918X_nr_devices++;
		
	if (isDmm)
		M918X_nr_dmms++;
	else
		M918X_nr_switches++;
	
	return 0;
}

// return the M918X_dev associated with the given pci_dev
// return NULL if no associated M918X_dev is found
static struct M918X_dev *M918X_find_dev(struct pci_dev * pci_dev) {
	int i;

	if (!M918X_devices) return NULL;
	
	for (i=0;i<M918X_MAX_NR_DEVICES;i++) {
		if (strcmp(pci_name(&M918X_devices[i].pci_dev),pci_name(pci_dev)) == 0) {
			return &M918X_devices[i];
		}
	}

	return NULL;

}

void M918X_device_remove(struct pci_dev *pdev)
{
	struct M918X_dev *sm_dev;
	dev_t devno;
	printk(KERN_INFO "M918X_device_remove\n");

	// container_of will not work in this context because
	// the pci_dev is not actually contained inside of the M918X_dev
	// therefore use the M918X_find_dev function
	//sm_dev = container_of(pdev, struct M918X_dev, pci_dev);
	sm_dev = M918X_find_dev(pdev);

	if (!sm_dev) {
		printk(KERN_ERR "M918X_device_remove:  cannot find M918X_dev for the pci_dev being removed\n");
	} else {
		iounmap(sm_dev->addr_bar0);

		devno = MKDEV(M918X_major, sm_dev->minor);
		unregister_chrdev_region(devno, M918X_MINORS_PER_DEVICE);
		cdev_del(&sm_dev->cdev);
	}

	pci_disable_device(pdev);

	pci_release_regions(pdev);

}


static int M918X_pci_init(void)
{
	int driver_major = M918X_DRIVER_MAJOR;
	int driver_minor = M918X_DRIVER_MINOR;
	int driver_build = M918X_DRIVER_BUILD;

	printk(KERN_INFO "M918X_pci_init - version %d.%02d.%03d \n",driver_major, driver_minor, driver_build);

	M918X_devices = kmalloc(M918X_MAX_NR_DEVICES * sizeof(struct M918X_dev), GFP_KERNEL);
	if (!M918X_devices) {
		printk(KERN_ERR "M918X_pci_init:  unable to allocate memory\n");
		return -ENOMEM;
	}
	memset(M918X_devices, 0, M918X_MAX_NR_DEVICES * sizeof(struct M918X_dev));

	return pci_register_driver(&M918X_pci_driver);
}

static void M918X_pci_exit(void)
{
	printk(KERN_INFO "M918X_pci_exit\n");
	pci_unregister_driver(&M918X_pci_driver);

	if (M918X_devices) kfree(M918X_devices);
}

module_init(M918X_pci_init);
module_exit(M918X_pci_exit);
