/*
 *   Copyright 2011, Agilent Technologies. All rights reserved.
 *
 *   Definitions of data structures used by the
 *   M918X kernel module.
 *
 *   This driver provides a hardware interface to the following instruments:
 *
 *   Agilent M9181, M9182, M9183 Digital Multimeters
 *   Signametrics SM2060, SM2064, SMX2055, SMX2060, SMX2064 Digital Multimeters
 *  
 */

#ifndef _M918X_DRV_H
#define _M918X_DRV_H

#ifdef __cplusplus
extern "C" {
#endif

#include <linux/version.h>

#include "M918X_id.h"
#include "M918X_user.h"

#define M918X_MAX_NR_DEVICES 32

	// bar number used for normal communications..
#define M918X_pci_bar  2

	// this resource provides access to various config registers
	// mostly accessed by the user through ioctl
#define M918X_PCI_CONFIG_RESOURCE 0

	// this resource is accessed by the user through mmap
	// and is used for all normal   inbyte outbyte operations 
	// that are passed to the dmm through the uart
#define M918X_PCI_DATA_RESOURCE 2

	// the number of minor device numbers needed by each device
#define M918X_MINORS_PER_DEVICE 1

	int M918X_file_open(struct inode *inode, struct file *filp);
	int M918X_file_release(struct inode *inode, struct file *filp);
	int M918X_file_mmap(struct file *filp,
			     struct vm_area_struct *vma);
#ifdef HAVE_UNLOCKED_IOCTL
	long M918X_unlocked_ioctl(struct file *filp,
			 unsigned int cmd, unsigned long arg);
#else
	int M918X_ioctl(struct inode *inode, struct file *filp,
			 unsigned int cmd, unsigned long arg);
#endif
	long M918X_compat_ioctl(struct file *filp,
			 unsigned int cmd, unsigned long arg);



	int M918X_device_probe(struct pci_dev *dev,
				const struct pci_device_id *id);
	void M918X_device_remove(struct pci_dev *dev);

	//struct pci_device_id  M918X_pci_device_id[] = 
	static struct pci_device_id M918X_pci_device_id[] __devinitdata = {
		//  {VENDOR_ID, DEVICE_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
		{SMX206X_VENDOR_ID, SMX206X_DEVICE_ID, SMX206X_VENDOR_ID, SMX2055_SUBSYSTEM_ID, 0, 0, 0},
		{SMX206X_VENDOR_ID, SMX206X_DEVICE_ID, SMX206X_VENDOR_ID, SMX2060_SUBSYSTEM_ID, 0, 0, 0},
		{SMX206X_VENDOR_ID, SMX206X_DEVICE_ID, SMX206X_VENDOR_ID, SMX2064_SUBSYSTEM_ID, 0, 0, 0},
		{SMX403X_VENDOR_ID, SMX403X_DEVICE_ID, SMX403X_VENDOR_ID, SMX4030_SUBSYSTEM_ID, 0, 0, 0},
		{SMX403X_VENDOR_ID, SMX403X_DEVICE_ID, SMX403X_VENDOR_ID, SMX4032_SUBSYSTEM_ID, 0, 0, 0},
		{AGILENT_VENDOR_ID, M9181_DEVICE_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
		{AGILENT_VENDOR_ID, M9182_DEVICE_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
		{AGILENT_VENDOR_ID, M9183_DEVICE_ID, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
		{}		// end of list
	};

	struct file_operations M918X_fops = {
		.owner = THIS_MODULE,
// in more recent kernels .ioctl has been removed and replaced with .unlocked_ioctl
#ifdef HAVE_UNLOCKED_IOCTL
		.unlocked_ioctl = M918X_unlocked_ioctl,
#else
		.ioctl = M918X_ioctl,
#endif
		.compat_ioctl = M918X_compat_ioctl,
		.open = M918X_file_open,
		.release = M918X_file_release,
		.mmap = M918X_file_mmap,
	};



	static struct pci_driver M918X_pci_driver = {
		.name = M918X_DRIVER_NAME,
		.id_table = M918X_pci_device_id,
		.probe = M918X_device_probe,
		.remove = M918X_device_remove,
		.suspend = NULL,
		.resume = NULL,
#ifndef SUSE
#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,23))
		.enable_wake = NULL,
#endif
#endif
		.shutdown = NULL,
	};


	struct M918X_dev {
		int type; //9181, 2064, 4032, etc
		int number; // 0, 1, 2, etc.. 0 indicates the first instrument of int type registered
		struct semaphore sem;
		struct cdev cdev;
		struct pci_dev pci_dev;
		void __iomem *addr_bar0;
		int minor;
	};

#ifdef __cplusplus
}
#endif


#endif				/* _M918X_DRV_H */
