/***************************************************************************** 
  hp548xx Instrument Driver EXAMPLE PROGRAM #1                                
                                                                              
  Illustrates the use of the standard VPP 3.1 functions:                      
                                                                              
      hp548xx_init                                                            
      hp548xx_close                                                           
      hp548xx_reset                                                           
      hp548xx_revision_query                                                  
      hp548xx_self_test                                                       
      hp548xx_error_message                                                   
      hp548xx_error_query                                                     

                                                                              
  Illustrates the use of several HP utility functions:                        
                                                                              
      hp548xx_errorQueryDetect                                                
           Sets the driver to check the instrument for errors.                

      hp548xx_cmd                                                             

           Sends a command string to the instrument.  In this example,        
           it sends a command which generates an error which then             
           illustrates the use of hp548xx_error_query and                     
           hp548xx_error_message (in the check_err macro).It is also used       
           to send a command that clears event status registers.              
                                                                              

  One of the most interesting things about this example is the error          
    detection portion, most of the second half of the program.  After         
    trying it out, you might think that setting errorQueryDetect to           
    true is a great idea!  In fact, it's not a bad idea for the hp548xx,        
    but beware of the potential performance impact.  With errorQueryDetect    
    set on, the instrument is queried at least once, if not several times,    

    for all functions which access the instrument.  If your application       
    is sensitive to performance, this might make the application too slow.    

  One effective strategy is to create and debug your application with         
    errorQueryDetect set on, then remove it and leave it off for production   
    applications.                                                             
                                                                              

 *****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "hp548xx.h"

void check_err (ViSession vi, ViStatus errStatus)
{
    ViInt32	inst_err;
    ViChar	err_message[256];

    if(VI_SUCCESS > errStatus)
    {
            /* Send a device clear to ensure communication with     		*/
            /*   the instrument.                                    		*/

        hp548xx_dcl(vi);


            /* If the driver is set to detect instrument errors,    		*/
            /*   and an instrument error is detected, the error     		*/
            /*   code is hp548xx_INSTR_ERROR_DETECTED (see  				*/
            /*   hp548xx.h).  In this case, query the instrument   			*/
            /*   for the error and display it.                      		*/
            /* Otherwise, the error is a driver error.  Query the   		*/
            /*   driver for the error and display it.               		*/

        if(hp548xx_INSTR_ERROR_DETECTED == errStatus)	
        {
            hp548xx_error_query(vi, &inst_err, err_message); 
            hp548xx_error_message(vi, inst_err, err_message); 
            printf("Instrument Error : %ld, %s\n", inst_err, err_message); 
        }
        else
        {
            hp548xx_error_message(vi, errStatus, err_message); 
            printf("Driver Error : %ld, %s\n", errStatus, err_message);
        }

            /* Optionally reset the instrument, close the          		*/
            /*   instrument handle, and exit the program.         		*/
    /* hp548xx_reset(vi); 	*/
    /* hp548xx_close(vi); 	*/
    /* exit(errStatus);    		*/	
    }
return;
}

void main ( )
{

ViStatus    err_status;
ViSession   vi;
ViChar      err_msg[256];
ViInt32     inst_err_nbr;
ViChar      inst_err_msg[256];
ViChar      driver_rev[50];
ViChar      instr_rev[50];

ViInt16     test_result;
ViChar      test_message[100];

/*  Initialize the instrument.  Note that this function will verify   	*/
/*    that the instrument specified is an hp548xx (id_query=VI_TRUE) 	*/
/*    and will send a reset to the instrument (do_reset=VI_TRUE).     	*/

err_status = hp548xx_init("GPIB0::7::INSTR", VI_TRUE, VI_TRUE, &vi);
if (   ( err_status < VI_SUCCESS )
    || ( vi == VI_NULL ) )
{
    printf("\ninit failed with return code %d.\n", err_status);
    if ( vi != VI_NULL )
    {
        hp548xx_error_message(vi,err_status,err_msg);
        printf("  Error Status:  %d\n", err_status);
        printf("  Error Message: %s\n", err_msg);
    }
    exit (err_status);
}

/*  Query the instrument for instrument revision and the driver for  	*/
/*    driver revision, and print both.                               	*/

err_status = hp548xx_revision_query(vi, driver_rev, instr_rev);
check_err(vi, err_status);

printf("  Driver Revision     : %s\n", driver_rev);
printf("  Instrument Revision : %s\n", instr_rev);


/*	Set the Timeout to 50 seconds 										*/
err_status = hp548xx_timeOut(vi, 50000);
check_err(vi, err_status);


/*  Run an instrument self-test.  Print the test result (0 or an     	*/
/*    instrument error number) and a message describing the error.   	*/

err_status = hp548xx_self_test(vi, &test_result, test_message);
check_err(vi, err_status);

printf("  Self Test Result    : %d\n", test_result);
printf("  Self Test Message   : %s\n", test_message);


/*  Reset the instrument.                                      			*/

err_status = hp548xx_reset(vi);
check_err(vi, err_status);


/*  Execute a SCPI command which is guaranteed to fail after reset.  	*/
/*    Note that the driver does not query the instrument for the     	*/
/*    error, and so no error is reported.  Instrument errors are     	*/
/*    not reported if errorQueryDetect is set to false (the          	*/
/*    default).                                                      	*/
/*  Further on in this example, errorQueryDetect is set to true and  	*/
/*    the same command is run to show how to detect instrument       	*/
/*    errors.                                                        	*/ 

err_status = hp548xx_cmd(vi, ":VIEW CHAN5");
check_err(vi, err_status);


/*  Clear the instrument's event status registers.  This in effect   	*/
/*    clears the instrument error.                                   	*/
/*  If this is not executed, the error_QueryDetect function below    	*/
/*    will report an error after it executes - the error from the    	*/
/*    command executed above.                                        	*/

err_status = hp548xx_cmd(vi, "*CLS");
check_err(vi, err_status);

/*  Execute a reset.  This executes *RST.                            	*/
/*    Note that the reset will not clear the instrument error.       	*/

err_status = hp548xx_reset(vi);
check_err(vi, err_status);

/*  Turn on error detection.  If error detection is not turned on,   	*/
/*    the driver will not attempt to detect instrument errors.       	*/

err_status = hp548xx_errorQueryDetect(vi, VI_TRUE);
check_err(vi, err_status);

/*  Execute a SCPI command which is guaranteed to fail after reset.  	*/
/*    This will force the execution of hp548xx_error_message, which  	*/
/*    returns driver errors, and hp548xx_error_query, which returns  	*/
/*    instrument errors.  Both of these functions are called in the  	*/
/*    macro check_err.                                                  */

err_status = hp548xx_cmd(vi, ":VIEW CHAN5");
check_err(vi, err_status);

/*  Reading instrument errors pops the error queue in the instrument 	*/
/*  If the instrument is queried again, the queue is empty.          	*/

err_status = hp548xx_error_query(vi,&inst_err_nbr, inst_err_msg);
err_status = hp548xx_error_message(vi,inst_err_nbr, inst_err_msg);
printf("\n  Inst. Error Number   : %d\n", inst_err_nbr);
printf("  Inst. Error Message  : %s\n", inst_err_msg);

/*  Close the instrument.                                            	*/

err_status = hp548xx_close(vi);
printf("\nProgram complete.");
}
