/*****************************************************************************/
/*  hp548xx.c                                                                */
/*  Copyright (C) 1998 Hewlett-Packard Company                               */
/*---------------------------------------------------------------------------*/
/*  Driver for hp548xx, Hewlett Packard Infiniium Oscilloscopes hp548xx      */
/*  Driver Version:                                                          */
/*---------------------------------------------------------------------------*/
/*  This driver is compatible with the following VXIplug&play standards:     */
/*    WIN32 System Framework revision 4.0                                    */
/*    VISA revision 1.0 (does not work with VTL)                             */
/*****************************************************************************/

#define hp548xx_REV_CODE    "A.00.02"                /* Driver Revision */

/*****************************************************************************/
/* Exactly one of the following must be defined for the program to           */
/*   compile correctly.                                                      */
/*                                                                           */
/* #define __hpux          (for compilation for HP-UX)                       */
/* #define WIN32           (for compilation for 32-bit Windows)              */
/*                                                                           */
/* In addition, if WIN32 is defined, _CVI_ may also be defined to indicate   */
/*   that the program is being compiled under LabWindows CVI 4.0 or          */
/*   greater.                                                                */
/*****************************************************************************/

/*****************************************************************************/
/* The following defines are intended to provide OS compatibility among      */
/*   Windows 32-bit and HP-UX C compilers.                                   */
/*****************************************************************************/

#ifdef __hpux
#define _fcvt    fcvt
#endif

#define _huge

/*****************************************************************************/
/* Instrument specific header files - for hp548xx  - Begin                   */
/* Instrument specific header files - for hp548xx  - End                     */

#include <stdlib.h>           /* prototype for malloc()  */
#include <string.h>           /* prototype for strcpy()  */
#include <stdio.h>            /* prototype for sprintf() */
#include <math.h>             /* prototype for pow()     */
/*#include <varargs.h>           prototype for vararg stuff in the      */
                              /*   delayQueryf and delayRead functions  */

#ifdef __hpux
#define _INCLUDE_HPUX_SOURCE                    USUALLY NEEDED
#define _INCLUDE_XOPEN_SOURCE_EXTENDED          USUALLY NEEDED
#include <sys/time.h>
#else
#include <windows.h>
#endif

#include "visa.h"
#include "hp548xx.h"

#define hp548xx_ERR_MSG_LENGTH 256   /* size of error message buffer     */
#define ZERO_DELAY          0       /* Number of milliseconds to delay. */
                                    /*   See doDelay function.          */


/*****************************************************************************/
/*  The string below is used to do the instrument identification in the      */
/*    init routine.  The string matches the first part of the instrument's   */
/*    response to it's *IDN? command.                                        */
/*****************************************************************************/

#define hp548xx_IDN_STRING  "HEWLETT-PACKARD,548"

/*****************************************************************************/
/*  VISA Globals data structure.                                             */
/*  It is desireable to store global variables in VISA rather than the       */
/*    driver.  This avoids having any state variables in the driver.  This   */
/*    structure defines the driver's global data.  Memory is allocated to    */
/*    this structure in the init function, which passes the pointer to VISA, */
/*    and every function which follows retrieves a pointer to it.            */
/*****************************************************************************/

struct hp548xx_globals
{
    ViStatus     errNumber;
    char         errMessage[160];
    ViBoolean    errQueryDetect;
    ViInt32      dTime;
    ViUInt16     interfaceType;
        
        /**************************/
        /*  Instrument Specific   */
        /**************************/

};  

/*****************************************************************************/
/*  Driver Message Strings.                                                  */
/*****************************************************************************/

#define hp548xx_MSG_VI_OPEN_ERR            "vi was zero.  Was the hp548xx_init() successful?"
#define hp548xx_MSG_CONDITION              "condition"
#define hp548xx_MSG_EVENT                  "event"	
#define hp548xx_MSG_EVENT_HDLR_INSTALLED   "event handler is already installed for event happening"
#define hp548xx_MSG_EVENT_HDLR_INST2       "Only 1 handler can be installed at a time."	
#define hp548xx_MSG_INVALID_HAPPENING      "is not a valid happening."
#define hp548xx_MSG_NOT_QUERIABLE          "is not queriable."	
#define hp548xx_MSG_INVALID_STATUS         "Parameter 2 is invalid in function hp548xx_error_message()."
#define hp548xx_MSG_INVALID_STATUS_VALUE   "is not a valid viStatus value."
#define hp548xx_MSG_INVALID_VI             "Parameter 1 is invalid in function hp548xx_error_message()."  \
                                           "Using an inactive ViSession may cause this error."	\
                                           "Was the instrument driver closed prematurely?"
#define hp548xx_MSG_NO_ERRORS              "No Errors"
#define hp548xx_MSG_SELF_TEST_FAILED       "Self test failed." 	
#define hp548xx_MSG_SELF_TEST_PASSED       "Self test passed."

#define hp548xx_MSG_BOOLEAN                "Expected 0 or 1; Got %hd"
#define hp548xx_MSG_REAL                   "Expected %lg to %lg; Got %lg"
#define hp548xx_MSG_INT                    "Expected %hd to %hd; Got %hd"
#define hp548xx_MSG_LONG                   "Expected %ld to %ld; Got %ld"
#define hp548xx_MSG_LOOKUP                 "Error converting string response to integer"
#define hp548xx_MSG_NO_MATCH               "Could not match string %s"

#define VI_ERROR_PARAMETER1_MSG           "Parameter 1 is invalid"
#define VI_ERROR_PARAMETER2_MSG           "Parameter 2 is invalid"
#define VI_ERROR_PARAMETER3_MSG           "Parameter 3 is invalid"
#define VI_ERROR_PARAMETER4_MSG           "Parameter 4 is invalid"
#define VI_ERROR_PARAMETER5_MSG           "Parameter 5 is invalid"
#define VI_ERROR_PARAMETER6_MSG           "Parameter 6 is invalid"
#define VI_ERROR_PARAMETER7_MSG           "Parameter 7 is invalid"
#define VI_ERROR_PARAMETER8_MSG           "Parameter 8 is invalid"
#define VI_ERROR_PARAMETER9_MSG           "Parameter 9 is invalid"
#define VI_ERROR_PARAMETER10_MSG          "Parameter 10 is invalid"
#define VI_ERROR_PARAMETER11_MSG          "Parameter 11 is invalid"
#define VI_ERROR_PARAMETER12_MSG          "Parameter 12 is invalid"
#define VI_ERROR_PARAMETER13_MSG          "Parameter 13 is invalid"
#define VI_ERROR_PARAMETER14_MSG          "Parameter 14 is invalid"
#define VI_ERROR_PARAMETER15_MSG          "Parameter 15 is invalid"
#define VI_ERROR_PARAMETER16_MSG          "Parameter 16 is invalid"
#define VI_ERROR_PARAMETER17_MSG          "Parameter 17 is invalid"
#define VI_ERROR_PARAMETER18_MSG          "Parameter 18 is invalid"
#define VI_ERROR_FAIL_ID_QUERY_MSG        "Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG       "ViSession (parameter 1) was not created by this driver"
#define INSTR_ERROR_NULL_PTR_MSG          "NULL pointer detected"
#define INSTR_ERROR_RESET_FAILED_MSG      "reset failed"
#define INSTR_ERROR_UNEXPECTED_MSG        "An unexpected error occurred"
#define INSTR_ERROR_DETECTED_MSG          "Instrument Error Detected, call hp548xx_error_query()."
#define INSTR_ERROR_LOOKUP_MSG            "String not found in table"
#define INSTR_ERROR_NO_LAST_COMMA         "no last comma found in IDN string"

#define FILE_OPEN_ERROR_MSG               "File I/O error - could not open specified file"
#define FILE_READ_ERROR_MSG               "File I/O error - could not read from file"
#define FILE_WRITE_ERROR_MSG              "File I/O error - could not write to file"
#define INSTR_ERROR_CMD_ERROR_MSG    	  "Generic syntax error"
#define INSTR_ERROR_INV_CHAR_MSG 		  "A syntactic element contains a character invalid for that type"
#define INSTR_ERROR_SYNTAX_MSG   		  "Unrecognized command or data type was encountered"
#define INSTR_ERROR_INV_SEPARATOR_MSG	  "Expecting separator but found illegal character"
#define INSTR_ERROR_DATA_TYPE_MSG		  "Found data element different than one allowed"
#define INSTR_ERROR_GET_ALLOW_MSG		  "A Group Execute Triggr was received within a program message"
#define INSTR_ERROR_PARAM_ALLOW_MSG       "More parameters than expected were received"
#define INSTR_ERROR_MISS_PARAM_MSG   	  "Fewer parameters than required were received"
#define INSTR_ERROR_MNEMONIC_LONG_MSG	  "Header or character data contains moe than 12 elements"
#define INSTR_ERROR_UNDEF_HEADER_MSG 	  "Header is syntactically correct but undefined"
#define INSTR_ERROR_INV_CHAR_NUM_MSG 	  "An invalid character for the data type being parsed was found"
#define INSTR_ERROR_NUM_OVERFLOW_MSG 	  "Number is too large or too small to be represented"
#define INSTR_ERROR_TOO_MANY_DIG_MSG 	  "The mantissa of a decimal data contained more than 255 digits"
#define INSTR_ERROR_NUM_DATA_ALLOW_MSG    "Legal numeric data element on illegal header position"
#define INSTR_ERROR_INV_SUFFIX_MSG   	  "Invalid or inappropriate sufix for the oscilloscope"
#define INSTR_ERROR_SUFFIX_ALLOW_MSG 	  "Suffix after a numeric element that does not allow suffixes"
#define INSTR_ERROR_INV_CHAR_DATA_MSG	  "Character data element is invalid or contains invalid character"
#define INSTR_ERROR_CHAR_DATA_LONG_MSG    "Character data is too long"
#define INSTR_ERROR_CHAR_DATA_ALLOW_MSG   "Character data element prohibited by the oscilloscope"
#define INSTR_ERROR_STR_DATA_MSG 	  	  "Generic string data error"
#define INSTR_ERROR_INV_STR_DATA_MSG 	  "String data element invalid for some reason"
#define INSTR_ERROR_STR_DATA_ALLOW_MSG    "String data element not allowed at the point of parsing"
#define INSTR_ERROR_BLOCK_DATA_MSG   	  "Generic block data error"
#define INSTR_ERROR_INV_BLOCK_DATA_MSG    "Invalid block data"
#define INSTR_ERROR_BLOCK_DATA_ALLOW_MSG  "Block data not allowed at this point in parsing"
#define INSTR_ERROR_EXPRESSION_MSG   	  "Generic expression error"
#define INSTR_ERROR_INV_EXP_MSG  		  "Invalid expression"
#define INSTR_ERROR_EXP_DATA_ALLOW_MSG    "Expression data not allowed at this point in parsing"
#define INSTR_ERROR_EXECUTION_MSG 		  "Generic execution error"
#define INSTR_ERROR_DATA_OUT_RANGE_MSG    "Data value is outside legal range"
#define INSTR_ERROR_TOO_MUCH_DATA_MSG	  "More data than the oscilloscope can handle was received"
#define INSTR_ERROR_ILL_PARAM_VALUE_MSG   "Illegal parameter value"
#define INSTR_ERROR_HW_MISING_MSG		  "Hardware missing"
#define INSTR_ERROR_FILE_NFOUND_MSG  	  "File name not found"
#define INSTR_ERROR_SYSTEM_MSG	 		  "System error has occurred"
#define INSTR_ERROR_QUEUE_OVERFLOW_MSG    "Error occurred but there was no room in the error queue"
#define INSTR_ERROR_QUERY_MSG			  "Generic query error"
#define INSTR_ERROR_Q_INTERRUPTED_MSG	  "Query interrupted"
#define INSTR_ERROR_Q_UNTERMINATED_MSG    "Query unterminated"
#define INSTR_ERROR_Q_DEADLOCKED_MSG 	  "Query deadlocked"
#define INSTR_ERROR_Q_UNTERMINATED_R_MSG  "Query unterminated after indefinite response"

/*****************************************************************************/
/*  Error table structure.                                                   */
/*    The first element of the table is the error number, the second is the  */
/*    error message.  Error numbers in the "VISA Generic" section of the     */
/*    table are defined in the VISA header files.  Error numbers in the      */
/*    "Instrument Specific" section are defined in the driver header file.   */
/*    All of the error messages are defined above.                           */
/*****************************************************************************/

struct instrErrStruct
{
	ViStatus errStatus;
	ViString errMessage;
};

const static struct instrErrStruct instrErrMsgTable[] =
{
        /**************************/
        /*  Template Messages     */
        /**************************/

    { VI_ERROR_FAIL_ID_QUERY,             VI_ERROR_FAIL_ID_QUERY_MSG },
    { VI_ERROR_PARAMETER1,                VI_ERROR_PARAMETER1_MSG },
    { VI_ERROR_PARAMETER2,                VI_ERROR_PARAMETER2_MSG },
    { VI_ERROR_PARAMETER3,                VI_ERROR_PARAMETER3_MSG },
    { VI_ERROR_PARAMETER4,                VI_ERROR_PARAMETER4_MSG },
    { VI_ERROR_PARAMETER5,                VI_ERROR_PARAMETER5_MSG },
    { VI_ERROR_PARAMETER6,                VI_ERROR_PARAMETER6_MSG },
    { VI_ERROR_PARAMETER7,                VI_ERROR_PARAMETER7_MSG },
    { VI_ERROR_PARAMETER8,                VI_ERROR_PARAMETER8_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER9,      VI_ERROR_PARAMETER9_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER10,     VI_ERROR_PARAMETER10_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER11,     VI_ERROR_PARAMETER11_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER12,     VI_ERROR_PARAMETER12_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER13,     VI_ERROR_PARAMETER13_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER14,     VI_ERROR_PARAMETER14_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER15,     VI_ERROR_PARAMETER15_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER16,     VI_ERROR_PARAMETER16_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER17,     VI_ERROR_PARAMETER17_MSG },
    { hp548xx_INSTR_ERROR_PARAMETER18,     VI_ERROR_PARAMETER18_MSG },
    { hp548xx_INSTR_ERROR_INV_SESSION,     INSTR_ERROR_INV_SESSION_MSG },
    { hp548xx_INSTR_ERROR_NULL_PTR,        INSTR_ERROR_NULL_PTR_MSG },
    { hp548xx_INSTR_ERROR_RESET_FAILED,    INSTR_ERROR_RESET_FAILED_MSG },
    { hp548xx_INSTR_ERROR_UNEXPECTED,      INSTR_ERROR_UNEXPECTED_MSG },
    { hp548xx_INSTR_ERROR_DETECTED,        INSTR_ERROR_DETECTED_MSG },
    { hp548xx_INSTR_ERROR_LOOKUP,          INSTR_ERROR_LOOKUP_MSG },
    { hp548xx_INSTR_NO_LAST_COMMA,         INSTR_ERROR_NO_LAST_COMMA }, 

        /**************************/
        /*  Instrument Specific   */
        /**************************/
	{ hp548xx_FILE_OPEN_ERROR,		        FILE_OPEN_ERROR_MSG 			},
	{ hp548xx_FILE_READ_ERROR,		  	    FILE_READ_ERROR_MSG 			},
	{ hp548xx_FILE_WRITE_ERROR,    	    	FILE_WRITE_ERROR_MSG 			},
	{ hp548xx_INSTR_ERROR_CMD_ERROR,    	INSTR_ERROR_CMD_ERROR_MSG    	},				
	{ hp548xx_INSTR_ERROR_INV_CHAR, 		INSTR_ERROR_INV_CHAR_MSG 		},				
	{ hp548xx_INSTR_ERROR_SYNTAX,   		INSTR_ERROR_SYNTAX_MSG   		},				
	{ hp548xx_INSTR_ERROR_INV_SEPARATOR,	INSTR_ERROR_INV_SEPARATOR_MSG	},				
	{ hp548xx_INSTR_ERROR_DATA_TYPE,		INSTR_ERROR_DATA_TYPE_MSG		},				
	{ hp548xx_INSTR_ERROR_GET_ALLOW,		INSTR_ERROR_GET_ALLOW_MSG		},				
	{ hp548xx_INSTR_ERROR_PARAM_ALLOW,      INSTR_ERROR_PARAM_ALLOW_MSG     },				
	{ hp548xx_INSTR_ERROR_MISS_PARAM,   	INSTR_ERROR_MISS_PARAM_MSG   	},				
	{ hp548xx_INSTR_ERROR_MNEMONIC_LONG,	INSTR_ERROR_MNEMONIC_LONG_MSG	},				
	{ hp548xx_INSTR_ERROR_UNDEF_HEADER, 	INSTR_ERROR_UNDEF_HEADER_MSG 	},				
	{ hp548xx_INSTR_ERROR_INV_CHAR_NUM, 	INSTR_ERROR_INV_CHAR_NUM_MSG 	},				
	{ hp548xx_INSTR_ERROR_NUM_OVERFLOW, 	INSTR_ERROR_NUM_OVERFLOW_MSG 	},				
	{ hp548xx_INSTR_ERROR_TOO_MANY_DIG, 	INSTR_ERROR_TOO_MANY_DIG_MSG 	},				
	{ hp548xx_INSTR_ERROR_NUM_DATA_ALLOW,   INSTR_ERROR_NUM_DATA_ALLOW_MSG  },				
	{ hp548xx_INSTR_ERROR_INV_SUFFIX,   	INSTR_ERROR_INV_SUFFIX_MSG   	},				
	{ hp548xx_INSTR_ERROR_SUFFIX_ALLOW, 	INSTR_ERROR_SUFFIX_ALLOW_MSG 	},				
	{ hp548xx_INSTR_ERROR_INV_CHAR_DATA,	INSTR_ERROR_INV_CHAR_DATA_MSG	},				
	{ hp548xx_INSTR_ERROR_CHAR_DATA_LONG,   INSTR_ERROR_CHAR_DATA_LONG_MSG  },				
	{ hp548xx_INSTR_ERROR_CHAR_DATA_ALLOW,  INSTR_ERROR_CHAR_DATA_ALLOW_MSG },				
	{ hp548xx_INSTR_ERROR_STR_DATA, 	  	INSTR_ERROR_STR_DATA_MSG 	  	},				
	{ hp548xx_INSTR_ERROR_INV_STR_DATA, 	INSTR_ERROR_INV_STR_DATA_MSG 	},				
	{ hp548xx_INSTR_ERROR_STR_DATA_ALLOW,   INSTR_ERROR_STR_DATA_ALLOW_MSG  },				
	{ hp548xx_INSTR_ERROR_BLOCK_DATA,   	INSTR_ERROR_BLOCK_DATA_MSG   	},				
	{ hp548xx_INSTR_ERROR_INV_BLOCK_DATA,   INSTR_ERROR_INV_BLOCK_DATA_MSG  },				
	{ hp548xx_INSTR_ERROR_BLOCK_DATA_ALLOW, INSTR_ERROR_BLOCK_DATA_ALLOW_MSG},				
	{ hp548xx_INSTR_ERROR_EXPRESSION,   	INSTR_ERROR_EXPRESSION_MSG   	},				
	{ hp548xx_INSTR_ERROR_INV_EXP,  		INSTR_ERROR_INV_EXP_MSG  		},				
	{ hp548xx_INSTR_ERROR_EXP_DATA_ALLOW,   INSTR_ERROR_EXP_DATA_ALLOW_MSG  },				
	{ hp548xx_INSTR_ERROR_EXECUTION, 		INSTR_ERROR_EXECUTION_MSG 		},				
	{ hp548xx_INSTR_ERROR_DATA_OUT_RANGE,   INSTR_ERROR_DATA_OUT_RANGE_MSG  },				
	{ hp548xx_INSTR_ERROR_TOO_MUCH_DATA,	INSTR_ERROR_TOO_MUCH_DATA_MSG	},				
	{ hp548xx_INSTR_ERROR_ILL_PARAM_VALUE,  INSTR_ERROR_ILL_PARAM_VALUE_MSG },				
	{ hp548xx_INSTR_ERROR_HW_MISING,		INSTR_ERROR_HW_MISING_MSG		},				
	{ hp548xx_INSTR_ERROR_FILE_NFOUND,  	INSTR_ERROR_FILE_NFOUND_MSG  	},				
	{ hp548xx_INSTR_ERROR_SYSTEM,	 		INSTR_ERROR_SYSTEM_MSG	 		},				
	{ hp548xx_INSTR_ERROR_QUEUE_OVERFLOW,   INSTR_ERROR_QUEUE_OVERFLOW_MSG  },				
	{ hp548xx_INSTR_ERROR_QUERY,			INSTR_ERROR_QUERY_MSG			},				
	{ hp548xx_INSTR_ERROR_Q_INTERRUPTED,	INSTR_ERROR_Q_INTERRUPTED_MSG	},				
	{ hp548xx_INSTR_ERROR_Q_UNTERMINATED,   INSTR_ERROR_Q_UNTERMINATED_MSG  },				
	{ hp548xx_INSTR_ERROR_Q_DEADLOCKED, 	INSTR_ERROR_Q_DEADLOCKED_MSG 	},				
	{ hp548xx_INSTR_ERROR_Q_UNTERMINATED_R, INSTR_ERROR_Q_UNTERMINATED_R_MSG}				
};


/*****************************************************************************/
/*****************************************************************************/
/********************                                     ********************/
/**********                                                         **********/
/****                   SUPPORTING MACROS AND FUNCTIONS                   ****/
/**                                                                         **/
/**-------------------------------------------------------------------------**/
/**     These macros and functions are not exposed to the user, but         **/
/**     are used by other functions in the driver.                          **/
/****                                                                     ****/
/**********                                                         **********/
/********************                                     ********************/
/*****************************************************************************/
/*****************************************************************************/



/*****************************************************************************/
/*  SWAP MACROS                                                              */
/*****************************************************************************/
/*  Reorders bytes where instrument byte ordering of integers and reals      */
/*    differs from the computing framework.                                  */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    dest                                                                   */
/*      The number to be reordered.  The reordered result is also stored in  */
/*      this variable on completion.                                         */
/*****************************************************************************/

#define SWAP_FLOAT64(dest)                                         \
 { unsigned char    src[8];                                        \
      *((double *)src) = *((double *)dest);                        \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[7];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[6];  \
      ((unsigned char *)(dest))[2] =  ((unsigned char*)(src))[5];  \
      ((unsigned char *)(dest))[3] =  ((unsigned char*)(src))[4];  \
      ((unsigned char *)(dest))[4] =  ((unsigned char*)(src))[3];  \
      ((unsigned char *)(dest))[5] =  ((unsigned char*)(src))[2];  \
      ((unsigned char *)(dest))[6] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[7] =  ((unsigned char*)(src))[0];  \
 }

#define SWAP_32(dest)                                              \
 { unsigned char    src[4];                                        \
      *((long *)src) = *((long *)dest);                            \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[3];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[2];  \
      ((unsigned char *)(dest))[2] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[3] =  ((unsigned char*)(src))[0];  \
 }

#define SWAP_16(dest)                                              \
 { unsigned char    src[2];                                        \
      *((int *)src) = *((int *)dest);                              \
      ((unsigned char *)(dest))[0] =  ((unsigned char*)(src))[1];  \
      ((unsigned char *)(dest))[1] =  ((unsigned char*)(src))[0];  \
 }


/*****************************************************************************/
/*  MACRO GET_GLOBALS                                                        */
/*****************************************************************************/
/*  Returns a pointer to the VISA globals storage.  This storage was         */
/*    allocated in init, but it is turned over to VISA to avoid holding      */
/*    state variables in the driver.                                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession            instrumentHandle                                  */
/*      The instrument's VISA address.                                       */
/*    struct hp548xx_globals  *thisPtr                                       */
/*      A pointer to the VISA globals storage                                */
/*    funcname                                                               */
/*      The name of the function to be stored with the context dependent     */
/*      error information.                                                   */
/*****************************************************************************/

#define GET_GLOBALS(instrumentHandle,thisPtr)                                \
{                                                                            \
    errStatus = viGetAttribute(instrumentHandle, VI_ATTR_USER_DATA,          \
                                                      (ViAddr) &thisPtr);    \
    if (errStatus < VI_SUCCESS)                                              \
        return errStatus;                                                    \
}


/*****************************************************************************/
/*  DELAY Functions                                                          */
/*  setDelay (located prior to init function) sets the delay time for the    */
/*    instrument.                                                            */
/*  doDelay uses delay time to slow down the driver after a read request.    */
/*---------------------------------------------------------------------------*/
/*  In some SCPI instruments, the instrument cannot change its state to      */
/*    'processing a read request' fast enough to keep another request        */
/*    from getting through to the instrument.  By setting a short delay      */
/*    (less than 100 microseconds should be sufficient) after each read,     */
/*    the driver will wait a sufficient amount of time before attempting     */
/*    to send the next string to the instrument.                             */
/*  DO NOT document these function in the fp file or the help file.  They    */
/*    are intended to address problems in specific instruments, problems     */
/*    that may only appear on faster CPU's.  They are designed so that the   */
/*    problem can be addressed quickly in the field once it is identified.   */
/*  If the problem is identified by driver developers, a 100 ms delay        */
/*    should be set in the hp548xx_init function.  The template version of   */
/*    init currently sets the delay to 0.                                    */
/*****************************************************************************/

/*****************************************************************************/
static void doDelay(
        ViInt32 delayTime)
/*****************************************************************************/
/*  Delays execution for a specified number of microseconds.                 */
/*  Platform   Code    Tick Resolution  Tick Increment  Call Overhead        */
/*   WIN32     32-bit  CPU dependent    CPU dependent   0-20 us              */
/*   HP-UX     32-bit  1 us             CPU dependent   0-20 us              */
/*  PARAMETERS                                                               */
/*    long           delayTime (in)                                          */
/*      The number of micro-seconds to delay before proceeding.              */
/*****************************************************************************/

{
long   ticksToCount;

        /*************************************************************/
        /*  WIN32 Version                                            */
        /*  WIN32 increments depend on the hardware.  If high-       */
        /*    frequency counting isn't supported, revert to common   */
        /*    standard functions (same as WIN16).                    */
        /*  The call overhead to execute doDelay if high-frequency   */
        /*    is supported is about 25-30 micro-seconds.             */
        /*  The call overhead to execute doDelay if high-frequency   */
        /*    is -NOT- supported is ~10 milli-seconds on             */
        /*    Windows 95 & NT.                                       */
        /*************************************************************/
#ifdef WIN32

#ifdef _CVI_
#else
__int64  ticksPerSecond;
__int64  startTicks;
__int64  nowTicks;
#endif
DWORD    msTicks;

if (delayTime == 0)
    return;

#ifdef _CVI_
#else
if (QueryPerformanceFrequency((LARGE_INTEGER *)&ticksPerSecond))
{
    ticksToCount = (long)((ticksPerSecond/1E6)*delayTime) + 1;
    QueryPerformanceCounter((LARGE_INTEGER *)&nowTicks);
    startTicks = nowTicks;
    while (nowTicks - startTicks < (DWORD)ticksToCount)
        QueryPerformanceCounter((LARGE_INTEGER *)&nowTicks);
}
else
#endif
{
    ticksToCount = (long)((1E3/1E6)*delayTime) + 1;
    msTicks = GetTickCount();
    while (GetTickCount() - msTicks < (DWORD)ticksToCount) ;
}

#endif

        /*************************************************************/
        /*  HP-UX Version                                            */
        /*  HP-UX counts in 1 micro-second increments.               */
        /*  The call overhead to execute doDelay on HP-UX is about   */
        /*    15-20 micro-seconds.                                   */
        /*************************************************************/
#ifdef __hpux

struct timeval t0, t1;

if (delayTime == 0)
    return;

/* printf("doDelay delaying %ld\n", ticks);  */ 
gettimeofday(&t0, NULL);
t0.tv_usec += delayTime;
t0.tv_sec += t0.tv_usec / 1000000;
t0.tv_usec = t0.tv_usec % 1000000;

gettimeofday(&t1, NULL);
while (t1.tv_sec < t0.tv_sec)
    gettimeofday(&t1, NULL);

if (t1.tv_sec > t0.tv_sec) return; /* get out quick if past delay time */

while (t1.tv_usec < t0.tv_usec)
    gettimeofday(&t1, NULL);

#endif

return;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus delayVScanf(
        ViSession  instrumentHandle,
        ViString   scanFmt,
        ... ) 
/*****************************************************************************/
/*  Same as viVScanf, but with a delay added if necessary.                   */
/*                                                                           */
/*  PARAMETERS (same as viVScanf)                                            */
/*    ViSession   instrumentHandle (in)                                      */
/*    ViString    scanFmt (in)                                               */
/*    ...                                                                    */
/*****************************************************************************/

{
ViStatus  errStatus;
va_list   va;
struct hp548xx_globals *thisPtr;
   
if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_USER_DATA,
                                             (ViAddr) &thisPtr)) < VI_SUCCESS)
    return errStatus;

va_start(va, scanFmt);
errStatus = viVScanf(instrumentHandle, scanFmt, va);
va_end(va);

if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

return errStatus;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus delayVPrintf(
        ViSession  instrumentHandle,
        ViString   printFmt,
        ... ) 
/*****************************************************************************/
/*  Same as viVPrintf, but with a delay added if necessary.                  */
/*                                                                           */
/*  PARAMETERS (same as viVPrintf)                                           */
/*    ViSession   instrumentHandle (in)                                      */
/*    ViString    printFmt (in)                                              */
/*    ...                                                                    */
/*****************************************************************************/

{
ViStatus  errStatus;
va_list   va;
struct hp548xx_globals *thisPtr;
   
if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_USER_DATA,
                                             (ViAddr) &thisPtr)) < VI_SUCCESS)
    return errStatus;

va_start(va, printFmt);
errStatus = viVPrintf(instrumentHandle, printFmt, va);
va_end(va);

if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

return errStatus;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus delayVQueryf(
        ViSession  instrumentHandle,
        ViString   printFmt,
        ViString   scanFmt,
        ... ) 
/*****************************************************************************/
/*  Same as viVQueryf, but with a delay added if necessary.                  */
/*                                                                           */
/*  PARAMETERS (same as viVQueryf)                                           */
/*    ViSession   instrumentHandle (in)                                      */
/*    ViString    printFmt (in)                                              */
/*    ViString    scanFmt (in)                                               */
/*    ...                                                                    */
/*****************************************************************************/

{
ViStatus  errStatus;
va_list   va;
struct hp548xx_globals *thisPtr;
   
if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_USER_DATA,
                                             (ViAddr) &thisPtr)) < VI_SUCCESS)
    return errStatus;

va_start(va, scanFmt);
errStatus = viVQueryf(instrumentHandle, printFmt, scanFmt, va);
va_end(va);

if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

return errStatus;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus delayRead(
        ViSession  instrumentHandle,
        ViPBuf     buf,
        ViUInt32   cnt,
        ViPUInt32  retCnt) 
/*****************************************************************************/
/*  Same as viRead, but with a delay added if necessary.                     */
/*                                                                           */
/*  PARAMETERS (same as viRead)                                              */
/*    ViSession   instrumentHandle (in)                                      */
/*    ViPBuf      buf (in/out)                                               */
/*    ViUInt32    cnt (in)                                                   */
/*    ViPUInt32   retCnt (in)                                                */
/*****************************************************************************/

{
ViStatus  errStatus;
struct hp548xx_globals *thisPtr;
   
if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_USER_DATA,
                                             (ViAddr) &thisPtr)) < VI_SUCCESS)
    return errStatus;

errStatus = viRead(instrumentHandle, buf, cnt, retCnt);

if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

return errStatus;
}        /* ----- end of function ----- */


/*****************************************************************************/
#define hp548xx_LOG_STATUS( instrumentHandle, thisPtr, status )		 	\
  return statusUpdate( instrumentHandle, thisPtr, status )  

static ViStatus statusUpdate(
        ViSession  instrumentHandle,
        struct     hp548xx_globals *thisPtr,
        ViStatus   s ) 
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession              instrumentHandle (in)                           */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViStatus               s (in)                                          */
/*      A driver error return code.                                          */
/*****************************************************************************/

{   
ViStatus  errStatus;
ViInt16   eventQ;

if (thisPtr)
    thisPtr->errNumber = s;

        /*************************************************************/
        /*  If error query detect is set on and the instrument       */
        /*    has not timed out, then ...                            */
        /*  Read the status byte into eventQ and check the error     */
        /*    bytes.                                                 */
        /*  Potential status byte errors are 0x04 (Query Error),     */
        /*    0x08 (Device Dependent Error), 0x10 (Execution Error), */
        /*    and 0x20 (Command Error).                              */
        /*************************************************************/

if (thisPtr && thisPtr->errQueryDetect && s != VI_ERROR_TMO)
{   
    if ((errStatus = delayVQueryf(instrumentHandle, "*ESR?\n", "%hd%*t",
                                                       &eventQ)) < VI_SUCCESS)
        return VI_ERROR_SYSTEM_ERROR;

    if( (0x04  | 0x08  | 0x10  | 0x20) & eventQ )
        return hp548xx_INSTR_ERROR_DETECTED;
}

return s;
}        /* ----- end of function ----- */


/*****************************************************************************/
/*  MACRO CHK_BOOLEAN                                                        */
/*****************************************************************************/
/*  Ref chk_boolean for info.                                                */
/*****************************************************************************/

#define CHK_BOOLEAN(chk_val, err)                          \
if (chk_boolean(thisPtr, chk_val))                         \
   {                                                       \
    thisPtr->errNumber = err;                              \
    return err;                                            \
   }


/*****************************************************************************/
static ViBoolean chk_boolean(
        struct     hp548xx_globals *thisPtr,
        ViBoolean  chk_val)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViBoolean              chk_val (in)                                    */
/*      Check this value to make sure it is a legal Boolean.                 */
/*****************************************************************************/

{
ViChar message[hp548xx_ERR_MSG_LENGTH];

if ( (chk_val != VI_TRUE) && (chk_val != VI_FALSE) )
{
    sprintf(message, hp548xx_MSG_BOOLEAN, chk_val);	
    strcpy(thisPtr->errMessage, message);				
    return VI_TRUE;
}

return VI_FALSE;
}        /* ----- end of function ----- */


/*****************************************************************************/
/*  MACRO CHK_REAL_RANGE                                                     */
/*****************************************************************************/
/*  Ref chk_real_range for info.                                             */
/*****************************************************************************/

#define CHK_REAL_RANGE(chk_val, min, max, err)             \
if (chk_real_range(thisPtr, chk_val, min, max))            \
   {                                                       \
    thisPtr->errNumber = err;                              \
    return err;                                            \
   }


/*****************************************************************************/
static ViBoolean chk_real_range( 
        struct    hp548xx_globals *thisPtr,
        ViReal64  chk_val,
        ViReal64  min,
        ViReal64  max)
/*****************************************************************************/
/*  Tests to see if a ViReal64 is in range.                                  */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViReal64             chk_val (in)                                      */
/*      The value to be checked.                                             */
/*    ViReal64             min (in)                                          */
/*      The bottom of the range.                                             */
/*    ViReal64             max (in)                                          */
/*      The top of the range.                                                */
/*****************************************************************************/

{
ViChar message[hp548xx_ERR_MSG_LENGTH];

if ( (chk_val < min) || (chk_val > max) )  			
{								
    sprintf(message, hp548xx_MSG_REAL, min, max, chk_val);	
    strcpy(thisPtr->errMessage, message);
    return VI_TRUE;
}

return VI_FALSE;
}        /* ----- end of function ----- */ 
 
  
/*****************************************************************************/
/*  MACRO CHK_INT_RANGE                                                      */
/*****************************************************************************/
/*  Ref chk_int_range for info.                                              */
/*****************************************************************************/

#define CHK_INT_RANGE(chk_val, min, max, err)              \
if (chk_int_range(thisPtr, chk_val, min, max) )            \
   {                                                       \
    thisPtr->errNumber = err;                                \
    return err;                                            \
   }


/*****************************************************************************/
static ViBoolean chk_int_range( 
        struct   hp548xx_globals *thisPtr,
        ViInt16  chk_val,
        ViInt16  min,
        ViInt16  max)
/*****************************************************************************/
/*  Tests to see if a ViInt16 is in range.                                   */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViInt16              chk_val (in)                                      */
/*      The value to be checked.                                             */
/*    ViInt16              min (in)                                          */
/*      The bottom of the range.                                             */
/*    ViInt16              max (in)                                          */
/*      The top of the range.                                                */
/*****************************************************************************/

{
ViChar message[hp548xx_ERR_MSG_LENGTH];

if ( (chk_val < min) || (chk_val > max) )  			
{								
    sprintf(message, hp548xx_MSG_INT, min, max, chk_val);	
    strcpy(thisPtr->errMessage, message);
    return VI_TRUE;
}

return VI_FALSE;
}        /* ----- end of function ----- */ 
   
   
/*****************************************************************************/
/*  MACRO CHK_LONG_RANGE                                                     */
/*****************************************************************************/
/*  Ref chk_long_range for info.                                             */
/*****************************************************************************/

#define CHK_LONG_RANGE(chk_val, min, max, err)             \
if (chk_long_range(thisPtr, chk_val, min, max))            \
   {                                                       \
    thisPtr->errNumber = err;                              \
    return err;                                            \
   }


/*****************************************************************************/
static ViBoolean chk_long_range( 
        struct  hp548xx_globals *thisPtr,
        ViInt32 chk_val,
        ViInt32 min,
        ViInt32 max)
/*****************************************************************************/
/*  Tests to see if a ViInt32 is in range.                                   */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViInt32              chk_val (in)                                      */
/*      The value to be checked.                                             */
/*    ViInt32              min (in)                                          */
/*      The bottom of the range.                                             */
/*    ViInt32              max (in)                                          */
/*      The top of the range.                                                */
/*****************************************************************************/

{
ViChar message[hp548xx_ERR_MSG_LENGTH];

if ( (chk_val < min) || (chk_val > max) )  			
{								
    sprintf(message, hp548xx_MSG_LONG, min, max, chk_val);	
    strcpy(thisPtr->errMessage, message);
    return VI_TRUE;
}

return VI_FALSE;
}        /* ----- end of function ----- */ 
   
   
/*****************************************************************************/
/*  MACRO CHK_ENUM                                                           */
/*****************************************************************************/
/*  Ref chk_enum for info.                                                   */
/*****************************************************************************/

#define CHK_ENUM( chk_val, limit, err )                    \
if (chk_enum( thisPtr, chk_val, limit) )                   \
   {                                                       \
    thisPtr->errNumber = err;                                \
    return err;                                            \
   }


/*****************************************************************************/
static ViBoolean chk_enum (
        struct hp548xx_globals *thisPtr,
        ViInt16 chk_val,
        ViInt16 limit)
/*****************************************************************************/
/* Chk_enum searches for a string in an array of strings.  It is used by     */
/* the CHK_ENUM macro                                                        */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    struct hp548xx_globals *thisPtr (in)                                   */
/*      A pointer to the VISA globals for this VISA session                  */
/*    ViInt16              chk_val (in)                                      */
/*      The value to be checked.                                             */
/*    ViInt16              limit (in)                                        */
/*      The upper limit of the enumeration values.                           */
/*****************************************************************************/

{
ViChar message[hp548xx_ERR_MSG_LENGTH];

if ( ( chk_val < 0 ) || (chk_val > limit) )  			
{								
    sprintf(message, hp548xx_MSG_INT, 0, limit, chk_val);	
    strcpy(thisPtr->errMessage, message);
    return VI_TRUE;
}

return VI_FALSE;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus _VI_FUNC setDelay (
        ViSession  instrumentHandle,
        ViInt32    delayTime)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViInt32    delayTime (in)                                              */
/*      The delayTime in microseconds.  Must be between 0 and 1,000.  In     */
/*      practice, should never be set higher than 100.                       */
/*****************************************************************************/

{
ViStatus errStatus = VI_SUCCESS ;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

CHK_LONG_RANGE(delayTime, 0L, 1000L, VI_ERROR_PARAMETER2);

thisPtr->dTime = delayTime;

thisPtr->errNumber = VI_SUCCESS;
return VI_SUCCESS;
}        /* ----- end of function ----- */


/*****************************************************************************/
static ViStatus _VI_FUNC initError(
        ViSession  vi,
        ViSession  defRM,
        ViStatus   errStatus)
/*****************************************************************************/
/*  This function is only called from hp548xx_init.                          */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession      vi (in)                                                 */
/*      Instrument Handle returned by viOpen.                                */
/*    ViSession      defRM (in)                                              */
/*      Resource Manager handle from viOpen.                                 */
/*    ViStatus       errStatus (in)                                          */
/*      A driver error return code.                                          */
/*****************************************************************************/

{
viClose(vi);
viClose(defRM);
vi=VI_NULL;
return errStatus;
}        /* ----- end of function ----- */


/*****************************************************************************/
/*****************************************************************************/
/********************                                     ********************/
/**********                                                         **********/
/****                          DRIVER FUNCTIONS                           ****/
/**                                                                         **/
/**-------------------------------------------------------------------------**/
/**     The functions which are exposed to the driver user start here.      **/
/**     They are in the following order:                                    **/
/**        VPP required functions.                                          **/
/**        HP required utility functions                                    **/
/**        HP passthrough functions                                         **/
/**        Instrument specific functions                                    **/
/****                                                                     ****/
/**********                                                         **********/
/********************                                     ********************/
/*****************************************************************************/
/*****************************************************************************/




/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_init(
        ViRsrc resourceName,
        ViBoolean IDQuery,
        ViBoolean resetDevice,
        ViPSession instrumentHandle)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession      resourceName (in)                                       */
/*      The instrument's VISA address.                                       */
/*      GPIB Example: "GPIB0::7::INSTR"                                      */
/*    ViBoolean      IDQuery (in)                                            */
/*      If VI_TRUE, validate instrument identity                             */
/*    ViBoolean      resetDevice (in)                                        */
/*      If VI_TRUE, perform an instrument reset                              */
/*    ViPSession     instrumentHandle (out)                                  */
/*      Pointer to Instrument Handle.  Returns VI_NULL if the init fails.    */
/*****************************************************************************/

{
struct     hp548xx_globals *thisPtr;
ViStatus   errStatus;
ViSession  defRM;
ViChar     idn_buf[256];

*instrumentHandle = VI_NULL;

        /*************************************************************/
        /* Find the Default Resource Manager.  Potential errors are  */
        /*   VI_ERROR_SYSTEM_ERROR, VI_ERROR_ALLOC                   */
        /*************************************************************/

if ((errStatus = viOpenDefaultRM(&defRM)) < VI_SUCCESS)
    return errStatus;

        /*************************************************************/
        /* Open a session to the instrument.  Potential errors are   */
        /*   VI_ERROR_NSUP_OPER, VI_ERROR_INV_RSRC_NAME,             */
        /*   VI_ERROR_INV_ACC_MODE, VI_ERROR_RSRC_NFOUND,            */
        /*   VI_ERROR_ALLOC                                          */
        /*************************************************************/

if ((errStatus = viOpen(defRM, resourceName, VI_NULL, VI_NULL,
                                              instrumentHandle)) < VI_SUCCESS)
{
    viClose(defRM);
    *instrumentHandle=VI_NULL;
    return errStatus;
}

        /*************************************************************/
        /* Allocate global memory for the session.  Inititalize the  */
        /*   memory.  Note that viSetAttribute should never fail,    */
        /*   as all of it's input conditions are satisfied.          */
        /*************************************************************/

if ((thisPtr =
         (struct hp548xx_globals *)malloc(sizeof(struct hp548xx_globals))) == 0)
{
    viClose(*instrumentHandle);
    viClose(defRM);
    *instrumentHandle=VI_NULL;
    return VI_ERROR_ALLOC;
}

if ((errStatus = viSetAttribute(*instrumentHandle, VI_ATTR_USER_DATA,
                                          (ViAttrState)thisPtr)) < VI_SUCCESS)
{
    viClose(*instrumentHandle);
    viClose(defRM);
    *instrumentHandle=VI_NULL;
    return errStatus;
}

        /*************************************************************/
        /* Set initial values of variables in global memory.         */
        /*************************************************************/

thisPtr->errNumber = VI_SUCCESS;
thisPtr->errMessage[0] = 0;
thisPtr->errQueryDetect = VI_FALSE;
thisPtr->dTime = 300;

if ((errStatus = viGetAttribute(*instrumentHandle, VI_ATTR_INTF_TYPE,
                                       &thisPtr->interfaceType)) < VI_SUCCESS)
    return initError(*instrumentHandle, defRM, errStatus);


        /*************************************************************/
        /* Validate the instrument's identity.                       */
        /*************************************************************/

if( VI_TRUE == IDQuery ) 
{
    if (thisPtr->interfaceType == VI_INTF_GPIB)
        if (viClear(*instrumentHandle) <  VI_SUCCESS)
            return initError(*instrumentHandle, defRM, VI_ERROR_FAIL_ID_QUERY);

    if (   (delayVQueryf(*instrumentHandle, "*IDN?\n", "%t",
                                                        idn_buf) < VI_SUCCESS)   
        || (strncmp(idn_buf, hp548xx_IDN_STRING, strlen(hp548xx_IDN_STRING))) ) 
        return initError(*instrumentHandle, defRM, VI_ERROR_FAIL_ID_QUERY);
}

        /*************************************************************/
        /* Reset the instrument if so instructed.                    */
        /*   GPIB reset includes a 488.2 device clear.               */
        /*************************************************************/

if( VI_TRUE == resetDevice ) 
{
    if (thisPtr->interfaceType == VI_INTF_GPIB)
        if (viClear(*instrumentHandle) <  VI_SUCCESS)
            return initError(*instrumentHandle, defRM,
                                             hp548xx_INSTR_ERROR_RESET_FAILED);

    if (hp548xx_reset(*instrumentHandle) <  VI_SUCCESS)
        return initError(*instrumentHandle, defRM,
                                             hp548xx_INSTR_ERROR_RESET_FAILED);
}


if ((errStatus = delayVPrintf(*instrumentHandle,":SYST:HEAD OFF\n")) < VI_SUCCESS)
    return statusUpdate(*instrumentHandle, thisPtr, errStatus);


return statusUpdate(*instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_close( 
        ViSession instrumentHandle)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession      instrumentHandle (in)                                   */
/*      Instrument Handle returned from hp548xx_init()                       */
/*****************************************************************************/

{
struct hp548xx_globals *thisPtr;
ViStatus errStatus = VI_SUCCESS ;
ViSession defRM;

GET_GLOBALS(instrumentHandle, thisPtr)

        /*************************************************************/
        /* Retrieve Resource Management session, free global memory  */
        /*   for the session (if it exists), and close the vi and    */
        /*   RM sessions.                                            */
        /*************************************************************/

if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_RM_SESSION, &defRM))
                                                                 < VI_SUCCESS)
{
    thisPtr->errNumber = errStatus;
    return errStatus;
}

if (thisPtr)		
    free(thisPtr);

return viClose(defRM);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_reset( 
        ViSession instrumentHandle)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession      instrumentHandle (in)                                   */
/*      Instrument Handle returned from hp548xx_init()                       */
/*****************************************************************************/

{
struct hp548xx_globals *thisPtr;
ViStatus errStatus = VI_SUCCESS  ;

GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVPrintf(instrumentHandle, "*RST\n")) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_self_test(
        ViSession instrumentHandle,
        ViPInt16  selfTestResult,
        ViPString selfTestMessage)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession   instrumentHandle (in)                                      */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPInt16    selfTestResult (out)                                       */
/*      Self-test result                                                     */
/*    ViPString   selfTestMessage[] (out)                                    */
/*      Self-test status message string, limited to 256 characters.          */
/*****************************************************************************/

{
struct hp548xx_globals *thisPtr;
ViStatus errStatus = VI_SUCCESS ;
int cnt;

*selfTestResult = 0; 
selfTestMessage[0] = 0; 

GET_GLOBALS(instrumentHandle, thisPtr)

   
/*    
if ((errStatus = delayVQueryf(instrumentHandle, "*TST?\n", "%hd%*t",
                                                selfTestResult)) < VI_SUCCESS)
{
    *selfTestResult = -1; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
}

switch( *selfTestResult)
{
    case 0:
        sprintf(selfTestMessage, hp548xx_MSG_SELF_TEST_PASSED);
        break;

    default:
        sprintf(selfTestMessage, hp548xx_MSG_SELF_TEST_FAILED);
        break;
}
*/
/****************************/
if ((errStatus = delayVPrintf(instrumentHandle, "SELF:SCOPETEST\n")) < VI_SUCCESS)
{
    *selfTestResult = -1; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
}

doDelay (200000000);

if ((errStatus = delayVPrintf(instrumentHandle, "SELF:SCOPETEST?\n")) < VI_SUCCESS)
{
    *selfTestResult = -1; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
} 
if ((errStatus = delayRead(instrumentHandle,(ViPBuf)selfTestMessage,50 ,&cnt)) < VI_SUCCESS)
{
    *selfTestResult = -1; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
} 

/****************************/

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_error_query(
        ViSession  instrumentHandle,
        ViPInt32   errorCode,
        ViPString  errorMessage)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession    instrumentHandle (in)                                     */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPInt32     errorCode (out)                                           */
/*      An instrument error code                                             */
/*    ViPString    errorMessage[] (out)                                      */
/*      Instrument error message string, limited to 256 characters.          */
/*****************************************************************************/

{
struct hp548xx_globals *thisPtr;
ViStatus errStatus = VI_SUCCESS ;

*errorCode = -1; 
errorMessage[0] = 0; 

GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVQueryf( instrumentHandle, ":SYST:ERR? STR\n", "%ld,%t",
                                       errorCode, errorMessage)) < VI_SUCCESS)
{
    *errorCode = -1; 
    errorMessage[0] = 0; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
}

	/* get rid of extra LF at the end of the error_message */
/*errorMessage[strlen(errorMessage)-1] = 0;*/

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_error_message(
        ViSession  instrumentHandle,
        ViStatus   statusCode,
        ViPString  message)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession   instrumentHandle (in)                                      */
/*      Instrument Handle returned from hp548xx_init()                       */
/*      May be VI_NULL for this function                                     */
/*    ViStatus    statusCode (in)                                            */
/*      The error return value from an instrument driver function            */
/*    ViPString   message[] (out)                                            */
/*      Error message string.  This is limited to 256 characters.            */
/*****************************************************************************/

{
struct    hp548xx_globals *thisPtr;
ViStatus  errStatus;  
ViInt32   idx;

message[0] = 0;

thisPtr = NULL;
if (instrumentHandle != VI_NULL)
    GET_GLOBALS(instrumentHandle, thisPtr)

if (statusCode == VI_SUCCESS)
{
    sprintf(message, hp548xx_MSG_NO_ERRORS);
    thisPtr->errNumber = VI_SUCCESS;
    return VI_SUCCESS;
}

        /*************************************************************/
        /* Search the error message table to see if the error is in  */
        /*   that table.  If it is, copy the corresponding error     */
        /*   message to the output error message.  If context        */
        /*   dependent error messages are being used, append the     */
        /*   context dependent information to the message.           */
        /* If the error is found in the table, exit the routine      */
        /*   successfully.                                           */
        /*************************************************************/

for (idx=0; idx < (sizeof instrErrMsgTable / 
	           sizeof(struct instrErrStruct)); idx++)
{
    if (instrErrMsgTable[idx].errStatus == statusCode)
    {
        if ((thisPtr) && (thisPtr->errNumber == statusCode))
        {
            sprintf(message, "%s;  %s",
                    instrErrMsgTable[idx].errMessage,
                    thisPtr->errMessage);
        }
        else
            strcpy(message, instrErrMsgTable[idx].errMessage);
        
        thisPtr->errNumber = VI_SUCCESS;
        return VI_SUCCESS;
    }
}

        /*************************************************************/
        /* Check to see if the error is a VTL/VISA error number      */
        /*   using viStatusDesc, which will also return the error    */
        /*   message.  If context dependent error messages are       */
        /*   used, append the context dependent information to the   */
        /*   message.                                                */
        /* If the error is found in the table, exit the routine      */
        /*   successfully.                                           */
        /*************************************************************/

errStatus = viStatusDesc(instrumentHandle, statusCode, message);
if (errStatus == VI_SUCCESS)
{
    if( (thisPtr) && (thisPtr->errNumber == statusCode))
    {
        strcat(message, ";  ");
        strcat(message, thisPtr->errMessage);
    }

    thisPtr->errNumber = VI_SUCCESS;
    return VI_SUCCESS;
}

        /*************************************************************/
        /*   At this point vi is either INVALID or VI_NULL           */
        /*************************************************************/

if (instrumentHandle == VI_NULL)
{
    strcpy(message, hp548xx_MSG_VI_OPEN_ERR);
    thisPtr->errNumber = VI_SUCCESS;
    return VI_SUCCESS;
} 

sprintf(message, hp548xx_MSG_INVALID_STATUS "  %ld"
	  hp548xx_MSG_INVALID_STATUS_VALUE, (long)statusCode);
	
thisPtr->errNumber = VI_ERROR_PARAMETER2;
return VI_ERROR_PARAMETER2;
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_revision_query(
        ViSession    instrumentHandle,
        ViPString    instrumentDriverRevision,
        ViPString    firmwareRevision)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession    instrumentHandle (in)                                     */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPString    instrumentDriverRevision[] (out)                          */
/*      Instrument driver revision.  This is limited to 256 characters.      */
/*    ViPString    firmwareRevision[] (out)                                  */
/*      Instrument firmware revision.  This is limited to 256 characters.    */
/*****************************************************************************/

{
struct    hp548xx_globals *thisPtr;
ViStatus  errStatus;
ViChar    temp_str1[256];         /* temp hold for instr rev string */
ViChar   *last_comma;             /* last comma in *IDN string */
int len;
	
	
GET_GLOBALS(instrumentHandle, thisPtr)

instrumentDriverRevision[0] = 0; 
firmwareRevision[0] = 0; 

sprintf(instrumentDriverRevision, "%s", hp548xx_REV_CODE);

        /*************************************************************/
        /* In the standard c template, the scan format string is     */
        /*   "%t%*t".  This removes white space from the end of the  */
        /*   returned string, but doesn't work if there are          */
        /*   embedded spaces in the returned string.  I used this    */
        /*   format string and resorted to stripping off spaces and  */
        /*   spaces and trailing non-printing characters below.      */
        /*************************************************************/

if ((errStatus = delayVQueryf( instrumentHandle, "*IDN?\n", "%t", temp_str1))
                                                                 < VI_SUCCESS)
{
    instrumentDriverRevision[0] = 0; 
    firmwareRevision[0]  = 0; 
    return statusUpdate(instrumentHandle, thisPtr, errStatus); 
}
    
        /*************************************************************/
        /* error and exit if last comma not found                    */
        /*************************************************************/

last_comma = strrchr(temp_str1,',');

if (!last_comma) 
{									    
    firmwareRevision[0] = 0;
    strcpy(thisPtr->errMessage, INSTR_ERROR_NO_LAST_COMMA);
    thisPtr->errNumber = hp548xx_INSTR_NO_LAST_COMMA;
    return hp548xx_INSTR_NO_LAST_COMMA; 
}

        /*************************************************************/
        /* The next statement assumes that the firmware revision of  */
        /*   the instrument follows the last comma/space pair.       */
        /*************************************************************/
        /* JMH Note: eliminate the comma at the end of the line.     */
        /*************************************************************/

strcpy(firmwareRevision, last_comma+1);
len=strlen(firmwareRevision)-1;
firmwareRevision[len]=0;

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS); 
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_timeOut (
        ViSession  instrumentHandle,
        ViInt32    setTimeOut)
/*****************************************************************************/
/*  Sets the current setting of the driver timeout value of the instrument   */
/*  in milliseconds.                                                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViInt32    setTimeOut (in)                                             */
/*      The period of time which VISA will allow an instrument command to    */
/*      wait before stopping it's execution.                                 */
/*****************************************************************************/

{
ViStatus errStatus = 0;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

CHK_LONG_RANGE(setTimeOut, hp548xx_TIMEOUT_MIN,
                                     hp548xx_TIMEOUT_MAX, VI_ERROR_PARAMETER2);

if ((errStatus = viSetAttribute(instrumentHandle, VI_ATTR_TMO_VALUE,
                                                    setTimeOut)) < VI_SUCCESS)
{
    thisPtr->errNumber = errStatus;
    return errStatus;
}

thisPtr->errNumber = VI_SUCCESS;
return VI_SUCCESS;
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_timeOut_Q (
        ViSession  instrumentHandle,
        ViPInt32   timeOut)
/*****************************************************************************/
/*  Returns the current setting of the timeout value of the instrument in    */
/*  milliseconds.                                                            */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPInt32   timeOut (out)                                               */
/*      The period of time which VISA will allow an instrument command to    */
/*      wait before stopping it's execution.                                 */
/*****************************************************************************/

{
ViStatus errStatus = 0;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = viGetAttribute(instrumentHandle, VI_ATTR_TMO_VALUE,
                                                       timeOut)) < VI_SUCCESS)
{
    thisPtr->errNumber = errStatus;
    return errStatus;
}

thisPtr->errNumber = VI_SUCCESS;
return VI_SUCCESS;
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_errorQueryDetect( 
        ViSession  instrumentHandle,
        ViBoolean  setErrorQueryDetect)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViBoolean  setErrorQueryDetect (in)                                    */
/*      Enables (VI_TRUE) or disables (VI_FALSE) automatic instrument error  */
/*      querying                                                             */
/*****************************************************************************/

{
ViStatus errStatus = 0;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

CHK_BOOLEAN(setErrorQueryDetect, VI_ERROR_PARAMETER2);

thisPtr->errQueryDetect = setErrorQueryDetect;

thisPtr->errNumber = VI_SUCCESS;
return VI_SUCCESS;
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_errorQueryDetect_Q( 
        ViSession   instrumentHandle,
        ViPBoolean  errorQueryDetect)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession   instrumentHandle (in)                                      */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPBoolean  errorQueryDetect (out)                                     */
/*      Equals VI_TRUE if automatic instrument error querying is being       */
/*      being performed.                                                     */
/*****************************************************************************/

{
ViStatus errStatus = 0;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

*errorQueryDetect = thisPtr->errQueryDetect;

thisPtr->errNumber = VI_SUCCESS;
return VI_SUCCESS;
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_dcl( 
        ViSession  instrumentHandle)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession instrumentHandle (in)                                        */
/*      Instrument Handle returned from hp548xx_init()                       */
/*****************************************************************************/

{
ViStatus errStatus = 0;
struct hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

if (thisPtr->interfaceType == VI_INTF_GPIB)
    if ((errStatus = viClear(instrumentHandle)) < VI_SUCCESS)
        return statusUpdate( instrumentHandle, thisPtr, errStatus);	

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */

 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_opc_Q(
        ViSession   instrumentHandle,
        ViPBoolean  instrumentReady)
/*****************************************************************************/
/*  Sends an *OPC? command to the instrument and returns VI_TRUE when all    */
/*  pending operations are complete.                                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession   instrumentHandle (in)                                      */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViPBoolean  instrumentReady (out)                                      */
/*      Returns VI_TRUE when pending operations are complete.                */
/*****************************************************************************/

{

ViStatus errStatus=VI_SUCCESS;

struct hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

thisPtr->dTime = 200;

if ((errStatus = delayVQueryf(instrumentHandle, "*OPC?\n", "%hd%*t",
                                               instrumentReady)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*
if ((errStatus = delayVPrintf(instrumentHandle,"*OPC?\n")) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

if ((errStatus = delayVScanf(instrumentHandle, "%hd%*t",
                                               instrumentReady)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
*/

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_readStatusByte_Q(
        ViSession  instrumentHandle,
        ViPInt16   statusByte)
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession instrumentHandle (in)                                        */
/*      Instrument Handle returned from hp548xx_init()                       */
/*    ViPInt16  statusByte (out)                                             */
/*      Returns the contents of the status byte                              */
/*****************************************************************************/

{
ViStatus  errStatus = 0;
struct    hp548xx_globals *thisPtr;
ViUInt16  stb;

GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = viReadSTB(instrumentHandle, &stb)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus );

*statusByte = (ViInt16)stb;

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmd( 
        ViSession instrumentHandle,
        ViString  sendStringCommand)
/*****************************************************************************/
/*  Send a scpi command, it does not look for a response                     */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession     instrumentHandle (in)                                    */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString      sendStringCommand (in)                                   */
/*      The SCPI command string to be sent to the instrument                 */
/*****************************************************************************/

{
ViStatus errStatus = VI_SUCCESS ;
struct   hp548xx_globals *thisPtr;

GET_GLOBALS(instrumentHandle, thisPtr)

errStatus = delayVPrintf(instrumentHandle,"%s\n", sendStringCommand);
   
return statusUpdate(instrumentHandle, thisPtr, errStatus);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdString_Q(
        ViSession       instrumentHandle,
        ViString        queryStringCommand,
        ViInt32         stringSize,
        ViChar _VI_FAR  stringResult[] )
/*****************************************************************************/
/*  PARAMETERS                                                               */
/*    ViSession       instrumentHandle (in)                                  */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString        queryStringCommand (in)                                */
/*      The SCPI command string to be sent to the instrument                 */
/*    ViInt32         stringSize (in)                                        */
/*      The size of the char array (result) passed to the function to hold   */
/*      the string returned by the instrument                                */
/*    ViChar _VI_FAR  stringResult[] (out)                                   */
/*      The string returned by the instrument                                */
/*****************************************************************************/

{
ViStatus  errStatus;
struct    hp548xx_globals *thisPtr;
int       sz;

GET_GLOBALS(instrumentHandle, thisPtr)

        /* Command strings must have at least one non-null character */
if(stringSize < 2)
   {
    thisPtr->errNumber = VI_ERROR_PARAMETER2;
    return VI_ERROR_PARAMETER2;
   }

sz = (int)stringSize;
if ((errStatus = delayVQueryf(instrumentHandle, "%s\n", "%#t",
                         queryStringCommand, &sz, stringResult)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
stringResult[sz]='\0';
      
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdInt(
        ViSession  instrumentHandle,
        ViString   sendIntegerCommand,
        ViInt32    sendInteger )
/*****************************************************************************/
/*  Sends an instrument command which takes a single integer parameter.      */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   sendIntegerCommand (in)                                     */
/*      The instrument command string to be sent to the instrument.          */
/*    ViPInt32   sendInteger (in)                                            */
/*      The integer sent to the instrument at the end of the instrument      */
/*      command.  Can be ViInt16 or ViInt32.                                 */
/*****************************************************************************/

{  
ViStatus errStatus = VI_SUCCESS ;
struct hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVPrintf(instrumentHandle, "%s %ld\n", sendIntegerCommand,
                                                   sendInteger)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdInt16_Q(
        ViSession  instrumentHandle,
        ViString   queryI16Command,
        ViPInt16   i16Result)
/*****************************************************************************/
/*  Sends scpi command and waits for a response that must be representable   */
/*    as an int16.  A non numeric instrument response returns zero in        */
/*    result.                                                                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   queryI16Command (in)                                        */
/*      The SCPI command string to be sent to the instrument.                */
/*    ViPInt16   i16Result (out)                                             */
/*      The integer returned from the instrument.                            */
/*****************************************************************************/

{
ViStatus errStatus;
struct   hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVQueryf(instrumentHandle, "%s\n", "%hd%*t",
                                    queryI16Command, i16Result)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdInt32_Q(
        ViSession instrumentHandle,
        ViString  queryI32Command,
        ViPInt32  i32Result)
/*****************************************************************************/
/*  Sends scpi command and waits for a response that must be representable   */
/*    as an int32.  A non numeric instrument response returns zero in        */
/*    result.                                                                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   queryI32Command (in)                                        */
/*      The SCPI command string to be sent to the instrument.                */
/*    ViPInt32   i32Result (out)                                             */
/*      The integer returned from the instrument.                            */
/*****************************************************************************/

{  
ViStatus  errStatus;
struct    hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVQueryf(instrumentHandle, "%s\n", "%ld%*t",
                                    queryI32Command, i32Result)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
        
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 

/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdInt16Arr_Q(
        ViSession instrumentHandle,
        ViString  queryI16ArrayCommand,
        ViInt32   i16ArraySize,
        ViInt16   _VI_FAR i16ArrayResult[],
        ViPInt32  i16ArrayCount)
/*****************************************************************************/
/*  This entry point assumes that the return result is a arbitrary block.    */
/*  Do not use this entry point for ascii values.  For that just get the     */
/*  data from _cmd_string_Q routine.                                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                        */
/*    ViString   queryI16ArrayCommand (in)                                   */
/*      The instrument command string to be sent to the instrument.          */
/*    ViInt32    i16ArraySize (in)                                           */
/*      The number of elements allocated to the results array.               */
/*    ViInt16    _VI_FAR i16ArrayResult[] (out)                              */
/*      Array of 16-bit integers - result from instrument.                   */
/*    ViPInt32   i16ArrayCount (out)                                         */
/*      The actual number of elements returned by the instrument             */
/*                                                                           */
/*  The function assumes that the data returned by the instrument has        */
/*    four components.                                                       */
/*    1 - An initial "#" taking one byte                                     */
/*    2 - A single ASCII digit indicating the size in bytes of a number      */
/*        which follows immediately.                                         */
/*    3 - A string number indicating the number of array elements to be      */
/*        returned by the instrument.                                        */
/*    4 - The arbitrary block being returned by the instrument.  In this     */
/*        case it is an array of 16-bit integers.                            */
/*                                                                           */
/*****************************************************************************/

{
ViStatus  errStatus ;
ViUInt32  cnt;
struct    hp548xx_globals * thisPtr;
char      lc[100];
ViInt32   i;
ViInt32   ArrSize;
ViInt32   nbytes;
char      _huge *Array;
	
GET_GLOBALS(instrumentHandle , thisPtr)

CHK_LONG_RANGE(i16ArraySize, hp548xx_CMDINT16ARR_Q_MIN,
                                hp548xx_CMDINT16ARR_Q_MAX, VI_ERROR_PARAMETER3)

if ((errStatus = delayVPrintf(instrumentHandle , "%s\n",
                                          queryI16ArrayCommand)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
			
if ((errStatus = delayRead(instrumentHandle,
                                           (ViPBuf)lc, 1, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
if (lc[0] != '#')
   {
    thisPtr->errNumber = VI_ERROR_INV_EXPR;
    return VI_ERROR_INV_EXPR;
   }
		
if ((errStatus = delayRead(instrumentHandle,
                                           (ViPBuf)lc, 1, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
nbytes = lc[0]-'0';
	    
if (nbytes)
{
    if ((errStatus = delayRead(instrumentHandle,
                                      (ViPBuf)lc, nbytes, &cnt)) < VI_SUCCESS)
        return statusUpdate(instrumentHandle, thisPtr, errStatus);
    lc[cnt] = '\0';
    ArrSize = atol(lc);
}
else
    ArrSize = i16ArraySize*sizeof(ViInt16);
		    
Array = (char *)malloc(ArrSize ); 
	
if (!Array )
   {
    thisPtr->errNumber = VI_ERROR_ALLOC;
    return VI_ERROR_ALLOC;
   }
	
if ((errStatus = delayRead(instrumentHandle,
                                  (ViPBuf)Array, ArrSize, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
if (thisPtr->interfaceType == VI_INTF_GPIB)
    if (errStatus == VI_SUCCESS_MAX_CNT)
        viClear(instrumentHandle );
ArrSize = cnt/sizeof(ViInt16);
ArrSize = (ArrSize > i16ArraySize) ? i16ArraySize : ArrSize;
	
/*copy only as many as the user allows*/
memcpy(i16ArrayResult, Array, (size_t)ArrSize*sizeof(ViInt16)); 

free(Array); 

*i16ArrayCount=ArrSize;

for (i = 0; i < *i16ArrayCount; i++)
    SWAP_16(&i16ArrayResult[i]);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);    	    	
}        /* ----- end of function ----- */

/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdInt32Arr_Q(
        ViSession instrumentHandle,
        ViString  queryI32ArrayCommand,
        ViInt32   i32ArraySize,
        ViInt32   _VI_FAR i32ArrayResult[],
        ViPInt32  i32ArrayCount)
/*****************************************************************************/
/*  This entry point assumes that the return result is a arbitrary block.    */
/*  Do not use this entry point for ascii values.  For that just get the     */
/*  data from _cmd_string_Q routine.                                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   queryI32ArrayCommand (in)                                   */
/*      The instrument command string to be sent to the instrument.          */
/*    ViInt32    i32ArraySize (in)                                           */
/*      The number of elements allocated to the results array.               */
/*    ViInt32    _VI_FAR i32ArrayResult[] (out)                              */
/*      Array of 32-bit integers - result from instrument.                   */
/*    ViPInt32   i32ArrayCount (out)                                         */
/*      The actual number of elements returned by the instrument             */
/*                                                                           */
/*  The function assumes that the data returned by the instrument has        */
/*    four components.                                                       */
/*    1 - An initial "#" taking one byte                                     */
/*    2 - A single ASCII digit indicating the size in bytes of a number      */
/*        which follows immediately.                                         */
/*    3 - A string number indicating the number of array elements to be      */
/*        returned by the instrument.                                        */
/*    4 - The arbitrary block being returned by the instrument.  In this     */
/*        case it is an array of 32-bit integers.                            */
/*                                                                           */
/*****************************************************************************/

{
ViStatus  errStatus ;
ViUInt32  cnt;
struct    hp548xx_globals * thisPtr;
char      lc[100];
ViInt32   i;
ViInt32   ArrSize;
ViInt32   nbytes;
char      _huge *Array;
	
GET_GLOBALS(instrumentHandle, thisPtr)

CHK_LONG_RANGE(i32ArraySize, hp548xx_CMDINT32ARR_Q_MIN,
                                hp548xx_CMDINT32ARR_Q_MAX, VI_ERROR_PARAMETER3)

if ((errStatus = delayVPrintf(instrumentHandle, "%s\n",
                                          queryI32ArrayCommand)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
			
if ((errStatus = delayRead(instrumentHandle,
                                 (ViPBuf)lc, 1, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
if (lc[0] != '#')
   {
    thisPtr->errNumber = VI_ERROR_INV_EXPR;
    return VI_ERROR_INV_EXPR;
   }
		
if ((errStatus = delayRead(instrumentHandle,
                                 (ViPBuf)lc, 1, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
nbytes = lc[0]-'0';

if (nbytes)
{
    if ((errStatus = delayRead(instrumentHandle,
                                 (ViPBuf)lc, nbytes, &cnt)) < VI_SUCCESS)
        return statusUpdate(instrumentHandle, thisPtr, errStatus);
    lc[cnt] = '\0';
    ArrSize = atol(lc);
}
else
    ArrSize = i32ArraySize*sizeof(ViInt32);
		    
Array = (char *)malloc(ArrSize ); 
	
if (!Array)
   {
    thisPtr->errNumber = VI_ERROR_ALLOC;
    return VI_ERROR_ALLOC;
   }

if ((errStatus = delayRead(instrumentHandle,
                                 (ViPBuf)Array, ArrSize, &cnt)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);
if (thisPtr->interfaceType == VI_INTF_GPIB)
    if (errStatus == VI_SUCCESS_MAX_CNT)
        viClear(instrumentHandle);
ArrSize = cnt/sizeof(ViInt32);
ArrSize = (ArrSize > i32ArraySize ) ? i32ArraySize : ArrSize;

 /*copy only as many as the user allows*/

memcpy(i32ArrayResult, Array, (size_t)ArrSize*sizeof(ViInt32));
free(Array);

*i32ArrayCount = ArrSize;

for (i = 0; i < *i32ArrayCount; i++)
    SWAP_32 (&i32ArrayResult[i]);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);    	    	
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdReal(
        ViSession  instrumentHandle,
        ViString   sendRealCommand,
        ViReal64   sendReal )
/*****************************************************************************/
/*  Sends an instrument command which takes a single real parameter.         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   sendRealCommand (in)                                        */
/*      The instrument command string to be sent to the instrument.          */
/*    ViReal64   sendReal (in)                                               */
/*      The real number sent to the instrument at the end of the instrument  */
/*      command.  Can be ViReal32 or ViReal64.                               */
/*****************************************************************************/

{
ViStatus errStatus = VI_SUCCESS ;
struct   hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVPrintf(instrumentHandle, "%s %g\n", sendRealCommand,
                                                     sendReal )) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */


/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_cmdReal64_Q(
    ViSession  instrumentHandle,
    ViString   queryRealCommand,
    ViPReal64  realResult)
/*****************************************************************************/
/*  Sends scpi command and waits for a response that must be representable   */
/*    as an real64.  A non numeric instrument response returns zero in       */
/*    result, in case of underflow undefined.                                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   queryRealCommand (in)                                       */
/*      The SCPI command string to be sent to the instrument.                */
/*    ViPReal64  realResult (out)                                            */
/*      The real returned from the instrument.                               */
/*****************************************************************************/

{   
ViStatus  errStatus;
struct    hp548xx_globals *thisPtr;
   
GET_GLOBALS(instrumentHandle, thisPtr)

if ((errStatus = delayVQueryf(instrumentHandle, "%s\n", "%lg%*t",
                                  queryRealCommand, realResult)) < VI_SUCCESS)
    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 


/*****************************************************************************/
static const char * const hp548xx_acquire_average_a[] = {"OFF","ON",0};
static const char * const hp548xx_acquire_sampling_mode_a[] = {"RTIM","ETIM","PDET",0};
static const char * const hp548xx_acquire_sampling_rate_a[] = {"AUTO",
	"5E-1","1E0","2.5E0","5E0","1E1","2.5E1","5E1","1E2","2.5E2","5E2","1E3","2.5E3","5E3",
	"1E4","2.5E4","5E4","1E5","2.5E5","5E5","1E6","2.5E6","5E6","1E7","2.5E7","5E7",
	"1E8","2.5E8","5E8","1E9","2E9","4E9","8E9",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_acquireSetup (ViSession instrumentHandle, 
						ViInt16 average,
						ViInt16 countvalue,
						ViInt16 completionpercent,
						ViInt16 samplingmode,
						ViInt32 memorydepth, 
						ViInt16 sampling_rate)
/*****************************************************************************/
/* Sets acquisition mode, averaging on/off, count, and completion.           */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    average (in)                                                */
/*      This parameter enables or disables averaging mode.                   */
/*    ViInt16    countvalue (in)                                             */
/*      This parameter sets the number of averages for the waveforms.        */
/*    ViInt16    completionpercent (in)                                      */
/*      It specifies the minimum completion criteria for an acquisition.     */
/*    ViInt16    samplingmode (in)                                           */
/*      Selects the type of acquisition mode to be set.                      */
/*    ViInt32    memorydepth (in)                                            */
/*      Sets the requested memory depth for an acquisition.                  */
/*    ViInt16    samplingrate (in)                                           */
/*      Sets the acquisition sample rate of the oscilloscope.                */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr; 
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/	
	CHK_ENUM( average, 1, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(countvalue, hp548xx_ACQ_COUNT_MIN , hp548xx_ACQ_COUNT_MAX , VI_ERROR_PARAMETER3)
	CHK_INT_RANGE(completionpercent, hp548xx_ACQ_PERCENT_MIN, hp548xx_ACQ_PERCENT_MAX, VI_ERROR_PARAMETER4)
	CHK_ENUM( samplingmode, 2, VI_ERROR_PARAMETER5 )
	if (memorydepth){ 
	CHK_LONG_RANGE(memorydepth, hp548xx_ACQ_POINTS_MIN , hp548xx_ACQ_POINTS_MAX , VI_ERROR_PARAMETER6)}
	CHK_ENUM( sampling_rate, 32, VI_ERROR_PARAMETER7 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:AVER %s\n",
					hp548xx_acquire_average_a[average])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:AVER:COUN %d\n",
					countvalue)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:COMP %d\n",
					completionpercent)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:MODE %s\n",
					hp548xx_acquire_sampling_mode_a[samplingmode])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if(memorydepth){
		if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:POIN %d\n",
						memorydepth)) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}else{
		if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:POIN AUTO\n")) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}
	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:SRAT %s\n",
					hp548xx_acquire_sampling_rate_a[sampling_rate])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_acquire_config_a[] = {"TCH","FCH",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_set54845Aconfig (ViSession instrumentHandle,
						ViInt16 config)
/*****************************************************************************/
/* Sets channel mode  for 54845 A.                                           */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    config (in)                                                 */
/*      Configures the HP 54845A oscilloscope to use 2 or 4 channels for 	 */
/*		acquisition.                                                         */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr; 
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/	
	CHK_ENUM( config, 1, VI_ERROR_PARAMETER2 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":ACQ:CONF %s\n",
					hp548xx_acquire_config_a[config])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_channelScale (ViSession instrumentHandle,
						ViInt16 channel_number, 
						ViReal64 rangevalue,
						ViReal64 offsetvalue)
/*****************************************************************************/
/* Sets Vertical range and offset.                                           */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    channel_number (in)                                         */
/*      Specifies the channel number.                                        */
/*    ViReal64   rangevalue (in)                                             */
/*      Full-Scale voltage of the specified channel number.                  */
/*    ViReal64   offsetvalue (in)                                            */
/*      Sets the voltage that is represented at the center of the display    */
/*		for the selected channel.                                            */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(channel_number, hp548xx_CH1 , hp548xx_CH4 , VI_ERROR_PARAMETER2)
	CHK_REAL_RANGE(rangevalue, hp548xx_RANGE_MIN , hp548xx_RANGE_MAX, VI_ERROR_PARAMETER3)	
	CHK_REAL_RANGE(offsetvalue, hp548xx_OFFSET_MIN , hp548xx_OFFSET_MAX, VI_ERROR_PARAMETER4)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:RANG %g\n",
					channel_number,rangevalue)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:OFFS %g\n",
					channel_number,offsetvalue)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_channel_bwlimit_a[] = {"OFF","ON",0};
static const char * const hp548xx_channel_display_a[] = {"OFF","ON",0};
static const char * const hp548xx_channel_input_a[] = {"DC","DC50","AC","LFR1","LFR2",0};
static const char * const hp548xx_channel_units_a[] = {"RAT","DEC",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_channelInputSetup (ViSession instrumentHandle,
						ViInt16 channel_number,
						ViInt16 bwlimit, 
						ViInt16 display,
						ViInt16 coupling,
						ViReal64 attenuationfactor,
						ViInt16 attenuation_units)
/*****************************************************************************/
/* Sets probe attenuation, coupling, bandwith limit, and display on/off.     */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    channel_number (in)                                         */
/*      Specifies the channel  number.                                       */
/*    ViInt16    bwlimit (in)                                                */
/*      Specifies the limits of the low pass filter.                         */
/*    ViInt16    display (in)                                                */
/*      Turns the display of the specified channel on or off.                */
/*    ViInt16    input (in)                                                  */
/*      Selects the input coupling.                                          */
/*    ViReal64   attenuationfactor (in)                                      */
/*      Sets the probe attenuation factor.                                   */
/*    ViInt16    attenuation_units (in)                                      */
/*      Specifies the mode for probe attenuation.                            */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(channel_number, hp548xx_CH1 , hp548xx_CH4 , VI_ERROR_PARAMETER2)
	CHK_ENUM( bwlimit, 1, VI_ERROR_PARAMETER3 )
	CHK_ENUM( display, 1, VI_ERROR_PARAMETER4 )
	CHK_ENUM( coupling, 4, VI_ERROR_PARAMETER5 )
	CHK_ENUM( attenuation_units, 1, VI_ERROR_PARAMETER7 )
	if (attenuation_units == hp548xx_CHANN_PROBE_RATIO){
		CHK_REAL_RANGE(attenuationfactor, hp548xx_CHANN_ATT_RAT_MIN , hp548xx_CHANN_ATT_RAT_MAX, VI_ERROR_PARAMETER6)	
	}else {
		CHK_REAL_RANGE(attenuationfactor, hp548xx_CHANN_ATT_DEC_MIN , hp548xx_CHANN_ATT_DEC_MAX, VI_ERROR_PARAMETER6)	
	}   
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:BWL %s\n",
					channel_number,hp548xx_channel_bwlimit_a[bwlimit])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:DISP %s\n",
					channel_number,hp548xx_channel_display_a[display])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:INP %s\n",
					channel_number,hp548xx_channel_input_a[coupling])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:PROB %g,%s\n",
					channel_number,attenuationfactor,hp548xx_channel_units_a[attenuation_units])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);


return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_save_recall_a[] = {"STOR:SET","REC:SET",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_saveRecallSetup (ViSession instrumentHandle,
						ViInt16 saverecall,
						ViInt16 memoryregisternumber)
/*****************************************************************************/
/* Saves / Recalls setup to/from the specified register.                     */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    saverecall (in)                                             */
/*      It specifies the setup to be saved/Recalled from the Instrument 	 */
/*      register.                                                            */
/*    ViInt16    memoryregisternumber (in)                                   */
/*      This parameter specifies the value of the Register(0 to 9), 		 */
/*      where the setup to/from saved/Recalled.                              */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( saverecall, 1, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(memoryregisternumber, hp548xx_COMM_REG0 , hp548xx_COMM_REG9 , VI_ERROR_PARAMETER3)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":%s %d\n",
					hp548xx_save_recall_a[saverecall],memoryregisternumber)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_disk_destination_a[] = {"SET","WMEM",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_loadFromDisk (ViSession instrumentHandle,
						ViString filename,
						ViInt16 destination,
						ViInt16 destinationnumber)
/*****************************************************************************/
/* Loads .wvf, .txt, or .set files from disk.                                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   filename (in)                                               */
/*      Specifies the filename to load data from.                            */
/*    ViInt16    destination (in)                                            */
/*      Specifies whether the load will be made   with the setup or 		 */
/*		the wmem command.                                                    */
/*    ViInt16    destinationnumber (in)                                      */
/*      Specifies the  w-memory number.                                      */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	if(filename==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER2);
	CHK_ENUM( destination, 1, VI_ERROR_PARAMETER3 )
	if (destination == hp548xx_DISK_WMEM ){
		CHK_INT_RANGE(destinationnumber, hp548xx_WMEM1 , hp548xx_WMEM4 , VI_ERROR_PARAMETER4)
	}
/*--Send instrument commands-------------------------------------------------*/	
	if (destination == hp548xx_DISK_SETUP){
		if ((errStatus = delayVPrintf(instrumentHandle,":DISK:LOAD \"%s\",%s\n",
						filename,hp548xx_disk_destination_a[destination])) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}else{
		if ((errStatus = delayVPrintf(instrumentHandle,":DISK:LOAD \"%s\",%s%d\n",
						filename,hp548xx_disk_destination_a[destination],destinationnumber)) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_disk_source_a[] = {"CHAN","FUNC","WMEM","SET",0};
static const char * const hp548xx_disk_format_a[] = {"INT","TEXT,YVAL","TEXT,VERB","TEXT,XYP",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_storeToDisk (ViSession instrumentHandle,
						ViString filename, 
						ViInt16 source,
						ViInt16 source_number, 
						ViInt16 format)
/*****************************************************************************/
/* Stores .wvf, .txt, .set,or .csv files to disk.                            */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   filename (in)                                               */
/*      Specifies the name of the file to be stored.                         */
/*    ViInt16    source (in)                                                 */
/*      Specifies the source to be stored.                                   */
/*    ViInt16    source_number (in)                                          */
/*      Represents the channel, function, or waveform memory number.         */
/*    ViInt16    format (in)                                                 */
/*      Specifies the format of the destination file.                        */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	if(filename==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER2);
	CHK_ENUM( source, 3, VI_ERROR_PARAMETER3 )
	CHK_INT_RANGE(source_number, hp548xx_CH_F_W_1 , hp548xx_CH_F_W_4 , VI_ERROR_PARAMETER4)		
	CHK_ENUM( format, 3, VI_ERROR_PARAMETER5 )
/*--Send instrument commands-------------------------------------------------*/	
	//setup
	if (source!=hp548xx_DISK_SOURCE_SETUP ){
		if ((errStatus = delayVPrintf(instrumentHandle,":VIEW %s%d\n",
						hp548xx_disk_source_a[source],source_number)) < VI_SUCCESS)
		    return statusUpdate(instrumentHandle, thisPtr, errStatus);
		    
		doDelay (100000);

		if ((errStatus = delayVPrintf(instrumentHandle,":DISK:STOR %s%d,\"%s\",%s\n",
						hp548xx_disk_source_a[source],source_number,filename,hp548xx_disk_format_a[format])) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}else{
		if ((errStatus = delayVPrintf(instrumentHandle,":DISK:STOR %s,\"%s\",%s\n",
						hp548xx_disk_source_a[source],filename,hp548xx_disk_format_a[format])) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_displayData (ViSession instrumentHandle,
						ViInt16 row_number, 
						ViInt16 col_number,
						ViString stringargument)
/*****************************************************************************/
/* Displays text line at row and column indicated.                           */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    row_number (in)                                             */
/*      This parameter sets the starting row on the screen for string to 	 */
/*		be Displayed.                                                        */
/*    ViInt16    col_number (in)                                             */
/*      This parameter sets the starting Column on the screen for string 	 */
/*		to be Displayed.                                                     */
/*    ViString   stringargument (in)                                         */
/*      It takes String to be Displayed at location specified by row and 	 */
/*      col parameters.                                                      */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;								
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(row_number,hp548xx_DISP_ROW_MIN,hp548xx_DISP_ROW_MAX, VI_ERROR_PARAMETER2)
	CHK_INT_RANGE(col_number,hp548xx_DISP_COL_MIN,hp548xx_DISP_COL_MAX, VI_ERROR_PARAMETER3)
	if(stringargument==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER4);
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":DISP:ROW %d\n",
					row_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":DISP:COL %d\n",
					col_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":DISP:LINE \"%s\"\n",
					stringargument)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_disp_filetype_a[] = {"PCX","BMP","EPS","PS","GIF",0};
static const char * const hp548xx_disp_screenmode_a[] = {"SCR","GRAT",0};
static const char * const hp548xx_disp_compression_a[] = {"OFF","ON",0};
static const char * const hp548xx_disp_inversion_a[] = {"NORM","INV",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_displayData_Q (ViSession instrumentHandle,
						ViString filename, 
						ViInt16 filetype,
						ViInt16 screenmode, 
						ViInt16 compression,
						ViInt16 inversion)
/*****************************************************************************/
/* Returns information about the captured data.                              */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   filename (in)                                               */
/*      Specifies the filename to store the display data.                    */
/*    ViInt16    filetype (in)                                               */
/*      Specifies the filetype.                                              */
/*    ViInt16    screenmode (in)                                             */
/*      Specifies the display settings.                                      */
/*    ViInt16    compression (in)                                            */
/*      Determines compression feature.                                      */
/*    ViInt16    inversion (in)                                              */
/*      Specifies inversion of the displayed file.                           */
/*****************************************************************************/
{
/*--Data acquisition variables-----------------------------------------------*/		
	ViInt16 header_length;
	ViChar term;
	ViInt32 bytes_read;                                            
	ViInt16 Acquired_length;
	ViChar data_h[100];				
    
    ViInt32 size,ret_size ;
    ViChar  *result;
    FILE *fp;

	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	if(filename==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER2);
	CHK_ENUM( filetype, 4, VI_ERROR_PARAMETER3 )
	CHK_ENUM( screenmode, 1, VI_ERROR_PARAMETER4 )
	CHK_ENUM( compression, 1, VI_ERROR_PARAMETER5 )
	CHK_ENUM( inversion, 1, VI_ERROR_PARAMETER6 )
/*--Open file to write data--------------------------------------------------*/	
	if( ( fp = fopen(filename, "wb"))==NULL) 
		return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_OPEN_ERROR);
/*--Send instrument command--------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":DISP:DATA? %s,%s,%s,%s\n",
					hp548xx_disp_filetype_a[filetype],hp548xx_disp_screenmode_a[screenmode],
					hp548xx_disp_compression_a[compression],hp548xx_disp_inversion_a[inversion])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Allocate memory for data buffer------------------------------------------*/	
	result = (ViChar *)calloc(hp548xx_MAX_SIZE, sizeof( ViChar ) );
	size=hp548xx_MAX_SIZE; 
/*--Request data header------------------------------------------------------*/	
	if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, 1, &ret_size)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if (data_h[0] != '#')
		return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_INV_EXPR);
    if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, 1, &ret_size)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	header_length = data_h[0]-'0';
	if (header_length)
	{
	    if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, header_length, &ret_size)) < VI_SUCCESS)
        	return statusUpdate(instrumentHandle, thisPtr, errStatus);
    	data_h[ret_size]='\0';
    	Acquired_length = atol(data_h);
	}
	else return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_INV_EXPR);
/*--Read data from instrument------------------------------------------------*/	
  	if ((errStatus = delayRead(instrumentHandle,(ViPBuf)result,size,&ret_size)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Write data to file-------------------------------------------------------*/	
    if ( fwrite(result, sizeof(ViChar),(ViInt32)ret_size, fp)<(ViInt32)ret_size)
    	return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_WRITE_ERROR);
    fclose(fp);
/*--Free allocated result memory---------------------------------------------*/	
    free(result);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_fft_display_a[] = {"OFF","ON",0};
static const char * const hp548xx_fft_op_type_a[] = {"CHAN","FUNC","WMEM","",0};
static const char * const hp548xx_fft_window_a[] = {"RECT","HANN","FLAT",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_funcFftSetup (ViSession instrumentHandle,
                        ViInt16 functionnumber, 
                        ViInt16 display,
                        ViInt16 operandtype,
                        ViReal64 operand,
                        ViReal64 centerfrequencyvalue,
                        ViInt16 window)
/*****************************************************************************/
/* Sets up center frequency and window for FFT.                              */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    functionnumber (in)                                         */
/*      Represents the selected fuction number.                              */
/*    ViInt16    display (in)                                                */
/*      Sets the display on/off for the specified function.                  */
/*    ViInt16    operandtype (in)                                            */
/*      Selects the operand type for the FFTMagnitude command.               */
/*    ViReal64   operand (in)                                                */
/*      Sets the operand for the FFTMagnitude command, the channel, 		 */
/*		waveform, or function number, or constant value.                     */
/*    ViReal64   centerfrequencyvalue (in)                                   */
/*      The parameter center_frequency sets the center frequency in Hertz    */
/*		when FFT is selected.                                                */
/*    ViInt16    window (in)                                                 */
/*      The parameter window allows selection of three windows for the FFT	 */
/* 		function.                                                            */
/*****************************************************************************/
{									    
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(functionnumber,hp548xx_FUNC1,hp548xx_FUNC4,VI_ERROR_PARAMETER2)		
	CHK_ENUM( display, 1, VI_ERROR_PARAMETER3 )
	CHK_ENUM( operandtype, 3, VI_ERROR_PARAMETER4 )
	if ( operandtype != hp548xx_CONSTANT ){
		if ((operand!=1) && (operand!=2) && (operand!=3) && (operand!=4))
			return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER5);
	}
	CHK_REAL_RANGE(centerfrequencyvalue, hp548xx_FFT_CFREQ_MIN , hp548xx_FFT_CFREQ_MAX, VI_ERROR_PARAMETER6)	
	CHK_ENUM( window, 2, VI_ERROR_PARAMETER7 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:DISP %d\n",
					functionnumber,display)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if (operandtype == hp548xx_CONSTANT) {
		if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:FFTM %s%g\n",
						functionnumber,hp548xx_fft_op_type_a[operandtype],operand)) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}else {
		if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:FFTM %s%d\n",
						functionnumber,hp548xx_fft_op_type_a[operandtype],(int)operand)) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}  
	if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:FFT:FREQ %g\n",
					functionnumber,centerfrequencyvalue)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:FFT:WIND %s\n",
					functionnumber,hp548xx_fft_window_a[window])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_fft_operation_a[] = {"ADD","SUBT","DIFF","DIV","MULT","VERS",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_funcOper (ViSession instrumentHandle,
                        ViInt16 functionnumber, 
                        ViInt16 display,
                        ViInt16 operation, 
                        ViInt16 operand1type,
                        ViReal64 operand1, 
                        ViInt16 operand2type,
                        ViReal64 operand2)
/*****************************************************************************/
/* Sets up function selection, horizontal and vertical ranges, and display	 */
/* on/off. Functions included: add, substract, divide, and multiply.         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    functionnumber (in)                                         */
/*      Represents the selected fuction number.                              */
/*    ViInt16    display (in)                                                */
/*      Sets the display on/off for the specified function.                  */
/*    ViInt16    operation (in)                                              */
/*      Selects the operation to be performed.                               */
/*    ViInt16    operand1type (in)                                           */
/*      Operand 1 type.                                                      */
/*    ViReal64   operand1 (in)                                               */
/*      Operand 1 value.                                                     */
/*    ViInt16    operand2type (in)                                           */
/*      Operand 2 type.                                                      */
/*    ViReal64   operand2 (in)                                               */
/*      Operand 2 value.                                                     */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(functionnumber,hp548xx_FUNC1,hp548xx_FUNC4,VI_ERROR_PARAMETER2)		
	CHK_ENUM( display, 1, VI_ERROR_PARAMETER3 )
	CHK_ENUM( operation, 5, VI_ERROR_PARAMETER4 )
	CHK_ENUM( operand1type, 3, VI_ERROR_PARAMETER5 )
	if ( operand1type != hp548xx_CONSTANT ){
		if ((operand1!=1) && (operand1!=2) && (operand1!=3) && (operand1!=4))
			return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER6);
	}
	CHK_ENUM( operand2type, 3, VI_ERROR_PARAMETER7 )
	if ( operand2type != hp548xx_CONSTANT ){
		if ((operand2!=1) && (operand2!=2) && (operand2!=3) && (operand2!=4))
			return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER8);
	}
/*--Send instrument commands-------------------------------------------------*/	
/*	printf(":FUNC:DISP:\n");
*/
	if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:DISP %d\n",
					functionnumber,display)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	doDelay (1000);
	
	switch (operation){
		case hp548xx_FUNC_OP_ADD:
		case hp548xx_FUNC_OP_SUB:
		case hp548xx_FUNC_OP_DIV:
		case hp548xx_FUNC_OP_MULT:
		case hp548xx_FUNC_OP_VERS:
			if (operand1type == hp548xx_CONSTANT) {
				if (operand2type == hp548xx_CONSTANT) {					 
/*					printf(":FUNC%d:%s %s%g,%s%g\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],operand1,
							hp548xx_fft_op_type_a[operand2type],operand2);
*/
					if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%g,%s%g\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],operand1,
							hp548xx_fft_op_type_a[operand2type],operand2)) < VI_SUCCESS)
		    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
				}else {
/*					printf(":FUNC%d:%s %s%g,%s%d\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],operand1,
							hp548xx_fft_op_type_a[operand2type],(int)operand2);
*/
					if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%g,%s%d\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],operand1,
							hp548xx_fft_op_type_a[operand2type],(int)operand2)) < VI_SUCCESS)
		    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
				}  
			}else {
				if (operand2type == hp548xx_CONSTANT) {
/*					printf(":FUNC%d:%s %s%d,%s%g\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],(int)operand1,
							hp548xx_fft_op_type_a[operand2type],operand2);
*/
					if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%d,%s%g\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],(int)operand1,
							hp548xx_fft_op_type_a[operand2type],operand2)) < VI_SUCCESS)
		    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
				}else {
/*					printf(":FUNC%d:%s %s%d,%s%d\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],(int)operand1,
							hp548xx_fft_op_type_a[operand2type],(int)operand2);
*/
					if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%d,%s%d\n",
							functionnumber,hp548xx_fft_operation_a[operation],
							hp548xx_fft_op_type_a[operand1type],(int)operand1,
							hp548xx_fft_op_type_a[operand2type],(int)operand2)) < VI_SUCCESS)
		    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
				}  
			}  
			break;
		case hp548xx_FUNC_OP_DIFF:
			if (operand1type == hp548xx_CONSTANT) {
				if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%g\n",
						functionnumber,hp548xx_fft_operation_a[operation],
						hp548xx_fft_op_type_a[operand1type],operand1)) < VI_SUCCESS)
	    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
			}else {
				if ((errStatus = delayVPrintf(instrumentHandle,":FUNC%d:%s %s%d\n",
						functionnumber,hp548xx_fft_operation_a[operation],
						hp548xx_fft_op_type_a[operand1type],(int)operand1)) < VI_SUCCESS)
	    			return statusUpdate(instrumentHandle, thisPtr, errStatus);
			}  
			break;
	}

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_hcopy_area_a[] = {"GRAT","SCR",0};
static const char * const hp548xx_hcopy_factors_a[] = {"OFF","ON",0};
static const char * const hp548xx_hcopy_image_a[] = {"NORM","INV","MON",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_hardCopySetup (ViSession instrumentHandle, 
						ViInt16 area,
						ViInt16 factors, 
						ViInt16 image)
/*****************************************************************************/
/* Sets up area, factors, and image definition for hardcopy.                 */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    area (in)                                                   */
/*      Sets up the area for hardcopy.                                       */
/*    ViInt16    factors (in)                                                */
/*      Determines whether the oscilloscope setup factors will be appended   */
/*		to scrren or graticule images.                                     */
/*    ViInt16    image (in)                                                  */
/*      Determines whether the image will be printed Normal, Invert, or 	 */
/*		Monochrome mode.                                                     */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;		  
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( area, 1, VI_ERROR_PARAMETER2 )
	CHK_ENUM( factors, 1, VI_ERROR_PARAMETER3 )
	CHK_ENUM( image, 2, VI_ERROR_PARAMETER4 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":HARD:AREA %s\n",
					hp548xx_hcopy_area_a[area])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":HARD:FACT %s\n",
					hp548xx_hcopy_factors_a[factors])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":HARD:IMAG %s\n",
					hp548xx_hcopy_image_a[image])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus); 
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_meas_source_a[] = {"CHAN","FUNC","WMEM","SET",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_measureAll_Q (ViSession instrumentHandle, 
						ViInt16 source,
						ViInt16 source_number,
						ViReal64 _VI_FAR resultsarray [])
/*****************************************************************************/
/* Returns values for: freq, period, +width,-width, risetime, falltime, 	 */
/* duty cycle, Vrms, Vmax, Vmin, Vtop, Vbase,Vavg, Vamp, Vpp, overshoot, 	 */
/* preshoot, Vlower, Vupper for selected source.                             */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Specifies whether the source will be a channel, a function, or a 	 */
/*		waveform memory.                                                     */
/*    ViInt16    source_number (in)                                          */
/*      Specifies the source for channel, function, or waveform memory.      */
/*    ViReal64   _VI_FAR resultsarray[] (out)                                */
/*      Array contains the returned values.                                  */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( source, 3, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(source_number, hp548xx_CH_F_W_1 , hp548xx_CH_F_W_4 , VI_ERROR_PARAMETER3)		
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":MEAS:SOUR %s%d\n",
					hp548xx_meas_source_a[source],source_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Query the instrument for specific measurements---------------------------*/	

	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:FREQ?\n","%lg",&resultsarray[hp548xx_MEAS_FREQUENCY])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PER?\n","%lg",&resultsarray[hp548xx_MEAS_PERIOD])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PWID?\n","%lg",&resultsarray[hp548xx_MEAS_PWIDTH])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:NWID?\n","%lg",&resultsarray[hp548xx_MEAS_NWIDTH])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:RIS?\n","%lg",&resultsarray[hp548xx_MEAS_RISETIME])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:FALL?\n","%lg",&resultsarray[hp548xx_MEAS_FALLTIME])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:DUTY?\n","%lg",&resultsarray[hp548xx_MEAS_DUTYCYCLE])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? CYCL,AC\n","%lg",&resultsarray[hp548xx_MEAS_VRMS_CYC_AC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? CYCL,DC\n","%lg",&resultsarray[hp548xx_MEAS_VRMS_CYC_DC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? DISP,AC\n","%lg",&resultsarray[hp548xx_MEAS_VRMS_DIS_AC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? DISP,DC\n","%lg",&resultsarray[hp548xx_MEAS_VRMS_DIS_DC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMAX?\n","%lg",&resultsarray[hp548xx_MEAS_VMAX])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMIN?\n","%lg",&resultsarray[hp548xx_MEAS_VMIN])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VTOP?\n","%lg",&resultsarray[hp548xx_MEAS_VTOP])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VBAS?\n","%lg",&resultsarray[hp548xx_MEAS_VBASE])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAV? CYCL\n","%lg",&resultsarray[hp548xx_MEAS_VAVG_CYC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAV? DISP\n","%lg",&resultsarray[hp548xx_MEAS_VAVG_DIS])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAMP?\n","%lg",&resultsarray[hp548xx_MEAS_VAMPLITUDE])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VPP?\n","%lg",&resultsarray[hp548xx_MEAS_VPP])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:OVER?\n","%lg",&resultsarray[hp548xx_MEAS_OVERSHOOT])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PRES?\n","%lg",&resultsarray[hp548xx_MEAS_PRESHOOT])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VLOW?\n","%lg",&resultsarray[hp548xx_MEAS_VLOW])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMID?\n","%lg",&resultsarray[hp548xx_MEAS_VMID])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VUP?\n","%lg",&resultsarray[hp548xx_MEAS_VUPP])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_meas_meas_a[] = {"FREQ?","PER?","PWID?","NWID?","RIS?",
		"FALL?","DUTY?","VRMS? CYCL,AC","VRMS? CYCL,DC","VRMS? DIS,AC","VRMS? DIS,DC","VMAX?",
		"VMIN?","VTOP?","VBAS?","VAV? CYC","VAV? DIS","VAMP?","VPP?","OVER?","PRE?","VLOW?","VMID?","VUPP?",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_measureOne_Q (ViSession instrumentHandle, 
						ViInt16 source,
						ViInt16 source_number, 
						ViInt16 measurement,
						ViPReal64 result)
/*****************************************************************************/
/* Returns values for specific measurement on selected source: freq, 		 */
/* period, +width,-width, risetime, falltime, duty cycle,Vrms,Vmax,Vmin, 	 */
/* Vtop, Vbase, Vavg, Vamp, Vpp, overshoot, preshoot, Vlower, 				 */
/* Vupper for selected source.                                               */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Specifies whether the source will be a channel, a function, or a 	 */
/*		waveform memory.                                                     */
/*    ViInt16    source_number (in)                                          */
/*      Specifies the source for channel, function, or waveform memory.      */
/*    ViInt16    measurement (in)                                            */
/*      Specify measurement.                                                 */
/*    ViPReal64  result (out)                                                */
/*      Returns measured value.                                              */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( source, 3, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(source_number, hp548xx_CH_F_W_1 , hp548xx_CH_F_W_4 , VI_ERROR_PARAMETER3)		
	CHK_ENUM( measurement,23, VI_ERROR_PARAMETER4)		
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":MEAS:SOUR %s%d\n",
					hp548xx_meas_source_a[source],source_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Query the instrument for specific measurement----------------------------*/	
	switch (measurement){
		case hp548xx_MEAS_FREQUENCY:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:FREQ?\n","%lg",result)) < VI_SUCCESS)
	           	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
	    case hp548xx_MEAS_PERIOD:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PER?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_PWIDTH:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PWID?\n","%lg",result)) < VI_SUCCESS)
              	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_NWIDTH:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:NWID?\n","%lg",result)) < VI_SUCCESS)
              	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_RISETIME:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:RIS?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_FALLTIME:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:FALL?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_DUTYCYCLE:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:DUTY?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VRMS_CYC_AC:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? CYCL,AC\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VRMS_CYC_DC:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? CYCL,DC\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VRMS_DIS_AC:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? DISP,AC\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VRMS_DIS_DC:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VRMS? DISP,DC\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VMAX:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMAX?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VMIN:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMIN?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VTOP:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VTOP?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VBASE:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VBAS?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VAVG_CYC:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAV? CYCL\n","%lg",result)) < VI_SUCCESS)
              	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VAVG_DIS:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAV? DISP\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VAMPLITUDE:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VAMP?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VPP:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VPP?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_OVERSHOOT:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:OVER?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_PRESHOOT:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:PRES?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VLOW:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VLOW?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VMID:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VMID?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
			break;
		case hp548xx_MEAS_VUPP:	  
			if ((errStatus = delayVQueryf(instrumentHandle,":MEAS:VUP?\n","%lg",result)) < VI_SUCCESS)
               	return statusUpdate(instrumentHandle, thisPtr, errStatus);
        	break;
    }
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_autoscale (ViSession instrumentHandle)
/*****************************************************************************/
/* Causes the oscilloscope to evaluate all input signals and find the 		 */
/* optimum conditions for displaying the signal.                             */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":AUT\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_bv_setting_a[] = {"BLAN","VIEW",0};
static const char * const hp548xx_bv_source_a[] = {"CHAN","FUNC","WMEM",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_blankView (ViSession instrumentHandle, 
						ViInt16 setting,
						ViInt16 source, 
						ViInt16 source_number)
/*****************************************************************************/
/* Turns on/off active channel, function, or waveform memory.                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    setting (in)                                                */
/*      Selects to turn on/off active channel, function, or waveform memory. */
/*    ViInt16    source (in)                                                 */
/*      Selects channel, function, or waveform memory.                       */
/*    ViInt16    source_number (in)                                          */
/*      Selects specific channel, function, or waveform memory.              */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( setting, 1, VI_ERROR_PARAMETER2 )
	CHK_ENUM( source, 2, VI_ERROR_PARAMETER3 )
	CHK_INT_RANGE(source_number,hp548xx_CH_F_W_1,hp548xx_CH_F_W_4, VI_ERROR_PARAMETER4)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":%s %s%d\n",
					hp548xx_bv_setting_a[setting],hp548xx_bv_source_a[source],source_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_dig_source_a[] = {"CHAN","FUNC",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_digitize (ViSession instrumentHandle, 
						ViInt16 source,
						ViInt16 source_number)
/*****************************************************************************/
/* Digitizes the signal.                                                     */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Selects channel or function.                                         */
/*    ViInt16    source_number (in)                                          */
/*      Selects specific channel or  function.                               */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( source, 1, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(source_number,hp548xx_CH_F_W_1,hp548xx_CH_F_W_4, VI_ERROR_PARAMETER3)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":DIG %s%d\n",
					hp548xx_dig_source_a[source],source_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_printScreen (ViSession instrumentHandle)
/*****************************************************************************/
/* Print to hardcopy device using factors set in hardcopy subsystem.         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":PRIN\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_run (ViSession instrumentHandle)
/*****************************************************************************/
/* Starts the oscilloscope running.                                          */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":RUN\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_stop (ViSession instrumentHandle)
/*****************************************************************************/
/* Stops the acquisition.                                                    */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":STOP\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_single (ViSession instrumentHandle)
/*****************************************************************************/
/* Causes the oscilloscope to make a single acquisition when the next 		 */
/* trigger event occurs.                                                     */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":SING\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_swav_source_a[] = {"CHAN","FUNC","WMEM",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_storeWaveform (ViSession instrumentHandle, 
						ViInt16 source,
						ViInt16 source_number, 
						ViInt16 destination)
/*****************************************************************************/
/* Stores a channel, function, or stored waveform to the indicated 			 */
/* waveform memory.                                                          */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Selects channel, function, or waveform memory.                       */
/*    ViInt16    source_number (in)                                          */
/*      Selects specific channel, function, or waveform memory.              */
/*    ViInt16    destination (in)                                            */
/*      Selects specific waveform memory to store waveform                   */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( source, 2, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(source_number,hp548xx_CH_F_W_1,hp548xx_CH_F_W_4, VI_ERROR_PARAMETER3)	
	CHK_INT_RANGE(destination,hp548xx_WMEM1,hp548xx_WMEM4, VI_ERROR_PARAMETER4)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":VIEW %s%d\n",
					hp548xx_swav_source_a[source],source_number)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
		    
	doDelay (100000);

	if ((errStatus = delayVPrintf(instrumentHandle,":STOR:WAV %s%d,WMEM%d\n",
					hp548xx_swav_source_a[source],source_number,destination)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_systemSetup_Q (ViSession instrumentHandle,
						ViString filename)
/*****************************************************************************/
/* Performs setup query and stores in the specified filename. Information    */
/* is written in a binary block  data format as defined in the IEEE 488.2    */
/* standard.                                                                 */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   filename (in)                                               */
/*      Specifies the filename for data destination.                         */
/*****************************************************************************/
{
    ViInt32 size,ret_size;
    ViChar  *result;
    FILE *fp;
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	if(filename==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER2);
/*--Send instrument command--------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":SYST:SET?\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Open file to store system setup------------------------------------------*/	
    if( ( fp = fopen(filename, "wb"))==NULL) 
    	return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_OPEN_ERROR);
/*--Allocate an array of hp548xx_MAX_SIZE to result--------------------------*/	
    result = (ViChar *)calloc(hp548xx_MAX_SIZE, sizeof( ViChar ) );
    size=hp548xx_MAX_SIZE;
/*--Read data from instrument------------------------------------------------*/	
  	if ((errStatus = delayRead(instrumentHandle,(ViPBuf)result,size,&ret_size)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Write data to file-------------------------------------------------------*/	
    if ( fwrite(result, sizeof(ViChar),(ViInt32)ret_size, fp)<(ViInt32)ret_size)
    	return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_WRITE_ERROR);
    fclose(fp);
/*--Free allocated result memory---------------------------------------------*/	
    free(result);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_systemSetup (ViSession instrumentHandle,
						ViString filename)
/*****************************************************************************/
/* This function puts system setup information in binary IEEE 488.2 format   */
/* to the instrument from the specified file.                                */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViString   filename (in)                                               */
/*      Specifies the binary filename to retrive the system Setup. 			 */
/*		This should be a Binary IEEE 488.2 format.                           */
/*****************************************************************************/
{
    ViInt32 numread,scpi_length,size;
    ViChar *data,*scpi_data;
    FILE *fp;

	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	if(filename==NULL)
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_PARAMETER2);
/*--Allocate memory for the data and scpi buffer-----------------------------*/	
	data = (ViChar *)calloc( (ViInt16)(14*1024+100), sizeof( ViChar ) );
	scpi_data=data;
	if( data == NULL )
    	return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_ALLOC); 
	scpi_length = sprintf( data,"SYST:SET ");
/*--Move pointer to the data start position----------------------------------*/	
	data=data+scpi_length;
/*--Open file and read setup data--------------------------------------------*/	
	if( (fp = fopen( filename, "rb" )) == NULL )
    	return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_OPEN_ERROR); 
    if ((numread = fread( data, sizeof( ViChar ),(ViInt16)(14*1024), fp))==0)
    	return statusUpdate(instrumentHandle, thisPtr, hp548xx_FILE_READ_ERROR); 
	fclose(fp);
/*--Return pointer to data starting position---------------------------------*/	
	data=scpi_data;
	if ((errStatus = viWrite(instrumentHandle,data,(numread+scpi_length),&size)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	data=scpi_data;
/*--Free buffer memory-------------------------------------------------------*/	
	free(data);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_tim_reference_a[] = {"LEFT","CENT","RIGH",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_timebaseRange (ViSession instrumentHandle,
						ViInt16 reference,
						ViReal64 position, 
						ViReal64 fullscalerange)
/*****************************************************************************/
/* Defines horizontal range, position and reference.                         */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    reference (in)                                              */
/*      Sets the delay reference to the left, center, or right side of 		 */
/*		the display.                                                         */
/*    ViReal64   position (in)                                               */
/*      Sets the time interval between the trigger event and the delay 		 */
/*		reference point.                                                     */
/*    ViReal64   fullscalerange (in)                                         */
/*      Full scale horizontal range.                                         */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( reference, 2, VI_ERROR_PARAMETER2 )
	CHK_REAL_RANGE(position, hp548xx_TIM_POS_MIN , hp548xx_TIM_POS_MAX, VI_ERROR_PARAMETER3)	
	CHK_REAL_RANGE(fullscalerange, hp548xx_TIM_RANGE_MIN , hp548xx_TIM_RANGE_MAX, VI_ERROR_PARAMETER4)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":TIM:REF %s\n",
					hp548xx_tim_reference_a[reference])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":TIM:POS %g\n",
					position)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TIM:RANG %g\n",
					fullscalerange)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_trig_edge_source_a[] = {"AUX","CHAN1","CHAN2","CHAN3","CHAN4","EXT","LINE",0};
static const char * const hp548xx_trig_edge_smode_a[] = {"AUTO","TRIG","SING",0};
static const char * const hp548xx_trig_edge_coupl_a[] = {"AC","DC","LFR","HFR",0};
static const char * const hp548xx_trig_edge_slope_a[] = {"POS","NEG",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_edgeTriggerSetup (ViSession instrumentHandle,
						ViInt16 source, 
						ViReal64 triggerlevel,
						ViInt16 sweepmode, 
						ViInt16 coupling,
						ViInt16 slope)
/*****************************************************************************/
/* Sets trigger mode to EDGE and defines edge trigger source, slope, 		 */
/* level, sweep, and coupling.                                               */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Selects the source for edge mode triggering. Also the  trigger 		 */
/*		input for the trigger level command.                                 */
/*    ViReal64   triggerlevel (in)                                           */
/*      The trigger level on the specified channel, external trigger, or 	 */
/*		auxiliary trigger input.                                             */
/*    ViInt16    sweepmode (in)                                              */
/*      Specifies the oscilloscope sweep mode.                               */
/*    ViInt16    coupling (in)                                               */
/*      Sets the trigger coupling when source is set to one of the channels. */
/*    ViInt16    slope (in)                                                  */
/*      Sets the slope for the previously selected source.                   */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( source, 6, VI_ERROR_PARAMETER2 )
	CHK_REAL_RANGE(triggerlevel, hp548xx_TRIG_LEVEL_MIN , hp548xx_TRIG_LEVEL_MAX, VI_ERROR_PARAMETER3)	
	CHK_ENUM( sweepmode, 2, VI_ERROR_PARAMETER4 )
	CHK_ENUM( coupling, 3, VI_ERROR_PARAMETER5 )
	CHK_ENUM( slope, 1, VI_ERROR_PARAMETER6 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:MODE EDGE\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:EDGE:SOURCE %s\n",
					hp548xx_trig_edge_source_a[source])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if (source != hp548xx_LINE ){	
		if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:LEV %s,%g\n",
						hp548xx_trig_edge_source_a[source],triggerlevel)) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
		if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:EDGE:SLOP %s\n",
						hp548xx_trig_edge_slope_a[slope])) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	}
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:SWE %s\n",
					hp548xx_trig_edge_smode_a[sweepmode])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((source>=hp548xx_CH1) && (source<=hp548xx_CH4))
		if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:EDGE:COUP %s\n",
						hp548xx_trig_edge_coupl_a[coupling])) < VI_SUCCESS)
	    	return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_trig_glitch_source_a[] = {"","CHAN1","CHAN2","CHAN3","CHAN4","EXT",0};
static const char * const hp548xx_trig_glitch_smode_a[] = {"AUTO","TRIG","SING",0};
static const char * const hp548xx_trig_glitch_polarity_a[] = {"POS","NEG",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_glitchTriggerSetup (ViSession instrumentHandle,
						ViInt16 source, 
						ViReal64 triggerlevel,
						ViInt16 sweepmode, 
						ViReal64 width,
						ViInt16 polarity)
/*****************************************************************************/
/* Defines source, level, polarity, width, and sweep for glitch mode.        */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Selects the source for glitch mode triggering. Also specifies the    */
/*		trigger input for the trigger level command.                         */
/*    ViReal64   triggerlevel (in)                                           */
/*      The trigger level on the specified channel, external trigger, or 	 */
/*		auxiliary trigger input.                                             */
/*    ViInt16    sweepmode (in)                                              */
/*      Specifies the oscilloscope sweep mode.                               */
/*    ViReal64   width (in)                                                  */
/*      Sets the glitch width. The scope will trigger on a pulse whose 		 */
/*		width is less than the specified width.                              */
/*    ViInt16    polarity (in)                                               */
/*      Defines the polarity of the glitch.                                  */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(source, hp548xx_CHAN1 , hp548xx_EXT , VI_ERROR_PARAMETER2)	
	CHK_REAL_RANGE(triggerlevel, hp548xx_TRIG_LEVEL_MIN , hp548xx_TRIG_LEVEL_MAX, VI_ERROR_PARAMETER3)	
	CHK_ENUM( sweepmode, 2, VI_ERROR_PARAMETER4 )
	CHK_REAL_RANGE(width, hp548xx_TRIG_WIDTH_MIN , hp548xx_TRIG_WIDTH_MAX , VI_ERROR_PARAMETER5)		
	CHK_ENUM( polarity, 1, VI_ERROR_PARAMETER6 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:MODE GLIT\n"
					)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:GLIT:SOURCE %s\n",
					hp548xx_trig_glitch_source_a[source])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:LEV %s,%g\n",
					hp548xx_trig_glitch_source_a[source],triggerlevel)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:SWE %s\n",
					hp548xx_trig_glitch_smode_a[sweepmode])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:GLIT:WIDTH %g\n",
					width)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":TRIG:GLIT:POL %s\n",
					hp548xx_trig_glitch_polarity_a[polarity])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
ViStatus _VI_FUNC hp548xx_waveformScaling_Q (ViSession instrumentHandle,
						ViPInt32 points,
						ViReal64 _VI_FAR data [])
/*****************************************************************************/
/* Returns values to calculate x,y values from waveform data. 				 */
/* Returns the following: number of points, x increment, y increment, 		 */
/* x origin, and y origin.                                                   */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViPInt16   points (out)                                                */
/*      Returns the points value in the current waveform preamble.           */
/*    ViReal64   _VI_FAR data[] (in)                                         */
/*      Returns the Xincrement,Xorigin,Yincrement, and Yorigin data in 		 */
/*		array form.                                                          */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Query the instrument for specific measurements---------------------------*/	
	if ((errStatus = delayVQueryf(instrumentHandle,":WAV:POIN?\n","%d",points)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":WAV:XINC?\n","%lg",&data[hp548xx_WAV_XINC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":WAV:XOR?\n","%lg",&data[hp548xx_WAV_XOR])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":WAV:YINC?\n","%lg",&data[hp548xx_WAV_YINC])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayVQueryf(instrumentHandle,":WAV:YOR?\n","%lg",&data[hp548xx_WAV_YOR])) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_wav_format_a[] = {"ASC","BYTE","LONG","WORD",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_waveformFormat (ViSession instrumentHandle,
						ViInt16 format)
/*****************************************************************************/
/* Sets data transmission mode for waveform data.                            */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    format (in)                                                 */
/*      Sets data transmission mode for waveform data.                       */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( format, 3, VI_ERROR_PARAMETER2 )
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":WAV:FORM %s\n",
					hp548xx_wav_format_a[format])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_wav_display_a[] = {"OFF","ON",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_displayWmem (ViSession instrumentHandle, 
						ViInt16 display,
						ViInt16 source)
/*****************************************************************************/
/* Displays the indicated waveform memory.                                   */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    display (in)                                                */
/*      Enables or disables the viewing of ths selected waveform memory.     */
/*    ViInt16    source (in)                                                 */
/*      Specifies the waveform memory number.                                */
/*****************************************************************************/
{
	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)
/*--Validate input parameters------------------------------------------------*/		
	CHK_ENUM( display, 1, VI_ERROR_PARAMETER2 )
	CHK_INT_RANGE(source,hp548xx_WMEM1,hp548xx_WMEM4, VI_ERROR_PARAMETER3)	
/*--Send instrument commands-------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":WMEM%d:DISP %s\n",
					source,hp548xx_wav_display_a[display])) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
static const char * const hp548xx_wav_ch_source_a[] = {"","CHAN1","CHAN2","CHAN3","CHAN4",0};
/*---------------------------------------------------------------------------*/	
ViStatus _VI_FUNC hp548xx_waveformArray_Q (ViSession instrumentHandle,
						ViInt16 source, 
						ViPInt32 points,
						ViReal64 _VI_FAR time_value [],
						ViReal64 _VI_FAR volts [])
/*****************************************************************************/
/* This application function converts waveform data to time and voltage 	 */
/* values.                                                                   */
/*                                                                           */
/*  PARAMETERS                                                               */
/*    ViSession  instrumentHandle (in)                                       */
/*      Instrument handle returned from hp548xx_init()                       */
/*    ViInt16    source (in)                                                 */
/*      Specifies the channel to take measurements from.                     */
/*    ViPInt32   points (out)                                                */
/*      Returns the number of data points.                                   */
/*    ViReal64   _VI_FAR time_value (out)                                    */
/*      Array contains the returned values for the time period.              */
/*    ViReal64   _VI_FAR volts (out)                                         */
/*      Array contains the returned values for voltage across time.          */
/*****************************************************************************/
{
/*--Data conversion variables------------------------------------------------*/		
	ViReal64 wav_param[4];
	ViReal64 xorg,xinc;				
	ViReal64 yorg,yinc;
/*--Data acquisition variables-----------------------------------------------*/		
	ViInt16 header_length;
	ViChar term;
	ViInt32 bytes_read;                                            
	ViInt16 Acquired_length;
/*--Data buffers-------------------------------------------------------------*/		
	ViChar data_h[100];
	ViPChar data;			    

	ViInt16 i;

	ViStatus errStatus = VI_SUCCESS ;
	struct   hp548xx_globals *thisPtr;
	GET_GLOBALS(instrumentHandle, thisPtr)

/*--Validate input parameters------------------------------------------------*/		
	CHK_INT_RANGE(source, hp548xx_CHAN1 , hp548xx_CHAN4 , VI_ERROR_PARAMETER2)	
	

/*--Send instrument commands and call specific functions---------------------*/	
	if ((errStatus = hp548xx_digitize (instrumentHandle,hp548xx_CHANNEL,source)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":CHAN%d:DISP ON\n",
					source)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = delayVPrintf(instrumentHandle,":WAV:SOUR CHAN%d\n",
					source)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	
	if ((errStatus = hp548xx_waveformFormat (instrumentHandle,hp548xx_WAV_FORMAT_BYTE )) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Request values to allow interpretation of raw data-----------------------*/	
	if ((errStatus = hp548xx_waveformScaling_Q (instrumentHandle,points,wav_param)) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);

	xinc=wav_param[hp548xx_WAV_XINC ];
	xorg=wav_param[hp548xx_WAV_XOR ];
	yinc=wav_param[hp548xx_WAV_YINC ];
	yorg=wav_param[hp548xx_WAV_YOR ];
	
/*--Request data header------------------------------------------------------*/	
	if ((errStatus = delayVPrintf(instrumentHandle,":WAVeform:DATA?\n")) < VI_SUCCESS)
	    return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, 1, &bytes_read)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
	if (data_h[0] != '#')
		return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_INV_EXPR);
    if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, 1, &bytes_read)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);

	header_length = data_h[0]-'0';
	if (header_length)
	{
	    if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data_h, header_length, &bytes_read)) < VI_SUCCESS)
        	return statusUpdate(instrumentHandle, thisPtr, errStatus);
    	data_h[bytes_read]='\0';
    	Acquired_length = atol(data_h);
	}
	else return statusUpdate(instrumentHandle, thisPtr, VI_ERROR_INV_EXPR);
    
/*--Allocate memory for data structures--------------------------------------*/	
 	data = 		(ViChar *)calloc(Acquired_length, sizeof( ViChar ) );
/*--Request data-------------------------------------------------------------*/	
	if ((errStatus = delayRead(instrumentHandle,(ViPBuf)data, Acquired_length, &bytes_read)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Read termination char----------------------------------------------------*/	
    if ((errStatus = delayRead(instrumentHandle,(ViPBuf)&term, 1, &bytes_read)) < VI_SUCCESS)
    	return statusUpdate(instrumentHandle, thisPtr, errStatus);
/*--Calculate time vs voltage info-------------------------------------------*/	
	for (i = 0; i < Acquired_length; i++)
	{
		time_value[i] =((i) * xinc) + xorg;	/* calculate time info */
	    volts[i] = ((data[i]) * yinc) + yorg;	/* calculate volt info */ 
  	}
  	
  	*points=Acquired_length;
	
return statusUpdate(instrumentHandle, thisPtr, VI_SUCCESS);
}        /* ----- end of function ----- */ 
/*****************************************************************************/
