#pragma warning(disable:4786)

#include <cassert>
#include "ElectronicLoad.h"

struct Model {
    char *modelNumber;
    bool areMultipleChannelsSupported;
};
static const Model modelTable[] = {
    {"6050A", true},
    {"6051A", true},
    {"6060A", false},
    {"6060B", false},
    {"6063A", false},
    {"6063B", false}
};
static const int modelTableSize = sizeof(modelTable)/sizeof(Model);

void ElectronicLoad::configureContinuousTransient(const Real64 &dutyCycle,
                                                  const Real64 &frequency,
                                                  const Int32 &mode,
                                                  const Real64 &immediateLevel,
                                                  const Real64 &transientLevel) {
    setMode(mode);
    switch (mode) {
    case MODE_CURRENT:
        setCurrent(CURR, immediateLevel);
        setCurrent(CURR_TLEV, transientLevel);
        break;
    case MODE_RESISTANCE:
        setResistance(RES, immediateLevel);
        setResistance(RES_TLEV, transientLevel);
        break;
    case MODE_VOLTAGE:
        setVoltage(VOLT, immediateLevel);
        setVoltage(VOLT_TLEV, transientLevel);
        break;
    default:
        assert(false);
    }
    setInputState(VI_TRUE);

    setTransient(VI_TRUE, TRAN_MODE_CONTINUOUS, dutyCycle, frequency);
}

void ElectronicLoad::configureInput(const Int32 &mode,
                                    const Real64 &level) {
    setMode(mode);
    switch (mode) {
    case MODE_CURRENT:
        setCurrent(CURR, level);
        break;
    case MODE_RESISTANCE:
        setResistance(RES, level);
        break;
    case MODE_VOLTAGE:
        setVoltage(VOLT, level);
        break;
    default:
        assert(false);
    }
    setInputState(VI_TRUE);
}

void ElectronicLoad::configureParallelLoads(const Int32 &numChannels,
                                            const AInt32 &channels,
                                            const Real64 &currentSlew,
                                            const Int32 &triggerSource,
                                            const Real64 &triggerTimer,
                                            const Real64 &triggerCurrent,
                                            const Boolean &transientState,
                                            const Int32 &transientMode,
                                            const Real64 &transientPulseWidth,
                                            const Real64 &transientCurrent) {
    pushChannel();

    for (int n = 0; n < numChannels; n += 1) {
        setChannel(channels[n]);
        setMode(MODE_CURRENT);
        setCurrent(CURR_SLEW, currentSlew);
        setCurrent(CURR_TRIG, triggerCurrent);
        setCurrent(CURR_TLEV, transientCurrent);
        if (transientState != VI_FALSE) {
            switch (transientMode) {
            case TRAN_MODE_PULSE:
            case TRAN_MODE_TOGGLE:
                setTransient(transientState, transientMode, transientPulseWidth, transientCurrent);
                break;
            default:
                throw transientMode.error;
            }
        }
    }

    popChannel();

    setTrigger(triggerSource, triggerTimer);
}

void ElectronicLoad::configureTrigger(const Int32 &triggerSource,
                                      const Int32 &mode,
                                      const Real64 &immediateLevel,
                                      const Real64 &triggerLevel) {
    setMode(mode);
    switch (mode) {
    case MODE_CURRENT:
        setCurrent(CURR, immediateLevel);
        setCurrent(CURR_TRIG, triggerLevel);
        break;
    case MODE_RESISTANCE:
        setResistance(RES, immediateLevel);
        setResistance(RES_TRIG, triggerLevel);
        break;
    case MODE_VOLTAGE:
        setVoltage(VOLT, immediateLevel);
        setVoltage(VOLT_TRIG, triggerLevel);
        break;
    default:
        assert(false);
    }
    setInputState(VI_TRUE);

    switch (triggerSource) {
    case TRIG_SOUR_BUS:
    case TRIG_SOUR_EXTERNAL:
    case TRIG_SOUR_HOLD:
        setTrigger(triggerSource, 0);
        break;
    default:
        throw triggerSource.error;
    }
}

void ElectronicLoad::configureTriggeredTransient(const Int32 &triggerSource,
                                                 const Int32 &transientMode,
                                                 const Real64 &pulseWidth,
                                                 const Int32 &mode,
                                                 const Real64 &immediateLevel,
                                                 const Real64 &transientLevel) {
    switch(triggerSource) {
    case TRIG_SOUR_BUS:
    case TRIG_SOUR_EXTERNAL:
    case TRIG_SOUR_HOLD:
        setTrigger(triggerSource, 0);
        break;
    default:
        throw triggerSource.error;
    }

    setMode(mode);
    switch (mode) {
    case MODE_CURRENT:
        setCurrent(CURR, immediateLevel);
        setCurrent(CURR_TLEV, transientLevel);
        break;
    case MODE_RESISTANCE:
        setResistance(RES, immediateLevel);
        setResistance(RES_TLEV, transientLevel);
        break;
    case MODE_VOLTAGE:
        setVoltage(VOLT, immediateLevel);
        setVoltage(VOLT_TLEV, transientLevel);
        break;
    default:
        assert(false);
    }
    setInputState(VI_TRUE);

    switch (transientMode) {
    case TRAN_MODE_PULSE:
    case TRAN_MODE_TOGGLE:
        setTransient(VI_TRUE, transientMode, pulseWidth, 0);
        break;
    default:
        throw transientMode.error;
    }
}

void ElectronicLoad::getChannel(const Int32 &channelType,
                                ViPInt32 channel) {
    switch (channelType) {
    case CHAN:
        readInt32("CHAN?", channel);
        break;
    case CHAN_MAX:
        readInt32("CHAN? MAX", channel);
        break;
    case CHAN_MIN:
        readInt32("CHAN? MIN", channel);
        break;
    default:
        throw channelType.error;
    }
}

void ElectronicLoad::getCurrent(const Int32 &currentType, ViPReal64 current) {
    switch (currentType) {
    case CURR:
        readReal64("CURR?", current);
        break;
    case CURR_MAX:
        readReal64("CURR? MAX", current);
        break;
    case CURR_MIN:
        readReal64("CURR? MIN", current);
        break;
    case CURR_SLEW:
        readReal64("CURR:SLEW?", current);
        break;
    case CURR_SLEW_MAX:
        readReal64("CURR:SLEW? MAX", current);
        break;
    case CURR_SLEW_MIN:
        readReal64("CURR:SLEW? MIN", current);
        break;
    case CURR_TLEV:
        readReal64("CURR:TLEV?", current);
        break;
    case CURR_TLEV_MAX:
        readReal64("CURR:TLEV? MAX", current);
        break;
    case CURR_TLEV_MIN:
        readReal64("CURR:TLEV? MIN", current);
        break;
    case CURR_TRIG:
        readReal64("CURR:TRIG?", current);
        break;
    case CURR_TRIG_MAX:
        readReal64("CURR:TRIG? MAX", current);
        break;
    case CURR_TRIG_MIN:
        readReal64("CURR:TRIG? MIN", current);
        break;
    default:
        throw currentType.error;
    }
}

void ElectronicLoad::getCurrentRange(const Int32 &currentRangeType,
                                     ViPReal64 currentRange) {
    switch (currentRangeType) {
    case CURR_RANG:
        readReal64("CURR:RANG?", currentRange);
        break;
    case CURR_RANG_MAX:
        readReal64("CURR:RANG? MAX", currentRange);
        break;
    case CURR_RANG_MIN:
        readReal64("CURR:RANG? MIN", currentRange);
        break;
    default:
        throw currentRangeType.error;
    }
}

void ElectronicLoad::getRegister(const Int32 &type,
                                 ViPInt32 registerValue) {
    switch (type) {
    case REG_STAT_QUES_NTR:
    case REG_STAT_QUES_PTR:
        throw type.error;
    default:
        break;
    }

    switch (type) {
    case REG_STAT_CHAN:
        readInt32("STAT:CHAN?", registerValue);
        break;
    case REG_STAT_CHAN_COND:
        readInt32("STAT:CHAN:COND?", registerValue);
        break;
    case REG_STAT_CHAN_ENAB:
        readInt32("STAT:CHAN:ENAB?", registerValue);
        break;
    case REG_STAT_CSUM:
        readInt32("STAT:CSUM?", registerValue);
        break;
    case REG_STAT_CSUM_ENAB:
        readInt32("STAT:CSUM:ENAB?", registerValue);
        break;
    default:
        ScpiInputInstrument::getRegister(type, registerValue);
    }
}

void ElectronicLoad::getResistance(const Int32 &resistanceType, ViPReal64 resistance) {
    switch (resistanceType) {
    case RES:
        readReal64("RES?", resistance);
        break;
    case RES_MAX:
        readReal64("RES? MAX", resistance);
        break;
    case RES_MIN:
        readReal64("RES? MIN", resistance);
        break;
    case RES_TLEV:
        readReal64("RES:TLEV?", resistance);
        break;
    case RES_TLEV_MAX:
        readReal64("RES:TLEV? MAX", resistance);
        break;
    case RES_TLEV_MIN:
        readReal64("RES:TLEV? MIN", resistance);
        break;
    case RES_TRIG:
        readReal64("RES:TRIG?", resistance);
        break;
    case RES_TRIG_MAX:
        readReal64("RES:TRIG? MAX", resistance);
        break;
    case RES_TRIG_MIN:
        readReal64("RES:TRIG? MIN", resistance);
        break;
    default:
        throw resistanceType.error;
    }
}

void ElectronicLoad::getResistanceRange(const Int32 &resistanceRangeType,
                                        ViPReal64 resistanceRange) {
    switch (resistanceRangeType) {
    case RES_RANG:
        readReal64("RES:RANG?", resistanceRange);
        break;
    case RES_RANG_MAX:
        readReal64("RES:RANG? MAX", resistanceRange);
        break;
    case RES_RANG_MIN:
        readReal64("RES:RANG? MIN", resistanceRange);
        break;
    default:
        throw resistanceRangeType.error;
    }
}

void ElectronicLoad::getVoltage(const Int32 &voltageType,
                                ViPReal64 voltage) {
    switch (voltageType) {
    case VOLT:
        readReal64("VOLT?", voltage);
        break;
    case VOLT_MAX:
        readReal64("VOLT? MAX", voltage);
        break;
    case VOLT_MIN:
        readReal64("VOLT? MIN", voltage);
        break;
    case VOLT_TLEV:
        readReal64("VOLT:TLEV?", voltage);
        break;
    case VOLT_TLEV_MAX:
        readReal64("VOLT:TLEV? MAX", voltage);
        break;
    case VOLT_TLEV_MIN:
        readReal64("VOLT:TLEV? MIN", voltage);
        break;
    case VOLT_TRIG:
        readReal64("VOLT:TRIG?", voltage);
        break;
    case VOLT_TRIG_MAX:
        readReal64("VOLT:TRIG? MAX", voltage);
        break;
    case VOLT_TRIG_MIN:
        readReal64("VOLT:TRIG? MIN", voltage);
        break;
    case VOLT_SLEW:
        readReal64("VOLT:SLEW?", voltage);
        break;
    case VOLT_SLEW_MAX:
        readReal64("VOLT:SLEW? MAX", voltage);
        break;
    case VOLT_SLEW_MIN:
        readReal64("VOLT:SLEW? MIN", voltage);
        break;
    default:
        throw voltageType.error;
    }
}

void ElectronicLoad::internalInit(const std::string &modelNumber) {
    Model model = {"", false};
    for (int n = 0; n < modelTableSize; n += 1)
        if (modelNumber == modelTable[n].modelNumber) {
            model = modelTable[n];
            break;
        }

    m_AreMultipleChannelsSupported = model.areMultipleChannelsSupported;
}

bool ElectronicLoad::isValidModel(const std::string &model) const {
    bool isValidModel = false;
    for (int n = 0; n < modelTableSize; n += 1)
        if (model == modelTable[n].modelNumber) {
            isValidModel = true;
            break;
        }
    return isValidModel;
}
    
void ElectronicLoad::measure(const Int32 &measurementType, ViPReal64 measurement) {
    beginMeasurement();
    try {
        switch (measurementType) {
        case MEAS_CURRENT:
            readReal64("MEAS:CURR?", measurement);
            break;
        case MEAS_POWER:
            readReal64("MEAS:POW?", measurement);
            break;
        case MEAS_VOLTAGE:
            readReal64("MEAS:VOLT?", measurement);
            break;
        default:
            throw measurementType.error;
        }
        endMeasurement();
    } catch (ViStatus) {
        endMeasurement();
        throw;
    }
}

void ElectronicLoad::setChannel(const Int32 &channel) {
    writeInt32("CHAN", channel);
}

void ElectronicLoad::setCurrent(const Int32 &currentType,
                                const Real64 &current) {
    switch (currentType) {
    case CURR:
        writeReal64("CURR", current);
        break;
    case CURR_MAX:
        write("CURR MAX");
        break;
    case CURR_MIN:
        write("CURR MIN");
        break;
    case CURR_SLEW:
        writeReal64("CURR:SLEW", current);
        break;
    case CURR_SLEW_MAX:
        write("CURR:SLEW MAX");
        break;
    case CURR_SLEW_MIN:
        write("CURR:SLEW MIN");
        break;
    case CURR_TLEV:
        writeReal64("CURR:TLEV", current);
        break;
    case CURR_TLEV_MAX:
        write("CURR:TLEV MAX");
        break;
    case CURR_TLEV_MIN:
        write("CURR:TLEV MIN");
        break;
    case CURR_TRIG:
        writeReal64("CURR:TRIG", current);
        break;
    case CURR_TRIG_MAX:
        write("CURR:TRIG MAX");
        break;
    case CURR_TRIG_MIN:
        write("CURR:TRIG MIN");
        break;
    default:
        throw currentType.error;
    }
}

void ElectronicLoad::setCurrentProtection(const Boolean &isEnabled,
                                          const Real64 &delay,
                                          const Real64 &currentLimit) {
    if (isEnabled) {
        writeReal64("CURR:PROT:DEL", delay);
        writeReal64("CURR:PROT", currentLimit);
    }
    writeBoolean("CURR:PROT:STAT", isEnabled);
}

void ElectronicLoad::setCurrentRange(const Int32 &currentRangeType,
                                     const Real64 &currentRange) {
    switch (currentRangeType) {
    case CURR_RANG:
        writeReal64("CURR:RANG", currentRange);
        break;
    case CURR_RANG_MAX:
        write("CURR:RANG MAX");
        break;
    case CURR_RANG_MIN:
        write("CURR:RANG MIN");
        break;
    default:
        throw currentRange.error;
    }
}

void ElectronicLoad::setInputShort(const Boolean &inputShort) {
    writeBoolean("INP:SHOR", inputShort);
}

void ElectronicLoad::setMode(const Int32 &mode) {
    switch (mode) {
    case MODE_CURRENT:
        write("MODE:CURR");
        break;
    case MODE_RESISTANCE:
        write("MODE:RES");
        break;
    case MODE_VOLTAGE:
        write("MODE:VOLT");
        break;
    default:
        throw mode.error;
    }
}

void ElectronicLoad::setPort(const Boolean &port) {
    writeBoolean("PORT0", port);
}

void ElectronicLoad::setRegister(const Int32 &type,
                                 const Int32 &registerValue) {
    switch (type) {
    case REG_STAT_QUES_NTR:
    case REG_STAT_QUES_PTR:
        throw type.error;
    default:
        break;
    }

    switch (type) {
    case REG_STAT_CHAN_ENAB:
        writeInt32("STAT:CHAN:ENAB", registerValue);
        break;
    case REG_STAT_CSUM_ENAB:
        writeInt32("STAT:CSUM:ENAB", registerValue);
        break;
    default:
        ScpiInputInstrument::setRegister(type, registerValue);
    }
}

void ElectronicLoad::setResistance(const Int32 &resistanceType,
                                   const Real64 &resistance) {
    switch (resistanceType) {
    case RES:
        writeReal64("RES", resistance);
        break;
    case RES_MAX:
        write("RES MAX");
        break;
    case RES_MIN:
        write("RES MIN");
        break;
    case RES_TLEV:
        writeReal64("RES:TLEV", resistance);
        break;
    case RES_TLEV_MAX:
        write("RES:TLEV MAX");
        break;
    case RES_TLEV_MIN:
        write("RES:TLEV MIN");
        break;
    case RES_TRIG:
        writeReal64("RES:TRIG", resistance);
        break;
    case RES_TRIG_MAX:
        write("RES:TRIG MAX");
        break;
    case RES_TRIG_MIN:
        write("RES:TRIG MIN");
        break;
    default:
        throw resistanceType.error;
    }
}

void ElectronicLoad::setResistanceRange(const Int32 &resistanceRangeType,
                                        const Real64 &resistanceRange) {
    switch (resistanceRangeType) {
    case RES_RANG:
        writeReal64("RES:RANG", resistanceRange);
        break;
    case RES_RANG_MAX:
        write("RES:RANG MAX");
        break;
    case RES_RANG_MIN:
        write("RES:RANG MIN");
        break;
    default:
        throw resistanceRange.error;
    }
}

void ElectronicLoad::setTransient(const Boolean &transientState,
                                  const Int32 &transientMode,
                                  const Real64 &dutyCycleOrPulseWidth,
                                  const Real64 &frequency) {
    if (transientState) {
        switch (transientMode) {
        case TRAN_MODE_CONTINUOUS:
            write("TRAN:MODE CONT");
            writeReal64("TRAN:DCYC", dutyCycleOrPulseWidth);
            writeReal64("TRAN:FREQ", frequency);
            break;
        case TRAN_MODE_PULSE:
            write("TRAN:MODE PULS");
            writeReal64("TRAN:TWID", dutyCycleOrPulseWidth);
            break;
        case TRAN_MODE_TOGGLE:
            write("TRAN:MODE TOGG");
            break;
        default:
            throw transientMode.error;
        }
    }
    writeBoolean("TRAN", transientState);
}

void ElectronicLoad::setTrigger(const Int32 &source,
                                const Real64 &timer) {
    if (!m_AreMultipleChannelsSupported) {
        switch (source) {
        case TRIG_SOUR_LINE:
        case TRIG_SOUR_TIMER:
            throw source.error;
            break;
        default:
            break;
        }
    }

    switch (source) {
    case TRIG_SOUR_BUS:
        write("TRIG:SOUR BUS");
        break;
    case TRIG_SOUR_EXTERNAL:
        write("TRIG:SOUR EXT");
        break;
    case TRIG_SOUR_HOLD:
        write("TRIG:SOUR HOLD");
        break;
    case TRIG_SOUR_LINE:
        write("TRIG:SOUR LINE");
        break;
    case TRIG_SOUR_TIMER:
        write("TRIG:SOUR TIM");
        writeReal64("TRIG:TIM", timer);
        break;
    default:
        throw source.error;
    }
}

void ElectronicLoad::setVoltage(const Int32 &voltageType,
                                const Real64 &voltage) {
    switch (voltageType) {
    case VOLT:
        writeReal64("VOLT", voltage);
        break;
    case VOLT_MAX:
        write("VOLT MAX");
        break;
    case VOLT_MIN:
        write("VOLT MIN");
        break;
    case VOLT_TLEV:
        writeReal64("VOLT:TLEV", voltage);
        break;
    case VOLT_TLEV_MAX:
        write("VOLT:TLEV MAX");
        break;
    case VOLT_TLEV_MIN:
        write("VOLT:TLEV MIN");
        break;
    case VOLT_TRIG:
        writeReal64("VOLT:TRIG", voltage);
        break;
    case VOLT_TRIG_MAX:
        write("VOLT:TRIG MAX");
        break;
    case VOLT_TRIG_MIN:
        write("VOLT:TRIG MIN");
        break;
    case VOLT_SLEW:
        writeReal64("VOLT:SLEW", voltage);
        break;
    case VOLT_SLEW_MAX:
        write("VOLT:SLEW MAX");
        break;
    case VOLT_SLEW_MIN:
        write("VOLT:SLEW MIN");
        break;
    default:
        throw voltageType.error;
    }
}