#if !defined(Runtime_h)
#define Runtime_h

#include <memory>
#include <sstream>
#include <vector>
#include "VisaSession.h"

class Runtime {
public:
    enum {
        ERROR_PARAMETER = _VI_ERROR + 0x3FFC0D30
    };

    class Boolean {
    public:
        Boolean(ViBoolean v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViInt32() const { return value == VI_FALSE ? 0 : 1; }
        ViBoolean value;
        ViStatus error;
    };

    class ABoolean {
    public:
        ABoolean(ViABoolean v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViABoolean() const { return value; }
        ViABoolean value;
        ViStatus error;
    };

    class Real64 {
    public:
        Real64(ViReal64 v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViReal64() const { return value; }
        ViReal64 value;
        ViStatus error;
    };

    class AReal64 {
    public:
        AReal64(ViAReal64 v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViAReal64() const { return value; }
        ViAReal64 value;
        ViStatus error;
    };

    class Int32 {
    public:
        Int32(ViInt32 v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViInt32() const { return value; }
        ViInt32 value;
        ViStatus error;
    };

    class AInt32 {
    public:
        AInt32(ViAInt32 v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViAInt32() const { return value; }
        ViAInt32 value;
        ViStatus error;
    };

    class Int16 {
    public:
        Int16(ViInt16 v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViInt32() const { return value; }
        ViInt16 value;
        ViStatus error;
    };

    class String {
    public:
        String(ViString v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViString() const { return value; }
        operator std::string() const { return value; }
        bool operator==(const ViString s) const { return std::string(s) == value; }
        ViString value;
        ViStatus error;
    };

    class AString {
    public:
        AString(ViAString v, ViStatus e = ERROR_PARAMETER) : value(v), error(e) {}
        operator ViAString() const { return value; }
        ViAString value;
        ViStatus error;
    };

    class ErrorHandler : public VisaErrorHandler {
    public:
        ErrorHandler() : m_Status(VI_SUCCESS) {}
        ViStatus getStatus() const { return m_Status; }
        void reset() { m_Status = VI_SUCCESS; }
        ViStatus status(ViStatus);
    private:
        ViStatus m_Status;
    };

    Runtime() : m_Session(&m_ErrorHandler) {}

    void clear() { m_Session.clear(); }
    void close() { m_Session.close(); }
    ErrorHandler &getErrorHandler() { return m_ErrorHandler; }
    const ErrorHandler &getErrorHandler() const { return m_ErrorHandler; }
    VisaSession &getSession() { return m_Session; }
    const VisaSession &getSession() const { return m_Session; }
    ViSession getSessionId() const { return m_Session.getSessionId(); }
    ViStatus getStatus() const { return getErrorHandler().getStatus(); }
    void getTimeout(ViPInt32 milliseconds) { m_Session.getAttribute(VI_ATTR_TMO_VALUE, milliseconds); }
    void open(ViRsrc resourceId) { m_Session.open(resourceId); }
    void setTimeout(ViInt32 milliseconds) { m_Session.setAttribute(VI_ATTR_TMO_VALUE, milliseconds); }
    void sleep(const ViInt32);
    void status(const ViStatus status) { getErrorHandler().status(status); }

    void checkRange(const Int16 &, ViInt16, ViInt16);
    void checkRange(const Int32 &, ViInt32, ViInt32);
    void checkRange(const Real64 &, ViReal64, ViReal64);

    void parseList(const std::string &, const bool, std::vector<std::string> *);
    void readBoolean(const std::string &command, ViPBoolean result) { read(command) >> *result; }
    void readInt16(const std::string &command, ViPInt16 result) { read(command) >> *result; }
    void readInt32(const std::string &command, ViPInt32 result) { read(command) >> *result; }
    void readReal64(const std::string &command, ViPReal64 result) { read(command) >> *result; }
    void readAReal64(const std::string &, const int, ViAReal64);
    void readString(const std::string &, const bool, ViPString);
    void readString(const std::string &, const bool, std::string *);
    void readAString(const std::string &, const bool, const int, ViAString, ViPInt32);
    std::string readString(const std::string &);
    void readList(const std::string &, const bool, std::vector<std::string> *);
    void write(const std::string &);
    void writeBoolean(const std::string &, ViBoolean);
    void writeABoolean(const std::string &, ViInt32, ViABoolean);
    void writeInt16(const std::string &, ViInt16);
    void writeInt32(const std::string &, ViInt32);
    void writeReal64(const std::string &, ViReal64);
    void writeAReal64(const std::string &, ViInt32, ViAReal64);
    void writeString(const std::string &, const std::string &);
    void writeString(const std::string &, const bool, const std::string &);
    void writeAString(const std::string &, const bool, ViInt32, ViAString);

protected:
    VisaSession m_Session;

    virtual void commandComplete() = 0;
    std::string dequote(const std::string &);
    void disableNotification() { m_IsNotificationEnabled = false; }
    void enableNotification() { m_IsNotificationEnabled = true; }
    std::string quote(const std::string &);

private:
    ErrorHandler m_ErrorHandler;
    bool m_IsNotificationEnabled;

    std::istringstream read(const std::string &);
};

inline
void Runtime::writeString(const std::string &command, 
                          const std::string &argument) {
    writeString(command, false, argument);
}

inline
std::ostream &operator<<(std::ostream &out, const Runtime::String &s) {
    return out << (char *)s;
}

#endif