#pragma warning(disable:4786)

#include <cassert>
#include <vpptype.h>

#include "ScpiInstrument.h"

ScpiInstrument::ScpiInstrument() : Runtime() {
}

void ScpiInstrument::abort() {
    write("ABOR");
}

void ScpiInstrument::beginMeasurement() {
    Runtime::getTimeout(&m_Timeout);
    Runtime::setTimeout(m_MeasurementTimeout);
}

void ScpiInstrument::checkNonNegativeInt(const Int32 &nonNegativeInt) {
    if (nonNegativeInt < 0)
        throw nonNegativeInt.error;
}

void ScpiInstrument::cmd(const String &command) {
    write(command);
}

void ScpiInstrument::cmdInt(const String &command, const Int32 &value) {
    writeInt32(command, value);
}

void ScpiInstrument::cmdInt_Q(const String &command, ViPInt32 result) {
    readInt32(command, result);
}

void ScpiInstrument::cmdString_Q(const String &command, const Int32 &size, ViChar result[]) {
    checkNonNegativeInt(size);
    strncpy(result, readString(command).c_str(), size - 1);
    if (size > 0)
        result[size - 1] = 0;
}

void ScpiInstrument::commandComplete() {
    switch (m_InstrumentErrorDetectionMode) {
    case INST_ERROR_DETECT_OFF:
        break;
    case INST_ERROR_DETECT_POLL:
        disableNotification();
        ViInt32 eventStatus;
        readInt32("*ESR?", &eventStatus);
        if ((eventStatus & 60) > 0) {
            enableNotification();
            throw ViStatus(INSTR_ERROR_DETECTED);
        }
        enableNotification();
        break;
    case INST_ERROR_DETECT_INTERRUPT:
        {
            bool isInstrumentError = true;
            try {
                ViEventType eventType;
                ViEvent context;
                m_Session.waitOnEvent(VI_EVENT_SERVICE_REQ, m_InstrumentErrorWait, &eventType, &context);
            } catch (ViStatus status) {
                if (status != VI_ERROR_TMO)
                    throw;
                isInstrumentError = false;
            }
            if (isInstrumentError) {
                disableNotification();
                readString("*ESR?");
                enableNotification();
               throw ViStatus(INSTR_ERROR_DETECTED);
            }
        }
        break;
    default:
        assert(false);
    }
}

void ScpiInstrument::dcl() {
    clear();
}

void ScpiInstrument::disableInstrumentErrorInterruption() {
    if (m_InstrumentErrorDetectionMode == INST_ERROR_DETECT_INTERRUPT) {
        write("*ESE 0");
        write("*SRE 0");

        m_Session.disableEvent(VI_EVENT_SERVICE_REQ, VI_QUEUE);
    }
}

void ScpiInstrument::doDelay(const Int32 &milliseconds) {
    checkNonNegativeInt(milliseconds);

    sleep(milliseconds);
}

void ScpiInstrument::enableInstrumentErrorInterruption(const Int32 &wait) {
    m_InstrumentErrorWait = wait;

    write("*ESE 60");
    write("*SRE 32");

    m_Session.enableEvent(VI_EVENT_SERVICE_REQ, VI_QUEUE, VI_NULL);
}

void ScpiInstrument::endMeasurement() {
    Runtime::setTimeout(m_Timeout);
}

void ScpiInstrument::error_message(ViStatus status, ViChar message[]) {
    struct MessageTableEntry {
        ViStatus status;
        ViString message;
    };

    static const struct MessageTableEntry messageTable[] = {
        {INSTR_ERROR_NSUP_FUNCTION, "Function unsupported for this model"},
        {ERROR_PARAMETER, "Unspecified parameter is invalid"},
        {ERROR_PARAMETER1, "Parameter 1 is invalid"},
        {ERROR_PARAMETER2, "Parameter 2 is invalid"},
        {ERROR_PARAMETER3, "Parameter 3 is invalid"},
        {ERROR_PARAMETER4, "Parameter 4 is invalid"},
        {ERROR_PARAMETER5, "Parameter 5 is invalid"},
        {ERROR_PARAMETER6, "Parameter 6 is invalid"},
        {ERROR_PARAMETER7, "Parameter 7 is invalid"},
        {ERROR_PARAMETER8, "Parameter 8 is invalid"},
        {INSTR_ERROR_DETECTED, "ScpiInstrument error detected"},
        {SUCCESS, "Success"}
    };
    static const unsigned int messageTableSize = sizeof(messageTable)/sizeof(MessageTableEntry);

    bool messageFound = false;
    for (unsigned int n = 0; n < messageTableSize; n += 1)
        if (messageTable[n].status == status) {
            strcpy(message, messageTable[n].message);
            messageFound = true;
            break;
        }

    if (!messageFound) {
        try {
            std::string statusDesc;
            getSession().statusDesc(status, &statusDesc);
            strcpy(message, statusDesc.c_str());
        } catch (ViStatus) {
            strcpy(message, "Unknown error");
        }
    }
}

void ScpiInstrument::error_query(ViPInt32 errorCode, ViChar message[]) {
    std::vector<std::string> list;
    readList("SYST:ERR?", false, &list);
    std::istringstream in(list[0]);
    in >> *errorCode;
    strcpy(message, dequote(list[1]).c_str());
}

void ScpiInstrument::getMeasTimeout(ViPInt32 milliseconds) {
    *milliseconds = m_MeasurementTimeout;
}

void ScpiInstrument::getRegister(const Int32 &type, ViPInt32 registerValue) {
    switch (type) {
    case REG_ESE:
        readInt32("*ESE?", registerValue);
        break;
    case REG_ESR:
        readInt32("*ESR?", registerValue);
        break;
    case REG_SRE:
        readInt32("*SRE?", registerValue);
        break;
    case REG_STB:
        readInt32("*STB?", registerValue);
        break;
    case REG_STAT_OPER:
        readInt32("STAT:OPER?", registerValue);
        break;
    case REG_STAT_OPER_COND:
        readInt32("STAT:OPER:COND?", registerValue);
        break;
    case REG_STAT_OPER_ENAB:
        readInt32("STAT:OPER:ENAB?", registerValue);
        break;
    case REG_STAT_OPER_NTR:
        readInt32("STAT:OPER:NTR?", registerValue);
        break;
    case REG_STAT_OPER_PTR:
        readInt32("STAT:OPER:PTR?", registerValue);
        break;
    case REG_STAT_QUES:
        readInt32("STAT:QUES?", registerValue);
        break;
    case REG_STAT_QUES_COND:
        readInt32("STAT:QUES:COND?", registerValue);
        break;
    case REG_STAT_QUES_ENAB:
        readInt32("STAT:QUES:ENAB?", registerValue);
        break;
    case REG_STAT_QUES_NTR:
        readInt32("STAT:QUES:NTR?", registerValue);
        break;
    case REG_STAT_QUES_PTR:
        readInt32("STAT:QUES:PTR?", registerValue);
        break;
    default:
        throw type.error;
    }
}

void ScpiInstrument::getTimeout(ViPInt32 milliseconds) {
    Runtime::getTimeout(milliseconds);
}

void ScpiInstrument::init(const String &resourceId, const Boolean &idQuery, const Boolean &resetInstrument) {
    disableNotification();

    open(resourceId);

    // Set default timeout
    setTimeout(2000);

    // Reset command parser
    getSession().clear();

    // Check programming language of the instrument.
    checkLanguage();

    m_DriverRevision = "B.00.00";
    m_InstrumentErrorDetectionMode = INST_ERROR_DETECT_OFF;
    m_InstrumentId = readString("*IDN?");
    m_MeasurementTimeout = DEFAULT_MEASUREMENT_TIMEOUT;

    // Validate the instrument's identity.
    if (idQuery && !isValidModel(m_Model = parseModel(m_InstrumentId)))
        throw ViStatus(ERROR_FAIL_ID_QUERY);

    // Reset the instrument if so instructed
    if (resetInstrument) 
        reset();

    internalInit(m_Model);

    enableNotification();
}

std::string ScpiInstrument::parseModel(const std::string &instrumentId) {
    std::string model(instrumentId);
    model.erase(0, model.find(',') + 1);
    model.resize(model.find(','));
    return model;
}

void ScpiInstrument::reset() {
    write("*RST;*CLS");
}

void ScpiInstrument::revision_query(ViChar driverRevision[], ViChar instrumentId[]) {
    strcpy(driverRevision, m_DriverRevision.c_str());
    strcpy(instrumentId, m_InstrumentId.c_str());
}

void ScpiInstrument::self_test(ViPInt32 result, ViChar message[]) {
    readInt32("*TST?", result);
    strcpy(message, readString("SYST:ERR?").c_str());
}

void ScpiInstrument::setInstrumentErrorDetection(const Int32 &mode, const Int32 &wait) {
    switch (mode) {
    case INST_ERROR_DETECT_OFF:
    case INST_ERROR_DETECT_POLL:
        disableInstrumentErrorInterruption();
        break;
    case INST_ERROR_DETECT_INTERRUPT:
        enableInstrumentErrorInterruption(wait);
        break;
    default:
        throw mode.error;
    }

    m_InstrumentErrorDetectionMode = mode;
}

void ScpiInstrument::setMeasTimeout(const Int32 &milliseconds) {
    checkNonNegativeInt(milliseconds);

    m_MeasurementTimeout = milliseconds;
}

void ScpiInstrument::setRegister(const Int32 &type, const Int32 &registerValue) {
    switch (type) {
    case REG_ESE:
        writeInt32("*ESE", registerValue);
        break;
    case REG_SRE:
        writeInt32("*SRE", registerValue);
        break;
    case REG_STAT_OPER_ENAB:
        writeInt32("STAT:OPER:ENAB", registerValue);
        break;
    case REG_STAT_OPER_NTR:
        writeInt32("STAT:OPER:NTR", registerValue);
        break;
    case REG_STAT_OPER_PTR:
        writeInt32("STAT:OPER:PTR", registerValue);
        break;
    case REG_STAT_QUES_ENAB:
        writeInt32("STAT:QUES:ENAB", registerValue);
        break;
    case REG_STAT_QUES_NTR:
        writeInt32("STAT:QUES:NTR", registerValue);
        break;
    case REG_STAT_QUES_PTR:
        writeInt32("STAT:QUES:PTR", registerValue);
        break;
    default:
        throw type.error;
    }
}

void ScpiInstrument::setTimeout(const Int32 &milliseconds) {
    checkNonNegativeInt(milliseconds);

    Runtime::setTimeout(milliseconds);
}

void ScpiInstrument::trigger(const Int32 &triggerType) {
    switch (triggerType) {
    case TRIG_BUS:
        write("*TRG");
        break;
    case TRIG_IMMEDIATE:
        write("TRIG");
        break;
    default:
        throw triggerType.error;
    }
}

void ScpiInstrument::wait() {
    ViBoolean b;
    readBoolean("*OPC?", &b);
}
