#ifndef ScpiInstrument_h
#define ScpiInstrument_h

#include <vpptype.h>

#include "Runtime.h"

class ScpiInstrument : virtual public Runtime {
public:
    // Function inputs
    enum {
        INST_ERROR_DETECT_OFF = 0,
        INST_ERROR_DETECT_POLL,
        INST_ERROR_DETECT_INTERRUPT,
        REG_ESE,
        REG_ESR,
        REG_SRE,
        REG_STB,
        REG_STAT_OPER,
        REG_STAT_OPER_COND,
        REG_STAT_OPER_ENAB,
        REG_STAT_OPER_NTR,
        REG_STAT_OPER_PTR,
        REG_STAT_QUES,
        REG_STAT_QUES_COND,
        REG_STAT_QUES_ENAB,
        REG_STAT_QUES_NTR,
        REG_STAT_QUES_PTR,
        TRIG_BUS,
        TRIG_IMMEDIATE
    };

    // Function outputs which are not function inputs
    enum {
        OUTP_STATUS_UNKNOWN = 0x00000000,
        OUTP_STATUS_OFF = 0x00000001,
        OUTP_STATUS_CV = 0x00000002,
        OUTP_STATUS_CC = 0x00000004,
        OUTP_STATUS_PROT = 0x00000008,
        OUTP_STATUS_UNREG = 0x00000010,
    };

    // Helper values such as register bits
    enum {
        ESR_OPC     = 0x0001,
        ESR_QYE     = 0x0004,
        ESR_DDE     = 0x0008,
        ESR_EXE     = 0x0010,
        ESR_CME     = 0x0020,
        ESR_PON     = 0x0080,
        OPER_CAL    = 0x0001,
        OPER_WTG    = 0x0020,
        OPER_CV     = 0x0100,
        OPER_CC     = 0x0400,
        QUES_OV     = 0x0001,
        QUES_OCP    = 0x0002,
        QUES_OT     = 0x0010,
        QUES_RI     = 0x0200,
        QUES_UNREG  = 0x0400,
        SRE_QUES    = 0x0008,
        SRE_MAV     = 0x0010,
        SRE_ESB     = 0x0020,
        SRE_OPER    = 0x0080,
        STB_QUES    = 0x0008,
        STB_MAV     = 0x0010,
        STB_ESB     = 0x0020,
        STB_MSS     = 0x0040,
        STB_RQS     = 0x0040,
        STB_OPER    = 0x0080
    };

    // Driver specific errors
    enum {
        INSTR_ERROR_DETECTED = (_VI_ERROR + 0x3FFC0D07),
        INSTR_ERROR_NSUP_FUNCTION
    };

    // Redefined VISA constants
    enum {
        DEFAULT_MEASUREMENT_TIMEOUT = 5000,
        ERROR_FAIL_ID_QUERY = VI_ERROR_FAIL_ID_QUERY,
        ERROR_PARAMETER1 = VI_ERROR_PARAMETER1,
        ERROR_PARAMETER2 = VI_ERROR_PARAMETER2,
        ERROR_PARAMETER3 = VI_ERROR_PARAMETER3,
        ERROR_PARAMETER4 = VI_ERROR_PARAMETER4,
        ERROR_PARAMETER5 = VI_ERROR_PARAMETER5,
        ERROR_PARAMETER6 = VI_ERROR_PARAMETER6,
        ERROR_PARAMETER7 = VI_ERROR_PARAMETER7,
        ERROR_PARAMETER8 = VI_ERROR_PARAMETER8,
        SUCCESS = VI_SUCCESS
    };

    // Constants specific to the C++ implementation
    enum {
        ERROR_PARAMETER = Runtime::ERROR_PARAMETER,
        INTERNAL_ERROR = ERROR_PARAMETER + 1,
    };

    ScpiInstrument();

    virtual void abort();
    virtual void close() { Runtime::close(); }
    virtual void cmd(const String &);
    virtual void cmdInt(const String &, const Int32 &);
    virtual void cmdInt_Q(const String &, ViPInt32);
    virtual void cmdString_Q(const String &, const Int32 &, ViChar[]);
    virtual void dcl();
    virtual void doDelay(const Int32 &);
    virtual void error_message(ViStatus, ViChar[]);
    virtual void error_query(ViPInt32, ViChar[]);
    virtual void getMeasTimeout(ViPInt32);
    virtual void getRegister(const Int32 &, ViPInt32);
    virtual void getTimeout(ViPInt32);
    virtual void init(const String &, const Boolean &, const Boolean &);
    virtual void reset();
    virtual void revision_query(ViChar[], ViChar[]);
    virtual void self_test(ViPInt32, ViChar[]);
    virtual void setInstrumentErrorDetection(const Int32 &, const Int32 &);
    virtual void setMeasTimeout(const Int32 &);
    virtual void setRegister(const Int32 &, const Int32 &);
    virtual void setTimeout(const Int32 &);
    virtual void trigger(const Int32 &);
    virtual void wait();

protected:
    std::string m_DriverRevision;
    std::string m_InstrumentId;
    ViInt32 m_MeasurementTimeout;
    std::string m_Model;
    ViInt32 m_Timeout;

    void beginMeasurement();
    void checkNonNegativeInt(const Int32 &);
    void endMeasurement();

    // Called by VisaSession at the end of every instrument command
    virtual void commandComplete();
    // Called by the init function after initialization has completed
    virtual void internalInit(const std::string &) = 0;

private:
    // Variables needed by setInstrumentErrorDetection
    ViInt32 m_InstrumentErrorDetectionMode;
    ViInt32 m_InstrumentErrorWait;

    // Support functions for setInstrumentErrorDetection
    void disableInstrumentErrorInterruption();
    void enableInstrumentErrorInterruption(const Int32 &);

    virtual void checkLanguage() = 0;
    virtual bool isValidModel(const std::string &) const = 0;
    static std::string parseModel(const std::string &);
};

#endif