///////////////////////////////////////////////////////////////////////////////
//              This program is property of
///////////////////////////////////////////////////////////////////////////////
//
//
//                   ****   **** 
//                 ***  /_  _  ***     HEWLETT
//                 **  / / /_/  **
//                 ***    /    ***     PACKARD
//                   ****   **** 
//
//
///////////////////////////////////////////////////////////////////////////////
//              Copyright (C) 1999 Hewlett-Packard Company
///////////////////////////////////////////////////////////////////////////////

 
//-----------------------------------------------------------------------------
// File name     : hpe8311aExample.c
//-----------------------------------------------------------------------------
// File contents : Example program to introduce to usage of hpe8311a_32.DLL
//-----------------------------------------------------------------------------


#include "visa.h"             // usage of visa-functions like viOpen()
#include "hpe8311a.h"         // usage of driver-functions like hpe8311a_init()
#include "hpe8311aExample.h"  // forward-declarations for this file

#include <stdio.h>
#include <string.h>


#define INSTRUMENT_RESOURCE_DESCRIPTION "GPIB-VXI0::111::INSTR"





//*****************************************************************************
void scpiCommandLoop(
  ViSession instrumentHandle
  )
//*****************************************************************************
{
  ViStatus errStatus;
  ViChar   cmd[HPE8311A_STRING_LENGTH];
  ViChar   response[60*1024];
  ViChar*  helpPtr;


  //---------------------------------------------------------------------------
  // Switch on automatic instrument-error quering
  errStatus = hpe8311a_errorQueryDetect( instrumentHandle, VI_TRUE );
  checkErrorOccured( instrumentHandle, errStatus );



  //---------------------------------------------------------------------------
  // Loop: 
  do
  {

    // Inform the user about the loop and the way to abort
    fflush( stdin );
    printf( "____________________________________________________________\n" );
    printf( "SCPI-COMMAND-LOOP! \n" );
    printf( "  - Enter valid scpi-commands like *idn? or *opt? \n" );
    printf( "  - To finish test press >ENTER< without typing a command! \n" );

    // Get the command from the user
    fgets( cmd, HPE8311A_STRING_LENGTH, stdin );

    // Be sure that the command is correct terminated
    helpPtr = strstr( cmd, "\n" );
    if( helpPtr != NULL )
      *helpPtr = 0;

    if( strlen(cmd) != 0 )
    {
      // Send the command to the instrument (and query data)
      if( strchr(cmd, '?') == NULL )
        errStatus = hpe8311a_cmd( instrumentHandle, cmd );
      else
      {
        errStatus = hpe8311a_cmdString_Q(
          instrumentHandle, 
          cmd, 
          60*1024, 
          response );

        if ( errStatus == VI_SUCCESS )
          printf( "  RESPONSE: %s\n", response);
      }

      checkErrorOccured( instrumentHandle, errStatus ); // Error-Handling!

    }// END if( strlen(cmd) != 0 )
  }
  while( strlen(cmd) != 0 );
}


//*****************************************************************************
void demonstrateAutomaticInstrumentErrorDetect(
  ViSession instrumentHandle
  )
//*****************************************************************************
{
  ViStatus errStatus;
  ViChar   response[60*1024];
  ViUInt8  count;
  ViChar*  scpiCommands[] = {
           "*idn?",
           "*opt?",
           ":Hello:Instrument!",
           ":How:Are:You?" };



  //---------------------------------------------------------------------------
  printf( "\n" );
  printf( "____________________________________________________________\n" );
  printf( "DEMONSTRATION OF DIFFERENCES BETWEEN: \n" );
  printf( "- AUTOMATIC query for instrument-errors and \n" );
  printf( "- NO automatic query \n" );
  printf( "\n" );



  //---------------------------------------------------------------------------
  // Switch off automatic instrument-error quering
  errStatus = hpe8311a_errorQueryDetect( instrumentHandle, VI_FALSE );
  checkErrorOccured( instrumentHandle, errStatus );

  printf( "AUTOMATIC INSTRUMENT-ERROR-DETECT IS SWITCHED OFF !\n" );
  printf( "\n" );

  for( count=0; count<4; count++ )
  {
    printf( "__Sending Command to the instrument: %s \n", scpiCommands[count] );

    if( strchr( scpiCommands[count], '?') == NULL )
      errStatus = hpe8311a_cmd( instrumentHandle, scpiCommands[count] );
    else
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle, 
        scpiCommands[count], 
        60*1024, 
        response );

      if ( errStatus == VI_SUCCESS )
        printf( "  RESPONSE: %s\n", response);
    }

    printf( "  Calling the error-check-routine! \n" );
    checkErrorOccured( instrumentHandle, errStatus ); // Error-Handling!
    printf( "\n" );
  }



  //---------------------------------------------------------------------------
  // Switch on automatic instrument-error quering
  errStatus = hpe8311a_errorQueryDetect( instrumentHandle, VI_TRUE );
  checkErrorOccured( instrumentHandle, errStatus );

  printf( "\n\n" );
  printf( "AUTOMATIC INSTRUMENT-ERROR-DETECT IS SWITCHED ON !\n" );
  printf( "  now actual and OLD errors are reported !\n" );
  printf( "\n" );


  for( count=0; count<4; count++ )
  {
    printf( "__Sending Command to the instrument: %s \n", scpiCommands[count] );

    if( strchr( scpiCommands[count], '?') == NULL )
      errStatus = hpe8311a_cmd( instrumentHandle, scpiCommands[count] );
    else
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle, 
        scpiCommands[count], 
        60*1024, 
        response );

      if ( errStatus == VI_SUCCESS )
        printf( "  RESPONSE: %s\n", response);
    }

    printf( "  Calling the error-check-routine! \n" );
    checkErrorOccured( instrumentHandle, errStatus ); // Error-Handling!
    printf( "\n" );
  }

}


//*****************************************************************************
void programInstrument(
  ViSession instrumentHandle
  )
//*****************************************************************************
{
  ViStatus errStatus;
  enum {
    RESET,
    SET_MODE,
    SET_PULSE_TYPE,
    SET_PERIOD,
    SET_OUTPUT,
    MACHINE_FINISHED,
    ERROR_OCCURED
  } stateMachine ;



  //---------------------------------------------------------------------------
  printf( "\n" );
  printf( "____________________________________________________________\n" );
  printf( "PROGRAM INSTRUMENT WITH SEVERAL PARAMETER: \n" );
  printf( "\n" );

    

  //---------------------------------------------------------------------------
  // Switch off automatic instrument-error querying
  errStatus = hpe8311a_errorQueryDetect( instrumentHandle, VI_FALSE );
  checkErrorOccured( instrumentHandle, errStatus );
  printf( "AUTOMATIC INSTRUMENT-ERROR-DETECT IS SWITCHED OFF !\n" );
  printf( "  this makes programming faster - but not safer!\n" );
  printf( "\n" );



  //---------------------------------------------------------------------------
  // Do the programming

  stateMachine = SET_MODE;

  do
  {
    switch( stateMachine )
    {
      //_______________________________________________________________________
      case  RESET:

            printf( "__calling function: hpe8311a_reset().. \n" );

            errStatus = hpe8311a_reset(
              instrumentHandle
              );

            stateMachine = SET_MODE;
            break;
      //_______________________________________________________________________
      case  SET_MODE:

            printf( "__calling function: hpe8311a_modeContinuousBurst().. \n" );
            printf( "  to etablish an apropriate instrument-mode \n" );

            errStatus = hpe8311a_modeContinuousBurst(
              instrumentHandle,
              200,
              HPE8311A_INTERNAL_OSCILLATOR,
              HPE8311A_RISING
              );

            stateMachine = SET_PULSE_TYPE;
            break;
      //_______________________________________________________________________
      case  SET_PULSE_TYPE:

            printf( "__calling function: hpe8311a_modePulseType().. \n" );
            printf( "  to select single-pulses for channel one \n" );

            errStatus = hpe8311a_modePulseType(
              instrumentHandle,
              HPE8311A_CHANNEL_1,
              HPE8311A_SINGLE_PULSES
              );

            stateMachine = SET_PERIOD;
            break;
      //_______________________________________________________________________
      case  SET_PERIOD:

            printf( "__calling function: hpe8311a_timePeriod().. \n" );
            printf( "  to set period on 12.3 uSec \n" );

            errStatus = hpe8311a_timePeriod(
              instrumentHandle,
              12.3e-6
              );

            stateMachine = SET_OUTPUT;
            break;
      //_______________________________________________________________________
      case  SET_OUTPUT:

            printf( "__calling function: hpe8311a_outputStateNormal().. \n" );
            printf( "  to switch output one on \n" );

            errStatus = hpe8311a_outputStateNormal(
              instrumentHandle,
              HPE8311A_CHANNEL_1,
              HPE8311A_OUTPUT_NORMAL_ON
              );

            stateMachine = ERROR_OCCURED;
            break;
      //_______________________________________________________________________
      case  ERROR_OCCURED:

            printf( "__Calling the error-check-routine for driver-errors.. \n" );

            checkErrorOccured(
              instrumentHandle,
              errStatus 
              );

            stateMachine = MACHINE_FINISHED;
            break;
      //_______________________________________________________________________
    };//END switch( stateMachine )


    if(  (errStatus<VI_SUCCESS)  &&  (stateMachine!=MACHINE_FINISHED)  )
      stateMachine = ERROR_OCCURED;

    printf("_Press ENTER to continue."); getc(stdin);
  }
  while( stateMachine != MACHINE_FINISHED );


  //---------------------------------------------------------------------------
  // Check the instrument-errors - since automatic 
  //   instrument-error-detect is switched off.


  // check driver-errors
  printf( "  Calling the error-check-routine for instrument-errors! \n" );
  checkErrorOccured(
    instrumentHandle,
    HPE8311A_INSTR_ERROR_DETECTED
    );

}


//*****************************************************************************
ViStatus checkErrorOccured(
  ViSession instrumentHandle,
  ViStatus  outsideErrStatus
  )
//*****************************************************************************
{
  ViInt32  instrumentError;
  ViChar   errorMessage[HPE8311A_STRING_LENGTH];
  ViStatus privateErrStatus = VI_SUCCESS;

  errorMessage[0] = '\0';


  if( outsideErrStatus < VI_SUCCESS )
  {
    //-------------------------------------------------------------------------
    // Only doing error-reporting, when there really occured an error:
    //   Stay silent when errState is SUCCESSfully.
    printf( "\n" );
    printf( "  ERROR-REQUEST! \n" );



    //-------------------------------------------------------------------------
    // Send a device clear to ensure communication with     
    //   the instrument.                                    
    hpe8311a_dcl( instrumentHandle );

  

    //-------------------------------------------------------------------------
    if( outsideErrStatus == HPE8311A_INSTR_ERROR_DETECTED )
    {

      // If the driver is set to detect instrument errors,    
      //   and an instrument error is detected, the error     
      //   code is HPE8311A_INSTR_ERROR_DETECTED (see         
      //   hpe8311a.h).

      // Query the instrument for pendant errors in a loop 
      //   until all errors are found and print them.
      // When there are no more instrument-errors, the instrument
      //   reports as instrumentError: VI_SUCCESS.
      do
      {
        privateErrStatus = hpe8311a_error_query( 
          instrumentHandle,
          &instrumentError,
          errorMessage );

        printf( "  <ERROR-INSTRUMENT> : %ld, %s\n", 
          instrumentError, 
          errorMessage ); 
      }
      while(  (instrumentError!=0) && (privateErrStatus==VI_SUCCESS) );

      outsideErrStatus = privateErrStatus;
        // When there occured an additonal driver error while querying the 
        //   instrument, we want to report it later.
        // When all was ok, it contains success-state, to avoid that there
        //   is reported a driver-error later.


    };//END if( outsideErrStatus == HPE8311A_INSTR_ERROR_DETECTED )



    //-------------------------------------------------------------------------
    // Variable outsideErrorStatus contains an driver-error-code
    //   eigher coming from outside the function or generated while querying 
    //   the instrument for instrument-errors before.
    // We have to ask again if there really an error occured, since the
    //   query for instrument-errors may be run without causing driver-errors.

    if( outsideErrStatus < VI_SUCCESS )
    {
      
      // Query the driver for that error-code and display it.               
      privateErrStatus = hpe8311a_error_message( 
        instrumentHandle, 
        outsideErrStatus, 
        errorMessage ); 

      if( privateErrStatus == VI_SUCCESS )
      {
        // function: hpe8311a_error_message found the error-message successfuly
        //   we can report errorNumber and errorMessage
        printf( "  <ERROR-DRIVER> : %ld, %s\n", 
          outsideErrStatus, 
          errorMessage );
      }
      else
      {
        if( errorMessage[0] != 0 )
        {
          // function: hpe8311a_error_message failed, but there is a message
          //   we can only report an errorMessage
          printf( "  <ERROR-DRIVER> : %ld, %s\n", 
            outsideErrStatus, 
            errorMessage );
        }
      };//END if( privateErrStatus == VI_SUCCESS )
    };//END if( outsideErrStatus < VI_SUCCESS )


    
    //-------------------------------------------------------------------------
    printf( "\n" );

  };//END if( outsideErrStatus < VI_SUCCESS )
  
  return( privateErrStatus );
}



//*****************************************************************************
// This is the work function main() has to do:
//   - Initialize the handle to the instument.
//   - Request some information.
//   - Calling some working-functions.
//   - Closing the handles to the instrument.
//---------------------------------------------------------------------------
// Keep attention to intitialize and close the instrument-handle carefully,
//   since there is memory to allocate and to free and important 
//   initializations to do.
//---------------------------------------------------------------------------
// Handle the return-values of the driver-functions well - they contain the 
//   last error-state of the driver. Use some function like checkErrorOccured()
//   to find out which error occured.
//*****************************************************************************
int main(
  int argc, 
  char argv[]
  )
//*****************************************************************************
{
  ViSession	instrumentHandle = VI_NULL;
  ViSession defaultRM = VI_NULL;
  ViStatus  errStatus = VI_SUCCESS;

  ViChar    resourceName[HPE8311A_STRING_LENGTH];

  ViChar		driverRevision[HPE8311A_STRING_LENGTH];
  ViChar		instrumentRevision[HPE8311A_STRING_LENGTH];

  ViUInt32  choice;



  //---------------------------------------------------------------------------
  // Initialize the instrument-handle to the desired instrument.
  //---------------------------------------------------------------------------
  // Always call this function in an application first, before using any
  //   other driver-functions.
  //---------------------------------------------------------------------------
  // Note that this function will:
  //  - verify that the instrument specified is an HPE8311A (IDQuery=VI_TRUE)
  //  - reset the instrument (resetDevice=VI_TRUE).
  //---------------------------------------------------------------------------
  // these are valid examples of supported resourceNames
  //  - VXI0::111::INSTR         --> pure VXI-bus
  //  - GPIB-VXI0::111::INSTR    --> GPIB interface to VXI-bus
  //---------------------------------------------------------------------------

  // Edit the resourceName refering to the bus-system and the address,
  //   or use VISA function viFindRsrc() and viFindNext() to automate.
  strcpy( resourceName, INSTRUMENT_RESOURCE_DESCRIPTION );
  
  
  printf( "\n\n" );
  printf( "____________________________________________________________\n" );
  printf( "MAIN-PROGRAM: Inititialize Instrument-Handle!\n" );
  printf( "  with description: %s\n", resourceName );

  // Initialize!
  errStatus = hpe8311a_init( 
    resourceName, 
    VI_TRUE, //IDQuery
    VI_TRUE, //resetDevice
    &instrumentHandle );

  // Error-Handling!
  if( errStatus < VI_SUCCESS )
    return checkErrorOccured( instrumentHandle, errStatus );
    // There is no use to continue with the program when 
    //   function hpe8311a_init() failed!


  // Maybe the instrument has already some errors?
  //  Let's find them out!
  checkErrorOccured( instrumentHandle, HPE8311A_INSTR_ERROR_DETECTED );



  //---------------------------------------------------------------------------
  //  Query the instrument for instrument revision and the driver for  
  //    driver revision, and print both.                               
  //---------------------------------------------------------------------------

  printf( "\n\n" );
  printf( "____________________________________________________________\n" );
  printf( "MAIN-PROGRAM: Request information!\n" );

  errStatus = hpe8311a_revision_query( instrumentHandle, driverRevision, instrumentRevision );

  // Error Handling!
  checkErrorOccured( instrumentHandle, errStatus );


  if( errStatus == VI_SUCCESS )
  {
    printf( "\n" );
    printf( "  Driver Revision     : %s\n", driverRevision );
    printf( "  Instrument Revision : %s\n", instrumentRevision );
  }



  //---------------------------------------------------------------------------
  // Call working-rounines here!!
  //---------------------------------------------------------------------------

  do
  {
    printf( "\n\n" );
    printf( "____________________________________________________________\n" );
    printf( "MAIN-PROGRAM: Select working-routines here!\n" );
    fflush( stdin );
    printf( "  \n" );
    printf( "  End the program----------------> 0 \n" );
    printf( "  \n" );
    printf( "  Enter SCPI-Commands \n" );
    printf( "  within a loop -----------------> 1 \n" );
    printf( "  \n" );
    printf( "  Demonstrate differences\n" );
    printf( "  between automatic \n" );
    printf( "  instrument-error-detect \n" );
    printf( "  switched on or off ------------> 2 \n" );
    printf( "  \n" );
    printf( "  Program instrument \n" );
    printf( "  with several parameter --------> 3 \n" );
    printf( "  \n" );

    scanf( "%i", &choice );


    switch( choice )
    {
      case 0:  
        break;
      case 1: scpiCommandLoop( instrumentHandle );
        break;
      case 2: demonstrateAutomaticInstrumentErrorDetect( instrumentHandle );
        break;
      case 3: programInstrument( instrumentHandle );
        break;
      default:
        printf( "Sorry, wrong choice, try again!\n" );
        break;
    }//END switch()

  }
  while( choice != 0 );  





  //---------------------------------------------------------------------------
  // Close the instrument-handle to the desired instrument.
  //---------------------------------------------------------------------------
  // Always call this function in an application last. After used it, no
  //   further driver-function should be called, without using function
  //   hpe8311a_init() again.
  //---------------------------------------------------------------------------

  printf( "\n\n" );
  printf( "____________________________________________________________\n" );
  printf( "MAIN-PROGRAM: Close Instrument-Handle!\n" );
  printf( "with description: %s\n", resourceName );

  errStatus = hpe8311a_close( instrumentHandle );
  checkErrorOccured( instrumentHandle, errStatus );

  printf( "\n\n" );
  printf( "____________________________________________________________\n" );
  printf( "MAIN-PROGRAM: FINISHED!\n" );
  printf( "Press any key.\n" );
  fflush( stdin );
  fgetc( stdin );

  return 0;
}




