///////////////////////////////////////////////////////////////////////////////
//              This program is property of
///////////////////////////////////////////////////////////////////////////////
//
//
//                   ****   ****
//                 ***  /_  _  ***     HEWLETT
//                 **  / / /_/  **
//                 ***    /    ***     PACKARD
//                   ****   ****
//
//
///////////////////////////////////////////////////////////////////////////////
//              Copyright (C) 1999 Hewlett-Packard Company
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// File name     : hpe8311a.c
//-----------------------------------------------------------------------------
// File contents : Driver for HPE8311A Pulse-Generator
//                 Driver Version: A 1.0.2.0
//-----------------------------------------------------------------------------
// Note          : This driver is compatible with the following
//                 VXIplug&play Standards:
//                 WIN32 System Framework revision 4.0
//                 VISA revision 1.1
//-----------------------------------------------------------------------------

//*****************************************************************************
// BASIC CHANGES
// differences between this driver and the jumpstart-driver
//*****************************************************************************

//=============================================================================
// DECISIONS:
//---------------------------------------------------------------------------
// Already all functions have to make a decision:
//   The Instrument to communicate with, may either be
//   -  a register-based Module, or
//   -  a scpi-command-based Device.
//---------------------------------------------------------------------------
// Depending on this decision:
//   the received scpi-command is passed to the instrument
//   -  via a special class, that has the ability to translate this commands
//      to data and transfer that data to the instrument
//      (register-based Module).
//   -  via VISA in-output functions
//      (scpi-command-based Device)
//      [remember that the jumpstart-driver uses VISA-in-output-functions
//      especially in delayXXX()-functions!! ].
//=============================================================================

//=============================================================================
// IO-FUNCTIONS FOR MESSAGE-BASED-MODULES (mb)
//---------------------------------------------------------------------------
// In the message-based-mode the module has a handshake, which is designed
//   to fit the needs of scpi-commands. Therefore own functions are used,
//   which are in fact quite similar to the functions used in rb-mode.
//=============================================================================

//=============================================================================
// IO-FUNCTIONS FOR REGISTER-BASED-MODULES (rb)
//---------------------------------------------------------------------------
// Originally these modules are register-based, but with changes of the
//   special-bios they are message-based now.
// It is possible to force these modules from the message-based-mode
//   back into register-based-mode, and then to use the its
//   register-based ability (this is good to program a new bios).
//=============================================================================


//=============================================================================
// PARSER FOR CONTROL-SEQUENCES in message-based-mode
//---------------------------------------------------------------------------
// All strings and arrays send via the command-functions (e.g. hpe8311a_cmd)
//   parse the scpi command and the parameter for the occurence of special
//   control-sequences. Where these are found, an escape is inserted before
//   that sequence.
//---------------------------------------------------------------------------
// There exists also a special command-function named:
//   hpe8311a_cmdWithControlSequence(). With this a control-sequence can be
//   send to the instrument avoiing the control-sequence-parser and with this
//   avoiing the insertation of ESC-character.
//---------------------------------------------------------------------------
// These are the control sequences:
//   0x21 '!'
//   0x7E '~'
//   0x5E '^'
//   0x05 ENQ
//   0x07 BEL
//   0x1B ESC
//=============================================================================

//=============================================================================
// REACTION OF FUNCTIONS,
//   depending on bios-State,
//   only for VXI/GPIB-VXI instruments:
//---------------------------------------------------------------------------
// The instruments bios-State could be loader (ur-bios) or special-bios.
//   Depending from the actual bios-State some functions of the driver are
//   usefull to call, others not (cause problems).
//---------------------------------------------------------------------------
// The actual bios-State is requested from the instrument while
//   hpe8311a_init() and is stored in the global data-store of the session.
//   When hpe8311a_init() finds the loader, a error will produced, but
//   hpe8311a_init() will be left with a valid session. The customer, who
//   reads the error, can still call with this session a utility function
//   like hpe8311a_biosSpecial() or hpe8311a_biosProgramFile() to get the
//   module back into the special-bios.
//---------------------------------------------------------------------------
// Also routines, which can change the bios-State like hpe8311a_biosSpecial()
//   or hpe8311a_biosProgramFile() request the bios-State and stores it.
//---------------------------------------------------------------------------
// The stored bios-State is used to look that functions, that can cause
//   errors, when the state is the wrong one.
//---------------------------------------------------------------------------
// When the bios-State changes accidently, while the application is running,
//   the functions are not looked, because an old bios-State is still stored!
//   That causes failures, but restarting the application will clear up what
//   has gone wrong. It might be better to request HPE8311A_ALWAYS the actual
//   bios-State, but that takes time which should be saved (normally the
//   module does not change the bios-State at all).
//=============================================================================


//*****************************************************************************

//#############################################################################
//#############################################################################
//####################                                     ####################
//##########                                                         ##########
//####                   DECLARATION AND #define DATA                      ####
//##                                                                         ##
//##-------------------------------------------------------------------------##
//##                                                                         ##
//##                                                                         ##
//####                                                                     ####
//##########                                                         ##########
//####################                                     ####################
//#############################################################################
//#############################################################################

//*****************************************************************************
// MANUFACTURER_ID
//-----------------------------------------------------------------------------
#define HEWLETT_PACKARD 0x0FFF

//*****************************************************************************
// HPE8311A_REV_CODE:
//-----------------------------------------------------------------------------
#define HPE8311A_REV_CODE    "A.01.01"  /* Driver Revision */

//*****************************************************************************
// TIME-OUT START VALUE:
//-----------------------------------------------------------------------------
#define HPE8311A_GLOBAL_TIMEOUT 5000

//*****************************************************************************
// Exactly one of the following must be defined for the program to
//  compile correctly.
//
// #define __hpux          (for compilation for HP-UX)
// #define WIN32           (for compilation for 32-bit Windows)
//
// In addition, if WIN32 is defined, _CVI_ may also be defined to indicate
//  that the program is being compiled under LabWindows CVI 4.0 or
//  greater.
//*****************************************************************************

//*****************************************************************************
// The following defines are intended to provide OS compatibility among
//  Windows 32-bit and HP-UX C compilers.
//*****************************************************************************

#ifdef __hpux
# define _fcvt    fcvt
#endif

#define _huge

//*****************************************************************************

#include <stdlib.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#ifdef __hpux
#  define _INCLUDE_HPUX_SOURCE
#  define _INCLUDE_XOPEN_SOURCE_EXTENDED
#  include <sys/time.h>
#endif

#if defined(_WIN32) || defined(WIN32)
#  define WIN32_LEAN_AND_MEAN
#  if _MSC_VER >= 1000
#     pragma warning(disable : 4514) // winnt.h; inline functions not used
#     pragma warning(push, 3)
#     include <windows.h>
#     include <mmsystem.h>
#     pragma warning(pop)
#  else
#     include <windows.h>
#     include <mmsystem.h>
#  endif        /* _MSC_VER */
#endif          /* _WIN32 */

// some defines to do the correct timout calculation
#ifdef _WIN32
#define TIME_T            DWORD
#define GET_TIME()        timeGetTime()
#define DIFF_TIME(t2, t1) (t2-t1) /* may be 1 ms wrong if counter overflows */
#else
#define TIME_T            time_t
#define GET_TIME()        time(NULL)
#define DIFF_TIME(t2, t1) (1000.0*difftime(t2-t1))
#endif

#include <vpptype.h>
#include <visa.h>
#include "hpe8311a.h"

//@~ #define HPE8311A_ERR_MSG_LENGTH 256 /* size of error message buffer */


#define A16_STATUS_REG_OFFS     4   // Offset of the status register within 
                                    // the A16 address space
#define A16_READY_BIT      (1 << 9) // mask for the A16 ready bit


//*****************************************************************************
// The strings below are used to do the instrument identification in the
//  init routine.  The strings match the first part of the instrument's
//  response to it's *IDN? command.
//-----------------------------------------------------------------------------
// These strings should match the instrument's model, but should not attempt
//  to match firmware revision as a rule.  For instance, suppose the
//  instrument responds to *IDN? with
//        HEWLETT-PACKARD, 83475B, rev A.01.14
//  The appropriate IDN_STRING would be
//        HEWLETT-PACKARD, 83475
//  or
//        HEWLETT-PACKARD, 83475B
//  depending on how similar letter revisions of the instrument are.
//-----------------------------------------------------------------------------
// The driver can be used with other instruments than the following listed.
//   In such a case turn the ID-query off or add a new define-string for
//   the desired instrument.
// When turned ID-query on. The following instruments are supported by the
//   driver. When adding a new define, remember to change then also the
//   init()-routine to take notice of the new identity-string.
//-----------------------------------------------------------------------------
#define HPE8311A_IDN_SUPPORTED 8

#define HPE8311A_IDN_NUMBER_HP81101A 81101
#define HPE8311A_IDN_NUMBER_HP81104A 81104
#define HPE8311A_IDN_NUMBER_HP81110A 81110
#define HPE8311A_IDN_NUMBER_HPE8305A  8305
#define HPE8311A_IDN_NUMBER_HPE8306A  8306
#define HPE8311A_IDN_NUMBER_HPE8309A  8309
#define HPE8311A_IDN_NUMBER_HPE8311A  8311
#define HPE8311A_IDN_NUMBER_HPE8312A  8312

#define HPE8311A_IDN_STRING_HP81101A  "HEWLETT-PACKARD,HP81101A"
#define HPE8311A_IDN_STRING_HP81104A  "HEWLETT-PACKARD,HP81104A"
#define HPE8311A_IDN_STRING_HP81110A  "HEWLETT-PACKARD,HP81110A"
#define HPE8311A_IDN_STRING_HPE8305A  "HEWLETT-PACKARD,E8305"
#define HPE8311A_IDN_STRING_HPE8306A  "HEWLETT-PACKARD,E8306"
#define HPE8311A_IDN_STRING_HPE8309A  "HEWLETT-PACKARD,E8309"
#define HPE8311A_IDN_STRING_HPE8311A  "HEWLETT-PACKARD,E8311"
#define HPE8311A_IDN_STRING_HPE8312A  "HEWLETT-PACKARD,E8312"

struct identityStc
{
  ViUInt32 identityNumber;
  ViChar identityStcing[HPE8311A_STRING_LENGTH];
};

struct identityStc identityStcI[HPE8311A_IDN_SUPPORTED] = {
  {HPE8311A_IDN_NUMBER_HP81101A, HPE8311A_IDN_STRING_HP81101A},
  {HPE8311A_IDN_NUMBER_HP81104A, HPE8311A_IDN_STRING_HP81104A},
  {HPE8311A_IDN_NUMBER_HP81110A, HPE8311A_IDN_STRING_HP81110A},
  {HPE8311A_IDN_NUMBER_HPE8305A, HPE8311A_IDN_STRING_HPE8305A},
  {HPE8311A_IDN_NUMBER_HPE8306A, HPE8311A_IDN_STRING_HPE8306A},
  {HPE8311A_IDN_NUMBER_HPE8309A, HPE8311A_IDN_STRING_HPE8309A},
  {HPE8311A_IDN_NUMBER_HPE8311A, HPE8311A_IDN_STRING_HPE8311A},
  {HPE8311A_IDN_NUMBER_HPE8312A, HPE8311A_IDN_STRING_HPE8312A}
};

//*****************************************************************************
// General Purpose Utility functions go here
//*****************************************************************************

/* --------------------------------------------------------------------------
 * The safe version of free() used with..
 * #define FREE(ptr) Free((void **)&(ptr))
 * -------------------------------------------------------------------------- */

#define FREE(ptr) Free((void **)&(ptr))

static void Free(void **ptr)
{
  if (*ptr)
  {
    free(*ptr);
    *ptr = NULL;
  }
}


//*****************************************************************************
// VISA Globals data structure.
// It is desireable to store global variables in VISA rather than the
//  driver.  This avoids having any state variables in the driver.  This
//  structure defines the driver's global data.  Memory is allocated to
//  this structure in the init function, which passes the pointer to VISA,
//  and every function which follows retrieves a pointer to it.
//-----------------------------------------------------------------------------
struct hpe8311a_globals
{
  ViStatus errNumber;
  ViChar errMessage[HPE8311A_STRING_LENGTH];
  ViBoolean errQueryDetect;
  ViInt32 dTime;
  ViUInt16 interfaceType;
  ViChar resourceName[HPE8311A_STRING_LENGTH];
  ViUInt16 modelCode;
  ViInt32 globalTimeOut;
  ViUInt32 deviceIdentity;
  ViUInt32 outputType;
  ViPChar pTransBuffer;       // NULL unless a transaction is in progress
  ViUInt32 TransBufferLength; // Current size of buffer
  struct hpe8311a_options {
    ViInt16 trans_unit;
  } options;
};

#define XACT_BUFF_SIZE  (30 * 1024)

//*****************************************************************************
// local unprotected...caller guarantees pThis!

static void transactionCancel(struct hpe8311a_globals * pThis)
{
  // free any transaction and reset length
  FREE(pThis->pTransBuffer);
  pThis->TransBufferLength = 0;
}



//*****************************************************************************
// Driver Message Strings.
//*****************************************************************************

#define HPE8311A_MSG_VI_OPEN_ERR            "vi was zero.  Was the hpe8311a_init() successful?"
#define HPE8311A_MSG_INVALID_STATUS         "Parameter 2 is invalid in function hpe8311a_error_message()."
#define HPE8311A_MSG_INVALID_STATUS_VALUE   "is not a valid ViStatus value."
#define HPE8311A_MSG_NO_ERRORS              "No Errors"
#define HPE8311A_MSG_SELF_TEST_FAILED       "Self test failed."
#define HPE8311A_MSG_SELF_TEST_PASSED       "Self test passed."

#define HPE8311A_MSG_BOOLEAN                "Expected 0 or 1; Got %hd"
#define HPE8311A_MSG_LONG                   "Expected %ld to %ld; Got %ld"

#define VI_ERROR_PARAMETER1_MSG           "Parameter 1 is invalid"
#define VI_ERROR_PARAMETER2_MSG           "Parameter 2 is invalid"
#define VI_ERROR_PARAMETER3_MSG           "Parameter 3 is invalid"
#define VI_ERROR_PARAMETER4_MSG           "Parameter 4 is invalid"
#define VI_ERROR_PARAMETER5_MSG           "Parameter 5 is invalid"
#define VI_ERROR_PARAMETER6_MSG           "Parameter 6 is invalid"
#define VI_ERROR_PARAMETER7_MSG           "Parameter 7 is invalid"
#define VI_ERROR_PARAMETER8_MSG           "Parameter 8 is invalid"
#define VI_ERROR_PARAMETER9_MSG           "Parameter 9 is invalid"
#define VI_ERROR_PARAMETER10_MSG          "Parameter 10 is invalid"
#define VI_ERROR_PARAMETER11_MSG          "Parameter 11 is invalid"
#define VI_ERROR_PARAMETER12_MSG          "Parameter 12 is invalid"
#define VI_ERROR_PARAMETER13_MSG          "Parameter 13 is invalid"
#define VI_ERROR_PARAMETER14_MSG          "Parameter 14 is invalid"
#define VI_ERROR_PARAMETER15_MSG          "Parameter 15 is invalid"
#define VI_ERROR_PARAMETER16_MSG          "Parameter 16 is invalid"
#define VI_ERROR_PARAMETER17_MSG          "Parameter 17 is invalid"
#define VI_ERROR_PARAMETER18_MSG          "Parameter 18 is invalid"
#define VI_ERROR_FAIL_ID_QUERY_MSG        "Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG       "Invalid session; operation cannot be performed without a valid VISA session"
#define INSTR_ERROR_NULL_PTR_MSG          "NULL pointer detected"
#define INSTR_ERROR_RESET_FAILED_MSG      "reset failed"
#define INSTR_ERROR_UNEXPECTED_MSG        "An unexpected error occurred"
#define INSTR_ERROR_DETECTED_MSG          "Instrument Error Detected, call hpe8311a_error_query()."
#define INSTR_ERROR_LOOKUP_MSG            "String not found in table"
#define INSTR_ERROR_NO_LAST_COMMA         "no last comma found in IDN string"
#define INSTR_ERROR_CANNOT_READ_OPTIONS_MSG "cannot read instrument options"

#define INSTR_ERROR_NOT_SUPPORTED_INSTR_TYPE \
        "Unsupported instrument-type detected!"

#define INSTR_ERROR_POUND_CHAR_NOT_IN_INSTRUMENT_ANSWER \
        "Instrument output-buffer: expected a '#'-char"

#define INSTR_ERROR_LENGTH_BYTE_CONTAINS_INVALID_VALUE \
        "Instrument output-buffer: length-byte invalid (0 < x < 10)"

#define INSTR_ERROR_OUTPUT_DATA_GREATER_READ_BUFFER \
        "Instrument output-buffer: data length is greater than provided buffer"

#define INSTR_ERROR_CANNOT_OPEN_FILE \
        "Cannot open the specified file!"

#define INSTR_ERROR_CANNOT_CLOSE_FILE \
        "Cannot close the specified file!"

#define INSTR_ERROR_SOURCE_FILE_IS_EMPTY \
        "Source file is empty!"

#define INSTR_ERROR_WRONG_STATE \
        "Module found in an unexpected ready-state!"

#define INSTR_ERROR_BIOS_FILE_CHECKSUM_ERROR \
        "Bios-File contains data with invalid checksum!"

#define INSTR_ERROR_CANT_SWAP_ODD_BYTE_QUANTITY \
        "Cannot swap odd byte-count: following byte may be overwritten!"

#define INSTR_ERROR_PROGRAM_BIOS_NOT_SUPPORTED_FOR_GPIB \
        "Function: biosProgram() is not supported by GPIB-devices!"

#define INSTR_ERROR_SPECIAL_BIOS_NOT_SUPPORTED_FOR_GPIB \
        "Function: biosSpecial() is not supported by GPIB-devices!"

#define INSTR_ERROR_FUNCTION_SPECIAL_BIOS_EXPECTED_URBIOS \
        "Function biosSpecial() expected modules bios-state in loader-state!"

#define INSTR_ERROR_DOR_GREATER_THAN_OUTPUT_BUFFER \
        "Value in data-out-register: block size larger than output buffer!"

#define INSTR_ERROR_CANNOT_SET_MODULE_TO_READY_STATE \
        "Cannot set module to ready state!"

#define INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT \
        "Driver internal handshake timeout occurred"

#define INSTR_ERROR_FOUND_MODULE_IN_URBIOS \
        "Instrument is in boot loader: use function hpe8311a_biosSpecial() to start firmware"

#define INSTR_ERROR_NOT_A_HEWLETT_PACKARD_MODULE \
        "Function: hpe8311a_init() failed - can only initialize a module from HEWLETT-PACKARD!"

#define INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE \
        "Instrument's answer is invalid (not in the permissible range)!"

#define INSTR_ERROR_UNEXPECTED_VALUE_IN_INSTRUMENTS_ANSWER \
        "Instrument's answer contains an unexpected value!"

#define INSTR_ERROR_SOURCE_FILE_IS_TOO_LARGE \
        "Source file is too large!"

#define INSTR_ERROR_CALIBRATION_FAILED \
        "Calibration of timing system failed!"

#define INSTR_ERROR_VARIABLE_OVERFLOW \
        "Overflow of driver-internal variable occurred!"

#define INSTR_ERROR_STATE_MACHINE_ERROR \
        "Driver internal error occurred - unexpected state"

#define INSTR_ERROR_NOT_SUPPORTED_FOR_VXI \
        "Called function not supported for a VXI/GPIB-VXI instrument!"

#define INSTR_ERROR_CANNOT_WRITE_TO_FILE \
        "Error occurred while writing data to a file!"

#define INSTR_ERROR_CANNOT_READ_FROM_FILE \
        "Error occurred while reading data from a file!"

#define INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT \
        "Error occurred while reading data from a file - file contents are corrupt!"

#define INSTR_ERROR_DOR_EQUAL_ZERO \
        "Value in data-out-register zero: instrument returned zero bytes!"

#define INSTR_ERROR_REQUESTED_ERROR_KOMMA_MISSED \
        "Error while requesting instrument errors - instruments answer is missing a comma!"

#define INSTR_ERROR_TOO_MANY_ACTIVE_TRIGGER_REPORTED \
        "Error while requesting active vxi-trigger-line: more than one active line reported!"

#define INSTR_ERROR_UNSUPPORTED_INTERFACE \
        "Unsupported Interface type!"

#define INSTR_ERROR_TRANSACTION_IN_PROG \
        "A Transaction is already in progress!"

#define INSTR_ERROR_TRANSACTION_NOT_IN_PROG \
        "No Transaction is in progress!"

#define INSTR_ERROR_TRANSACTION_BUFFER_OVERFLOW \
        "Transaction buffer overflow...transaction voided and canceled!"

//*****************************************************************************
// Error table structure.
//  The first element of the table is the error number, the second is the
//  error message.  Error numbers in the "VISA Generic" section of the
//  table are defined in the VISA header files.  Error numbers in the
//  "Instrument Specific" section are defined in the driver header file.
//  All of the error messages are defined above.
//*****************************************************************************

struct instrErrStruct
{
  ViStatus errStatus;
  ViString errMessage;
};

static const struct instrErrStruct instrErrMsgTable[] =
{
  {VI_ERROR_FAIL_ID_QUERY,    VI_ERROR_FAIL_ID_QUERY_MSG},

  {VI_ERROR_PARAMETER1, VI_ERROR_PARAMETER1_MSG},
  {VI_ERROR_PARAMETER2, VI_ERROR_PARAMETER2_MSG},
  {VI_ERROR_PARAMETER3, VI_ERROR_PARAMETER3_MSG},
  {VI_ERROR_PARAMETER4, VI_ERROR_PARAMETER4_MSG},
  {VI_ERROR_PARAMETER5, VI_ERROR_PARAMETER5_MSG},
  {VI_ERROR_PARAMETER6, VI_ERROR_PARAMETER6_MSG},
  {VI_ERROR_PARAMETER7, VI_ERROR_PARAMETER7_MSG},
  {VI_ERROR_PARAMETER8, VI_ERROR_PARAMETER8_MSG},

  {HPE8311A_INSTR_ERROR_PARAMETER9,   VI_ERROR_PARAMETER9_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER10,  VI_ERROR_PARAMETER10_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER11,  VI_ERROR_PARAMETER11_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER12,  VI_ERROR_PARAMETER12_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER13,  VI_ERROR_PARAMETER13_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER14,  VI_ERROR_PARAMETER14_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER15,  VI_ERROR_PARAMETER15_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER16,  VI_ERROR_PARAMETER16_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER17,  VI_ERROR_PARAMETER17_MSG},
  {HPE8311A_INSTR_ERROR_PARAMETER18,  VI_ERROR_PARAMETER18_MSG},

  {HPE8311A_INSTR_ERROR_INV_SESSION,    INSTR_ERROR_INV_SESSION_MSG},
  {HPE8311A_INSTR_ERROR_NULL_PTR,       INSTR_ERROR_NULL_PTR_MSG},
  {HPE8311A_INSTR_ERROR_RESET_FAILED,   INSTR_ERROR_RESET_FAILED_MSG},
  {HPE8311A_INSTR_ERROR_UNEXPECTED,     INSTR_ERROR_UNEXPECTED_MSG},
  {HPE8311A_INSTR_ERROR_DETECTED,       INSTR_ERROR_DETECTED_MSG},
  {HPE8311A_INSTR_ERROR_LOOKUP,         INSTR_ERROR_LOOKUP_MSG},
  {HPE8311A_INSTR_NO_LAST_COMMA,        INSTR_ERROR_NO_LAST_COMMA},

  {HPE8311A_INSTR_ERROR_NOT_SUPPORTED_INSTR_TYPE,            INSTR_ERROR_NOT_SUPPORTED_INSTR_TYPE},
  {HPE8311A_INSTR_ERROR_POUND_CHAR_NOT_IN_INSTRUMENT_ANSWER, INSTR_ERROR_POUND_CHAR_NOT_IN_INSTRUMENT_ANSWER},
  {HPE8311A_INSTR_ERROR_LENGTH_BYTE_CONTAINS_INVALID_VALUE,  INSTR_ERROR_LENGTH_BYTE_CONTAINS_INVALID_VALUE},
  {HPE8311A_INSTR_ERROR_OUTPUT_DATA_GREATER_READ_BUFFER,     INSTR_ERROR_OUTPUT_DATA_GREATER_READ_BUFFER},
  {HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE,                    INSTR_ERROR_CANNOT_OPEN_FILE},
  {HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE,                   INSTR_ERROR_CANNOT_CLOSE_FILE},
  {HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_EMPTY,                INSTR_ERROR_SOURCE_FILE_IS_EMPTY},
  {HPE8311A_INSTR_ERROR_WRONG_STATE,                         INSTR_ERROR_WRONG_STATE},
  {HPE8311A_INSTR_ERROR_BIOS_FILE_CHECKSUM_ERROR,            INSTR_ERROR_BIOS_FILE_CHECKSUM_ERROR},
  {HPE8311A_INSTR_ERROR_CANT_SWAP_ODD_BYTE_QUANTITY,         INSTR_ERROR_CANT_SWAP_ODD_BYTE_QUANTITY},
  {HPE8311A_INSTR_ERROR_PROGRAM_BIOS_NOT_SUPPORTED_FOR_GPIB, INSTR_ERROR_PROGRAM_BIOS_NOT_SUPPORTED_FOR_GPIB},
  {HPE8311A_INSTR_ERROR_SPECIAL_BIOS_NOT_SUPPORTED_FOR_GPIB, INSTR_ERROR_SPECIAL_BIOS_NOT_SUPPORTED_FOR_GPIB},

  {HPE8311A_INSTR_ERROR_FUNCTION_SPECIAL_BIOS_EXPECTED_URBIOS,   INSTR_ERROR_FUNCTION_SPECIAL_BIOS_EXPECTED_URBIOS},

  {HPE8311A_INSTR_ERROR_DOR_GREATER_THAN_OUTPUT_BUFFER,   INSTR_ERROR_DOR_GREATER_THAN_OUTPUT_BUFFER},
  {HPE8311A_INSTR_ERROR_CANNOT_SET_MODULE_TO_READY_STATE, INSTR_ERROR_CANNOT_SET_MODULE_TO_READY_STATE},
  {HPE8311A_INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT,          INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT},
  {HPE8311A_INSTR_ERROR_FOUND_MODULE_IN_URBIOS,           INSTR_ERROR_FOUND_MODULE_IN_URBIOS},
  {HPE8311A_INSTR_ERROR_NOT_A_HEWLETT_PACKARD_MODULE,     INSTR_ERROR_NOT_A_HEWLETT_PACKARD_MODULE},
  {HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE,   INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE},

  {HPE8311A_INSTR_ERROR_UNEXPECTED_VALUE_IN_INSTRUMENTS_ANSWER,   INSTR_ERROR_UNEXPECTED_VALUE_IN_INSTRUMENTS_ANSWER},

  {HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_TOO_LARGE,   INSTR_ERROR_SOURCE_FILE_IS_TOO_LARGE},
  {HPE8311A_INSTR_ERROR_CALIBRATION_FAILED,         INSTR_ERROR_CALIBRATION_FAILED},
  {HPE8311A_INSTR_ERROR_VARIABLE_OVERFLOW,          INSTR_ERROR_VARIABLE_OVERFLOW},
  {HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR,        INSTR_ERROR_STATE_MACHINE_ERROR},
  {HPE8311A_INSTR_ERROR_NOT_SUPPORTED_FOR_VXI,      INSTR_ERROR_NOT_SUPPORTED_FOR_VXI},
  {HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE,       INSTR_ERROR_CANNOT_WRITE_TO_FILE},
  {HPE8311A_INSTR_ERROR_CANNOT_READ_FROM_FILE,      INSTR_ERROR_CANNOT_READ_FROM_FILE},
  {HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT,   INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT},
  {HPE8311A_INSTR_ERROR_DOR_EQUAL_ZERO,             INSTR_ERROR_DOR_EQUAL_ZERO},

  {HPE8311A_INSTR_ERROR_REQUESTED_ERROR_KOMMA_MISSED,     INSTR_ERROR_REQUESTED_ERROR_KOMMA_MISSED},
  {HPE8311A_INSTR_ERROR_TOO_MANY_ACTIVE_TRIGGER_REPORTED, INSTR_ERROR_TOO_MANY_ACTIVE_TRIGGER_REPORTED},
  {HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE,            INSTR_ERROR_UNSUPPORTED_INTERFACE},
  {HPE8311A_INSTR_ERROR_CANNOT_READ_OPTIONS,              INSTR_ERROR_CANNOT_READ_OPTIONS_MSG},

  {HPE8311A_INSTR_ERROR_TRANSACTION_IN_PROG,              INSTR_ERROR_TRANSACTION_IN_PROG},
  {HPE8311A_INSTR_ERROR_TRANSACTION_NOT_IN_PROG,          INSTR_ERROR_TRANSACTION_NOT_IN_PROG},
  {HPE8311A_INSTR_ERROR_TRANSACTION_BUFFER_OVERFLOW,      INSTR_ERROR_TRANSACTION_BUFFER_OVERFLOW}

  // {HPE8311A_},
};
//*****************************************************************************
// Defines for state-machines
//-----------------------------------------------------------------------------
#define STATE_MACHINE_NO_FINISH                         0
#define STATE_MACHINE_DO_FINISH                         1
#define STATE_MACHINE_FINISH                            2
#define STATE_MACHINE_ERROR_OCCURRED                    3

#define STATE_MACHINE_GENERAL_COMMANDS                 10

#define STATE_MACHINE_BURST_LENGTH                     20
#define STATE_MACHINE_PERIOD_SOURCE                    21
#define STATE_MACHINE_CLOCK_EDGE                       22
#define STATE_MACHINE_TRIGGER_SOURCE                   23
#define STATE_MACHINE_TRIGGER_SLOPE                    24
#define STATE_MACHINE_GATE_SOURCE                      25
#define STATE_MACHINE_GATE_LEVEL                       26
#define STATE_MACHINE_PLL_PERIOD_FREQUENCY             27
#define STATE_MACHINE_REQUEST_TRIGGER_COUPLING         28
#define STATE_MACHINE_REQUEST_TRIGGER_SOURCE           29

#define STATE_MACHINE_REQUEST_TIMEOUT                  40
#define STATE_MACHINE_SET_NEW_TIMEOUT                  41
#define STATE_MACHINE_RESTORE_OLD_TIMEOUT              42

#define STATE_MACHINE_CALIBRATE_INSTRUMENT             50


#define STATE_MACHINE_OPEN_FILE                        60
#define STATE_MACHINE_CLOSE_FILE                       61

#define STATE_MACHINE_MALLOC_MEMORY                    70
#define STATE_MACHINE_FREE_MEMORY                      71

#define STATE_MACHINE_PREPARE_DATA                     80
#define STATE_MACHINE_ANALYZE_RESULTS                  81
#define STATE_MACHINE_RETURN_RESULTS                   82

//*****************************************************************************
// Global String Array for source or gate parameter
//-----------------------------------------------------------------------------
ViChar *sourceArray[] = {
  "IMM",
  "INT2",
  "EXT2",
  "EXT",
  "MAN",
  "ECLT0",
  "ECLT1",
  "TTLT0",
  "TTLT1",
  "TTLT2",
  "TTLT3",
  "TTLT4",
  "TTLT5",
  "TTLT6",
  "TTLT7"
};

/* --------------------------------------------------------------------------
 * MACROS
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Macros for both debug and release versions
 * -------------------------------------------------------------------------- */

// the instrument loader-state
#define LOADER_STATE  0x999

// most functions can't handle loader-state.
#define CHECK_URBIOS \
{ \
  if( thisPtr->modelCode == LOADER_STATE ) \
  { \
    DBG_PRINT("Module found in loader-state!"); \
    ERR_RETURN( HPE8311A_INSTR_ERROR_FOUND_MODULE_IN_URBIOS ); \
  } \
}

/* --------------------------------------------------------------------------
 * The debug macros
 * -------------------------------------------------------------------------- */

#ifdef DEBUG_VERBOSE

# if defined(_MSC_VER)
#   pragma message("Compiling with debug output")
# endif

// Routine to write into a debug file
static FILE *debugFile = NULL;         // pointer to a file
static ViChar debugFilename[] = {"c:/hpe83dbg.txt"};
static ViChar debugMessage[HPE8311A_STRING_LENGTH];

static void DebugPrintf(char debMessage[])
{
  // for speed we never close the file ... auto-done at program close
  if(NULL == debugFile) 
  {
    if(NULL != (debugFile = fopen(debugFilename, "w")))
      fprintf(debugFile, debMessage);
  }
  else
  {
    fprintf(debugFile, debMessage);
  }
}

// a quick-write (non-formatted)
static void _dbgString(const char * msg)
{
  // for speed we never close the file ... auto-done at program close
  if(NULL == debugFile) 
  {
    if(NULL != (debugFile = fopen(debugFilename, "w")))
      fputs(msg, debugFile);
  }
  else
  {
    fputs(msg, debugFile);
  }

}

// Our debug print routine..
static void _dbgPrintf(const char * format,...)
{
  va_list args;
  va_start(args, format);
  (void) vsprintf(debugMessage, format, args);
  va_end(args);

  DebugPrintf(debugMessage);
}

# define NL "\r\n"

// our real trace-out routine..
static ViStatus trace_out(ViStatus err_status, const char * pFuncName)
{
  if (err_status != VI_SUCCESS)
  {
    _dbgPrintf(NL "TRACE_OUT: %s failed; error-code: %i", pFuncName, err_status);
  }
  else
  {
    _dbgString(NL "TRACE_OUT: ");
    _dbgString(pFuncName);
  }
  return err_status;
}

# define FF    " \n[%s: %d]"
# define FI    __FILE__, __LINE__

# define DBG_PRINT(string)         _dbgPrintf(NL "DBG_PRINT   " FUNC_NAME " [Line %d]: " string, __LINE__)
# define DBG_PRINT1(fmt,v1)        _dbgPrintf(NL fmt FF,(v1), FI)
# define DBG_MSG(msg)              _dbgString(NL "DBG_MSG   " msg)

# define TRACE_IN() \
  _dbgString(NL "TRACE_IN: " FUNC_NAME)

# define TRACE_VAR(fmt, var) \
  _dbgPrintf(NL "TRACE_VAR " FUNC_NAME " [Line %d]: " #var " = " fmt, __LINE__, (var))

# define TRACE_OUT(ErrStatus) return(trace_out((ErrStatus), FUNC_NAME))

# define DBG_SHOW_LOADER_STATE \
    if( thisPtr->modelCode == LOADER_STATE ) \
      DBG_PRINT("Found module in loader-state"); \
    else \
      DBG_PRINT("Found module in special-bios")

# define ERR_RETURN(_err)          TRACE_OUT(thisPtr ? thisPtr->errNumber = (_err) : (_err))

# define ERR_CHK_RETURN(_err) \
  if ((_err) < VI_SUCCESS) \
    { ERR_RETURN(_err); } \
  return (VI_SUCCESS)

#else           /* DEBUG_VERBOSE */
/* ------------------------------------------------------------------------ */

// A true 'no-op' that doesn't give a warning
#define NO_OP  ((void) 0)

# define DBG_PRINT(string)         NO_OP  /* string MUST be in quotes */
# define DBG_PRINT1(fmt,v1)        NO_OP  /* format MUST be in quotes */
# define DBG_MSG(msg)              NO_OP

# define TRACE_IN()                NO_OP  /* funcname MUST NOT be in quotes */
# define TRACE_VAR(vName, vVal)    NO_OP
# define TRACE_OUT(ErrStatus)      return((ErrStatus))

# define DBG_SHOW_LOADER_STATE     NO_OP

// do not use this in static functions!
# define ERR_RETURN(_err)          return(thisPtr ? thisPtr->errNumber = (_err) : (_err))

# define ERR_CHK_RETURN(_err) \
  if ((_err) < VI_SUCCESS) \
    { ERR_RETURN (_err); } \
  return (VI_SUCCESS)

#endif
/* This is the end of DEBUG_VERBOSE
 * ...DO NOT (!!!NOT!!!) comment the #endif above 
 */
/* ------------------------------------------------------------------------ */

//#############################################################################
//#############################################################################
//####################                                     ####################
//##########                                                         ##########
//####                   SUPPORTING MACROS AND FUNCTIONS                   ####
//##                                                                         ##
//##-------------------------------------------------------------------------##
//##     These macros and functions are not exposed to the user, but         ##
//##     are used by other functions in the driver.                          ##
//####                                                                     ####
//##########                                                         ##########
//####################                                     ####################
//#############################################################################
//#############################################################################

//*****************************************************************************
// FORWARD DHPE8311A_ECLARATIONS
// for mb-functions
//*****************************************************************************
static ViStatus _VI_FUNC mbWriteDIR(
    ViSession instrumentHandle,
    ViUInt32 dataInRegister,
    ViUInt32 value
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC mbReadDOR(
    ViSession instrumentHandle,
    ViUInt32 dataOutRegister,
    ViPUInt32 value
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC mbWriteInBuf(
    ViSession instrumentHandle,
    ViString writeBuffer,
    ViUInt32 bytesToWrite
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC mbReadOutBuf(
    ViSession instrumentHandle,
    ViPString readBuffer,
    ViUInt32 bytesToRead
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC mbSendSCPIandDATA(
    ViSession instrumentHandle,
    ViPString scpiCommandAndData,
    ViUInt32 sendBytesCount
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC mbSendSCPIreadDATA(
    ViSession instrumentHandle,
    ViPString scpiCommand,
    ViPString readBuffer,
    ViUInt32 readBufferSize,
    ViPUInt32 readBytesCount
);
//*****************************************************************************
// FORWARD DHPE8311A_ECLARATIONS
// for rb-functions
//*****************************************************************************
static ViStatus _VI_FUNC rbWriteA24Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt32 writeValue
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbReadA24Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt32 * requestedValue
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbReadA16Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt16 * requestedValue
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbWriteBlock(
    ViSession instrumentHandle,
    const ViUInt16 * data,
    ViUInt32 wordCount
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbCheckExpectedReadyState(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbWriteA24RegisterUINT(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState,
    ViInt16 regNr,
    ViInt32 writeValue
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbWriteBlockSmall(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState,
    ViInt16 regNr,
    ViUInt32 commandNr,
    const ViUInt16 * blockData,
    ViUInt32 byteCount
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbEraseBootProm(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViPInt8 foundState
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbSpecialBios(
    ViSession instrumentHandle,
    ViInt8 * foundState
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbProgramBiosFile(
    ViSession instrumentHandle,
    const ViChar filename[]
);
//---------------------------------------------------------------------------
static ViStatus _VI_FUNC rbProgramBiosBuffer(
    ViSession instrumentHandle,
    const ViChar * buffer
);
//*****************************************************************************

//*****************************************************************************
// This function converts a string to ViInt32 with extra error checking
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | integer                                              ViInt32*       OUT
//  |   ---------------------------------------------------------------------
//  |  | Destination is 32bit signed integer.
//   ------------------------------------------------------------------------
//  | string                                               ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | Source is string, with length limited to HPE8311A_STRING_LENGTH
//*****************************************************************************
static ViStatus convertViStringToInt32(
    ViInt32 * integer,
    ViChar * string
)
//*****************************************************************************
{
  // test out the input parameters
  if (integer == NULL)
    return VI_ERROR_PARAMETER1;

  if ((string == NULL) || (strlen(string) >= HPE8311A_STRING_LENGTH))
  {
    *integer = HPE8311A_INVALID_VALUE;
    return VI_ERROR_PARAMETER2;
  }

  *integer = strtol(string, NULL, 10);

  // verify the conversion
  if (ERANGE == errno)
  {
    *integer = HPE8311A_INVALID_VALUE;
    return HPE8311A_INSTR_ERROR_VARIABLE_OVERFLOW;
  }

  return VI_SUCCESS;
}

//*****************************************************************************
// This function converts a string to ViInt16 with extra error checking
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | integer                                              ViInt16*       OUT
//  |   ---------------------------------------------------------------------
//  |  | Destination is 16bit signed integer.
//   ------------------------------------------------------------------------
//  | string                                               ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | Source is string, with length limited to HPE8311A_STRING_LENGTH
//*****************************************************************************
static ViStatus convertViStringToInt16(
    ViInt16 * integer,
    ViChar * string
)
//*****************************************************************************
{
  // use the 32-bit function
  ViInt32 v_int32;
  ViStatus errStatus = convertViStringToInt32(&v_int32, string);

  // we have different limits...
  if(VI_SUCCESS == errStatus)
  {
    if(v_int32 < SHRT_MIN || v_int32 > SHRT_MAX)
      return HPE8311A_INSTR_ERROR_VARIABLE_OVERFLOW;

    *integer = (ViInt16) v_int32;
  }

  return errStatus;
}

//*****************************************************************************
// This function converts a string to ViUInt32 with extra error checking
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | integer                                              ViInt32*       OUT
//  |   ---------------------------------------------------------------------
//  |  | Destination is 32bit unsigned integer.
//   ------------------------------------------------------------------------
//  | string                                               ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | Source is string, with length limited to HPE8311A_STRING_LENGTH
//*****************************************************************************
static ViStatus convertViStringToUInt32(
    ViUInt32 * integer,
    ViChar * string
)
//*****************************************************************************
{
  // test out the input parameters
  if (integer == NULL)
    return VI_ERROR_PARAMETER1;

  if ((string == NULL) || (strlen(string) >= HPE8311A_STRING_LENGTH))
  {
    *integer = HPE8311A_INVALID_VALUE;
    return VI_ERROR_PARAMETER2;
  }

  *integer = strtoul(string, NULL, 10);

  // verify the conversion
  if (ERANGE == errno)
  {
    *integer = HPE8311A_INVALID_VALUE;
    return HPE8311A_INSTR_ERROR_VARIABLE_OVERFLOW;
  }

  return VI_SUCCESS;
}

//*****************************************************************************
// This function converts a string to ViUInt16 with extra error checking
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | integer                                              ViUInt16*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Destination is 16bit unsigned integer.
//   ------------------------------------------------------------------------
//  | string                                               ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | Source is string, with length limited to HPE8311A_STRING_LENGTH
//*****************************************************************************
static ViStatus convertViStringToUInt16(
    ViUInt16 * integer,
    ViChar * string
)
//*****************************************************************************
{
  // use the 32-bit function
  ViUInt32 v_uint32;
  ViStatus errStatus = convertViStringToUInt32(&v_uint32, string);

  // we have special limits...
  if(VI_SUCCESS == errStatus)
  {
    if(v_uint32 > USHRT_MAX)
      return HPE8311A_INSTR_ERROR_VARIABLE_OVERFLOW;

    *integer = (ViUInt16) v_uint32;
  }

  return errStatus;
}

//*****************************************************************************
// MACRO GET_GLOBALS;
//*****************************************************************************
// PARAMETERS
//  ViSession            hInstr
//    The instrument's VISA address.
//  struct hpe8311a_globals  *pThis
//    A pointer to the VISA globals storage
//*****************************************************************************
// Sets pThis to point to the VISA globals storage.  This storage was
//  allocated in init, but it is turned over to VISA to avoid holding
//  STAte variables in the driver.
//*****************************************************************************

static ViStatus get_globals(ViSession hInstr, struct hpe8311a_globals ** ppThis)
{
  ViStatus errStatus;

  if (hInstr == VI_NULL)
    return (HPE8311A_INSTR_ERROR_INV_SESSION);

  *ppThis = NULL;

  errStatus = viGetAttribute(hInstr, VI_ATTR_USER_DATA, (ViAddr) ppThis);

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  if (*ppThis == NULL)
    return (HPE8311A_INSTR_ERROR_NULL_PTR);

  return VI_SUCCESS;
}

#define GET_GLOBALS(hInstr, pThis)     \
  if( (errStatus = get_globals((hInstr), &(pThis) )) < VI_SUCCESS ) \
    return errStatus

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// DELAY Functions
// setDelay (located prior to init function) sets the delay time for the
//  instrument.
// doDelay uses delay time to slow down the driver after a read request.
//-----------------------------------------------------------------------------
// In some SCPI instruments, the instrument cannot change its STAte to
//  'processing a read request' fast enough to keep another request
//  from getting through to the instrument.  By setting a short delay
//  (less than 100 microseconds should be sufficient) after each read,
//  the driver will wait a sufficient amount of time before attempting
//  to send the next string to the instrument.
// DO NOT document these function in the fp file or the help file.  They
//  are intended to address problems in specific instruments, problems
//  that may only appear on faster CPU's.  They are designed so that the
//  problem can be addressed quickly in the field once it is identified.
// If the problem is identified by driver developers, a 100 ms delay
//  should be set in the hpe8311a_init function.  The template version of
//  init currently sets the delay to 0.
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//  long           delayTime (in)
//    The number of micro-seconds to delay before proceeding.
//*****************************************************************************
// Delays execution for a specified number of microseconds.
// Platform   Code    Tick Resolution  Tick Increment  Call Overhead
// WIN32     32-bit  CPU dependent    CPU dependent   0-20 us
// HP-UX     32-bit  1 us             CPU dependent   0-20 us
//*****************************************************************************
static void doDelay(ViInt32 delayTime)
//*****************************************************************************
{

  long ticksToCount;
  //-----------------------------------------------------------
  // WIN32 Version
  // WIN32 increments depend on the hardware.  If high-
  // frequency counting isn't supported, revert to common
  // STAndard functions (same as WIN16).
  // The call overhead to execute doDelay if high-frequency
  // is supported is about 25-30 micro-seconds.
  // The call overhead to execute doDelay if high-frequency
  // is -NOT- supported is ~10 milli-seconds on
  // Windows 95 & NT.
  //-----------------------------------------------------------

#ifdef WIN32

#ifdef _CVI_
# else
  __int64 ticksPerSecond;
  __int64 STArtTicks;
  __int64 nowTicks;
#endif          /* _CVI_ */

  DWORD msTicks;
  if (delayTime == 0)
    return;

#ifdef _CVI_
# else

  if (QueryPerformanceFrequency((LARGE_INTEGER *) & ticksPerSecond))
  {
    ticksToCount = (long) ((ticksPerSecond / 1E6) * delayTime) + 1;
    (void) QueryPerformanceCounter((LARGE_INTEGER *) & nowTicks);
    STArtTicks = nowTicks;
    while (nowTicks - STArtTicks < (DWORD) ticksToCount)
      (void) QueryPerformanceCounter((LARGE_INTEGER *) & nowTicks);
  }
  else
#endif          /* _CVI_ */

  {
    ticksToCount = (long) ((1E3 / 1E6) * delayTime) + 1;
    msTicks = GetTickCount();
    while (GetTickCount() - msTicks < (DWORD) ticksToCount)
    {;
    }
  }

#endif          /* WIN32 */

  //-----------------------------------------------------------
  // HP-UX Version
  // HP-UX counts in 1 micro-second increments.
  // The call overhead to execute doDelay on HP-UX is about
  // 15-20 micro-seconds.
  //-----------------------------------------------------------

#ifdef __hpux

  struct timeval t0, t1;
  if (delayTime == 0)
    return;

  gettimeofday(&t0, NULL);
  t0.tv_usec += delayTime;
  t0.tv_sec += t0.tv_usec / 1000000;
  t0.tv_usec = t0.tv_usec % 1000000;

  gettimeofday(&t1, NULL);

  while (t1.tv_sec < t0.tv_sec)
    gettimeofday(&t1, NULL);

  if (t1.tv_sec > t0.tv_sec)
    return;                     // get out quick if past delay time

  while (t1.tv_usec < t0.tv_usec)
    gettimeofday(&t1, NULL);

#endif          /* __hpux */

  return;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS (same as viVScanf)
//  ViSession   instrumentHandle (in)
//  ViString    scanFmt (in)
//  ...
//*****************************************************************************
// Same as viVScanf, but with a delay added if necessary.
//*****************************************************************************
static ViStatus delayVScanf(
    ViSession instrumentHandle,
    ViString scanFmt,
  ...)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "delayVScanf"

  ViStatus errStatus;
  va_list va;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // error if we're in the middle of a transaction...
  if(thisPtr->pTransBuffer)
    return HPE8311A_INSTR_ERROR_TRANSACTION_IN_PROG;

  va_start(va, scanFmt);

  errStatus = viVScanf(
    instrumentHandle,
    scanFmt,
    va);

  va_end(va);

  if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

  return errStatus;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS (same as viVPrintf)
//  ViSession   instrumentHandle (in)
//  ViString    printFmt (in)
//  ...
//*****************************************************************************
// Same as viVPrintf, but with a delay added if necessary.
//*****************************************************************************
static ViStatus delayVPrintf(
    ViSession instrumentHandle,
    ViString printFmt,
  ...)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "delayVPrintf"

  ViStatus errStatus;
  va_list va;
  ViString pFormatStr;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  va_start(va, printFmt);

  if(thisPtr->pTransBuffer)
  {
    // is this the middle of a transaction?
    if(NULL != printFmt)
    {
      // put result into a temp buffer
      ViChar buff[256] = {0};
      ViUInt32 sendBytesCount;

      (void) vsprintf(buff, printFmt, va);
      sendBytesCount = strlen(buff);

      // check buffer overflow....
      if((thisPtr->TransBufferLength + sendBytesCount + 4) > XACT_BUFF_SIZE)
      {
        transactionCancel(thisPtr);
        return HPE8311A_INSTR_ERROR_TRANSACTION_BUFFER_OVERFLOW;
      }

      // copy new command and delimiter to transaction buffer
      (void) strcpy(thisPtr->pTransBuffer + thisPtr->TransBufferLength, buff);
      thisPtr->TransBufferLength += sendBytesCount;

      // delimit every command...if there's a \n at the end then overwrite it
      if(thisPtr->pTransBuffer[thisPtr->TransBufferLength - 1] == 0x0A)
        --thisPtr->TransBufferLength;

      thisPtr->pTransBuffer[thisPtr->TransBufferLength++] = ';';

      // done for now
      va_end(va);
      return VI_SUCCESS;
    }

    // end of transaction...overwrite the last ';' with a '\n'
    thisPtr->pTransBuffer[thisPtr->TransBufferLength - 1] = 0x0A;

    // and send it
    pFormatStr = thisPtr->pTransBuffer;
  }
  else
  {
    pFormatStr = printFmt;
  }

  errStatus = viVPrintf(
    instrumentHandle,
    pFormatStr,
    va
    );

  va_end(va);

  if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

  return errStatus;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS (same as viVQueryf)
//   ViSession   instrumentHandle (in)
//   ViString    printFmt (in)
//   ViString    scanFmt (in)
//   ...
//*****************************************************************************
// Same as viVQueryf, but with a delay added if necessary.
//*****************************************************************************
static ViStatus delayVQueryf(
    ViSession instrumentHandle,
    ViString printFmt,
    ViString scanFmt,
  ...)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "delayVQueryf"

  ViStatus errStatus;
  va_list va;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // error if we're in the middle of a transaction...
  if(thisPtr->pTransBuffer)
    return HPE8311A_INSTR_ERROR_TRANSACTION_IN_PROG;

  va_start(va, scanFmt);

  errStatus = viVQueryf(
    instrumentHandle,
    printFmt,
    scanFmt,
    va);

  va_end(va);

  if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

  return errStatus;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS (same as viRead)
//   ViSession   instrumentHandle (in)
//   ViPBuf      buf (in/out)
//   ViUInt32    cnt (in)
//   ViPUInt32   retCnt (in)
//*****************************************************************************
// Same as viRead, but with a delay added if necessary.
//*****************************************************************************
static ViStatus delayRead(
    ViSession instrumentHandle,
    ViPBuf buf,
    ViUInt32 cnt,
    ViPUInt32 retCnt)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "delayRead"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  errStatus = viRead(
    instrumentHandle,
    buf,
    cnt,
    retCnt
    );

  if (thisPtr->dTime > 0)
    doDelay(thisPtr->dTime);

  return errStatus;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession              instrumentHandle (in)
//   struct hpe8311a_globals *thisPtr (in)
//     A pointer to the VISA globals for this VISA session
//   ViStatus               CallersError (in)
//     A driver error return code.
//*****************************************************************************
static ViStatus statusUpdate(
    ViSession instrumentHandle,
    struct hpe8311a_globals * thisPtr,
    ViStatus CallersError)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "statusUpdate"

  ViStatus errStatus;
  ViInt32 eventQ;

  // just return CallersError if we're in the middle of a transaction...
  
  if(thisPtr->pTransBuffer)
    return CallersError;

  // We will ONLY do the instrument error check if we received a 
  // VI_SUCCESS on the last call.  
  // This is fundamental to the way in which this function works
  // ...it must ONLY be called by routines which use the static 
  //    (low-level) access functions.  Those functions return errors when
  //    they cannot communicate with the instrument...not when the instrument
  //    detects an error in the passed SCPI command.
  
  if (CallersError < VI_SUCCESS)
    ERR_RETURN (CallersError);
    
  if (!thisPtr->errQueryDetect)
    return VI_SUCCESS;
    
  TRACE_IN();

  // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // If error query detect is set on and the
  // instrument has not timed out, then ...
  // Read the Status byte into eventQ and check
  // the error bytes.
  // Potential Status byte errors are
  // 0x04 (Query Error),
  // 0x08 (Device Dependent Error),
  // 0x10 (Execution Error),
  // and 0x20 (Command Error).
  // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "*ESR?",                // SCPI
      readString,
      HPE8311A_STRING_LENGTH - 1, // reserve for terminator
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    readString[validLength] = '\0'; // terminate
    eventQ = atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(instrumentHandle,
      "*ESR?\n",
      "%hd%*t",
      &eventQ
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);
      
    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  if ((0x04 | 0x08 | 0x10 | 0x20) & eventQ)
    ERR_RETURN (HPE8311A_INSTR_ERROR_DETECTED);
    
  return VI_SUCCESS;
}
//******************************************************************************
//******************************************************************************
// PARAMETERS
//   struct hpe8311a_globals *thisPtr (in)
//     A pointer to the VISA globals for this VISA session
//   ViBoolean              chk_val (in)
//     Check this value to make sure it is a legal Boolean.
//******************************************************************************
static ViBoolean chk_boolean(
    struct hpe8311a_globals * thisPtr,
    ViBoolean chk_val
)
//******************************************************************************
{
  if ((chk_val != VI_TRUE) && (chk_val != VI_FALSE))
  {
    if (thisPtr)
      sprintf(thisPtr->errMessage, HPE8311A_MSG_BOOLEAN, chk_val);
    return VI_TRUE;
  }

  return VI_FALSE;
}
//*****************************************************************************
// MACRO CHK_LONG_RANGE
//*****************************************************************************
// Ref chk_long_range for info.
//*****************************************************************************

#define CHK_LONG_RANGE(chk_val, min, max, err) \
if (chk_long_range(thisPtr, (ViInt32)(chk_val), (ViInt32)(min), (ViInt32)(max))) \
{ \
  ERR_RETURN ((err)); \
}

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   struct hpe8311a_globals *thisPtr (in)
//     A pointer to the VISA globals for this VISA session
//   ViInt32              chk_val (in)
//     The value to be checked.
//   ViInt32              min (in)
//     The bottom of the range.
//   ViInt32              max (in)
//     The top of the range.
//*****************************************************************************
// Tests to see if a ViInt32 is in range.
//*****************************************************************************
static ViBoolean chk_long_range(
    struct hpe8311a_globals * thisPtr,
    ViInt32 chk_val,
    ViInt32 min,
    ViInt32 max
)
//*****************************************************************************
{
  if ((chk_val < min) || (chk_val > max))
  {
    if (thisPtr)
      sprintf(thisPtr->errMessage, HPE8311A_MSG_LONG, min, max, chk_val);
    return VI_TRUE;
  }

  return VI_FALSE;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession      vi (in)
//     Instrument Handle returned by viOpen.
//   ViSession      defRM (in)
//     Resource Manager handle from viOpen.
//   struct hpe8311a_globals*  heapData
//     Global data, malloced by init()
//   ViStatus       errStatus (in)
//     A driver error return code.
//*****************************************************************************
// This function is only called from hpe8311a_init.
//*****************************************************************************
static ViStatus _VI_FUNC initError(
    ViPSession pInstrumentHandle,
    ViSession defRM,
    struct hpe8311a_globals * thisPtr,
    ViStatus errStatus
)
//*****************************************************************************
{
  (void) viClose(*pInstrumentHandle);
  (void) viClose(defRM);
  *pInstrumentHandle = VI_NULL;
  FREE(thisPtr);
  return (errStatus);
}

#define INIT_ERR_RETURN(_err) \
  return(initError(pInstrumentHandle, defRM, thisPtr, (_err)))

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// IO-FUNCTIONS FOR MESSAGE-BASED-MODULES (mb)
//   (use them only driver internal)
//-----------------------------------------------------------------------------
// Originally these modules are register-based, but with changes of the
//   special-bios they are message-based now. It is possible to force these
//   modules back into register-based-modus, and then to use the
//   register-based ability (to program a new bios).
//-----------------------------------------------------------------------------
// MB-LOW-LEVEL-FUNCTIONS:
//   They are created for low-level access to the interface of the modules.
//   Remember, that these functions do their actions without taking care
//   about the ready-state of the module. So, use them carefuly and with the
//   permanent knowledge of the current module-state.
//   - to write to a Data-In-Register use
//     mbWriteDIR();
//   - to read from a Data-Out-Register use
//     mbReadDOR();
//   - to write to the Input-Buffer use
//     mbWriteInBuf();
//   - to read from a Output-Buffer use
//     mbReadOutBuf();
//-----------------------------------------------------------------------------
// MB-HIGH-LEVEL-FUNCTIONS:
//   They are created to perform input and output to the module, following its
//   handshake-conventions. With this functions no more thought to the
//   communication flowchart has to be done. The abilities of the
//   mb-low-level-functions are used inside of this functions.
//   - to send a SCPI-command to the module
//     and to write DATA to the module use
//     mbSendSCPIandDATA();
//   - to send a SCPI-command to the module
//     and to read DATA (commands answer) from the module use
//     mbSendSCPIreadDATA();
//   Remember, that the DATA are in some RAW-FORM. How to interpretate them
//   (whether int, float, string, array) is the work of the user-function.
//-----------------------------------------------------------------------------
// The jump-STArt-sceleton has got the command-functions like hpe8311a_cmd().
//   It is recommended that only these functions use the io-functions for
//   message-based-modules (mb), because they already do all the work of
//   interpretating the DATA-format.
//-----------------------------------------------------------------------------
// All other functions, that means the jump-STArt-functions and the
//   instrument-specific-functions use the command-functions like hpe8311a_cmd()
//   of the jump-STArt-sceleton. With this, nearly no more consideration has
//   to be done about the DATA-format sent to and received from the instrument.
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//128kB    A24-ADRESS-SPACE of every module
//---------------------------------------------------------------------------
//0x1FFFF  BLOCK-MEMORY used for block-data input into instrument
//0x18800  30kB
#define    BASE_BLOCK_INPUT_BUFFER  0x18800UL
//---------------------------------------------------------------------------
//0x187FF  BLOCK-MEMORY used for block-data output out of instrument
//0x11000  30kB (possible 60kB but dangerous!)
#define    BASE_BLOCK_OUTPUT_BUFFER 0x11000UL
//---------------------------------------------------------------------------
//0x1FFFF  BLOCK-MEMORY used for block-transfer
//0x11000  60kB
#define    BASE_BLOCK_MEMORY_REG    0x11000UL
//---------------------------------------------------------------------------
//0x10BFF  DATA-OUT-REGISTER
//0x10800  256*4B = 1kB = 256 Registers
#define    BASE_DATA_OUT_REG        0x10800UL
//---------------------------------------------------------------------------
//0x107FF  DATA-IN-REGISTER
//0x10000  512*4B = 2kB = 512 read/write-Registers
#define    BASE_DATA_IN_REG         0x10000UL
//---------------------------------------------------------------------------
//0x0FFFF  MODULE-HARDWARE-REGISTER
//0x00000  64kB (Do not use!!!)
// #define    BASE_MODULE_HARDWARE_REG 0x00000
//*****************************************************************************

//*****************************************************************************
// Data-Out-Register --> for module-ready-state:
//   valid values are:
//   - DOV_READY (0)
//   - DOV_BUSY (3)
#define DOR_STATUS 256
//---------------------------------------------------------------------------
// Data-Out-Register --> for number of valid bytes in output-buffer:
#define DOR_BLK_BYTES_VALID 265
//---------------------------------------------------------------------------
// Data-In-Register --> for number of valid bytes in input-buffer:
#define DIR_BLK_BYTES_VALID 300
//*****************************************************************************

//*****************************************************************************
// Values for the STAte Register (DOR_STATE):
// Note: only values < 64k are allowed because this value must be polled
// and there is now way to poll 32bit at once!
// *** DO NOT CHANGE THE VALUE OF ANY OF THESE ENUMS (used in VB) ***
//---------------------------------------------------------------------------
enum DOV_STAte
{
  DOV_READY,                    // action successfully completed
  DOV_READY_AFTER_REQUEST,      // requested simpl/complex value valid
  DOV_READY_FOR_NEXT_BLK,       // partial block transfer done (i/o)
  DOV_BUSY,                     // action still in execution
  DOV_ERROR,                    // error occurred during execution
  DOV_READY_WITHIN_TRANSACTION, // ready for next parameter
  // insert new values above this line
  DOV_STATE_FORCE_INT = 0x10000
};
//*****************************************************************************

//*****************************************************************************
// registers in A16 address space, these numbers are register numbers,
// not offsets like in vxi.h, so don't use the wrong values !
//---------------------------------------------------------------------------
#define A16_ID_REG			        0 /* read */
#define A16_DEV_TYPE_REG	        	1 /* read */
#define A16_STATUS_REG		         	2 /* read */

//*****************************************************************************
// special Bits of these A16-adress-space-registers
//---------------------------------------------------------------------------
#define DIR_BIT			        0x40 /* DIR bit in A16 Status */

//*****************************************************************************

/*lint -e749 local enumeration constant not referenced... */

enum
{
  // STAte Register --------------------------------------------------------
  DOR_STATE = 256,              // must be polled
  // STAte of Measurement --------------------------------------------------
  DOR_FRONT_INTR = 0,           // can generate interrupts on workSTAtion
  // used to signal that a frontend
  // generated an interrupt
  // Value: bit
  // 0    Interrupt on ch. 1
  // 1    Interrupt on ch. 2
  // Note:  Only used by HP81110A and COMBO

  DOR_MEAS_STATE = 1,           // can generate interrupts on workSTAtion
  // used to signal changes of the measument
  // STAte
  // Note:  Not used by HP81110A

  DOR_DOR_PROC_INIT = 2,        // can generate interrupts on workSTAtion
  // used to signal that the
  // data-out-interrupt process is now
  // active
  // Value: Don't care (HPE8311A_ALWAYS 0)
  // Note:  Only used by HP81110A and COMBO

  DOR_SEQUENCER_INTR = 3,       // can generate interrupts on workSTAtion
  // used to signal sequencer interrupts
  // Value: Don't care (HPE8311A_ALWAYS 0)
  // Note:  Only uesd by COMBO

  // Block Memory ----------------------------------------------------------
  DOR_BLK_BYTES_AVAIL = 256 + 8,// block Memory length (60k today) poll

  // Result of a simple Request Register -----------------------------------
  DOR_SIMPLE_REQUEST = 256 + 16,// must be polled

  // first module specific data-out register -------------------------------
  DOR_FIRST_MODULE = 300,
  DOR_FORCE_INT = 0x10000
};
//*****************************************************************************

//*****************************************************************************
// Data In Registers DIR_, associated Values DIV_
//*****************************************************************************
enum
{
  // Configuration of board (Termination, Feedback, ...) -------------------
  DIR_VXI_TERM = 0,
  DIR_VXI_HPE8311A_ECL_TRIGGER,
  DIR_VXI_HPE8311A_TTL_FEEDBACK,

  // Configuration of the DOR access process -------------------------------
  DIR_DOR_PROC_STATE = 8,

  // Configuration of Channels ---------------------------------------------

  // Request Errors --------------------------------------------------------
  DIR_REQ_ERRORS = 16,

  // Command Registers -----------------------------------------------------
  DIR_USR_COMMAND = 32,         // only user commands
  DIR_EXP_COMMAND,              // user and expert commands
  DIR_FAB_COMMAND,              // user, expert, factory commands
  DIR_LAB_COMMAND,              // all commands

  // Block Transfers RAW ---------------------------------------------------
  DIR_BTF_BIOS_PROM = 77,       // w for program prom

  // Execution Mode Register -----------------------------------------------
  DIR_EXEC_MODE = 120,

  // Pin specific SI Unit Registers: ---------------------------------------
  // module specific or general???
  DIR_FIRST_VOLTAGE = 128,      // use only in module specific h-file
  DIR_FIRST_CURRENT = 160,      // use only in module specific h-file
  DIR_FIRST_RESISTOR = 192,     // use only in module specific h-file
  DIR_FIRST_TIME = 224,         // use only in module specific h-file

  // Pin specific other Registers: -----------------------------------------
  // module specific or general???
  DIR_FIRST_OTHER = 256,        // use only in module specific h-file

  // Complex Parameters (may be Pin dependant): ----------------------------
  DIR_SET_COMPLEX = 384,
  DIR_SET_COMPLEX_RUNNING,

  // Request Registers (may be Pin dependant): -----------------------------
  DIR_REQ_SIMPLE = 400,
  DIR_REQ_SIMPLE_RUNNING,
  DIR_REQ_COMPLEX,
  DIR_REQ_COMPLEX_RUNNING,

DIR_FORCE_INT = 0x10000};
//*****************************************************************************

//*****************************************************************************
// Values for the command registers (DIR_xxx_COMMAND):
//*****************************************************************************
enum
{
  // General user commands: ------------------------------------------------
  DIV_UC_LOW_GENERAL = 0x00000,
  DIV_UC_PROGRAM_MODE,
  DIV_UC_ARMED_MODE,
  DIV_UC_RUN_MODE,

  DIV_UC_READY,                 // just used to enforce a ready STAte of the
  // module

  DIV_UC_RST0,

  DIV_UC_ARM_WITHOUT_RESET,     // do not reset the error rate counters

  DIV_UC_CONNECT,
  DIV_UC_DISCONNECT,

  DIV_UC_GET_SELFTEST_ERRORS,
  DIV_UC_GET_PWR_UP_ERRORS,

  // Module specific user commands -----------------------------------------
  DIV_UC_LOW_MODULE = 0x10000,

  // General Expert Commands: ----------------------------------------------
  DIV_EC_LOW_GENERAL = 0x20000,
  DIV_EC_ENABLE,
  DIV_EC_DISABLE,
  DIV_EC_UR_BIOS,
  DIV_EC_ERASE_BIOS_PROM,
  DIV_EC_BOOT,

  // Module specific expert commands ---------------------------------------
  DIV_EC_LOW_MODULE = 0x30000,

  // General Factory Commands: ---------------------------------------------
  DIV_FC_LOW_GENERAL = 0x40000,
  DIV_FC_ENABLE,
  DIV_FC_DISABLE,
  DIV_FC_BURN_CAL_DATA,
  DIV_FC_BURN_EEPROM,

  // Module specific expert commands ---------------------------------------
  DIV_FC_LOW_MODULE = 0x50000,

  // General Lab Commands: -------------------------------------------------
  DIV_LC_LOW_GENERAL = 0x60000,
  DIV_LC_ENABLE,
  DIV_LC_DISABLE,
  DIV_LC_LABIO_ENABLE,
  DIV_LC_LABIO_DISABLE,

  // Module specific Lab commands ------------------------------------------
  DIV_LC_LOW_MODULE = 0x70000,

  // End Marker ------------------------------------------------------------
  DIV_MAX_CMD_NO = 0x80000
};
//*****************************************************************************

//*****************************************************************************
// Control Structures in Block Memory for Block Transfers
// BIV   = Blockmemory In Values
// BOV   = Blockmemory Out Values
// BV    = Blockmemory (In and Out) Values
//---------------------------------------------------------------------------
// RAW Block Memory Transfers
//enum BIV_RawFormats
//{
//   BIV_RAW_BIT2,
//   BIV_RAW_BYTE,
//   BIV_RAW_WORD,
//   BIV_RAW_WORD4,
//   BIV_RAW_ISP_STREAM,
//   BIV_RAW_FPGA_STREAM,
//
//   BIV_RAW_FORCE_INT = 0x10000
//};
//---------------------------------------------------------------------------
//lint -e754 local structure member not referenced
struct BIV_RawIOControl
{
  ViUInt32 eFormat;	  // BIV_RawFormats
  ViUInt32 uParam;    // Memory Bank, device number, ...
  ViUInt32 uLength;
  ViUInt32 uOffset;
  ViUInt32 uReserved[4];
};
//*****************************************************************************

//*****************************************************************************
// Values for the Block Transfer Registers (DIR_BTF_x):
//---------------------------------------------------------------------------
enum DIV_BTF_Dir
{
  DIV_BTF_READ,
  DIV_BTF_WRITE,
  DIV_BTF_DONE,
  DIV_BTF_WRITE_AND_DONE,
DIV_BTF_FORCE_INT = 0x10000};
//*****************************************************************************

//*****************************************************************************
// A shared buffer to prepare the contents of the blockmemory before a
// transfer to a module.
//---------------------------------------------------------------------------
#define MAX_BLK_LEN ( 60*1024 )
//*****************************************************************************

//*****************************************************************************
static void _VI_FUNC mbFloatToString(
    ViReal64 num,
    ViChar val[]
)
//*****************************************************************************
{

#ifndef _CVI_

  // val must be at least 20 chars long
  int dec, sign;
  char *x = ecvt(num, 12, &dec, &sign);
  char lc[256];
  if (sign)
    strcpy(val, "-.");
  else
    strcpy(val, ".");

  strcat(val, x);
  strcat(val, "E");
  strcat(val, itoa(dec, lc, 10));

#else

    sprintf(val, "%lG", num);

#endif          /* _CVI_ */

  return;
}
//-----------------------------------------------------------------------------
//Funktion..: Exchange (swap) Databits - depending on the targetSystem
//-----------------------------------------------------------------------------
//Parameter.: wordArray - value, whose bytes need to be swap (or not)
//            byteQuantity - size of the array
//-----------------------------------------------------------------------------
//Return....: swaped (or not swaped) value
//-----------------------------------------------------------------------------
static ViStatus _VI_FUNC BlockDataSwapBytes(
    ViUInt16 * wordArray,
    ViUInt32 byteQuantity
)
//-----------------------------------------------------------------------------
{

#ifdef _WIN32

  // working on Windows means working on an Intel-Processor
  // Intel stores data on an other way than Motorola
  // (see bigEndian and liHPE8311A_TTLeEndian)
  // therefore lowerWord and higher Word need to be exchanged

  ViUInt8 *helpP = (ViUInt8 *) (wordArray);
  ViUInt8 temp;
  ViUInt32 count;
  if (byteQuantity % 2 == 1)
  {
    // ... this means it is an odd value...
    // ...that is really dangerous!!
    // It is not garanteed, that the byte after the last vaid byte
    // is allowed to be overwritten!!
    return (HPE8311A_INSTR_ERROR_CANT_SWAP_ODD_BYTE_QUANTITY);
  }

  for (
    count = 0;
    count < (byteQuantity / 2);
    count++
    )
  {
    temp = *helpP;
    *helpP = *(helpP + 1);
    *(helpP + 1) = temp;
    helpP++;
    helpP++;
  }

  return (VI_SUCCESS);

# elif _HPUX

# else
#   error "Wrong Target"
#endif
}
//-----------------------------------------------------------------------------
//Funktion..: Exchange (swap) Databits - depending on the targetSystem
//-----------------------------------------------------------------------------
//Parameter.: doubleWord - value, whose bytes need to be swap (or not)
//            quantity   - size of the array
//-----------------------------------------------------------------------------
//Return....: swaped (or not swaped) value
//-----------------------------------------------------------------------------
static void BlockDataSwapWords(
    ViUInt32 * pWordArray,
    ViUInt32 quantity
)
//-----------------------------------------------------------------------------
{

#ifdef _WIN32

  // assume Windows always on a little-endian processor

  while (quantity--)
  {
    *pWordArray = (*pWordArray << 16) | (*pWordArray >> 16);
    pWordArray++;
  }

# elif _HPUX
  // do nothing

# else
#   error "Wrong Target"
#endif
}
//*****************************************************************************
// MB-LOW-LEVEL-FUNCTION:
//     to write to a Data-In-Register use
//     mbWriteDIR();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | dataInRegister                                       ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide register-number to write value to
//  |  |   Allowed register-numbers are:
//  |  |   - DIR_BLK_BYTES_VALID (300)
//   ------------------------------------------------------------------------
//  | value                                                ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to write into the specified register.
//  |  |   This value means the number of valid bytes, which have
//  |  |   been written to the input-buffer before. The instrument
//  |  |   is asked to read these bytes.
//*****************************************************************************
static ViStatus _VI_FUNC mbWriteDIR(
    ViSession instrumentHandle,
    ViUInt32 dataInRegister,
    ViUInt32 value
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbWriteDIR"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  CHK_LONG_RANGE(dataInRegister, 300L, 300L, VI_ERROR_PARAMETER2);
  CHK_LONG_RANGE(value, 0L, 30 * 1024L, VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  // Write doubleWord to doubleWord-Register:
  // A doubleWord must be separated into two Words
  // These two words must be transeved with 16bit operations each
  // loverWord (bit15-00) <---> higher address
  // upperWord (bit31-16) <---> lower address

  //-------------------------------------------------------------------------
  // Write upperWord to doubleWord-Register
  // must be written first - or an unallowed Interrupt is caused

  errStatus = viOut16(
    instrumentHandle,
    VI_A24_SPACE,
    (dataInRegister * 4 + BASE_DATA_IN_REG),
    (ViUInt16) (value >> 16)
    );

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // write lowerWord to doubleWord-Register
  // must be written last - to cause Interrupt

  return viOut16(
    instrumentHandle,
    VI_A24_SPACE,
    (dataInRegister * 4 + BASE_DATA_IN_REG + 2),
    (ViUInt16) (value)
    );

}
//*****************************************************************************
// MB-LOW-LEVEL-FUNCTION:
//     to read from a Data-Out-Register use
//     mbReadDOR();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | dataOutRegister                                      ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide register-number to read value from.
//  |  |   Allowed register-numbers are:
//  |  |   - DOR_STATUS (256)
//  |  |   - DOR_BLK_BYTES_VALID (265)
//  |  |   - ??? (?)
//   ------------------------------------------------------------------------
//  | value                                                ViPUInt32      OUT
//  |   ---------------------------------------------------------------------
//  |  | This is the value, the function has read out of the
//  |  |   specified data-out-register.
//  |  | Depending on the provided data-out-register-number it has
//  |  |   different valid meanings:
//  |  |   - dataOutRegister...: DOR_STATUS (256):
//  |  |     meaning...........: module ready STAte
//  |  |     range.............: DOV_READY (0) or DOV_BUSY (3)
//  |  |   - dataOutRegister...: DOR_BLK_BYTES_VALID (265)
//  |  |     meaning...........: Number of valid bytes in output-buffer
//  |  |     range.............: max.: size of output-buffer
//  |  |   - dataOutRegister...: ???
//  |  |     meaning...........: The language parser Status byte
//  |  |     range.............: ???
//*****************************************************************************
static ViStatus _VI_FUNC mbReadDOR(
    ViSession instrumentHandle,
    ViUInt32 dataOutRegister,
    ViPUInt32 value
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbReadDOR"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViUInt16 readWord = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // check range of input-parameter
  switch (dataOutRegister)
  {
    //-----------------------------------------------------------------------
  case DOR_STATUS:              // DOR_STATUS (256)
  case DOR_BLK_BYTES_VALID:     // DOR_BLK_BYTES_VALID (265)
    // case ???:              //The language parser Status byte

    // all is o.k. -- nothing specific to do

    break;
    //-----------------------------------------------------------------------
  default:                      // parameter dataOutRegister is out of
    // allowed range

    return VI_ERROR_PARAMETER2;
    //-----------------------------------------------------------------------
  }

  //-------------------------------------------------------------------------
  // Read doubleWord from doubleWord-Register:
  // A doubleWord must be separated into two Words
  // These two words must be transeved with 16bit operations each
  // loverWord (bit15-00) <---> higher address
  // upperWord (bit31-16) <---> lower address

  //-------------------------------------------------------------------------
  // Read upperWord from doubleWord-Register
  errStatus = viIn16(
    instrumentHandle,
    VI_A24_SPACE,
    (dataOutRegister * 4 + BASE_DATA_OUT_REG),
    &readWord
    );

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  *value = ((ViUInt32) (readWord) << 16);

  //-------------------------------------------------------------------------
  // Read loverWord from doubleWord-Register

  errStatus = viIn16(
    instrumentHandle,
    VI_A24_SPACE,
    (dataOutRegister * 4 + BASE_DATA_OUT_REG + 2),
    &readWord
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  *value = (*value | (ViUInt32) (readWord));

  //-------------------------------------------------------------------------
  // check the value, which was read out of the instrument
  switch (dataOutRegister)
  {

  case DOR_STATUS:              // value must be either DOV_READY or DOV_BUSY
    if ((*value != DOV_READY) && (*value != DOV_BUSY))
    {
      // reading a wrong STAte from the module (very suspicious)
      return HPE8311A_INSTR_ERROR_WRONG_STATE;
    }
    break;

  case DOR_BLK_BYTES_VALID:     // output buffer has max size of 60K
    if (*value > (60 * 1024))
    {
      // read a value greater than buffer size (also very suspicious)
      return HPE8311A_INSTR_ERROR_DOR_GREATER_THAN_OUTPUT_BUFFER;
    }
    break;

  default:
    break;

  }

  return errStatus;
}
//*****************************************************************************
// MB-LOW-LEVEL-FUNCTION:
//     to write to the Input-Buffer use
//     mbWriteInBuf();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | writeBuffer                                          ViString        IN
//  |   ---------------------------------------------------------------------
//  |  | Pointer to a dataBlock.
//  |  | This dataBlock may contain integers or chars.
//   ------------------------------------------------------------------------
//  | bytesToWrite                                         ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Information about the quantity of valid
//  |  | bytes in the writeBuffer.
//*****************************************************************************
static ViStatus _VI_FUNC mbWriteInBuf(
    ViSession instrumentHandle,
    ViString writeBuffer,
    ViUInt32 bytesToWrite
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbWriteInBuf"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  CHK_LONG_RANGE(bytesToWrite, 0L, (60 * 1024L) / 2, VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  // Think about byte-order.
  errStatus = BlockDataSwapBytes(
    (ViUInt16 *) writeBuffer,
    bytesToWrite
    );

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Get the data to the instrument.

  return viMoveOut16(
    instrumentHandle,
    VI_A24_SPACE,
    BASE_BLOCK_INPUT_BUFFER,
    (ViBusSize) ((bytesToWrite + 1) / 2),
    (ViAUInt16) (writeBuffer)
    );

}
//*****************************************************************************
// MB-LOW-LEVEL-FUNCTION:
//     to read from a Output-Buffer use
//     mbReadOutBuf();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | readBuffer                                                          OUT
//  |   ---------------------------------------------------------------------
//  |  | Pointer to a dataBlock.
//  |  |
//   ------------------------------------------------------------------------
//  | bytesToRead                                                          IN
//  |   ---------------------------------------------------------------------
//  |  | Information how many words to read out of Instruments Buffer
//  |  |
//*****************************************************************************
static ViStatus _VI_FUNC mbReadOutBuf(
    ViSession instrumentHandle,
    ViPString readBuffer,
    ViUInt32 bytesToRead
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbReadOutBuf"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  CHK_LONG_RANGE(bytesToRead, 0L, 60 * 1024L, VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  // Get the data from the instrument.

  errStatus = viMoveIn16(
    instrumentHandle,
    VI_A24_SPACE,
    BASE_BLOCK_OUTPUT_BUFFER,
    (ViBusSize) ((bytesToRead + 1) / 2),
    (ViAUInt16) (readBuffer)
    );

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  return BlockDataSwapBytes(
    (ViUInt16 *) readBuffer,
    bytesToRead
    );

}
//*****************************************************************************
// MB-LOW-LEVEL-FUNCTION
//     to perform handshake between the driver and the instrument.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//*****************************************************************************
static ViStatus _VI_FUNC mbPerformHandshake(
    ViSession instrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbPerformHandshake"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;  // for error-handling
  ViUInt32 moduleReadyState;    // ready-sate to read from the instrument
  ViUInt16 A16StatusReg;        // Value of the A16 status register

  // Variables used for the handshake
  ViInt32 timeOut;              // Driver Variable set by user.
  ViReal64 deltaTime;           // Difference between start-time and
  // actual-time
  TIME_T startTime;             // Start-time of handshake
  TIME_T actualTime;            // Actual-time

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //---------------------------------------------------------------------------
  // Get the actual used value of timeOut.

  errStatus = hpe8311a_timeOut_Q(
    instrumentHandle,
    &timeOut
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // The instrument uses a two level handshake. That is there are 2 registers
  // where a busy is signalled to the world outside.
  // Register #1 is the A16 Status-Register that contains a READY-bit.
  // Register #2 is the A24 Status-Register that shows the current state of
  //             the instruments software.
  //
  // So first the A16 has to say the interface is ready, next the software
  // has to be ready.

  //---------------------------------------------------------------------------
  // Check the A16 status register
  startTime = GET_TIME();

  do
  {
    errStatus = viIn16(
      instrumentHandle,
      VI_A16_SPACE, 
      A16_STATUS_REG_OFFS, 
      &A16StatusReg
      );

    if (errStatus < VI_SUCCESS)
      return errStatus;

    // break out of loop if ready
    if ((A16StatusReg & A16_READY_BIT) == A16_READY_BIT)
      break;

    // this doesn't happen very often
    DBG_MSG("A16 not ready");

    // Do not stay longer than the allowed timeout in the loop.
    actualTime = GET_TIME();
    deltaTime = DIFF_TIME(actualTime, startTime);

    // wait here for a bit
    while(actualTime == GET_TIME()) {;}
    
  } while ( (ViInt32) deltaTime <= timeOut );

  //----------------------------------------------------------------------------
  // Check if we timed out
  
  if ( (A16StatusReg & A16_READY_BIT) != A16_READY_BIT)
  {
    DBG_MSG("A16 Timeout Occurred");
    errStatus = HPE8311A_INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT; 
  }
  else
  {
    //----------------------------------------------------------------------------
    // Now the A24 status register

    startTime = GET_TIME();

    do
    {
      errStatus = mbReadDOR(
        instrumentHandle,
        DOR_STATUS,
        &moduleReadyState
        );

      if (errStatus < VI_SUCCESS)
        return errStatus;

      // get out if we're done!
      if (moduleReadyState == DOV_READY)
        return VI_SUCCESS;

      // This happens a LOT on fast systems!
      DBG_MSG("A24 not ready");
    
      // Do not stay longer than the allowed timeout in the loop.
      actualTime = GET_TIME();
      deltaTime = DIFF_TIME(actualTime, startTime);

      // wait here for a bit
      while(actualTime == GET_TIME()) {;}

    } while ( (ViInt32) deltaTime <= timeOut );

    DBG_MSG("A24 Timeout Occurred");
    errStatus = HPE8311A_INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT;
  }

  return (errStatus);
}
//*****************************************************************************
// MB-HIGH-LEVEL-FUNCTION
//     to send a SCPI-command to the module
//     and to write DATA to the module use
//     mbSendSCPIandDATA();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | scpiCommandAndData                                   ViPString       IN
//  |   ---------------------------------------------------------------------
//  |  | Scpi-command and the appended DATA to send
//  |  | to the instrument in form of a string.
//  |  | NOTE; This ptr sent as NULL indicates the completion of a transaction
//   ------------------------------------------------------------------------
//  | sendBytesCount                                       ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Quantity of bytes, provided in the
//  |  | scpiCommandAndData string. This includes the
//  |  | bytes of the scpi-command an the DATA
//  |  | NOTE; This must be 0 at the completion of a transaction!
//*****************************************************************************
static ViStatus _VI_FUNC mbSendSCPIandDATA(
    ViSession instrumentHandle,
    ViPString scpiCommandAndData,
    ViUInt32 sendBytesCount
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbSendSCPIandDATA"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;  // for error-handling
  ViUInt32 sendSize;            // valid-bytes to put to instruments
  // input-memory
  ViUInt32 helpSendSize;        // help-value for calling mbWriteInBuf()
  ViChar *sendStrPtr;           // pointer on heap for send-data
  ViChar *helpSendStrPtr;       // help-pointer for send-data
  ViChar *helpScpiCommand;      // help-pointer for source-data
  ViUInt32 counter;             // counts up to sendBytesCount

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // 8/99; SCR; added transaction mode

  if(thisPtr->pTransBuffer)
  {
    // check buffer overflow....
    if((thisPtr->TransBufferLength + sendBytesCount + 4) > XACT_BUFF_SIZE)
    {
      transactionCancel(thisPtr);
      return HPE8311A_INSTR_ERROR_TRANSACTION_BUFFER_OVERFLOW;
    }

    // we're going to add scpiCommandAndData (transaction in progress)
    // or an end-string (transaction finish) to the buffer
    sendStrPtr = thisPtr->pTransBuffer + thisPtr->TransBufferLength;

  }
  else
  {
    if(NULL == scpiCommandAndData)
      return HPE8311A_INSTR_ERROR_NULL_PTR;

    //-------------------------------------------------------------------------
    // Get the scpiCommandAndData on the heap,
    // because there have to be added some bytes later.
    // To do this save, there has to be an own space,
    // otherwise foreign data will be destroyed.

    // Malloc the heap space.
    sendStrPtr = (ViChar *) malloc(sendBytesCount + 4);

    // Check whether malloc worked.
    if (sendStrPtr == NULL)
    {
      // malloc did not work
      return VI_ERROR_ALLOC;
    }
  }

  // Copy the data from scpiCommandAndData to the new heap space.
  // Remember: there may be byte-data, which contain 0x00,
  // so it is not save to use strcpy!
  // NOTE; the passed ptr can be null in transactions!

  helpSendStrPtr = sendStrPtr;

  if(scpiCommandAndData)
  {
    helpScpiCommand = scpiCommandAndData;

    for (counter = 0; counter < sendBytesCount; counter++)
    {
      *helpSendStrPtr++ = *helpScpiCommand++;
    }

    //-------------------------------------------------------------------------
    // if we're in a transaction keep our counter accurate and get out.
    // scpiCommandAndData will be NULL when we want to complete the transaction
    // so we don't even get here.

    if(thisPtr->pTransBuffer)
    {
      *helpSendStrPtr++ = ';';  // delimiter...don't forget to add 1 to counter
      thisPtr->TransBufferLength += (sendBytesCount + 1);
      return VI_SUCCESS;
    }
  }
  else
  {
    // complete a transaction...we're going to send the whole buffer 
    // so init the ptr and byte count
    sendStrPtr = thisPtr->pTransBuffer;
    sendBytesCount = thisPtr->TransBufferLength;
  }


  //---------------------------------------------------------------------------
  // Append "~n" and '\0' to scpi-command.
  // Remember, there has to be space provided, for this additional chars.

  *helpSendStrPtr++ = '~';
  *helpSendStrPtr++ = 'n';
  *helpSendStrPtr++ = '\0';

  sendSize = sendBytesCount + 2;

  //---------------------------------------------------------------------------
  // pad output buffer if required

  // Variable sendSize contains the correct value, do not change it.
  helpSendSize = sendSize;

  if (helpSendSize % 2 == 1)
  {
    // ...this means helpSendSize is an odd value:
    // Here, some preparation is made to make mbWriteInBuf() and with
    // that BlockDataSwapBytes() safe!
    *helpSendStrPtr++ = '\0';
    helpSendSize++;
    // now helpSendSize has an even value
  }

  //---------------------------------------------------------------------------
  // Wait until module-ready-state is ready.

  errStatus = mbPerformHandshake(instrumentHandle);

  if (errStatus < VI_SUCCESS)
  {
    if(NULL == thisPtr->pTransBuffer) 
      FREE(sendStrPtr);
    return errStatus;
  }

  //---------------------------------------------------------------------------
  // copy command into input-buffer

  errStatus = mbWriteInBuf(
    instrumentHandle,
    sendStrPtr,
    helpSendSize
    );

  // do not need the send-data any more
  // ...but NEVER free a transaction buffer in here!

  if(NULL == thisPtr->pTransBuffer) 
    FREE(sendStrPtr);             

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // write number of bytes into Data-In-Register 300

  errStatus = mbWriteDIR(
    instrumentHandle,
    DIR_BLK_BYTES_VALID,
    sendSize
    );

  return errStatus;
}
//*****************************************************************************
// MB-HIGH-LEVEL-FUNCTION
//     to send a SCPI-command to the module
//     and to read DATA (commands answer) from the module use
//     mbSendSCPIreadDATA();
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | scpiCommand                                          ViPString       IN
//  |   ---------------------------------------------------------------------
//  |  | SCPI-Command to send to the instrument.
//  |  |   No data is appended, because this is a query.
//   ------------------------------------------------------------------------
//  | readBuffer                                           ViPString      OUT
//  |   ---------------------------------------------------------------------
//  |  | Pointer to the buffer where the instruments answer
//  |  |   has to be written to. This buffer is provided by the
//  |  |   user of this routine.
//   ------------------------------------------------------------------------
//  | readBufferSize                                       ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Size of the readBuffer.
//   ------------------------------------------------------------------------
//  | readBytesCount                                       ViUInt32       OUT
//  |   ---------------------------------------------------------------------
//  |  | Counter for the number of bytes read from the
//  |  |   instrument
//*****************************************************************************
static ViStatus _VI_FUNC mbSendSCPIreadDATA(
    ViSession instrumentHandle,
    ViPString scpiCommand,
    ViPString readBuffer,
    ViUInt32 readBufferSize,
    ViPUInt32 readBytesCount
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "mbSendSCPIreadDATA"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;  // for error-handling
  ViUInt32 moduleValidBytes;    // valid-bytes in instruments output-memory
  ViUInt32 helpModuleValidBytes;// help-value for calling mbReadOutBuf()
  ViUInt32 sendSize;            // valid-bytes to put to instruments
  // input-memory
  ViUInt32 helpSendSize;        // help-value for calling mbWriteInBuf()
  ViChar *sendReadStrPtr;       // pointer on heap for send/read-data
  ViChar *helpSendReadStrPtr;   // help-pointer for send/read-data
  ViChar *helpReadBuffer;       // help-pointer to save ReadBuffer
  ViUInt32 counter;             // counts up to moduleValidBytes

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // error if we're in the middle of a transaction...
  if(thisPtr->pTransBuffer)
    return HPE8311A_INSTR_ERROR_TRANSACTION_IN_PROG;

  //---------------------------------------------------------------------------
  // Get the scpiCommand on the heap,
  // because there have to be added some bytes later.
  // To do this save, there has to be an own space,
  // otherwise foreign data will be destroyed.

  // Get number of bytes to send.
  sendSize = strlen(scpiCommand);

  // Mallocate the heap space.
  sendReadStrPtr = (ViChar *) malloc(sendSize + 4);

  // Check whether malloc worked.
  if (sendReadStrPtr == NULL)
  {
    // malloc did not work
    return VI_ERROR_ALLOC;
  }

  strcpy(sendReadStrPtr, scpiCommand);

  //---------------------------------------------------------------------------
  // Append "~n" and '\0' to scpi-command.
  // Remember, there has to be space provided, for this additional chars.

  helpSendReadStrPtr = sendReadStrPtr + sendSize;
  *helpSendReadStrPtr++ = '~';
  *helpSendReadStrPtr++ = 'n';
  *helpSendReadStrPtr++ = '\0';

  sendSize += 2;

  //---------------------------------------------------------------------------
  // copy command into input-buffer

  // Variable sendSize contains the correct value, do not change it.
  helpSendSize = sendSize;

  if (helpSendSize % 2 == 1)
  {
    // ...this means sendSize is an odd value:
    // Here, some preparation is made to make mbWriteInBuf() and with
    // that BlockDataSwapBytes() more save!
    *helpSendReadStrPtr = '\0';
    helpSendSize++;
    // now sendSize has an even value
  }

  //---------------------------------------------------------------------------
  // Wait until module-ready-state is ready.

  errStatus = mbPerformHandshake(instrumentHandle);

  if (errStatus < VI_SUCCESS)
    return errStatus;

  errStatus = mbWriteInBuf(
    instrumentHandle,
    sendReadStrPtr,
    helpSendSize
    );

  FREE(sendReadStrPtr);         // do not need the send-data any more

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // write number of bytes into Data-In-Register 300

  errStatus = mbWriteDIR(
    instrumentHandle,
    DIR_BLK_BYTES_VALID,
    sendSize
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // Wait until module-ready-state is ready.

  errStatus = mbPerformHandshake(instrumentHandle);

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // read number of bytes valid in output-buffer
  // from Data-Out-Register

  errStatus = mbReadDOR(
    instrumentHandle,
    DOR_BLK_BYTES_VALID,
    &moduleValidBytes
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //---------------------------------------------------------------------------
  // Is there something in the buffer ?

  if (moduleValidBytes == 0)
    return (HPE8311A_INSTR_ERROR_DOR_EQUAL_ZERO);
  // Nothing in the buffer!
  // We made a query and the instrument answered nothing?
  // That's really impertinent!

  //---------------------------------------------------------------------------
  // if there is something in the output buffer --> read this!

  if (moduleValidBytes > 0)
  {
    // Check if the provided buffer is big enough for the bytes to read.
    if (moduleValidBytes > readBufferSize)
      return (HPE8311A_INSTR_ERROR_OUTPUT_DATA_GREATER_READ_BUFFER);

    // Mallocate the heap space.
    sendReadStrPtr = (ViChar *) malloc(moduleValidBytes + 3);

    // Check whether malloc worked.
    if (sendReadStrPtr == NULL)
      return VI_ERROR_ALLOC;

    // Variable moduleValidBytes contains the correct value, do not change
    // it.
    helpModuleValidBytes = moduleValidBytes;

    if (helpModuleValidBytes % 2 == 1)
    {
      // ...this means sendSize is an odd value:
      // Here, some preparation is made to make mbReadOutBuf() and with
      // that BlockDataSwapBytes() more save!
      helpModuleValidBytes++;
      // now sendSize has an even value
    }

    errStatus = mbReadOutBuf(
      instrumentHandle,
      sendReadStrPtr,
      helpModuleValidBytes
      );

    if (errStatus < VI_SUCCESS)
    {
      FREE(sendReadStrPtr);
      *readBytesCount = 0;
      readBuffer[0] = '\0';
      return errStatus;
    }

    // Copy the valid values from the sendReadStrPtr to the readBuffer:
    // Remember: the data inside of sendReadStrPtr may contain eigher
    // chars or integers. When there occures 0x00 it is not shure
    // if end-of-string or just an integer is meant. Therfore strcpy()
    // is not usable.

    // The sendReadStrPtr is still needed to free space.
    helpSendReadStrPtr = sendReadStrPtr;
    // The readBuffer is still needed to pass back the values.
    helpReadBuffer = readBuffer;

    for (counter = 0; counter < moduleValidBytes; counter++)
    {
      *helpReadBuffer++ = *helpSendReadStrPtr++;
    }

    // The user of this routine wants to know how many valid bytes
    // are in the readBuffer now.
    *readBytesCount = moduleValidBytes;

    FREE(sendReadStrPtr);       // do not need the read-data any more
  }
  else
  {                             // if(  moduleValidBytes > 0  )
    *readBytesCount = 0;
    readBuffer[0] = '\0';
  }

  return (VI_SUCCESS);
}
//*****************************************************************************
// MB-UTILITY-FUNCTION
//     this function creates space on the heap and copies in that a provided
//     string.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | sourceString                                         ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | String to be copied on the heap.
//   ------------------------------------------------------------------------
//  | heapStringPtr                                        ViChar**    IN/OUT
//  |   ---------------------------------------------------------------------
//  |  | This pointer points to a pointer provided by the user. The users
//  |  |   pointer should not refer to any data, ist should only exist, or
//  |  |   data will go lost. The routine mallocs new space on the heap and
//  |  |   makes via the heapStringPtr the user provided pointer point on the
//  |  |   new space. Of course the space is filled with usefull data, coming
//  |  |   from the sourceString.
//*****************************************************************************
static ViStatus _VI_FUNC mbGetOnHeapString(
    ViChar * sourceString,
    ViChar ** heapStringPtr
)
//-----------------------------------------------------------------------------
{
  ViUInt32 sourceStringLength = 0;
  ViChar *tempHeapStringPtr = NULL;

  if(NULL == sourceString || NULL == heapStringPtr)
    return (VI_ERROR_INV_PARAMETER);

  sourceStringLength = strlen(sourceString);

  tempHeapStringPtr = (ViChar *) malloc(
    sourceStringLength + 1
    );

  if (tempHeapStringPtr == NULL)
    return (VI_ERROR_ALLOC);

  memcpy(
    tempHeapStringPtr,
    sourceString,
    sourceStringLength
    );

  tempHeapStringPtr[sourceStringLength] = 0;
  *heapStringPtr = tempHeapStringPtr;

  return (VI_SUCCESS);
}
//*****************************************************************************
// MB-UTILITY-FUNCTION
//     this function creates space on the heap and copies in it some data
//     provided inside an array.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | sourceArray                                         ViChar*         IN
//  |   ---------------------------------------------------------------------
//  |  | Array to be copied on the heap.
//   ------------------------------------------------------------------------
//  | sourceArraySize                                     ViUInt32         IN
//  |   ---------------------------------------------------------------------
//  |  | Size of the array to be copied on the heap.
//   ------------------------------------------------------------------------
//  | heapArrayPtr                                        ViChar**     IN/OUT
//  |   ---------------------------------------------------------------------
//  |  | This pointer points to an pointer provided by the user. The users
//  |  |   pointer should not refer to any data, ist should only exist, or
//  |  |   data will go lost. The routine mallocs new space on the heap and
//  |  |   makes via the heapArrayPtr the user provided pointer point on the
//  |  |   new space. Of course the space is filled with usefull data, coming
//  |  |   from the sourceArray.
//*****************************************************************************
static ViStatus _VI_FUNC mbGetOnHeapArray(
    ViChar * pSourceArray,
    ViInt32 sourceArraySize,
    ViChar ** heapArrayPtr
)
//-----------------------------------------------------------------------------
{
  ViChar *tempHeapArrayPtr = NULL;
  tempHeapArrayPtr = (ViChar *) malloc((size_t) (sourceArraySize + 1));

  if (tempHeapArrayPtr == NULL)
    return (VI_ERROR_ALLOC);

  memcpy(tempHeapArrayPtr, pSourceArray, (size_t) sourceArraySize);

  tempHeapArrayPtr[sourceArraySize] = 0;
  *heapArrayPtr = tempHeapArrayPtr;

  return (VI_SUCCESS);
}
//*****************************************************************************
// MB-UTILITY-FUNCTION
//   For the parser of the instrument these control
//   sequences have to be preceeded by an additional
//   escape. This means, the ESC has to be inserted.
//---------------------------------------------------------------------------
// These control sequences are:
//   0x21 '!'
//   0x7E '~'
//   0x5E '^'
//   0x05 ENQ
//   0x07 BEL
//   0x1B ESC
//---------------------------------------------------------------------------
// Important: This routine assumes to get a pointer on the heap, otherwise
//   an error will occure.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | heapStringPtr                                        ViChar**    IN/OUT
//  |   ---------------------------------------------------------------------
//  |  | This is a pointer to the string on the heap. The present control
//  |  |   sequences will be counted, the neccesary escapes will be inserted,
//  |  |   the space allocation / deallocation the heap is also performed.
//*****************************************************************************
static ViStatus _VI_FUNC insertEscapesString(
    ViChar ** heapStringPtr
)
//-----------------------------------------------------------------------------
{
  ViUInt32 heapStringPtrSize = 0;
  ViUInt32 additionalBytes = 0;
  ViUInt32 count = 0;
  ViChar *sourceDataPtr = NULL;
  ViChar *destinationDataPtr = NULL;
  ViChar *tempHeapStringPtr = NULL;
  ViChar actualByte;
  //---------------------------------------------------------------------------
  // First calculate how many control sequences are in the string.

  additionalBytes = 0;
  sourceDataPtr = *heapStringPtr;
  heapStringPtrSize = strlen(*heapStringPtr);

  for (count = 0; count < heapStringPtrSize; count++)
  {
    switch (*sourceDataPtr)
    {
    case 0x21:                  // '!'
    case 0x7E:                  // '~'
    case 0x5E:                  // '^'
    case 0x05:                  // ENQ
    case 0x07:                  // BEL
    case 0x1B:                  // ESC
      additionalBytes++;
      break;

    default:
      break;
    }                           // END switch()

    sourceDataPtr++;
  }                             // END for()

  //---------------------------------------------------------------------------
  // Malloc new space

  tempHeapStringPtr = (ViChar *) malloc(heapStringPtrSize + additionalBytes + 1);

  if (tempHeapStringPtr == NULL)
    return (VI_ERROR_ALLOC);

  //---------------------------------------------------------------------------
  // Copy data and insert the escape

  destinationDataPtr = tempHeapStringPtr;
  sourceDataPtr = *heapStringPtr;

  for (count = 0; count < heapStringPtrSize; count++)
  {
    // Get a char from the source.
    actualByte = sourceDataPtr[count];

    switch (actualByte)
    {
    case 0x21:                  // '!'
    case 0x7E:                  // '~'
    case 0x5E:                  // '^'
    case 0x05:                  // ENQ
    case 0x07:                  // BEL
    case 0x1B:                  // ESC
      // Insert additional ESC when control sequence occurred
      *destinationDataPtr++ = 0x1B;
      *destinationDataPtr++ = actualByte; /*lint !e796 !e797 poss.
                                           * creation/access of oob ptr. */
      break;

    default:                    // Insert this char on the heap.
      *destinationDataPtr++ = actualByte;
      break;

    }                           // END switch()
  }                             // END for()

  *destinationDataPtr = 0;      // terminate the string

  //---------------------------------------------------------------------------
  // Now exchange the pointers and kill the old allocation.

  FREE(*heapStringPtr);
  *heapStringPtr = tempHeapStringPtr;

  return (VI_SUCCESS);
}
//*****************************************************************************
// MB-UTILITY-FUNCTION
//   For the parser of the instrument these control
//   sequences have to be preceeded by an additional
//   escape. This means, the ESC has to be inserted.
//---------------------------------------------------------------------------
// These control sequences are:
//   0x21 '!'
//   0x7E '~'
//   0x5E '^'
//   0x05 ENQ
//   0x07 BEL
//   0x1B ESC
//---------------------------------------------------------------------------
// Important: This routine assumes to get a pointer on the heap, otherwise
//   an error will occure.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | heapArrayPtr                                         ViChar**    IN/OUT
//  |   ---------------------------------------------------------------------
//  |  | This is a pointer to the Array on the heap. The present control
//  |  |   sequences will be counted, the neccesary escapes will be inserted,
//  |  |   the space allocation / deallocation the heap is also performed.
//   ------------------------------------------------------------------------
//  | oldHeapArrayPtrSize                                  ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | This is the count for the valid bytes inside the provided array.
//   ------------------------------------------------------------------------
//  | newHeapArrayPtrSize                                  ViUInt32*      OUT
//  |   ---------------------------------------------------------------------
//  |  | This is the count for the valid bytes inside the manipulated array.
//  |  |   It is larger when escapes were inserted by the routine.
//*****************************************************************************
static ViStatus _VI_FUNC insertEscapesArray(
    ViChar ** heapArrayPtr,
    ViUInt32 oldHeapArrayPtrSize,
    ViUInt32 * newHeapArrayPtrSize
)
//-----------------------------------------------------------------------------
{
  ViUInt32 additionalBytes = 0;
  ViUInt32 count = 0;
  ViChar *sourceDataPtr = NULL;
  ViChar *destinationDataPtr = NULL;
  ViChar *tempHeapArrayPtr = NULL;
  ViChar actualByte;
  //---------------------------------------------------------------------------
  // First calculate how many control sequences are in the string.

  additionalBytes = 0;
  sourceDataPtr = *heapArrayPtr;

  for (count = 0; count < oldHeapArrayPtrSize; count++)
  {
    switch (*sourceDataPtr)
    {
    case 0x21:                  // '!'
    case 0x7E:                  // '~'
    case 0x5E:                  // '^'
    case 0x05:                  // ENQ
    case 0x07:                  // BEL
    case 0x1B:                  // ESC
      additionalBytes++;
      break;
    default:
      break;
    }                           // END switch()

    sourceDataPtr++;
  }                             // END for()

  //---------------------------------------------------------------------------
  // Malloc new space

  *newHeapArrayPtrSize = oldHeapArrayPtrSize + additionalBytes;

  tempHeapArrayPtr = (ViChar *) malloc(
    *newHeapArrayPtrSize + 1
    );

  if (tempHeapArrayPtr == NULL)
  {
    *newHeapArrayPtrSize = 0;
    return (VI_ERROR_ALLOC);
  }

  //---------------------------------------------------------------------------
  // Copy data and insert the escape

  destinationDataPtr = tempHeapArrayPtr;
  sourceDataPtr = *heapArrayPtr;

  for (count = 0; count < oldHeapArrayPtrSize; count++)
  {
    // Get a char from the source.
    actualByte = sourceDataPtr[count];

    switch (actualByte)
    {
    case 0x21:                  // '!'
    case 0x7E:                  // '~'
    case 0x5E:                  // '^'
    case 0x05:                  // ENQ
    case 0x07:                  // BEL
    case 0x1B:                  // ESC
      // Insert additional ESC when control
      // sequence occurred
      *destinationDataPtr = 0x1B;
      destinationDataPtr++;
      *destinationDataPtr = actualByte;
      destinationDataPtr++;
      break;

    default:                    // Insert this char on the heap.
      *destinationDataPtr = actualByte;
      destinationDataPtr++;
      break;

    }                           // END switch()
  }                             // END for()

  *destinationDataPtr = 0;      // terminate the Array

  //---------------------------------------------------------------------------
  // Now exchange the pointers and kill the old allocation.

  FREE(*heapArrayPtr);
  *heapArrayPtr = tempHeapArrayPtr;

  return (VI_SUCCESS);
}
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// FUNCTIONS FOR REGISTER-BASED-MODULES (rb)
//   (use them only driver internal)
//-----------------------------------------------------------------------------
// Originally these modules are register-based, but with changes of the
//   special-bios they are message-based now.
//-----------------------------------------------------------------------------
//It is possible to force these modules back into register-based-modus, and
//   then to use its register-based ability.
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// OPERATRIONS WITHOUT HANDSHAKE-PROTOKOL
// These routines expect the module to be in the right STAte.
// It is the users turn to be shure about this.
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Write doubleWord to the Data-In-Register
// A24-Space of the Module.
//---------------------------------------------------------------------------
// Very low-level-access, does not support any handshake-protocol.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | regNr                                                ViInt16         IN
//  |   ---------------------------------------------------------------------
//  |  | (registerNumber): selects the desired Register.
//   ------------------------------------------------------------------------
//  | writeValue                                           ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | contains the desired value to write
//*****************************************************************************
static ViStatus _VI_FUNC rbWriteA24Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt32 writeValue
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbWriteA24Reg"

  // Write doubleWord to doubleWord-Register:
  // A doubleWord must be separated into two Words
  // These two words must be transeved with 16bit operations each
  // loverWord (bit15-00) <---> higher address
  // upperWord (bit31-16) <---> lower address

  ViUInt16 writeWord = 0;
  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-----------------------------------------------------------------------
  // Write upperWord to doubleWord-Register
  // must be written first - or an unallowed Interrupt is caused

  writeWord = (ViUInt16) (writeValue >> 16);

  errStatus = viOut16(
    instrumentHandle,
    VI_A24_SPACE,
    (ViBusAddress) ((ViUInt16) regNr * 4 + BASE_DATA_IN_REG),
    writeWord
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //-----------------------------------------------------------------------
  // write lowerWord to doubleWord-Register
  // must be written last - to cause Interrupt

  writeWord = (ViUInt16) (writeValue);

  errStatus = viOut16(
    instrumentHandle,
    VI_A24_SPACE,
    (ViBusAddress) ((ViUInt16) regNr * 4 + BASE_DATA_IN_REG + 2),
    writeWord
    );

  return errStatus;
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Read doubleWord form the Data-Out-Register
// A24-Space of the Module.
//---------------------------------------------------------------------------
// Very low-level-access, does not support any handshake-protocol.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | regNr                                                ViInt16         IN
//  |   ---------------------------------------------------------------------
//  |  | (registerNumber): selects the desired Register.
//   ------------------------------------------------------------------------
//  | requestedValue                                       ViUInt32*      OUT
//  |   ---------------------------------------------------------------------
//  |  | doubleWord which is read out of the desired Register
//*****************************************************************************
static ViStatus _VI_FUNC rbReadA24Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt32 * requestedValue
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbReadA24Reg"

  // Read doubleWord from doubleWord-Register:
  // A doubleWord must be separated into two Words
  // These two words must be transeved with 16bit operations each
  // loverWord (bit15-00) <---> higher address
  // upperWord (bit31-16) <---> lower address

  ViStatus errStatus;
  ViUInt16 requestedWord = 0;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-----------------------------------------------------------------------
  // Read upperWord from doubleWord-Register
  // must be read first - or an unallowed Interrupt is caused
  errStatus = viIn16(
    instrumentHandle,
    VI_A24_SPACE,
    (ViBusAddress) ((ViUInt16) regNr * 4 + BASE_DATA_OUT_REG),
    &requestedWord
    );

  if (errStatus < VI_SUCCESS)
  {
    *requestedValue = 0;
    return errStatus;
  }

  *requestedValue = ((ViUInt32) (requestedWord) << 16);

  //-----------------------------------------------------------------------
  // Read loverWord from doubleWord-Register
  // must be read last
  errStatus = viIn16(
    instrumentHandle,
    VI_A24_SPACE,
    (ViBusAddress) ((ViUInt16) regNr * 4 + BASE_DATA_OUT_REG + 2),
    &requestedWord
    );

  if (errStatus < VI_SUCCESS)
  {
    *requestedValue = 0;
    return errStatus;
  }

  *requestedValue = (*requestedValue | (ViUInt32) (requestedWord));

  TRACE_VAR("%d", *requestedValue);

  //-----------------------------------------------------------------------
  // When reached here, all work successful done.
  return (VI_SUCCESS);
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Read Word from the Data-In-Register
// A16-Space of the Module
//---------------------------------------------------------------------------
// Very low-level-access, does not support any handshake-protocol.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | regNr                                                ViInt16         IN
//  |   ---------------------------------------------------------------------
//  |  | (registerNumber): selects the desired Register.
//   ------------------------------------------------------------------------
//  | requestedValue                                       ViUInt16*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Word which is read out of the desired Register
//*****************************************************************************
static ViStatus _VI_FUNC rbReadA16Reg(
    ViSession instrumentHandle,
    ViInt16 regNr,
    ViUInt16 * requestedValue
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbReadA16Reg"

  ViUInt16 value;
  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  errStatus = viIn16(
    instrumentHandle,
    VI_A16_SPACE,
    (ViBusAddress) ((ViUInt16) regNr * 2),
    &value
    );

  if (errStatus < VI_SUCCESS)
  {
    *requestedValue = 0;
    return errStatus;
  }
  else
    *requestedValue = value;

  TRACE_VAR("%d", *requestedValue);

  //-----------------------------------------------------------------------
  // When reached here, all work successful done.
  return (VI_SUCCESS);
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// WriteBlock to the BlockMemory
// A24-Space of the Module
//---------------------------------------------------------------------------
// Very low-level-access, does not support any handshake-protocol.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | data                                                 ViInt16*        IN
//  |   ---------------------------------------------------------------------
//  |  | Pointer to a dataBlock (Array of words).
//   ------------------------------------------------------------------------
//  | byteCount                                            ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Information about the quantity of valid bytes in the array.
//*****************************************************************************
static ViStatus _VI_FUNC rbWriteBlock(
    ViSession instrumentHandle,
    const ViUInt16 * data,
    ViUInt32 byteCount
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbWriteBlock"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // Think about byte-order.
  errStatus = BlockDataSwapBytes(
    (ViUInt16 *) data,
    byteCount
    );

  //-------------------------------------------------------------------------
  // Get the data to the instrument.

  errStatus = viMoveOut16(
    instrumentHandle,
    VI_A24_SPACE,
    BASE_BLOCK_MEMORY_REG,
    (ViBusSize) ((byteCount + 1) / 2),
    (ViAUInt16) (data)
    );

  return (errStatus);
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Check the modules execution STAte.
//---------------------------------------------------------------------------
// Very low-level-access, does not support any handshake-protocol.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | expectedSTAte                                        ViInt8          IN
//  |   ---------------------------------------------------------------------
//  |  | Expected module ready STAte.
//   ------------------------------------------------------------------------
//  | foundState                                           ViInt8         OUT
//  |   ---------------------------------------------------------------------
//  |  | Actual module ready STAte.
//*****************************************************************************
static ViStatus _VI_FUNC rbCheckExpectedReadyState(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbCheckExpectedReadyState"

  ViStatus errStatus;
  ViUInt16 valueA16;
  ViUInt32 valueA24;
  struct hpe8311a_globals *thisPtr;
# define PERFORM_TIMEOUT
#ifdef PERFORM_TIMEOUT
  ViInt32 timeOut;              // Driver Variable set by user.
  ViReal64 deltaTime;
  TIME_T startTime;
  TIME_T actualTime;
#endif

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //---------------------------------------------------------------------------
  // Get the actual used value of timeOut.

#ifdef PERFORM_TIMEOUT

  errStatus = hpe8311a_timeOut_Q(
    instrumentHandle,
    &timeOut
    );

  if (errStatus < VI_SUCCESS)
    return errStatus;

  timeOut *= 10;

#endif

  //---------------------------------------------------------------------------
  // check if module is ready for data input (bit 6 in a16 Status reg.)
  // wait until module is ready for data input
  // bit is set, when ready

#ifdef PERFORM_TIMEOUT
  startTime = GET_TIME();
#endif

  do
  {
    errStatus = rbReadA16Reg(
      instrumentHandle,
      A16_STATUS_REG,
      &valueA16
      );

    if (errStatus < VI_SUCCESS)
      return (errStatus);

    valueA16 = (ViUInt16) (valueA16 & DIR_BIT);

#ifdef PERFORM_TIMEOUT

    // Keep watch of module perhaps HPE8311A_ALWAYS answeres same STAte!
    // Do not stay longer than the allowed timeout in the loop.

    actualTime = GET_TIME();
    deltaTime = DIFF_TIME(actualTime, startTime);

    if ((ViInt32) deltaTime > timeOut)
    {
      DBG_PRINT("driver-timeout!");
      return HPE8311A_INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT;
    }

#endif

  }
  while (valueA16 == 0 && errStatus == VI_SUCCESS);

  //---------------------------------------------------------------------------
  // wait until module is ready for data input (A24-Register)
  // bit is set, when ready

#ifdef PERFORM_TIMEOUT
  startTime = GET_TIME();
#endif

  do
  {
    errStatus = rbReadA24Reg(
      instrumentHandle,
      DOR_STATE,
      &valueA24
      );

    if (errStatus < VI_SUCCESS)
      return (errStatus);

#ifdef PERFORM_TIMEOUT

    // Keep watch of module perhaps HPE8311A_ALWAYS answeres same STAte!
    // Do not stay longer than the allowed timeout in the loop.
    actualTime = GET_TIME();
    deltaTime = DIFF_TIME(actualTime, startTime);

    if ((ViInt32) deltaTime > timeOut)
    {
      DBG_PRINT("driver-timeout!");
      return HPE8311A_INSTR_ERROR_DRIVER_INTERNAL_TIMEOUT;
    }

#endif

  }
  while (valueA24 == DOV_BUSY && errStatus == VI_SUCCESS);

#ifdef DEBUG_VERBOSE
  switch (valueA24)
  {
  case DOV_READY:
    DebugPrintf("DOV_READY\n");
    break;
  case DOV_READY_AFTER_REQUEST:
    DebugPrintf("DOV_READY_AFTER_REQUEST\n");
    break;
  case DOV_READY_FOR_NEXT_BLK:
    DebugPrintf("DOV_READY_FOR_NEXT_BLK\n");
    break;
  case DOV_BUSY:
    DebugPrintf("DOV_BUSY\n");
    break;
  case DOV_ERROR:
    DebugPrintf("DOV_ERROR\n");
    break;
  case DOV_READY_WITHIN_TRANSACTION:
    DebugPrintf("DOV_READY_WITHIN_TRANSACTION\n");
    break;
  case DOV_STATE_FORCE_INT:
    DebugPrintf("DOV_STATE_FORCE_INT\n");
    break;
  default:
    DebugPrintf("DO NOT KNOW THIS STATE!!!\n");
  }
  DebugPrintf("...\n");
#endif          
/* end of DEBUG_VERBOSE */

  *foundState = (ViInt8) valueA24;

  if (expectedSTAte != *foundState)
    return (HPE8311A_INSTR_ERROR_WRONG_STATE);

  return (VI_SUCCESS);
}
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
// FUNCTIONS FOR REGISTER-BASED-MODULES (rb)
// OPERATRIONS WITH HANDSHAKE-PROTOKOL
// Routines provide abilities to wait until modul has reached the ready-state
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Writes a register in a24 (data-in-register) with protocol.
//---------------------------------------------------------------------------
// If the module isn't in the expected STAte, then nothing
//   will be done and the error will be reported.
// If the module is in the expected STAte, then the register
//   access will be executed and the return value VI_SUCCESS.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | expectedSTAte                                        ViInt8          IN
//  |   ---------------------------------------------------------------------
//  |  | Expected module ready STAte.
//   ------------------------------------------------------------------------
//  | foundState                                           ViInt8         OUT
//  |   ---------------------------------------------------------------------
//  |  | Actual module ready STAte.
//   ------------------------------------------------------------------------
//  | regNr                                                ViInt16         IN
//  |   ---------------------------------------------------------------------
//  |  | Register Number to write writeValue to.
//   ------------------------------------------------------------------------
//  | writeValue                                           ViInt32         IN
//  |   ---------------------------------------------------------------------
//  |  | Value to write to the desired register.
//*****************************************************************************
static ViStatus _VI_FUNC rbWriteA24RegisterUINT(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState,
    ViInt16 regNr,
    ViInt32 writeValue
)
//-----------------------------------------------------------------------------
{
  ViStatus errStatus;
  ViInt8 newFoundState;
  errStatus = rbCheckExpectedReadyState(
    instrumentHandle,
    expectedSTAte,
    &newFoundState
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return errStatus;

  errStatus = rbWriteA24Reg(
    instrumentHandle,
    (ViInt16) regNr,
    (ViUInt32) writeValue
    );

  return errStatus;
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Write to the block memory with protocol (<60 KByte).
//---------------------------------------------------------------------------
// If the module isn't in the expected STAte, then nothing
//   will be done and the error will be reported.
// If the module is in the expected STAte, then the register
//   access will be executed and the return value VI_SUCCESS.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | expectedSTAte                                        ViInt8          IN
//  |   ---------------------------------------------------------------------
//  |  | Expected module ready STAte.
//   ------------------------------------------------------------------------
//  | foundState                                           ViInt8         OUT
//  |   ---------------------------------------------------------------------
//  |  | Actual module ready STAte.
//   ------------------------------------------------------------------------
//  | regNr                                                ViInt16         IN
//  |   ---------------------------------------------------------------------
//  |  | Register Number to write the commandNr to.
//   ------------------------------------------------------------------------
//  | commandNr                                            ViInt32         IN
//  |   ---------------------------------------------------------------------
//  |  | Command Number to write to the desired register.
//  |  |   This command describes, kind of block memory access:
//  |  |   -> DIV_BTF_WRITE
//  |  |   -> DIV_BTF_DONE
//   ------------------------------------------------------------------------
//  | blockData                                            ViUInt16*       IN
//  |   ---------------------------------------------------------------------
//  |  | Pointer to the data to transfer.
//   ------------------------------------------------------------------------
//  | byteCount                                            ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Number of valid bytes to transfer.
//*****************************************************************************
static ViStatus _VI_FUNC rbWriteBlockSmall(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViInt8 * foundState,
    ViInt16 regNr,
    ViUInt32 commandNr,
    const ViUInt16 * blockData,
    ViUInt32 byteCount
)
//-----------------------------------------------------------------------------
{
  ViStatus errStatus;
  ViInt8 newFoundState;
  //-------------------------------------------------------------------------
  // Check the right module-state

  errStatus = rbCheckExpectedReadyState(
    instrumentHandle,
    expectedSTAte,
    &newFoundState
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Write the data into the instrument.

  errStatus = rbWriteBlock(
    instrumentHandle,
    blockData,
    byteCount
    );

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Inform instrument about the new data.

  errStatus = rbWriteA24Reg(
    instrumentHandle,
    regNr,
    commandNr
    );

  return (errStatus);
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Erase the boot-flash-prom on a module sector 1-7). The module is running
//   in loader after erasing the flash-prom.
//---------------------------------------------------------------------------
// If the module isn't in the expected STAte, then nothing
//   will be done and the error will be reported.
// If the module is in the expected STAte, then the register
//   access will be executed and the return value VI_SUCCESS.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | expectedSTAte                                        ViInt8          IN
//  |   ---------------------------------------------------------------------
//  |  | Expected module ready STAte.
//   ------------------------------------------------------------------------
//  | foundState                                           ViInt8         OUT
//  |   ---------------------------------------------------------------------
//  |  | Actual module ready STAte.
//*****************************************************************************
static ViStatus _VI_FUNC rbEraseBootProm(
    ViSession instrumentHandle,
    ViInt8 expectedSTAte,
    ViPInt8 foundState
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbEraseBootProm"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViInt8 newFoundState;
  ViReal64 deltaTime;
  TIME_T startTime;
  TIME_T actualTime;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // Switch module into loader mode.

  // Make module accepting expert command.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    expectedSTAte,
    &newFoundState,
    DIR_EXP_COMMAND,
    DIV_EC_ENABLE
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return errStatus;

  // Send expert command "switch to loader mode" to the instrument.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    expectedSTAte,
    &newFoundState,
    DIR_EXP_COMMAND,
    DIV_EC_UR_BIOS
    );

  startTime = GET_TIME();

  do
  {
    actualTime = GET_TIME();
    deltaTime = DIFF_TIME(actualTime, startTime);
  }
  while (deltaTime < 2000.0);

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return errStatus;

  //-------------------------------------------------------------------------
  // Erase boot-flash-prom (sector 1-7)

  // Make module accepting expert command.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    expectedSTAte,
    &newFoundState,
    DIR_EXP_COMMAND,
    DIV_EC_ENABLE
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return errStatus;

  DBG_PRINT("MODULE_IO: erase boot-flash-prom!");

  // Send expert command "erase the bios-prom" to the instrument.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    expectedSTAte,
    &newFoundState,
    DIR_EXP_COMMAND,
    DIV_EC_ERASE_BIOS_PROM
    );

  *foundState = newFoundState;

  return errStatus;
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Force a module to jump from it's loader to the
//   special-bios and reboots the module.
//---------------------------------------------------------------------------
// If the module isn't in the expected STAte, then nothing
//   will be done and the error will be reported.
// If the module is in the expected STAte, then the register
//   access will be executed and the return value VI_SUCCESS.
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | expectedSTAte                                        ViInt8          IN
//  |   ---------------------------------------------------------------------
//  |  | Expected module ready STAte.
//   ------------------------------------------------------------------------
//  | foundState                                           ViInt8         OUT
//  |   ---------------------------------------------------------------------
//  |  | Actual module ready STAte.
//*****************************************************************************
static ViStatus _VI_FUNC rbSpecialBios(
    ViSession instrumentHandle,
    ViInt8 * foundState
)
//-----------------------------------------------------------------------------
{
#  undef FUNC_NAME
# define FUNC_NAME "rbSpecialBios"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViInt8 newFoundState;
  ViReal64 deltaTime;
  TIME_T startTime;
  TIME_T actualTime;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // Perhaps the module is busy?
  // Now calling rbCheckExpectedReadyState because of ist handshake ability.
  errStatus = rbCheckExpectedReadyState(
    instrumentHandle,
    DOV_READY,                  // maybe something else!
    &newFoundState
    );

  *foundState = newFoundState;

  if (errStatus != HPE8311A_INSTR_ERROR_WRONG_STATE)
  {
    if (errStatus < VI_SUCCESS)
      return (errStatus);
  }

  //-------------------------------------------------------------------------
  DBG_PRINT("kill all pending errors!");

  // Get module into ready STAte - kill all pending errors.
  // Do use rbWriteA24RegisterUINT now, because module-state known from
  // above!
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    *foundState,
    &newFoundState,
    DIR_USR_COMMAND,
    DIV_UC_READY
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // jump from loader in special-bios

  // Make module accepting expert command.
  // Now module should be synchrinized!
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    DOV_READY,
    &newFoundState,
    DIR_USR_COMMAND,
    DIV_EC_ENABLE
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Send expert command "boot-module" to the instrument.
  // After this, the module is in special-bios mode.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    *foundState,
    &newFoundState,
    DIR_EXP_COMMAND,
    DIV_EC_BOOT
    );

  *foundState = newFoundState;

  startTime = GET_TIME();

  do
  {
    actualTime = GET_TIME();
    deltaTime = DIFF_TIME(actualTime, startTime);
  }
  while (deltaTime < 1000.0);

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  // With this command, module errors from instrument may rise up.
  // Also this command may take a lot of time.

  //-------------------------------------------------------------------------
  DBG_PRINT("get synchronized with handshake!");

  // Booting special-bios may take really a long time.
  // Best way to check this is using the handshake ability
  // of rbCheckExpectedReadyState().
  errStatus = rbCheckExpectedReadyState(
    instrumentHandle,
    *foundState,
    &newFoundState
    );

  *foundState = newFoundState;

  // Maybe, there occures an error from the instrument!!!
  // So it is not shure to find DOV_READY!!!
  // Don't mind about wrong STAte, just go on!!!
  if (errStatus != HPE8311A_INSTR_ERROR_WRONG_STATE)
  {
    if (errStatus < VI_SUCCESS)
      return (errStatus);
  }

  //-------------------------------------------------------------------------
  DBG_PRINT("kill all pending errors!");

  // When there occurred an error from the instrument:
  // Get module into ready STAte - kill all pending errors!!
  // This is useful to program proms. This kind of errors will
  // occure again, when power cycling of module is performed.
  errStatus = rbWriteA24RegisterUINT(
    instrumentHandle,
    *foundState,
    &newFoundState,
    DIR_USR_COMMAND,
    DIV_UC_READY
    );

  *foundState = newFoundState;

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  DBG_PRINT("get synchronized with handshake!");

  // Check if module is really in ready-state
  errStatus = rbCheckExpectedReadyState(
    instrumentHandle,
    DOV_READY,
    &newFoundState
    );

  *foundState = newFoundState;

  return (errStatus);
}
/* --------------------------------------------------------------------------
 * The real worker
 * -------------------------------------------------------------------------- */

static ViUInt8 HexChar2Nibble(ViUInt8 chAsc)
{
  if (chAsc >= '0' && chAsc <= '9')
    return (ViUInt8) (chAsc - '0');

  if (chAsc >= 'A' && chAsc <= 'F')
    return (ViUInt8) (chAsc - 'A' + 10);

  if (chAsc >= 'a' && chAsc <= 'f')
    return (ViUInt8) (chAsc - 'a' + 10);

  return 0x0;
}


//-----------------------------------------------------------------------------
// Function      : convert array of ascci value to a array of bytes
//-----------------------------------------------------------------------------
// Parameter     : src           : ptr of source string
//                 dest          : ptr of destination string
//-----------------------------------------------------------------------------
// Return        : ptr to the first character in the destination
//                 string
//-----------------------------------------------------------------------------
static ViUInt8 *Ascii2Byte(
    ViUInt8 * dest,
    const ViChar * src
)
//-----------------------------------------------------------------------------
{
  ViUInt8 *d = dest;
  ViUInt8 chAsc;
  ViUInt8 chByte = 0;
  int bLeftNibble = 1;
  while ((chAsc = (ViUInt8) * src++) != '\0' && chAsc != '\r' && chAsc != '\n')
  {
    if (bLeftNibble)
    {
      chByte = HexChar2Nibble(chAsc);
      chByte <<= 4;
    }
    else
    {
      chByte = (ViUInt8) (chByte + HexChar2Nibble(chAsc));
      *dest++ = chByte;
    }

    // toggle for next nibble
    bLeftNibble = !bLeftNibble;
  }

  // if it's the right nibble's turn then we haven't put the last
  // one into dest buffer yet.
  if (bLeftNibble)
    *dest++ = chByte;

  return (d);
}

//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Programm the BIOS of a module
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | filename                                             ViPString       IN
//  |   ---------------------------------------------------------------------
//  |  | Name of the BIOS-file on the PC.
//*****************************************************************************
static ViStatus _VI_FUNC rbProgramBiosFile(
    ViSession instrumentHandle,
    const ViChar filename[]
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "rbProgramBiosFile"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViStatus errStatusBackup = VI_SUCCESS;
  FILE *fptr = NULL;            // pointer to a file
  ViUInt32 dataCount = 0;       // counter to count the bytes inside a file
  ViChar *bufferP = NULL;       // pointer to a large data-space on the heap
  ViChar *helpBufP = NULL;      // help-pointer to the same data-space
  ViChar fileChar;              // char, read out of the file

  enum
  {
    OPEN_FILE,
    COUNT_DATA,
    MALLOC_SPACE,
    MOVE_BIOS,
    FREE_SPACE,
    CLOSE_FILE,
    FINISHED
  };

  ViInt8 STAte = OPEN_FILE;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  errStatusBackup = VI_SUCCESS;

  while (STAte != FINISHED)
  {
    switch (STAte)
    {
      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case OPEN_FILE:
      fptr = fopen(filename, "r");

      if (fptr == NULL)
      {
        DBG_PRINT1("OPEN_FILE [%s] failed", filename);
        STAte = FINISHED;
        errStatusBackup = HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE;
      }
      else
        STAte = COUNT_DATA;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case COUNT_DATA:
      // count the bytes inside of the file
      // (it's a text file so we'll do this the long way.)
      dataCount = 0;

      /*lint -e668 (poss. null ptr...fptr) */

      while (fgetc(fptr) != EOF)
        dataCount++;

      // go back to beginning...
      if (fseek(fptr, 0, SEEK_SET))
      {
        errStatusBackup = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
        STAte = CLOSE_FILE;
      }
      else
      {
        // file was empty?
        if (dataCount == 0)
        {
          DBG_PRINT1("Source file [%s] is empty", filename);
          errStatusBackup = HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_EMPTY;
          STAte = CLOSE_FILE;
          break;
        }
        else
          STAte = MALLOC_SPACE;
      }

       /*lint +e668 */

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case MALLOC_SPACE:
      // malloc space on heap, the file may be quite large
      // need additional byte for EOF
      bufferP = (ViChar *) malloc(dataCount + 1);

      // malloc did not work?
      if (bufferP == NULL)
      {
        DBG_PRINT1("Malloc failed [Not enough memory free to read file %s]", filename);
        errStatusBackup = VI_ERROR_ALLOC;
        STAte = CLOSE_FILE;
      }
      else
        STAte = MOVE_BIOS;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case MOVE_BIOS:
      helpBufP = bufferP;

      /*lint -e668 -e613 pass/use null ptr (fptr) */

      while (EOF != (fileChar = (ViChar) fgetc(fptr)))
        *helpBufP++ = fileChar;

      *helpBufP = (ViChar)EOF;  //lint !e794 ... rbProgramBiosBuffer need the EOF !!

      errStatusBackup = rbProgramBiosBuffer(
        instrumentHandle,
        (const ViChar *) bufferP
        );

      /*lint +e668 +e613 */

      STAte = FREE_SPACE;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case FREE_SPACE:
      FREE(bufferP);
      STAte = CLOSE_FILE;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case CLOSE_FILE:

      if (fptr)
      {
        if (fclose(fptr) == EOF)
        {
          DBG_PRINT1("File Close failed [file %s]", filename);
          errStatusBackup = HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE;
        }
      }

      STAte = FINISHED;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case FINISHED:              // DO NOT PUT CODE HERE
      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    }                           /*lint !e744 switch statement has no default */
  }                             // END while(  STAte != FINISHED )

  // Get an error, which was backuped.
  if (errStatus == VI_SUCCESS)
    errStatus = errStatusBackup;

  return (errStatus);
}
//*****************************************************************************
// FUNCTION FOR REGISTER-BASED-MODULES (rb)
// Programm the BIOS of a module
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     viSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init( )
//   ------------------------------------------------------------------------
//  | buffer                                               ViPString       IN
//  |   ---------------------------------------------------------------------
//  |  | Contains all the data to program.
//*****************************************************************************
static ViStatus _VI_FUNC rbProgramBiosBuffer(
    ViSession instrumentHandle,
    const ViChar * buffer
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "rbProgramBiosBuffer"

  union PtrConv
  {
    ViUInt8 *pu8;
    ViUInt16 *pu16;
    ViUInt32 *pu32;
    void *pv;
  };

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViStatus errStatusBackup;
  ViInt8 expectedSTAte = DOV_READY;
  ViInt8 foundState = DOV_READY;
  union PtrConv pBlkMem;

  enum
  {
    ERASE_PROM,
    CHECK_BIOS,
    PROGRAM_PROM,
    BOOT_MODULE,
    CHECK_MODULE,
    ERROR_OCCURED,
    FINISHED
  };

  ViInt32 STAte;
  // helpPointer to the buffer
  const ViChar * helpP = buffer;

  // block memory
  struct BIV_RawIOControl *header;

  ViUInt8 *bm_ptr;
  ViInt32 blk_mem_len;
  ViInt32 max_len;
  // header info
  ViUInt32 num_s_rec;
  ViUInt32 num_bytes;
  // flags
  ViInt8 prog_prom_flag;
  ViInt8 transfer_flag;
  ViInt8 error_flag;
  // s-record
  ViInt8 i;
  ViInt8 buf_char;
  ViInt32 src_pos;
  ViChar src_buffer[100];
  ViChar *src;
  ViUInt8 dest_buffer[100];
  ViUInt8 *dest;
  ViUInt8 length;
  ViUInt32 checksum;
  // initialize the pointer to the block memory buffer
  ViUInt8 *blk_mem_buffer;
  blk_mem_buffer = (ViUInt8 *) malloc(MAX_BLK_LEN * sizeof(ViUInt8));
  if (blk_mem_buffer == NULL)
    return (VI_ERROR_ALLOC);    // malloc did not work.

  pBlkMem.pu8 = blk_mem_buffer;
  header = (struct BIV_RawIOControl *) (pBlkMem.pv);

  src = (ViChar *) & src_buffer[0];
  dest = (ViUInt8 *) & dest_buffer[0];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  error_flag = 0;
  errStatusBackup = VI_SUCCESS;
  STAte = CHECK_BIOS;

  while (STAte != FINISHED)
  {
    switch (STAte)
    {

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    case CHECK_BIOS:

      DBG_PRINT("CHECK_BIOS: Verifying checksums of BIOS-file.");

      // reading ALL lines
      // STArting..............................................
      do
      {
        src_pos = 0;
        buf_char = 0;

        // reading A line
        // STArting...............................................
        while ((src_pos < 99) &&
          (buf_char != EOF) &&
          (buf_char != '\n')
          )
        {
          buf_char = *helpP++;

          if ((buf_char != EOF) &&
            (((buf_char >= 'A') && (buf_char <= 'Z')) ||
              ((buf_char >= 'a') && (buf_char <= 'z')) ||
              ((buf_char >= '0') && (buf_char <= '9')) ||
              (buf_char == ' ')
            )
            )
            src[src_pos++] = (char) (buf_char & 0xFF);
        }                       // END while
        // reading a line
        // finished...............................................

        src[src_pos] = 0;

        if (buf_char != EOF)
          dest = Ascii2Byte(dest, src);

        // Calculate checksum
        checksum = 0;
        length = dest[1];
        for (i = 1; i <= length; i++)
          checksum += dest[i];

        if ((~checksum & 0xff) != dest[length + 1])
        {
          DBG_PRINT("CheckBios: Checksum error !");
          errStatusBackup = HPE8311A_INSTR_ERROR_BIOS_FILE_CHECKSUM_ERROR;
          STAte = FINISHED;
        }

      } while ((buf_char != EOF) && (STAte != FINISHED));
      // reading ALL lines
      // finished..............................................

      // set helpP to the begin of the buffer
      helpP = buffer;

      if (error_flag == 1)
        STAte = FINISHED;
      else
        STAte = ERASE_PROM;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case ERASE_PROM:

      DBG_PRINT("SECTION: ERASE_PROM - START!");

      errStatus = rbCheckExpectedReadyState(
        instrumentHandle,
        DOV_READY,
        &foundState
        );

      if (errStatus != HPE8311A_INSTR_ERROR_WRONG_STATE)
      {
        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
          STAte = FINISHED;
        }
      }

      expectedSTAte = foundState;
      errStatus = VI_SUCCESS;

      // Get module into ready STAte - kill all pending errors

      if (error_flag == 0)
      {
        errStatus = rbWriteA24RegisterUINT(
          instrumentHandle,
          expectedSTAte,
          &foundState,
          DIR_USR_COMMAND,
          DIV_UC_READY
          );

        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
          STAte = FINISHED;
        }
      }

      if (error_flag == 0)
      {
        // Now, it should be possible to erase the prom

        errStatus = rbEraseBootProm(
          instrumentHandle,
          DOV_READY,
          &foundState
          );

        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
          STAte = ERROR_OCCURED;
        }

        expectedSTAte = foundState;
      }

      if (error_flag == 0)
        STAte = PROGRAM_PROM;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case PROGRAM_PROM:

      DBG_PRINT("PROGRAM_PROM: program flash-prom.");

      // program flash-prom with s-records

      blk_mem_len = MAX_BLK_LEN - sizeof(*header) - 1000;

      num_s_rec = 0;
      max_len = blk_mem_len;

      bm_ptr = (ViUInt8 *) (header + 1);

      prog_prom_flag = 1;
      transfer_flag = 0;

      while (prog_prom_flag || transfer_flag)
      {
        if (transfer_flag)
        {

          if (num_s_rec > 0)
          {
            num_bytes = (ViUInt32) bm_ptr - (ViUInt32) pBlkMem.pv;
            *(ViUInt8 *) bm_ptr = 0;
            num_bytes = (num_bytes + 1) & (~1);

            header->uParam = num_s_rec;
            header->uLength = num_bytes;

            DBG_PRINT1("rbProgramBiosBuffer(): PROGRAM_PROM: write %d bytes", num_bytes);

            //-----------------------------------------------------
            // Think about byte-order:
            // The first values of the block-transfer-data are the
            // header-data, they are strored in uint32 format.
            // To get motorola-format and intel-format conform,
            // the values of the header-data have to be exchaned:

            errStatus = BlockDataSwapBytes(
              pBlkMem.pu16,
              sizeof(*header)
              );

            if (errStatus < VI_SUCCESS)
            {
              errStatusBackup = errStatus;
              error_flag = 1;
              STAte = ERROR_OCCURED;
              break;
            }

            BlockDataSwapWords(
              pBlkMem.pu32,
              (sizeof(*header) / sizeof(ViUInt32))
              );

            errStatus = rbWriteBlockSmall(
              instrumentHandle,
              expectedSTAte,
              &foundState,
              DIR_BTF_BIOS_PROM,
              DIV_BTF_WRITE,
              (ViUInt16 *) pBlkMem.pu16,
              num_bytes
              );

            if (errStatus < VI_SUCCESS)
            {
              errStatusBackup = errStatus;
              error_flag = 1;
              STAte = ERROR_OCCURED;
              break;
            }

            expectedSTAte = DOV_READY_FOR_NEXT_BLK;
          }                     // END if( num_s_rec > 0 )

          num_s_rec = 0;
          max_len = blk_mem_len;
          bm_ptr = (ViUInt8 *) (header + 1);
        }
        else
        {                       // if( transfer_flag )
          src_pos = 0;
          buf_char = 0;

          // parse in A
          // line.........................................................
          while ((src_pos < 99) &&
            (buf_char != EOF) &&
            (buf_char != '\n')
            )
          {
            buf_char = *helpP++;

            if ((buf_char != EOF) &&
              (((buf_char >= 'A') && (buf_char <= 'Z')) ||
                ((buf_char >= 'a') && (buf_char <= 'z')) ||
                ((buf_char >= '0') && (buf_char <= '9')) ||
                (buf_char == ' ')
              )
              )
              src[src_pos++] = (char) (buf_char & 0xFF);
          }                     // END while - line parsing is finished
          // ..................................

          src[src_pos] = 0;     // terminate the line

          if (buf_char != EOF)
            dest = Ascii2Byte(dest, src);
          else
            prog_prom_flag = 0;

        }                       // END if( transfer_flag )

        if (((dest[1] + 2) < max_len) && prog_prom_flag)
        {
          for (i = 0; i < (dest[1] + 2); i++)
            *bm_ptr++ = dest[i];

          num_s_rec += 1;
          max_len -= (dest[1] + 2);
          transfer_flag = 0;
        }
        else
        {
          if (transfer_flag)
            transfer_flag = 0;
          else
            transfer_flag = 1;
        }

      }                         // END while( prog_prom_flag || transfer_flag
      // )

      if (error_flag == 1)
        STAte = ERROR_OCCURED;
      else
        STAte = CHECK_MODULE;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case CHECK_MODULE:

      DBG_PRINT("CHECK_MODULE: check module and find errors.");

      if (error_flag == 0)
      {
        errStatus = rbWriteA24RegisterUINT(
          instrumentHandle,
          expectedSTAte,
          &foundState,
          DIR_BTF_BIOS_PROM,
          DIV_BTF_DONE
          );

        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
        }
      }

      if (error_flag == 0)
      {
        errStatus = rbCheckExpectedReadyState(
          instrumentHandle,
          DOV_READY,
          &foundState
          );

        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
        }

        expectedSTAte = foundState;
      }

      if (error_flag == 1)
        STAte = ERROR_OCCURED;
      else
        STAte = BOOT_MODULE;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case BOOT_MODULE:

      DBG_PRINT("BOOT_MODULE: reboot module.");

      if (error_flag == 0)
      {
        errStatus = rbSpecialBios(
          instrumentHandle,
          &foundState
          );

        if (errStatus < VI_SUCCESS)
        {
          error_flag = 1;
          errStatusBackup = errStatus;
        }
        else
        {
          DBG_PRINT("BOOT_MODULE: bios successfully programmed.");
        }
      }

      if (error_flag == 1)
        STAte = ERROR_OCCURED;
      else
        STAte = FINISHED;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case ERROR_OCCURED:

      DBG_PRINT("ERROR: programming of module failed.");

      // If we are here, then a serious error occoured
      // to ensure further operability of the module,
      // force the module to the ready STAte.

      // Maybe, the module is still busy. This can be checked
      // with the handshake ability of rbCheckExpectedReadyState()!
      errStatus = rbCheckExpectedReadyState(
        instrumentHandle,
        DOV_READY,
        &foundState);

      if (errStatus != HPE8311A_INSTR_ERROR_WRONG_STATE)
      {
        if (errStatus < VI_SUCCESS)
          errStatusBackup = errStatus;
      }

      expectedSTAte = foundState;
      errStatus = VI_SUCCESS;

      // get module into ready STAte - kill all pending errors
      errStatus = rbWriteA24RegisterUINT(
        instrumentHandle,
        expectedSTAte,
        &foundState,
        DIR_USR_COMMAND,
        DIV_UC_READY
        );

      // in every case:
      STAte = FINISHED;

      break;

      // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    case FINISHED:
      // DO NOT PUT CODE HERE
      break;

    }                           /*lint !e744 switch statement has no default */
  }                             // END while(  STAte != FINISHED )

  // Get an error, which was perhaps backuped.
  if (errStatus == VI_SUCCESS)
    errStatus = errStatusBackup;

  return (errStatus);
}


//#############################################################################
//#############################################################################
//####################                                     ####################
//##########                                                         ##########
//####                          DRIVER FUNCTIONS                           ####
//##                                                                         ##
//##-------------------------------------------------------------------------##
//##     The functions which are exposed to the driver user STArt here.      ##
//##     They are in the following order:                                    ##
//##          VXIplug&play Required Functions                                ##
//##          STAndard HP Functions                                          ##
//##          Status Functions                                               ##
//##          Passthrough Functions                                          ##
//##          Subsystem Functions                                            ##
//##          Important Dircet Functions                                     ##
//##          Application Functions                                          ##
//####                                                                     ####
//##########                                                         ##########
//####################                                     ####################
//#############################################################################
//#############################################################################

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//@@ VXIplug&play Required Functions                                         @@
//@@-------------------------------------------------------------------------@@
//@@ These functions are mandated by the VXI plug&play specifications and    @@
//@@ must be part of all drivers.                                            @@
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession      resourceName (in)
//     The instrument's VISA address.
//     GPIB Example: "GPIB0::20::INSTR"
//   ViBoolean      IDQuery (in)
//     If VI_TRUE, validate instrument identity
//   ViBoolean      resetDevice (in)
//     If VI_TRUE, perform an instrument reset
//   ViPSession     instrumentHandle (out)
//     Pointer to Instrument Handle.  Returns VI_NULL if the init fails.
//*****************************************************************************
// Initializes the instrument session and returns a valid instrument handle.
// This will HPE8311A_ALWAYS be the first function to be executed to control an
// instrument.
// The instrument handle returned has to be used as the first parameter in all
// other driver functions.  hpe8311a_init can also optionally reset the
// instrument and/or verify the instrument's identity.
// The instrument is identified using the response to the "*IDN?" query and it
// is suggested that the response up to the Model Number be used for comparison.
// The firmware revision should not be considered.
// A model version number may or may not be included.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_init(
    ViRsrc resourceName,
    ViBoolean IDQuery,
    ViBoolean resetDevice,
    ViPSession pInstrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_init"

  struct hpe8311a_globals * thisPtr = NULL;
  ViStatus errStatus;
  ViSession defRM;
  ViChar idn_buf[HPE8311A_STRING_LENGTH] = {0};
  ViChar opt_buf[HPE8311A_STRING_LENGTH] = {0};
  ViUInt32 validLength;
  ViUInt16 manufacturerID;
  ViInt8 count;
  ViChar *pcOptHelper;

  *pInstrumentHandle = VI_NULL;
  
  //-----------------------------------------------------------
  // Verify the input parameters: resourceName
  //-----------------------------------------------------------

  TRACE_IN();                   

  // unvalid pointer ??
  if (resourceName == NULL)
    return (VI_ERROR_PARAMETER1);

  // string empty ??
  if (strlen(resourceName) == 0)
    return (VI_ERROR_PARAMETER1);

  // string is too long ??
  if (strlen(resourceName) > HPE8311A_STRING_LENGTH)
    return (VI_ERROR_PARAMETER1);

  // wrong interface type ??
  if (strstr(resourceName, "ASRL") != NULL)
    return (HPE8311A_INSTR_ERROR_NOT_SUPPORTED_INSTR_TYPE);

  //-----------------------------------------------------------
  // Verify the input parameter: IDQuery
  //-----------------------------------------------------------

  if (chk_boolean(NULL, IDQuery))
    return (VI_ERROR_PARAMETER2);

  //-----------------------------------------------------------
  // Verify the input parameter: resetDevice
  //-----------------------------------------------------------

  if (chk_boolean(NULL, resetDevice))
    return (VI_ERROR_PARAMETER3);

  //-----------------------------------------------------------
  // Find the Default Resource Manager.  Potential errors are
  // VI_ERROR_SYSTEM_ERROR, VI_ERROR_ALLOC
  //-----------------------------------------------------------

  errStatus = viOpenDefaultRM(&defRM);

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-----------------------------------------------------------
  // Open a session to the instrument.  Potential errors are
  // VI_ERROR_NSUP_OPER, VI_ERROR_INV_RSRC_NAME,
  // VI_ERROR_INV_ACC_MODE, VI_ERROR_RSRC_NFOUND,
  // VI_ERROR_ALLOC
  //-----------------------------------------------------------

  errStatus = viOpen(
    defRM,
    resourceName,
    VI_NULL,
    2500,
    pInstrumentHandle
    );

  if (errStatus < VI_SUCCESS)
  {
    (void) viClose(defRM);
    return (errStatus);
  }

  //------------------------------------------------------------
  // Allocate global memory for the session.  Inititalize the
  // memory.  Note that viSetAttribute should never fail,
  // as all of it's input conditions are satisfied.
  //------------------------------------------------------------

  thisPtr = (struct hpe8311a_globals *) malloc(sizeof(struct hpe8311a_globals));

  if (thisPtr == NULL)
    INIT_ERR_RETURN (VI_ERROR_ALLOC);

  // init so we know where we started from
  memset(thisPtr, 0, sizeof(struct hpe8311a_globals));

  errStatus = viSetAttribute(
    *pInstrumentHandle,
    VI_ATTR_USER_DATA,
    (ViAttrState) thisPtr
    );

  if (errStatus < VI_SUCCESS)
    INIT_ERR_RETURN (errStatus);

  //-----------------------------------------------------------
  // Set initial values of variables in global memory.
  //-----------------------------------------------------------

  thisPtr->errNumber = VI_SUCCESS;
  thisPtr->errMessage[0] = 0;
  thisPtr->errQueryDetect = VI_FALSE;
  thisPtr->dTime = 0;
  thisPtr->modelCode = 0;

  //-----------------------------------------------------------
  // Set actual data from the resource-manager.
  //-----------------------------------------------------------

  errStatus = viGetAttribute(
    *pInstrumentHandle,
    VI_ATTR_INTF_TYPE,
    &thisPtr->interfaceType
    );

  if (errStatus < VI_SUCCESS)
    INIT_ERR_RETURN (errStatus);

  errStatus = viGetAttribute(
    *pInstrumentHandle,
    VI_ATTR_RSRC_NAME,
    thisPtr->resourceName
    );

  if (errStatus < VI_SUCCESS)
    INIT_ERR_RETURN (errStatus);

  //-----------------------------------------------------------
  // Set new value for timeOut.
  //-----------------------------------------------------------

  errStatus = hpe8311a_timeOut(
    *pInstrumentHandle,
    HPE8311A_GLOBAL_TIMEOUT
    );

  if (errStatus < VI_SUCCESS)
    INIT_ERR_RETURN (errStatus);

  switch (thisPtr->interfaceType)
  {

  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    //----------------------------------------------------------------
    // Get manufacturerID to see, whether the module
    // comes from Hewlett-Packard at all.

    // Do not need handshake here. Is only manufacturerID.
    errStatus = rbReadA16Reg(
      *pInstrumentHandle,
      A16_ID_REG,
      &manufacturerID
      );

    // manufacturerID only inside the lower 3 nibbles.
    manufacturerID = (ViUInt16) (manufacturerID & 0x0FFF);

    if (manufacturerID != HEWLETT_PACKARD)
      INIT_ERR_RETURN (HPE8311A_INSTR_ERROR_NOT_A_HEWLETT_PACKARD_MODULE);

    //----------------------------------------------------------------
    // Get model code to see, whether module
    // is already in register-mode, that
    // means loader-state.

    // Do not need handshake here. Is only model-code.
    errStatus = rbReadA16Reg(
      *pInstrumentHandle,
      A16_DEV_TYPE_REG,
      &(thisPtr->modelCode)
      );

    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (errStatus);

    // modelCode only inside the three last nibbles
    thisPtr->modelCode = (ViUInt16) (thisPtr->modelCode & 0x0FFF);

    CHECK_URBIOS;

    //----------------------------------------------------------------
    // Module is not in loader-state.
    // Switch module from register-based-mode to scpi-command-based-mode.

    errStatus = mbPerformHandshake(*pInstrumentHandle);

    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (errStatus);

    // do this by writing some stuff into DIR 300
    errStatus = mbWriteDIR(
      *pInstrumentHandle,
      DIR_BLK_BYTES_VALID,
      0
      );
    
    //----------------------------------------------------------------
    // query instrument ident

    errStatus = mbSendSCPIreadDATA(
      *pInstrumentHandle,
      "*IDN?",
      idn_buf,
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (errStatus);

    idn_buf[validLength] = '\0';
    
    break;

  case VI_INTF_GPIB:

    //----------------------------------------------------------------

    //----------------------------------------------------------------
    

    //----------------------------------------------------------------
    // query instrument ident, when so requested

    errStatus = viClear(*pInstrumentHandle);

    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (VI_ERROR_FAIL_ID_QUERY);

    errStatus = delayVQueryf(
      *pInstrumentHandle,
      "*IDN?\n",
      "%t",
      idn_buf
      );

    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (VI_ERROR_FAIL_ID_QUERY);

    // Last char in idn-buf is a lineFeed, kill this.
    idn_buf[strlen(idn_buf) - 1] = 0;
    
    break;

  default:
    INIT_ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch()

  //-----------------------------------------------------------
  // Set the global variable for instrument identification
  //-----------------------------------------------------------

  thisPtr->deviceIdentity = 0;

  for (count = 0; count < HPE8311A_IDN_SUPPORTED; count++)
  {
    if (!strncmp(
        idn_buf,
        identityStcI[count].identityStcing,
        strlen(identityStcI[count].identityStcing)
      )
      )
    {
      thisPtr->deviceIdentity = identityStcI[count].identityNumber;
      break;
    }
  }

  //-----------------------------------------------------------
  // Verify the instruments ident, when so requested
  //-----------------------------------------------------------

  if (IDQuery == VI_TRUE)
  {

    DBG_PRINT("analyzing the instruments ID now!");

    //---------------------------------------------------------
    if (
      !(thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8311A)
      )
      //---------------------------------------------------------
    {
      INIT_ERR_RETURN (VI_ERROR_FAIL_ID_QUERY);
    }

    DBG_PRINT("ID of instrument conforms with driver ID!");

  }                             // END if( IDQuery == VI_TRUE )

  //----------------------------------------------------------------
  // we are still there, so it has to be the correct device, or 
  // someone told us to ignore the ID string
  //
  // reset device when requested
  if (resetDevice == VI_TRUE)
  {
    DBG_PRINT("resetting the instrument!");

    errStatus = hpe8311a_reset(*pInstrumentHandle);
  
    if (errStatus < VI_SUCCESS)
      INIT_ERR_RETURN (HPE8311A_INSTR_ERROR_RESET_FAILED);
  }

  //----------------------------------------------------------------
  // unfortunatell we have to know what kind of output is
  // installed to the instrument.
  errStatus = hpe8311a_cmdString_Q(*pInstrumentHandle, "*opt?", sizeof(opt_buf), opt_buf);

  if (errStatus < VI_SUCCESS)
    INIT_ERR_RETURN (HPE8311A_INSTR_ERROR_CANNOT_READ_OPTIONS);

  // extract the product number of channel 1 (channel 2 is always the same, or missing)
  pcOptHelper = strchr(opt_buf, ',');
  // we do not need the channel 2 info, so cut of everything starting at the comma
  if (pcOptHelper != NULL)
  {
    ViChar *stopstring; // for conversion to integer (points to an optional revision character)

    *pcOptHelper = 0x00;
    // opt_buf holds now the type of output 1 in string format
    // now we need the product number of the output
    thisPtr->outputType = (ViUInt32) strtol(opt_buf, &stopstring, 10);

    // just to keep it running...
    if (opt_buf == stopstring) // oops, no number
      thisPtr->outputType = 81111L;
  }
  else
  {
    // strange things happen... to keep the whole thing running fake a standard 
    // output (HP81111A)
    thisPtr->outputType = 81111L;
  }
  //----------------------------------------------------------------
  // now do the customization depending on the output type
  switch (thisPtr->outputType)
  {
  case 8312:
  case 81112:
    thisPtr->options.trans_unit = 0;
    break;
  default:
    thisPtr->options.trans_unit = 1;
    break;
  };

  return (statusUpdate(*pInstrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession      instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
// Closes the VISA session associated with the instrument handle and frees all
// resources associated with that session.  This typically will be the last
// call to the driver from the test program.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_close(
    ViSession instrumentHandle)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_close"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViSession defRM;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-----------------------------------------------------------
  // Retrieve Resource Management session, free global memory
  // for the session (if it exists), and close the instrumentHandle and
  // RM sessions.
  //-----------------------------------------------------------

  errStatus = viGetAttribute(instrumentHandle, VI_ATTR_RM_SESSION, &defRM);
  
  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  // cancel any current transaction...just to be sure
  transactionCancel(thisPtr);

  FREE(thisPtr);

  return (viClose(defRM));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession      instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
// Returns the instrument to a known STAte.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_reset(ViSession instrumentHandle)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_reset"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    CHECK_URBIOS;
    errStatus = mbSendSCPIandDATA(instrumentHandle, "*RST", 4);
    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            
    
    // GPIB reset includes a 488.2 device clear
    errStatus = viClear(instrumentHandle);

    if (errStatus < VI_SUCCESS)
      break;
      
    errStatus = delayVPrintf(instrumentHandle, "*RST\n");
    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession   instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt16    selfTestResult (out)
//     Self-test result
//   ViPString   selfTestMessage[] (out)
//     Self-test Status message string, limited to 256 characters.
//*****************************************************************************
// Performs a self-test of the instrument and returns the self-test result
// and a text error message.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_self_test(
    ViSession instrumentHandle,
    ViPInt16 selfTestResult,
    ViPString selfTestMessage)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_self_test"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViInt32 oldTimeOut = 0;
  ViInt8 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;
  *selfTestResult = HPE8311A_INVALID_VALUE;
  selfTestMessage[0] = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  stateMachine = STATE_MACHINE_REQUEST_TIMEOUT;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_REQUEST_TIMEOUT:

      // Query actual value of time-out
      errStatus = hpe8311a_timeOut_Q(
        instrumentHandle,
        &oldTimeOut
        );

      stateMachine = STATE_MACHINE_SET_NEW_TIMEOUT;

      break;

    case STATE_MACHINE_SET_NEW_TIMEOUT:

      // Set the new value, needed for self-test
      // since the self-test needs a long time
      errStatus = hpe8311a_timeOut(
        instrumentHandle,
        30000                   // 30 seconds
        );

      stateMachine = STATE_MACHINE_GENERAL_COMMANDS;

      break;

    case STATE_MACHINE_GENERAL_COMMANDS:

      switch (thisPtr->interfaceType)
      {
        //-------------------------------------------------------------------------
      case VI_INTF_VXI:
      case VI_INTF_GPIB_VXI:
        {
        ViChar readString[HPE8311A_STRING_LENGTH];
        ViUInt32 validLength = 0;

        errStatus = mbSendSCPIreadDATA(
          instrumentHandle,
          "*TST?",
          readString,           // selfTestResult
          HPE8311A_STRING_LENGTH - 1,
          &validLength
          );

        readString[validLength] = '\0';

        if (errStatus == VI_SUCCESS)
          *selfTestResult = (ViInt16) atoi(readString);
        }

        break;
        //-------------------------------------------------------------------------
      case VI_INTF_GPIB:
    
        errStatus = delayVQueryf(
          instrumentHandle,
          "*TST?\n",
          "%hd%*t",
          selfTestResult
          );

        break;
        //-------------------------------------------------------------------------
      default:
        ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

      }                         // END switch( thisPtr->interfaceType )

      stateMachine = STATE_MACHINE_RESTORE_OLD_TIMEOUT;

      break;

    case STATE_MACHINE_RESTORE_OLD_TIMEOUT:

      errStatus = hpe8311a_timeOut(
        instrumentHandle,
        oldTimeOut
        );

      stateMachine = STATE_MACHINE_ANALYZE_RESULTS;

      break;

    case STATE_MACHINE_ANALYZE_RESULTS:

      switch (*selfTestResult)
      {
        //---------------------------------------------------------------
      case 0:
        sprintf(selfTestMessage, HPE8311A_MSG_SELF_TEST_PASSED);
        break;
        //---------------------------------------------------------------
      default:
        sprintf(selfTestMessage, HPE8311A_MSG_SELF_TEST_FAILED);
        break;
        //---------------------------------------------------------------
      }                         // END switch( *selfTestResult)

      stateMachine = STATE_MACHINE_DO_FINISH;

      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      // Try to restore the old time out, although an error already
      // occurred - perhaps it works.
      (void) hpe8311a_timeOut(
        instrumentHandle,
        oldTimeOut
        );

      stateMachine = STATE_MACHINE_DO_FINISH;
      stateMachineErrorOccurred = VI_TRUE;

      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession    instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32     errorCode (out)
//     An instrument error code
//   ViPString    errorMessage[] (out)
//     Instrument error message string, limited to 256 characters.
//*****************************************************************************
// Queries the instrument for errors and reports the top error in the
// instrument's error queue.  This function will get the error number and the
// detailed error message from the instrument.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_error_query(
    ViSession instrumentHandle,
    ViPInt32 errorCode,
    ViPString errorMessage)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_error_query"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViChar *helpPtr;
  *errorCode = HPE8311A_INVALID_VALUE;
  errorMessage[0] = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViChar tempString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "SYST:ERR?",
      readString,               // errorCode, errorMessage
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // Terminate unterminated readString
    readString[validLength - 1] = 0;

    // Separate error-code and write it back
    helpPtr = strstr(readString, ",");

    if (helpPtr == NULL)
      ERR_RETURN (HPE8311A_INSTR_ERROR_REQUESTED_ERROR_KOMMA_MISSED);

    *helpPtr = 0;               // terminate

    strcpy(tempString, readString);
    *errorCode = atoi(tempString);

    // Separate error-message and write it back

    helpPtr++;                  // overjump char '\0'
    helpPtr++;                  // overjump char >"<

    strcpy(errorMessage, helpPtr);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "SYST:ERR?\n",
      "%ld,%t",
      errorCode,
      errorMessage
      );

    if (errStatus < VI_SUCCESS)
    {
      *errorCode = HPE8311A_INVALID_VALUE;
      errorMessage[0] = 0;

      ERR_RETURN (errStatus);
    }

    // get rid of extra LF at the end of the error_message
    errorMessage[strlen(errorMessage) - 1] = 0;

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession   instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//     May be VI_NULL for this function
//   ViStatus    statusCode (in)
//     The error ERR_RETURN value from an instrument driver function
//   ViPString   message[] (out)
//     Error message string.  This is limited to 256 characters.
//*****************************************************************************
// Returns a text string for an error number. The driver developer is
// responsible for providing a text string in this routine for any and all
// errors that are not returned by VISA. (i.e. those generated within the
// driver).
// No instrument communication is associated with this function.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_error_message(
    ViSession instrumentHandle,
    ViStatus statusCode,
    ViPString message
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_error_message"

  struct hpe8311a_globals *thisPtr = NULL;
  ViStatus errStatus;
  size_t idx;
  message[0] = 0;
  TRACE_IN();

  // Do not use macro GET_GLOBALS here
  // since it will return an error when thisPtr is NULL !

  if (instrumentHandle != VI_NULL)
  {
    errStatus = viGetAttribute(
      instrumentHandle,
      VI_ATTR_USER_DATA,
      (ViAddr) & thisPtr
      );

    if (errStatus < VI_SUCCESS)
    {
      sprintf(message, "Cannot retreive error-message - function hpe8311a_error_message() failed");
      ERR_RETURN (errStatus);
    }
  }

  if (statusCode == VI_SUCCESS)
  {
    sprintf(message, HPE8311A_MSG_NO_ERRORS);
    if(thisPtr) 
      thisPtr->errNumber = VI_SUCCESS;
    return VI_SUCCESS;
  }

  //-----------------------------------------------------------
  // Search the error message table to see if the error is in
  // that table.  If it is, copy the corresponding error
  // message to the output error message.  If context
  // dependent error messages are being used, append the
  // context dependent information to the message.
  // If the error is found in the table, exit the routine
  // successfully.
  //-----------------------------------------------------------

  for (idx = 0;
    idx < (sizeof instrErrMsgTable / sizeof(struct instrErrStruct));
    idx++
    )
  {
    if (instrErrMsgTable[idx].errStatus == statusCode)
    {
      strcpy(message, instrErrMsgTable[idx].errMessage);

      if (thisPtr != NULL)
      {
        if (thisPtr->errNumber == statusCode)
        {
          sprintf(
            message,
            "%s;  %s",
            instrErrMsgTable[idx].errMessage,
            thisPtr->errMessage
            );

          thisPtr->errNumber = VI_SUCCESS;
        }
      }

      return VI_SUCCESS;
    }
  }

  //-----------------------------------------------------------
  // Check to see if the error is a VTL/VISA error number
  // using viStatusDesc, which will also ERR_RETURN the error
  // message.  If context dependent error messages are
  // used, append the context dependent information to the
  // message.
  // If the error is found in the table, exit the routine
  // successfully.
  //-----------------------------------------------------------

  errStatus = viStatusDesc(instrumentHandle, statusCode, message);

  if (errStatus == VI_SUCCESS)
  {
    if (thisPtr != NULL)
    {
      if (thisPtr->errNumber == statusCode)
      {
        strcat(message, ";  ");
        strcat(message, thisPtr->errMessage);
      }
      thisPtr->errNumber = VI_SUCCESS;
    }

    return VI_SUCCESS;
  }

  //-----------------------------------------------------------
  // At this point instrumentHandle is either INVALID or VI_NULL
  //-----------------------------------------------------------

  if (instrumentHandle == VI_NULL)
  {
    strcpy(message, HPE8311A_MSG_VI_OPEN_ERR);
    if (thisPtr != NULL)
      thisPtr->errNumber = VI_SUCCESS;
    return VI_SUCCESS;
  }

  sprintf(message, HPE8311A_MSG_INVALID_STATUS "  %ld"
    HPE8311A_MSG_INVALID_STATUS_VALUE, (long) statusCode);

  if (thisPtr != NULL)
    thisPtr->errNumber = VI_ERROR_PARAMETER2;

  return VI_ERROR_PARAMETER2;
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession    instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPString    instrumentDriverRevision[] (out)
//     Instrument driver revision.  This is limited to 256 characters.
//   ViPString    firmwareRevision[] (out)
//     Instrument firmware revision.  This is limited to 256 characters.
//*****************************************************************************
// Queries the instrument and driver for the current revision number.  The
// instruments firmware revision is obtained from the response to the *IDN?
// command.  The driver revision is hard coded in the driver, and will STArt
// with A.00.00.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_revision_query(
    ViSession instrumentHandle,
    ViPString instrumentDriverRevision,
    ViPString firmwareRevision)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_revision_query"

  struct hpe8311a_globals *thisPtr;
  ViStatus errStatus;
  ViChar *last_comma;           // last comma in *IDN string
  ViChar temp_str1[HPE8311A_STRING_LENGTH];
  ViUInt32 validLength = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  instrumentDriverRevision[0] = 0;
  firmwareRevision[0] = 0;

  sprintf(instrumentDriverRevision, "%s", HPE8311A_REV_CODE);

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "*IDN?",
      temp_str1,
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
    {
      instrumentDriverRevision[0] = 0;
      firmwareRevision[0] = 0;
      ERR_RETURN (errStatus);
    }

    temp_str1[validLength] = '\0';

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // error and exit if last comma not found
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    last_comma = strrchr(temp_str1, ',');

    if (!last_comma)
    {
      firmwareRevision[0] = 0;
      strcpy(thisPtr->errMessage, INSTR_ERROR_NO_LAST_COMMA);
      ERR_RETURN (HPE8311A_INSTR_NO_LAST_COMMA);
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // The next STAtement assumes that the firmware
    // revision of the instrument follows the last
    // comma/space pair.
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    strcpy(firmwareRevision, last_comma + 1);

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // In the STAndard c template, the scan format
    // string is "%t%*t".  This removes white space
    // from the end of the returned string, but doesn't
    // work if there are embedded spaces in the
    // returned string.  I used this format string and
    // resorted to stripping off spaces and spaces and
    // trailing non-printing characters below.
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    errStatus = delayVQueryf(
      instrumentHandle,
      "*IDN?\n",
      "%t",
      temp_str1
      );

    // Last char in temp_str1 is a lineFeed, kill this.
    temp_str1[strlen(temp_str1) - 1] = 0;

    if (errStatus < VI_SUCCESS)
    {
      instrumentDriverRevision[0] = 0;
      firmwareRevision[0] = 0;

      ERR_RETURN (errStatus);
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // error and exit if last comma not found
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    last_comma = strrchr(temp_str1, ',');

    if (!last_comma)
    {
      firmwareRevision[0] = 0;
      strcpy(thisPtr->errMessage, INSTR_ERROR_NO_LAST_COMMA);
      ERR_RETURN (HPE8311A_INSTR_NO_LAST_COMMA);
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // The next STAtement assumes that the firmware
    // revision of the instrument follows the last
    // comma/space pair.
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    strcpy(firmwareRevision, last_comma + 1);

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//@@ STAndard HP Functions                                                   @@
//@@-------------------------------------------------------------------------@@
//@@ These STAndard HP functions must be present in all                      @@
//@@ HP VISA-based drivers.                                                  @@
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViInt32    setTimeOut (in)
//     The period of time which VISA will allow an instrument command to
//     wait before stopping it's execution.
//*****************************************************************************
// This routine is a driver internal routine.  It sets the amount of time a
// driver should wait on I/O, in milliseconds, before returning a timeout
// error.
//*****************************************************************************
// Note that the default timeout is driver dependent.  However, it
// should be set to a value that does not cause timeouts during normal
// operation.  The driver is allowed to override the default and set timeout
// in certain long routines.  Examples include self-test and long measurements.
// If these routines override this command they should be documented in the
// help file.  It may also be advisable to provide a separate timeout call for
// these functions.  If a routine overrides the STAndard timeout, it must
// restore the default timeout value before completing.
// The default timeout for HP VISA is 2 Seconds.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeOut(
    ViSession instrumentHandle,
    ViInt32 setTimeOut)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeOut"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  CHK_LONG_RANGE(
    setTimeOut,
    HPE8311A_TIMEOUT_MIN,
    HPE8311A_TIMEOUT_MAX,
    VI_ERROR_PARAMETER2
    );

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB_VXI:
    // Well, here is something to explain!
    // We recognized, that there may occure problems while
    // using viStetAttribute, dependent from the used constellation.
    // To avoid this, no viSetAttribute() is used together with GPIB-VXI
    // instruments within this driver. Instead this we use our own global
    // variable timeOut to keep the time-out-value.

    thisPtr->globalTimeOut = setTimeOut;

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:
  case VI_INTF_VXI:

    // Set global variable on requested value.
    thisPtr->globalTimeOut = setTimeOut;

    // AND set the visa-attribute.
    errStatus = viSetAttribute(
      instrumentHandle,
      VI_ATTR_TMO_VALUE,
      (ViAttrState) setTimeOut
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32   timeOut (out)
//     The period of time which VISA will allow an instrument command to
//     wait before stopping it's execution.
//*****************************************************************************
// Returns the current setting of the timeout value of the instrument in
// milliseconds.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeOut_Q(
    ViSession instrumentHandle,
    ViPInt32 timeOut)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeOut_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB_VXI:
    // Well, here is something to explain!
    // We recognized, that there may occure problems while
    // using viStetAttribute, dependent from the used constellation.
    // To avoid this, no viSetAttribute() is used together with GPIB-VXI
    // instruments within this driver. Instead this we use our own global
    // variable timeOut to keep the time-out-value.

    *timeOut = thisPtr->globalTimeOut;
    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:
  case VI_INTF_VXI:

    errStatus = viGetAttribute(
      instrumentHandle,
      VI_ATTR_TMO_VALUE,
      timeOut
      );

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViBoolean  setErrorQueryDetect (in)
//     Enables (VI_TRUE) or disables (VI_FALSE) automatic instrument error
//     querying
//*****************************************************************************
// This routine is an internal driver routine.  It does not perform I/O with
// the instrument.  Instead it sets a flag, detectError which, if true,
// instructs a centralized error handling routine in the driver to
// non-destructively query the instrument for the presence of errors.
// The method of querying the instrument is somewhat instrument dependent.
// In IEEE 488.2 compliant instruments this is accomplished by querying the
// instrument's event Status register and examining the four bits relating to
// instrument errors.  However, it must be noted that the event Status register
// is read destructive.  Other bits in the register which may matter in the
// driver's operation, such as operation complete should be preserved until
// the next explicit query of the register, or until a *CLS command is
// received.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_errorQueryDetect(
    ViSession instrumentHandle,
    ViBoolean setErrorQueryDetect)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_errorQueryDetect"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  if (chk_boolean(thisPtr, setErrorQueryDetect))
    ERR_RETURN (VI_ERROR_PARAMETER2);

  thisPtr->errQueryDetect = setErrorQueryDetect;
  return (VI_SUCCESS);
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession   instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPBoolean  errorQueryDetect (out)
//     Equals VI_TRUE if automatic instrument error querying is being
//     being performed.
//*****************************************************************************
// Queries the error query detect flag.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_errorQueryDetect_Q(
    ViSession instrumentHandle,
    ViPBoolean errorQueryDetect)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_errorQueryDetect_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *errorQueryDetect = thisPtr->errQueryDetect;
  return (VI_SUCCESS);
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
// Send a device clear message to the instrument, which, in turn clears all
// input and output buffers, clears any pending operations, and places the
// instrument parser into a STAte where it is ready to accept input. This
// routine does not generate any unique error conditions beyond those returned
// by VISA (i.e. timeout, query interrupted, etc.).
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_dcl(ViSession instrumentHandle)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_dcl"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    // The VXI based instruments don't really support the dcl command, but
    // for comapatibility with the other family members we are patient and
    // don't generate an error.
    DBG_PRINT("not supported for VXI-instruments!");
    return VI_SUCCESS;
    
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = viClear(instrumentHandle);
    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}

//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//@@ Status Functions                                                        @@
//@@-------------------------------------------------------------------------@@
//@@ The Status routines are HP recommended functions and should be part of  @@
//@@ all HP VISA-based drivers if the commands is supported.  (*OPC? and     @@
//@@ *STB? should be supported on most instruments.)  These functions        @@
//@@ provide a STAndard and easy method of providing Status polling.         @@
//@@-------------------------------------------------------------------------@@
//@@ Note that callbacks (interrupts) are not implemented in the             @@
//@@ STAndard set.                                                           @@
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession   instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViPBoolean  instrumentReady (out)
//     Returns VI_TRUE when pending operations are complete.
//*****************************************************************************
// Sends an *OPC? command to the instrument and returns VI_TRUE when all
// pending operations are complete.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_opc_Q(
    ViSession instrumentHandle,
    ViPBoolean instrumentReady)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_opc_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt32 requestedValue;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  thisPtr->dTime = 200;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "*OPC?",
      readString,               // instrumentReady
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    requestedValue = atoi(readString);

    if (requestedValue == 0)
      *instrumentReady = VI_FALSE;
    else
    {
      if (requestedValue == 1)
        *instrumentReady = VI_TRUE;
      else
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "*OPC?\n"
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = delayVScanf(
      instrumentHandle,
      "%hd%*t",
      instrumentReady
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt16  STAtusByte (out)
//     Returns the contents of the Status byte
//*****************************************************************************
// Returns the instrument's primary Status byte.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_readStatusByte_Q(
    ViSession instrumentHandle,
    ViPInt16 STAtusByte)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_readStatusByte_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViUInt16 stb;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "*STB?",                  // SCPI
      readString,               // STAtusByte
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *STAtusByte = (ViInt16) atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = viReadSTB(
      instrumentHandle,
      &stb
      );

    if (errStatus < VI_SUCCESS)
      break;

    *STAtusByte = (ViInt16) stb;

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession vi (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32  operationEventRegister (out)
//     Returns the contents of the operation event register
//*****************************************************************************
// Returns the contents of the instrument's operation event Status register.
// This routine is only required for IEEE 488.2 instruments.  Note that if the
// user makes this call, automatic error detection (_errorQueryDetect_Q)
// may miss an instrument error or other event that is automatically handled
// in the driver.  Some instruments use non-STAndard forms of the SCPI command
// which look similar to the STAndard form.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_operEvent_Q(
    ViSession instrumentHandle,
    ViPInt32 operationEventRegister)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_operEvent_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "STAT:OPER:EVEN?",        // scpi
      readString,               // operationEventRegister
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *operationEventRegister = atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "STAT:OPER:EVEN?\n",
      "%ld%*t",
      operationEventRegister
      );

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);
  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32  operationConditionRegister (out)
//     Returns the contents of the operation condition register
//*****************************************************************************
// Returns the contents of the instrument's operation condition register.
// This routine is only required for IEEE 488.2 instruments. Some instruments
// use non-STAndard forms of the SCPI command which look similar to the
// STAndard form.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_operCond_Q(
    ViSession instrumentHandle,
    ViPInt32 operationConditionRegister)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_operCond_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "STAT:OPER:COND?",        // scpi
      readString,               // operationConditionRegister
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *operationConditionRegister = atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "STAT:OPER:COND?\n",
      "%ld%*t",
      operationConditionRegister
      );

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32  questionableEventRegister (out)
//     Returns the contents of the questionable event register
//*****************************************************************************
// Returns the contents of the instrument's data questionable Status register.
// This routine is only required for IEEE 488.2 instruments. Some instruments
// use non-STAndard forms of the SCPI command which look similar to the
// STAndard form.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_quesEvent_Q(
    ViSession instrumentHandle,
    ViPInt32 questionableEventRegister)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_quesEvent_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "STAT:QUES:EVEN?",        // scpi
      readString,               // questionableEventRegister
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *questionableEventRegister = atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "STAT:QUES:EVEN?\n",
      "%ld%*t",
      questionableEventRegister
      );

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession instrumentHandle (in)
//     Instrument Handle returned from hpe8311a_init()
//   ViPInt32  questionableConditionRegister (out)
//     Returns the contents of the questionable condition register
//*****************************************************************************
// Returns the contents of the instrument's data questionable condition
// register.  This routine is only required for IEEE 488.2 instruments.
// Some instruments use non-STAndard forms of the SCPI command which look
// similar to the STAndard form.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_quesCond_Q(
    ViSession instrumentHandle,
    ViPInt32 questionableConditionRegister)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_quesCond_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      "STAT:QUES:COND?",        // scpi
      readString,               // questionableConditionRegister
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *questionableConditionRegister = atoi(readString);
    }

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "STAT:QUES:COND?\n",
      "%ld%*t",
      questionableConditionRegister
      );

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//@@ PASSTHROUGH FUNCTIONS                                                   @@
//@@-------------------------------------------------------------------------@@
//@@ These functions are called "passthrough" functions because they enable  @@
//@@ the user to pass the instrument's commands through to the instrument,   @@
//@@ and to receive results back from the instrument.                        @@
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession     instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString      sendStringCommand (in)
//     The SCPI command string to be sent to the instrument
//*****************************************************************************
// This routine allows the user to send an arbitrary command string.
// The only parameter to this function will be the command string, which is
// limited to 256 characters. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdWithControlSequence(
    ViSession instrumentHandle,
    ViString sendStringCommand
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdWithControlSequence"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendStringCommand,
      strlen(sendStringCommand)
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s\n",
      sendStringCommand
      );

    break;
    
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession     instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString      sendStringCommand (in)
//     The SCPI command string to be sent to the instrument
//*****************************************************************************
// This routine allows the user to send an arbitrary command string.
// The only parameter to this function will be the command string, which is
// limited to 256 characters. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmd(
    ViSession instrumentHandle,
    ViString sendStringCommand
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmd"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViUInt32 sendSize = 0;
  ViChar *sendStringCommandPtr = NULL;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      sendStringCommand,
      &sendStringCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(&sendStringCommandPtr);

    if (errStatus < VI_SUCCESS || NULL == sendStringCommandPtr)
      break;

    sendSize = strlen(sendStringCommandPtr);

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendStringCommandPtr,
      sendSize
      );

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(instrumentHandle, "%s\n", sendStringCommand);
    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // switch( thisPtr->interfaceType )

  FREE(sendStringCommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession     instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString      sendStringCommand (in)
//     The SCPI command string to be sent to the instrument
//   ViString      sendString (in)
//     String-Parameter to be sent to the instrument
//*****************************************************************************
// This routine allows the user to send an arbitrary command string.
// The only parameter to this function will be the command string, which is
// limited to 256 characters. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdString(
    ViSession instrumentHandle,
    ViString sendStringCommand,
    ViString sendString
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdString"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *sendStringCommandPtr = NULL;
  ViChar *sendStringPtr = NULL;
  ViChar *sendParameterPtr = NULL;
  ViUInt32 sendSize = 0;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Preparations for Parameter sendStringCommand

    errStatus = mbGetOnHeapString(
      sendStringCommand,
      &sendStringCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendStringCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Preparations for Parameter sendString

    errStatus = mbGetOnHeapString(
      sendString,
      &sendStringPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendStringPtr
      );

    if (errStatus < VI_SUCCESS
      || NULL == sendStringCommandPtr
      || NULL == sendStringPtr
      )
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Connecting Parameters...mallocs space on heap

    sendSize =
      strlen(sendStringCommandPtr)
      + strlen(sendStringPtr)
      + 1;                      // between sendStringCommand, sendString:
    // blank

    sendParameterPtr = (ViChar *) malloc(sendSize + 1);
    // Remember the final '\0' or free will fail!

    if (sendParameterPtr == NULL)
    {
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    strcpy(sendParameterPtr, sendStringCommandPtr);
    strcat(sendParameterPtr, " ");
    strcat(sendParameterPtr, sendStringPtr);

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendParameterPtr,
      sendSize
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s %s\n",
      sendStringCommand,
      sendString
      );

    break;
    
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(sendStringCommandPtr);
  FREE(sendStringPtr);
  FREE(sendParameterPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession       instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString        queryStringCommand (in)
//     The SCPI command string to be sent to the instrument
//   ViInt32         stringSize (in)
//     The size of the char array (result) passed to the function to hold
//     the string returned by the instrument
//   ViChar _VI_FAR  stringResult[] (out)
//     The string returned by the instrument
//*****************************************************************************
// This routine allows the user to send an arbitrary command string, and
// returns a single string parameter. The returned string will be null
// terminated. The command string is limited to 256 characters.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdString_Q(
    ViSession instrumentHandle,
    ViString queryStringCommand,
    ViInt32 stringSize,
    ViChar stringResult[]
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdString_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  int sz;
  ViUInt32 validLength = 0;
  ViChar *queryStringCommandPtr = NULL;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // Command strings must have at least one non-null character
  if (stringSize < 2)
    ERR_RETURN (VI_ERROR_PARAMETER2);

  sz = (int) stringSize;
  stringResult[0] = '\0';

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      queryStringCommand,
      &queryStringCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryStringCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryStringCommandPtr,    // scpi
      stringResult,             // stringResult
      (ViUInt32) stringSize,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    // Terminate the data to make ready for strings.
    // But only do this in the known memory space.
    if ((ViInt32) validLength < stringSize)
      stringResult[validLength] = '\0';

    break;
    
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%#t",
      queryStringCommand,
      &sz,
      stringResult
      );

    if (errStatus < VI_SUCCESS)
      break;

    // There may be a line-feed at the end of the string
    stringResult[sz - 1] = '\0';

    break;
    
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryStringCommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   sendIntegerCommand (in)
//     The instrument command string to be sent to the instrument.
//   ViPInt32   sendInteger (in)
//     The integer sent to the instrument at the end of the instrument
//     command.  Can be ViInt16 or ViInt32.
//*****************************************************************************
// This routine allows the user to send an arbitrary command string, followed
// by a single integer parameter.  A white space character is sent between the
// command string and the parameter. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdInt(
    ViSession instrumentHandle,
    ViString sendIntegerCommand,
    ViInt32 sendInteger
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdInt"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar sendIntegerString[HPE8311A_STRING_LENGTH];
  ViChar *sendParameterPtr = NULL;
  ViChar *sendIntegerCommandPtr = NULL;
  ViUInt32 sendSize = 0;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;

    errStatus = mbGetOnHeapString(
      sendIntegerCommand,
      &sendIntegerCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendIntegerCommandPtr
      );

    if (errStatus < VI_SUCCESS || NULL == sendIntegerCommandPtr)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Preparations for Parameter sendIntegerCommand

    // convert the int-value to a string
    sprintf(sendIntegerString, "%d", sendInteger);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Connecting Parameters...malloc space on heap

    // calculate the quantity of bytes to send
    sendSize = strlen(sendIntegerCommandPtr)
      + 1                       // space
      + strlen(sendIntegerString);

    sendParameterPtr = (ViChar *) malloc(sendSize + 1);
    // Remember the final '\0' or free will fail!

    if (sendParameterPtr == NULL)
    {
      // malloc did not work
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // insert the scpi-command
    strcpy(sendParameterPtr, sendIntegerCommandPtr);
    // insert a space
    strcat(sendParameterPtr, " ");
    // insert the converted value
    strcat(sendParameterPtr, sendIntegerString);

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendParameterPtr,
      sendSize
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s %ld\n",
      sendIntegerCommand,
      sendInteger
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(sendIntegerCommandPtr);
  FREE(sendParameterPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryI16Command (in)
//     The SCPI command string to be sent to the instrument.
//   ViPInt16   i16Result (out)
//     The integer returned from the instrument.
//*****************************************************************************
// Sends scpi command and waits for a response that must be representable
//   as an ViInt16.  A non numeric instrument response returns zero in
//   result.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdInt16_Q(
    ViSession instrumentHandle,
    ViString queryI16Command,
    ViPInt16 i16Result
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdInt16_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *queryI16CommandPtr = NULL;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *i16Result = 0;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;
    CHECK_URBIOS;

    errStatus = mbGetOnHeapString(
      queryI16Command,
      &queryI16CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryI16CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryI16CommandPtr,       // scpi
      readString,               // i16Result
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    errStatus = convertViStringToInt16(i16Result, readString);
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%hd%*t",
      queryI16Command,
      i16Result
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryI16CommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryI32Command (in)
//     The SCPI command string to be sent to the instrument.
//   ViPInt32   i32Result (out)
//     The integer returned from the instrument.
//*****************************************************************************
// Sends scpi command and waits for a response that must be representable
//   as an ViInt32.  A non numeric instrument response returns zero in
//   result.
// Non-integer numbers will be rounded.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdInt32_Q(
    ViSession instrumentHandle,
    ViString queryI32Command,
    ViPInt32 i32Result
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdInt32_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *queryI32CommandPtr = NULL;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *i32Result = 0;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      queryI32Command,
      &queryI32CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryI32CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryI32CommandPtr,       // scpi
      readString,               // i32Result
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    errStatus = convertViStringToInt32(i32Result, readString);
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%ld%*t",
      queryI32Command,
      i32Result
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryI32CommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   sendUIntegerCommand (in)
//     The instrument command string to be sent to the instrument.
//   ViPUInt32   sendUInteger (in)
//     The integer sent to the instrument at the end of the instrument
//     command.  Can be ViUInt16 or ViUInt32.
//*****************************************************************************
// This routine allows the user to send an arbitrary command string, followed
// by a single integer parameter.  A white space character is sent between the
// command string and the parameter. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdUInt(
    ViSession instrumentHandle,
    ViString sendUIntegerCommand,
    ViUInt32 sendUInteger
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdUInt"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar sendUIntegerString[HPE8311A_STRING_LENGTH];
  ViChar *sendParameterPtr = NULL;
  ViChar *sendUIntegerCommandPtr = NULL;
  ViUInt32 sendSize = 0;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      sendUIntegerCommand,
      &sendUIntegerCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendUIntegerCommandPtr
      );

    if (errStatus < VI_SUCCESS || NULL == sendUIntegerCommandPtr)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Preparations for Parameter sendUIntegerCommand

    // convert the int-value to a string
    sprintf(sendUIntegerString, "%d", sendUInteger);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Connecting Parameters...malloc space on heap

    // calculate the quantity of bytes to send
    sendSize = strlen(sendUIntegerCommandPtr)
      + 1                       // space
      + strlen(sendUIntegerString);

    sendParameterPtr = (ViChar *) malloc(sendSize + 1);
    // Remember the final '\0' or free will fail!

    if (sendParameterPtr == NULL)
    {
      // malloc did not work
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // insert the scpi-command
    strcpy(sendParameterPtr, sendUIntegerCommandPtr);
    // insert a space
    strcat(sendParameterPtr, " ");
    // insert the converted value
    strcat(sendParameterPtr, sendUIntegerString);

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendParameterPtr,
      sendSize
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s %ld\n",
      sendUIntegerCommand,
      sendUInteger
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(sendUIntegerCommandPtr);
  FREE(sendParameterPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryUI16Command (in)
//     The SCPI command string to be sent to the instrument.
//   ViPUInt16  ui16Result (out)
//     The unsignedinteger returned from the instrument.
//*****************************************************************************
// Sends scpi command and waits for a response that must be representable
//   as an ViUInt16.  A non numeric instrument response returns zero in
//   result.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdUInt16_Q(
    ViSession instrumentHandle,
    ViString queryUI16Command,
    ViPUInt16 ui16Result
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdUInt16_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *queryUI16CommandPtr = NULL;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *ui16Result = 0;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      queryUI16Command,
      &queryUI16CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryUI16CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryUI16CommandPtr,      // scpi
      readString,               // ui16Result
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    errStatus = convertViStringToUInt16(ui16Result, readString);
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%hd%*t",
      queryUI16Command,
      ui16Result
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryUI16CommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryUI32Command (in)
//     The SCPI command string to be sent to the instrument.
//   ViPUInt32  ui32Result (out)
//     The unsigned integer returned from the instrument.
//*****************************************************************************
// Sends scpi command and waits for a response that must be representable
//   as an ViUInt32.  A non numeric instrument response returns zero in
//   result.
// Non-integer numbers will be rounded.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdUInt32_Q(
    ViSession instrumentHandle,
    ViString queryUI32Command,
    ViPUInt32 ui32Result
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdUInt32_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *queryUI32CommandPtr = NULL;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *ui32Result = 0;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      queryUI32Command,
      &queryUI32CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryUI32CommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryUI32CommandPtr,      // scpi
      readString,               // ui32Result
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    errStatus = convertViStringToUInt32(ui32Result, readString);
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%ld%*t",
      queryUI32Command,
      ui32Result
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryUI32CommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
// DESCRIPTION OF THE DATA to send to the instrument:
//---------------------------------------------------------------------------
//    -  sendI8ArrayCommand
//       the scpi-command itself
//    -  ' '
//       just a space to sign where scpi-command ends
//    -  '#'
//       just a char to sign where the data begin
//    -  lengthByte
//       this is one char, used to describe the quantity of bytes
//       used in the following sendI8ArraySizeString
//    -  sendI8ArraySizeString
//       this is a string which contains the number of valid bytes
//       contained in the following sendI8Array
//    -  sendI8Array
//       This are the data given to this function. They are transported
//       without transforming to the instrument as a chain of bytes.
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   sendI8ArrayCommand (in)
//     The instrument command string to be sent to the instrument.
//   ViPChar i8Array (in)
//     Data to send to the instrument
//   ViInt32 i8ArraySize (in)
//     Number of Elements of the Array to send to the instrument.
//*****************************************************************************
// Sends an instrument command which takes the contents of an 8bit-integer
// array to the instrument. No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdInt8Arr(
    ViSession instrumentHandle,
    ViString sendI8ArrayCommand,
    ViPChar sendI8Array,
    ViInt32 sendI8ArraySize
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdInt8Arr"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *sendParameterPtr = NULL;
  ViChar *sendI8ArrayCommandPtr = NULL;
  ViChar *sendI8ArrayPtr = NULL;
  ViChar *helpDataPtr = NULL;
  ViUInt32 sendI8ArrayCommandSize = 0;
  ViUInt32 sendSize = 0;
  ViUInt32 lengthByte = 0;
  char lengthByteString[2];
  char sendI8ArraySizeString[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      sendI8ArrayCommand,
      &sendI8ArrayCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendI8ArrayCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Preparations for Parameter sendI8Array

    sprintf(sendI8ArraySizeString, "%d", sendI8ArraySize);
    // Do this, before ESC is inserted into the array,
    // since the ESC and the control-sequence represent one character.

    errStatus = mbGetOnHeapArray(
      sendI8Array,
      sendI8ArraySize,
      &sendI8ArrayPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesArray(
      &sendI8ArrayPtr,
      (ViUInt32) sendI8ArraySize,
      (ViUInt32 *) (&sendI8ArraySize)
      );

    if (errStatus < VI_SUCCESS
      || NULL == sendI8ArrayCommandPtr
      || NULL == sendI8ArrayPtr
      )
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Connecting Parameters

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // GET ENOUGH SPACE ON HEAP FOR sendStrPtr:
    // First calculate the quantity of bytes used to
    // send the instrument the sendI8ArraySizeString.
    lengthByte = strlen(sendI8ArraySizeString);

    // And now calculate the amount of space to malloc:
    // sendSize =
    // the length of the sendCommand
    // + 1 for ' '
    // + 1 for '#'
    // + 1 for the lengthByte
    // + the length of the sendI8ArraySizeString
    // + the length of sendI8Array == sendI8ArraySize
    sendI8ArrayCommandSize = strlen(sendI8ArrayCommandPtr);
    sendSize = 
      (sendI8ArrayCommandSize + 3 + lengthByte + (ViUInt32)sendI8ArraySize);

    // The input-buffer has only got 30*1024 byte
    // ...function does not accept larger buffers.
    // Remember to the final '\n'!
    if (sendSize + 1 > 30 * 1024)
    {
      errStatus = VI_ERROR_PARAMETER4;
      break;
    }

    sendParameterPtr = (ViChar *) malloc(sendSize + 1);
    // Remember the final '\0' or free will fail!

    if (sendParameterPtr == NULL)
    {
      // malloc did not work
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // CONSTRUCT THE DATA TO SEND
    // insert the scpi-command
    strcpy(sendParameterPtr, sendI8ArrayCommandPtr);
    // append ' ' and '#'
    strcat(sendParameterPtr, " #");
    // insert the lengthByte
    sprintf(lengthByteString, "%d", lengthByte);
    strcat(sendParameterPtr, lengthByteString);
    // insert the sendI8ArraySizeString
    strcat(sendParameterPtr, sendI8ArraySizeString);
    // Insert now all the data from the sendI8Array
    // ( using memcpy, maybe '0' in the array )
    helpDataPtr = sendParameterPtr
      + sendI8ArrayCommandSize
      + 3
      + lengthByte;
    memcpy(helpDataPtr, sendI8ArrayPtr, (size_t) sendI8ArraySize);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendParameterPtr,
      sendSize
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s %*b\n",
      sendI8ArrayCommand,
      sendI8ArraySize,
      sendI8Array
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  // free the space
  FREE(sendI8ArrayCommandPtr);
  FREE(sendI8ArrayPtr);
  FREE(sendParameterPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryI8ArrayCommand (in)
//     The instrument command string to be sent to the instrument.
//   ViInt32    i8ArraySize (in)
//     The number of elements allocated to the results array.
//   ViPChar   _VI_FAR i8ArrayResult (out)
//     Array of 8-bit integers - result from instrument.
//   ViPInt32   i8ArrayCount (out)
//     The actual number of elements returned by the instrument
//*****************************************************************************
// The function assumes that the data returned by the instrument has
//   four components.
//   1 - An initial "#" taking one byte
//   2 - A single ASCII digit indicating the size in bytes of a number
//       which follows immediately.
//   3 - A string number indicating the number of array elements to be
//       returned by the instrument.
//   4 - The arbitrary block being returned by the instrument.  In this
//       case it is an array of 8-bit integers.
//       Only this block is returned in the i8ArrayResult.
//*****************************************************************************
// This entry point assumes that the ERR_RETURN result is a arbitrary block.
// Do not use this entry point for ascii values.  For that just get the
// data from _cmd_string_Q routine.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdInt8Arr_Q(
    ViSession instrumentHandle,
    ViString queryI8ArrayCommand,
    ViInt32 i8ArraySize,
    ViPChar i8ArrayResult,
    ViPInt32 i8ArrayCount
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdInt8Arr_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  // variables for reading gpib
  ViUInt32 cnt;
  ViChar lc[100];
  ViInt32 ArrSize;
  ViInt32 nbytes;
  ViChar _huge *Array = NULL;
  ViChar *queryI8ArrayCommandPtr = NULL;
  ViChar *readString = NULL;
  ViChar *analyzePtr = NULL;
  ViUInt32 readBytes = 0;
  ViUInt8 lengthByte = 0;
  ViChar dataSizeString[HPE8311A_STRING_LENGTH];
  ViUInt32 dataSize = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // check
  CHK_LONG_RANGE(i8ArraySize,
    HPE8311A_CMDINT8ARR_Q_MIN,
    HPE8311A_CMDINT8ARR_Q_MAX,
    VI_ERROR_PARAMETER3
    );

  *i8ArrayCount = 0;
  i8ArrayResult[0] = '\0';

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      queryI8ArrayCommand,
      &queryI8ArrayCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryI8ArrayCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    // check if provided array has more fields than the
    // instrument in the output-buffer...malloc as req'd
    CHK_LONG_RANGE(
      i8ArraySize,
      0L,
      60 * 1024L,
      VI_ERROR_PARAMETER3
      );

    // malloc space as big as output-buffer
    readString = (ViPChar) malloc(60 * 1024 + 2 + 1);
    // Remember the final '\0' or free will fail!

    if (readString == NULL)
    {
      // malloc did not work
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    // read data from instrument
    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryI8ArrayCommand,      // scpi
      readString,               // i8ArrayResult
      60 * 1024,
      &readBytes
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    // First reference to the data.
    analyzePtr = readString;

    if (analyzePtr[0] != '#')
    {
      errStatus = HPE8311A_INSTR_ERROR_POUND_CHAR_NOT_IN_INSTRUMENT_ANSWER;
      break;
    }

    // First byte is done.
    analyzePtr++;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // The second Byte from instruments answer..
    // describes length of following string-number

    lengthByte = (ViUInt8) analyzePtr[0];
    lengthByte -= 0x30;

    if (lengthByte > 9)
    {
      errStatus = HPE8311A_INSTR_ERROR_LENGTH_BYTE_CONTAINS_INVALID_VALUE;
      break;
    }

    // Second byte is done.
    analyzePtr++;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // This string-number indicates the number of arrays
    // that were read from the instrument

    // Get this string number into a string-variable.
    memcpy(
      dataSizeString,
      analyzePtr,
      (size_t) lengthByte * sizeof(ViInt8)
      );

    // Terminate this new string-variable.
    dataSizeString[lengthByte] = '\0';

    // And convert it to a integer.
    dataSize = (ViUInt32) (unsigned int) atoi(dataSizeString);

    if (dataSize > (ViUInt32) i8ArraySize)
    {
      errStatus = HPE8311A_INSTR_ERROR_OUTPUT_DATA_GREATER_READ_BUFFER;
      break;
    }

    // Now length bytes are done.
    analyzePtr += lengthByte;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Perform check and free memory

    // Now copy this quantity of data into the users
    // memory. But only as many as the user allows.
    memcpy(
      i8ArrayResult,
      analyzePtr,
      (size_t) dataSize * sizeof(ViInt8)
      );

    *i8ArrayCount = (ViInt32)dataSize;

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s\n",
      queryI8ArrayCommand
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Read first byte of instruments answer

    errStatus = delayRead(
      instrumentHandle,
      (ViPBuf) lc,
      1,
      &cnt
      );

    if (errStatus < VI_SUCCESS)
      break;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // First Byte has to be an #

    if (lc[0] != '#')
    {
      errStatus = HPE8311A_INSTR_ERROR_POUND_CHAR_NOT_IN_INSTRUMENT_ANSWER;
      break;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Read second Byte from instruments answer...
    // ...this describes length of following string-number

    errStatus = delayRead(
      instrumentHandle,
      (ViPBuf) lc,
      1,
      &cnt
      );

    if (errStatus < VI_SUCCESS)
      break;

    // convert ascii to hex
    nbytes = lc[0] - '0';

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Read string-number from instruments answer...
    // ...this string-number indicates the number of array
    // elements to be read from the instrument!

    if (nbytes)
    {
      errStatus = delayRead(
        instrumentHandle,
        (ViPBuf) lc,
        (ViUInt32) nbytes,
        &cnt);

      if (errStatus < VI_SUCCESS)
        break;

      lc[cnt] = '\0';           // terminate end of data
      ArrSize = atol(lc);       // convert ascii to long
    }
    else
      ArrSize = i8ArraySize * (ViInt32)sizeof(ViInt8); // to malloc max. space

    // mallocate enough space for the instrument data on the heap
    // ----------------
    Array = (char *) malloc((size_t) ArrSize);
    if (Array == NULL)          // malloc did not work
    {
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Read significant block-data from the instrument!\n" );

    errStatus = delayRead(
      instrumentHandle,
      (ViPBuf) Array,
      (ViUInt32) ArrSize,
      &cnt);

    if (errStatus < VI_SUCCESS) 
      break;

    if (errStatus > VI_SUCCESS) 
      errStatus = viClear(instrumentHandle);

    ArrSize = (ViInt32) (cnt / sizeof(ViInt8));
    ArrSize = (ArrSize > i8ArraySize) ? i8ArraySize : ArrSize;

    // copy only as many as the user allows
    memcpy(i8ArrayResult, Array, (size_t) ArrSize * sizeof(ViInt8));

    *i8ArrayCount = ArrSize;

    break;

    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch()

  FREE(queryI8ArrayCommandPtr);
  FREE(readString);
  FREE(Array);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   sendRealCommand (in)
//     The instrument command string to be sent to the instrument.
//   ViReal64   sendReal (in)
//     The real number sent to the instrument at the end of the instrument
//     command.  Can be ViReal32 or ViReal64.
//*****************************************************************************
// This routine allows the user to send an arbitrary command string,
// followed by a single floating point parameter.
// A white space character is sent between the command string and parameter.
// No result from Instrument is returned.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdReal(
    ViSession instrumentHandle,
    ViString sendRealCommand,
    ViReal64 sendReal64
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdReal"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar sendRealString[HPE8311A_STRING_LENGTH];
  ViChar *sendParameterPtr = NULL;
  ViChar *sendRealCommandPtr = NULL;
  ViUInt32 sendSize = 0;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:

    CHECK_URBIOS;
    errStatus = mbGetOnHeapString(
      sendRealCommand,
      &sendRealCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &sendRealCommandPtr
      );

    if (errStatus < VI_SUCCESS || NULL == sendRealCommandPtr)
      break;

    // convert the float-value to a string
    mbFloatToString(sendReal64, sendRealString);

    // Connecting Parameters
    sendSize =
      strlen(sendRealCommandPtr)
      + 1                       // space
      + strlen(sendRealString);

    sendParameterPtr = (ViChar *) malloc(sendSize + 1);
    // Remember the final '\0' or free will fail!

    if (sendParameterPtr == NULL)
    {
      errStatus = VI_ERROR_ALLOC;
      break;
    }

    // insert the scpi-command
    strcpy(sendParameterPtr, sendRealCommand);
    // insert a space
    strcat(sendParameterPtr, " ");
    // insert the converted value
    strcat(sendParameterPtr, sendRealString);

    errStatus = mbSendSCPIandDATA(
      instrumentHandle,
      sendParameterPtr,
      sendSize
      );

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVPrintf(
      instrumentHandle,
      "%s %g\n",
      sendRealCommand,
      sendReal64
      );

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(sendRealCommandPtr);
  FREE(sendParameterPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//*****************************************************************************
//*****************************************************************************
// PARAMETERS
//   ViSession  instrumentHandle (in)
//     Instrument handle returned from hpe8311a_init()
//   ViString   queryRealCommand (in)
//     The SCPI command string to be sent to the instrument.
//   ViPReal64  realResult (out)
//     The real returned from the instrument.
//*****************************************************************************
// Sends scpi command and waits for a response that must be representable
//   as an real64.  A non numeric instrument response returns zero in
//   result, in case of underflow undefined.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_cmdReal64_Q(
    ViSession instrumentHandle,
    ViString queryRealCommand,
    ViPReal64 realResult
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_cmdReal64_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar *queryRealCommandPtr = NULL;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  *realResult = 0;

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:
    {
    ViChar readString[HPE8311A_STRING_LENGTH];
    ViUInt32 validLength = 0;

    CHECK_URBIOS;

    errStatus = mbGetOnHeapString(
      queryRealCommand,
      &queryRealCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = insertEscapesString(
      &queryRealCommandPtr
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = mbSendSCPIreadDATA(
      instrumentHandle,
      queryRealCommandPtr,      // scpi
      readString,               // realResult
      HPE8311A_STRING_LENGTH - 1,
      &validLength
      );

    if (errStatus < VI_SUCCESS)
      break;

    readString[validLength] = '\0';
    *realResult = atof(readString);
    }

    break;
    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:

    errStatus = delayVQueryf(
      instrumentHandle,
      "%s\n",
      "%lg%*t",
      queryRealCommand,
      realResult);

    break;
    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }

  FREE(queryRealCommandPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//@@ SUBSYSTEM FUNCTIONS                                                     @@
//@@ Essential set of related capabilities                                   @@
//@@-------------------------------------------------------------------------@@
//@@ CATEGORIES:                                                             @@
//@@  * MODE-FUNCTIONS                                                       @@
//@@    continuous - mode                                                    @@
//@@    triggered - mode                                                     @@
//@@    gated - mode                                                         @@
//@@    external witdth - mode                                               @@
//@@  * TIME-FUNCTIONS                                                       @@
//@@  * LEVEL-FUNCTIONS                                                      @@
//@@  * PATTERN-FUNCTIONS                                                    @@
//@@  * LIMIT-FUNCTIONS                                                      @@
//@@  * CONFIG-FUNCTIONS                                                     @@
//@@  * OUTPUT-FUNCTIONS                                                     @@
//@@  * NOT-GROUPABLE-FUNCTIONS                                              @@
//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// MODE-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// General used
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_modePulseType()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - General used
//---------------------------------------------------------------------------
// This function selects, whether the provided channel generates
//   - a single pulse per period or
//   - a double pulse per period.
//---------------------------------------------------------------------------
// To put the instrument completely to a desired puls-mode or burst-mode use
//   also the functions:
//   - modeContinuousPulses() or
//     modeContinuousBurst() or
//   - modeTriggeredPulses() or
//     modeTriggeredBurst() or
//   - modeGatedPulses() or
//     modeGatedBurst()
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeDoubleDelay() to adjust the delay between
//   the leading edges of the first and second pulse.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DOUBle[1|2][:STATe] OFF
//   [:SOURce]:PULSe:DOUBle[1|2][:STATe] ON
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | pulseType                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select pulse-type of selected channel:
//  |  | Range:
//  |  |  - HPE8311A_SINGLE_PULSES
//  |  |  - HPE8311A_DOUBLE_PULSES
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modePulseType(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 pulseType
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modePulseType"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // Only perform function when modules bios-state is in special-bios
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (pulseType)
  {

  case HPE8311A_SINGLE_PULSES:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_DOUBLE_PULSES:
    strcpy(scpiParameter, "ON");
    break;

  default:
    DBG_PRINT("failed: wrong parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB1",
      scpiParameter
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB2",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB1",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB2",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modePulseType_Q()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - General used
//---------------------------------------------------------------------------
// Use this function to find out whether the puls-type of the selected channel
//   is single-pulses or double pulses.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:DOUBle[1|2][:STATe]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | pulseType                                            ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Requests the actual pulse-type of selected channel:
//  |  | Range:
//  |  |  - HPE8311A_SINGLE_PULSES
//  |  |  - HPE8311A_DOUBLE_PULSES
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modePulseType_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 pulseType
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modePulseType_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // Only perform function when modules bios-state is in special-bios
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DOUB1?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DOUB2?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "0"))
    *pulseType = HPE8311A_SINGLE_PULSES;
  else
  {
    if (!strcmp(requestedString, "1"))
      *pulseType = HPE8311A_DOUBLE_PULSES;
    else
    {
      // the value got from the instrument is not within the allowed range
      *pulseType = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modePatternDataFormat()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - General used
//---------------------------------------------------------------------------
// This function selects, whether the provided channel works
//   with the data-format:
//   - HPE8311A_NRZ (no ERR_RETURN to zero):
//     a leading-edge is generated for a 0->1 data transition,
//     a trailing-edge is generated for a 1->0 data transition.
//   - HPE8311A_RZ (ERR_RETURN to zero):
//     A single pulse is generated in each pulse-period with pattern-data
//     TRUE, no single pulse is generated in each pulse-period with
//     pattern-data FALSE.
//---------------------------------------------------------------------------
// To put the instrument completely to a desired pattern-mode use also
//   the functions:
//   - modeContinuousPattern() or
//   - modeTriggeredPattern() or
//   - modeGatedPattern()
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :DIGital[:STIMulus]:SIGNal[1|2]:FORMat RZ
//   :DIGital[:STIMulus]:SIGNal[1|2]:FORMat NRZ
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//   ------------------------------------------------------------------------
//  | dataFormat                                           ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select data-format for selected channel:
//  |  | Range:
//  |  |  - HPE8311A_NRZ
//  |  |  - HPE8311A_RZ
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modePatternDataFormat(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 dataFormat
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modePatternDataFormat"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  //-------------------------------------------------------------------------
  // Only perform function when modules bios-state is in special-bios
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (dataFormat)
  {

  case HPE8311A_NRZ:
    strcpy(scpiParameter, "NRZ");
    break;

  case HPE8311A_RZ:
    strcpy(scpiParameter, "RZ");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":DIG:SIGN1:FORM",
      scpiParameter
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":DIG:SIGN2:FORM",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":DIG:SIGN1:FORM",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":DIG:SIGN2:FORM",
      scpiParameter
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------

  return (errStatus);
}
//*****************************************************************************
// hpe8311a_modePatternDataFormat_Q()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - General used
//---------------------------------------------------------------------------
// Use this function to find out whether the data-format of the selected
//   channel is HPE8311A_NRZ (non ERR_RETURN to zero) or HPE8311A_RZ (retrun to zero).
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:SIGNal[1|2]:FORMat?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//   ------------------------------------------------------------------------
//  | dataFormat                                           ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Requests the actual data-format of selected channel:
//  |  | Range:
//  |  |  - HPE8311A_NRZ
//  |  |  - HPE8311A_RZ
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modePatternDataFormat_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 dataFormat
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modePatternDataFormat_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":DIG:SIGN1:FORM?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":DIG:SIGN2:FORM?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    return (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "NRZ"))
    *dataFormat = HPE8311A_NRZ;
  else
  {
    if (!strcmp(requestedString, "RZ"))
      *dataFormat = HPE8311A_RZ;
    else
    {
      // the value got from the instrument is not within the allowed range
      *dataFormat = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *dataFormat);
  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Mode - Continuous
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_modeContinuousPulses()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - continuous Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Continuous-Pulses.
//   - Pulse-periods are generated continuously.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out not used in continuous pulse mode.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe POS
//   :TRIGger:SLOPe NEG
//   :TRIGger:COUNt 1
//   :ARM:SOURce IMM
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeContinuousPulses(
    ViSession instrumentHandle,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeContinuousPulses"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;

  GET_GLOBALS(instrumentHandle, thisPtr);
  
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :ARM:SOUR IMM; :TRIG:COUN 1");

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_FINISH;
      break;

    case STATE_MACHINE_FINISH:

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeContinuousBurst()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - continuous Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Continuous-Burst.
//   - A burst of pulse-periods is repeated continuously.
//   - The output signal is the same as in continuous-pulses-mode,
//     but the strobe-out now marks the beginning and end of each burst.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out rises at the STArt of the first pulse-period in a burst
//     and falls at the STArt of the last pulse-period.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGGer:COUNt <burstLength>
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe POS
//   :TRIGger:SLOPe NEG
//   :ARM:SOURce IMM
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | burstLength                                          ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set burstLength on
//  |  |   (number of pulse-periods per burst).
//  |  | Range: 2 to 65536
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeContinuousBurst(
    ViSession instrumentHandle,
    ViUInt32 burstLength,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeContinuousBurst"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :ARM:SOUR IMM");

      stateMachine = STATE_MACHINE_BURST_LENGTH;

      break;

    case STATE_MACHINE_BURST_LENGTH:

      strcat(completeModeCommand, "; :TRIG:COUN ");
      sprintf(scpiParameter, "%d", burstLength);
      strcat(completeModeCommand, scpiParameter);

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  } while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeContinuousPattern()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - continuous Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Continuous-Pattern.
//   - A pattern of pulses is repeated continuously.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out is bit programmable, like the outputs.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//     hpe8311a_modePatternDataFormat().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe POS
//   :TRIGger:SLOPe NEG
//   :ARM:SOURce IMM
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern ON
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to the HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeContinuousPattern(
    ViSession instrumentHandle,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeContinuousPattern"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT ON; :ARM:SOUR IMM");

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Mode - Triggered
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_modeTriggeredPulses
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Triggered Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Triggered-Pulses.
//   - Single pulse-periods are triggered by an active edge at the selected
//     arming source.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out not used in triggered pulse mode.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :TRIGger:SOURce IMM
//   :TRIGGer:COUNt 1
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//   :ARM:SENSe EDGE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerSource                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//  -------------------------------------------------------------------------
//  | triggerSlope                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger at the EXT-Input:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_BOTH
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeTriggeredPulses(
    ViSession instrumentHandle,
    ViUInt16 triggerSource,
    ViUInt16 triggerSlope
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeTriggeredPulses"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;

  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :TRIG:SOUR IMM; :TRIG:COUN 1; :ARM:SENS EDGE");

      stateMachine = STATE_MACHINE_TRIGGER_SOURCE;

      break;

    case STATE_MACHINE_TRIGGER_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (triggerSource)
      {

      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[triggerSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( triggerSource )

      stateMachine = STATE_MACHINE_TRIGGER_SLOPE;

      break;

    case STATE_MACHINE_TRIGGER_SLOPE:

      switch (triggerSlope)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_BOTH:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(triggerSlope)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeTriggeredBurst()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Triggered Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Triggered-Burst.
//   - A burst of pulse-periods is triggered by an active edge at the
//     selected arming source.
//   - The output signal is the same as in triggered-pulses-mode,
//     but the strobe out now marks the beginning and end of each burst.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out rises at the STArt of the first pulse-period in a burst
//     and falls at the STArt of the last pulse-period.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGGer:COUNt <burstLength>
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe POS
//   :TRIGger:SLOPe NEG
//   :ARM:SOURce INT2
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :ARM:PERiod <value>
//   :ARM:FREQuency <value>
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//   :ARM:SENSe EDGE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | burstLength                                          ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set burstLength on
//  |  | Range: 2 to 65536
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | triggerSource                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//   ------------------------------------------------------------------------
//  | triggerSlope                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_BOTH
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | pllPeriodFrequencyFormat                             ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the format of pll-period or pll-frequency setting:
//  |  |  - HPE8311A_VALUE_IS_PERIOD
//  |  |  - HPE8311A_VALUE_IS_FREQUENCY
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | pllPeriodFrequencyValue                              ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set pll-period or pll-frequency on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeTriggeredBurst(
    ViSession instrumentHandle,
    ViUInt32 burstLength,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge,
    ViUInt16 triggerSource,
    ViUInt16 triggerSlope,
    ViUInt16 pllPeriodFrequencyFormat,
    ViReal64 pllPeriodFrequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeTriggeredBurst"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :ARM:SENS EDGE");

      stateMachine = STATE_MACHINE_BURST_LENGTH;

      break;

    case STATE_MACHINE_BURST_LENGTH:

      strcat(completeModeCommand, "; :TRIG:COUN ");
      sprintf(scpiParameter, "%d", burstLength);
      strcat(completeModeCommand, scpiParameter);

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_TRIGGER_SOURCE;

      break;

    case STATE_MACHINE_TRIGGER_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (triggerSource)
      {

      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[triggerSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER5;
        break;

      }                         // END switch( triggerSource )

      stateMachine = STATE_MACHINE_TRIGGER_SLOPE;

      break;

    case STATE_MACHINE_TRIGGER_SLOPE:

      switch (triggerSlope)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_BOTH:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER6;
        break;

      }                         // END switch(triggerSlope)

      stateMachine = STATE_MACHINE_PLL_PERIOD_FREQUENCY;

      break;

    case STATE_MACHINE_PLL_PERIOD_FREQUENCY:

      // prepare trigger frequency/period value
      mbFloatToString(pllPeriodFrequencyValue, scpiParameter);

      switch (pllPeriodFrequencyFormat)
      {

      case HPE8311A_VALUE_IS_PERIOD:
        strcat(completeModeCommand, "; :ARM:PER ");
        strcat(completeModeCommand, scpiParameter);
        break;

      case HPE8311A_VALUE_IS_FREQUENCY:
        strcat(completeModeCommand, "; :ARM:FREQ ");
        strcat(completeModeCommand, scpiParameter);
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER7;
        break;

      }                         // END switch(triggerSlope)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeTriggeredPattern()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Triggered Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Triggered-Pattern.
//   - A pattern of pulses is triggered by an active edge form
//     the selected arming source.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out is bit programmable, like the outputs.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//     hpe8311a_modePatternDataFormat().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe NEG
//   :TRIGger:SLOPe POS
//   :ARM:SOURce INT2
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :ARM:PERiod <value>
//   :ARM:FREQuency <value>
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern ON
//   :ARM:SENSe EDGE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to the HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | triggerSource                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//   ------------------------------------------------------------------------
//  | triggerSlope                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_BOTH
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | pllPeriodFrequencyFormat                             ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the format of pll-period or pll-frequency setting:
//  |  |  - HPE8311A_VALUE_IS_PERIOD
//  |  |  - HPE8311A_VALUE_IS_FREQUENCY
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | pllPeriodFrequencyValue                              ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set pll-period or pll-frequency on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeTriggeredPattern(
    ViSession instrumentHandle,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge,
    ViUInt16 triggerSource,
    ViUInt16 triggerSlope,
    ViUInt16 pllPeriodFrequencyFormat,
    ViReal64 pllPeriodFrequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeTriggeredPattern"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT ON; :ARM:SENS EDGE");

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_TRIGGER_SOURCE;

      break;

    case STATE_MACHINE_TRIGGER_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (triggerSource)
      {

      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[triggerSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch( triggerSource )

      stateMachine = STATE_MACHINE_TRIGGER_SLOPE;

      break;

    case STATE_MACHINE_TRIGGER_SLOPE:

      switch (triggerSlope)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_BOTH:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER5;
        break;

      }                         // END switch(triggerSlope)

      stateMachine = STATE_MACHINE_PLL_PERIOD_FREQUENCY;

      break;

    case STATE_MACHINE_PLL_PERIOD_FREQUENCY:

      // prepare Period/Frequency value
      mbFloatToString(pllPeriodFrequencyValue, scpiParameter);

      switch (pllPeriodFrequencyFormat)
      {

      case HPE8311A_VALUE_IS_PERIOD:
        strcat(completeModeCommand, "; :ARM:PER ");
        strcat(completeModeCommand, scpiParameter);
        break;

      case HPE8311A_VALUE_IS_FREQUENCY:
        strcat(completeModeCommand, "; :ARM:FREQ ");
        strcat(completeModeCommand, scpiParameter);
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER6;
        break;

      }                         // END switch(triggerSlope)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Mode - Gated
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_modeGatedPulses()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Gated Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Gated-Pulses.
//   - Pulse-periods are gated by an active level at the
//     selected arming source.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out not used in continuous pulse mode.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe NEG
//   :TRIGger:SLOPe POS
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :TRIGger:COUNt 1
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//   :ARM:SENSe LEVel
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | gateSource                                           ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//   ------------------------------------------------------------------------
//  | gateLevel                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger:
//  |  |  - HPE8311A_LOW_LEVEL
//  |  |  - HPE8311A_HIGH_LEVEL
//  |  |  - HPE8311A_ALWAYS
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeGatedPulses(
    ViSession instrumentHandle,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge,
    ViUInt16 gateSource,
    ViUInt16 gateLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeGatedPulses"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:
      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :ARM:SENS LEV; :TRIG:COUN 1");
      stateMachine = STATE_MACHINE_PERIOD_SOURCE;
      break;

    case STATE_MACHINE_PERIOD_SOURCE:
      strcat(completeModeCommand, "; :TRIG:SOUR ");
      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;
      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_GATE_SOURCE;

      break;

    case STATE_MACHINE_GATE_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (gateSource)
      {

      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[gateSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch( gateSource )

      stateMachine = STATE_MACHINE_GATE_LEVEL;

      break;

    case STATE_MACHINE_GATE_LEVEL:

      switch (gateLevel)
      {

      case HPE8311A_LOW_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_HIGH_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_ALWAYS:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER5;
        break;

      }                         // END switch(gateLevel)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  } while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeGatedBurst()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Gated Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Gated-Burst.
//   - Bursts of pulse-periods are gated by an active level at
//     the selected arming source.
//   - The output signal is the same as in gated-pulses-mode,
//     but the strobe out now marks the beginning and end of each burst.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out rises at the STArt of the first pulse-period in a burst
//     and falls at the STArt of the last pulse-period.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePulseType().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGGer:COUNt <burstLength>
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe NEG
//   :TRIGger:SLOPe POS
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :ARM:EWIDTH[:STATe] OFF
//   :DIGital:PATTern OFF
//   :ARM:SENSe LEVel
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | burstLength                                          ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set burstLength on
//  |  | Range: 2 to 65536
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | gateSource                                           ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//   ------------------------------------------------------------------------
//  | gateLevel                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger:
//  |  |  - HPE8311A_LOW_LEVEL
//  |  |  - HPE8311A_HIGH_LEVEL
//  |  |  - HPE8311A_ALWAYS
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeGatedBurst(
    ViSession instrumentHandle,
    ViUInt32 burstLength,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge,
    ViUInt16 gateSource,
    ViUInt16 gateLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeGatedBurst"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT OFF; :ARM:SENS LEV");

      stateMachine = STATE_MACHINE_BURST_LENGTH;

      break;

    case STATE_MACHINE_BURST_LENGTH:

      strcat(completeModeCommand, "; :TRIG:COUN ");
      sprintf(scpiParameter, "%d", burstLength);
      strcat(completeModeCommand, scpiParameter);

      stateMachine = STATE_MACHINE_PERIOD_SOURCE;

      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {

      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;

      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_GATE_SOURCE;

      break;

    case STATE_MACHINE_GATE_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (gateSource)
      {

      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[gateSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER5;
        break;

      }                         // END switch( gateSource )

      stateMachine = STATE_MACHINE_GATE_LEVEL;

      break;

    case STATE_MACHINE_GATE_LEVEL:

      switch (gateLevel)
      {

      case HPE8311A_LOW_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_HIGH_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_ALWAYS:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER6;
        break;

      }                         // END switch(gateLevel)

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_modeGatedPattern()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - Gated Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: Gated-Pattern.
//   - A pattern of pulses is gated by an active level at
//     the selected arming source.
//   - Trigger-out marks each pulse-period.
//   - Strobe-out is bit programmable, like the outputs.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_modePatternDataFormat().
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:SOURce IMM
//   :TRIGger:SOURce INT2
//   :TRIGger:SOURce EXT2
//   :TRIGger:SLOPe NEG
//   :TRIGger:SLOPe POS
//   :ARM:SOURce EXT
//   :ARM:SOURce MAN
//   :ARM:SLOPe NEG
//   :ARM:SLOPe POS
//   :ARM:SLOPe EITH
//   :ARM:EWIDTH:[STATe] OFF
//   :DIGital:PATTern ON
//   :ARM:SENSe LEVel
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pulsePeriodSource                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of pulse-period:
//  |  |  - HPE8311A_INTERNAL_OSCILLATOR
//  |  |  - HPE8311A_INTERNAL_PLL
//  |  |  - HPE8311A_CLOCK_INPUT
//   ------------------------------------------------------------------------
//  | externalClockEdge                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Synchronize the Pulse-period to HPE8311A_RISING or HPE8311A_FALLING edge at CLK-IN:
//  |  |  - HPE8311A_FALLING
//  |  |  - HPE8311A_RISING
//  |  |  - HPE8311A_UNTOUCHED
//   ------------------------------------------------------------------------
//  | gateSource                                           ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//   ------------------------------------------------------------------------
//  | gateLevel                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select slope of trigger:
//  |  |  - HPE8311A_LOW_LEVEL
//  |  |  - HPE8311A_HIGH_LEVEL
//  |  |  - HPE8311A_ALWAYS
//  |  |  - HPE8311A_UNTOUCHED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeGatedPattern(
    ViSession instrumentHandle,
    ViUInt16 pulsePeriodSource,
    ViUInt16 externalClockEdge,
    ViUInt16 gateSource,
    ViUInt16 gateLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeGatedPattern"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.
  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID OFF; :DIG:PATT ON; :ARM:SENS LEV");
      stateMachine = STATE_MACHINE_PERIOD_SOURCE;
      break;

    case STATE_MACHINE_PERIOD_SOURCE:

      strcat(completeModeCommand, "; :TRIG:SOUR ");

      switch (pulsePeriodSource)
      {
      case HPE8311A_INTERNAL_OSCILLATOR:
      case HPE8311A_INTERNAL_PLL:
      case HPE8311A_CLOCK_INPUT:
        strcat(completeModeCommand, sourceArray[pulsePeriodSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;
      }                         // END switch( pulsePeriodSource )

      stateMachine = STATE_MACHINE_CLOCK_EDGE;
      break;

    case STATE_MACHINE_CLOCK_EDGE:

      switch (externalClockEdge)
      {

      case HPE8311A_FALLING:
        strcat(completeModeCommand, "; :TRIG:SLOP NEG");
        break;

      case HPE8311A_RISING:
        strcat(completeModeCommand, "; :TRIG:SLOP POS");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER3;
        break;

      }                         // END switch(externalClockEdge)

      stateMachine = STATE_MACHINE_GATE_SOURCE;
      break;

    case STATE_MACHINE_GATE_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (gateSource)
      {

      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[gateSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;

      }                         // END switch( gateSource )

      stateMachine = STATE_MACHINE_GATE_LEVEL;
      break;

    case STATE_MACHINE_GATE_LEVEL:

      switch (gateLevel)
      {
      case HPE8311A_LOW_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP NEG");
        break;

      case HPE8311A_HIGH_LEVEL:
        strcat(completeModeCommand, "; :ARM:SLOP POS");
        break;

      case HPE8311A_ALWAYS:
        strcat(completeModeCommand, "; :ARM:SLOP EITH");
        break;

      case HPE8311A_UNTOUCHED:
        break;

      default:
        errStatus = VI_ERROR_PARAMETER5;
        break;

      }                         // END switch(gateLevel)

      stateMachine = STATE_MACHINE_FINISH;
      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  } while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Mode - External width
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_modeExternalWidth()
//*****************************************************************************
// CATEGORY: MODE-FUNCTIONS - External-width Mode
//---------------------------------------------------------------------------
// This functions configures the instruments mode into: External-width.
//   - The pulse-width is determined by an external signal.
//---------------------------------------------------------------------------
// The period, delay, and width of the output pulse are not programmable in
//   this mode as they are determined by the external signal.
//---------------------------------------------------------------------------
// To set the instrument completely to the desired mode use additionaly
//   hpe8311a_configInputExternal().
//---------------------------------------------------------------------------
// Used scpi-command:
//   :ARM:EWIDTH[:STATe] ON
//   :ARM:SOUR EXT
//   :ARM:SOUR MAN
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerSource                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of trigger:
//  |  |  - HPE8311A_EXTERNAL_INPUT
//  |  |  - HPE8311A_MANUAL
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_modeExternalWidth(
    ViSession instrumentHandle,
    ViUInt16 triggerSource
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_modeExternalWidth"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  ViInt8 stateMachine;
  ViInt8 stateMachineFinish;
  ViChar completeModeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument
  // therfore STArt a stateMachine.

  stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
  stateMachineFinish = STATE_MACHINE_NO_FINISH;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_GENERAL_COMMANDS:

      strcpy(completeModeCommand, ":ARM:EWID ON");

      stateMachine = STATE_MACHINE_TRIGGER_SOURCE;

      break;

    case STATE_MACHINE_TRIGGER_SOURCE:

      strcat(completeModeCommand, "; :ARM:SOUR ");

      switch (triggerSource)
      {

      case HPE8311A_EXTERNAL_INPUT:
      case HPE8311A_MANUAL:
      case HPE8311A_VXI_ECL_0:
      case HPE8311A_VXI_ECL_1:
      case HPE8311A_VXI_TTL_0:
      case HPE8311A_VXI_TTL_1:
      case HPE8311A_VXI_TTL_2:
      case HPE8311A_VXI_TTL_3:
      case HPE8311A_VXI_TTL_4:
      case HPE8311A_VXI_TTL_5:
      case HPE8311A_VXI_TTL_6:
      case HPE8311A_VXI_TTL_7:
        strcat(completeModeCommand, sourceArray[triggerSource]);
        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch( triggerSource )

      stateMachine = STATE_MACHINE_FINISH;

      break;

    case STATE_MACHINE_FINISH:
      // finish stateMachine

      errStatus = hpe8311a_cmd(
        instrumentHandle,
        completeModeCommand
        );

      stateMachineFinish = STATE_MACHINE_DO_FINISH;

      break;

    default:
      errStatus = HPE8311A_INSTR_ERROR_STATE_MACHINE_ERROR;
      stateMachineFinish = STATE_MACHINE_DO_FINISH;
      break;

    }                           // END switch( stateMachine )

  }
  while (!errStatus && !stateMachineFinish);

  ERR_CHK_RETURN (errStatus);
}
//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// TIMING-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Period
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timePeriod()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Period
//---------------------------------------------------------------------------
// This function sets the instruments pulse-period.
//---------------------------------------------------------------------------
// hpe8311a_timeFrequency() is complementary to this function.
//---------------------------------------------------------------------------
// Select the period source for the pulse-period using :TRIGger:SOURce.
//   The currently selected source is programmed by this command. Note that
//   the specified limits and available resolution depend on the selected
//   source.
//---------------------------------------------------------------------------
// You cannot set the pulse-period if you have selected the CLOCK-IN
//   connector as the period source (:TRIG:SOUR EXT).
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:PERiod
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | periodValue                                          ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set period on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timePeriod(
    ViSession instrumentHandle,
    ViReal64 periodValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timePeriod"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal(instrumentHandle, ":PULS:PER", periodValue);

}
//*****************************************************************************
// hpe8311a_timePeriod_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Period
//---------------------------------------------------------------------------
// This function requests the pulse-period from the instrument.
//---------------------------------------------------------------------------
// hpe8311a_timeFrequency_Q() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:PERiod?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | periodValue                                          ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested period to.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timePeriod_Q(
    ViSession instrumentHandle,
    ViReal64 * periodValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timePeriod_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal64_Q(instrumentHandle, ":PULS:PER?", periodValue);

}
//*****************************************************************************
// hpe8311a_timePeriodMeasure()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Time Period
//---------------------------------------------------------------------------
// This function measures the period provided at external CLOCK-Input.
//---------------------------------------------------------------------------
// hpe8311a_timeFrequencyMeasure() is complementary to this function.
//---------------------------------------------------------------------------
// If the CLOCK-IN connector is the selected pulse-period source, you can
//   then read the measured value with hpe8311a_timePeriod_Q().
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:PERiod:AUTO ONCE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timePeriodMeasure(
    ViSession instrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timePeriodMeasure"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmd(instrumentHandle, ":PULS:PER:AUTO ONCE");

}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Frequency
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeFrequency()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Frequency
//---------------------------------------------------------------------------
// This function sets the instruments frequency of pulses.
//---------------------------------------------------------------------------
// hpe8311a_timePeriod() is complementary to this function.
//---------------------------------------------------------------------------
// Select the frequency source for the pulse frequency using :TRIGger:SOURce.
//   The currently selected source is programmed by this command. Note that
//   the specified limits and available resolution depend on the selected
//   source.
//---------------------------------------------------------------------------
// You cannot set the pulse frequency if you have selected the CLOCK-IN
//   connector as the frequency source (:TRIG:SOUR EXT).
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:FREQuency[:FIXed] <frequencyValue>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | frequencyValue                                       ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set frequency on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeFrequency(
    ViSession instrumentHandle,
    ViReal64 frequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeFrequency"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal(instrumentHandle, ":FREQ", frequencyValue);

}
//*****************************************************************************
// hpe8311a_timeFrequency_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Frequency
//---------------------------------------------------------------------------
// This function requests the frequency of pulses from the instrument.
//---------------------------------------------------------------------------
// hpe8311a_timePeriod_Q() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:FREQuency[:FIXed]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | frequencyValue                                       ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested frequency to.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeFrequency_Q(
    ViSession instrumentHandle,
    ViReal64 * frequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeFrequency_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal64_Q(instrumentHandle, ":FREQ?", frequencyValue);

}
//*****************************************************************************
// hpe8311a_timeFrequencyMeasure()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Frequency
//---------------------------------------------------------------------------
// This function measures the frequency provided at external CLOCK-Input.
//---------------------------------------------------------------------------
// hpe8311a_timePeriodMeasure() is complementary to this function.
//---------------------------------------------------------------------------
// If the CLOCK-IN connector is the selected pulse frequency source, you can
//   then read the measured value with hpe8311a_timeFrequency_Q().
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:FREQuency[:FIXed]:AUTO ONCE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeFrequencyMeasure(
    ViSession instrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeFrequencyMeasure"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmd(instrumentHandle, ":FREQ:AUTO ONCE");

}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Delay
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeDelay()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Delay
//---------------------------------------------------------------------------
// This function sets the instruments delay parameter. "Delay" means the
//   the delay of the pulses leading-edge within the pulse-period.
//---------------------------------------------------------------------------
// There are four different units for the value of delay available.
//  - HPE8311A_UNIT_SEC (second)
//  - HPE8311A_UNIT_PCT (percent)
//  - HPE8311A_UNIT_DEG
//  - HPE8311A_UNIT_RAD
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType() to activate the desired
//   pulse-type in the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DELay[1|2]:UNIT SEC
//   [:SOURce]:PULSe:DELay[1|2]:UNIT PCT
//   [:SOURce]:PULSe:DELay[1|2]:UNIT DEG
//   [:SOURce]:PULSe:DELay[1|2]:UNIT RAD
//   [:SOURce]:PULSe:DELay[1|2] <value>
//*****************************************************************************
//   PARAMETER Name                                        Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | delayUnit                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit for provided value of delay:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//  |  |  - HPE8311A_UNIT_DEG
//  |  |  - HPE8311A_UNIT_RAD
//   ------------------------------------------------------------------------
//  | delayValue                                           ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set delay on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelay(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 delayUnit,
    ViReal64 delayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelay"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeCommand[HPE8311A_STRING_LENGTH];
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar valueString[HPE8311A_STRING_LENGTH];
  
  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Prepare the complete command to send to the instrument.

  switch (delayUnit)
  {

  case HPE8311A_UNIT_SEC:
    strcpy(unitString, "SEC");
    break;

  case HPE8311A_UNIT_PCT:
    strcpy(unitString, "PCT");
    break;

  case HPE8311A_UNIT_DEG:
    strcpy(unitString, "DEG");
    break;

  case HPE8311A_UNIT_RAD:
    strcpy(unitString, "RAD");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch( delayUnit )

  mbFloatToString(delayValue, valueString);

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:DEL1:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DEL1 ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " "); // do need additional unit!
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :PULS:DEL2:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DEL2 ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_1:

    strcpy(completeCommand, ":PULS:DEL1:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DEL1 ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_2:

    strcpy(completeCommand, ":PULS:DEL2:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DEL2 ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  default:
    DBG_PRINT("failed: wrong Parameter 2!");
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelay_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Delay
//---------------------------------------------------------------------------
// This function requests the delay parameter from the instrument. "Delay"
//   means the the delay of the pulses leading-edge within the pulse-period.
//---------------------------------------------------------------------------
// There are four different units for the value of delay available.
//  - HPE8311A_UNIT_SEC (second)
//  - HPE8311A_UNIT_PCT (percent)
//  - HPE8311A_UNIT_DEG
//  - HPE8311A_UNIT_RAD
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType_Q() to request the actual used
//   pulse-type from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DELay[1|2]:UNIT?
//   [:SOURce]:PULSe:DELay[1|2]?
//*****************************************************************************
//   PARAMETER Name                                        Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | delayUnit                                            ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for actual used value of delay:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//  |  |  - HPE8311A_UNIT_DEG
//  |  |  - HPE8311A_UNIT_RAD
//   ------------------------------------------------------------------------
//  | delayValue                                           ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used value delay is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelay_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 delayUnit,
    ViPReal64 delayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelay_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DEL1:UNIT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:DEL1?",
      delayValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DEL2:UNIT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:DEL2?",
      delayValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "SEC"))
    *delayUnit = HPE8311A_UNIT_SEC;
  else if (!strcmp(requestedString, "S"))
    *delayUnit = HPE8311A_UNIT_SEC;
  else if (!strcmp(requestedString, "PCT"))
    *delayUnit = HPE8311A_UNIT_PCT;
  else if (!strcmp(requestedString, "DEG"))
    *delayUnit = HPE8311A_UNIT_DEG;
  else if (!strcmp(requestedString, "RAD"))
    *delayUnit = HPE8311A_UNIT_RAD;
  else
  {
    // the value got from the instrument is not within the allowed range
    *delayUnit = HPE8311A_INVALID_VALUE;
    errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
  }

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelayHold()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Delay
//---------------------------------------------------------------------------
// This function sets the instruments delay parameter. "Delay" means the
//   the delay of the pulses leading-edge within the pulse-period.
//---------------------------------------------------------------------------
// There are two delay hold-modes available.
//   - HPE8311A_HOLD_ABSOLUTE
//     is the absolute delay from the STArt of a pulse-period to the
//     STArt of the leading-edge of the pulse. The absolute delay is
//     independent of the pulse-period so the leading-edge does not move
//     relative to the STArt of the period if you change the period.
//   - HPE8311A_HOLD_RATIO
//     is the delay from the STArt of the pulse-period to the STArt
//     of the leading-edge expressed as a percentage or in degrees
//     (360 = 1 pulse-period) of the pulse-period.
//     In this format if you change the period, the leading-edge moves
//     relative to the STArt of the period in order to maintain the
//     percentage delay or the phase delay.
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType() to activate the desired
//   pulse-type in the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DELay[1|2]:HOLD TIME
//   [:SOURce]:PULSe:DELay[1|2]:HOLD PRATio
//*****************************************************************************
//   PARAMETER Name                                        Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | delayHoldMode                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the hold-mode for delay:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayHold(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 delayHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayHold"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (delayHoldMode)
  {

  case HPE8311A_HOLD_ABSOLUTE:
    strcpy(scpiParameter, "TIME");
    break;

  case HPE8311A_HOLD_RATIO:
    strcpy(scpiParameter, "PRAT");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:DEL1:HOLD ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:DEL2:HOLD ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DEL1:HOLD",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DEL2:HOLD",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelayHold_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Delay
//---------------------------------------------------------------------------
// This function requests the delay parameter from the instrument. "Delay"
//   means the the delay of the pulses leading-edge within the pulse-period.
//---------------------------------------------------------------------------
// There are two delay hold-modes available.
//   - HPE8311A_HOLD_ABSOLUTE
//     is the absolute delay from the STArt of a pulse-period to the
//     STArt of the leading-edge of the pulse. The absolute delay is
//     independent of the pulse-period so the leading-edge does not move
//     relative to the STArt of the period if you change the period.
//   - HPE8311A_HOLD_RATIO
//     is the delay from the STArt of the pulse-period to the STArt
//     of the leading-edge expressed as a percentage or in degrees
//     (360 = 1 pulse-period) of the pulse-period.
//     In this format if you change the period, the leading-edge moves
//     relative to the STArt of the period in order to maintain the
//     percentage delay or the phase delay.
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType_Q() to request the actual used
//   pulse-type from the instrument.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:DELay[1|2]:HOLD?
//*****************************************************************************
//   PARAMETER Name                                        Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | delayHoldMode                                        ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used hold-mode for delay:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayHold_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 delayHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayHold_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "PULS:DEL1:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "PULS:DEL2:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "TIME"))
    *delayHoldMode = HPE8311A_HOLD_ABSOLUTE;
  else
  {
    if (!strcmp(requestedString, "PRAT"))
      *delayHoldMode = HPE8311A_HOLD_RATIO;
    else
    {
      // the value got from the instrument is not within the allowed range
      *delayHoldMode = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Double-Delay
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeDelayDouble()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Double-Delay
//---------------------------------------------------------------------------
// This function sets the instruments double-delay parameter. "Double-Delay"
//   means the delay between the leading edges of the two pulses in
//   double-pulse mode The first pulse HPE8311A_ALWAYS STArts at the STArt of the
//   pulse-period.
//---------------------------------------------------------------------------
// There are two different units for the value of double-delay available.
//  - HPE8311A_UNIT_SEC (second)
//  - HPE8311A_UNIT_PCT (percent)
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType() to activate the desired
//   pulse-type in the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:UNIT SEC
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:UNIT PCT
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | doubleDelayUnit                                      ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit for provided value of double-delay:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | doubleDelayValue                                     ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set double-delay on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayDouble(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 doubleDelayUnit,
    ViReal64 doubleDelayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayDouble"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar doubleDelayString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (doubleDelayUnit)
  {

  case HPE8311A_UNIT_SEC:
    strcpy(unitString, "SEC");
    break;

  case HPE8311A_UNIT_PCT:
    strcpy(unitString, "PCT");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  mbFloatToString(doubleDelayValue, doubleDelayString);

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:DOUB1:DEL:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DOUB1:DEL ");
    strcat(completeCommand, doubleDelayString);

    strcat(completeCommand, "; :PULS:DOUB2:DEL:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DOUB2:DEL ");
    strcat(completeCommand, doubleDelayString);

    break;

  case HPE8311A_CHANNEL_1:

    strcpy(completeCommand, ":PULS:DOUB1:DEL:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DOUB1:DEL ");
    strcat(completeCommand, doubleDelayString);

    break;

  case HPE8311A_CHANNEL_2:

    strcpy(completeCommand, "; :PULS:DOUB2:DEL:UNIT ");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "; :PULS:DOUB2:DEL ");
    strcat(completeCommand, doubleDelayString);

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelayDouble_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Double-Delay
//---------------------------------------------------------------------------
// This function requests the double-delay parameter from the instrument.
//   "Double-Delay" means the delay between the leading edges of the two
//   pulses in double-pulse mode The first fulse HPE8311A_ALWAYS STArts at the STArt
//   of the pulse-period.
//---------------------------------------------------------------------------
// There are two different units for the value of double-delay available.
//  - HPE8311A_UNIT_SEC (second)
//  - HPE8311A_UNIT_PCT (percent)
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType_Q() to request the actual used
//   pulse-type from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:UNIT?
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | doubleDelayUnit                                      ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for actual used value of double-delay:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | doubleDelayValue                                     ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Report the actual used value double-delay is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayDouble_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 doubleDelayUnit,
    ViPReal64 doubleDelayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayDouble_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "PULS:DOUB1:DEL:UNIT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      "PULS:DOUB1:DEL?",
      doubleDelayValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "PULS:DOUB2:DEL:UNIT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      "PULS:DOUB2:DEL?",
      doubleDelayValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "S"))
    *doubleDelayUnit = HPE8311A_UNIT_SEC;
  else if (!strcmp(requestedString, "SEC"))
    *doubleDelayUnit = HPE8311A_UNIT_SEC;
  else if (!strcmp(requestedString, "PCT"))
    *doubleDelayUnit = HPE8311A_UNIT_PCT;
  else
  {
    // the value got from the instrument is not within the allowed range
    *doubleDelayUnit = HPE8311A_INVALID_VALUE;
    errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
  }

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelayDoubleHold()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Double-Delay
//---------------------------------------------------------------------------
// This function sets the instruments double-delay parameter. "Double-Delay"
//   means the delay between the leading edges of the two pulses in
//   double-pulse mode The first fulse HPE8311A_ALWAYS STArts at the STArt of the
//   pulse-period.
//---------------------------------------------------------------------------
// There are two delay hold-modes available.
//   - HPE8311A_HOLD_ABSOLUTE
//     is the absolute delay from the STArt of a pulse-period to the
//     STArt of the leading-edge of the pulse. The absolute delay is
//     independent of the pulse-period so the leading-edge does not move
//     relative to the STArt of the period if you change the period.
//   - HPE8311A_HOLD_RATIO
//     is the delay from the STArt of the pulse-period to the STArt
//     of the leading-edge expressed as a percentage of the pulse-period.
//     In this format if you change the period, the leading-edge moves
//     relative to the STArt of the period in order to maintain the
//     the phase delay.
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType() to activate the desired
//   pulse-type in the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:HOLD TIME
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:HOLD PRATio
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | doubleDelayHoldMode                                  ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the hold-mode for double-delay:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayDoubleHold(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 doubleDelayHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayDoubleHold"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeCommand[HPE8311A_STRING_LENGTH];
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (doubleDelayHoldMode)
  {

  case HPE8311A_HOLD_ABSOLUTE:
    strcpy(scpiParameter, "TIME");
    break;

  case HPE8311A_HOLD_RATIO:
    strcpy(scpiParameter, "PRAT");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:DOUB1:DEL:HOLD ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:DOUB2:DEL:HOLD ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB1:DEL:HOLD",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:DOUB2:DEL:HOLD",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeDelayDoubleHold_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Double-Delay
//---------------------------------------------------------------------------
// This function requests the double-delay parameter from the instrument.
//   "Double-Delay" means the delay between the leading edges of the two
//   pulses in double-pulse mode The first fulse HPE8311A_ALWAYS STArts at the STArt
//   of the pulse-period.
//---------------------------------------------------------------------------
// There are two delay hold-modes available.
//   - HPE8311A_HOLD_ABSOLUTE
//     is the absolute delay from the STArt of a pulse-period to the
//     STArt of the leading-edge of the pulse. The absolute delay is
//     independent of the pulse-period so the leading-edge does not move
//     relative to the STArt of the period if you change the period.
//   - HPE8311A_HOLD_RATIO
//     is the delay from the STArt of the pulse-period to the STArt
//     of the leading-edge expressed as a percentage or in degrees
//     (360 = 1 pulse-period) of the pulse-period.
//     In this format if you change the period, the leading-edge moves
//     relative to the STArt of the period in order to maintain the
//     percentage delay or the phase delay.
//---------------------------------------------------------------------------
// Use also function hpe8311a_modePulseType_Q() to request the actual used
//   pulse-type from the instrument.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:DOUBle[1|2]:DELay:HOLD?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | doubleDelayHoldMode                                  ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used hold-mode for double-delay:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeDelayDoubleHold_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 doubleDelayHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeDelayDoubleHold_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DOUB1:DEL:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:DOUB2:DEL:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "TIME"))
    *doubleDelayHoldMode = HPE8311A_HOLD_ABSOLUTE;
  else if (!strcmp(requestedString, "PRAT"))
    *doubleDelayHoldMode = HPE8311A_HOLD_RATIO;
  else
  {
    // the value got from the instrument is not within the allowed range
    *doubleDelayHoldMode = HPE8311A_INVALID_VALUE;
    errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
  }

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - width
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeWidth()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function sets the instruments width parameter.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:WIDTH[1|2]  <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | widthValue                                           ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set width on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidth(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 widthValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidth"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar widthString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    mbFloatToString(widthValue, widthString);

    strcpy(completeCommand, ":SOUR:PULS:WIDT1 ");
    strcat(completeCommand, widthString);
    strcat(completeCommand, "; :SOUR:PULS:WIDT2 ");
    strcat(completeCommand, widthString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":SOUR:PULS:WIDT1",
      widthValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":SOUR:PULS:WIDT2",
      widthValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidth_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function requests the width parameter from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DCYCle[1|2]?
//   [:SOURce]:PULSe:TDELay[1|2]?
//   [:SOURce]:PULSe:WIDTH[1|2]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | widthValue                                           ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Report the actual used value width is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidth_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * widthValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidth_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:WIDT1?",
      widthValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:WIDT2?",
      widthValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthDutyCycle()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function sets the instruments duty-cycle parameter.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DCYCle[1|2] <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | dutyCycleValue                                       ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set duty-cycle on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthDutyCycle(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 dutyCycleValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthDutyCycle"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar dutyCycleString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    mbFloatToString(dutyCycleValue, dutyCycleString);

    strcpy(completeCommand, ":PULS:DCYC1 ");
    strcat(completeCommand, dutyCycleString);
    strcat(completeCommand, "; :PULS:DCYC2 ");
    strcat(completeCommand, dutyCycleString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":PULS:DCYC1",
      dutyCycleValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":PULS:DCYC2",
      dutyCycleValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthDutyCycle_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function requests the duty-cycle parameter from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:DCYCle[1|2]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | dutyCycleValue                                       ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Report the actual used value duty-cycle is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthDutyCycle_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * dutyCycleValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthDutyCycle_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:DCYC1?",
      dutyCycleValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:DCYC2?",
      dutyCycleValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthTrailingDelay()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function sets the instruments trailing-delay parameter.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TDELay[1|2] <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingDelayValue                                   ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set trailing-delay on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthTrailingDelay(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 trailingDelayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthTrailingDelay"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar trailingDelayString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    mbFloatToString(trailingDelayValue, trailingDelayString);

    strcpy(completeCommand, ":PULS:TDEL1 ");
    strcat(completeCommand, trailingDelayString);
    strcat(completeCommand, "; :PULS:TDEL2 ");
    strcat(completeCommand, trailingDelayString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":PULS:TDEL1",
      trailingDelayValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":PULS:TDEL2",
      trailingDelayValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthTrailingDelay_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function requests the trailing-delay parameter from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TDELay[1|2]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingDelayValue                                   ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Report the actual used value trailing-delay is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthTrailingDelay_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * trailingDelayValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthTrailingDelay_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TDEL1?",
      trailingDelayValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TDEL2?",
      trailingDelayValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthHold()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function sets the instruments width parameter.
//---------------------------------------------------------------------------
// There are three hold-modes for width available.
//   - width
//     is the absolute pulse-width measured from start of the leading-edge to
//     start of the trailing-edge. In this format the pulse-width is
//     independent of changes in pulse-period and delay.
//   - HPE8311A_DUTY_CYCLE
//     is the pulse-width measured from start of the leading-edge to start of
//     the trailing-edge expressed as a percentage of the period. In this
//     format if you adjust the period, the absolute width is adjusted to
//     maintain the dutycycle. You cannot have the width format set to
//     dutycycle and the leading/trailing-edge format set to percentage of
//     width at the same time.
//   - HPE8311A_TRAILING_DELAY
//     is the absolute delay from the start of the pulse-period to the start
//     of the trailing-edge. In this format the trailing-edge remains
//     fixed relative to the start of the pulse-period if you adjust the
//     pulse-delay (leading-edge delay) or the pulse-period.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:HOLD[1|2] WIDTH
//   [:SOURce]:PULSe:HOLD[1|2] DCYCle
//   [:SOURce]:PULSe:HOLD[1|2] TDELay
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | widthHoldMode                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the hold-mode for width:
//  |  | Range:
//  |  |  - HPE8311A_WIDTH
//  |  |  - HPE8311A_DUTY_CYCLE
//  |  |  - HPE8311A_TRAILING_DELAY
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthHold(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 widthHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthHold"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (widthHoldMode)
  {

  case HPE8311A_WIDTH:
    strcpy(scpiParameter, "WIDT");
    break;

  case HPE8311A_DUTY_CYCLE:
    strcpy(scpiParameter, "DCYC");
    break;

  case HPE8311A_TRAILING_DELAY:
    strcpy(scpiParameter, "TDEL");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch(widthHoldMode)

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:HOLD1 ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:HOLD2 ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:HOLD1",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:HOLD2",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_timeWidthHold_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - width
//---------------------------------------------------------------------------
// This function requests the width parameter from the instrument.
//---------------------------------------------------------------------------
// There are three hold-modes for width available.
//   - HPE8311A_WIDTH
//     is the absolute pulse-width measured from start of the leading-edge to
//     start of the trailing-edge. In this format the pulse-width is
//     independent of changes in pulse-period and delay.
//   - HPE8311A_DUTY_CYCLE
//     is the pulse-width measured from start of the leading-edge to start of
//     the trailing-edge expressed as a percentage of the period. In this
//     format if you adjust the period, the absolute width is adjusted to
//     maintain the dutycycle. You cannot have the width format set to
//     dutycycle and the leading/trailing-edge format set to percentage of
//     width at the same time.
//   - HPE8311A_TRAILING_DELAY
//     is the absolute delay from the start of the pulse-period to the start
//     of the trailing-edge. In this format the trailing-edge remains
//     fixed relative to the start of the pulse-period if you adjust the
//     pulse-delay (leading-edge delay) or the pulse-period.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:HOLD[1|2]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | widthHoldMode                                        ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used hold-mode for width:
//  |  | Range:
//  |  |  - HPE8311A_WIDTH
//  |  |  - HPE8311A_DUTY_CYCLE
//  |  |  - HPE8311A_TRAILING_DELAY
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeWidthHold_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 widthHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeWidthHold_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:HOLD1?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:HOLD2?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "WIDT"))
    *widthHoldMode = HPE8311A_WIDTH;
  else
  {
    if (!strcmp(requestedString, "DCYC"))
      *widthHoldMode = HPE8311A_DUTY_CYCLE;
    else
    {
      if (!strcmp(requestedString, "TDEL"))
        *widthHoldMode = HPE8311A_TRAILING_DELAY;
      else
      {
        // the value got from the instrument is not within the allowed range
        *widthHoldMode = HPE8311A_INVALID_VALUE;
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
      }
    }
  }

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Leading Edge
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeEdgeLeading()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Leading Edge
//---------------------------------------------------------------------------
// This function sets the instruments leading-edge parameter. "Leading Edge"
//   means the transition-time of the pulse, measured from 10% to
//   90% of pulse amplitude. Note: The value of leading-edge is
//   independently programmable within certain ranges only.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeHold() to select the hold mode. Note:
//   the hold mode is set for leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT SEC
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT PCT
//   [:SOURce]:PULSe:TRANsition[1|2]:LEADing <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | leadingEdgeUnit                                      ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit for provided value of leadingEdge:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | leadingEdgeValue                                     ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set leading-edge on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeLeading(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 leadingEdgeUnit,
    ViReal64 leadingEdgeValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeLeading"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar valueString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Prepare the complete command to send to the instrument.
  completeCommand[0] = 0x00;

  switch (leadingEdgeUnit)
  {

  case HPE8311A_UNIT_SEC:
    strcpy(unitString, "SEC");
    break;

  case HPE8311A_UNIT_PCT:
    // make sure we've got seconds if we can't do percent
    if (thisPtr->options.trans_unit == 0)
      ERR_RETURN (VI_ERROR_PARAMETER3);

    strcpy(unitString, "PCT");
    break;

  default:

    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  mbFloatToString(leadingEdgeValue, valueString);

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN1:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }
    strcat(completeCommand, ":PULS:TRAN1:LEAD ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " "); // do need additional unit!
    strcat(completeCommand, unitString);

    if (thisPtr->options.trans_unit != 0)
    {
      strcat(completeCommand, "; :PULS:TRAN2:UNIT ");
      strcat(completeCommand, unitString);
    }
    strcat(completeCommand, "; :PULS:TRAN2:LEAD ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_1:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN1:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }
    strcat(completeCommand, ":PULS:TRAN1:LEAD ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_2:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN2:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }
    strcat(completeCommand, ":PULS:TRAN2:LEAD ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_timeEdgeLeading_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Leading Edge
//---------------------------------------------------------------------------
// This function requests the leading-edge parameter from the instrument.
//   "Leading Edge" means the transition-time of the pulse, measured from
//   10% to 90% of pulse amplitude.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeHold_Q() to request the actual used
//   hold mode from the instrument. Note: this hold mode is valid for
//   leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT?
//   [:SOURce]:PULSe:TRANsition[1|2]:LEADing?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | leadingEdgeUnit                                      ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for actual used value of leadingEdge:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | leadingEdgeValue                                     ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used value leadingEdge is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeLeading_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 leadingEdgeUnit,
    ViPReal64 leadingEdgeValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeLeading_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    if (thisPtr->options.trans_unit != 0)
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle,
        ":PULS:TRAN1:UNIT?",
        HPE8311A_STRING_LENGTH,
        requestedString
        );
    }

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TRAN1:LEAD?",
      leadingEdgeValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    if (thisPtr->options.trans_unit != 0)
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle,
        ":PULS:TRAN2:UNIT?",
        HPE8311A_STRING_LENGTH,
        requestedString
       );
    }

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TRAN2:LEAD?",
      leadingEdgeValue
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (thisPtr->options.trans_unit != 0)
  {
    if (!strcmp(requestedString, "SEC"))
      *leadingEdgeUnit = HPE8311A_UNIT_SEC;
    else
    {
      if (!strcmp(requestedString, "PCT"))
        *leadingEdgeUnit = HPE8311A_UNIT_PCT;
      else
      {
        // the value got from the instrument is not within the allowed range
        *leadingEdgeUnit = HPE8311A_INVALID_VALUE;
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
      }
    }
  }
  else
    *leadingEdgeUnit = HPE8311A_UNIT_SEC;


  ERR_CHK_RETURN(errStatus);
}

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Trailing Edge
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeEdgeTrailing()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Trailing Edge
//---------------------------------------------------------------------------
// This function sets the instruments trailing-edge parameter.
//   "Trailing-Edge" means the transition-time of the pulse, measured
//   from 90% to 10% of pulse amplitude. Note: The value of trailing-edge
//   is independently programmable within certain ranges only.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeHold() to select the hold mode. Note:
//   the hold mode is set for leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeTrailingCoupling() to specify the
//   coupling mode between trailing-edge and leading-edge.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT SEC
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT PCT
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingEdgeUnit                                     ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit for provided value of trailingEdge:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | trailingEdgeValue                                    ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set trailingEdge on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeTrailing(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 trailingEdgeUnit,
    ViReal64 trailingEdgeValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeTrailing"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar valueString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Prepare the complete command to send to the instrument.
  completeCommand[0] = 0x00;

  switch (trailingEdgeUnit)
  {

  case HPE8311A_UNIT_SEC:
    strcpy(unitString, "SEC");
    break;

  case HPE8311A_UNIT_PCT:
    // make sure we've got seconds if we can't do percent
    if (thisPtr->options.trans_unit == 0)
      ERR_RETURN (VI_ERROR_PARAMETER3);

    strcpy(unitString, "PCT");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  mbFloatToString(trailingEdgeValue, valueString);

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN1:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }

    strcat(completeCommand, ":PULS:TRAN1:TRA ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " "); // do need additional unit!
    strcat(completeCommand, unitString);

    if (thisPtr->options.trans_unit != 0)
    {
      strcat(completeCommand, "; :PULS:TRAN2:UNIT ");
      strcat(completeCommand, unitString);
    }
    strcat(completeCommand, "; :PULS:TRAN2:TRA ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_1:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN1:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }
    strcat(completeCommand, ":PULS:TRAN1:TRA ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  case HPE8311A_CHANNEL_2:

    if (thisPtr->options.trans_unit != 0)
    {
      strcpy(completeCommand, ":PULS:TRAN2:UNIT ");
      strcat(completeCommand, unitString);
      strcat(completeCommand, "; ");
    }
    strcat(completeCommand, ":PULS:TRAN2:TRA ");
    strcat(completeCommand, valueString);
    strcat(completeCommand, " ");
    strcat(completeCommand, unitString);

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN(errStatus);
}

//*****************************************************************************
// hpe8311a_timeEdgeTrailing_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Trailing Edge
//---------------------------------------------------------------------------
// This function requests the trailing-edge parameter from the instrument.
//   "Trailing-Edge" means the transition-time of the pulse, measured
//   from 90% to 10% of pulse amplitude.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeHold_Q() to request the actual used
//   hold mode from the instrument. Note: this hold mode is valid for
//   leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// Use also function hpe8311a_timeEdgeTrailingCoupling() to request the
//   actual used coupling mode between trailing-edge and leading-edge
//   from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:UNIT?
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingEdgeUnit                                     ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for actual used value of trailingEdge:
//  |  | Range:
//  |  |  - HPE8311A_UNIT_SEC (second)
//  |  |  - HPE8311A_UNIT_PCT (percent)
//   ------------------------------------------------------------------------
//  | trailingEdgeValue                                    ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used value trailing-edge is set on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeTrailing_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 trailingEdgeUnit,
    ViPReal64 trailingEdgeValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeTrailing_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    if (thisPtr->options.trans_unit != 0)
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle,
        ":PULS:TRAN1:UNIT?",
        HPE8311A_STRING_LENGTH,
        requestedString
        );
    }
    

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TRAN1:TRA?",
      trailingEdgeValue
      );

    break;

  case HPE8311A_CHANNEL_2:

    if (thisPtr->options.trans_unit != 0)
    {
      errStatus = hpe8311a_cmdString_Q(
        instrumentHandle,
        ":PULS:TRAN2:UNIT?",
        HPE8311A_STRING_LENGTH,
        requestedString
        );
    }

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":PULS:TRAN2:TRA?",
      trailingEdgeValue
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (thisPtr->options.trans_unit != 0)
  {
    if (!strcmp(requestedString, "SEC"))
      *trailingEdgeUnit = HPE8311A_UNIT_SEC;
    else
    {
      if (!strcmp(requestedString, "PCT"))
        *trailingEdgeUnit = HPE8311A_UNIT_PCT;
      else
      {
        // the value got from the instrument is not within the allowed range
        *trailingEdgeUnit = HPE8311A_INVALID_VALUE;
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
      }
    }
  }
  else
  {
    *trailingEdgeUnit = HPE8311A_UNIT_SEC;
  }

  //-------------------------------------------------------------------------

  DBG_PRINT1("hpe8311a_timeEdgeTrailing_Q() returns: trailingEdgeUnit: %i\n", *trailingEdgeUnit);
  DBG_PRINT1("hpe8311a_timeEdgeTrailing_Q() returns: trailingEdgeValue: %e\n", *trailingEdgeValue);

  ERR_CHK_RETURN(errStatus);
}

//*****************************************************************************
// hpe8311a_timeEdgeTrailingCoupling()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Trailing Edge
//---------------------------------------------------------------------------
// This function sets the instruments coupling-mode parameter for automatic
//   coupling of the pulse trailing-edge transition-time to the leading-edge
//   transition-time. Note: this couples the edge-hold, the unit and the
//   value of the trailing-edge to the settings of the leading-edge.
//---------------------------------------------------------------------------
// There are two coupling-modes available:
//   - HPE8311A_COUPLING_OFF:
//     The trailing-edge transition time is automatically set to the same
//     value as the leading-edge, and is updated automatically each time
//     the leading-edge transition-time changes.
//   - HPE8311A_COUPLING_ON:
//     The trailing-edge transition time is independently programmable.
//   - HPE8311A_COUPLING_ONCE:
//     The trailing-edge transition time is set ONCE to the same value as the
//     leading-edge.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling:AUTO OFF
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling:AUTO ON
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling:AUTO ONCE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingEdgeCoupling                                 ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the coupling mode from trailingEdge to leadingEdge:
//  |  | Range:
//  |  |  - HPE8311A_COUPLING_OFF
//  |  |  - HPE8311A_COUPLING_ON
//  |  |  - HPE8311A_COUPLING_ONCE
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeTrailingCoupling(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 trailingEdgeCoupling
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeTrailingCoupling"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (trailingEdgeCoupling)
  {

  case HPE8311A_COUPLING_OFF:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_COUPLING_ON:
    strcpy(scpiParameter, "ON");
    break;

  case HPE8311A_COUPLING_ONCE:
    strcpy(scpiParameter, "ONCE");
    break;

  default:
    DBG_PRINT("failed: wrong Parameter 3!");
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:TRAN1:TRA:AUTO ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:TRAN2:TRA:AUTO ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRAN1:TRA:AUTO",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRAN2:TRA:AUTO",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN(errStatus);
}

//*****************************************************************************
// hpe8311a_timeEdgeTrailingCoupling_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Trailing Edge
//---------------------------------------------------------------------------
// This function requests the coupling-mode parameter for automatic coupling
//   of the pulse trailing-edge transition-time to the leading-edge
//   transition-time from the insrument. Note: this couples the edge-hold,
//   the unit and the value of the trailing-edge to the settings of the
//   leading-edge.
//---------------------------------------------------------------------------
// There are two coupling-modes available:
//   - HPE8311A_COUPLING_OFF:
//     The trailing-edge transition time is automatically set to the same
//     value as the leading-edge, and is updated automatically each time
//     the leading-edge transition-time changes.
//   - HPE8311A_COUPLING_ON:
//     The trailing-edge transition time is independently programmable.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:TRAiling:AUTO?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | trailingEdgeCoupling                                 ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used coupling mode
//  |  |   from trailingEdge to leadingEdge.
//  |  | Range:
//  |  |  - HPE8311A_COUPLING_OFF
//  |  |  - HPE8311A_COUPLING_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeTrailingCoupling_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 trailingEdgeCoupling
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeTrailingCoupling_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRAN1:TRA:AUTO?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRAN2:TRA:AUTO?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "0"))
    *trailingEdgeCoupling = HPE8311A_COUPLING_OFF;
  else
  {
    if (!strcmp(requestedString, "1"))
      *trailingEdgeCoupling = HPE8311A_COUPLING_ON;
    else
    {
      // the value got from the instrument is not within the allowed range
      *trailingEdgeCoupling = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *trailingEdgeCoupling);
  ERR_CHK_RETURN(errStatus);
}

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Time - Edge Hold
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_timeEdgeHold()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Edge Hold
//---------------------------------------------------------------------------
// This function sets the instruments edge-hold parameter. Note: this hold
//   mode is valid for leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// There are two hold-modes for edge available:
//   - HPE8311A_HOLD_ABSOLUTE
//     The absolute transition-time measured from 10% to 90% of pulse
//     amplitude. In this format the edge is independent of the pulse-width.
//   - HPE8311A_HOLD_RATIO
//     The absolute transition-time expressed as a percentage of pulse-width.
//     In this format if you adjust the pulse-width, the transition-time is
//     adjusted to maintain the edge-time as a percentage of the width.
//     You cannot have the width format set to dutycycle and the leading/
//     trailing-edge format set to percentage of width at the same time.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRANsition[1|2]:HOLD TIME
//   [:SOURce]:PULSe:TRANsition[1|2]:HOLD WRATio
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | edgeHoldMode                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the actual hold-mode for leading and trailing Edge:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeHold(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 edgeHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeHold"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (edgeHoldMode)
  {

  case HPE8311A_HOLD_ABSOLUTE:
    strcpy(scpiParameter, "TIME");
    break;

  case HPE8311A_HOLD_RATIO:
    strcpy(scpiParameter, "WRAT");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":PULS:TRAN1:HOLD ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:TRAN2:HOLD ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRAN1:HOLD",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRAN2:HOLD",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN(errStatus);
}

//*****************************************************************************
// hpe8311a_timeEdgeHold_Q()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS - Edge Hold
//---------------------------------------------------------------------------
// This function requests the edge-hold parameter from the instrument. Note:
//   this hold mode is valid for leading-edge and trailing-edge in one time.
//---------------------------------------------------------------------------
// There are two hold-modes for edge available:
//   - HPE8311A_HOLD_ABSOLUTE
//     The absolute transition-time measured from 10% to 90% of pulse
//     amplitude. In this format the edge is independent of the pulse-width.
//   - HPE8311A_HOLD_RATIO
//     The absolute transition-time expressed as a percentage of pulse-width.
//     In this format if you adjust the pulse-width, the transition-time is
//     adjusted to maintain the edge-time as a percentage of the width.
//     You cannot have the width format set to dutycycle and the leading/
//     trailing-edge format set to percentage of width at the same time.
//---------------------------------------------------------------------------
// Used scpi-command:
//   [:SOURce]:PULSe:TRANsition[1|2]:HOLD?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | edgeHoldMode                                         ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used hold-mode for leading and trailing-edge:
//  |  | Range:
//  |  |  - HPE8311A_HOLD_ABSOLUTE
//  |  |  - HPE8311A_HOLD_RATIO
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeEdgeHold_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 edgeHoldMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeEdgeHold_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRAN1:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRAN2:HOLD?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "TIME"))
    *edgeHoldMode = HPE8311A_HOLD_ABSOLUTE;
  else
  {
    if (!strcmp(requestedString, "WRAT"))
      *edgeHoldMode = HPE8311A_HOLD_RATIO;
    else
    {
      // the value got from the instrument is not within the allowed range
      *edgeHoldMode = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *edgeHoldMode);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_timeCalibration()
//*****************************************************************************
// CATEGORY: TIMING-FUNCTIONS
//---------------------------------------------------------------------------
// This function recalibrates the timing system. The timing-circuits for
//   VCO-period, delay and width are calibrated in reference to the internal
//   PPL-reference. When the device is switched off and on again, the factory
//   calibration data are activated again.
// The function returns, whether the desired calibration has passed.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :CALibration[:ALL]
//   :CALibration[:ALL]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_timeCalibration(
    ViSession instrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_timeCalibration"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt32 calibrationResult;
  ViInt8 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;
  ViInt32 oldTimeOut = HPE8311A_INVALID_VALUE;
  ViBoolean timeOutChanged = VI_FALSE;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  stateMachine = STATE_MACHINE_REQUEST_TIMEOUT;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_REQUEST_TIMEOUT:

      errStatus = hpe8311a_timeOut_Q(
        instrumentHandle,
        &oldTimeOut
        );

      stateMachine = STATE_MACHINE_SET_NEW_TIMEOUT;

      break;

    case STATE_MACHINE_SET_NEW_TIMEOUT:

      errStatus = hpe8311a_timeOut(
        instrumentHandle,
        60000
        );

      if (errStatus == VI_SUCCESS)
        timeOutChanged = VI_TRUE;

      stateMachine = STATE_MACHINE_CALIBRATE_INSTRUMENT;

      break;

    case STATE_MACHINE_CALIBRATE_INSTRUMENT:

      errStatus = hpe8311a_cmdInt32_Q(
        instrumentHandle,
        ":CAL?",
        &calibrationResult
        );

      if (calibrationResult != 0)
        errStatus = HPE8311A_INSTR_ERROR_CALIBRATION_FAILED;

      stateMachine = STATE_MACHINE_RESTORE_OLD_TIMEOUT;

      break;

    case STATE_MACHINE_RESTORE_OLD_TIMEOUT:

      errStatus = hpe8311a_timeOut(
        instrumentHandle,
        oldTimeOut
        );

      stateMachine = STATE_MACHINE_DO_FINISH;

      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      // try to write it back, without verifying ERR_RETURN-value
      (void) hpe8311a_timeOut(
        instrumentHandle,
        oldTimeOut
        );

      stateMachine = STATE_MACHINE_DO_FINISH;
      stateMachineErrorOccurred = VI_TRUE;
      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  ERR_CHK_RETURN (errStatus);
}
//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// LEVEL-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//*****************************************************************************
// hpe8311a_levelHighLow()
//*****************************************************************************
// CATEGORY: LEVEL-FUNCTIONS - High/Low-Level
//---------------------------------------------------------------------------
// This function programs the high-level and low-level of the OUTPUT signal
//   in current or voltage.
// This enables either of the voltage or the current subsystems for the
//   specified channel.
//---------------------------------------------------------------------------
// The available current range is limited by the combination of:
//   - Specified current or voltage limits
//     hpe8311a_limit()
//   - Actual OUTPUT Impedance setting:
//     hpe8311a_configOutputImpedance()
//   - Actual Expected Load impedance setting:
//     hpe8311a_configOutputLoadCompensation()
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:HOLD[1|2] VOLT
//   [:SOURce]:HOLD[1|2] CURR
//   [:SOURce]:CURRent[1|2]:LEVel:HIGH <value>
//   [:SOURce]:CURRent[1|2]:LEVel:LOW  <value>
//   [:SOURce]:VOLTage[1|2]:LEVel:HIGH <value>
//   [:SOURce]:VOLTage[1|2]:LEVel:LOW  <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | levelUnit                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit of the level-values:
//  |  |  - HPE8311A_LEVELS_VOLT
//  |  |  - HPE8311A_LEVELS_AMPERE
//   ------------------------------------------------------------------------
//  | levelValueHigh                                       ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set high-level on
//   ------------------------------------------------------------------------
//  | levelValueLow                                        ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set low-level on
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_levelHighLow(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 levelUnit,
    ViReal64 levelValueHigh,
    ViReal64 levelValueLow
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_levelHighLow"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar levelStringHigh[HPE8311A_STRING_LENGTH];
  ViChar levelStringLow[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  mbFloatToString(levelValueHigh, levelStringHigh);
  mbFloatToString(levelValueLow, levelStringLow);

  switch (levelUnit)
  {

  case HPE8311A_LEVELS_VOLT:
    strcpy(unitString, "VOLT");
    break;

  case HPE8311A_LEVELS_AMPERE:
    strcpy(unitString, "CURR");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch(levelUnit)

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":HOLD1 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:HIGH ");
    strcat(completeCommand, levelStringHigh);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:LOW ");
    strcat(completeCommand, levelStringLow);

    strcat(completeCommand, "; :HOLD2 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:HIGH ");
    strcat(completeCommand, levelStringHigh);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:LOW ");
    strcat(completeCommand, levelStringLow);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    strcpy(completeCommand, ":HOLD1 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:HIGH ");
    strcat(completeCommand, levelStringHigh);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:LOW ");
    strcat(completeCommand, levelStringLow);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_2:

    strcpy(completeCommand, ":HOLD2 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:HIGH ");
    strcat(completeCommand, levelStringHigh);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:LOW ");
    strcat(completeCommand, levelStringLow);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch(channel)

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_levelHighLow_Q()
//*****************************************************************************
// CATEGORY: LEVEL-FUNCTIONS - High/Low-Level
//---------------------------------------------------------------------------
// This function requests the actual used high-level and low-level in
//   current or voltage of the OUTPUT signal from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:HOLD[1|2]?
//   [:SOURce]:CURRent[1|2]:LEVel:HIGH?
//   [:SOURce]:CURRent[1|2]:LEVel:LOW?
//   [:SOURce]:VOLTage[1|2]:LEVel:HIGH?
//   [:SOURce]:VOLTage[1|2]:LEVel:LOW?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | levelUnit                                            ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for the level-values:
//  |  |  - HPE8311A_LEVELS_VOLT
//  |  |  - HPE8311A_LEVELS_AMPERE
//   ------------------------------------------------------------------------
//  | levelValueHigh                                       ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the value of the actual used high-level.
//   ------------------------------------------------------------------------
//  | levelValueLow                                        ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the value of the actual used low-level.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_levelHighLow_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 levelUnit,
    ViPReal64 levelValueHigh,
    ViPReal64 levelValueLow
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_levelHighLow_Q"


  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":HOLD1?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    // Analyze the requested value
    if (!strcmp(requestedString, "VOLT"))
      *levelUnit = HPE8311A_LEVELS_VOLT;
    else
    {
      if (!strcmp(requestedString, "CURR"))
        *levelUnit = HPE8311A_LEVELS_AMPERE;
      else
      {
        // the value got from the instrument is not within the allowed range
        *levelUnit = HPE8311A_INVALID_VALUE;
        ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
      }
    }

    switch (*levelUnit)
    {

    case HPE8311A_LEVELS_VOLT:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT1:LEV:HIGH?",
        levelValueHigh
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT1:LEV:LOW?",
        levelValueLow
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    case HPE8311A_LEVELS_AMPERE:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR1:LEV:HIGH?",
        levelValueHigh
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR1:LEV:LOW?",
        levelValueLow
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    default:
      ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);

    }                           // END switch(levelUnit)

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":HOLD2?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    // Analyze the requested value
    if (!strcmp(requestedString, "VOLT"))
      *levelUnit = HPE8311A_LEVELS_VOLT;
    else
    {
      if (!strcmp(requestedString, "CURR"))
        *levelUnit = HPE8311A_LEVELS_AMPERE;
      else
      {
        // the value got from the instrument is not within the allowed range
        *levelUnit = HPE8311A_INVALID_VALUE;
        ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
      }
    }

    switch (*levelUnit)
    {

    case HPE8311A_LEVELS_VOLT:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT2:LEV:HIGH?",
        levelValueHigh
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT2:LEV:LOW?",
        levelValueLow
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    case HPE8311A_LEVELS_AMPERE:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR2:LEV:HIGH?",
        levelValueHigh
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR2:LEV:LOW?",
        levelValueLow
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    default:
      ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);

    }                           // END switch(levelUnit)

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch(channel)

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *levelUnit);
  TRACE_VAR("%e", *levelValueHigh);
  TRACE_VAR("%e", *levelValueLow);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_levelOffsetAmplitude()
//*****************************************************************************
// CATEGORY: LEVEL-FUNCTIONS - Offset/Amplitude-Level
//---------------------------------------------------------------------------
// This function programs the offset-level and the amplitude-level in current
//   or voltage of the OUTPUT signal.
// This enables either of the voltage or the current subsystems for the
//   specified channel.
//---------------------------------------------------------------------------
// The available current range is limited by the combination of:
//   - Specified ampere or voltage limits
//     hpe8311a_limit()
//   - Actual OUTPUT Impedance setting:
//     hpe8311a_configOutputImpedance()
//   - Actual Expected Load impedance setting:
//     hpe8311a_configOutputLoadCompensation()
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:HOLD[1|2] VOLT
//   [:SOURce]:HOLD[1|2] CURR
//   [:SOURce]:CURRent[1|2]:LEVel:AMPLitude <value>
//   [:SOURce]:CURRent[1|2]:LEVel:OFFSet    <value>
//   [:SOURce]:VOLTage[1|2]:LEVel:AMPLitude <value>
//   [:SOURce]:VOLTage[1|2]:LEVel:OFFSet    <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | levelUnit                                            ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the unit of the level-values:
//  |  |  - HPE8311A_LEVELS_VOLT
//  |  |  - HPE8311A_LEVELS_AMPERE
//   ------------------------------------------------------------------------
//  | levelValueOffset                                     ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set offset on
//   ------------------------------------------------------------------------
//  | levelValueAmplitude                                  ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set amplitude on
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_levelOffsetAmplitude(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 levelUnit,
    ViReal64 levelValueOffset,
    ViReal64 levelValueAmplitude
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_levelOffsetAmplitude"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar unitString[HPE8311A_STRING_LENGTH];
  ViChar levelStringOffset[HPE8311A_STRING_LENGTH];
  ViChar levelStringAmplitude[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  mbFloatToString(levelValueOffset, levelStringOffset);
  mbFloatToString(levelValueAmplitude, levelStringAmplitude);

  switch (levelUnit)
  {

  case HPE8311A_LEVELS_VOLT:
    strcpy(unitString, "VOLT");
    break;

  case HPE8311A_LEVELS_AMPERE:
    strcpy(unitString, "CURR");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch(levelUnit)

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":HOLD1 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:AMPL ");
    strcat(completeCommand, levelStringAmplitude);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:OFFS ");
    strcat(completeCommand, levelStringOffset);

    strcat(completeCommand, "; :HOLD2 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:AMPL ");
    strcat(completeCommand, levelStringAmplitude);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:OFFS ");
    strcat(completeCommand, levelStringOffset);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    strcpy(completeCommand, ":HOLD1 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:AMPL ");
    strcat(completeCommand, levelStringAmplitude);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "1:LEV:OFFS ");
    strcat(completeCommand, levelStringOffset);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_2:

    strcpy(completeCommand, ":HOLD2 ");
    strcat(completeCommand, unitString);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:AMPL ");
    strcat(completeCommand, levelStringAmplitude);

    strcat(completeCommand, "; :");
    strcat(completeCommand, unitString);
    strcat(completeCommand, "2:LEV:OFFS ");
    strcat(completeCommand, levelStringOffset);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch(channel)

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_levelOffsetAmplitude_Q()
//*****************************************************************************
// CATEGORY: LEVEL-FUNCTIONS - Offset/Amplitude-Level
//---------------------------------------------------------------------------
// This function requests the actual used offset-level and the amplitude-
//   level in current or voltage of the OUTPUT signal from the instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:HOLD[1|2]?
//   [:SOURce]:CURRent[1|2]:LEVel:AMPLitude?
//   [:SOURce]:CURRent[1|2]:LEVel:OFFSet?
//   [:SOURce]:VOLTage[1|2]:LEVel:AMPLitude?
//   [:SOURce]:VOLTage[1|2]:LEVel:OFFSet?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | levelUnit                                            ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used unit for the level-values:
//  |  |  - HPE8311A_LEVELS_VOLT
//  |  |  - HPE8311A_LEVELS_AMPERE
//   ------------------------------------------------------------------------
//  | levelValueOffset                                     ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the value of the actual used offset.
//   ------------------------------------------------------------------------
//  | levelValueAmplitude                                  ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the value of the actual used amplitude.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_levelOffsetAmplitude_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 levelUnit,
    ViPReal64 levelValueOffset,
    ViPReal64 levelValueAmplitude
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_levelOffsetAmplitude_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":HOLD1?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    // Analyze the requested value
    if (!strcmp(requestedString, "VOLT"))
      *levelUnit = HPE8311A_LEVELS_VOLT;
    else
    {
      if (!strcmp(requestedString, "CURR"))
        *levelUnit = HPE8311A_LEVELS_AMPERE;
      else
      {
        // the value got from the instrument is not within the allowed range
        *levelUnit = HPE8311A_INVALID_VALUE;
        ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
      }
    }

    switch (*levelUnit)
    {

    case HPE8311A_LEVELS_VOLT:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT1:LEV:OFFS?",
        levelValueOffset
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT1:LEV:AMPL?",
        levelValueAmplitude
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    case HPE8311A_LEVELS_AMPERE:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR1:LEV:OFFS?",
        levelValueOffset
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR1:LEV:AMPL?",
        levelValueAmplitude
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    default:
      ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);

    }                           // END switch( *levelUnit )

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":HOLD2?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    if (errStatus < VI_SUCCESS)
      break;

    // Analyze the requested value
    if (!strcmp(requestedString, "VOLT"))
      *levelUnit = HPE8311A_LEVELS_VOLT;
    else
    {
      if (!strcmp(requestedString, "CURR"))
        *levelUnit = HPE8311A_LEVELS_AMPERE;
      else
      {
        // the value got from the instrument is not within the allowed range
        *levelUnit = HPE8311A_INVALID_VALUE;
        ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
      }
    }

    switch (*levelUnit)
    {

    case HPE8311A_LEVELS_VOLT:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT2:LEV:OFFS?",
        levelValueOffset
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":VOLT2:LEV:AMPL?",
        levelValueAmplitude
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    case HPE8311A_LEVELS_AMPERE:

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR2:LEV:OFFS?",
        levelValueOffset
        );

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = hpe8311a_cmdReal64_Q(
        instrumentHandle,
        ":CURR2:LEV:AMPL?",
        levelValueAmplitude
        );

      if (errStatus < VI_SUCCESS)
        break;

      break;

    default:
      ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);

    }                           // END switch( *levelUnit )

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch(channel)

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *levelUnit);
  TRACE_VAR("%e", *levelValueOffset);
  TRACE_VAR("%e", *levelValueAmplitude);
  ERR_CHK_RETURN (errStatus);
}

//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// PATTERN-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//*****************************************************************************
// hpe8311a_patternUpdate()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Update
//---------------------------------------------------------------------------
// This function sets the instruments parameter for updating the pattern.
//   "Update" means the the automatic updating of the pattern generating
//   hardware following a hpe8311a_patternDataDownloadXXX() function-call.
//---------------------------------------------------------------------------
// Disable the automatic updating if you want to set up new pattern
//   data in the instrument without affecting the pattern which is currently
//   being generated. You can then update the hardware with the new pattern
//   data by calling hpe8311a_patternUpdate( iHandle, HPE8311A_UPDATE_ONCE ).
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :DIGital[:STIMulus]:PATTern:UPDate OFF
//   :DIGital[:STIMulus]:PATTern:UPDate ON
//   :DIGital[:STIMulus]:PATTern:UPDate ONCE
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | patternUpdateMode                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select pattern update mode
//  |  | Range:
//  |  |  - HPE8311A_UPDATE_OFF
//  |  |  - HPE8311A_UPDATE_ONCE
//  |  |  - HPE8311A_UPDATE_CONTINUOUS
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternUpdate(
    ViSession instrumentHandle,
    ViUInt16 patternUpdateMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternUpdate"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (patternUpdateMode)
  {

  case HPE8311A_UPDATE_OFF:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_UPDATE_ONCE:
    strcpy(scpiParameter, "ONCE");
    break;

  case HPE8311A_UPDATE_CONTINUOUS:
    strcpy(scpiParameter, "ON");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch(patternUpdateMode)

  errStatus = hpe8311a_cmdString(
    instrumentHandle,
    ":DIG:PATT:UPD",
    scpiParameter
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternUpdate_Q()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Update
//---------------------------------------------------------------------------
// This function requests the actual used parameter for updating the pattern.
//   from the instrument. "Update" means the the automatic updating of the
//   pattern generating hardware following a hpe8311a_patternDataDownloadXX()
//   function-call.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:UPDate?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | patternUpdateMode                                    ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the selected pattern update mode from the instrument.
//  |  | Range:
//  |  |  - HPE8311A_UPDATE_OFF
//  |  |  - HPE8311A_UPDATE_CONTINUOUS
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternUpdate_Q(
    ViSession instrumentHandle,
    ViPUInt16 patternUpdateMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternUpdate_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  errStatus = hpe8311a_cmdString_Q(
    instrumentHandle,
    ":DIG:PATT:UPD?",
    HPE8311A_STRING_LENGTH,
    requestedString
    );

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "OFF"))
    *patternUpdateMode = HPE8311A_UPDATE_OFF;
  else
  {
    if (!strcmp(requestedString, "ON"))
      *patternUpdateMode = HPE8311A_UPDATE_CONTINUOUS;
    else
    {
      // the value got from the instrument is not within the allowed range
      *patternUpdateMode = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *patternUpdateMode);
  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternDataDownloadBuffer()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Download
//---------------------------------------------------------------------------
// This function sets the pattern provided by the user in a data-buffer
//   to one or all channels starting from Bit 1.
//---------------------------------------------------------------------------
// The <IEEE-data> are an arbitrary block of program data as defined in
//   IEEE 488.2 7.7.6.2, for example:
//   - #1541213
//     # start of block
//     1 Length of the length of the data
//     5 Length of the data
//     41213 5 bytes of data
//   - #2161000100010001000
//     # start of block
//     2 Length of the length of the data
//     16 Length of the data
//     10...00 16 bytes of data
//---------------------------------------------------------------------------
// Note: the data provided in the data-buffer must not contain the header
//   like described before. Only the raw pattern-data are needed - the
//   IEEE conform header is added automatically by this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:DATA[1|2|3] <IEEE-data>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_ALL
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | dataBuffer                                           ViPUInt8        IN
//  |   ---------------------------------------------------------------------
//  |  | Pointer to the buffer containing the data to transfer into the
//  |  | instrument. Every pattern-bit uses one byte within the buffer.
//   ------------------------------------------------------------------------
//  | patternLength                                        ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Length of the Provided pattern,
//  |  | that means also size of the Provided buffer in bytes.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataDownloadBuffer(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt8 dataBuffer,
    ViUInt32 patternLength
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataDownloadBuffer"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_DATA_CHANNEL_ALL:

    errStatus = hpe8311a_cmdInt8Arr(
      instrumentHandle,
      "DIG:PATT:DATA",
      (ViInt8 *) dataBuffer,
      (ViInt32) patternLength
      );

    break;

  case HPE8311A_DATA_CHANNEL_1:

    errStatus = hpe8311a_cmdInt8Arr(
      instrumentHandle,
      "DIG:PATT:DATA1",
      (ViInt8 *) dataBuffer,
      (ViInt32) patternLength
      );

    break;

  case HPE8311A_DATA_CHANNEL_2:

    errStatus = hpe8311a_cmdInt8Arr(
      instrumentHandle,
      "DIG:PATT:DATA2",
      (ViInt8 *) dataBuffer,
      (ViInt32) patternLength
      );

    break;

  case HPE8311A_DATA_CHANNEL_3:

    errStatus = hpe8311a_cmdInt8Arr(
      instrumentHandle,
      "DIG:PATT:DATA3",
      (ViInt8 *) dataBuffer,
      (ViInt32) patternLength
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternDataDownloadBuffer_Q()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Download
//---------------------------------------------------------------------------
// This function requests the actual used pattern from one instruments
//   channels starting from Bit 1 and writes this data into the specified
//   memory-buffer.
//---------------------------------------------------------------------------
// The <IEEE-data> are an arbitrary block of program data as defined in
//   IEEE 488.2 7.7.6.2, for example:
//   - #1541213
//     # start of block
//     1 Length of the length of the data
//     5 Length of the data
//     41213 5 bytes of data
//   - #2161000100010001000
//     # start of block
//     2 Length of the length of the data
//     16 Length of the data
//     10...00 16 bytes of data
//---------------------------------------------------------------------------
// Note: the data provided in the data-buffer do not contain the header
//   like described before. Only the raw pattern-data are supported - the
//   IEEE conform header is removed automatically by this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:DATA[1|2|3]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | dataBuffer                                           ViPUInt8       OUT
//  |   ---------------------------------------------------------------------
//  |  | Pointer to the buffer to write the requested instrument data to.
//  |  | Every pattern-bit uses one byte within the buffer.
//   ------------------------------------------------------------------------
//  | patternLengthReturned                                ViPUInt32      OUT
//  |   ---------------------------------------------------------------------
//  |  | Quantity of valid bytes returned by the instrument and
//  |  |   written into the specified buffer.
//   ------------------------------------------------------------------------
//  | bufferSize                                           ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Size of the provided buffer in bytes.
//   ------------------------------------------------------------------------
//  | convertBinaryToAscii                                 ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | The expected data to read from the instrument will have binary
//  |  |   format and need to be transformed into ASCII-format, before
//  |  |   written into the provided buffer:
//  |  | Range:
//  |  |  - HPE8311A_CONVERT_OFF
//  |  |  - HPE8311A_CONVERT_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataDownloadBuffer_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt8 dataBuffer,
    ViPUInt32 patternLengthReturned,
    ViUInt32 bufferSize,
    ViUInt16 convertBinaryToAscii
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataDownloadBuffer_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViUInt32 count;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_DATA_CHANNEL_ALL:

    errStatus = hpe8311a_cmdInt8Arr_Q(
      instrumentHandle,
      ":DIG:PATT:DATA?",
      (ViInt32) bufferSize,
      (ViPChar) dataBuffer,
      (ViInt32 *) patternLengthReturned
      );

    break;

  case HPE8311A_DATA_CHANNEL_1:

    errStatus = hpe8311a_cmdInt8Arr_Q(
      instrumentHandle,
      ":DIG:PATT:DATA1?",
      (ViInt32) bufferSize,
      (ViPChar) dataBuffer,
      (ViInt32 *) patternLengthReturned
      );

    break;

  case HPE8311A_DATA_CHANNEL_2:

    errStatus = hpe8311a_cmdInt8Arr_Q(
      instrumentHandle,
      ":DIG:PATT:DATA2?",
      (ViInt32) bufferSize,
      (ViPChar) dataBuffer,
      (ViInt32 *) patternLengthReturned
      );

    break;

  case HPE8311A_DATA_CHANNEL_3:

    errStatus = hpe8311a_cmdInt8Arr_Q(
      instrumentHandle,
      ":DIG:PATT:DATA3?",
      (ViInt32) bufferSize,
      (ViPChar) dataBuffer,
      (ViInt32 *) patternLengthReturned
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch(channel)

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
  {
    dataBuffer[0] = 0;
    *patternLengthReturned = 0;

    ERR_RETURN (errStatus);
  }

  //-------------------------------------------------------------------------
  // Convert binary-data to ascii-data when so requested.

  switch (convertBinaryToAscii)
  {

  case HPE8311A_CONVERT_OFF:
    // nothing to do!
    break;

  case HPE8311A_CONVERT_ON:

    for (count = 0; count < (*patternLengthReturned); count++)
      dataBuffer[count] += 0x30;

    break;

  default:
    errStatus = VI_ERROR_PARAMETER6;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *patternLengthReturned);
  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
#define PATTERN_FILE_HEADER_ROWS 7
#define PATTERN_FILE_HEADER_STRING_0 "#==============================================================================\n"
#define PATTERN_FILE_HEADER_STRING_1 "# This is a pattern-file for Hewlett-Packard HPE8311A instruments!\n"
#define PATTERN_FILE_HEADER_STRING_2 "#==============================================================================\n"
#define PATTERN_FILE_HEADER_STRING_3 "# The file may be edited by the user!\n"
#define PATTERN_FILE_HEADER_STRING_4 "#   Use >#< for comments!\n"
#define PATTERN_FILE_HEADER_STRING_5 "#   Use max. 80 chars in a line!\n"
#define PATTERN_FILE_HEADER_STRING_6 "#==============================================================================\n"
#define PATTERN_FILE_CULOMS_ALLOWED 80

//*****************************************************************************
// hpe8311a_patternDataDownloadFile()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Download
//---------------------------------------------------------------------------
// This function sets the pattern provided by the user in a file to one
//   or all channels starting from Bit 1.
//---------------------------------------------------------------------------
// The <IEEE-data> are an arbitrary block of program data as defined in
//   IEEE 488.2 7.7.6.2, for example:
//   - #1541213
//     # start of block
//     1 Length of the length of the data
//     5 Length of the data
//     41213 5 bytes of data
//   - #2161000100010001000
//     # start of block
//     2 Length of the length of the data
//     16 Length of the data
//     10...00 16 bytes of data
//---------------------------------------------------------------------------
// Note: the data provided in the file must not contain the header
//   like described before. Only the raw pattern-data are needed - the
//   IEEE conform header is added automatically by this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:DATA[1|2|3] <IEEE-data>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_ALL
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | fileName                                             ViString        IN
//  |   ---------------------------------------------------------------------
//  |  | File containing the data to transfer into the instrument.
//  |  | Every pattern-bit uses one byte within the file.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataDownloadFile(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViString fileName
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataDownloadFile"


  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  FILE *patternFile = NULL;
  ViChar gotChar;
  ViUInt32 patternLength = 0;
  ViUInt8 *patternBuffer = NULL;
  ViUInt32 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;
  ViBoolean remarkDetected = VI_FALSE;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (fileName == NULL)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(fileName) == 0)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(fileName) > HPE8311A_STRING_LENGTH)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  stateMachine = STATE_MACHINE_OPEN_FILE;
  patternFile = NULL;
  patternBuffer = NULL;

  do
  {

    switch (stateMachine)
    {
    case STATE_MACHINE_OPEN_FILE:

      patternFile = fopen(fileName, "rb");

      if (patternFile == NULL)
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE;

      stateMachine = STATE_MACHINE_MALLOC_MEMORY;
      break;

    case STATE_MACHINE_MALLOC_MEMORY:

      // Malloc space on heap
      patternBuffer = (ViUInt8 *) malloc(60 * 1024 * sizeof(ViUInt8));

      if (patternBuffer == NULL)
        errStatus = VI_ERROR_ALLOC;

      stateMachine = STATE_MACHINE_PREPARE_DATA;
      break;

    case STATE_MACHINE_PREPARE_DATA:

      // Copy file-contents into the new buffer
      patternLength = 0;

      /*lint -e668 -e613 poss. null ptr (patternFile) */

      while (
        (EOF != (gotChar = (ViChar) fgetc(patternFile)))
        && (VI_SUCCESS == errStatus)
        )
      {

        if (remarkDetected)
        {
          //-----------------------------------------------------------

          // We are in the remark-mode and skip all chars until
          // a new line occures.

          if (gotChar == '\n')
            remarkDetected = VI_FALSE;
        }
        else
        {
          //-----------------------------------------------------------

          // We are not in the remark mode and have to analyze
          // the chars. But we only accept those chars we are
          // interested in;
          // 0  <= gotChar <=  7
          // '0' <= gotChar <= '7'

          // We are not interested in all the others except
          // '\n' and '#'

          if (gotChar != '\n')
          {
            if (gotChar == '#')
            {
              remarkDetected = VI_TRUE;
            }
            else
            {

              if (
                ((0 <= gotChar) && (gotChar <= 7)) ||
                (('0' <= gotChar) && (gotChar <= '7'))
                )
              {
                patternBuffer[patternLength++] = (ViUInt8) gotChar;

                if (patternLength > (60 * 1024 - 1))
                {
                  errStatus = HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_TOO_LARGE;
                }
              }
            }
          }
        }
      } // END while()

      if (patternLength == 0)
        errStatus = HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_EMPTY;

      /*lint +e668 +e613 */

      stateMachine = STATE_MACHINE_GENERAL_COMMANDS;
      break;

    case STATE_MACHINE_GENERAL_COMMANDS:

      switch (channel)
      {

      case HPE8311A_DATA_CHANNEL_ALL:
      case HPE8311A_DATA_CHANNEL_1:
      case HPE8311A_DATA_CHANNEL_2:
      case HPE8311A_DATA_CHANNEL_3:

        errStatus = hpe8311a_patternDataDownloadBuffer(
          instrumentHandle,
          channel,
          patternBuffer,
          patternLength
          );

        break;

      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;

      }                         // END switch()

      // Send the scpi-commands to the instrument to perform the query

      stateMachine = STATE_MACHINE_CLOSE_FILE;
      break;

    case STATE_MACHINE_CLOSE_FILE:

      // Close file
      if (patternFile)
      {
        // only report an error if we don't already have one
        if (EOF == fclose(patternFile) && errStatus == VI_SUCCESS)
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE;
      }

      stateMachine = STATE_MACHINE_FREE_MEMORY;
      break;

    case STATE_MACHINE_FREE_MEMORY:

      FREE(patternBuffer);
      stateMachine = STATE_MACHINE_DO_FINISH;
      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      stateMachine = STATE_MACHINE_CLOSE_FILE;
      stateMachineErrorOccurred = VI_TRUE;
      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternDataDownloadFile_Q()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Download
//---------------------------------------------------------------------------
// This function requests the actual used pattern from one instruments
//   channels starting from Bit 1 and writes the requested data into the
//   specified file.
//---------------------------------------------------------------------------
// The <IEEE-data> are an arbitrary block of program data as defined in
//   IEEE 488.2 7.7.6.2, for example:
//   - #1541213
//     # start of block
//     1 Length of the length of the data
//     5 Length of the data
//     41213 5 bytes of data
//   - #2161000100010001000
//     # start of block
//     2 Length of the length of the data
//     16 Length of the data
//     10...00 16 bytes of data
//---------------------------------------------------------------------------
// Note: the data provided by this function in the file do not contain the
//   header like described before. Only the raw pattern-data are supported -
//   the IEEE conform header is removed automatically by this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:DATA[1|2|3]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | fileName                                             ViString        IN
//  |   ---------------------------------------------------------------------
//  |  | File to write to the data transfered out of the instrument.
//  |  | Every pattern-bit uses one byte within the file.
//   ------------------------------------------------------------------------
//  | convertBinaryToAscii                                 ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | The expected data to read from the instrument will have binary
//  |  |   format and need to be transformed into ASCII-format, before
//  |  |   written into the provided file:
//  |  | Range:
//  |  |  - HPE8311A_CONVERT_OFF
//  |  |  - HPE8311A_CONVERT_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataDownloadFile_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViString fileName,
    ViUInt16 convertBinaryToAscii
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataDownloadFile_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  FILE *patternFile;
  ViUInt8 *patternBuffer;
  ViUInt32 patternLengthReturned = 0;
  ViUInt32 count;
  ViInt8 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;
  ViInt8 actualColumn;
  ViChar actualString[HPE8311A_STRING_LENGTH];
  ViChar actualTimeDate[HPE8311A_STRING_LENGTH];

  ViChar *fileHeader[PATTERN_FILE_HEADER_ROWS] = {
    PATTERN_FILE_HEADER_STRING_0,
    PATTERN_FILE_HEADER_STRING_1,
    PATTERN_FILE_HEADER_STRING_2,
    PATTERN_FILE_HEADER_STRING_3,
    PATTERN_FILE_HEADER_STRING_4,
    PATTERN_FILE_HEADER_STRING_5,
    PATTERN_FILE_HEADER_STRING_6
  };

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (fileName == NULL)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(fileName) == 0)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(fileName) > HPE8311A_STRING_LENGTH)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  stateMachine = STATE_MACHINE_OPEN_FILE;
  patternFile = NULL;
  patternBuffer = NULL;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_OPEN_FILE:

      //---------------------------------------------------------------
      // Create the new file
      patternFile = fopen(fileName, "wb");

      if (patternFile == NULL)
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE;
        break;
      }

      //---------------------------------------------------------------
      // Write file-header into the new file
      for (count = 0; count < PATTERN_FILE_HEADER_ROWS; count++)
      {
        if (fputs(fileHeader[count], patternFile) == EOF)
        {
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
          break;
        }
      }

      //---------------------------------------------------------------
      // Write device information into the file

      // init for error
      errStatus = HPE8311A_INSTR_ERROR_NOT_SUPPORTED_INSTR_TYPE;

      for (count = 0; count < HPE8311A_IDN_SUPPORTED; count++)
      {
        if ((identityStcI[count]).identityNumber == thisPtr->deviceIdentity)
        {
          sprintf(actualString, "# DEVICE: %s \n", (identityStcI[count]).identityStcing);

          if (fputs(actualString, patternFile) == EOF)
          {
            errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
          }
          else
          {
            errStatus = VI_SUCCESS;
          }
          break;
        }
      }

      if(errStatus < VI_SUCCESS)
        break;

      //---------------------------------------------------------------
      // Write additional information into the file
      // TIME

#if defined (_WIN32) && !defined (_CVI_)

      (void) _strtime(actualTimeDate);
      sprintf(actualString, "# TIME: %s \n", actualTimeDate);

      if (fputs(actualString, patternFile) == EOF)
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

#endif          /* _WIN32 */

      //---------------------------------------------------------------
      // Write additional information into the file
      // DATE

#if defined (_WIN32) && !defined (_CVI_)

      (void) _strdate(actualTimeDate);
      sprintf(actualString, "# DATE: %s \n", actualTimeDate);

      if (fputs(actualString, patternFile) == EOF)
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

#endif          /* _WIN32 */

      //---------------------------------------------------------------
      // Write additional information into the file
      // CHANNEL

      sprintf(actualString, "# CHANNEL: %u \n", channel);

      if (fputs(actualString, patternFile) == EOF)
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

      if (fputs(fileHeader[0], patternFile) == EOF)
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

      stateMachine = STATE_MACHINE_MALLOC_MEMORY;

      break;

    case STATE_MACHINE_MALLOC_MEMORY:

      // Malloc space on heap
      patternBuffer = (ViUInt8 *) (malloc(60 * 1024 * sizeof(ViInt8)));

      if (patternBuffer == NULL)
        errStatus = VI_ERROR_ALLOC;

      stateMachine = STATE_MACHINE_GENERAL_COMMANDS;

      break;

    case STATE_MACHINE_GENERAL_COMMANDS:

      // Send the scpi-commands to the instrument to perform the query

      switch (channel)
      {
        //-------------------------------------------------------------
      case HPE8311A_DATA_CHANNEL_ALL:
      case HPE8311A_DATA_CHANNEL_1:
      case HPE8311A_DATA_CHANNEL_2:
      case HPE8311A_DATA_CHANNEL_3:

        errStatus = hpe8311a_patternDataDownloadBuffer_Q(
          instrumentHandle,
          channel,
          patternBuffer,
          &patternLengthReturned,
          60 * 1024,
          HPE8311A_CONVERT_OFF
          );

        break;
        //-------------------------------------------------------------
      default:
        errStatus = VI_ERROR_PARAMETER2;
        break;
        //-------------------------------------------------------------
      }                         // END switch( channel )

      stateMachine = STATE_MACHINE_ANALYZE_RESULTS;

      break;

    case STATE_MACHINE_ANALYZE_RESULTS:

      // Convert binary-data to ascii-data when so requested.
      switch (convertBinaryToAscii)
      {
        //-------------------------------------------------------------
      case HPE8311A_CONVERT_OFF:
        // nothing to do!
        break;
        //-------------------------------------------------------------
      case HPE8311A_CONVERT_ON:
        for (count = 0; count < patternLengthReturned; count++)
          patternBuffer[count] += 0x30; /*lint !e613 !e668 */
        break;
        //-------------------------------------------------------------
      default:
        errStatus = VI_ERROR_PARAMETER4;
        break;
        //-------------------------------------------------------------
      }                         // END switch()

      stateMachine = STATE_MACHINE_RETURN_RESULTS;

      break;

    case STATE_MACHINE_RETURN_RESULTS:

      // Copy buffer-contents into the new file

      actualColumn = 1;

      for (count = 0; count < patternLengthReturned; count++)
      {
        if (actualColumn >= PATTERN_FILE_CULOMS_ALLOWED)
        {
          fputc('\n', patternFile); /*lint !e613 !e668 */
          actualColumn = 1;
        }

        if (fputc(patternBuffer[count], patternFile) == EOF)  /*lint !e613 !e668 */
        {
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
          break;
        }

        actualColumn++;
      }

      stateMachine = STATE_MACHINE_CLOSE_FILE;

      break;

    case STATE_MACHINE_CLOSE_FILE:

      if (patternFile)
      {
        // only report an error if we don't already have one
        if (EOF == fclose(patternFile) && errStatus == VI_SUCCESS)
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE;
      }

      stateMachine = STATE_MACHINE_FREE_MEMORY;
      break;

    case STATE_MACHINE_FREE_MEMORY:

      // Free mallocated heap-space
      FREE(patternBuffer);
      stateMachine = STATE_MACHINE_DO_FINISH;

      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      stateMachine = STATE_MACHINE_CLOSE_FILE;
      stateMachineErrorOccurred = VI_TRUE;
      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternDataPreset()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - Preset
//---------------------------------------------------------------------------
// This function sets up clock data starting from bit 1 with value 1.
//---------------------------------------------------------------------------
// The parameter <patternDivider> is used as the divider to generate a
//   CLOCK%patternDivider sequence (squarewave if HPE8311A_NRZ data is selected).
//   The parameter <patternLength> determines the length of the sequence.
//   - patternDivider = 0
//     special Case!
//     then the sequence defined by <length> is filled with zeros.
//   - patternDivider = 1
//     special Case!
//     then the sequence is filled with ones.
//   - patternDivider=2 Sequence = 101010101010101....
//   - patternDivider=3 Sequence = 100100100100100....
//   - patternDivider=4 Sequence = 110011001100110....
//   - patternDivider=5 Sequence = 110001100011000....
//   - patternDivider=6 Sequence = 111000111000111....
//   - patternDivider=7 Sequence = 111000011100001....
//   - patternDivider=8 Sequence = 111100001111000....
//---------------------------------------------------------------------------
// Note: To produce a CONTINUOUS squarewave the pattern length must
//   be a multiple of the selected divider.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:PRESet[1|2|3] <patternDivider>,<patternLength>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_ALL
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | patternDivider                                       ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Period of the generated pattern results of provided value
//  |  |   multiplied with the actual used puls-period ( Clock/n ).
//   ------------------------------------------------------------------------
//  | patternLength                                        ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | pattern length
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataPreset(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt32 patternDivider,
    ViUInt32 patternLength
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataPreset"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter1[HPE8311A_STRING_LENGTH];
  ViChar scpiParameter2[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  sprintf(scpiParameter1, "%d", patternDivider);
  sprintf(scpiParameter2, "%d", patternLength);

  switch (channel)
  {

  case HPE8311A_DATA_CHANNEL_ALL:

    strcpy(completeCommand, ":DIG:PATT:PRES ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_1:

    strcpy(completeCommand, ":DIG:PATT:PRES1 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_2:

    strcpy(completeCommand, ":DIG:PATT:PRES2 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_3:

    strcpy(completeCommand, ":DIG:PATT:PRES3 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_patternDataPRBS()
//*****************************************************************************
// CATEGORY: PATTERN-FUNCTIONS - PRBS
//---------------------------------------------------------------------------
// This function sets up PRBS data starting from bit 1.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :DIGital[:STIMulus]:PATTern:PRBS[1|2|3] <polynomExponent>,<patternLength>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_DATA_CHANNEL_ALL
//  |  |  - HPE8311A_DATA_CHANNEL_1
//  |  |  - HPE8311A_DATA_CHANNEL_2
//  |  |  - HPE8311A_DATA_CHANNEL_3
//   ------------------------------------------------------------------------
//  | polynomExponent                                      ViUInt8         IN
//  |   ---------------------------------------------------------------------
//  |  | Exponent of the prbs (pseudo random binary sequence)
//  |  |   polynom: 2^(n-1).
//  |  | Range: 7 to 12.
//   ------------------------------------------------------------------------
//  | patternLength                                        ViUInt32        IN
//  |   ---------------------------------------------------------------------
//  |  | Length of the randomized generated pattern.
//  |  |   Determines how many bits of the PRBS sequence are used. If
//  |  |   <patternLength> is longer than the PRBS, the PRBS is repeated
//  |  |   as necessary to achieve the required length.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_patternDataPRBS(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 polynomExponent,
    ViUInt32 patternLength
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_patternDataPRBS"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter1[HPE8311A_STRING_LENGTH];
  ViChar scpiParameter2[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  sprintf(scpiParameter1, "%d", polynomExponent);
  sprintf(scpiParameter2, "%d", patternLength);

  switch (channel)
  {

  case HPE8311A_DATA_CHANNEL_ALL:

    strcpy(completeCommand, ":DIG:PATT:PRBS ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_1:

    strcpy(completeCommand, ":DIG:PATT:PRBS1 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_2:

    strcpy(completeCommand, ":DIG:PATT:PRBS2 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  case HPE8311A_DATA_CHANNEL_3:

    strcpy(completeCommand, ":DIG:PATT:PRBS3 ");
    strcat(completeCommand, scpiParameter1);
    strcat(completeCommand, ", ");
    strcat(completeCommand, scpiParameter2);

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// SETTING-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_settingStore()
//*****************************************************************************
// CATEGORY: SETTING-FUNCTIONS
//---------------------------------------------------------------------------
// The actual instrument set-up-information is transfered to the instruments
//   internal memory (RAM). There are 10 different settings storeable.
//---------------------------------------------------------------------------
// This set-up-information includes all parameter and mode settings, but
//   does not include the status group registers. The stored setting is
//   hold in the memory until the instrument is power-cycled.
//---------------------------------------------------------------------------
// hpe8311a_settingRecall() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   *SAV
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | settingNumber                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the setting-number to store the setting to.
//  |  | Range:
//  |  |   HPE8311A_SETTING_1
//  |  |   HPE8311A_SETTING_2
//  |  |   HPE8311A_SETTING_3
//  |  |   HPE8311A_SETTING_4
//  |  |   HPE8311A_SETTING_5
//  |  |   HPE8311A_SETTING_6
//  |  |   HPE8311A_SETTING_7
//  |  |   HPE8311A_SETTING_8
//  |  |   HPE8311A_SETTING_9
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_settingStore(
    ViSession instrumentHandle,
    ViUInt16 settingNumber
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_settingStore"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (
    (settingNumber < HPE8311A_SETTING_1) ||
    (settingNumber > HPE8311A_SETTING_9)
    )
    ERR_RETURN (VI_ERROR_PARAMETER2);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  errStatus = hpe8311a_cmdUInt(
    instrumentHandle,
    "*SAV",
    settingNumber
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_settingRecall()
//*****************************************************************************
// CATEGORY: SETTING-FUNCTIONS
//---------------------------------------------------------------------------
// The stored instrument set-up-information is transfered from the
//   internal memory and used as the actual setting.
// There may be 10 different settings stored.
//---------------------------------------------------------------------------
// This set-up-information includes all parameter and mode settings, but
//   does not include the status group registers.
//---------------------------------------------------------------------------
// hpe8311a_settingStore() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   *RCL
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | settingNumber                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the setting-number describing the desired stored setting.
//  |  | Range:
//  |  |   HPE8311A_SETTING_DEFAULT
//  |  |   HPE8311A_SETTING_1
//  |  |   HPE8311A_SETTING_2
//  |  |   HPE8311A_SETTING_3
//  |  |   HPE8311A_SETTING_4
//  |  |   HPE8311A_SETTING_5
//  |  |   HPE8311A_SETTING_6
//  |  |   HPE8311A_SETTING_7
//  |  |   HPE8311A_SETTING_8
//  |  |   HPE8311A_SETTING_9
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_settingRecall(
    ViSession instrumentHandle,
    ViUInt16 settingNumber
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_settingRecall"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (settingNumber > HPE8311A_SETTING_9)
    ERR_RETURN (VI_ERROR_PARAMETER2);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  errStatus = hpe8311a_cmdUInt(
    instrumentHandle,
    "*RCL",
    settingNumber
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************

#define SETTING_FILE_HEADER_ROWS 6
#define SETTING_FILE_HEADER_STRING_0 "===============================================================================\n"
#define SETTING_FILE_HEADER_STRING_1 "This is a setting-file for Hewlett-Packard HPE8311A instruments!\n"
#define SETTING_FILE_HEADER_STRING_2 "===============================================================================\n"
#define SETTING_FILE_HEADER_STRING_3 "Do not edit this file or setting will become unuseable!\n"
#define SETTING_FILE_HEADER_STRING_4 "Corrupt setting-files do cause unexpected instrument-state!\n"
#define SETTING_FILE_HEADER_STRING_5 "===============================================================================\n"
#define SETTING_FILE_HEADER_CHECK_LENGTH  "#checkLength#"
#define SETTING_FILE_HEADER_CHECK_SUMCRC  "#checkSumCRC#"

//*****************************************************************************
// crcUpdate()
//*****************************************************************************
// This function must be called once for each value which is
//   to be included in the CRC for messages to be transmitted.
//---------------------------------------------------------------------------
// This function is called once for each value which is included
//   in the CRC of a received message, AND once for each of the two CRC
//   value at the end of the received message. If the resulting
//   CRC is zero, then the message has been correctly received.
//---------------------------------------------------------------------------
// Calling sequence:
//   crc = crcUpdate(crc,next_value);
//*****************************************************************************
static ViUInt16 crcUpdate(
    ViUInt16 crc,
    ViUInt8 crcValue
)
//*****************************************************************************
{
  ViUInt32 x;
  ViUInt16 i;
  // "x" will contain the value to be processed in bits 0-7 and the CRC
  // in bits 8-23. Bit 24 will be used to test for overflow, and then cleared
  // to prevent the sign bit of "x" from being set to 1. Bits 25-31 are not
  // used. ("x" is treated as though it is a 32 bit register).

  // Get the CRC and the value
  x = ((ViUInt32) crc << 8) | crcValue;

  for (i = 0; i < 8; i++)
  {
    // Shift the high-order bit of the value into the low-order bit of the
    // CRC, and shift the high-order bit of the CRC into bit 24.
    x = x << 1;

    // Test to see if the old high-order bit of the CRC was a 1.
    if (x & 0x01000000)
      x = x ^ 0x01102100;
    // If the old high-order bit of the CRC was a 1, exclusive-or it with a
    // one
    // to set it to 0, and exclusive-or the CRC with hex 1021 to produce the
    // CCITT-recommended CRC generator of: X**16 + X**12 + X**5 + 1. To
    // produce
    // the CRC generator of: X**16 + X**15 + X**2 + 1, change the constant
    // from
    // 0x01102100 to 0x01800500. This will exclusive-or the CRC with hex 8005
    // and produce the same CRC that IBM uses for their synchr. transmission
    // protocols.
  }

  return ((ViUInt16) ((x & 0x00ffff00) >> 8));
}
//*****************************************************************************
// crcFinish()
//*****************************************************************************
//  This function must be called once after all the value in a block
//    have been processed for a message which is to be TRANSMITTED. It
//    returns the calculated CRC bytes, which should be transmitted as the
//    two value following the block. The first of these 2 bytes
//    must be taken from the high-order byte of the CRC, and the second
//    must be taken from the low-order byte of the CRC.
//---------------------------------------------------------------------------
//  This routine is NOT called for a message which has been RECEIVED.
//---------------------------------------------------------------------------
//   Calling sequence:
//       crc = crcFinish(crc);
//*****************************************************************************
static ViUInt16 crcFinish(
    ViUInt16 crc
)
//*****************************************************************************
{
  // Call crcUpdate twice, passing it a value of hex 00 each time, to
  // flush out the last 16 bits from the CRC calculation, and ERR_RETURN the
  // result as the value of this function.
  return (crcUpdate(crcUpdate(crc, '\0'), '\0'));
}
//*****************************************************************************
// hpe8311a_settingFile()
//*****************************************************************************
// CATEGORY: SETTING-FUNCTIONS
//---------------------------------------------------------------------------
// Instrument set-up-information, stored in a special data-file, are
//   transfered to the instrument.
//---------------------------------------------------------------------------
// This set-up-information includes all parameter and mode settings, but
//   does not include the contents of the instrument setting memories and
//   the status group registers. The data format of the file is binary
//   should not be edited.
//---------------------------------------------------------------------------
// hpe8311a_settingFile_Q() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :SYST:SET
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | settingSourceFileName                                ViString        IN
//  |   ---------------------------------------------------------------------
//  |  | Complete string containing the name of the setting-file
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_settingFile(
    ViSession instrumentHandle,
    ViString settingSourceFileName
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_settingFile"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  FILE *settingFile = NULL;
  ViInt32 gotInt;
  ViChar gotChar;
  ViUInt32 settingLength = 0;
  ViUInt32 verifySettingLength = 0;
  ViChar *settingBuffer = NULL;
  ViUInt32 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;

  ViChar *fileHeader[SETTING_FILE_HEADER_ROWS] = {
    SETTING_FILE_HEADER_STRING_0,
    SETTING_FILE_HEADER_STRING_1,
    SETTING_FILE_HEADER_STRING_2,
    SETTING_FILE_HEADER_STRING_3,
    SETTING_FILE_HEADER_STRING_4,
    SETTING_FILE_HEADER_STRING_5
  };
  ViChar requestedLine[HPE8311A_STRING_LENGTH];
  ViUInt32 count;
  ViUInt16 checkSumCRC = 0;
  ViUInt16 verifyCheckSumCRC = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (settingSourceFileName == NULL)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(settingSourceFileName) == 0)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(settingSourceFileName) > HPE8311A_STRING_LENGTH)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  stateMachine = STATE_MACHINE_OPEN_FILE;
  settingFile = NULL;
  settingBuffer = NULL;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_OPEN_FILE:

      settingFile = fopen(settingSourceFileName, "rb");

      if (settingFile == NULL)
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE;

      stateMachine = STATE_MACHINE_MALLOC_MEMORY;

      break;

    case STATE_MACHINE_MALLOC_MEMORY:

      settingBuffer = (ViChar *) malloc(60 * 1024 * sizeof(ViChar));

      if (settingBuffer == NULL)
        errStatus = VI_ERROR_ALLOC;

      stateMachine = STATE_MACHINE_PREPARE_DATA;

      break;

    case STATE_MACHINE_PREPARE_DATA:

      //---------------------------------------------------------------
      // Verify the file-contents.
      // First there has to be the correct setting-file-header.
      for (count = 0; count < SETTING_FILE_HEADER_ROWS; count++)
      {
        if (fgets(requestedLine, HPE8311A_STRING_LENGTH, settingFile) == NULL)  /*lint !e613 !e668 */
        {
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_READ_FROM_FILE;
          break;
        }

        if (strcmp(requestedLine, fileHeader[count]) != 0)
        {
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          break;
        }
      }

      if (errStatus < VI_SUCCESS)
        break;

      //---------------------------------------------------------------
      // Verify the file-contents.
      // And we know the valid bytes to transfer to the instrument.
      // This is written like this:  #checkLength#validBytesString#

      // Read #checkLength#
      if (NULL == fgets(requestedLine,
          (int) (1 + strlen(SETTING_FILE_HEADER_CHECK_LENGTH)),
          settingFile)
        )                       /*lint !e668 !e613 */
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_READ_FROM_FILE;
        break;
      }
      if (strcmp(requestedLine, SETTING_FILE_HEADER_CHECK_LENGTH) != 0)
      {
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
        break;
      }

      // Read the validBytesString
      for (count = 0; count < HPE8311A_STRING_LENGTH; count++)
      {

        gotInt = fgetc(settingFile);  /*lint !e668 !e613 */

        switch (gotInt)
        {
        case EOF:
          // failure
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          count = HPE8311A_STRING_LENGTH;
          break;
        case '#':
          // end of validBytesString found
          requestedLine[count] = 0; // terminate
          count = HPE8311A_STRING_LENGTH;
          break;
        default:
          requestedLine[count] = (ViChar) gotInt; // memorize
          break;
        }                       // END switch()

        if (count == HPE8311A_STRING_LENGTH - 1)
        {
          // the validBytesString seems to be too long
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          break;
        }

      }                         // END for()

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = convertViStringToUInt32(&settingLength, requestedLine);

      if (errStatus < VI_SUCCESS)
        break;

      if (settingLength == 0)
      {
        errStatus = HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_EMPTY;
        break;
      }

      if (settingLength >= (60 * 1024))
      {
        errStatus = HPE8311A_INSTR_ERROR_SOURCE_FILE_IS_TOO_LARGE;
        break;
      }

      //---------------------------------------------------------------
      // Verify the file-contents.
      // And we know the valid bytes to transfer to the instrument.
      // This is written like this:  #checkSumCRC#checkSumString#

      // Read #checkSumCRC#
      if (NULL == fgets(requestedLine,
          (int) (1 + strlen(SETTING_FILE_HEADER_CHECK_SUMCRC)),
          settingFile)
        )                       /*lint !e668 !e613 */
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_READ_FROM_FILE;
        break;
      }
      if (strcmp(requestedLine, SETTING_FILE_HEADER_CHECK_SUMCRC) != 0)
      {
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
        break;
      }

      // Read the validBytesString
      for (count = 0; count < HPE8311A_STRING_LENGTH; count++)
      {

        gotInt = fgetc(settingFile);  /*lint !e668 !e613 */

        switch (gotInt)
        {
        case EOF:
          // failure
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          count = HPE8311A_STRING_LENGTH;
          break;
        case '#':
          // end of validBytesString found
          requestedLine[count] = 0; // terminate
          count = HPE8311A_STRING_LENGTH;
          break;
        default:
          requestedLine[count] = (ViChar) gotInt; // memorize
          break;
        }                       // END switch()

        if (count == HPE8311A_STRING_LENGTH - 1)
        {
          // the validBytesString seems to be too long
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          break;
        }
      }                         // END for()

      if (errStatus < VI_SUCCESS)
        break;

      errStatus = convertViStringToUInt16(&checkSumCRC, requestedLine);

      if (errStatus < VI_SUCCESS)
        break;

      //---------------------------------------------------------------
      // Copy file-contents into the new buffer

      verifySettingLength = 0;
      verifyCheckSumCRC = 0;

      /* NOTE; lint gets a little confused with null ptr's 
       * in while loops 
       */
      /*lint -e668 -e613 pass/use null ptr (settingFile) */

      while ((gotInt = fgetc(settingFile)) != EOF)
      {
        gotChar = (ViChar) (gotInt & 0x000000FF);
        settingBuffer[verifySettingLength++] = gotChar;
        verifyCheckSumCRC = crcUpdate(verifyCheckSumCRC, (ViUInt8) gotChar);

        if (verifySettingLength > settingLength)
        {
          errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
          break;
        }
      }

      // The while loop before was left with too long length?
      if (errStatus < VI_SUCCESS)
        break;

      // The while loop before was left with an read-error?
      if (ferror(settingFile) != VI_FALSE)
      {
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
        break;
      }

      if (feof(settingFile) == VI_FALSE)
      {
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
        break;
      }

      /*lint +e668 +e613 */

      if (verifySettingLength != settingLength)
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;
      // EOF occurred to early, expected more data!

      verifyCheckSumCRC = crcFinish(verifyCheckSumCRC);

      if (verifyCheckSumCRC != checkSumCRC)
        errStatus = HPE8311A_INSTR_ERROR_FILE_CONTENTS_IS_CORRUPT;

      stateMachine = STATE_MACHINE_GENERAL_COMMANDS;

      break;

    case STATE_MACHINE_GENERAL_COMMANDS:

      errStatus = hpe8311a_cmdInt8Arr(
        instrumentHandle,
        "SYST:SET",
        settingBuffer,
        (ViInt32) settingLength
        );

      stateMachine = STATE_MACHINE_CLOSE_FILE;

      break;

    case STATE_MACHINE_CLOSE_FILE:

      // Close file
      if (settingFile)
      {
        // only report an error if we don't already have one
        if (EOF == fclose(settingFile) && errStatus == VI_SUCCESS)
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE;
      }

      stateMachine = STATE_MACHINE_FREE_MEMORY;
      break;

    case STATE_MACHINE_FREE_MEMORY:

      FREE(settingBuffer);
      stateMachine = STATE_MACHINE_DO_FINISH;
      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      stateMachine = STATE_MACHINE_CLOSE_FILE;
      stateMachineErrorOccurred = VI_TRUE;
      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_settingFile_Q()
//*****************************************************************************
// CATEGORY: SETTING-FUNCTIONS
//---------------------------------------------------------------------------
// Instrument set-up-information are requested from the instrument and
//   stored in a special data-file.
//---------------------------------------------------------------------------
// This set-up-information includes all parameter and mode settings, but
//   does not include the contents of the instrument setting memories and
//   the status group registers. The data format of the file is binary
//   should not be edited.
//---------------------------------------------------------------------------
// hpe8311a_settingFile() is complementary to this function.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :SYST:SET?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | settingDestinationFileName                           ViString        IN
//  |   ---------------------------------------------------------------------
//  |  | Complete string containing the name of the setting-file.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_settingFile_Q(
    ViSession instrumentHandle,
    ViString settingDestinationFileName
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_settingFile_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  FILE *settingFile = NULL;
  ViUInt32 requestedSettingLength = 0;
  ViChar *settingBuffer = NULL;
  ViUInt32 stateMachine;
  ViBoolean stateMachineErrorOccurred = VI_FALSE;
  ViUInt32 count;

  ViChar *fileHeader[SETTING_FILE_HEADER_ROWS] = {
    SETTING_FILE_HEADER_STRING_0,
    SETTING_FILE_HEADER_STRING_1,
    SETTING_FILE_HEADER_STRING_2,
    SETTING_FILE_HEADER_STRING_3,
    SETTING_FILE_HEADER_STRING_4,
    SETTING_FILE_HEADER_STRING_5
  };
  ViChar settingLengthString[HPE8311A_STRING_LENGTH];
  ViUInt16 checkSumCRC = 0;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Verify parameter range

  if (settingDestinationFileName == NULL)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(settingDestinationFileName) == 0)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  if (strlen(settingDestinationFileName) > HPE8311A_STRING_LENGTH)
    ERR_RETURN (VI_ERROR_PARAMETER3);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  stateMachine = STATE_MACHINE_MALLOC_MEMORY;
  settingFile = NULL;
  settingBuffer = NULL;

  do
  {

    switch (stateMachine)
    {

    case STATE_MACHINE_MALLOC_MEMORY:

      settingBuffer = (ViChar *) malloc(60 * 1024 * sizeof(ViChar));

      if (settingBuffer == NULL)
        errStatus = VI_ERROR_ALLOC;

      stateMachine = STATE_MACHINE_GENERAL_COMMANDS;

      break;

    case STATE_MACHINE_GENERAL_COMMANDS:

      errStatus = hpe8311a_cmdInt8Arr_Q(
        instrumentHandle,
        "SYST:SET?",
        60 * 1024,
        settingBuffer,
        (ViPInt32) (&requestedSettingLength)
        );

      // check if transfer has been OK, but no data has been received
      if ( (errStatus == VI_SUCCESS) && (requestedSettingLength == 0) )
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;

      stateMachine = STATE_MACHINE_OPEN_FILE;

      break;

    case STATE_MACHINE_OPEN_FILE:

      settingFile = fopen(settingDestinationFileName, "wb");

      if (settingFile == NULL)
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_OPEN_FILE;

      stateMachine = STATE_MACHINE_RETURN_RESULTS;

      break;

    case STATE_MACHINE_RETURN_RESULTS:

      //---------------------------------------------------------------
      // Write file-header into the new file

      for (count = 0; count < SETTING_FILE_HEADER_ROWS; count++)
      {
        if (fputs(fileHeader[count], settingFile) == EOF) //lint !e668
        {
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
          break;
        }
      }

      if (errStatus < VI_SUCCESS)
        break;

      //---------------------------------------------------------------
      // Write byte-count file-header into the new file

      sprintf(
        settingLengthString,
        "%s%u#",
        SETTING_FILE_HEADER_CHECK_LENGTH,
        requestedSettingLength
        );

      if (fputs(settingLengthString, settingFile) == EOF) //lint !e668
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

      //---------------------------------------------------------------
      // Write check-sum file-header into the new file
      checkSumCRC = 0;

      for (count = 0; count < requestedSettingLength; count++)
        checkSumCRC = crcUpdate(checkSumCRC, (ViUInt8) settingBuffer[count]); /*lint !e613 !e668 */

      checkSumCRC = crcFinish(checkSumCRC);

      sprintf(
        settingLengthString,
        "%s%u#",
        SETTING_FILE_HEADER_CHECK_SUMCRC,
        checkSumCRC
        );

      if (fputs(settingLengthString, settingFile) == EOF) //lint !e668
      {
        errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
        break;
      }

      //---------------------------------------------------------------
      // Copy buffer-contents into the new file

      for (count = 0; count < requestedSettingLength; count++)
      {
        if (fputc(settingBuffer[count], settingFile) == EOF) //lint !e668 !e613
        {
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_WRITE_TO_FILE;
          break;
        }
      }

      stateMachine = STATE_MACHINE_CLOSE_FILE;

      break;

    case STATE_MACHINE_CLOSE_FILE:

      // Close file
      if (settingFile)
      {
        // only report an error if we don't already have one
        if (EOF == fclose(settingFile) && errStatus == VI_SUCCESS)
          errStatus = HPE8311A_INSTR_ERROR_CANNOT_CLOSE_FILE;
      }

      stateMachine = STATE_MACHINE_FREE_MEMORY;
      break;

    case STATE_MACHINE_FREE_MEMORY:

      FREE(settingBuffer);
      stateMachine = STATE_MACHINE_DO_FINISH;
      break;

    case STATE_MACHINE_ERROR_OCCURRED:

      stateMachine = STATE_MACHINE_CLOSE_FILE;
      stateMachineErrorOccurred = VI_TRUE;
      break;

    }                           /*lint !e744 switch statement has no default */

    if ((errStatus < VI_SUCCESS) && (stateMachineErrorOccurred == VI_FALSE))
      stateMachine = STATE_MACHINE_ERROR_OCCURRED;

  } while (stateMachine != STATE_MACHINE_DO_FINISH);

  ERR_CHK_RETURN (errStatus);
}
//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// LIMIT-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//*****************************************************************************
// hpe8311a_limitState()
//*****************************************************************************
// CATEGORY: LIMIT-FUNCTIONS - state
//---------------------------------------------------------------------------
// This function switches the output limits for the specified channels on or
//   off. When you switch on the output limits the output-levels cannot be
//   programed beyond the programmed limits, until you switch off the
//   output-limits. The limits apply whether you program High/Low levels or
//   Amplitude/Offset levels.
//---------------------------------------------------------------------------
// Note: The current and voltage limits are are HPE8311A_ALWAYS enabled/disabled
//   together, not independently.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:CURRent[1|2]:LIMit:STATe ON
//   [:SOURce]:CURRent[1|2]:LIMit:STATe OFF
//   [:SOURce]:VOLTage[1|2]:LIMit:STATe ON
//   [:SOURce]:VOLTage[1|2]:LIMit:STATe OFF
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | limitEnable                                          ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Output-limits enabled or disabled.
//  |  |   With Limits ON the output-levels cannot be set outside the limits.
//  |  | Range:
//  |  |  - HPE8311A_LIMITS_ENABLE_OFF
//  |  |  - HPE8311A_LIMITS_ENABLE_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_limitState(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 limitEnable
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_limitState"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (limitEnable)
  {

  case HPE8311A_LIMITS_ENABLE_OFF:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_LIMITS_ENABLE_ON:
    strcpy(scpiParameter, "ON");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":CURR1:LIM:STAT ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :CURR2:LIM:STAT ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":CURR1:LIM:STAT",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":CURR2:LIM:STAT",
      scpiParameter
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_limitState_Q()
//*****************************************************************************
// CATEGORY: LIMIT-FUNCTIONS - state
//---------------------------------------------------------------------------
// This function requests from the instrument whether the output limits for
//   the specified channel is switched on or off.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:CURRent[1|2]:LIMit:STATe?
//   [:SOURce]:VOLTage[1|2]:LIMit:STATe?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | limitEnable                                          ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports whether output-limits are enabled or disabled.
//  |  |   With Limits ON the output-levels cannot be set outside the limits.
//  |  | Range:
//  |  |  - HPE8311A_LIMITS_ENABLE_OFF
//  |  |  - HPE8311A_LIMITS_ENABLE_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_limitState_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 limitEnable
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_limitState_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":CURR1:LIM:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":CURR2:LIM:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "0"))
    *limitEnable = HPE8311A_LIMITS_ENABLE_OFF;
  else
  {
    if (!strcmp(requestedString, "1"))
      *limitEnable = HPE8311A_LIMITS_ENABLE_ON;
    else
    {
      // the value got from the instrument is not within the allowed range
      *limitEnable = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *limitEnable);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_limit()
//*****************************************************************************
// CATEGORY: LIMIT-FUNCTIONS - general used
//---------------------------------------------------------------------------
// This function sets:
//   - the high-level current limit.
//   - the low-level current limit.
//   - the high-level voltage limit.
//   - the low-level voltage limit.
//---------------------------------------------------------------------------
// Use function hpe8311a_limitState() to switch on or off current limiting.
//   Then the high-level current cannot be set above the programmed limit.
//---------------------------------------------------------------------------
// Note: The current is NOT limited by the OUTPUT hardware, this is only a
//   software limit.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:CURRent[1|2]:LIMit:HIGH <value>
//   [:SOURce]:CURRent[1|2]:LIMit:LOW  <value>
//   [:SOURce]:VOLTage[1|2]:LIMit:HIGH <value>
//   [:SOURce]:VOLTage[1|2]:LIMit:LOW  <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | limitsVoltageHighLevel                               ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set limitsVoltageHighLevel on.
//   ------------------------------------------------------------------------
//  | limitsVoltageLowLevel                                ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set limitsVoltageLowLevel on.
//   ------------------------------------------------------------------------
//  | limitsCurrentHighLevel                               ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set limitsCurrentHighLevel on.
//   ------------------------------------------------------------------------
//  | limitsCurrentLowLevel                                ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set limitsCurrentLowLevel on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_limit(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 limitsVoltageHighLevel,
    ViReal64 limitsVoltageLowLevel,
    ViReal64 limitsCurrentHighLevel,
    ViReal64 limitsCurrentLowLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_limit"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar voltageHighString[HPE8311A_STRING_LENGTH];
  ViChar voltageLowString[HPE8311A_STRING_LENGTH];
  ViChar currentHighString[HPE8311A_STRING_LENGTH];
  ViChar currentLowString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  mbFloatToString(limitsVoltageHighLevel, voltageHighString);
  mbFloatToString(limitsVoltageLowLevel, voltageLowString);
  mbFloatToString(limitsCurrentHighLevel, currentHighString);
  mbFloatToString(limitsCurrentLowLevel, currentLowString);

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":VOLT1:LIM:HIGH ");
    strcat(completeCommand, voltageHighString);

    strcat(completeCommand, "; :VOLT1:LIM:LOW ");
    strcat(completeCommand, voltageLowString);

    strcat(completeCommand, "; :CURR1:LIM:HIGH ");
    strcat(completeCommand, currentHighString);

    strcat(completeCommand, "; :CURR1:LIM:LOW ");
    strcat(completeCommand, currentLowString);

    strcat(completeCommand, "; :VOLT2:LIM:HIGH ");
    strcat(completeCommand, voltageHighString);

    strcat(completeCommand, "; :VOLT2:LIM:LOW ");
    strcat(completeCommand, voltageLowString);

    strcat(completeCommand, "; :CURR2:LIM:HIGH ");
    strcat(completeCommand, currentHighString);

    strcat(completeCommand, "; :CURR2:LIM:LOW ");
    strcat(completeCommand, currentLowString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    strcpy(completeCommand, ":VOLT1:LIM:HIGH ");
    strcat(completeCommand, voltageHighString);

    strcat(completeCommand, "; :VOLT1:LIM:LOW ");
    strcat(completeCommand, voltageLowString);

    strcat(completeCommand, "; :CURR1:LIM:HIGH ");
    strcat(completeCommand, currentHighString);

    strcat(completeCommand, "; :CURR1:LIM:LOW ");
    strcat(completeCommand, currentLowString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_2:

    strcpy(completeCommand, ":VOLT2:LIM:HIGH ");
    strcat(completeCommand, voltageHighString);

    strcat(completeCommand, "; :VOLT2:LIM:LOW ");
    strcat(completeCommand, voltageLowString);

    strcat(completeCommand, "; :CURR2:LIM:HIGH ");
    strcat(completeCommand, currentHighString);

    strcat(completeCommand, "; :CURR2:LIM:LOW ");
    strcat(completeCommand, currentLowString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_limit_Q()
//*****************************************************************************
// CATEGORY: LIMIT-FUNCTIONS - general used
//---------------------------------------------------------------------------
// This function requests the actual used values from the instrument:
//   - the high-level current limit.
//   - the low-level current limit.
//   - the high-level voltage limit.
//   - the low-level voltage limit.
//---------------------------------------------------------------------------
// Note: The current is NOT limited by the OUTPUT hardware, this is only a
//   software limit.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:CURRent[1|2]:LIMit:HIGH?
//   [:SOURce]:CURRent[1|2]:LIMit:LOW?
//   [:SOURce]:VOLTage[1|2]:LIMit:HIGH?
//   [:SOURce]:VOLTage[1|2]:LIMit:LOW?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | limitsVoltageHighLevel                               ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used High-Level Voltage-Limit from instrument.
//   ------------------------------------------------------------------------
//  | limitsVoltageLowLevel                                ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used Low-Level Voltage-Limit from instrument.
//   ------------------------------------------------------------------------
//  | limitsCurrentHighLevel                               ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used High-Level Current-Limit from instrument.
//   ------------------------------------------------------------------------
//  | limitsCurrentLowLevel                                ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the actual used Low-Level Current-Limit from instrument.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_limit_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * limitsVoltageHighLevel,
    ViReal64 * limitsVoltageLowLevel,
    ViReal64 * limitsCurrentHighLevel,
    ViReal64 * limitsCurrentLowLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_limit_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":VOLT1:LIM:HIGH?",
      limitsVoltageHighLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":VOLT1:LIM:LOW?",
      limitsVoltageLowLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":CURR1:LIM:HIGH?",
      limitsCurrentHighLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":CURR1:LIM:LOW?",
      limitsCurrentLowLevel
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":VOLT2:LIM:HIGH?",
      limitsVoltageHighLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":VOLT2:LIM:LOW?",
      limitsVoltageLowLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":CURR2:LIM:HIGH?",
      limitsCurrentHighLevel
      );

    if (errStatus < VI_SUCCESS)
      break;

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":CURR2:LIM:LOW?",
      limitsCurrentLowLevel
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch( channel )

  //-------------------------------------------------------------------------

  TRACE_VAR("%e", *limitsVoltageHighLevel);
  TRACE_VAR("%e", *limitsVoltageLowLevel);
  TRACE_VAR("%e", *limitsCurrentHighLevel);
  TRACE_VAR("%e", *limitsCurrentLowLevel);
  ERR_CHK_RETURN (errStatus);
}

//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// CONFIG-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Config - Outputs
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_configOutputImpedance()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Output
//---------------------------------------------------------------------------
// This function configures the source impedance of the specified OUTPUT
//   connectors. Note that only two settings are available.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:IMPedance[:INTernal] <internalImpedance>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | internalImpedance                                    ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set the output-impedance of selected channel.
//  |  | Range: 50Ohm or 1kOhm.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configOutputImpedance(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 internalImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configOutputImpedance"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar impedanceString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    mbFloatToString(internalImpedance, impedanceString);

    strcpy(completeCommand, ":OUTP1:IMP ");
    strcat(completeCommand, impedanceString);
    strcat(completeCommand, "; :OUTP2:IMP ");
    strcat(completeCommand, impedanceString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":OUTP1:IMP",
      internalImpedance
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":OUTP2:IMP",
      internalImpedance
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;
  }

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_configOutputImpedance_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Output
//---------------------------------------------------------------------------
// This function requests the acutual configured source impedance of the
//   specified OUTPUT connector from the instrument.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:IMPedance[:INTernal]?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | internalImpedance                                    ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports value of the used output-impedance of selected channel.
//  |  | Range: 50Ohm or 1kOhm.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configOutputImpedance_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * internalImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configOutputImpedance_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":OUTP1:IMP?",
      internalImpedance
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":OUTP2:IMP?",
      internalImpedance
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch( channel )

  //-------------------------------------------------------------------------

  TRACE_VAR("%e", *internalImpedance);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_configOutputLoadCompensation()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Output
//---------------------------------------------------------------------------
// This function configures the expected load impedance of the device-under-
//   test at the OUTPUT connectors. If you have a non-50 W load, the output
//   levels at the device-under-test will not be the levels you program
//   or set via the frontpanel unless you set the expected load using this
//   function.
//---------------------------------------------------------------------------
// Note: with the HP 81112A 3.8V/330MHz output channels it is not possible
//   to set load impedance, it expects 50 W loads.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:IMPedance:EXTernal <loadCompensation>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | loadCompensation                                     ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set the load-compensation of selected channel
//  |  | Range: 100mOhm up to 1MOhm.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configOutputLoadCompensation(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 loadCompensation
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configOutputLoadCompensation"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar compensationString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    mbFloatToString(loadCompensation, compensationString);

    strcpy(completeCommand, ":OUTP1:IMP:EXT ");
    strcat(completeCommand, compensationString);
    strcat(completeCommand, "; :OUTP2:IMP:EXT ");
    strcat(completeCommand, compensationString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":OUTP1:IMP:EXT",
      loadCompensation
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal(
      instrumentHandle,
      ":OUTP2:IMP:EXT",
      loadCompensation
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_configOutputLoadCompensation_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Output
//---------------------------------------------------------------------------
// This function requests the actual configured load impedance expected
//   at the OUTPUT connectors.
//---------------------------------------------------------------------------
// Note: with the HP 81112A 3.8V/330MHz output channels it is not pos-sible
//   to set load impedance, it expects 50 W loads.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:IMPedance:EXTernal?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | loadCompensation                                     ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports value of the used load-compensation of selected channel.
//  |  | Range: 100mOhm up to 1MOhm.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configOutputLoadCompensation_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViReal64 * loadCompensation
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configOutputLoadCompensation_Q"


  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":OUTP1:IMP:EXT?",
      loadCompensation
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":OUTP2:IMP:EXT?",
      loadCompensation
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  TRACE_VAR("%e", *loadCompensation);
  ERR_CHK_RETURN (errStatus);
}

//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Config - Inputs
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_configInputExternal()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Input
//---------------------------------------------------------------------------
// This function configurates the triggering threshold AND
//   the input impedance of the EXT-INPUT connector. Note that only
//   two settings are available for the impedance.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :ARM:LEVel     <inputExternalThreshold>
//   :ARM:IMPedance <inputExternalImpedance>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | inputExternalThreshold                               ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set threshold-level of external-input on
//  |  | value
//   ------------------------------------------------------------------------
//  | inputExternalImpedance                               ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set the impedance-value of external-input on
//  |  | Range: 50 Ohm or 10kOhm
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configInputExternal(
    ViSession instrumentHandle,
    ViReal64 inputExternalThreshold,
    ViReal64 inputExternalImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configInputExternal"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar thresholdString[HPE8311A_STRING_LENGTH];
  ViChar impedanceString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  mbFloatToString(inputExternalThreshold, thresholdString);
  mbFloatToString(inputExternalImpedance, impedanceString);

  strcpy(completeCommand, ":ARM:LEV ");
  strcat(completeCommand, thresholdString);
  strcat(completeCommand, "; :ARM:IMP ");
  strcat(completeCommand, impedanceString);

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_configInputExternal_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Input
//---------------------------------------------------------------------------
// This function requests the actual used configuration of the triggering
//   threshold AND the input impedance of the EXT-INPUT connector from the
//   instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :ARM:LEVel?
//   :ARM:IMPedance?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | inputExternalThreshold                               ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested threshold-level of
//  |  | external-input to.
//   ------------------------------------------------------------------------
//  | inputExternalImpedance                               ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested impedance-value of
//  |  | external-input to.
//  |  | Range: 50 Ohm or 10kOhm
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configInputExternal_Q(
    ViSession instrumentHandle,
    ViReal64 * inputExternalThreshold,
    ViReal64 * inputExternalImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configInputExternal_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query
  // for :ARM:LEV?

  errStatus = hpe8311a_cmdReal64_Q(
    instrumentHandle,
    ":ARM:LEV?",
    inputExternalThreshold
    );

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query
  // for :ARM:IMP?

  errStatus = hpe8311a_cmdReal64_Q(
    instrumentHandle,
    ":ARM:IMP?",
    inputExternalImpedance
    );

  //-------------------------------------------------------------------------

  TRACE_VAR("%e", *inputExternalThreshold);
  TRACE_VAR("%e", *inputExternalImpedance);
  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_configInputClock()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Input
//---------------------------------------------------------------------------
// This function configurates the triggering threshold AND
//   the input impedance of the CLOCK-IN connector. Note that only
//   two settings are available for the impedance.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:LEVel     <inputClockThreshold>
//   :TRIGger:IMPedance <inputClockImpedance>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | inputClockThreshold                                  ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set threshold-level of clock-input on.
//   ------------------------------------------------------------------------
//  | inputClockImpedance                                  ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value to set the impedance-value of clock-input on
//  |  | Range: 50 Ohm or 10kOhm
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configInputClock(
    ViSession instrumentHandle,
    ViReal64 inputClockThreshold,
    ViReal64 inputClockImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configInputClock"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar thresholdString[HPE8311A_STRING_LENGTH];
  ViChar impedanceString[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  mbFloatToString(inputClockThreshold, thresholdString);
  mbFloatToString(inputClockImpedance, impedanceString);

  strcpy(completeCommand, ":TRIG:LEV ");
  strcat(completeCommand, thresholdString);
  strcat(completeCommand, "; :TRIG:IMP ");
  strcat(completeCommand, impedanceString);

  errStatus = hpe8311a_cmd(
    instrumentHandle,
    completeCommand
    );

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_configInputClock_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Input
//---------------------------------------------------------------------------
// This function requests the actual used configuration of the triggering
//   threshold AND the input impedance of the CLOCK-IN connector from the
//   instrument.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :TRIGger:LEVel?
//   :TRIGger:IMPedance?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | inputClockThreshold                                  ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Requests the value threshold-level of clock-input is set on.
//   ------------------------------------------------------------------------
//  | inputClockImpedance                                  ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Requests the value impedance of clock-input is set on.
//  |  | Range: 50 Ohm or 10kOhm
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configInputClock_Q(
    ViSession instrumentHandle,
    ViReal64 * inputClockThreshold,
    ViReal64 * inputClockImpedance
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configInputClock_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query
  // for :TRIG:LEV?

  errStatus = hpe8311a_cmdReal64_Q(
    instrumentHandle,
    ":TRIG:LEV?",
    inputClockThreshold
    );

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query
  // for :TRIG:IMP?

  errStatus = hpe8311a_cmdReal64_Q(
    instrumentHandle,
    ":TRIG:IMP?",
    inputClockImpedance
    );

  //-------------------------------------------------------------------------

  TRACE_VAR("%e", *inputClockThreshold);
  TRACE_VAR("%e", *inputClockImpedance);
  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Config - Pll
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//*****************************************************************************
// hpe8311a_configPllReference()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - PLL
//---------------------------------------------------------------------------
// This function sets the reference source for the PLL, and when selected the
//   external reference (CLOCK-IN connector) this function also sets the
//   expected reference frequency.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:ROSCillator:SOURce INTernal1
//   [:SOURce]:ROSCillator:SOURce INTernal2
//   [:SOURce]:ROSCillator:SOURce EXTernal
//   [:SOURce]:ROSCillator:EXTernal:FREQuency <value>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pllSourceSelect                                      ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select source of the Pll:
//  |  |  - HPE8311A_PLL_SOURCE_CLOCK_IN:
//  |  |      Lock the PLL to a reference signal at the
//  |  |      CLOCK-IN connector.
//  |  |  - HPE8311A_PLL_SOURCE_INTERNAL_1:
//  |  |      Lock the PLL to its internal reference.
//  |  |
//  |  |  - HPE8311A_PLL_SOURCE_INTERNAL_2:
//  |  |      Lock the PLL to reference on vxi-backplane.
//  |  |
//   ------------------------------------------------------------------------
//  | pllExternalFrequencyValue                            ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide value of reference frequency expected at clock-input
//  |  |   for the internal pll.
//  |  | Range: 5MHz or 10 MHz.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configPllReference(
    ViSession instrumentHandle,
    ViUInt16 pllSourceSelect,
    ViReal64 pllExternalFrequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configPllReference"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (pllSourceSelect)
  {

  case HPE8311A_PLL_SOURCE_INTERNAL_1:

    if (
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HP81101A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HP81104A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HP81110A)
      )
      errStatus = hpe8311a_cmd(
        instrumentHandle,
        ":ROSC:SOUR INT"
        );

    if (
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8305A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8306A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8309A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8311A) ||
      (thisPtr->deviceIdentity == HPE8311A_IDN_NUMBER_HPE8312A)
      )
      errStatus = hpe8311a_cmd(
        instrumentHandle,
        ":ROSC:SOUR INT1"
        );

    break;

  case HPE8311A_PLL_SOURCE_INTERNAL_2:

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      ":ROSC:SOUR INT2"         // on vxi-backplane
      );

    break;

  case HPE8311A_PLL_SOURCE_CLOCK_IN:
    {
    ViChar frequencyString[HPE8311A_STRING_LENGTH];
    ViChar completeCommand[HPE8311A_STRING_LENGTH];

    mbFloatToString(pllExternalFrequencyValue, frequencyString);

    strcpy(completeCommand, ":ROSC:SOUR EXT");
    strcat(completeCommand, "; :ROSC:EXT:FREQ ");
    strcat(completeCommand, frequencyString);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );
    }

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_configPllReference_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - PLL
//---------------------------------------------------------------------------
// This function requests the actual used reference source for the PLL, and
//   when selected the external reference (CLOCK-IN connector) this function
//   also requests the expected reference frequency.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:ROSCillator:SOURce?
//   [:SOURce]:ROSCillator:EXTernal:FREQuency?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pllSourceSelect                                      ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Report the actual source of the Pll:
//  |  |  - HPE8311A_PLL_SOURCE_CLOCK_IN:
//  |  |      Lock the PLL to a reference signal at the
//  |  |      CLOCK-IN connector.
//  |  |  - HPE8311A_PLL_SOURCE_INTERNAL_1:
//  |  |      Lock the PLL to its internal reference.
//  |  |  - HPE8311A_PLL_SOURCE_INTERNAL_2:
//  |  |      Lock the PLL to reference on vxi-backplane.
//   ------------------------------------------------------------------------
//  | pllExternalFrequencyValue                            ViPReal64      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports the frequency depending on pllSourceSelect:
//  |  |   - HPE8311A_PLL_SOURCE_INTERNAL_1:
//  |  |     HPE8311A_PLL_SOURCE_INTERNAL_2:
//  |  |     These are static values - no value is reported.
//  |  |   - HPE8311A_PLL_SOURCE_CLOCK_IN:
//  |  |     Reference-frequency provided at clock-input is reported.
//  |  | Range: 5MHz or 10 MHz.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configPllReference_Q(
    ViSession instrumentHandle,
    ViPUInt16 pllSourceSelect,
    ViPReal64 pllExternalFrequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configPllReference_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;
  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  errStatus = hpe8311a_cmdString_Q(
    instrumentHandle,
    ":ROSC:SOUR?",
    HPE8311A_STRING_LENGTH,
    requestedString
    );

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "INT"))
    *pllSourceSelect = HPE8311A_PLL_SOURCE_INTERNAL_1;
  else if (!strcmp(requestedString, "INT2"))
    *pllSourceSelect = HPE8311A_PLL_SOURCE_INTERNAL_2;
  else if (!strcmp(requestedString, "EXT"))
    *pllSourceSelect = HPE8311A_PLL_SOURCE_CLOCK_IN;
  else
  {
    // the value got from the instrument is not within the allowed range
    *pllSourceSelect = HPE8311A_INVALID_VALUE;
    ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
  }

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (*pllSourceSelect)
  {

  case HPE8311A_PLL_SOURCE_INTERNAL_1:
  case HPE8311A_PLL_SOURCE_INTERNAL_2:
    *pllExternalFrequencyValue = 0;

    break;

  case HPE8311A_PLL_SOURCE_CLOCK_IN:

    errStatus = hpe8311a_cmdReal64_Q(
      instrumentHandle,
      ":ROSC:EXT:FREQ?",
      pllExternalFrequencyValue
      );

    break;

  default:
    errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *pllSourceSelect);
  TRACE_VAR("%e", *pllExternalFrequencyValue);
  ERR_CHK_RETURN (errStatus);
}
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// Config - Trigger-Levels
//00000000000000000000000000000000000000000000000000000000000000000000000000000

//
//

//
//*****************************************************************************
// hpe8311a_configTriggerOutLevel()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Trigger
//---------------------------------------------------------------------------
// This functions sets the output levels at the TRIGGER-OUT
//   and STROBE-OUT connector.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   [:SOURce]:PULSe:TRIGger[1|2]:VOLTage HPE8311A_TTL
//   [:SOURce]:PULSe:TRIGger[1|2]:VOLTage HPE8311A_ECL
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerOuput                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | selects the trigger-output (trigger or strobe)
//  |  |  - HPE8311A_OUTPUT_ALL
//  |  |  - HPE8311A_OUTPUT_TRIGGER
//  |  |  - HPE8311A_OUTPUT_STROBE
//   ------------------------------------------------------------------------
//  | triggerLevel                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select the level of selected trigger output
//  |  |  - HPE8311A_TTL
//  |  |  - HPE8311A_ECL
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configTriggerOutLevel(
    ViSession instrumentHandle,
    ViUInt16 triggerOuput,
    ViUInt16 triggerLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configTriggerOutLevel"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  switch (triggerLevel)
  {

  case HPE8311A_TTL:
    strcpy(scpiParameter, "TTL");
    break;

  case HPE8311A_ECL:
    strcpy(scpiParameter, "ECL");
    break;

  case HPE8311A_STA:
    // this is an undocumented secret, pssst!
    strcpy(scpiParameter, "STA");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);
    break;

  }                             // END switch( trigger Level )

  switch (triggerOuput)
  {

  case HPE8311A_OUTPUT_ALL:

    strcpy(completeCommand, ":PULS:TRIG1:VOLT ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :PULS:TRIG2:VOLT ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_OUTPUT_TRIGGER:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRIG1:VOLT",
      scpiParameter
      );

    break;

  case HPE8311A_OUTPUT_STROBE:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":PULS:TRIG2:VOLT",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//

//
//

//
//*****************************************************************************
// hpe8311a_configTriggerOutLevel_Q()
//*****************************************************************************
// CATEGORY: CONFIG-FUNCTIONS - Trigger-Level
//---------------------------------------------------------------------------
// This functions requests the actual used output levels at the TRIGGER-OUT
//   and STROBE-OUT connector from the instrument.
//---------------------------------------------------------------------------
// Equivalent to scpi-commands?:
//   [:SOURce]:PULSe:TRIGger[1|2]:VOLTage?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerOuput                                         ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | selects the trigger-output (trigger or strobe)
//  |  |  - HPE8311A_OUTPUT_TRIGGER
//  |  |  - HPE8311A_OUTPUT_STROBE
//   ------------------------------------------------------------------------
//  | triggerLevel                                         ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Requests the actual level of selected output
//  |  |  - HPE8311A_TTL
//  |  |  - HPE8311A_ECL
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_configTriggerOutLevel_Q(
    ViSession instrumentHandle,
    ViUInt16 triggerOuput,
    ViPUInt16 triggerLevel
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_configTriggerOutLevel_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument to perform the query

  switch (triggerOuput)
  {

  case HPE8311A_OUTPUT_TRIGGER:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRIG1:VOLT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_OUTPUT_STROBE:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":PULS:TRIG2:VOLT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "TTL"))
    *triggerLevel = HPE8311A_TTL;
  else
  {
    if (!strcmp(requestedString, "ECL"))
      *triggerLevel = HPE8311A_ECL;
    else
    {
      // this is an undocumented secret, pssst!
      if (!strcmp(requestedString, "STA"))
        *triggerLevel = HPE8311A_STA;
      else
      {
        // the value got from the instrument is not within the allowed range
        *triggerLevel = HPE8311A_INVALID_VALUE;
        errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
      }
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", triggerLevel);
  ERR_CHK_RETURN (errStatus);
}
//

//
//






//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// OUTPUT-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//*****************************************************************************
// hpe8311a_outputMath()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function sets the mathematic-mode for the instruments outputs.
//   Note: two channels have to be installed in the instrument to use
//   this feature.
//---------------------------------------------------------------------------
// There are two mathematic-modes provided:
//   - HPE8311A_MATH_OFF
//     No addition of outputs. Every outut works for itself.
//   - HPE8311A_MATH_PLUS
//     Signals from HPE8311A_BOTH channels are added at OUTPUT 1.
//     OUTPUT 2 is not used.
//---------------------------------------------------------------------------
// This allows you to for example:
//   - Generate 3 and 4 level waveforms
//   - Simulate single or repeated glitches
//   - Generate pulse transitions with a step-change in slew-rate
//   - Simulate overshoot and undershoot
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :CHANnel:MATH OFF
//   :CHANnel:MATH PLUS
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | mathMode                                             ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select whether output-channels should work seperated or added.
//  |  |  - HPE8311A_MATH_OFF
//  |  |  - HPE8311A_MATH_PLUS
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputMath(
    ViSession instrumentHandle,
    ViUInt16 mathMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputMath"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (mathMode)
  {

  case HPE8311A_MATH_OFF:

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      ":CHAN:MATH OFF"
      );

    break;

  case HPE8311A_MATH_PLUS:

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      ":CHAN:MATH PLUS"
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_outputMath_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function requests the mathematic-mode for the instruments outputs
//   from the instrument.
//---------------------------------------------------------------------------
// There are two mathematic-modes provided:
//   - HPE8311A_MATH_OFF
//     No addition of outputs. Every outut works for itself.
//   - HPE8311A_MATH_PLUS
//     Signals from HPE8311A_BOTH channels are added at OUTPUT 1.
//     OUTPUT 2 is not used.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :CHANnel:MATH?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | mathMode                                             ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports whether output-channels actual work seperated or added.
//  |  | Range:
//  |  |  - HPE8311A_MATH_OFF
//  |  |  - HPE8311A_MATH_PLUS
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputMath_Q(
    ViSession instrumentHandle,
    ViPUInt16 mathMode
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputMath_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  errStatus = hpe8311a_cmdString_Q(
    instrumentHandle,
    ":CHAN:MATH?",
    HPE8311A_STRING_LENGTH,
    requestedString
    );

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "OFF"))
    *mathMode = HPE8311A_MATH_OFF;
  else
  {
    if (!strcmp(requestedString, "PLUS"))
      *mathMode = HPE8311A_MATH_PLUS;
    else
    {
      *mathMode = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", mathMode);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_outputStateNormal()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - state
//---------------------------------------------------------------------------
// This function switches the normal OUTPUTs on or off.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :OUTPut[1|2]:NORMal:STATe OFF
//   :OUTPut[1|2]:NORMal:STATe ON
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputNormalState                                    ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Selects whether selected normal output to switch on or off.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_NORMAL_OFF
//  |  |  - HPE8311A_OUTPUT_NORMAL_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputStateNormal(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 outputNormalState
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputStateNormal"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (outputNormalState)
  {

  case HPE8311A_OUTPUT_NORMAL_OFF:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_OUTPUT_NORMAL_ON:
    strcpy(scpiParameter, "ON");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":OUTP1:NORM:STAT ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :OUTP2:NORM:STAT ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP1:NORM:STAT",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP2:NORM:STAT",
      scpiParameter
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_outputStateNormal_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - state
//---------------------------------------------------------------------------
// This function requests from the instrument, wheter the desired normal
//   OUTPUT is switched on or off.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:NORMal:STATe?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputNormalState                                    ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports whether selected normal output is switched on or off.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_NORMAL_OFF
//  |  |  - HPE8311A_OUTPUT_NORMAL_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputStateNormal_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 outputNormalState
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputStateNormal_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "OUTP1:NORM:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "OUTP2:NORM:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "0"))
    *outputNormalState = HPE8311A_OUTPUT_NORMAL_OFF;
  else
  {
    if (!strcmp(requestedString, "1"))
      *outputNormalState = HPE8311A_OUTPUT_NORMAL_ON;
    else
    {
      // the value got from the instrument is not within the allowed range
      *outputNormalState = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *outputNormalState);
  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_outputStateComplement()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - state
//---------------------------------------------------------------------------
// This function switches the complement OUTPUTs on or off.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :OUTPut[1|2]:COMPlement:STATe OFF
//   :OUTPut[1|2]:COMPlement:STATe ON
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputComplementstate                                ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select whether selected complement output to switch on or off.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_COMPLEMENT_OFF
//  |  |  - HPE8311A_OUTPUT_COMPLEMENT_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputStateComplement(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 outputComplementState
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "_VI_FUNC"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (outputComplementState)
  {

  case HPE8311A_OUTPUT_COMPLEMENT_OFF:
    strcpy(scpiParameter, "OFF");
    break;

  case HPE8311A_OUTPUT_COMPLEMENT_ON:
    strcpy(scpiParameter, "ON");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":OUTP1:COMP:STAT ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :OUTP2:COMP:STAT ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP1:COMP:STAT",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP2:COMP:STAT",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_outputStateComplement_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - State
//---------------------------------------------------------------------------
// This function requests from the instrument, wheter the desired complement
//   OUTPUT is switched on or off.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:COMPlement:STATe?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputComplementState                                ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports whether selected complement output is switched on or off.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_COMPLEMENT_OFF
//  |  |  - HPE8311A_OUTPUT_COMPLEMENT_ON
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputStateComplement_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 outputComplementState
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "_VI_FUNC"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":OUTP1:COMP:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      ":OUTP2:COMP:STAT?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "0"))
    *outputComplementState = HPE8311A_OUTPUT_COMPLEMENT_OFF;
  else
  {
    if (!strcmp(requestedString, "1"))
      *outputComplementState = HPE8311A_OUTPUT_COMPLEMENT_ON;
    else
    {
      // the value got from the instrument is not within the allowed range
      *outputComplementState = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *outputComplementState);
  ERR_CHK_RETURN (errStatus);
}

//*****************************************************************************
// hpe8311a_outputPolarity()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - State
//---------------------------------------------------------------------------
// This function switches the polarity of normal && complement OUTPUTs
//   between inverted-mode or not-inverted-mode.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :OUTPut[1|2]:POLarity NORM
//   :OUTPut[1|2]:POLarity INV
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_ALL
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputStatePolarity                                  ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select whether selected normal AND Complement output to
//  |  |   work in inverted or not inverted mode.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_POLARITY_NORMAL
//  |  |  - HPE8311A_OUTPUT_POLARITY_INVERTED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputPolarity(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViUInt16 outputStatePolarity
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputPolarity"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar scpiParameter[HPE8311A_STRING_LENGTH];
  ViChar completeCommand[HPE8311A_STRING_LENGTH];

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (outputStatePolarity)
  {

  case HPE8311A_OUTPUT_POLARITY_NORMAL:
    strcpy(scpiParameter, "NORM");
    break;

  case HPE8311A_OUTPUT_POLARITY_INVERTED:
    strcpy(scpiParameter, "INV");
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER3);

  }                             // END switch()

  switch (channel)
  {

  case HPE8311A_CHANNEL_ALL:

    strcpy(completeCommand, ":OUTP1:POL ");
    strcat(completeCommand, scpiParameter);
    strcat(completeCommand, "; :OUTP2:POL ");
    strcat(completeCommand, scpiParameter);

    errStatus = hpe8311a_cmd(
      instrumentHandle,
      completeCommand
      );

    break;

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP1:POL",
      scpiParameter
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString(
      instrumentHandle,
      ":OUTP2:POL",
      scpiParameter
      );

    break;

  default:
    errStatus = VI_ERROR_PARAMETER2;
    break;

  }                             // END switch()

  ERR_CHK_RETURN (errStatus);
}
//*****************************************************************************
// hpe8311a_outputPolarity_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS - State
//---------------------------------------------------------------------------
// This function requests from the instrument, whether the desired outputs
//   polarity is switched to normal-mode or to complement-mode.
//---------------------------------------------------------------------------
// Used scpi-commands:
//   :OUTPut[1|2]:POLarity?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | channel                                              ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Select channel-number:
//  |  | Range:
//  |  |  - HPE8311A_CHANNEL_1
//  |  |  - HPE8311A_CHANNEL_2
//   ------------------------------------------------------------------------
//  | outputStatePolarity                                  ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports whether selected normal AND Complement output
//  |  |   actual work in inverted or not inverted mode.
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_POLARITY_NORMAL
//  |  |  - HPE8311A_OUTPUT_POLARITY_INVERTED
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputPolarity_Q(
    ViSession instrumentHandle,
    ViUInt16 channel,
    ViPUInt16 outputStatePolarity
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputPolarity_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH] = {0};

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  switch (channel)
  {

  case HPE8311A_CHANNEL_1:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "OUTP1:POL?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  case HPE8311A_CHANNEL_2:

    errStatus = hpe8311a_cmdString_Q(
      instrumentHandle,
      "OUTP2:POL?",
      HPE8311A_STRING_LENGTH,
      requestedString
      );

    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch()

  //-------------------------------------------------------------------------

  if (errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  //-------------------------------------------------------------------------
  // Analyze the requested values

  if (!strcmp(requestedString, "NORM"))
    *outputStatePolarity = HPE8311A_OUTPUT_POLARITY_NORMAL;
  else
  {
    if (!strcmp(requestedString, "INV"))
      *outputStatePolarity = HPE8311A_OUTPUT_POLARITY_INVERTED;
    else
    {
      // the value got from the instrument is not within the allowed range
      *outputStatePolarity = HPE8311A_INVALID_VALUE;
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
    }
  }

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *outputStatePolarity);
  ERR_CHK_RETURN (errStatus);
}


//*****************************************************************************
// hpe8311a_outputSourceVxiTrigger()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function sets the VXI trigger output source.
// NOTE; the source should be set BEFORE enabling the trigger line.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTP:TTLT:SOUR :OUTP:ECLT:SOUR
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerOutput                                        ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Selects which output will source the vxi trigger,
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_SAME_AS_STROBE_OUT
//  |  |  - HPE8311A_OUTPUT_TRIGGER
//  |  |  - HPE8311A_OUTPUT_STROBE
//*****************************************************************************

ViStatus _VI_FUNC hpe8311a_outputSourceVxiTrigger(
  ViSession instrumentHandle,
  ViUInt16 triggerOutput
  )
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputSourceVxiTrigger"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar cmdString[HPE8311A_STRING_LENGTH] = {0};
  ViUInt16 triggerLine; 
  char * pSource = "";

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  // which source are we going to set?
  switch (triggerOutput)
  {

  case HPE8311A_OUTPUT_TRIGGER:
    pSource = "TRIG1";
    break;

  case HPE8311A_OUTPUT_STROBE:
    pSource = "TRIG2";
    break;

  case HPE8311A_OUTPUT_SAME_AS_STROBE_OUT:
    pSource = "TOUT";
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }

  // set 'em ALL!!!
  for (triggerLine = HPE8311A_VXI_ECL_0; triggerLine <= HPE8311A_VXI_TTL_7; triggerLine++)
  {
    strcat(cmdString, ":OUTP:");
    strcat(cmdString, sourceArray[triggerLine]);
    strcat(cmdString, ":SOUR ");
    strcat(cmdString, pSource);
    if (triggerLine < HPE8311A_VXI_TTL_7)
      strcat(cmdString, "; ");
  }

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  return hpe8311a_cmd(instrumentHandle, cmdString);

}


//*****************************************************************************
// hpe8311a_outputSourceVxiTrigger_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function returns the VXI trigger output source currently used.
// NOTE; If the VXI trigger is not enabled the function will ERR_RETURN
//       HPE8311A_OUTPUT_SAME_AS_STROBE_OUT.
//       Call hpe8311a_outputEnableVxiTrigger() to enable a specific trigger.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTP:TTLT#:SOUR :OUTP:ECLT#:SOUR
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerOutput                                        ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports which line is sourcing the trigger output
//  |  | Range:
//  |  |  - HPE8311A_OUTPUT_SAME_AS_STROBE_OUT
//  |  |  - HPE8311A_OUTPUT_TRIGGER
//  |  |  - HPE8311A_OUTPUT_STROBE
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputSourceVxiTrigger_Q(
  ViSession instrumentHandle,
  ViPUInt16 triggerOutput
  )
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputSourceVxiTrigger_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar requestedString[HPE8311A_STRING_LENGTH];
  ViChar cmdString[HPE8311A_STRING_LENGTH];
  ViUInt16 activeTriggerLine = HPE8311A_VXI_TRG_NONE; 

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  if(NULL == triggerOutput)
    ERR_RETURN (VI_ERROR_PARAMETER2);

  // must not default to HPE8311A_OUTPUT_SAME_AS_STROBE_OUT!
  *triggerOutput = HPE8311A_OUTPUT_TRIGGER;

  // find out which trigger line is currently on...
  errStatus = hpe8311a_outputEnableVxiTrigger_Q(instrumentHandle, &activeTriggerLine);

  if(errStatus < VI_SUCCESS)
    ERR_RETURN (errStatus);

  // and find out the source of that trigger line...that's what the caller wants
  switch ( activeTriggerLine )
  {
  case HPE8311A_VXI_TRG_NONE:
    ERR_RETURN (VI_SUCCESS);

  case HPE8311A_VXI_TTL_0:
  case HPE8311A_VXI_TTL_1:
  case HPE8311A_VXI_TTL_2:
  case HPE8311A_VXI_TTL_3:
  case HPE8311A_VXI_TTL_4:
  case HPE8311A_VXI_TTL_5:
  case HPE8311A_VXI_TTL_6:
  case HPE8311A_VXI_TTL_7:
  case HPE8311A_VXI_ECL_0:
  case HPE8311A_VXI_ECL_1:
    sprintf(cmdString, ":OUTP:%s:SOUR?", sourceArray[activeTriggerLine]); //lint !e661
  	break;

  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE);
  }

  errStatus = hpe8311a_cmdString_Q(
    instrumentHandle, cmdString, HPE8311A_STRING_LENGTH, requestedString
    );

  if (errStatus == VI_SUCCESS)
  {
    // so what did we get?
  
    if ( !strcmp(requestedString, "TOUT") )
  	  *triggerOutput = HPE8311A_OUTPUT_SAME_AS_STROBE_OUT;

    else if ( !strcmp(requestedString, "TRIG1") )
  	  *triggerOutput = HPE8311A_OUTPUT_TRIGGER;

    else if ( !strcmp(requestedString, "TRIG2") )
  	  *triggerOutput = HPE8311A_OUTPUT_STROBE;

    else
       errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
  }


  ERR_CHK_RETURN (errStatus); 
}


//*****************************************************************************
// hpe8311a_outputEnableVxiTrigger()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function switches the trigger-lines on the vxi-backplane on or off.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:HPE8311A_TTLTrg<n>|HPE8311A_ECLTrg<n>[:STATe] ON|OFF|0|1
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerLine                                          ViUInt16        IN
//  |   ---------------------------------------------------------------------
//  |  | Selects the vxi-trigger-line, which should be used after the
//  |  |   function call. The trigger-line used up to now is swiched off
//  |  |   automatically. To use no trigger-line use parameter
//  |  |   HPE8311A_VXI_TRG_NONE.
//  |  | Range:
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//  |  |  - HPE8311A_VXI_TRG_NONE
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputEnableVxiTrigger(
    ViSession instrumentHandle,
    ViUInt16 triggerLine
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputEnableVxiTrigger"


  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeCommand[HPE8311A_STRING_LENGTH];
  ViUInt16 actualTriggerLine;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  //-------------------------------------------------------------------------
  // Prepare the complete command to send to the instrument.

  switch (triggerLine)
  {
  case HPE8311A_VXI_ECL_0:
  case HPE8311A_VXI_ECL_1:
  case HPE8311A_VXI_TTL_0:
  case HPE8311A_VXI_TTL_1:
  case HPE8311A_VXI_TTL_2:
  case HPE8311A_VXI_TTL_3:
  case HPE8311A_VXI_TTL_4:
  case HPE8311A_VXI_TTL_5:
  case HPE8311A_VXI_TTL_6:
  case HPE8311A_VXI_TTL_7:
  case HPE8311A_VXI_TRG_NONE:
    // all is ok
    break;

  default:
    ERR_RETURN (VI_ERROR_PARAMETER2);

  }                             // END switch( triggerLine )

  completeCommand[0] = '\0';

  // switch all off, except that one which should be switched on
  for (actualTriggerLine = HPE8311A_VXI_ECL_0; 
       actualTriggerLine <= HPE8311A_VXI_TTL_7; 
       actualTriggerLine++)
  {
    strcat(completeCommand, ":OUTP:");
    strcat(completeCommand, sourceArray[actualTriggerLine]);

    if (actualTriggerLine == triggerLine)
      strcat(completeCommand, " ON");
    else
      strcat(completeCommand, " OFF");

    if (actualTriggerLine < HPE8311A_VXI_TTL_7)
      strcat(completeCommand, "; ");
  }

  //-------------------------------------------------------------------------
  // Send the scpi-commands to the instrument

  return hpe8311a_cmd(instrumentHandle, completeCommand);

}
//*****************************************************************************
// hpe8311a_outputEnableVxiTrigger_Q()
//*****************************************************************************
// CATEGORY: OUTPUT-FUNCTIONS
//---------------------------------------------------------------------------
// This function requests from the instrument the actual used
//   vxi-trigger-line.
//---------------------------------------------------------------------------
// Used scpi-command:
//   :OUTPut[1|2]:HPE8311A_TTLTrg<n>|HPE8311A_ECLTrg<n>?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | triggerLine                                          ViPUInt16      OUT
//  |   ---------------------------------------------------------------------
//  |  | Reports which trigger-line is switched on, or reports HPE8311A_VXI_TRG_NONE
//  |  |   when all swiched off.
//  |  | Range:
//  |  |  - HPE8311A_VXI_ECL_0
//  |  |  - HPE8311A_VXI_ECL_1
//  |  |  - HPE8311A_VXI_TTL_0
//  |  |  - HPE8311A_VXI_TTL_1
//  |  |  - HPE8311A_VXI_TTL_2
//  |  |  - HPE8311A_VXI_TTL_3
//  |  |  - HPE8311A_VXI_TTL_4
//  |  |  - HPE8311A_VXI_TTL_5
//  |  |  - HPE8311A_VXI_TTL_6
//  |  |  - HPE8311A_VXI_TTL_7
//  |  |  - HPE8311A_VXI_TRG_NONE
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_outputEnableVxiTrigger_Q(
    ViSession instrumentHandle,
    ViPUInt16 triggerLine
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_outputEnableVxiTrigger_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViChar completeCommand[HPE8311A_STRING_LENGTH];
  ViUInt16 actualTriggerLine = 0;
  ViInt32 foundTriggerState = 0;
  ViBoolean activeTriggerLineFound = VI_FALSE;  // not found
  ViUInt16 activeTriggerLine = HPE8311A_VXI_TRG_NONE; // no trigger-lines are
  // on

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  *triggerLine = HPE8311A_INVALID_VALUE;  // undefined value until now

  //-------------------------------------------------------------------------
  // Perform a loop, which requestes to all triggerLines, whether they
  // are active or not. This loop recocnizes visa-errors and errors
  // in the instruments response, like range-error. All triggerLines
  // are requested to test, that there is only one active!

  for (actualTriggerLine = HPE8311A_VXI_ECL_0; actualTriggerLine <= HPE8311A_VXI_TTL_7; actualTriggerLine++)
  {
    strcpy(completeCommand, ":OUTP:");
    strcat(completeCommand, sourceArray[actualTriggerLine]);
    strcat(completeCommand, "?");

    errStatus = hpe8311a_cmdInt32_Q(
      instrumentHandle,
      completeCommand,
      &foundTriggerState
      );

    // visa error occurred ?
    if (errStatus < VI_SUCCESS)
      break;

    // instruments answer not in valid range ?
    if ((foundTriggerState < 0) || (foundTriggerState > 1))
    {
      errStatus = HPE8311A_INSTR_ERROR_INSTRUMENT_ANSWER_NOT_IN_RANGE;
      break;
    }

    // already found an active trigger line ?
    // (normally there can only be one active! )
    if ((activeTriggerLineFound == VI_TRUE) && (foundTriggerState == 1))
    {
      errStatus = HPE8311A_INSTR_ERROR_TOO_MANY_ACTIVE_TRIGGER_REPORTED;
      break;
    }

    // memorize, whether already found an active trigger-line!
    if (foundTriggerState == 1)
    {
      activeTriggerLineFound = VI_TRUE;
      activeTriggerLine = actualTriggerLine;
    }

  }                             // END for()

  *triggerLine = activeTriggerLine;

  //-------------------------------------------------------------------------

  TRACE_VAR("%i", *triggerLine);
  ERR_CHK_RETURN (errStatus);
}
//#############################################################################
//00000000000000000000000000000000000000000000000000000000000000000000000000000
// ARMING-FUNCTIONS
//00000000000000000000000000000000000000000000000000000000000000000000000000000
//#############################################################################

//*****************************************************************************
// hpe8311a_armFrequency()
//*****************************************************************************
// CATEGORY: Arming - Functions
//---------------------------------------------------------------------------
// This function sets the frequency of the PLL (INTernal2) when it is used as
//   the :ARM:SOURce for internal triggering of pulses, bursts or patterns.
//---------------------------------------------------------------------------
// If you are using the PLL as :TRIGger:SOURce to set the pulse frequency,
//   use function hpe8311a_timeFrequency()
//---------------------------------------------------------------------------
// Used scpi-command:
//   :ARM:FREQuency <frequencyValue>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | frequencyValue                                       ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set frequency of PLL on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_armFrequency(
    ViSession instrumentHandle,
    ViReal64 frequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_armFrequency"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal(instrumentHandle, ":ARM:FREQ", frequencyValue);

}
//*****************************************************************************
// hpe8311a_armFrequency_Q()
//*****************************************************************************
// CATEGORY: Arming - Functions
//---------------------------------------------------------------------------
// This function requests the actual used frequency of the PLL (INTernal2)
//   when it is used as the :ARM:SOURce for internal triggering of pulses,
//   bursts or patterns from the instrument.
//---------------------------------------------------------------------------
// If you are using the PLL as :TRIGger:SOURce to request the pulse frequency,
//   use function hpe8311a_timeFrequency_Q()
//---------------------------------------------------------------------------
// Used scpi-command:
//   :ARM:FREQuency?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | frequencyValue                                       ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested frequency of PLL to.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_armFrequency_Q(
    ViSession instrumentHandle,
    ViReal64 * frequencyValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_armFrequency_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal64_Q(instrumentHandle, ":ARM:FREQ?", frequencyValue);

}
//*****************************************************************************
// hpe8311a_armPeriod()
//*****************************************************************************
// CATEGORY: Arming - Functions
//---------------------------------------------------------------------------
// This function sets the period of the PLL (INTernal2) when it is used as
//   the :ARM:SOURce for internal triggering of pulses, bursts or patterns.
//---------------------------------------------------------------------------
// If you are using the PLL as :TRIGger:SOURce to set the pulse-period,
//   use function hpe8311a_timePeriod()
//---------------------------------------------------------------------------
// Used scpi-command:
//   :ARM:PERiod <periodValue>
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | periodValue                                          ViReal64        IN
//  |   ---------------------------------------------------------------------
//  |  | Provide Value to set period of PLL on.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_armPeriod(
    ViSession instrumentHandle,
    ViReal64 periodValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_armPeriod"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal(instrumentHandle, ":ARM:PER", periodValue);

}
//*****************************************************************************
// hpe8311a_armPeriod_Q()
//*****************************************************************************
// CATEGORY: Arming - Functions
//---------------------------------------------------------------------------
// This function requests the actual used period of the PLL (INTernal2)
//   when it is used as the :ARM:SOURce for internal triggering of pulses,
//   bursts or patterns from the instrument.
//---------------------------------------------------------------------------
// If you are using the PLL as :TRIGger:SOURce to request the pulse-period,
//   use function hpe8311a_timePeriod_Q()
//---------------------------------------------------------------------------
// Used scpi-command:
//   :ARM:PERiod?
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | periodValue                                          ViReal64*      OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested period of PLL to.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_armPeriod_Q(
    ViSession instrumentHandle,
    ViReal64 * periodValue
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_armPeriod_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();
  CHECK_URBIOS;

  return hpe8311a_cmdReal64_Q(instrumentHandle, ":ARM:PER?", periodValue);

}
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_biosProgramFile(
    ViSession instrumentHandle,
    ViString fileName
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_biosProgramFile"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 newFoundState;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Get model code to see, whether module
    // is already in register-mode, that
    // means loader-state.

    // Do not need handshake here. Is only model-code.
    errStatus = rbReadA16Reg(
      instrumentHandle,
      A16_DEV_TYPE_REG,
      &(thisPtr->modelCode)
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // modelCode only inside the three last nibbles
    thisPtr->modelCode = (ViUInt16) (thisPtr->modelCode & 0x0FFF);
    DBG_SHOW_LOADER_STATE;

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Switch module from message-mode to register-mode.
    // But only if not in loader-state.

    if (thisPtr->modelCode != LOADER_STATE)
    {
      // Model is not in loader-state.
      errStatus = hpe8311a_cmd(
        instrumentHandle,
        ":DIAG:BIOS:LOADER"
        );

      if (errStatus < VI_SUCCESS)
        ERR_RETURN (errStatus);
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Program the bios

    // Call the subroutine to program the bios.
    errStatus = rbProgramBiosFile(
      instrumentHandle,
      fileName
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // synchronize with handshake

    errStatus = rbCheckExpectedReadyState(
      instrumentHandle,
      DOV_READY,                // may be other
      &newFoundState
      );

    if (errStatus != HPE8311A_INSTR_ERROR_WRONG_STATE && errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // Module is booted in special-bios-mode by rbProgramBiosFile().
    // Switch back into scpi-parser-mode
    errStatus = rbWriteA24RegisterUINT(
      instrumentHandle,
      newFoundState,
      &newFoundState,
      DIR_BLK_BYTES_VALID,
      0
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Get model code to see, whether module
    // is already in register-mode (loader-state)

    // Do not need handshake here. Is only model-code.
    errStatus = rbReadA16Reg(
      instrumentHandle,
      A16_DEV_TYPE_REG,
      &(thisPtr->modelCode)
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // modelCode only inside the three last nibbles
    thisPtr->modelCode = (ViUInt16) (thisPtr->modelCode & 0x0FFF);

    DBG_SHOW_LOADER_STATE;
    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            

    DBG_PRINT("Program-Bios not supported for GPIB-Instruments!");
    ERR_RETURN (HPE8311A_INSTR_ERROR_PROGRAM_BIOS_NOT_SUPPORTED_FOR_GPIB);


    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}

//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_biosSpecial(
    ViSession instrumentHandle
)
//*****************************************************************************
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_biosSpecial"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;
  ViInt8 newFoundState;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  switch (thisPtr->interfaceType)
  {
    //-------------------------------------------------------------------------
  case VI_INTF_VXI:
  case VI_INTF_GPIB_VXI:        
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Get model code to see, whether module
    // is already in register-mode, that
    // means loader-state.

    // Do not need handshake here. Is only model-code.
    errStatus = rbReadA16Reg(
      instrumentHandle,
      A16_DEV_TYPE_REG,
      &(thisPtr->modelCode)
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // modelCode only inside the three last nibbles
    thisPtr->modelCode = (ViUInt16) (thisPtr->modelCode & 0x0FFF);

    DBG_SHOW_LOADER_STATE;

    // Leave this routine, when loader-state not found.
    if (thisPtr->modelCode != LOADER_STATE)
    {
      // Model is not in loader-state.
      // Do not do further actions now!!
      ERR_RETURN (HPE8311A_INSTR_ERROR_FUNCTION_SPECIAL_BIOS_EXPECTED_URBIOS);
    }

    // kill pending errors

    errStatus = rbWriteA24Reg(
      instrumentHandle,
      32,
      0x4
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // switch to special-bios now

    errStatus = rbSpecialBios(
      instrumentHandle,
      &newFoundState
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Module is booted now in special-bios-mode.
    // Switch back into scpi-parser-mode
    errStatus = rbWriteA24Reg(
      instrumentHandle,
      DIR_BLK_BYTES_VALID,
      0
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Get model code to see, whether module
    // is already in register-mode, that
    // means loader-state.

    // Do not need handshake here. Is only model-code.
    errStatus = rbReadA16Reg(
      instrumentHandle,
      A16_DEV_TYPE_REG,
      &(thisPtr->modelCode)
      );

    if (errStatus < VI_SUCCESS)
      ERR_RETURN (errStatus);

    // modelCode only inside the three last nibbles
    thisPtr->modelCode = (ViUInt16) (thisPtr->modelCode & 0x0FFF);
    DBG_SHOW_LOADER_STATE;

    break;

    //-------------------------------------------------------------------------
  case VI_INTF_GPIB:            
    ERR_RETURN (HPE8311A_INSTR_ERROR_SPECIAL_BIOS_NOT_SUPPORTED_FOR_GPIB);


    //-------------------------------------------------------------------------
  default:
    ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);

  }                             // END switch( thisPtr->interfaceType )

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}


//*****************************************************************************
// hpe8311a_transactionStart()
//*****************************************************************************
// CATEGORY: Transaction - Functions
//---------------------------------------------------------------------------
// This function begins a "transaction"; all commands until calling
// hpe8311a_transactionFinish() or hpe8311a_transactionCancel() will
// be buffered.
//---------------------------------------------------------------------------
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_transactionStart(
  ViSession  instrumentHandle
  )
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_transactionStart"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // transaction in progress?
  if(thisPtr->pTransBuffer)
    ERR_RETURN (HPE8311A_INSTR_ERROR_TRANSACTION_IN_PROG);

  // alloc a buffer
  if(NULL == (thisPtr->pTransBuffer = (ViChar *) malloc(XACT_BUFF_SIZE)))
    ERR_RETURN (VI_ERROR_ALLOC);

  // a good starting point..
  memset(thisPtr->pTransBuffer, 0, XACT_BUFF_SIZE);
  thisPtr->TransBufferLength = 0;

  return (VI_SUCCESS);
}



//*****************************************************************************
// hpe8311a_transactionFinish()
//*****************************************************************************
// CATEGORY: Transaction - Functions
//---------------------------------------------------------------------------
// This function completes a current transaction by sending the buffer
// contents and flushing the buffer.
//---------------------------------------------------------------------------
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_transactionFinish(
  ViSession  instrumentHandle
  )
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_transactionFinish"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // transaction in progress?
  if(NULL == thisPtr->pTransBuffer)
    ERR_RETURN (HPE8311A_INSTR_ERROR_TRANSACTION_NOT_IN_PROG);

  // only send the buffer if it's not empty
  if(thisPtr->TransBufferLength)
  {
    switch (thisPtr->interfaceType)
    {
      //-------------------------------------------------------------------------
    case VI_INTF_VXI:
    case VI_INTF_GPIB_VXI:
      // transmit the buffer
      CHECK_URBIOS;
      errStatus = mbSendSCPIandDATA(instrumentHandle, NULL, 0);
      break;

    case VI_INTF_GPIB:
      // pure GPIB-based instruments
      errStatus = delayVPrintf(instrumentHandle, NULL);
      break;

    default:
      ERR_RETURN (HPE8311A_INSTR_ERROR_UNSUPPORTED_INTERFACE);
    }
  }

  // free the transaction buffer regardless...
  transactionCancel(thisPtr);

  return (statusUpdate(instrumentHandle, thisPtr, errStatus));
}


//*****************************************************************************
// hpe8311a_transactionCancel()
//*****************************************************************************
// CATEGORY: Transaction - Functions
//---------------------------------------------------------------------------
// This function unconditionally cancels any current transaction.
// ALL DATA IS LOST!
//---------------------------------------------------------------------------
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_transactionCancel(
  ViSession  instrumentHandle
  )
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_transactionCancel"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  transactionCancel(thisPtr);

  return (VI_SUCCESS);
}


//*****************************************************************************
// hpe8311a_transaction_Q()
//*****************************************************************************
// CATEGORY: Transaction - Functions
//---------------------------------------------------------------------------
// This function queries for the current transaction status and buffer size
//---------------------------------------------------------------------------
//*****************************************************************************
//  PARAMETER Name                                         Type     Direction
//   ------------------------------------------------------------------------
//  | instrumentHandle                                     ViSession       IN
//  |   ---------------------------------------------------------------------
//  |  | Instrument Handle returned from hpe8311a_init()
//   ------------------------------------------------------------------------
//  | pTransactionInProgress                               ViBoolean*     OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested transaction status.
//   ------------------------------------------------------------------------
//  | pCurrBufferLength                                    ViUInt32*     OUT
//  |   ---------------------------------------------------------------------
//  |  | Provide variable to report the requested transaction buffer size.
//*****************************************************************************
ViStatus _VI_FUNC hpe8311a_transaction_Q(
  ViSession  instrumentHandle,
  ViPBoolean pTransactionInProgress,
  ViPUInt32 pCurrBufferLength
  )
{
#  undef FUNC_NAME
# define FUNC_NAME "hpe8311a_transaction_Q"

  ViStatus errStatus;
  struct hpe8311a_globals *thisPtr;

  GET_GLOBALS(instrumentHandle, thisPtr);
  TRACE_IN();

  // junk not allowed
  if(NULL == pTransactionInProgress || NULL == pCurrBufferLength)
    ERR_RETURN (HPE8311A_INSTR_ERROR_NULL_PTR);

  if(thisPtr->pTransBuffer)
  {
    *pTransactionInProgress = VI_TRUE;
    *pCurrBufferLength = thisPtr->TransBufferLength;
  }
  else
  {
    *pTransactionInProgress = VI_FALSE;
    *pCurrBufferLength = 0;
  }

  return (VI_SUCCESS);
}
