
/*#line 1 "hpe1476.cs"*/
/* Copyright (C) 1995 - 2001 Agilent Technologies */
/* VXIplug&play Instrument Driver for the Agilent E1476*/

/*=============================*/
/*           HPE1476           */
/*=============================*/
/* Framework Revision: WIN95 4.0 */
/*                     WINNT 4.0 */
/* Driver Revision:    A.02.06 */
/* Firmware Revision:  A.09.00 */
/*=============================*/

/*=============================*/
/* Source Files:               */
/*=============================*/
/*   hpe1476.c                 */
/*   hpe1476.h                 */
/*=============================*/

/*=============================*/
/* WIN Framework Files         */
/*=============================*/
/*   hpe1476.def               */
/*   hpe1476.rc                */
/*   hpe1476.bas               */
/*=============================*/
#include <stdlib.h>		/* prototype for malloc() */
#include <string.h>		/* prototype for strcpy() */
#include <stdio.h>		/* prototype for sprintf() */
#include <windows.h>
#include "cscpiv.h"		/* includes cscpi and visa calls */
#define INSTR_CALLBACKS		/* needed to get handler types in hpe1476.h */
#include "hpe1476.h"

#define hpe1476_MODEL_CODE 0x0218	/* 1476 */
/* codes for this family are FF00-FF07*/
#define hpe1476_MANF_ID     4095	
#define hpe1476_IDN_STRING  "HEWLETT-PACKARD,SWITCHBOX"

#define hpe1476_REV_CODE "A.02.06"  /* Driver Revision */

#define hpe1476_ERR_MSG_LENGTH 256  /* size of error message buffer */

#define hpe1476_MAX_STAT_HAP 37		    /* number of happenings ??? */
#define hpe1476_MAX_STAT_REG 4		    /* number of IEEE 488.2 status registers */

/* this has to match the index of the ESR register in hpe1476_accessInfo[] */
#define hpe1476_ESR_REG_IDX 2

/* this needs to match the index location of hpe1476_USER_ERROR_HANLDER
 * in the hpe1476_statusHap[] array.  Normally, this is zero.
 */
#define hpe1476_USER_ERROR_HANDLER_IDX 0

/*=============================================================== 
 *
 *  All messages are stored in this area to aid in localization 
 *
 *=============================================================== 
 */


#define hpe1476_MSG_VI_OPEN_ERR 				\
	"vi was zero.  Was the hpe1476_init() successful?"

#define hpe1476_MSG_CONDITION					\
	"condition"
	/* hpe1476_statCond_Q() */

#define hpe1476_MSG_EVENT						\
	"event"	
	/* hpe1476_statEvent_Q() */

#define hpe1476_MSG_EVENT_HDLR_INSTALLED				\
	"event handler is already installed for event happening"
	/* hpe1476_statEvent_Q() */

#define hpe1476_MSG_EVENT_HDLR_INST2				\
	"Only 1 handler can be installed at a time."	
	/* hpe1476_statEvent_Q() */


#define hpe1476_MSG_INVALID_HAPPENING				\
	"is not a valid happening."
	/* hpe1476_statCond_Q() */
	/* hpe1476_statEven_Q() */
	/* hpe1476_statEvenHdlr() */
	/* hpe1476_statEvenHdlr_Q() */

#define hpe1476_MSG_NOT_QUERIABLE					\
	"is not queriable."	
	/* hpe1476_statCond_Q() */
	/* hpe1476_statEven_Q() */


#define hpe1476_MSG_IN_FUNCTION					\
	"in function" 		
	/* hpe1476_error_message() */

#define hpe1476_MSG_INVALID_STATUS					\
  	"Parameter 2 is invalid"				\
	"in function hpe1476_error_message()."
	/* hpe1476_error_message() */

#define hpe1476_MSG_INVALID_STATUS_VALUE				\
	"is not a valid viStatus value."
	/* hpe1476_error_message() */

#define  hpe1476_MSG_INVALID_VI					\
  	"Parameter 1 is invalid"				\
	" in function hpe1476_error_message()"			\
	".  Using an inactive ViSession may cause this error."	\
	"  Was the instrument driver closed prematurely?"
	/* hpe1476_message_query() */

#define hpe1476_MSG_NO_ERRORS					\
	"No Errors"
	/* hpe1476_error_message() */

#define hpe1476_MSG_SELF_TEST_FAILED 				\
	"Self test failed." 	
	/* hpe1476_self_test() */

#define hpe1476_MSG_SELF_TEST_PASSED 				\
	"Self test passed."
	/* hpe1476_self_test() */

/* the following messages are used by the functions to check parameters */

#define hpe1476_MSG_BOOLEAN   "Expected 0 or 1; Got %d"

#define hpe1476_MSG_REAL   "Expected %lg to %lg; Got %lg"
   
#define hpe1476_MSG_INT   "Expected %hd to %hd; Got %hd"

#define hpe1476_MSG_LONG   "Expected %ld to %ld; Got %ld"

#define hpe1476_MSG_LOOKUP "Error converting string response to integer"

#define hpe1476_MSG_NO_MATCH "Could not match string %s"

/* 
 * static error message 
 */

#define VI_ERROR_PARAMETER1_MSG				\
	"Parameter 1 is invalid"

#define VI_ERROR_PARAMETER2_MSG				\
	"Parameter 2 is invalid"

#define VI_ERROR_PARAMETER3_MSG				\
	"Parameter 3 is invalid"

#define VI_ERROR_PARAMETER4_MSG				\
	"Parameter 4 is invalid"

#define VI_ERROR_PARAMETER5_MSG				\
	"Parameter 5 is invalid"

#define VI_ERROR_PARAMETER6_MSG				\
	"Parameter 6 is invalid"

#define VI_ERROR_PARAMETER7_MSG				\
	"Parameter 7 is invalid"

#define VI_ERROR_PARAMETER8_MSG				\
	"Parameter 8 is invalid"

#define VI_ERROR_FAIL_ID_QUERY_MSG				\
	"Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG 				\
	"ViSession (parmeter 1) was not created by this driver"

#define INSTR_ERROR_NULL_PTR_MSG				\
	"NULL pointer detected"

#define INSTR_ERROR_RESET_FAILED_MSG				\
	"reset failed"

#define INSTR_ERROR_UNEXPECTED_MSG 				\
	"An unexpected error occurred"

#define INSTR_ERROR_DETECTED_MSG			\
	"Instrument Error Detected, call hpe1476_error_query()."

#define INSTR_ERROR_LOOKUP_MSG   				\
	"String not found in table"

/*================================================================*/

#define hpe1476_DEBUG_CHK_THIS( vi, thisPtr) 			\
	/* check for NULL user data */				\
	if( 0 == thisPtr)					\
	{							\
 		hpe1476_LOG_STATUS(                             	\
		  vi, NULL, hpe1476_INSTR_ERROR_INV_SESSION );	\
	}							\
	{							\
		ViSession defRM;				\
								\
		/* This should never fail */			\
		errStatus = viGetAttribute( vi,                 \
			VI_ATTR_RM_SESSION, &defRM);		\
		if( VI_SUCCESS > errStatus )			\
		{						\
 			hpe1476_LOG_STATUS(				\
			  vi, NULL, hpe1476_INSTR_ERROR_UNEXPECTED );	\
		}						\
		if( defRM != thisPtr->defRMSession)		\
		{						\
 			hpe1476_LOG_STATUS(				\
			  vi, NULL, hpe1476_INSTR_ERROR_INV_SESSION );	\
		}						\
	}

#define hpe1476_DEBUG_CHK_NULL_PTR( vi, ptr) 			\
	if( 0 == ptr) 						\
	{							\
 		hpe1476_LOG_STATUS( 				\
		   vi, NULL, hpe1476_INSTR_ERROR_NULL_PTR );	\
	}

#define hpe1476_CDE_INIT( funcname)  				\
	strcpy(thisPtr->errFuncName, funcname);			\
	thisPtr->errNumber = VI_SUCCESS;			\
	thisPtr->errMessage[0] = 0;

#define hpe1476_CDE_MESSAGE( message ) 	 			\
	strcpy(thisPtr->errMessage, message)

#define hpe1476_CDE_ERROR( status)					\
	thisPtr->errNumber = status		




struct hpe1476_eventHdlrStruct
{
	hpe1476_InstrEventHandler eventHandler;
	ViAddr userData;
};

struct hpe1476_globals
{
	ViSession			defRMSession;

	ViStatus			errNumber;
	char 				errFuncName[40];
	char				errMessage[160];

	ViBoolean			errQueryDetect;

	ViInt32 			driverEventArray[hpe1476_MAX_STAT_REG];
	ViInt32 			numEventHandlers;
	struct hpe1476_eventHdlrStruct	eventHandlerArray[hpe1476_MAX_STAT_HAP];
	ViBoolean			e1406; /* 1 if gpib to 1406*/

        /* Setting this to VI_TRUE,
         * will prevent I/O from occuring in an SRQ
         * This is needed, because VTL 3.0 doesn't provide
         * an atomic write/read operations.  Don't want
         * to do I/O if the driver is in the middle of
         * a read from the instrument.
         */
        ViBoolean                       blockSrqIO;
        ViInt32                         countSrqIO;
};
/* add the following to the globals data structure */

/* Note: for debugging, you can add __FILE__, __LINE__ as parameters
 * to hpe1476_LOG_STATUS, and ViString filename, and ViInt32 lineNumber to
 * hpe1476_statusUpdate() in order to determine exactly where an error
 * occured in a driver.
 */
#define hpe1476_LOG_STATUS( vi, thisPtr, status ) 	\
  return hpe1476_statusUpdate( vi, thisPtr, status)

/* declare this here since it is called by statusUpdate */
static void hpe1476_srqTraverse(ViSession vi, ViInt32 eventReg);

static ViStatus hpe1476_statusUpdate(
  ViSession vi,
  struct hpe1476_globals *thisPtr,
  ViStatus status
  )
{
	ViInt32  eventQ;

	/* ALL functions exit through here.
	 *
	 * this allows errors to be trapped and
	 * a user error handler could be called.
	 */

	/* can do nothing without state structure*/
	if (!thisPtr) return status;

	/* Check if user wants to query the instrument state.
         * This is only done, if the vi is valid and 
         * no other errors has occured.
         */
	if( VI_TRUE == thisPtr->errQueryDetect && 
            VI_SUCCESS <= status &&
            vi != VI_NULL )
	{
		/* assume IEEE 488.2 Instrument and query standard
                 * status event register for a parser error
                 */	
		if (thisPtr->e1406)
		{
			thisPtr->blockSrqIO = VI_TRUE;

			status = viPrintf(vi, "*esr?\n");
			if( VI_SUCCESS <= status) 
			{

				status = viScanf(vi, "%ld%*t\n", &eventQ);
			}  /* viPrintf was successful */
		} 
		else
		{
			{{ struct {double p1[1];} out_;
/*#line 324 "hpe1476.cs"*/
{extern com_esr_q();if (!instr_query(vi,com_esr_q,(void*)0,&out_)){
/*#line 324 "hpe1476.cs"*/
*&eventQ=(*(HPSL_INT16*)out_.p1).num;}}}}
		}
		if( VI_SUCCESS <= status)
		{

		    /* update driver's copy of the esr register 
       		     * This index 'hpe1476_ESR_REG_IDX' should 
		     * match the access event register array 
		     * and correspond to the 
      		     * standard event status register.
       		     */
		    thisPtr->driverEventArray[hpe1476_ESR_REG_IDX] = 
		      thisPtr->driverEventArray[hpe1476_ESR_REG_IDX] | eventQ;

		    /* see if the instrument thinks an error occured */
		    if( (  0x04 /* Query Error */
                      | 0x08 /* Device Dependent Error */
                      | 0x10 /* Execution Error */
                      | 0x20 /* Command Error */
                    ) & eventQ )
		    {
			/* set our error status, to indicate that a 
			 * instrument error occured.
                         */
			status =  hpe1476_INSTR_ERROR_DETECTED;
		    } 
		} /* INST_QUERY or viScanf was successful */
	} /* if we need to query the instrument */	

        /* check if we need to service SRQ events */
        if( 0 != thisPtr->countSrqIO && VI_NULL != vi )
        {
            ViStatus errStatus;

            /* suspend event processing for now */
            errStatus = viEnableEvent( vi,
              VI_EVENT_SERVICE_REQ, VI_SUSPEND_HNDLR, VI_NULL);
            if( VI_SUCCESS > errStatus )
            {
                /* an existing error message has priority */
                if( VI_SUCCESS <= status)
                {
                    /* no existing error message, so set new error */
                    status = errStatus;
                }
            } /* if - an error occured */
            else
            {
                /* reset the srq counter */
                thisPtr->countSrqIO = 0;

                /* recursively traverse status byte */
                hpe1476_srqTraverse( vi, 0);

                /* re-enable event processing */
                errStatus = viEnableEvent( vi, VI_EVENT_SERVICE_REQ,
                  VI_HNDLR, VI_NULL);
                if( VI_SUCCESS > errStatus )
                {
                    /* an existing error message has priority */
                    if( VI_SUCCESS <= status)
                    {
                        /* no existing error message, so set new error */
                        status = errStatus;
                    }
                } /* if - an error occured */

            } /* else - no error in viEnableEvent */

        } /* if - srq needs servicing */

        /* unblock IO in SRQs */
        thisPtr->blockSrqIO = VI_FALSE;

	/* store the context dependent error number */
	hpe1476_CDE_ERROR(status);

	/* if an error occurs, see if the user has an error handler enabled */
        if( status != hpe1476_INSTR_ERROR_DETECTED &&
	    VI_SUCCESS > status &&
	    thisPtr->eventHandlerArray[
	        hpe1476_USER_ERROR_HANDLER_IDX].eventHandler)
	{
		/* call the users handler */
		thisPtr->eventHandlerArray[
		    hpe1476_USER_ERROR_HANDLER_IDX].eventHandler(
			vi, (ViInt32)status, 
			thisPtr->eventHandlerArray[
			    hpe1476_USER_ERROR_HANDLER_IDX].userData);
	}

	/* return the appropriate error number */
	return status;
}





/* 
 * Error Message Structures
 */

struct instrErrStruct
{
	ViStatus errStatus;
	ViString errMessage;
};

const static struct instrErrStruct instrErrMsgTable[] =
{
        { VI_ERROR_PARAMETER1,	VI_ERROR_PARAMETER1_MSG },
        { VI_ERROR_PARAMETER2,	VI_ERROR_PARAMETER2_MSG },
        { VI_ERROR_PARAMETER3,	VI_ERROR_PARAMETER3_MSG },
        { VI_ERROR_PARAMETER4,	VI_ERROR_PARAMETER4_MSG },
        { VI_ERROR_PARAMETER5,	VI_ERROR_PARAMETER5_MSG },
        { VI_ERROR_PARAMETER6,	VI_ERROR_PARAMETER6_MSG },
        { VI_ERROR_PARAMETER7,	VI_ERROR_PARAMETER7_MSG },
        { VI_ERROR_PARAMETER8,	VI_ERROR_PARAMETER8_MSG },
        { VI_ERROR_FAIL_ID_QUERY,	VI_ERROR_FAIL_ID_QUERY_MSG },

	{ hpe1476_INSTR_ERROR_INV_SESSION,	INSTR_ERROR_INV_SESSION_MSG },
        { hpe1476_INSTR_ERROR_NULL_PTR,	INSTR_ERROR_NULL_PTR_MSG },
        { hpe1476_INSTR_ERROR_RESET_FAILED,INSTR_ERROR_RESET_FAILED_MSG },
        { hpe1476_INSTR_ERROR_UNEXPECTED,	INSTR_ERROR_UNEXPECTED_MSG },
	{ hpe1476_INSTR_ERROR_DETECTED,	INSTR_ERROR_DETECTED_MSG },
	{ hpe1476_INSTR_ERROR_LOOKUP,	INSTR_ERROR_LOOKUP_MSG },
};

/* macros for testing parameters */
#define hpe1476_CHK_BOOLEAN( my_val, err ) if( hpe1476_chk_boolean( thisPtr, my_val) ) hpe1476_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1476_chk_boolean(
  struct hpe1476_globals *thisPtr,
  ViBoolean my_val)
{
   char message[hpe1476_ERR_MSG_LENGTH];
   if( (my_val != VI_TRUE) && (my_val != VI_FALSE) )
   {
      /* true = parameter is invalid */
      sprintf(message, hpe1476_MSG_BOOLEAN, my_val);	
      hpe1476_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }

   /* false = okay */
   return VI_FALSE;
}


#define hpe1476_CHK_REAL_RANGE( my_val, min, max, err ) if( hpe1476_chk_real_range( thisPtr, my_val, min, max) ) hpe1476_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1476_chk_real_range( 
  struct hpe1476_globals *thisPtr,
  ViReal64 my_val,
  ViReal64 min,
  ViReal64 max)
{
   char message[hpe1476_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1476_MSG_REAL, min, max, my_val);	
      hpe1476_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
#define hpe1476_CHK_INT_RANGE( my_val, min, max, err ) if( hpe1476_chk_int_range( thisPtr, my_val, min, max) ) hpe1476_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1476_chk_int_range( 
  struct hpe1476_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 min,
  ViInt16 max)
{
   char message[hpe1476_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1476_MSG_INT, min, max, my_val);	
      hpe1476_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1476_CHK_LONG_RANGE( my_val, min, max, err ) if( hpe1476_chk_long_range( thisPtr, my_val, min, max) ) hpe1476_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1476_chk_long_range( 
  struct hpe1476_globals *thisPtr,
  ViInt32 my_val,
  ViInt32 min,
  ViInt32 max)
{
   char message[hpe1476_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1476_MSG_LONG, min, max, my_val);	
      hpe1476_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1476_CHK_ENUM( my_val, limit, err ) if( hpe1476_chk_enum( thisPtr, my_val, limit) ) hpe1476_LOG_STATUS( vi, thisPtr, err);

/* utility routine which searches for a string in an array of strings. */
/* This is used by the CHK_ENUM macro */
static ViBoolean hpe1476_chk_enum (
  struct hpe1476_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 limit)
{
    char message[hpe1476_ERR_MSG_LENGTH];

    if ( ( my_val < 0 ) || (my_val > limit) )  			
    {								
        sprintf(message, hpe1476_MSG_INT, 0, limit, my_val);	
        hpe1476_CDE_MESSAGE(message);				
        /* true = parameter is invalid */
        return VI_TRUE;
    }

    return VI_FALSE;
}



/*  ==========================================================================  
     This function searches an array of strings for a specific string and     
     returns its index.  If successful, a VI_SUCCESS is returned, 
     else hpe1476_INSTR_ERROR_LOOKUP is returned.
    ======================================================================== */
ViStatus hpe1476_findIndex (struct hpe1476_globals *thisPtr,
			const char * const array_of_strings[],
			  /*last entry in array must be 0 */
			const char *string, /* string read from instrument */
			ViPInt16 index) /* result index */
{
    ViInt16 i;
    ViInt16 my_len;
    char search_str[20];
    char message[80];

    strcpy(search_str, string);

  /* get rid of newline if present in string */
  /* needed because %s includes newline in some VTL's */
    my_len = strlen(search_str);
    if (search_str[my_len - 1] == '\n')
       search_str[my_len - 1] = '\0';

    for (i = 0; array_of_strings[i]; i++)
    {
	if (!strcmp (array_of_strings[i], search_str))
	{
	    *index = i;
	    return VI_SUCCESS;
	}
    }

  /* if we got here, we did not find it */
    sprintf(message, hpe1476_MSG_NO_MATCH, search_str);	
    hpe1476_CDE_MESSAGE(message);				
    return hpe1476_INSTR_ERROR_LOOKUP;
}


static const ViStatus cscpi_error_to_vi_error[] = {
	VI_ERROR_SYSTEM_ERROR,
	VI_ERROR_SYSTEM_ERROR, /* should never happen*/
	VI_ERROR_FAIL_ID_QUERY, /* 2 wrong card type*/
	VI_ERROR_ALLOC,		/* 3 out of memory*/
	VI_ERROR_INV_RSRC_NAME, /* 4 bad address*/
	VI_ERROR_RSRC_NFOUND,	/* 5 open failed*/
	VI_ERROR_SYSTEM_ERROR, /*  6 open go system error*/
	VI_ERROR_SYSTEM_ERROR, /*  7 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  8 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  9 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  10 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  11 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  12 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  13 internal errors*/
	VI_ERROR_SYSTEM_ERROR, /*  14 internal errors*/
};


/****************************************************************************
hpe1476_init
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | InstrDesc                                         ViRsrc      IN
  |   ---------------------------------------------------------------------
  |  | The Instrument Description.
  |  |
  |  | Examples: VXI::5, GPIB-VXI::128::INSTR
   ------------------------------------------------------------------------
  | id_query                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | if( VI_TRUE) Perform In-System Verification.
  |  | if(VI_FALSE) Do not perform In-System Verification
   ------------------------------------------------------------------------
  | do_reset                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | IF( VI_TRUE) Perform Reset Operation.
  |  | if(VI_FALSE) Do not perform Reset operation
   ------------------------------------------------------------------------
  | vi                                                ViPSession  OUT
  |   ---------------------------------------------------------------------
  |  | Instrument Handle. This is VI_NULL if an error occurred
  |  | during the init.

*****************************************************************************/

ViStatus _VI_FUNC hpe1476_init (
  ViRsrc InstrDesc, 
  ViBoolean id_query,
  ViBoolean do_reset,
  ViPSession vi )
{
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;
	ViUInt16 intf;

	ViUInt16 manfId;
	ViUInt16 modelCode;

	ViInt32 idx;

	*vi = VI_NULL;

	/* Find the Default Resource Manager */
	errStatus = viOpenDefaultRM( &defRM);
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_SYSTEM_ERROR 
		 *         VI_ERROR_ALLOC
		 */
		hpe1476_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* Open a session to the instrument */
	errStatus = viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi);
	if( VI_SUCCESS > errStatus)
	{
		/* can't open using vtl, try register based open*/
		intf = VI_INTF_VXI;
		goto gpib_check_done;
	}
	
	if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
	{
		viClose( *vi);
		viClose( defRM);
		*vi = VI_NULL;
		hpe1476_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* check for 1406 internal driver by reading *IDN? of 0 and
		*IDN? of secondary for this address*/
	if (intf == VI_INTF_GPIB_VXI)
	{
		char gpibdesc[64];
		ViUInt16 num, primary, secondary;
		ViSession vi1406;
		char idn_buf[256];

		/* Can't use VI_ATTR_INTF_NUM since
		/* HP VTL 3.0 has a defect which returns 7 instead of 0*/
		/* Read the number from the descriptor instead, assuming
			it's GPIB-VXIn.
		/* Set the GPIB board number is the same had the GPIB-VXI 
			board number*/
		if (memcmp(InstrDesc, "GPIB-VXI", 8) || 
			InstrDesc[8] < '0' || InstrDesc[8] > '9')
			num = 0;
		else
			num = (ViUInt16) atoi(InstrDesc+8);
		viGetAttribute(*vi, VI_ATTR_GPIB_PRIMARY_ADDR, &primary);
		sprintf(gpibdesc, "GPIB%d::%d::0", num,	primary);
		errStatus = viOpen( defRM, gpibdesc, VI_NULL, VI_NULL, 
					&vi1406);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		/* try sending an IDN? to 1406 */
		viSetAttribute(vi1406, VI_ATTR_TMO_VALUE, 1000); 
		errStatus = viClear(vi1406);
		if (errStatus < VI_SUCCESS)
		{
			viClose(vi1406);
			goto gpib_check_done;
		}

		errStatus = viPrintf(vi1406, "*IDN?\n");
		if (errStatus < VI_SUCCESS)
		{
			viClose(vi1406);
			goto gpib_check_done;
		}

		/* it worked, try getting response and check it*/
		errStatus = viScanf( vi1406, "%t", idn_buf);
		viClose(vi1406);
		if (errStatus < VI_SUCCESS || 
				memcmp(idn_buf, "HEWLETT-PACKARD,E140", 20))
			goto gpib_check_done;

		/* try sending an IDN? to the card drivers*/
		viSetAttribute(*vi, VI_ATTR_TMO_VALUE, 1000); 
		errStatus = viClear(*vi);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		errStatus = viPrintf(*vi, "*IDN?\n");
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		/* it worked, try getting response*/
		errStatus = viScanf( *vi, "%t", idn_buf);
		if (errStatus < VI_SUCCESS)
			goto gpib_check_done;

		/* got an answer so switch vi to gpib mode */
		viGetAttribute(*vi, VI_ATTR_GPIB_SECONDARY_ADDR, &secondary);
		sprintf(gpibdesc, "GPIB%d::%d::%d", num, primary, secondary);
		viClose(*vi);
		errStatus = viOpen( defRM, gpibdesc, VI_NULL, VI_NULL, vi);
		if( errStatus < VI_SUCCESS)
		{ /* this should never happen*/	
			viClose( defRM);
			*vi = VI_NULL;
			hpe1476_LOG_STATUS( *vi, NULL, errStatus);
		}
		intf = VI_INTF_GPIB;
	}

gpib_check_done:

	/* get memory for instance specific globals */
	thisPtr = (struct hpe1476_globals *)malloc(sizeof( struct hpe1476_globals) );
	if( 0 == thisPtr)
	{
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1476_LOG_STATUS(*vi, NULL, VI_ERROR_ALLOC);
	}

	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		/* associate memory with session, should not fail because
		 *   session is valid; and attribute is defined, supported,
		 *   and writable.
		 */
		errStatus = viSetAttribute( 
		              *vi, 
			      VI_ATTR_USER_DATA, 
		              (ViAttrState)thisPtr); 
		if( VI_SUCCESS > errStatus)
		{
			viClose( *vi);
			viClose( defRM);	/* also closes vi session */
			*vi = VI_NULL;
			hpe1476_LOG_STATUS(*vi, NULL, errStatus);
		}
		thisPtr->e1406 = 1;
	}
	else
	{
		/* reg based drivers open their own sessions*/
		viClose( defRM);	/* also closes vi session */

		/* start pos if not started*/
		os_init();/* STARTUP */

		{extern switch_header_fn();*vi = os_open(InstrDesc,switch_header_fn);}

		if (!*vi)
		{
			if (cscpi_open_error >= 0 && cscpi_open_error <= 14)
				hpe1476_LOG_STATUS(*vi, NULL,
			       	cscpi_error_to_vi_error[cscpi_open_error]);
			else
				hpe1476_LOG_STATUS(*vi, NULL, VI_ERROR_SYSTEM_ERROR);
		}

		/* associate memory with session */
		instr_setdata(*vi, thisPtr); 
		thisPtr->e1406 = 0;
	}
	/* initialize instance globals */
	viGetAttribute( *vi, VI_ATTR_RM_SESSION, &thisPtr->defRMSession);
        thisPtr->errNumber = VI_SUCCESS;
	thisPtr->errFuncName[0] = 0;
	thisPtr->errMessage[0] = 0;
	thisPtr->errQueryDetect = VI_FALSE;
        thisPtr->blockSrqIO = VI_FALSE;
        thisPtr->countSrqIO = 0;


	for( idx=0;idx<hpe1476_MAX_STAT_REG; idx++)
	{
	    thisPtr->driverEventArray[idx] = 0;
	}
	thisPtr->numEventHandlers = 0;
	for( idx=0;idx<hpe1476_MAX_STAT_HAP; idx++)
	{
	    thisPtr->eventHandlerArray[idx].eventHandler = NULL;
	    thisPtr->eventHandlerArray[idx].userData = 0;
	}
	
	if( VI_TRUE == do_reset ) 
	{
		/* call the reset function to reset the instrument */
		if( hpe1476_dcl(*vi)<VI_SUCCESS || hpe1476_reset(*vi)< VI_SUCCESS )
		{
			/* ignore any errors in PREFIX_close */
			hpe1476_close( *vi);
                        *vi=VI_NULL;
			hpe1476_LOG_STATUS( *vi, NULL, 
			  hpe1476_INSTR_ERROR_RESET_FAILED);
		}
                Sleep(100);
	}

	if( VI_TRUE == id_query ) 
	{
		char     idn_buf[256];

		/* find the interface type */
		if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
		{	
			/* Errors: VI_ERROR_NSUP_ATTR */

			/* ignore any errors in PREFIX_close */
			hpe1476_close( *vi);
                        *vi=VI_NULL;
	
			hpe1476_LOG_STATUS( *vi, NULL, errStatus);
		}

		switch( intf)
		{
			case VI_INTF_GPIB:
				if (
				    viClear(*vi) < VI_SUCCESS ||
				    viPrintf(*vi, "*IDN?\n") < VI_SUCCESS ||
				    viScanf(*vi, "%t", idn_buf) < VI_SUCCESS ||
				    /* check for a idn match */
				    strncmp(idn_buf, hpe1476_IDN_STRING, 
					    strlen(hpe1476_IDN_STRING))   
				    )
                		{   
					/* ignore any errors in PREFIX_close */
					hpe1476_close( *vi);
                                        *vi=VI_NULL;
					hpe1476_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
                		}

				break;

			case VI_INTF_VXI:
			case VI_INTF_GPIB_VXI:
	
				/* find the VXI manfacturer's ID */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MANF_ID, &manfId);
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
	
					/* ignore any errors in PREFIX_close */
					hpe1476_close( *vi);
                                        *vi=VI_NULL;
	
					hpe1476_LOG_STATUS( *vi, NULL,
					  errStatus);
				}

				/* find the instrument's model code */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MODEL_CODE, 
		       		(ViPAttrState)(&modelCode)); 
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
					/* Note: this should never happen 
					 *   with a VXI instrument
					 */

					/* ignore any errors in PREFIX_close */
					hpe1476_close( *vi);
                                        *vi=VI_NULL;
					hpe1476_LOG_STATUS( *vi, NULL, errStatus);
				}

				if( (manfId != hpe1476_MANF_ID) || 
				    ((modelCode&0xFFF8) != hpe1476_MODEL_CODE) )
				{
					/* ignore any errors in PREFIX_close */
					hpe1476_close( *vi);
                                        *vi = VI_NULL;

					hpe1476_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
				}

				break;

			default:
				/* ignore any errors in PREFIX_close */
				hpe1476_close( *vi);
				*vi = VI_NULL;
				hpe1476_LOG_STATUS( *vi, NULL,
				  hpe1476_INSTR_ERROR_UNEXPECTED);

		}	

	} /* if - id_query */

	hpe1476_LOG_STATUS( *vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1476_close
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_close (
  ViSession vi )
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_INV_SESSION	*/
		hpe1476_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1476_DEBUG_CHK_THIS( vi, thisPtr);

	/* free memory */
	if( thisPtr)		
	{
            ViStatus e;
	    /* make sure there is something to free */
	    if (thisPtr->e1406)
	    {
		/* retrieve Resource Management session */
		defRM = thisPtr->defRMSession;
		/* close the vi and RM sessions */
		e = viClose( defRM);
		free( thisPtr);
                return e;
	    }
	    else
	    {
		free( thisPtr);
		os_close(vi);
	    }

	}	
        else
        {
		os_close(vi);
        }

	return VI_SUCCESS;
}


/****************************************************************************
hpe1476_reset
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_reset (
  ViSession vi )
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_INV_SESSION	*/
		hpe1476_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1476_CDE_INIT( "hpe1476_reset");

	if (thisPtr->e1406)
	{
		errStatus = viPrintf( vi, "*RST\n");
	}
	else
	{
		{{ 
/*#line 1065 "hpe1476.cs"*/
{extern sw_com_rst();instr_send(vi,sw_com_rst,(void*)0);}}}
	}
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
	}

	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}


/****************************************************************************
hpe1476_self_test
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | test_result                                       ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | Numeric result from self-test operation
  |  |
  |  | 0 = no error ( test passed)
  |  | 1 = failure
   ------------------------------------------------------------------------
  | test_message                                      ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Self-test status message.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_self_test (
  ViSession vi,
  ViPInt16 test_result,
  ViChar _VI_FAR test_message[])
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*test_result = -1; 
	test_message[0] = 0; 

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_INV_SESSION	*/
		hpe1476_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1476_CDE_INIT( "hpe1476_self_test");

	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

		errStatus = viPrintf( vi, "*TST?\n");
		if( VI_SUCCESS > errStatus)
		{	
		
			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

		errStatus = viScanf( vi, "%hd%*t", test_result);
	}
	else
	{
		{{ struct {double p1[1];} out_;
/*#line 1144 "hpe1476.cs"*/
{extern sw_com_tst_q();if (!instr_query(vi,sw_com_tst_q,(void*)0,&out_)){
/*#line 1144 "hpe1476.cs"*/
*test_result=(*(HPSL_INT16*)out_.p1).num;}}}}
	}
	if( VI_SUCCESS > errStatus)
	{
		*test_result = -1; 
		
		hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
	}

	switch(*test_result % 100)
	{
           case 0:
              sprintf(test_message, hpe1476_MSG_SELF_TEST_PASSED);
              break;
	   /* from self_test.h in firmware source*/
           case 1:
              sprintf(test_message, "Firmware failed on card %d", 
			*test_result/100);
              break;
           case 2:
              sprintf(test_message, "Bus error on card %d", 
			*test_result/100);
              break;
           case 3:
              sprintf(test_message, "Card ID failed on card %d", 
			*test_result/100);
              break;
           case 4:
              sprintf(test_message, "Bad data on card %d", 
			*test_result/100);
              break;
           case 5:
              sprintf(test_message, "Bad register values on card %d", 
			*test_result/100);
              break;
           case 10:
              sprintf(test_message, "Interrupt failed on card %d", 
			*test_result/100);
              break;
           case 11:
              sprintf(test_message, "Busy timer failed on card %d", 
			*test_result/100);
              break;
           case 13:
              sprintf(test_message, "Interrupt too soon on card %d", 
			*test_result/100);
              break;
           case 30:
              sprintf(test_message, "Module ID incorrect on card %d", 
			*test_result/100);
              break;
           default:
              sprintf(test_message, hpe1476_MSG_SELF_TEST_FAILED);
              break;
        }

	
	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1476_error_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | error_number                                      ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | Instruments error code
   ------------------------------------------------------------------------
  | error_message                                     ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Instrument's error message.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_error_query (
  ViSession vi,
  ViPInt32 error_number,
  ViChar _VI_FAR error_message[])
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*error_number = -1; 
	error_message[0] = 0; 

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_INV_SESSION	*/
		hpe1476_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1476_CDE_INIT( "hpe1476_error_query");

	/* error_message[256] = 0; */

	/* get the error number */
	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

	        errStatus = viPrintf( vi, "SYST:ERR?\n");
		if( VI_SUCCESS > errStatus)
		{

			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get the error number */
		errStatus = viScanf( vi, "%ld,%t", error_number, error_message);
		/* check for error during the scan */
        	if( VI_SUCCESS > errStatus)
		{
			*error_number = -1; 
			error_message[0] = 0; 
			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get rid of extra LF at the end of the error_message */
		error_message[strlen(error_message)-1] = 0;
	}
	else
	{
        	{{ struct {double p1[1];double p2[1];} out_;
/*#line 1282 "hpe1476.cs"*/
{extern syst_err_q();if (!instr_query(vi,syst_err_q,(void*)0,&out_)){
/*#line 1282 "hpe1476.cs"*/
*error_number=(*(HPSL_INT16*)out_.p1).num;
/*#line 1282 "hpe1476.cs"*/
(void)strcpy((error_message),*(HPSL_STRING*)out_.p2);}}}}
	}

	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1476_error_message
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init().
  |  | This may be VI_NULL.
   ------------------------------------------------------------------------
  | error_number                                      ViStatus    IN
  |   ---------------------------------------------------------------------
  |  | The error return value from an instrument driver function
   ------------------------------------------------------------------------
  | message                                           ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Error message string.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_error_message (
  ViSession vi,
  ViStatus error_number,
  ViChar _VI_FAR message[])
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;  
	ViInt32 idx;

	/* initialize output parameters */
	message[0] = 0;

	thisPtr = NULL;

	/* try to find a thisPtr */
	if( VI_NULL != vi)
	{
	    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
		if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
		{
			errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
		}
		else
		{
			errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
		}
		if( VI_SUCCESS > errStatus)
		{
			/* Errors: VI_ERROR_INV_SESSION	*/
			strcpy( message, hpe1476_MSG_INVALID_VI);
			hpe1476_LOG_STATUS( vi, NULL, errStatus);
		}
		hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
	} 

	if( VI_SUCCESS == error_number)
	{
		sprintf( message, hpe1476_MSG_NO_ERRORS);
		hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* return the static error message */
	for(idx=0; 
	    idx < (sizeof instrErrMsgTable / 
	           sizeof( struct instrErrStruct));
	    idx++)
	{
		/* check for a matching error number */
		if( instrErrMsgTable[idx].errStatus == error_number)
		{
			if( (thisPtr) &&
			    (thisPtr->errNumber == error_number))
			{
				/* context dependent error
				 * message is available.
				 */
				sprintf( message,
				  "%s " hpe1476_MSG_IN_FUNCTION " %s() %s",
		          	  instrErrMsgTable[idx].errMessage,
				  thisPtr->errFuncName,
				  thisPtr->errMessage);
			}
			else
			{
				/* No context dependent eror 
				 * message available so copy 
				 * the static error message
				 */
				strcpy( message,
		          	  instrErrMsgTable[idx].errMessage);

			}
            
			hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
		}
	}


	/* try VTL/VISA to see if it exists there 
	 *
	 * At this point vi is either VALID or VI_NULL
	 */
	errStatus = viStatusDesc(vi, error_number, message);
	if( VI_SUCCESS == errStatus)
	{
		/* check for a context dependent error message */
		if( (thisPtr) &&
		    (thisPtr->errNumber == error_number))
		{
			/* context dependent error
			 * message is available.
			 */
			strcat( message, " ");
			strcat( message, hpe1476_MSG_IN_FUNCTION);
			strcat( message, " ");
			strcat( message, thisPtr->errFuncName);
			strcat( message, "() ");
			strcat( message, thisPtr->errMessage);
		}

		/* VTL found an error message, so return success */
		hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* if we have a VI_NULL, then we need to return a error message */
	if( VI_NULL == vi)
	{
		strcpy(message, hpe1476_MSG_VI_OPEN_ERR);
		hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	} 

	/* user passed in a invalid status */
	sprintf( message, 
	  hpe1476_MSG_INVALID_STATUS
	  "  %ld"
	  hpe1476_MSG_INVALID_STATUS_VALUE
	  , (long)error_number );
	
	hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
}


/****************************************************************************
hpe1476_revision_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession      IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | driver_rev                                ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument driver revision.  This is limited to 256 characters.
   ------------------------------------------------------------------------
  | instr_rev                                 ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument firmware revision.  This is limited to 256 characters.

*****************************************************************************/

ViStatus _VI_FUNC hpe1476_revision_query (
  ViSession vi,
  ViChar _VI_FAR driver_rev[],
  ViChar _VI_FAR instr_rev[])
{
	ViUInt16 intf;
	struct hpe1476_globals *thisPtr;
	ViStatus errStatus;
        char temp_str[256];		/* temp hold for instr rev string */
        char *last_comma;		/* last comma in *IDN string */

	/* initialize output parameters */
        driver_rev[0] = 0; 
	instr_rev[0] = 0; 

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
			vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_INV_SESSION	*/
		hpe1476_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1476_CDE_INIT( "hpe1476_revision_query");

	sprintf( driver_rev, "%s", hpe1476_REV_CODE);

	if (thisPtr->e1406)
	{
		thisPtr->blockSrqIO = VI_TRUE;

		if( errStatus = viPrintf( vi, "*IDN?\n"))
		{
        		driver_rev[0] = 0; 

			hpe1476_LOG_STATUS( vi, thisPtr, errStatus); 
		}

		if( errStatus = viScanf( vi, "%s%*t", temp_str))
		{
        		driver_rev[0] = 0; 
			instr_rev[0] = 0; 
		 
			hpe1476_LOG_STATUS( vi, thisPtr, errStatus); 
		}
	}
	else
	{
		{{ struct {double p1[1];} out_;
/*#line 1506 "hpe1476.cs"*/
{extern com_idn_q();if (!instr_query(vi,com_idn_q,(void*)0,&out_)){
/*#line 1506 "hpe1476.cs"*/
(void)memcpy((temp_str),(*(HPSL_ARB_ASCII*)out_.p1).buffer,(*(HPSL_ARB_ASCII*)out_.p1).length);
/*#line 1506 "hpe1476.cs"*/
(temp_str)[(*(HPSL_ARB_ASCII*)out_.p1).length]='\0';}}}}
	}
	
        last_comma = strrchr(temp_str,',');
	/* error and exit if last comma not found */
        if (!last_comma) 
        {
           instr_rev[0] = 0;
	   hpe1476_CDE_MESSAGE("no last comma found in IDN string" ); 
	   hpe1476_LOG_STATUS( vi, thisPtr, hpe1476_INSTR_ERROR_UNEXPECTED); 
        }

        strcpy(instr_rev, last_comma+1);
	
	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS); 
}

/****************************************************************************
hpe1476_timeOut
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | timeOut                                           ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | This value sets the I/O timeout for all functions in
  |  | the driver. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Changes the timeout value of the instrument.  Input is in     */
/*           milliseconds.                                                 */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1476_timeOut (ViSession vi, ViInt32 timeOut)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;
   

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(
			vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus)
   {
	/* Errors: VI_ERROR_INV_SESSION	*/
	hpe1476_LOG_STATUS( vi, NULL, errStatus);
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_timeOut" );

   hpe1476_CHK_LONG_RANGE( timeOut, 1, 2147483647, VI_ERROR_PARAMETER2 );

   if (thisPtr->e1406)
   {
	errStatus = viSetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut);
   }
   else
   {
	errStatus = instr_settimeout(vi, timeOut);
   }
   if ( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_timeOut_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | pTimeOut                                          ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | This is the minimum timeout period that the driver
  |  | can be set to. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Returns the current setting of the timeout value of the       */
/*           instrument in milliseconds.                                   */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1476_timeOut_Q (ViSession vi, ViPInt32 timeOut)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS( vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_timeOut_Q" );

   if (thisPtr->e1406)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut );
   }
   else
   {
	errStatus = instr_gettimeout(vi, timeOut );
   }
   if( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_errorQueryDetect
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | errorQueryDetect                                  ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | Boolean which enables (VI_TRUE) or disables (VI_FALSE)
  |  | automatic instrument error querying.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_errorQueryDetect( 
  ViSession vi, 
  ViBoolean errDetect)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus ) 
   { 
	hpe1476_LOG_STATUS( vi, NULL, errStatus ); 
   }

   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_errorQueryDetect" );

   hpe1476_CHK_BOOLEAN( errDetect, VI_ERROR_PARAMETER2 );

   thisPtr->errQueryDetect = errDetect;

   hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_errorQueryDetect_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | pErrDetect                                        ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | Boolean indicating if automatic instrument error
  |  | querying is performed.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_errorQueryDetect_Q( 
  ViSession vi, 
  ViPBoolean pErrDetect)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus ) 
   { 
	hpe1476_LOG_STATUS( vi, NULL, errStatus ); 
   }

   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_errorQueryDetect_Q" );

   *pErrDetect = thisPtr->errQueryDetect;

   hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_dcl
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_dcl( 
  ViSession vi)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus ) 
   { 
	hpe1476_LOG_STATUS( vi, NULL, errStatus ); 
   }

   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_dcl" );

   if (thisPtr->e1406)
   {
	errStatus = viClear(vi);
	if( VI_SUCCESS > errStatus) { hpe1476_LOG_STATUS( vi, thisPtr, errStatus ); }	
   }
   else
   {
   	{extern switch_hpsl_dcas();instr_clear(vi,switch_hpsl_dcas);}
   }

   hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_opc
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_opc (ViSession vi )
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_opc" );

   if (thisPtr->e1406)
   {
	errStatus = viPrintf(vi,"*OPC\n");
	if( VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	{{ 
/*#line 1823 "hpe1476.cs"*/
{extern sw_com_opc();instr_send(vi,sw_com_opc,(void*)0);}}}
   }

   hpe1476_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_opc_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | opc                                               ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | returns a 1 when all pending instruments commands are completed

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_opc_Q (ViSession vi, ViPInt16 opc)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_opc_Q" );
 
   if (thisPtr->e1406)
   {
	thisPtr->blockSrqIO = VI_TRUE;

	errStatus = viPrintf(vi,"*OPC?\n");
	if( VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}

	errStatus = viScanf(vi, "%hd%*t", opc);
	if( VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	{{ struct {double p1[1];} out_;
/*#line 1883 "hpe1476.cs"*/
{extern sw_com_opc_q();if (!instr_query(vi,sw_com_opc_q,(void*)0,&out_)){
/*#line 1883 "hpe1476.cs"*/
*opc=1;}}}}
   }

   hpe1476_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_wai
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_wai (ViSession vi)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if ( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_wai" );

   if (thisPtr->e1406)
   {
	errStatus = viPrintf(vi,"*WAI\n");
	if (VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	{{ 
/*#line 1931 "hpe1476.cs"*/
{extern sw_com_wai();instr_send(vi,sw_com_wai,(void*)0);}}}
   }

   hpe1476_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1476_trg
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_trg (ViSession vi)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if ( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_trg" );

   if (thisPtr->e1406)
   {
	errStatus = viPrintf(vi,"*TRG\n");
	if ( VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	{{ 
/*#line 1979 "hpe1476.cs"*/
{extern sw_special_trig();instr_send(vi,sw_special_trig,(void*)0);}}}
   }

   hpe1476_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}


/****************************************************************************
hpe1476_readStatusByte_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | statusByte                                        ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | returns the contents of the status byte

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_readStatusByte_Q(ViSession vi, ViPInt16 statusByte)
{
   ViUInt16 intf;
   ViStatus errStatus = 0;
   struct hpe1476_globals *thisPtr;
   ViUInt16 stb;

   viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
   if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
   {
	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   }
   else
   {
	errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
   }
   if ( VI_SUCCESS > errStatus)
   {
      hpe1476_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1476_CDE_INIT( "hpe1476_readStatusByte_Q" );

   if (thisPtr->e1406)
   {
	errStatus = viReadSTB(vi,&stb);
	if( VI_SUCCESS > errStatus)
	{
		hpe1476_LOG_STATUS(vi, thisPtr, errStatus );
	}
   }
   else
   {
	instr_poll(vi,&stb);
   }

   *statusByte = (ViInt16)stb;

   hpe1476_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

ViInt32 hpe1476_statusHap[hpe1476_MAX_STAT_HAP] = {
hpe1476_USER_ERROR_HANDLER,

hpe1476_ESR_OPC,
hpe1476_ESR_QUERY_ERROR,
hpe1476_ESR_DEVICE_DEPENDENT_ERROR,
hpe1476_ESR_EXECUTION_ERROR,
hpe1476_ESR_COMMAND_ERROR,

hpe1476_OPER_SCAN_DONE
};


/* Assumes we have driver copies of the event register.  This is needed
 * because in IEEE 488.2, the event register are cleared after they are 
 * read.  Since the event register contains several events, we need to
 * keep this information around to pass back to the user.
 */


struct hpe1476_statusAccess {
ViInt32 registerIdx; 
ViString condQry;
ViString eventQry;
ViString enableCmd;
};

const struct hpe1476_statusAccess hpe1476_accessInfo[hpe1476_MAX_STAT_REG] = {
{0, 	"",	 		"*STB?",		"*SRE"},
{400, 	"STAT:QUES:COND?",	"STAT:QUES:EVEN?",	"STAT:QUES:ENAB"},
{600,	"",			"*ESR?",			"*ESE"},
{800, 	"STAT:OPER:COND?",	"STAT:OPER:EVEN?",	"STAT:OPER:ENAB"}
};

/* this will return the index associated with the happening */
ViBoolean hpe1476_findHappeningIdx(
  ViInt32 happening,
  ViPInt32 pIdx)
{
	/* Note: this is a linear search, for faster access this
	 * could be done as a binary search since the data is arrange
	 * in order numerically.
	 */
	for( *pIdx=0; *pIdx<hpe1476_MAX_STAT_HAP; *pIdx = *pIdx + 1)
	{
		if( hpe1476_statusHap[*pIdx] == happening)
		{
			return VI_TRUE;
		}	
	}

	return VI_FALSE;
}

/* this will return the index that corresponds with regNum */
static ViBoolean hpe1476_findAccessIdx(
  ViInt32 regNum,
  ViPInt32 pIdx )
{
	for(*pIdx=0; *pIdx<hpe1476_MAX_STAT_REG; *pIdx = *pIdx + 1)
	{
		if( regNum == hpe1476_accessInfo[*pIdx].registerIdx)
		{	
			return VI_TRUE;
		}
	}
	return VI_FALSE;
}

static ViStatus hpe1476_readAllEvents( 
  ViSession vi)
{
	ViUInt16 intf;
	ViInt32 idx;
	ViStatus errStatus;
	ViInt32 eventResp;
        struct hpe1476_globals *thisPtr;

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
   	if( VI_SUCCESS > errStatus) 
   	{
   	   hpe1476_LOG_STATUS( vi, NULL, errStatus );
   	}
   	hpe1476_DEBUG_CHK_THIS( vi, thisPtr );
        

	/* read all events and update driver events */
	for(idx=0; idx<hpe1476_MAX_STAT_REG; idx++)
	{
		/* if there is an event query */
		if( 0 != strlen(hpe1476_accessInfo[idx].eventQry))
		{
			/* okay - query the instrument for event */
			if (thisPtr->e1406)
			{
				thisPtr->blockSrqIO = VI_TRUE;

				errStatus = viPrintf(vi,
				  "%s\n", hpe1476_accessInfo[idx].eventQry);
				if( VI_SUCCESS > errStatus)
				{
					return errStatus;
				}

				/* get the response */
				errStatus = viScanf(vi, "%ld%*t", &eventResp);
				if( VI_SUCCESS > errStatus )
				{
					return errStatus;
				}
			}
			else
			{
				ViChar resp_str[100];

				cscpi_exe(vi,
					  hpe1476_accessInfo[idx].eventQry,
					  strlen(hpe1476_accessInfo[idx].eventQry),
					  resp_str,
					  sizeof(resp_str)
					  );
				/* get the response */
				eventResp = atol(resp_str);
			}

			/* update the driver events copy of instr events */
			thisPtr->driverEventArray[idx] = 	
			  thisPtr->driverEventArray[idx] | eventResp;

		} /* if - an event query exists */

	} /* for - all status registers */

	return VI_SUCCESS;

} /* hpe1476_readAllEvents() */


/* this recursive routine traverse the IEEE 488.2 status structures and
 * processes events.
 */
static void hpe1476_srqTraverse(
  ViSession vi,
  ViInt32 eventReg)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
	ViInt32 accessIdx;
	ViInt32 hapIdx;
	ViUInt16 Status;
	ViInt32 eventVal;
	ViInt32 happening;
	ViInt32 bitPos;
	ViStatus errStatus;

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
   	if( VI_SUCCESS > errStatus) 
	{
		/* error couldn't get attribute */
		return;  
	}

	if( VI_FALSE == hpe1476_findAccessIdx( eventReg, &accessIdx))
	{
		/* couldn't find register num 
		 *   don't know what to do so leave 
		 */
		return;
	}

	if( 0 == eventReg )
	{
		/* need to read status byte through VTL instead of SCPI */
		if (thisPtr->e1406)
		{
			if( viReadSTB( vi, &Status))
			{
				/* an error occured */
				return;
			}
		}
		else
		{
			instr_poll(vi,&Status);
		}
		eventVal = (ViInt32)Status;
	}
	else
	{
		/* find the event in the table */
		if( 0 != strlen(hpe1476_accessInfo[accessIdx].eventQry))
		{
			/* query instrument to find what event(s) occured. */
			if (thisPtr->e1406)
			{
				if( viPrintf(vi,"%s\n",
					 hpe1476_accessInfo[accessIdx].eventQry))
				{
					/* an error occured */
					return;
				}

				if( viScanf(vi,"%ld%*t\n", &eventVal)) 
				{
					/* an error occured */
					return;
				}
			}
			else
			{
				ViChar resp_str[100];

				cscpi_exe(vi,
					  hpe1476_accessInfo[accessIdx].eventQry,
					  strlen(hpe1476_accessInfo[accessIdx].eventQry),
					  resp_str,
					  sizeof(resp_str)
					  );
				/* get the response */
				eventVal = atol(resp_str);
			}
		}
		else
		{
			/* error - no events for this status register */
			return;
		}
	} /* else - not status byte */

	/* update driver events */
	thisPtr->driverEventArray[accessIdx] = 
	  thisPtr->driverEventArray[accessIdx] | eventVal;

	for( bitPos = 0; bitPos < 16; bitPos++)
	{
		/* check for an event occurance */
		if( eventVal & (1 << bitPos) )
		{
			/* find happening index */
			happening = eventReg + bitPos + 1;
			if( VI_TRUE == 
			       hpe1476_findHappeningIdx( happening, &hapIdx) )
			{
				/* does event have a handler enabled? */
				if( thisPtr->eventHandlerArray[hapIdx].eventHandler)
				{
					/* call the users handler */
			    	thisPtr->eventHandlerArray[hapIdx].eventHandler(
			    	  vi, happening, 
			    	  thisPtr->eventHandlerArray[hapIdx].userData);
				}				
			}

		} /* if - event occured */

		/* check for more status registers */
		if( VI_TRUE == hpe1476_findAccessIdx( happening*100, &accessIdx))
		{
			/* need to traverse these other registers */
			hpe1476_srqTraverse( vi, happening*100);
		}

	} /* for - all bits in the event */
}


/* this is the SRQ event handler for hpib*/
ViStatus _VI_FUNCH hpe1476_srqHdlr(
  ViSession vi,
  ViEventType eventType,
  ViEvent event,
  ViAddr userData)
{
	/* the only reason we should get called is for a Service Request */	

	/* for VTL 3.0 we always need to return VI_SUCCESS, therefore
	 * we will ignore any VTL errors at this level.
	 */

        struct hpe1476_globals *thisPtr;

        if(VI_SUCCESS > 
	  viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )
	{
		/* error couldn't get attribute, 
		 * but we lie and return VI_SUCCESS */
		return VI_SUCCESS;  
	}

	/* check if it is okay to do I/O in SRQ handler */
	if( VI_TRUE == thisPtr->blockSrqIO)
	{
	    /* not okay to do I/O so just update srq count */
	    thisPtr->countSrqIO = thisPtr->countSrqIO + 1;
     	    return VI_SUCCESS;  
	}

	/* suspend event processing for now */
	viEnableEvent( vi, VI_EVENT_SERVICE_REQ, VI_SUSPEND_HNDLR, VI_NULL);

	/* recursively traverse status byte */
	hpe1476_srqTraverse( vi, 0);

	/* re-enable event processing */
       	viEnableEvent( vi, VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);

	/* always return VI_SUCCESS */
	return VI_SUCCESS;
} 


/* this is the simulated SRQ event handler for register based*/
ViStatus _VI_FUNCH hpe1476_rb_srqHdlr(
  ViSession vi,
  ViEventType eventType,
  ViEvent event,
  ViAddr userData)
{
	hpe1476_srqTraverse( vi, 0);
	return VI_SUCCESS;
} 



/****************************************************************************
hpe1476_statEven_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens. These can refer
  |  | to conditions or events. All of the conditions listed in the
  |  | hpe1476_statCond_Q() function will be detected as events as
  |  | well.   An event may be registered when a condition changes
  |  | state from  VI_FALSE to VI_TRUE.
  |  |
   ------------------------------------------------------------------------
  | pEvent                                            ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | VI_TRUE  = event occured sometime between event readings
  |  | VI_FALSE = the event did not occur between event readings

*****************************************************************************/

ViStatus _VI_FUNC hpe1476_statEven_Q(
  ViSession vi,
  ViInt32	happening,
  ViPBoolean	pEvent
)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	ViInt32 accessIdx;
	ViInt32 event_q;
	ViInt32 regNum;
	ViInt32 bitNum;
	ViInt32 mask;
	char errMsg[256];

        /* initialize output parameters */
	*pEvent = VI_FALSE;

    	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1476_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1476_CDE_INIT( "hpe1476_statEven_Q");

	/* check validity of happening */
	if( (hpe1476_USER_ERROR_HANDLER == happening) ||
            (VI_FALSE == hpe1476_findHappeningIdx( happening, &hapIdx) ))
	{
		sprintf(errMsg,"%hd " hpe1476_MSG_INVALID_HAPPENING, happening);
		hpe1476_CDE_MESSAGE( errMsg);
		
		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	regNum = happening / 100;
	bitNum = happening % 100;

	/* get access index */
	if( VI_FALSE == hpe1476_findAccessIdx( regNum * 100, &accessIdx))
	{
		
		hpe1476_LOG_STATUS( vi, thisPtr, hpe1476_INSTR_ERROR_UNEXPECTED);
	}

	/* see if we can query the event from the instrument */
	if( 0 == strlen(hpe1476_accessInfo[accessIdx].eventQry))
	{
		char errMsg[256];

		/* unable to query the event */
		sprintf(errMsg, hpe1476_MSG_EVENT
		  "%hd" hpe1476_MSG_NOT_QUERIABLE, happening);
		hpe1476_CDE_MESSAGE(errMsg); 
		
		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}
	else
	{
		/* okay - query the instrument for event */
		if (thisPtr->e1406)
		{
			thisPtr->blockSrqIO = VI_TRUE;

			errStatus = viPrintf(vi,
			  "%s\n", hpe1476_accessInfo[accessIdx].eventQry);
			if( 0 > errStatus)
			{

				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
			}

			/* get the response */
			errStatus = viScanf(vi, "%ld%*t", &event_q);
			if( 0 > errStatus )
			{

				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
			}
		}
		else
		{
			ViChar resp_str[100];

			cscpi_exe(vi,
				  hpe1476_accessInfo[accessIdx].eventQry,
				  strlen(hpe1476_accessInfo[accessIdx].eventQry),
				  resp_str,
				  sizeof(resp_str)
				  );
			/* get the response */
			event_q = atol(resp_str);
		}

		/* update the driver events copy of instrument events */
		thisPtr->driverEventArray[accessIdx] = 	
		  thisPtr->driverEventArray[accessIdx] | event_q;

		/* check state of event bit */
		mask = 1 << (bitNum -1);
		if( thisPtr->driverEventArray[accessIdx] & mask )
		{
			*pEvent = VI_TRUE;

			/* clear event bit in driver events */
			thisPtr->driverEventArray[accessIdx] = 	
		 	  thisPtr->driverEventArray[accessIdx] & (~mask);
		}
		else
		{
			*pEvent = VI_FALSE;

			/* event bit in driver event array
			 * is already cleared 
			 */
		}
	}

	
	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	


/****************************************************************************
hpe1476_statEvenHdlr
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens. These can refer to
  |  | conditions or events. Happenings are enumerated as ViInt32
  |  | numbers.  Only events can have happenings associated with it.
  |  |
   ------------------------------------------------------------------------
  | eventHandler                           hpe1476_InstrEventHandler  IN
  |   ---------------------------------------------------------------------
  |  | This is either NULL or a pointer to the user specified
  |  | event handler.  A NULL disables the event handler.
  |  |
  |  | An event handler has the following prototype:
  |  |
  |  | typedef void (_VI_FUNCH _VI_PTR  hpe1476_InstrEventHandler)(
  |  | ViSession vi,
  |  | ViInt32 happening,
  |  | ViAddr userData
  |  | );
   ------------------------------------------------------------------------
  | userData                                          ViAddr      IN
  |   ---------------------------------------------------------------------
  |  | This is a pointer that is passed to the handler when the
  |  | specified event occurs.  This can be used by the programmer
  |  | to pass additional information to the handler.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_statEvenHdlr(
  ViSession vi,
  ViInt32 happening,
  hpe1476_InstrEventHandler eventHandler,
  ViAddr userData
)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	ViInt32 accessIdx;
	ViInt32 propagate;
	ViInt32 regNum;
	ViInt32 bitNum;
	ViInt32 enableMask;
	char errMsg[80];

        /* initialize output parameters */

	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1476_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1476_CDE_INIT( "hpe1476_statEvenHdlr");

	if( hpe1476_USER_ERROR_HANDLER == happening)
	{
		/* store handler and user data */
		thisPtr->eventHandlerArray[
                   hpe1476_USER_ERROR_HANDLER_IDX].eventHandler = eventHandler;
		thisPtr->eventHandlerArray[
                   hpe1476_USER_ERROR_HANDLER_IDX].userData = userData;
		hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* check validity of happening */
	if( VI_FALSE == hpe1476_findHappeningIdx( happening, &hapIdx) )
	{

		sprintf(errMsg,"%ld " hpe1476_MSG_INVALID_HAPPENING, 
		  (long)happening);
		hpe1476_CDE_MESSAGE( errMsg);
		
		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	regNum = happening / 100;
	bitNum = happening % 100;

	/* suspend SRQ events - if handler is installed */
	if( thisPtr->e1406 && 0 > thisPtr->numEventHandlers)
	{
		errStatus = viEnableEvent( vi,VI_EVENT_SERVICE_REQ ,
		  VI_SUSPEND_HNDLR, VI_NULL);
		if( errStatus)
		{
                	
                	hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}
	}	

	/* see if we want to enable or disable the handler */
	if( NULL != eventHandler)
	{
		/* see if event handler already exists */		
		if( NULL == thisPtr->eventHandlerArray[hapIdx].eventHandler )
		{
			/* okay - no event handler */
			thisPtr->numEventHandlers =
			  thisPtr->numEventHandlers + 1;
		}
		else
		{
			/* reenable SRQ events - if handler is installed */
			if( thisPtr->e1406 && 0 > thisPtr->numEventHandlers)
			{
				errStatus = viEnableEvent( vi,
				  VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);
				if( errStatus)
				{
               			 	
                			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}
			}	

			/* error - event handler already exists */
			sprintf(errMsg, 
			  hpe1476_MSG_EVENT_HDLR_INSTALLED
			  " %ld.  "
			  hpe1476_MSG_EVENT_HDLR_INST2, (long)happening);
			
			hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
			
		}

		/* store handler and user data */
		thisPtr->eventHandlerArray[hapIdx].eventHandler = eventHandler;
		thisPtr->eventHandlerArray[hapIdx].userData = userData;

		/* do we need to install the event handler? */
		if( 1 == thisPtr->numEventHandlers)
		{
			/* install the event handler */
			if (thisPtr->e1406)
			{
				errStatus = viInstallHandler(vi, 
				  VI_EVENT_SERVICE_REQ, hpe1476_srqHdlr, VI_NULL);
				if( errStatus)
				{
               				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}

				/* enable events */
				errStatus = viEnableEvent( vi,
				  VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);
				if( errStatus)
				{
        	        		hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}
			}
			else
			{
				instr_onsrq(vi,hpe1476_rb_srqHdlr);
			}
		}

	} /* if - install event handler */
	else
	{
		/* see if event handler already exists */		
		if( NULL != thisPtr->eventHandlerArray[hapIdx].eventHandler )
		{
			/* okay - no event handler */
			thisPtr->numEventHandlers =
			  thisPtr->numEventHandlers - 1;
		}

		/* clear handler and user data */
		thisPtr->eventHandlerArray[hapIdx].eventHandler = NULL;
		thisPtr->eventHandlerArray[hapIdx].userData = NULL;

		/* do we need to uninstall the event handler? */
		if( 0 == thisPtr->numEventHandlers)
		{
			/* disable SRQ events */
			if (thisPtr->e1406)
			{
				errStatus = viDisableEvent( vi,
				  VI_EVENT_SERVICE_REQ ,VI_HNDLR);
				if( VI_SUCCESS > errStatus)
				{

	     		          	hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}

				/* flush any pending SRQ events */
				errStatus = viDiscardEvents( vi,
				  VI_EVENT_SERVICE_REQ ,VI_SUSPEND_HNDLR);
				if( VI_SUCCESS > errStatus)
				{
        	       			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}

				/* uninstall the event handler */
				errStatus = viUninstallHandler(vi, 
				  VI_EVENT_SERVICE_REQ, hpe1476_srqHdlr, VI_NULL);
				if( VI_SUCCESS > errStatus &&
				    VI_ERROR_INV_HNDLR_REF != errStatus)
				{
        	       			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
				}
			}
			else
			{
				instr_onsrq(vi,0);
			}

		}

		/* remove all stale events */
		errStatus = hpe1476_readAllEvents( vi);
		if( VI_SUCCESS > errStatus)
		{
               		hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

	} /* else - remove event handler */

	/* enable event bits to propagate through the IEEE 488.2 
	 * status system and eventually assert SRQ.
	 */
	propagate = happening;
	while( propagate > 0)
	{
		regNum = propagate / 100;
		bitNum = propagate % 100;

		/* get access index */
		if( VI_FALSE == hpe1476_findAccessIdx( regNum * 100, &accessIdx))
		{
			hpe1476_CDE_MESSAGE( "hpe1476_findAccessIdx failed");
			hpe1476_LOG_STATUS( vi, thisPtr, hpe1476_INSTR_ERROR_UNEXPECTED);
		}

		/* query enable value from instrument */
		if (thisPtr->e1406)
		{
			thisPtr->blockSrqIO = VI_TRUE;

			errStatus = viPrintf( vi, "%s?\n",
				 hpe1476_accessInfo[accessIdx].enableCmd);
			if( VI_SUCCESS > errStatus)
			{
				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
			}

			/* read back the response */
			errStatus = viScanf( vi, "%ld%*t", &enableMask);
			if( VI_SUCCESS > errStatus)
			{
				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
			}
		}
		else
		{
			ViChar cmd_str[100];
			ViChar resp_str[100];

			sprintf(cmd_str, "%s?\n", 
			        hpe1476_accessInfo[accessIdx].enableCmd);

			cscpi_exe( vi, cmd_str, strlen(cmd_str),
				  resp_str, sizeof(resp_str) );
			/* get the response */
			enableMask = atol(resp_str);
		}

		/* enable/disable  the corresponding event bit */
		if( NULL != eventHandler)
		{
			/* enable event bit */
			enableMask = enableMask | (1 << (bitNum - 1));
		}	
		else
		{
			/* disable event bit */
			enableMask = enableMask & (~(1 << (bitNum - 1)));
		}

		/* write back the new enable mask */
		if (thisPtr->e1406)
		{
			errStatus = viPrintf( vi, "%s %hd\n",
			   hpe1476_accessInfo[accessIdx].enableCmd,
			   enableMask);
			if( VI_SUCCESS > errStatus)
			{
			
				hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
			}
		}
		else
		{
			ViChar cmd_str[100];

			sprintf(cmd_str, "%s %hd\n", 
			        hpe1476_accessInfo[accessIdx].enableCmd,
			        enableMask);
			cscpi_exe( vi, cmd_str, strlen(cmd_str), 0, 0);
		}

		if( NULL == eventHandler)
		{
			/* for disabling events, only disable the
			 * lowest level.
			 */
			propagate = 0;
		}
		else
		{
			propagate = propagate / 100;
		}

	} /* while - propagate */

	
	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	


/****************************************************************************
hpe1476_statEvenHdlr_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32 happeningIN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens.  These can refer to
  |  | conditions or events.  Happenings are enumerated as ViInt32
  |  | numbers.
  |  |
   ------------------------------------------------------------------------
  | pEventHandler                     hpe1476_InstrPEventHandler      OUT
  |   ---------------------------------------------------------------------
  |  | This is the definition of a hpe1476_InstrPEventHandler:
  |  |
  |  | typedef void (_VI_PTR _VI_PTR hpe1476_InstrPEventHandler)(
  |  | ViSession vi,
  |  | ViInt32 happening,
  |  | ViAddr userData
  |  | );
   ------------------------------------------------------------------------
  | pUserData                                         ViPAddr     OUT
  |   ---------------------------------------------------------------------
  |  | This is a pointer to the userData that was registered
  |  | with the handler.

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_statEvenHdlr_Q(
  ViSession vi,
  ViInt32 happening,
  hpe1476_InstrPEventHandler pEventHandler,
  ViPAddr pUserData)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	char errMsg[80];

        /* initialize output parameters */

	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1476_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1476_CDE_INIT( "hpe1476_statEvenHdlr_Q");

	/* check validity of happening */
	if( hpe1476_USER_ERROR_HANDLER == happening)
	{
		hapIdx = hpe1476_USER_ERROR_HANDLER_IDX;
	}
	else if( VI_FALSE == hpe1476_findHappeningIdx( happening, &hapIdx) )
	{
		sprintf(errMsg,"%hd " hpe1476_MSG_INVALID_HAPPENING, happening);
		hpe1476_CDE_MESSAGE( errMsg);
		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	*((void **)pEventHandler) = (void *)thisPtr->eventHandlerArray[hapIdx].eventHandler;
	*pUserData = thisPtr->eventHandlerArray[hapIdx].userData;

	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/****************************************************************************
hpe1476_statEvenHdlrDelAll
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_statEvenHdlrDelAll(
  ViSession vi)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
        ViStatus errStatus;

        /* initialize output parameters */

	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
				vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1476_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1476_CDE_INIT( "hpe1476_statEvenHdlrDelAll");

	/* disable SRQ events */
	if (thisPtr->e1406)
	{
		errStatus = viDisableEvent( vi,VI_EVENT_SERVICE_REQ ,VI_HNDLR);
		if( VI_SUCCESS > errStatus)
		{
        	       	hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* flush any pending SRQ events */
		errStatus = viDiscardEvents( vi,VI_EVENT_SERVICE_REQ ,VI_SUSPEND_HNDLR);
		if( VI_SUCCESS > errStatus)
		{
        	       	hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* uninstall the event handler */
		errStatus = viUninstallHandler(vi, 
		  VI_EVENT_SERVICE_REQ, hpe1476_srqHdlr, VI_NULL);
		if( VI_SUCCESS > errStatus &&
		    VI_ERROR_INV_HNDLR_REF != errStatus)
		{
        		hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}
	}
	else
	{
		instr_onsrq(vi,0);
	}

	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/****************************************************************************
hpe1476_statEvenClr
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1476_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1476_statEvenClr(
  ViSession vi
)
{
	ViUInt16 intf;
        struct hpe1476_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 idx;

	viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
	if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
	{
		errStatus = viGetAttribute(
			vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	}
	else
	{
		errStatus = instr_getdata( vi,  (ViAddr) &thisPtr);
	}
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1476_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1476_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1476_CDE_INIT( "hpe1476_statEvenClr");

	/* clear instrument events */
	if (thisPtr->e1406)
	{
		errStatus = viPrintf(vi, "*CLS\n");
		if( VI_SUCCESS > errStatus)
		{
			hpe1476_LOG_STATUS( vi, thisPtr, errStatus);
		}
	}
	else
	{
		{{ 
/*#line 3080 "hpe1476.cs"*/
{extern sw_cls();instr_send(vi,sw_cls,(void*)0);}}}
	}

	/* clear driver copy of instrument events */
	for(idx=0; idx<hpe1476_MAX_STAT_REG; idx++)
	{
		thisPtr->driverEventArray[idx] = 0;
	}

	hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_abor
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  The hpe1476_abor function stops a scan in progress when
 *            trigger source is hpe1476_TRIG_SOUR_BUS or
 *           hpe1476_TRIG_SOUR_HOLD.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_abor(ViSession vi)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_abor" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"ABOR\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ 
/*#line 3142 "hpe1476.cs"*/
{extern sw_abort();instr_send(vi,sw_abort,(void*)0);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_armCoun
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_armCoun  allows
 *            scanning cycles to occur a multiple of times (1
 *            to 32767) with one hpe1476_initImm function and when
 *           hpe1476_initCont is set to 0 (off).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 armCoun
 * IN        
 *            The number of arm counts.
 * 
 *      MAX = hpe1476_ARM_COUN_MAX   32767
 *      MIN = hpe1476_ARM_COUN_MIN   1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_armCoun(ViSession vi,
  ViInt16 armCoun)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_armCoun" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(armCoun
                         ,hpe1476_ARM_COUN_MIN
                         ,hpe1476_ARM_COUN_MAX
                         ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"ARM:COUN %d\n",armCoun);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3211 "hpe1476.cs"*/
in_.p1_type= 0;((HPSL_INT16*)in_.p1)->num=(armCoun);
/*#line 3211 "hpe1476.cs"*/
((HPSL_INT16*)in_.p1)->suffix=0;
/*#line 3211 "hpe1476.cs"*/
{extern sw_trigcoun();instr_send(vi,sw_trigcoun,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_armCoun_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_armCoun_Q returns current number of scanning
 *            cycles set by hpe1476_armCoun.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPInt16 armCoun
 * OUT       
 *            Returns current number of arm counts.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_armCoun_Q(ViSession vi,
  ViPInt16 armCoun)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_armCoun_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"ARM:COUN?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%hd%*t",armCoun);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;struct {double p1[1];} out_;
/*#line 3276 "hpe1476.cs"*/
in_.p1_type= -1;
/*#line 3276 "hpe1476.cs"*/
{extern sw_trigcoun_q();if (!instr_query(vi,sw_trigcoun_q,&in_,&out_)){
/*#line 3276 "hpe1476.cs"*/
*armCoun=(*(HPSL_INT16*)out_.p1).num;}}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_closCardChan
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_closCardChan closes the multiplexer channel specified
 *           (0-63,90-94) on the card specified (1-99).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 Card_Number
 * IN        
 *            Numeric value of the card in a switchbox.  In a single-card
 *           switchbox, the card number is 1.
 * 
 *      MAX = hpe1476_CARD_MAX   99
 *      MIN = hpe1476_CARD_MIN   1
 * 
 * PARAM 3 : ViInt16 Channel
 * IN        
 *            Numeric value of the channel to be closed (0-63, 90-94).
 * 
 *      MAX = hpe1476_CHANNEL_NUMBER_MAX   94
 *      MIN = hpe1476_CHANNEL_NUMBER_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_closCardChan(ViSession vi,
  ViInt16 Card_Number,
  ViInt16 Channel)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_closCardChan" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(Card_Number
                         ,hpe1476_CARD_MIN
                         ,hpe1476_CARD_MAX
                         ,VI_ERROR_PARAMETER2);

    hpe1476_CHK_INT_RANGE(Channel
                         ,hpe1476_CHANNEL_NUMBER_MIN
                         ,hpe1476_CHANNEL_NUMBER_MAX
                         ,VI_ERROR_PARAMETER3);

    
    Card_Number *= 100;
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"CLOS (@%d)\n",Card_Number+Channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        long old_error;
    
        old_error = instr_error_head(vi);
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3362 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 3362 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 3362 "hpe1476.cs"*/
bl_1[0].number=Card_Number+Channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 3362 "hpe1476.cs"*/
{extern sw_clos();instr_send(vi,sw_clos,&in_);}}}
        if (old_error != instr_error_head(vi))
        { /* must have been bad card number*/
            hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_initCont
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_initCont enables or
 *            disables continuous scanning cycles for the switchbox.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViBoolean initCont
 * IN        
 *           0 = continuous scanning disabled
 *           1 = continuous scanning enabled
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_initCont(ViSession vi,
  ViBoolean initCont)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_initCont" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_BOOLEAN(initCont,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"INIT:CONT %u\n",initCont);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3428 "hpe1476.cs"*/
in_.p1_type= 0;*(HPSL_BOOLEAN*)in_.p1= ((initCont)!=0);
/*#line 3428 "hpe1476.cs"*/
{extern sw_initcont();instr_send(vi,sw_initcont,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_initCont_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_initCont_Q queries the scanning state.
 *            With continuous scanning enabled,
 *            the function returns 1.  With continuous scanning disabled,
 *            the function returns 0.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPBoolean initCont
 * OUT       
 *           Returns 0 when continuous scanning is disabled.
 *           Returns 1 when continuous scanning is enabled.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_initCont_Q(ViSession vi,
  ViPBoolean initCont)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_initCont_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"INIT:CONT?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%hd%*t",initCont);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {double p1[1];} out_;
/*#line 3496 "hpe1476.cs"*/
{extern sw_initcont_q();if (!instr_query(vi,sw_initcont_q,(void*)0,&out_)){
/*#line 3496 "hpe1476.cs"*/
*initCont=*(HPSL_BOOLEAN*)out_.p1;}}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_initImm
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_initImm  starts the scanning cycle
 *            and closes the first channel in the channel list.  Successive
 *            triggers from the source specified by the hpe1476_trigSour
 *            function advances the scan through the channel list.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_initImm(ViSession vi)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_initImm" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"INIT:IMM\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ 
/*#line 3552 "hpe1476.cs"*/
{extern sw_init();instr_send(vi,sw_init,(void*)0);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_openCardChan
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_openCardChan opens the multiplexer channel specified
 *           (0-63,90-94) on the card specified (1-99).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 Card_Number
 * IN        
 *            Numeric value of the card in a switchbox.  In a single-card
 *           switchbox, the card number is 1.
 * 
 *      MAX = hpe1476_CARD_MAX   99
 *      MIN = hpe1476_CARD_MIN   1
 * 
 * PARAM 3 : ViInt16 Channel
 * IN        
 *            Numeric value of the channel to be opened (0-63, 90-94).
 * 
 *      MAX = hpe1476_CHANNEL_NUMBER_MAX   94
 *      MIN = hpe1476_CHANNEL_NUMBER_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_openCardChan(ViSession vi,
  ViInt16 Card_Number,
  ViInt16 Channel)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_openCardChan" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(Card_Number
                         ,hpe1476_CARD_MIN
                         ,hpe1476_CARD_MAX
                         ,VI_ERROR_PARAMETER2);

    hpe1476_CHK_INT_RANGE(Channel
                         ,hpe1476_CHANNEL_NUMBER_MIN
                         ,hpe1476_CHANNEL_NUMBER_MAX
                         ,VI_ERROR_PARAMETER3);

    
    Card_Number *= 100;
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OPEN (@%d)\n",Card_Number+Channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        long old_error;
    
        old_error = instr_error_head(vi);
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3638 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 3638 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 3638 "hpe1476.cs"*/
bl_1[0].number=Card_Number+Channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 3638 "hpe1476.cs"*/
{extern sw_open();instr_send(vi,sw_open,&in_);}}}
        if (old_error != instr_error_head(vi))
        { /* must have been bad card number*/
            hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpEcltStat_M
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpEcltStat_M Selects and enables which ECL Trigger
 *            bus line (0 or 1) will output a trigger when a channel is
 *           closed during a scan.  This is also used to disable a selected
 *           ECL Trigger bus line.  "Line" specifies the ECL Trigger
 *            bus line (0 or 1) and "mode" enables
 *            (1) or disables (0) the specified ECLTrg bus line.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 ecltNum
 * IN        
 *            Numeric suffix for header ECLT.
 * 
 *      MAX = hpe1476_ECLTNUM_MAX   1
 *      MIN = hpe1476_ECLTNUM_MIN   0
 * 
 * PARAM 3 : ViBoolean outpEcltStat
 * IN        
 *            Enables or disables the ECL trigger bus line.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpEcltStat_M(ViSession vi,
  ViInt16 ecltNum,
  ViBoolean outpEcltStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpEcltStat_M" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(ecltNum
                         ,hpe1476_ECLTNUM_MIN
                         ,hpe1476_ECLTNUM_MAX
                         ,VI_ERROR_PARAMETER2);

    hpe1476_CHK_BOOLEAN(outpEcltStat,VI_ERROR_PARAMETER3);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OUTP:ECLT%d:STAT %u\n",ecltNum,outpEcltStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short sub1;char sub_pad[6];short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3720 "hpe1476.cs"*/
in_.sub1= (ecltNum) - -8;
/*#line 3720 "hpe1476.cs"*/
in_.p1_type= 0;*(HPSL_BOOLEAN*)in_.p1= ((outpEcltStat)!=0);
/*#line 3720 "hpe1476.cs"*/
{extern sw_trigttl();instr_send(vi,sw_trigttl,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpEcltStat_M_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpEcltStat_M_Q  Queries the present state of the
 *           specified ECL Trigger bus line.  The function
 *            returns "1" if the specified bus line is enabled or "0" if the
 *           specified bus line is disabled.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 ecltNum
 * IN        
 *            Numeric suffix for header ECLT.
 * 
 *      MAX = hpe1476_ECLTNUM_MAX   1
 *      MIN = hpe1476_ECLTNUM_MIN   0
 * 
 * PARAM 3 : ViPBoolean outpEcltStat
 * OUT       
 *            Returns the state of the selected ECL trigger bus line.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpEcltStat_M_Q(ViSession vi,
  ViInt16 ecltNum,
  ViPBoolean outpEcltStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpEcltStat_M_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(ecltNum
                         ,hpe1476_ECLTNUM_MIN
                         ,hpe1476_ECLTNUM_MAX
                         ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OUTP:ECLT%d:STAT?\n",ecltNum);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%hd%*t",outpEcltStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short sub1;char sub_pad[6];} in_;struct {double p1[1];} out_;
/*#line 3800 "hpe1476.cs"*/
in_.sub1= (ecltNum) - -8;
/*#line 3800 "hpe1476.cs"*/
{extern sw_trigttl_q();if (!instr_query(vi,sw_trigttl_q,&in_,&out_)){
/*#line 3800 "hpe1476.cs"*/
*outpEcltStat=*(HPSL_BOOLEAN*)out_.p1;}}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpStat
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpStat enables/disables the "Trig Out" port to be
 *           pulsed when a channel is closed during a scan.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViBoolean outpStat
 * IN        
 *            Enables or disables the Trig Out port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpStat(ViSession vi,
  ViBoolean outpStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpStat" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_BOOLEAN(outpStat,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OUTP %u\n",outpStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 3861 "hpe1476.cs"*/
in_.p1_type= 0;*(HPSL_BOOLEAN*)in_.p1= ((outpStat)!=0);
/*#line 3861 "hpe1476.cs"*/
{extern sw_trigout();instr_send(vi,sw_trigout,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpStat_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpStat_Q  Queries the present state of the Trig Out
 *           port. The function returns "1" if the port is enabled or "0" if
 *           the port is disabled.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPBoolean outpStat
 * OUT       
 *            Returns the enabling state of the Trig Out port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpStat_Q(ViSession vi,
  ViPBoolean outpStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpStat_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OUTP:STAT?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%hd%*t",outpStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {double p1[1];} out_;
/*#line 3927 "hpe1476.cs"*/
{extern sw_trigout_q();if (!instr_query(vi,sw_trigout_q,(void*)0,&out_)){
/*#line 3927 "hpe1476.cs"*/
*outpStat=*(HPSL_BOOLEAN*)out_.p1;}}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpTtltStat_M
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpTtltStat_M Selects
 *            and enables which TTL Trigger
 *            bus line (0 to 7) will output a trigger
 *            when a channel is closed during a scan.  This is also used to
 *           disable a selected TTL Trigger bus line.  "Line" specifies the
 *           TTL Trigger bus line (0 to 7) and "mode" enables
 *            (1) or disables (0) the specified TTL Trigger bus line.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 ttltNum
 * IN        
 *            Numeric suffix for header TTLT.
 * 
 *      MAX = hpe1476_TTLTNUM_MAX   7
 *      MIN = hpe1476_TTLTNUM_MIN   0
 * 
 * PARAM 3 : ViBoolean outpTtltStat
 * IN        
 *            Enables or disables the TTLT trigger bus line.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpTtltStat_M(ViSession vi,
  ViInt16 ttltNum,
  ViBoolean outpTtltStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpTtltStat_M" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(ttltNum
                         ,hpe1476_TTLTNUM_MIN
                         ,hpe1476_TTLTNUM_MAX
                         ,VI_ERROR_PARAMETER2);

    hpe1476_CHK_BOOLEAN(outpTtltStat,VI_ERROR_PARAMETER3);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OUTP:TTLT%d:STAT %u\n",ttltNum,outpTtltStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short sub1;char sub_pad[6];short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 4006 "hpe1476.cs"*/
in_.sub1= ttltNum;
/*#line 4006 "hpe1476.cs"*/
in_.p1_type= 0;*(HPSL_BOOLEAN*)in_.p1= ((outpTtltStat)!=0);
/*#line 4006 "hpe1476.cs"*/
{extern sw_trigttl();instr_send(vi,sw_trigttl,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_outpTtltStat_M_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_outpTtltStat_M_Q  Queries the present state of the
 *           specified TTL Trigger bus line.  The function returns "1" if the
 *           specified TTLTrg bus line is enabled or "0" if disabled.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 ttltNum
 * IN        
 *            Numeric suffix for header TTLT.
 * 
 *      MAX = hpe1476_TTLTNUM_MAX   7
 *      MIN = hpe1476_TTLTNUM_MIN   0
 * 
 * PARAM 3 : ViPBoolean outpTtltStat
 * OUT       
 *            Returns the state of the selected TTLT trigger bus line.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_outpTtltStat_M_Q(ViSession vi,
  ViInt16 ttltNum,
  ViPBoolean outpTtltStat)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_outpTtltStat_M_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(ttltNum
                         ,hpe1476_TTLTNUM_MIN
                         ,hpe1476_TTLTNUM_MAX
                         ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OUTP:TTLT%d:STAT?\n",ttltNum);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%hd%*t",outpTtltStat);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short sub1;char sub_pad[6];} in_;struct {double p1[1];} out_;
/*#line 4085 "hpe1476.cs"*/
in_.sub1= ttltNum;
/*#line 4085 "hpe1476.cs"*/
{extern sw_trigttl_q();if (!instr_query(vi,sw_trigttl_q,&in_,&out_)){
/*#line 4085 "hpe1476.cs"*/
*outpTtltStat=*(HPSL_BOOLEAN*)out_.p1;}}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routClos
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routClos closes a multiplexer channel specified.
 *            The channel is in the form ccnn where cc = card
 *            number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Numeric value of the channel to be closed in the form ccnn
 *           where cc = card number (01-99) and nn = channel number (00-63 or
 *           90-94).
 * 
 *      MAX = hpe1476_CHANNEL_MAX   999999
 *      MIN = hpe1476_CHANNEL_MIN   100
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routClos(ViSession vi,
  ViInt32 channel)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routClos" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_LONG_RANGE(channel
                          ,hpe1476_CHANNEL_MIN
                          ,hpe1476_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"CLOS (@%ld)\n",channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	long old_error;
    
    	old_error = instr_error_head(vi);
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 4158 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 4158 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 4158 "hpe1476.cs"*/
bl_1[0].number=channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 4158 "hpe1476.cs"*/
{extern sw_clos();instr_send(vi,sw_clos,&in_);}}}
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routClosList
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routClosList closes multiplexer
 *            channels specified in the channelList.
 *            The channelList is in the form "ccnn", "ccnn,ccnn",
 *            or "ccnn:ccnn" where cc = card number (01-99) and nn = channel
 *           number (00-63 or 90-94).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViString channelList
 * IN        
 *            Channel list of the channels to be closed. The channelList is
 *           in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc = card
 *           number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routClosList(ViSession vi,
  ViString channelList)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routClosList" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"CLOS (@%s)\n",channelList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	char *cmd = malloc(strlen(channelList)+20);
    	long old_error;
    
    	if (!cmd)
    	{
    		hpe1476_CDE_ERROR( VI_ERROR_ALLOC );
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
    	}
    	strcpy(cmd, "CLOS (@");
    	strcat(cmd, channelList);
    	strcat(cmd, ")");
    	old_error = instr_error_head(vi);
    	cscpi_exe(vi, cmd, strlen(cmd), 0, 0);
    	free(cmd);
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routClosList_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routClosList_Q returns
 *            the current state of the channels queried.  The channelList is
 *           in the form "ccnn".  The function returns a comma separated list
 *           of 1's and 0's, 1 for each closed channel and 0 for each open
 *           channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViString channelList
 * IN        
 *            Channel list of the channels to be queried. The channelList is
 *           in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc = card
 *           number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * PARAM 3 : ViChar _VI_FAR routClosList[]
 * OUT       
 *           A comma separated list of 1's and 0's,
 *           1 for each closed channel and 0 for open channel.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routClosList_Q(ViSession vi,
  ViString channelList,
  ViChar _VI_FAR routClosList[])
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routClosList_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"CLOS? (@%s)\n",channelList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",routClosList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        /* remove trailing newline */
        if (routClosList[0])
    	routClosList[strlen(routClosList)-1] = 0;	
        else
        {
    	hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    else
    {
    	char *cmd = malloc(strlen(channelList)+20);
    	long old_error;
    
    	if (!cmd)
    	{
    		hpe1476_CDE_ERROR( VI_ERROR_ALLOC );
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
    	}
    	strcpy(cmd, "CLOS? (@");
    	strcat(cmd, channelList);
    	strcat(cmd, ")");
    	old_error = instr_error_head(vi);
    	cscpi_exe(vi, cmd, strlen(cmd), routClosList, 32700);
    	/* remove trailing newline */
    	if (routClosList[0])
    		routClosList[strlen(routClosList)-1] = 0;	
    	else
    	{
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    	free(cmd);
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routClos_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routClos_Q returns
 *            the current state of the channel queried.  The channel is in
 *           the form ccnn.  The function returns 1 if the channel is closed
 *           or returns 0 if the channel is open.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Numeric value of the channel to be queried in the form ccnn
 *           where cc = card number (01-99) and nn = channel number (00-63 or
 *           90-94).
 * 
 *      MAX = hpe1476_CHANNEL_MAX   999999
 *      MIN = hpe1476_CHANNEL_MIN   100
 * 
 * PARAM 3 : ViPBoolean routClos
 * OUT       
 *           Returns the state of the channel queried.
 *           0 = Open channel
 *           1 = Closed channel
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routClos_Q(ViSession vi,
  ViInt32 channel,
  ViPBoolean routClos)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routClos_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_LONG_RANGE(channel
                          ,hpe1476_CHANNEL_MIN
                          ,hpe1476_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        char str[10];
    
        *str = 0;
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"CLOS? (@%ld)\n",channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",str);
        *routClos = (*str=='1');
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	char str[10];
    	long old_error;
    
    	*str = 0;
    	old_error = instr_error_head(vi);
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;struct {double p1[1];} out_;
/*#line 4443 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 4443 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 4443 "hpe1476.cs"*/
bl_1[0].number=channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 4443 "hpe1476.cs"*/
{extern sw_clos_q();if (!instr_query(vi,sw_clos_q,&in_,&out_)){
/*#line 4443 "hpe1476.cs"*/
(void)strcpy((str),*(HPSL_STRING*)out_.p1);}}}}
    	*routClos = (*str=='1');
    	if (!*str || old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routOpen
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routOpen opens a multiplexer channel specified.
 *            The channel
 *            is in the form ccnn where cc = card
 *            number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Numeric value of the channel to be opened in the form ccnn
 *           where cc = card number (01-99) and nn = channel number (00-63 or
 *           90-94).
 * 
 *      MAX = hpe1476_CHANNEL_MAX   999999
 *      MIN = hpe1476_CHANNEL_MIN   100
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routOpen(ViSession vi,
  ViInt32 channel)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routOpen" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_LONG_RANGE(channel
                          ,hpe1476_CHANNEL_MIN
                          ,hpe1476_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OPEN (@%ld)\n",channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	long old_error;
    
    	old_error = instr_error_head(vi);
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 4522 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 4522 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 4522 "hpe1476.cs"*/
bl_1[0].number=channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 4522 "hpe1476.cs"*/
{extern sw_open();instr_send(vi,sw_open,&in_);}}}
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routOpenList
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routOpenList opens the
 *            multiplexer channels specified
 *            in the channelList.  The channelList
 *            is in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc =
 *           card number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViString channelList
 * IN        
 *            Channel list of the channels to be opened. The channelList is
 *           in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc = card
 *           number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routOpenList(ViSession vi,
  ViString channelList)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routOpenList" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"OPEN (@%s)\n",channelList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	char *cmd = malloc(strlen(channelList)+20);
    	long old_error;
    
    	if (!cmd)
    	{
    		hpe1476_CDE_ERROR( VI_ERROR_ALLOC );
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
    	}
    	strcpy(cmd, "OPEN (@");
    	strcat(cmd, channelList);
    	strcat(cmd, ")");
    	old_error = instr_error_head(vi);
    	cscpi_exe(vi, cmd, strlen(cmd), 0, 0);
    	free(cmd);
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routOpenList_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routOpenList_Q returns the
 *            current state of the channels queried.  The channelList is in
 *           the form "ccnn".   The function returns a comma separated list
 *           of 1's and 0's, 1 for each open channel and 0 for each closed
 *           channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViString channelList
 * IN        
 *            Channel list of the channels to be queried. The channelList is
 *           in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc = card
 *           number (01-99) and nn = channel number (00-63 or 90-94).
 * 
 * PARAM 3 : ViChar _VI_FAR routOpenList[]
 * OUT       
 *           A comma separated list of 1's and 0's,
 *           1 for each open channel and 0 for closed channel.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routOpenList_Q(ViSession vi,
  ViString channelList,
  ViChar _VI_FAR routOpenList[])
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routOpenList_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OPEN? (@%s)\n",channelList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",routOpenList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        /* remove trailing newline */
        if (routOpenList[0])
    	routOpenList[strlen(routOpenList)-1] = 0;	
        else
        {
    	hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    else
    {
    	char *cmd = malloc(strlen(channelList)+20);
    	long old_error;
    
    	if (!cmd)
    	{
    		hpe1476_CDE_ERROR( VI_ERROR_ALLOC );
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
    	}
    	strcpy(cmd, "OPEN? (@");
    	strcat(cmd, channelList);
    	strcat(cmd, ")");
    	old_error = instr_error_head(vi);
    	cscpi_exe(vi, cmd, strlen(cmd), routOpenList, 32700);
    	/* remove trailing newline */
    	if (routOpenList[0])
    		routOpenList[strlen(routOpenList)-1] = 0;	
    	else
    	{
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    	free(cmd);
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routOpen_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routOpen_Q returns the
 *            current state of the channel queried.  The channel
 *            is in the form ccnn.  The function returns 1 if the channel is
 *           open or returns 0 if the channel is closed.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Numeric value of the channel to be queried in the form ccnn
 *           where cc = card number (01-99) and nn = channel number (00-63 or
 *           90-94).
 * 
 *      MAX = hpe1476_CHANNEL_MAX   999999
 *      MIN = hpe1476_CHANNEL_MIN   100
 * 
 * PARAM 3 : ViPBoolean routOpen
 * OUT       
 *           Returns the state of the channel queried.
 *           0 = Closed channel
 *           1 = Open channel
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routOpen_Q(ViSession vi,
  ViInt32 channel,
  ViPBoolean routOpen)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routOpen_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_LONG_RANGE(channel
                          ,hpe1476_CHANNEL_MIN
                          ,hpe1476_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        char str[10];
    
        *str = 0;
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OPEN? (@%ld)\n",channel);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",str);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        *routOpen = (*str=='1');
    }
    else
    {
    	char str[10];
    	long old_error;
    
    	*str = 0;
    	old_error = instr_error_head(vi);
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;struct {double p1[1];} out_;
/*#line 4807 "hpe1476.cs"*/
HPSL_CHANNEL_LIST bl_1[1];
/*#line 4807 "hpe1476.cs"*/
in_.p1_type= 5;
/*#line 4807 "hpe1476.cs"*/
bl_1[0].number=channel;*(HPSL_CHANNEL_PTR*)in_.p1=bl_1;bl_1[0].type= 1;bl_1[0].next= 0;
/*#line 4807 "hpe1476.cs"*/
{extern sw_open_q();if (!instr_query(vi,sw_open_q,&in_,&out_)){
/*#line 4807 "hpe1476.cs"*/
(void)strcpy((str),*(HPSL_STRING*)out_.p1);}}}}
    	*routOpen = (*str=='1');
    	if (!*str || old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routScanList
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routScanList defines the
 *            channels to be scanned.  The
 *            channelList is in the form "ccnn", "ccnn,ccnn",
 *            or "ccnn:ccnn" where cc = card number (01-99) and nn = channel
 *           number (00-63 or 93).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViString channelList
 * IN        
 *            Channel list of the channels to be scanned. The channelList is
 *           in the form "ccnn", "ccnn,ccnn", or "ccnn:ccnn" where cc = card
 *           number (01-99) and nn = channel number (00-63).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_routScanList(ViSession vi,
  ViString channelList)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routScanList" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"SCAN (@%s)\n",channelList);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	char *cmd = malloc(strlen(channelList)+20);
    	long old_error;
    
    	if (!cmd)
    	{
    		hpe1476_CDE_ERROR( VI_ERROR_ALLOC );
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_ALLOC );
    	}
    	strcpy(cmd, "SCAN (@");
    	strcat(cmd, channelList);
    	strcat(cmd, ")");
    	old_error = instr_error_head(vi);
    	cscpi_exe(vi, cmd, strlen(cmd), 0, 0);
    	if (old_error != instr_error_head(vi))
    	{ /* must have been bad channel list*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    	free(cmd);
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routScanMode
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routScanMode sets the multiplexer
 *            channels defined by the hpe1476_routScanList
 *            function for None, Volts, 2-wire Ohms, or 4-wire Ohms
 *           measurements.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 routScanMode
 * IN        
 *            The scan mode type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                       Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_ROUT_SCAN_MODE_NONE          0  None
 *      hpe1476_ROUT_SCAN_MODE_FRES          1  4-Wire Resistance
 *      hpe1476_ROUT_SCAN_MODE_RES           2  Resistance
 *      hpe1476_ROUT_SCAN_MODE_VOLT          3  Voltage
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_routScanMode_routScanMode_a[] = {"NONE",
        "FRES","RES","VOLT",0};
ViStatus _VI_FUNC hpe1476_routScanMode(ViSession vi,
  ViInt16 routScanMode)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routScanMode" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_ENUM(routScanMode,3,VI_ERROR_PARAMETER2);
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"SCAN:MODE %s\n",hpe1476_routScanMode_routScanMode_a[routScanMode]);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 4965 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_FRESNONE04[];char f_=0;
/*#line 4965 "hpe1476.cs"*/
in_.p1_type= 2;*(HPSL_ENUM*)in_.p1=char_to_enum(Eswitch_enum_FRESNONE04,hpe1476_routScanMode_routScanMode_a[routScanMode],vi, &f_);
/*#line 4965 "hpe1476.cs"*/
if (!f_) {extern sw_scan_mode();instr_send(vi,sw_scan_mode,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routScanMode_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routScanMode_Q returns the current state of the
 *            scan mode.  The function returns
 *            hpe1476_ROUT_SCAN_MODE_NONE,
 *            hpe1476_ROUT_SCAN_MODE_FRES,
 *            hpe1476_ROUT_SCAN_MODE_RES,
 *            or hpe1476_ROUT_SCAN_MODE_VOLT.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPInt16 routScanMode
 * OUT       
 *            Returns the current scan mode type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                       Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_ROUT_SCAN_MODE_NONE          0  None
 *      hpe1476_ROUT_SCAN_MODE_FRES          1  4-Wire Resistance
 *      hpe1476_ROUT_SCAN_MODE_RES           2  Resistance
 *      hpe1476_ROUT_SCAN_MODE_VOLT          3  Voltage
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_routScanMode_Q_routScanMode_a[] = {
        "NONE","FRES","RES","VOLT",0};
ViStatus _VI_FUNC hpe1476_routScanMode_Q(ViSession vi,
  ViPInt16 routScanMode)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;
    char routScanMode_str[32];

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routScanMode_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SCAN:MODE?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%s%*t",routScanMode_str);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {double p1[1];} out_;
/*#line 5045 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_FRESNONE04[];
/*#line 5045 "hpe1476.cs"*/
{extern sw_scan_mode_q();if (!instr_query(vi,sw_scan_mode_q,(void*)0,&out_)){
/*#line 5045 "hpe1476.cs"*/
enum_to_char(routScanMode_str, Eswitch_enum_FRESNONE04, *(HPSL_ENUM*)out_.p1);}}}}
    }
    errStatus = hpe1476_findIndex(thisPtr,hpe1476_routScanMode_Q_routScanMode_a,
        routScanMode_str, routScanMode);
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, thisPtr, errStatus ); }

    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routScanPort
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_routScanPort enables/disables
 *            the closing of the VSA, VSB, CS, RTA, and RTB  tree switches
 *           during scanning. hpe1476_routScanPort
 *           hpe1476_ROUT_SCAN_PORT_ABUS allows the switch driver to close
 *           tree switches connecting channels to the analog bus and is
 *           required to make scanning measurements from the analog bus. For
 *           correct measurement switching, set the measurement mode with the
 *           hpe1476_routScanMode function. The hpe1476_routScanPort
 *           hpe1476_ROUT_SCAN_PORT_NONE function prevents closing the tree
 *           switches during scan operation, which is useful if the
 *           measurement instrument is not connected to the ananlog bus.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 routScanPort
 * IN        
 *            The scan port type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                       Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_ROUT_SCAN_PORT_NONE          0  None
 *      hpe1476_ROUT_SCAN_PORT_ABUS          1  Analog Bus
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_routScanPort_routScanPort_a[] = {"NONE",
        "ABUS",0};
ViStatus _VI_FUNC hpe1476_routScanPort(ViSession vi,
  ViInt16 routScanPort)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routScanPort" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_ENUM(routScanPort,1,VI_ERROR_PARAMETER2);
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"SCAN:PORT %s\n",hpe1476_routScanPort_routScanPort_a[routScanPort]);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 5126 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_ABUSNONE02[];char f_=0;
/*#line 5126 "hpe1476.cs"*/
in_.p1_type= 2;*(HPSL_ENUM*)in_.p1=char_to_enum(Eswitch_enum_ABUSNONE02,hpe1476_routScanPort_routScanPort_a[routScanPort],vi, &f_);
/*#line 5126 "hpe1476.cs"*/
if (!f_) {extern sw_scan_port();instr_send(vi,sw_scan_port,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_routScanPort_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  The hpe1476_routScanPort_Q function returns the current
 *            hpe1476_routScanPort setting (hpe1476_ROUT_SCAN_PORT_NONE or
 *            hpe1476_ROUT_SCAN_PORT_ABUS).
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPInt16 routScanPort
 * OUT       
 *            Returns the current scan port type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                       Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_ROUT_SCAN_PORT_NONE          0  None
 *      hpe1476_ROUT_SCAN_PORT_ABUS          1  Analog Bus
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_routScanPort_Q_routScanPort_a[] = {
        "NONE","ABUS",0};
ViStatus _VI_FUNC hpe1476_routScanPort_Q(ViSession vi,
  ViPInt16 routScanPort)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;
    char routScanPort_str[32];

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_routScanPort_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SCAN:PORT?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%s%*t",routScanPort_str);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {double p1[1];} out_;
/*#line 5201 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_ABUSNONE02[];
/*#line 5201 "hpe1476.cs"*/
{extern sw_scan_port_q();if (!instr_query(vi,sw_scan_port_q,(void*)0,&out_)){
/*#line 5201 "hpe1476.cs"*/
enum_to_char(routScanPort_str, Eswitch_enum_ABUSNONE02, *(HPSL_ENUM*)out_.p1);}}}}
    }
    errStatus = hpe1476_findIndex(thisPtr,hpe1476_routScanPort_Q_routScanPort_a,
        routScanPort_str, routScanPort);
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, thisPtr, errStatus ); }

    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_systCdes_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_systCdes_Q returns
 *            the description of a selected module (card) in a switchbox.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 card
 * IN        
 *            Numeric value of the module (card) to be queried in a
 *           switchbox.
 * 
 *      MAX = hpe1476_CARD_MAX   99
 *      MIN = hpe1476_CARD_MIN   1
 * 
 * PARAM 3 : ViChar _VI_FAR systCdes[]
 * OUT       
 *            Returns the description of the selected module (card).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_systCdes_Q(ViSession vi,
  ViInt16 card,
  ViChar _VI_FAR systCdes[])
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_systCdes_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(card
                         ,hpe1476_CARD_MIN
                         ,hpe1476_CARD_MAX
                         ,VI_ERROR_PARAMETER2);

    *systCdes = 0;
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SYST:CDES? %d\n",card); 
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",systCdes);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        /* remove trailing newline */
        if (systCdes[0])
    	systCdes[strlen(systCdes)-1] = 0;	
        else
        {
    	hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    else
    {
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;struct {double p1[1];} out_;
/*#line 5292 "hpe1476.cs"*/
in_.p1_type= 0;((HPSL_INT16*)in_.p1)->num=(card);
/*#line 5292 "hpe1476.cs"*/
((HPSL_INT16*)in_.p1)->suffix=0;
/*#line 5292 "hpe1476.cs"*/
{extern sw_systcdes_q();if (!instr_query(vi,sw_systcdes_q,&in_,&out_)){
/*#line 5292 "hpe1476.cs"*/
(void)strcpy((systCdes),*(HPSL_STRING*)out_.p1);}}}}
    	if (!*systCdes)
    	{ /* must have been bad card*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_systCpon
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_systCpon opens all
 *            channels of a selected or all modules (cards) in a switchbox.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 card
 * IN        
 *            Numeric value of the module (card) to be reset in a switchbox.
 * 
 *      MAX = hpe1476_CARD_MAX   99
 *      MIN = hpe1476_CARD_MIN   1
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_systCpon(ViSession vi,
  ViInt16 card)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_systCpon" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(card
                         ,hpe1476_CARD_MIN
                         ,hpe1476_CARD_MAX
                         ,VI_ERROR_PARAMETER2);

    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"SYST:CPON %d\n",card);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
    	long old_error;
    
    	old_error = instr_error_head(vi);
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 5366 "hpe1476.cs"*/
in_.p1_type= 0;((HPSL_INT16*)in_.p1)->num=(card);
/*#line 5366 "hpe1476.cs"*/
((HPSL_INT16*)in_.p1)->suffix=0;
/*#line 5366 "hpe1476.cs"*/
{extern sw_systcpon();instr_send(vi,sw_systcpon,&in_);}}}
    	if (old_error != instr_error_head(vi))
    	{ /* must have been card*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_systCtyp_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_systCtyp_Q returns the module
 *            (card) type of a selected module in a switchbox.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 card
 * IN        
 *            Numeric value of the module (card) to be queried in a
 *           switchbox.
 * 
 *      MAX = hpe1476_CARD_MAX   99
 *      MIN = hpe1476_CARD_MIN   1
 * 
 * PARAM 3 : ViChar _VI_FAR systCtyp[]
 * OUT       
 *            Returns the module (card) type of the selected module.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1476_systCtyp_Q(ViSession vi,
  ViInt16 card,
  ViChar _VI_FAR systCtyp[])
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_systCtyp_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_INT_RANGE(card
                         ,hpe1476_CARD_MIN
                         ,hpe1476_CARD_MAX
                         ,VI_ERROR_PARAMETER2);

    *systCtyp = 0;
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SYST:CTYP? %d\n",card); 
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%t",systCtyp);
        if (errStatus < VI_SUCCESS)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        /* remove trailing newline */
        if (systCtyp[0])
    	systCtyp[strlen(systCtyp)-1] = 0;	
        else
        {
    	hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
        }
    }
    else
    {
    	{{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;struct {double p1[1];} out_;
/*#line 5457 "hpe1476.cs"*/
in_.p1_type= 0;((HPSL_INT16*)in_.p1)->num=(card);
/*#line 5457 "hpe1476.cs"*/
((HPSL_INT16*)in_.p1)->suffix=0;
/*#line 5457 "hpe1476.cs"*/
{extern sw_systctyp_q();if (!instr_query(vi,sw_systctyp_q,&in_,&out_)){
/*#line 5457 "hpe1476.cs"*/
(void)memcpy((systCtyp),(*(HPSL_ARB_ASCII*)out_.p1).buffer,(*(HPSL_ARB_ASCII*)out_.p1).length);
/*#line 5457 "hpe1476.cs"*/
(systCtyp)[(*(HPSL_ARB_ASCII*)out_.p1).length]='\0';}}}}
    	if (!*systCtyp)
    	{ /* must have been bad card*/
    		hpe1476_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
    	}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_trigSour
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_trigSour specifies
 *            the trigger source to advance the channel list during scanning.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViInt16 trigSour
 * IN        
 *            The trigger source type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                   Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_TRIG_SOUR_BUS            0  Software Trigger
 *      hpe1476_TRIG_SOUR_ECLT0          1  ECLTRG0
 *      hpe1476_TRIG_SOUR_ECLT1          2  ECLTRG1
 *      hpe1476_TRIG_SOUR_HOLD           3  Hold
 *      hpe1476_TRIG_SOUR_IMM            4  Immediate
 *      hpe1476_TRIG_SOUR_TTLT0          5  TTLTRG0
 *      hpe1476_TRIG_SOUR_TTLT1          6  TTLTRG1
 *      hpe1476_TRIG_SOUR_TTLT2          7  TTLTRG2
 *      hpe1476_TRIG_SOUR_TTLT3          8  TTLTRG3
 *      hpe1476_TRIG_SOUR_TTLT4          9  TTLTRG4
 *      hpe1476_TRIG_SOUR_TTLT5         10  TTLTRG5
 *      hpe1476_TRIG_SOUR_TTLT6         11  TTLTRG6
 *      hpe1476_TRIG_SOUR_TTLT7         12  TTLTRG7
 *      hpe1476_TRIG_SOUR_EXT           13  Trig In BNC
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_trigSour_trigSour_a[] = {"BUS","ECLT0",
        "ECLT1","HOLD","IMM","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4",
        "TTLT5","TTLT6","TTLT7","EXT",0};
ViStatus _VI_FUNC hpe1476_trigSour(ViSession vi,
  ViInt16 trigSour)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_trigSour" );

    /* Perform Error Checking on Each Parameter */
    hpe1476_CHK_ENUM(trigSour,13,VI_ERROR_PARAMETER2);
    if (thisPtr->e1406)
    {
        errStatus = viPrintf(vi,"TRIG:SOUR %s\n",hpe1476_trigSour_trigSour_a[trigSour]);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {short p1_type; char p1_type_pad[6]; double p1[1];} in_;
/*#line 5542 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_BUSDBUS25[];char f_=0;
/*#line 5542 "hpe1476.cs"*/
in_.p1_type= 2;*(HPSL_ENUM*)in_.p1=char_to_enum(Eswitch_enum_BUSDBUS25,hpe1476_trigSour_trigSour_a[trigSour],vi, &f_);
/*#line 5542 "hpe1476.cs"*/
if (!f_) {extern sw_trigsour();instr_send(vi,sw_trigsour,&in_);}}}
    }
    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1476_trigSour_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  hpe1476_trigSour_Q returns the current trigger source for
 *            the switchbox.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument.
 * 
 * PARAM 2 : ViPInt16 trigSour
 * OUT       
 *            Returns the current trigger source type.
 *            The types are defined as macros and are shown below:
 * 
 *      Macro Name                   Value  Description
 *      -----------------------------------------------------------
 *      hpe1476_TRIG_SOUR_BUS            0  Software Trigger
 *      hpe1476_TRIG_SOUR_ECLT0          1  ECLTRG0
 *      hpe1476_TRIG_SOUR_ECLT1          2  ECLTRG1
 *      hpe1476_TRIG_SOUR_HOLD           3  Hold
 *      hpe1476_TRIG_SOUR_IMM            4  Immediate
 *      hpe1476_TRIG_SOUR_TTLT0          5  TTLTRG0
 *      hpe1476_TRIG_SOUR_TTLT1          6  TTLTRG1
 *      hpe1476_TRIG_SOUR_TTLT2          7  TTLTRG2
 *      hpe1476_TRIG_SOUR_TTLT3          8  TTLTRG3
 *      hpe1476_TRIG_SOUR_TTLT4          9  TTLTRG4
 *      hpe1476_TRIG_SOUR_TTLT5         10  TTLTRG5
 *      hpe1476_TRIG_SOUR_TTLT6         11  TTLTRG6
 *      hpe1476_TRIG_SOUR_TTLT7         12  TTLTRG7
 *      hpe1476_TRIG_SOUR_EXT           13  Trig In BNC
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1476_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1476_trigSour_Q_trigSour_a[] = {"BUS",
        "ECLT0","ECLT1","HOLD","IMM","TTLT0","TTLT1","TTLT2","TTLT3",
        "TTLT4","TTLT5","TTLT6","TTLT7","EXT",0};
ViStatus _VI_FUNC hpe1476_trigSour_Q(ViSession vi,
  ViPInt16 trigSour)
{
    ViUInt16 intf;
    ViStatus errStatus = 0;
    struct hpe1476_globals *thisPtr;
    char trigSour_str[32];

    viGetAttribute( vi, VI_ATTR_INTF_TYPE, &intf);
    if (intf!=VI_INTF_VXI && intf!=VI_INTF_GPIB_VXI)
    {
        errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    }
    else
    {
        errStatus = instr_getdata(vi, (ViAddr) &thisPtr);
    }
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, 0, errStatus ); }

    hpe1476_DEBUG_CHK_THIS( vi, thisPtr );

    hpe1476_CDE_INIT( "hpe1476_trigSour_Q" );

    /* Perform Error Checking on Each Parameter */
    if (thisPtr->e1406)
    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"TRIG:SOUR?\n");
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
        errStatus = viScanf(vi,"%s%*t",trigSour_str);
        if (VI_SUCCESS > errStatus)
        {
            hpe1476_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    else
    {
        {{ struct {double p1[1];} out_;
/*#line 5629 "hpe1476.cs"*/
extern CSCPI_IMPORT void *Eswitch_enum_BUSDBUS25[];
/*#line 5629 "hpe1476.cs"*/
{extern sw_trigsour_q();if (!instr_query(vi,sw_trigsour_q,(void*)0,&out_)){
/*#line 5629 "hpe1476.cs"*/
enum_to_char(trigSour_str, Eswitch_enum_BUSDBUS25, *(HPSL_ENUM*)out_.p1);}}}}
    }
    errStatus = hpe1476_findIndex(thisPtr,hpe1476_trigSour_Q_trigSour_a,
        trigSour_str, trigSour);
    if (VI_SUCCESS > errStatus){ hpe1476_LOG_STATUS( vi, thisPtr, errStatus ); }

    hpe1476_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
