/* Copyright (C) 1997-2001 Agilent Technologies */
/* VXIplug&play Instrument Driver for the Agilent E1441 */

/*================================*/
/*           HPE1441              */
/*================================*/
/* Framework Revision: WIN95 4.0  */
/*                     WINNT 4.0  */
/* Driver Revision:    A.01.02    */
/* Firmware Revision:  A.01.00    */
/*================================*/

/*================================*/
/* Source Files:                  */
/*================================*/
/*   hpe1441.c                    */
/*   hpe1441.h                    */
/*================================*/

/*================================*/
/* WIN95/WINNT Framework Files    */
/*================================*/
/*   hpe1441.def                  */
/*   hpe1441.rc                   */
/*   hpe1441.bas                  */
/*================================*/

#define MSGB_DRIVER

#include <stdlib.h>		/* prototype for malloc() */
#include <string.h>		/* prototype for strcpy() */
#include <stdio.h>		/* prototype for sprintf() */
#include <malloc.h>
#include <math.h>
#include <windows.h>

#ifdef WIN32
#define _fmemcpy memcpy
#endif

#include "visa.h"
#define INSTR_CALLBACKS		/* needed to get handler types in hpe1441.h */
#include "hpe1441.h"	

#define hpe1441_MODEL_CODE 0x0244
#define hpe1441_MANF_ID     4095
#define hpe1441_IDN_STRING  "HEWLETT-PACKARD,E1441"
                                                   
#define hpe1441_REV_CODE "A.01.02"  /* Driver Revision */

#define hpe1441_ERR_MSG_LENGTH 256  /* size of error message buffer */

#define hpe1441_MAX_STAT_HAP 7 		    /* number of happenings ??? */
#define hpe1441_MAX_STAT_REG 3		    /* number of IEEE 488.2 status registers */

/* this has to match the index of the ESR register in hpe1441_accessInfo[] */
#define hpe1441_ESR_REG_IDX 2

/* this needs to match the index location of hpe1441_USER_ERROR_HANLDER
 * in the hpe1441_statusHap[] array.  Normally, this is zero.
 */
#define hpe1441_USER_ERROR_HANDLER_IDX 0

/*=============================================================== 
 *
 *  All messages are stored in this area to aid in localization 
 *
 *=============================================================== 
 */


#define hpe1441_MSG_VI_OPEN_ERR 				\
	"vi was zero.  Was the hpe1441_init() successful?"

#define hpe1441_MSG_CONDITION					\
	"condition"
	/* hpe1441_statCond_Q() */

#define hpe1441_MSG_EVENT						\
	"event"	
	/* hpe1441_statEvent_Q() */

#define hpe1441_MSG_EVENT_HDLR_INSTALLED				\
	"event handler is already installed for event happening"
	/* hpe1441_statEvent_Q() */

#define hpe1441_MSG_EVENT_HDLR_INST2				\
	"Only 1 handler can be installed at a time."	
	/* hpe1441_statEvent_Q() */


#define hpe1441_MSG_INVALID_HAPPENING				\
	"is not a valid happening."
	/* hpe1441_statCond_Q() */
	/* hpe1441_statEven_Q() */
	/* hpe1441_statEvenHdlr() */
	/* hpe1441_statEvenHdlr_Q() */

#define hpe1441_MSG_NOT_QUERIABLE					\
	"is not queriable."	
	/* hpe1441_statCond_Q() */
	/* hpe1441_statEven_Q() */


#define hpe1441_MSG_IN_FUNCTION					\
	"in function" 		
	/* hpe1441_error_message() */

#define hpe1441_MSG_INVALID_STATUS					\
  	"Parameter 2 is invalid"				\
	"in function hpe1441_error_message()."
	/* hpe1441_error_message() */

#define hpe1441_MSG_INVALID_STATUS_VALUE				\
	"is not a valid viStatus value."
	/* hpe1441_error_message() */

#define  hpe1441_MSG_INVALID_VI					\
  	"Parameter 1 is invalid"				\
	" in function hpe1441_error_message()"			\
	".  Using an inactive ViSession may cause this error."	\
	"  Was the instrument driver closed prematurely?"
	/* hpe1441_message_query() */

#define hpe1441_MSG_NO_ERRORS					\
	"No Errors"
	/* hpe1441_error_message() */

#define hpe1441_MSG_SELF_TEST_FAILED 				\
	"Self test failed." 	
	/* hpe1441_self_test() */

#define hpe1441_MSG_SELF_TEST_PASSED 				\
	"Self test passed."
	/* hpe1441_self_test() */
	
/* the following messages are used by the functions to check parameters */

#define hpe1441_MSG_BOOLEAN   "Expected 0 or 1; Got %hd"

#define hpe1441_MSG_REAL   "Expected %lg to %lg; Got %lg"
   
#define hpe1441_MSG_INT   "Expected %hd to %hd; Got %hd"

#define hpe1441_MSG_LONG   "Expected %ld to %ld; Got %ld"

#define hpe1441_MSG_LOOKUP "Error converting string response to integer"

#define hpe1441_MSG_NO_MATCH "Could not match string %s"

/* 
 * static error message 
 */

#define VI_ERROR_PARAMETER1_MSG				\
	"Parameter 1 is invalid"

#define VI_ERROR_PARAMETER2_MSG				\
	"Parameter 2 is invalid"

#define VI_ERROR_PARAMETER3_MSG				\
	"Parameter 3 is invalid"

#define VI_ERROR_PARAMETER4_MSG				\
	"Parameter 4 is invalid"

#define VI_ERROR_PARAMETER5_MSG				\
	"Parameter 5 is invalid"

#define VI_ERROR_PARAMETER6_MSG				\
	"Parameter 6 is invalid"

#define VI_ERROR_PARAMETER7_MSG				\
	"Parameter 7 is invalid"

#define VI_ERROR_PARAMETER8_MSG				\
	"Parameter 8 is invalid"

#define VI_ERROR_PARAMETER9_MSG				\
	"Parameter 9 is invalid"

#define VI_ERROR_PARAMETER10_MSG			\
	"Parameter 10 is invalid"

#define VI_ERROR_PARAMETER11_MSG			\
	"Parameter 11 is invalid"

#define VI_ERROR_PARAMETER12_MSG			\
	"Parameter 12 is invalid"

#define VI_ERROR_PARAMETER13_MSG			\
	"Parameter 13 is invalid"

#define VI_ERROR_PARAMETER14_MSG			\
	"Parameter 14 is invalid"

#define VI_ERROR_PARAMETER15_MSG			\
	"Parameter 15 is invalid"

#define VI_ERROR_PARAMETER16_MSG			\
	"Parameter 16 is invalid"

#define VI_ERROR_PARAMETER17_MSG			\
	"Parameter 17 is invalid"

#define VI_ERROR_PARAMETER18_MSG			\
	"Parameter 18 is invalid"

#define VI_ERROR_FAIL_ID_QUERY_MSG				\
	"Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG 				\
	"ViSession (parmeter 1) was not created by this driver"

#define INSTR_ERROR_NULL_PTR_MSG				\
	"NULL pointer detected"

#define INSTR_ERROR_RESET_FAILED_MSG				\
	"reset failed"

#define INSTR_ERROR_UNEXPECTED_MSG 				\
	"An unexpected error occurred"

#define INSTR_ERROR_DETECTED_MSG			\
	"Instrument Error Detected, call hpe1441_error_query()."

#define INSTR_ERROR_LOOKUP_MSG   				\
	"String not found in table"


/*================================================================*/

#define hpe1441_DEBUG_CHK_THIS( vi, thisPtr) 			\
	/* check for NULL user data */				\
	if( 0 == thisPtr)					\
	{							\
 		hpe1441_LOG_STATUS(                             	\
		  vi, NULL, hpe1441_INSTR_ERROR_INV_SESSION );	\
	}							\
	{							\
		ViSession defRM;				\
								\
		/* This should never fail */			\
		errStatus = viGetAttribute( vi,                 \
			VI_ATTR_RM_SESSION, &defRM);		\
		if( VI_SUCCESS > errStatus )			\
		{						\
 			hpe1441_LOG_STATUS(			\
			  vi, NULL, hpe1441_INSTR_ERROR_UNEXPECTED );	\
		}						\
		if( defRM != thisPtr->defRMSession)		\
		{						\
 			hpe1441_LOG_STATUS(			\
			  vi, NULL, hpe1441_INSTR_ERROR_INV_SESSION );	\
		}						\
	}

#define hpe1441_DEBUG_CHK_NULL_PTR( vi, ptr) 			\
	if( 0 == ptr) 						\
	{							\
 		hpe1441_LOG_STATUS( 				\
		   vi, NULL, hpe1441_INSTR_ERROR_NULL_PTR );	\
	}

#define hpe1441_CDE_INIT( funcname)  				\
	strcpy(thisPtr->errFuncName, funcname);			\
	thisPtr->errNumber = VI_SUCCESS;			\
	thisPtr->errMessage[0] = 0;

#define hpe1441_CDE_MESSAGE( message ) 	 			\
	strcpy(thisPtr->errMessage, message)

#define hpe1441_CDE_ERROR( status)					\
	thisPtr->errNumber = status		




struct hpe1441_eventHdlrStruct
{
	hpe1441_InstrEventHandler eventHandler;
	ViAddr userData;
};

struct hpe1441_globals
{
	ViSession			defRMSession;

	ViStatus			errNumber;
	char 				errFuncName[40];
	char				errMessage[160];

	ViBoolean			errQueryDetect;
        long				myDelay;

	ViInt32 			driverEventArray[hpe1441_MAX_STAT_REG];
	ViInt32 			numEventHandlers;
	struct hpe1441_eventHdlrStruct	eventHandlerArray[hpe1441_MAX_STAT_HAP];

	/* Setting this to VI_TRUE, 
	 * will prevent I/O from occuring in an SRQ
	 * This is needed, because VTL 3.0 doesn't provide
	 * an atomic write/read operations.  Don't want
	 * to do I/O if the driver is in the middle of
	 * a read from the instrument.
	 */
	ViBoolean			blockSrqIO;
	ViInt32                         countSrqIO;
};

#ifdef WIN32

#include <windows.h>
#include <winbase.h>


long setDelay( double val )
{
//Delay should be absolute (e.g. 100 micro sec = 1e-4)

double slice;
__int64 count;


if( !QueryPerformanceFrequency( (LARGE_INTEGER *)&count ) )
        {
                //hdw doens't have high perfomance count so use getickcount
                slice = 1e-3; //units for gettick count
        }
else
        {
                slice = 1.0/count; //Seconds per tick
        }

return (long)(val/slice)+1;

}

void doDelay( long ticks )
{
__int64 startval,tmp;

 if(!QueryPerformanceCounter( (LARGE_INTEGER *)&startval))
        {
                DWORD sval;
                sval = GetTickCount();
                while( GetTickCount()-sval  < (DWORD)ticks) ;
                return;
        }
 tmp = startval;
 while(tmp-startval < (DWORD)ticks)
 {
        QueryPerformanceCounter( (LARGE_INTEGER *)&tmp);
 }

}

#else
long setDelay( double val)
{
  DWORD ts;
  DWORD te;
  ViInt32 count;
  ts = GetTickCount();
  count=0;
  while(count<100000) count++;
  te = GetTickCount();
  if((te-ts)>=100) // looks like (te-ts)/100 uS per count
  {
    return (long)(val/((te-ts)/100000000.0))+1;
  }
  ts = GetTickCount();
  count=0;
  while(count<1000000) count++;
  te = GetTickCount();
  if((te-ts)>=100) // looks like (te-ts) nS per count
  {
    return (long)(val/((te-ts)/1000000000.0))+1;
  }
  ts = GetTickCount();
  count=0;
  while(count<10000000) count++;
  te = GetTickCount();
  if((te-ts)>=100) // looks like (te-ts)/10 nS per count
  {
    return (long)(val/((te-ts)/10000000000.0))+1;
  }
  ts = GetTickCount();
  count=0;
  while(count<100000000) count++;
  te = GetTickCount();
  if((te-ts)>=100) // looks like (te-ts)/100 nS per count
  {
    return (long)(val/((te-ts)/100000000000.0))+1;
  }
  // just how fast is that clock?  Give up -- use GetTickCount
  return (long)(-((val/1e-3)+1));
}

void doDelay( long ticks )
{
  DWORD t = GetTickCount();
  long count;
  if(ticks < 0)
  {
    ticks = -ticks;
    while( GetTickCount()-t < (DWORD)ticks) ;
  }
  else
  {
    count = 0;
    while(count<ticks) count++;
  }
}
#endif


/* add the following to the globals data structure */

/* Note: for debugging, you can add __FILE__, __LINE__ as parameters
 * to hpe1441_LOG_STATUS, and ViString filename, and ViInt32 lineNumber to
 * hpe1441_statusUpdate() in order to determine exactly where an error
 * occured in a driver.
 */
#define hpe1441_LOG_STATUS( vi, thisPtr, status ) 	\
  return hpe1441_statusUpdate( vi, thisPtr, status)

/* declare this here since it is called by statusUpdate */
static void hpe1441_srqTraverse(ViSession vi, ViInt32 eventReg);

static ViStatus hpe1441_statusUpdate( ViSession vi,
  struct hpe1441_globals *thisPtr, ViStatus status)
{
	ViInt32  eventQ;

	/* ALL functions exit through here.
	 *
	 * this allows errors to be trapped and
	 * a user error handler could be called.
	 */

	/* can do nothing without state structure*/
	if (!thisPtr) return status;

	/* Check if user wants to query the instrument state.
         * This is only done, if the vi is valid and 
         * no other errors has occured.
         */
	if( VI_TRUE == thisPtr->errQueryDetect && 
            VI_SUCCESS <= status &&
            vi != VI_NULL )
	{
	    /* assume IEEE 488.2 Instrument and query standard
             * status event register for a parser error
             */		
	    thisPtr->blockSrqIO = VI_TRUE;

	    status = viPrintf(vi, "*esr?\n");
	    if( VI_SUCCESS > status) return VI_ERROR_SYSTEM_ERROR;
	    {

		status = viScanf(vi, "%ld%*t\n", &eventQ);
  if(thisPtr)doDelay(thisPtr->myDelay);
	        if( VI_SUCCESS > status) return VI_ERROR_SYSTEM_ERROR;
		{

	  	    /* update driver's copy of the esr register 
       		     * This index 'hpe1441_ESR_REG_IDX' should 
		     * match the access event register array 
		     * and correspond to the 
      		     * standard event status register.
       		     */
		    thisPtr->driverEventArray[hpe1441_ESR_REG_IDX] = 
		      thisPtr->driverEventArray[hpe1441_ESR_REG_IDX] | eventQ;

		    /* see if the instrument thinks an error occured */
		    if( (  0x04 /* Query Error */
                      | 0x08 /* Device Dependent Error */
                      | 0x10 /* Execution Error */
                      | 0x20 /* Command Error */
                    ) & eventQ )
		    {
			/* set our error status, to indicate that a 
			 * instrument error occured.
                         */
			status =  hpe1441_INSTR_ERROR_DETECTED;
		    }
		} /* viScanf was successful */
	    } /* viPrinf was successful */
	} /* if we need to query the instrument */	

	/* check if we need to service SRQ events */
	if( 0 != thisPtr->countSrqIO && VI_NULL != vi )
        {
            ViStatus errStatus;

            /* suspend event processing for now */
            errStatus = viEnableEvent( vi, 
              VI_EVENT_SERVICE_REQ, VI_SUSPEND_HNDLR, VI_NULL);
            if( VI_SUCCESS > errStatus )
            {
                /* an existing error message has priority */
                if( VI_SUCCESS <= status)
                {
                    /* no existing error message, so set new error */
                    status = errStatus;
                }
            } /* if - an error occured */
            else
            { 
	        /* reset the srq counter */
	        thisPtr->countSrqIO = 0;

                /* recursively traverse status byte */
                hpe1441_srqTraverse( vi, 0);

                /* re-enable event processing */
                errStatus = viEnableEvent( vi, VI_EVENT_SERVICE_REQ, 
                  VI_HNDLR, VI_NULL);
                if( VI_SUCCESS > errStatus )
                {
                    /* an existing error message has priority */
                    if( VI_SUCCESS <= status)
                    {
                        /* no existing error message, so set new error */
                        status = errStatus;
                    }
                } /* if - an error occured */

            } /* else - no error in viEnableEvent */

	} /* if - srq needs servicing */

        /* unblock IO in SRQs */
	thisPtr->blockSrqIO = VI_FALSE;

	/* store the context dependent error number */
	hpe1441_CDE_ERROR(status);

	/* if an error occurs, see if the user has an error handler enabled */
        if( status != hpe1441_INSTR_ERROR_DETECTED &&
	    VI_SUCCESS > status &&
	    thisPtr->eventHandlerArray[
	        hpe1441_USER_ERROR_HANDLER_IDX].eventHandler)
        {
		/* call the users handler */
		thisPtr->eventHandlerArray[
                    hpe1441_USER_ERROR_HANDLER_IDX].eventHandler(
		      vi, (ViInt32)status, 
	    	      thisPtr->eventHandlerArray[
                          hpe1441_USER_ERROR_HANDLER_IDX].userData);
	}

	/* return the appropriate error number */
	return status;
}





/* 
 * Error Message Structures
 */

struct instrErrStruct
{
	ViStatus errStatus;
	ViString errMessage;
};

const static struct instrErrStruct instrErrMsgTable[] =
{
        { VI_ERROR_PARAMETER1,	VI_ERROR_PARAMETER1_MSG },
        { VI_ERROR_PARAMETER2,	VI_ERROR_PARAMETER2_MSG },
        { VI_ERROR_PARAMETER3,	VI_ERROR_PARAMETER3_MSG },
        { VI_ERROR_PARAMETER4,	VI_ERROR_PARAMETER4_MSG },
        { VI_ERROR_PARAMETER5,	VI_ERROR_PARAMETER5_MSG },
        { VI_ERROR_PARAMETER6,	VI_ERROR_PARAMETER6_MSG },
        { VI_ERROR_PARAMETER7,	VI_ERROR_PARAMETER7_MSG },
        { VI_ERROR_PARAMETER8,	VI_ERROR_PARAMETER8_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER9,	VI_ERROR_PARAMETER9_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER10,	VI_ERROR_PARAMETER10_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER11,	VI_ERROR_PARAMETER11_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER12,	VI_ERROR_PARAMETER12_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER13,	VI_ERROR_PARAMETER13_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER14,	VI_ERROR_PARAMETER14_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER15,	VI_ERROR_PARAMETER15_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER16,	VI_ERROR_PARAMETER16_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER17,	VI_ERROR_PARAMETER17_MSG },
        { hpe1441_INSTR_ERROR_PARAMETER18,	VI_ERROR_PARAMETER18_MSG },
        { VI_ERROR_FAIL_ID_QUERY,	VI_ERROR_FAIL_ID_QUERY_MSG },

	{ hpe1441_INSTR_ERROR_INV_SESSION,	INSTR_ERROR_INV_SESSION_MSG },
        { hpe1441_INSTR_ERROR_NULL_PTR,	INSTR_ERROR_NULL_PTR_MSG },
        { hpe1441_INSTR_ERROR_RESET_FAILED,INSTR_ERROR_RESET_FAILED_MSG },
        { hpe1441_INSTR_ERROR_UNEXPECTED,	INSTR_ERROR_UNEXPECTED_MSG },
	{ hpe1441_INSTR_ERROR_DETECTED,	INSTR_ERROR_DETECTED_MSG },
	{ hpe1441_INSTR_ERROR_LOOKUP,	INSTR_ERROR_LOOKUP_MSG }
};

/* macros for testing parameters */
#define hpe1441_CHK_BOOLEAN( my_val, err ) if( hpe1441_chk_boolean( thisPtr, my_val) ) hpe1441_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1441_chk_boolean(
  struct hpe1441_globals *thisPtr,
  ViBoolean my_val)
{
   char message[hpe1441_ERR_MSG_LENGTH];
   if( (my_val != VI_TRUE) && (my_val != VI_FALSE) )
   {
      /* true = parameter is invalid */
      sprintf(message, hpe1441_MSG_BOOLEAN, my_val);	
      hpe1441_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }

   /* false = okay */
   return VI_FALSE;
}


#define hpe1441_CHK_REAL_RANGE( my_val, min, max, err ) if( hpe1441_chk_real_range( thisPtr, my_val, min, max) ) hpe1441_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1441_chk_real_range( 
  struct hpe1441_globals *thisPtr,
  ViReal64 my_val,
  ViReal64 min,
  ViReal64 max)
{
   char message[hpe1441_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1441_MSG_REAL, min, max, my_val);	
      hpe1441_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
#define hpe1441_CHK_INT_RANGE( my_val, min, max, err ) if( hpe1441_chk_int_range( thisPtr, my_val, min, max) ) hpe1441_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1441_chk_int_range( 
  struct hpe1441_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 min,
  ViInt16 max)
{
   char message[hpe1441_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1441_MSG_INT, min, max, my_val);	
      hpe1441_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1441_CHK_LONG_RANGE( my_val, min, max, err ) if( hpe1441_chk_long_range( thisPtr, my_val, min, max) ) hpe1441_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1441_chk_long_range( 
  struct hpe1441_globals *thisPtr,
  ViInt32 my_val,
  ViInt32 min,
  ViInt32 max)
{
   char message[hpe1441_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )  			
   {								
      sprintf(message, hpe1441_MSG_LONG, min, max, my_val);	
      hpe1441_CDE_MESSAGE(message);				
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
} 
   
   
#define hpe1441_CHK_ENUM( my_val, limit, err ) if( hpe1441_chk_enum( thisPtr, my_val, limit) ) hpe1441_LOG_STATUS( vi, thisPtr, err);

/* utility routine which searches for a string in an array of strings. */
/* This is used by the CHK_ENUM macro */
static ViBoolean hpe1441_chk_enum (
  struct hpe1441_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 limit)
{
    char message[hpe1441_ERR_MSG_LENGTH];

    if ( ( my_val < 0 ) || (my_val > limit) )  			
    {								
        sprintf(message, hpe1441_MSG_INT, 0, limit, my_val);	
        hpe1441_CDE_MESSAGE(message);				
        /* true = parameter is invalid */
        return VI_TRUE;
    }

    return VI_FALSE;
}



/*  ==========================================================================  
     This function searches an array of strings for a specific string and     
     returns its index.  If successful, a VI_SUCCESS is returned, 
     else hpe1441_INSTR_ERROR_LOOKUP is returned.
    ======================================================================== */
ViStatus hpe1441_findIndex (struct hpe1441_globals *thisPtr,
			const char * const array_of_strings[],
			  /*last entry in array must be 0 */
			const char *string, /* string read from instrument */
			ViPInt16 index) /* result index */
{
    ViInt16 i;
    ViInt16 my_len;
    char search_str[20];
    char message[80];

    strcpy(search_str, string);

  /* get rid of newline if present in string */
  /* needed because %s includes newline in some VTL's */
    my_len = strlen(search_str);
    if (search_str[my_len - 1] == '\n')
       search_str[my_len - 1] = '\0';

    for (i = 0; array_of_strings[i]; i++)
    {
	if (!strcmp (array_of_strings[i], search_str))
	{
	    *index = i;
	    return VI_SUCCESS;
	}
    }

  /* if we got here, we did not find it */
    sprintf(message, hpe1441_MSG_NO_MATCH, search_str);	
    hpe1441_CDE_MESSAGE(message);				
    return hpe1441_INSTR_ERROR_LOOKUP;
}

#define GetGlobals(vi,thisPtr)\
{\
        errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);\
        if( VI_SUCCESS > errStatus)\
        {\
        hpe1441_LOG_STATUS( vi, NULL, errStatus);\
        }\
}

/****************************************************************************
hpe1441_init
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | InstrDesc                                         ViRsrc      IN
  |   ---------------------------------------------------------------------
  |  | The Instrument Description.
  |  |
  |  | Examples: VXI::5, GPIB-VXI::80::INSTR
   ------------------------------------------------------------------------
  | id_query                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | if( VI_TRUE) Perform In-System Verification.
  |  | if(VI_FALSE) Do not perform In-System Verification
   ------------------------------------------------------------------------
  | do_reset                                          ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | IF( VI_TRUE) Perform Reset Operation.
  |  | if(VI_FALSE) Do not perform Reset operation
   ------------------------------------------------------------------------
  | vi                                                ViPSession  OUT
  |   ---------------------------------------------------------------------
  |  | Instrument Handle. This is VI_NULL if an error occurred
  |  | during the init.

*****************************************************************************/

ViStatus _VI_FUNC hpe1441_init(ViRsrc InstrDesc, ViBoolean id_query, ViBoolean do_reset, ViPSession vi )
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;
        ViUInt16 intf;

	ViUInt16 manfId;
	ViUInt16 modelCode;

        ViInt32 idx;

	*vi = VI_NULL;

	/* Find the Default Resource Manager */
	errStatus = viOpenDefaultRM( &defRM);
	if( VI_SUCCESS > errStatus)
	{
		/* Errors: VI_ERROR_SYSTEM_ERROR 
		 *         VI_ERROR_ALLOC
		 */
		hpe1441_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* Open a session to the instrument */
	errStatus = viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi);
	if( VI_SUCCESS > errStatus)
	{	
		viClose( defRM);
		/* Errors: VI_ERROR_NSUP_OPER
		 *         VI_ERROR_INV_RSRC_NAME
		 *         VI_ERROR_INV_ACC_MODE
		 *         VI_ERROR_RSRC_NFOUND
		 *         VI_ERROR_ALLOC
		 */
		*vi = VI_NULL;
		hpe1441_LOG_STATUS( *vi, NULL, errStatus);
	}

	/* get memory for instance specific globals */
	thisPtr = (struct hpe1441_globals *)malloc(sizeof( struct hpe1441_globals) );
	if( 0 == thisPtr)
	{
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1441_LOG_STATUS(*vi, NULL, VI_ERROR_ALLOC);
	}

	/* associate memory with session, should not fail because
	 *   session is valid; and attribute is defined, supported,
	 *   and writable.
	 */
	errStatus = viSetAttribute( 
	              *vi, 
		      VI_ATTR_USER_DATA, 
	              (ViAttrState)thisPtr); 
	if( VI_SUCCESS > errStatus)
	{
		viClose( *vi);
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		hpe1441_LOG_STATUS(*vi, NULL, errStatus);
	}

	/* initialize instance globals */
	thisPtr->defRMSession = defRM;
        thisPtr->errNumber = VI_SUCCESS;
	thisPtr->errFuncName[0] = 0;
	thisPtr->errMessage[0] = 0;
	thisPtr->errQueryDetect = VI_FALSE;
	thisPtr->blockSrqIO = VI_FALSE;
	thisPtr->countSrqIO = 0;
        thisPtr->myDelay = setDelay( 100e-6 );


	for( idx=0;idx<hpe1441_MAX_STAT_REG; idx++)
	{
	    thisPtr->driverEventArray[idx] = 0;
	}
	thisPtr->numEventHandlers = 0;
	for( idx=0;idx<hpe1441_MAX_STAT_HAP; idx++)
	{
	    thisPtr->eventHandlerArray[idx].eventHandler = NULL;
	    thisPtr->eventHandlerArray[idx].userData = 0;
	}
	
	if( VI_TRUE == do_reset ) 
	{
	     /* dev clr and call the reset function to reset the instrument */
		if( viClear(*vi) < VI_SUCCESS || hpe1441_reset(*vi) < VI_SUCCESS )
		{
			/* ignore any errors in PREFIX_close */
			hpe1441_close( *vi);
                        *vi=VI_NULL;
			hpe1441_LOG_STATUS( *vi, NULL, 
			  hpe1441_INSTR_ERROR_RESET_FAILED);
		}
                Sleep(100);
	}

	if( VI_TRUE == id_query ) 
	{
		char     idn_buf[256];
		
		/* find the interface type */
		if( errStatus = viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf) )
		{	
			/* Errors: VI_ERROR_NSUP_ATTR */

			/* ignore any errors in PREFIX_close */
			hpe1441_close( *vi);
                        *vi=VI_NULL;
	
			hpe1441_LOG_STATUS( *vi, NULL, errStatus);
		}

		switch( intf)
		{
			case VI_INTF_GPIB:
				if (
				    viClear(*vi) < VI_SUCCESS ||
				    viPrintf(*vi, "*IDN?\n") < VI_SUCCESS ||
				    viScanf(*vi, "%t", idn_buf) < VI_SUCCESS ||
				    /* check for a idn match */
				    strncmp(idn_buf, hpe1441_IDN_STRING, 
					    strlen(hpe1441_IDN_STRING))   
				    )
                		{   
					/* ignore any errors in PREFIX_close */
					hpe1441_close( *vi);
                                        *vi=VI_NULL;
					hpe1441_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
                		}

				break;

			case VI_INTF_VXI:
			case VI_INTF_GPIB_VXI:
	
				/* find the VXI manfacturer's ID */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MANF_ID, &manfId);
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
	
					/* ignore any errors in PREFIX_close */
					hpe1441_close( *vi);
                                        *vi=VI_NULL;
	
					hpe1441_LOG_STATUS( *vi, NULL,
					  errStatus);
				}

				/* find the instrument's model code */
				errStatus = viGetAttribute( *vi, 
				    VI_ATTR_MODEL_CODE, 
		       		(ViPAttrState)(&modelCode)); 
				if( VI_SUCCESS > errStatus)
				{	
					/* Errors: VI_ERROR_NSUP_ATTR */
					/* Note: this should never happen 
					 *   with a VXI instrument
					 */

					/* ignore any errors in PREFIX_close */
					hpe1441_close( *vi);
                                        *vi=VI_NULL;
					hpe1441_LOG_STATUS( *vi, NULL, errStatus);
				}

				if( (manfId != hpe1441_MANF_ID) || 
				    (modelCode != hpe1441_MODEL_CODE) )
				{
					/* ignore any errors in PREFIX_close */
					hpe1441_close( *vi);
                                        *vi = VI_NULL;

					hpe1441_LOG_STATUS( *vi, NULL,
					  VI_ERROR_FAIL_ID_QUERY);
				}

				break;

			default:
				/* ignore any errors in PREFIX_close */
				hpe1441_close( *vi);
				*vi = VI_NULL;
				hpe1441_LOG_STATUS( *vi, NULL,
				  hpe1441_INSTR_ERROR_UNEXPECTED);

		}	

	} /* if - id_query */

	hpe1441_LOG_STATUS( *vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1441_close
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_close (
  ViSession vi )
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;

        GetGlobals( vi,thisPtr );
	hpe1441_DEBUG_CHK_THIS( vi, thisPtr);

	/* retrieve Resource Management session */
	defRM = thisPtr->defRMSession;

	/* free memory */
	if( thisPtr)		
	{
		/* make sure there is something to free */
		free( thisPtr);
	}	

	/* close the vi and RM sessions */
	return viClose( defRM);
}


/****************************************************************************
hpe1441_reset
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_reset (
  ViSession vi )
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;

        GetGlobals(vi,thisPtr)
	hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1441_CDE_INIT( "hpe1441_reset");

	errStatus = viPrintf( vi, "*RST\n");
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}


/****************************************************************************
hpe1441_self_test
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | test_result                                       ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | Numeric result from self-test operation
  |  |
  |  | 0 = no error ( test passed)
  |  | 1 = failure
   ------------------------------------------------------------------------
  | test_message                                      ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Self-test status message.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_self_test (
  ViSession vi,
  ViPInt16 test_result,
  ViChar _VI_FAR test_message[])
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*test_result = -1; 
	test_message[0] = 0; 

        GetGlobals(vi,thisPtr)
	hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1441_CDE_INIT( "hpe1441_self_test");

	thisPtr->blockSrqIO = VI_TRUE;

	errStatus = viPrintf( vi, "*TST?\n");
	if( VI_SUCCESS > errStatus)
	{	
		
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	errStatus = viScanf( vi, "%hd%*t", test_result);
        if(thisPtr) doDelay(thisPtr->myDelay);

	if( VI_SUCCESS > errStatus)
	{
		*test_result = -1; 
		
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	switch( *test_result)
	{
           case 0:
              sprintf(test_message, hpe1441_MSG_SELF_TEST_PASSED);
              break;

	      /*??? add instrument specific cases here*/

           default:
              sprintf(test_message, hpe1441_MSG_SELF_TEST_FAILED);
              break;
        }

	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1441_error_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | error                                      ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | Instruments error code
   ------------------------------------------------------------------------
  | error_message                                     ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Instrument's error message.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_error_query (
  ViSession vi,
  ViPInt32 error,
  ViChar _VI_FAR error_message[])
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;

	/* initialize output parameters */
	*error = -1; 
	error_message[0] = 0; 

        GetGlobals(vi,thisPtr)
	hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1441_CDE_INIT( "hpe1441_error_query");

	thisPtr->blockSrqIO = VI_TRUE;

        errStatus = viPrintf( vi, "SYST:ERR?\n");
	if( VI_SUCCESS > errStatus)
	{
		
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* get the error number */
	errStatus = viScanf( vi, "%ld,%t", error, error_message);
        if(thisPtr)doDelay(thisPtr->myDelay);

	/* check for error during the scan */
        if( VI_SUCCESS > errStatus)
	{
		*error = -1; 
		error_message[0] = 0; 
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* get rid of extra LF at the end of the error_message */
	error_message[strlen(error_message)-1] = 0;

	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}

/****************************************************************************
hpe1441_error_message
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init().
  |  | This may be VI_NULL.
   ------------------------------------------------------------------------
  | error                                      ViStatus    IN
  |   ---------------------------------------------------------------------
  |  | The error return value from an instrument driver function
   ------------------------------------------------------------------------
  | message                                           ViChar _VI_FAR []OUT
  |   ---------------------------------------------------------------------
  |  | Error message string.  This is limited to 256 characters.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_error_message (
  ViSession vi,
  ViStatus error,
  ViChar _VI_FAR message[])
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;  
	ViInt32 idx;

	/* initialize output parameters */
	message[0] = 0;

	thisPtr = NULL;

	/* try to find a thisPtr */
	if( VI_NULL != vi)
	{
                GetGlobals(vi,thisPtr)
		hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
	} 

	if( VI_SUCCESS == error)
	{
		sprintf( message, hpe1441_MSG_NO_ERRORS);
		hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* return the static error message */
	for(idx=0; 
	    idx < (sizeof instrErrMsgTable / 
	           sizeof( struct instrErrStruct));
	    idx++)
	{
		/* check for a matching error number */
		if( instrErrMsgTable[idx].errStatus == error)
		{
			if( (thisPtr) &&
			    (thisPtr->errNumber == error))
			{
				/* context dependent error
				 * message is available.
				 */
				sprintf( message,
				  "%s " hpe1441_MSG_IN_FUNCTION " %s() %s",
		          	  instrErrMsgTable[idx].errMessage,
				  thisPtr->errFuncName,
				  thisPtr->errMessage);
			}
			else
			{
				/* No context dependent eror 
				 * message available so copy 
				 * the static error message
				 */
				strcpy( message,
		          	  instrErrMsgTable[idx].errMessage);

			}
            
			hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
		}
	}


	/* try VTL/VISA to see if it exists there 
	 *
	 * At this point vi is either VALID or VI_NULL
	 */
	errStatus = viStatusDesc(vi, error, message);
	if( VI_SUCCESS == errStatus)
	{
		/* check for a context dependent error message */
		if( (thisPtr) &&
		    (thisPtr->errNumber == error))
		{
			/* context dependent error
			 * message is available.
			 */
			strcat( message, " ");
			strcat( message, hpe1441_MSG_IN_FUNCTION);
			strcat( message, " ");
			strcat( message, thisPtr->errFuncName);
			strcat( message, "() ");
			strcat( message, thisPtr->errMessage);
		}

		/* VTL found an error message, so return success */
		hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* if we have a VI_NULL, then we need to return a error message */
	if( VI_NULL == vi)
	{
		strcpy(message, hpe1441_MSG_VI_OPEN_ERR);
		hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	} 

	/* user passed in a invalid status */
	sprintf( message, 
	  hpe1441_MSG_INVALID_STATUS
	  "  %ld"
	  hpe1441_MSG_INVALID_STATUS_VALUE
	  , (long)error );
	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
}


/****************************************************************************
hpe1441_revision_query
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession      IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | driver_rev                                ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument driver revision.  This is limited to 256 characters.
   ------------------------------------------------------------------------
  | instr_rev                                 ViChar _VI_FAR []      OUT
  |   ---------------------------------------------------------------------
  |  | Instrument firmware revision.  This is limited to 256 characters.

*****************************************************************************/

ViStatus _VI_FUNC hpe1441_revision_query (
  ViSession vi,
  ViChar _VI_FAR driver_rev[],
  ViChar _VI_FAR instr_rev[])
{
	struct hpe1441_globals *thisPtr;
	ViStatus errStatus;
        char temp_str[256];		/* temp hold for instr rev string */
        char *last_comma;		/* last comma in *IDN string */

	/* initialize output parameters */
        driver_rev[0] = 0; 
	instr_rev[0] = 0; 

        GetGlobals(vi,thisPtr)
	hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
	hpe1441_CDE_INIT( "hpe1441_revision_query");

	sprintf( driver_rev, "%s", hpe1441_REV_CODE);

	thisPtr->blockSrqIO = VI_TRUE;

	if( errStatus = viPrintf( vi, "*IDN?\n"))
	{
        	driver_rev[0] = 0; 
		 
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus); 
	}

	if( errStatus = viScanf( vi, "%s%*t", temp_str))
	{
        	driver_rev[0] = 0; 
		instr_rev[0] = 0; 
                if(thisPtr)doDelay(thisPtr->myDelay);
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus); 
	}
                if(thisPtr)doDelay(thisPtr->myDelay);
        
        last_comma = strrchr(temp_str,',');
	/* error and exit if last comma not found */
        if (!last_comma) 
        {
           instr_rev[0] = 0;
	   hpe1441_CDE_MESSAGE("no last comma found in IDN string" ); 
	   hpe1441_LOG_STATUS( vi, thisPtr, hpe1441_INSTR_ERROR_UNEXPECTED); 
        }

        strcpy(instr_rev, last_comma+1);
                if(thisPtr)doDelay(thisPtr->myDelay);
	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS); 
}

/****************************************************************************
hpe1441_timeOut
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | timeOut                                           ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | This value sets the I/O timeout for all functions in
  |  | the driver. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Changes the timeout value of the instrument.  Input is in     */
/*           milliseconds.                                                 */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1441_timeOut (ViSession vi, ViInt32 timeOut)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;
   

   GetGlobals(vi,thisPtr)
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_timeOut" );

   hpe1441_CHK_LONG_RANGE( timeOut, 1, 2147483647, VI_ERROR_PARAMETER2 );

   errStatus = viSetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut);
   if ( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_timeOut_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | pTimeOut                                          ViPInt32    OUT
  |   ---------------------------------------------------------------------
  |  | This is the minimum timeout period that the driver
  |  | can be set to. It is specified in milliseconds.

*****************************************************************************/
/* ----------------------------------------------------------------------- */
/* Purpose:  Returns the current setting of the timeout value of the       */
/*           instrument in milliseconds.                                   */
/* ----------------------------------------------------------------------- */
ViStatus _VI_FUNC hpe1441_timeOut_Q (ViSession vi, ViPInt32 timeOut)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_timeOut_Q" );

   errStatus = viGetAttribute(vi, VI_ATTR_TMO_VALUE, timeOut );
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_errorQueryDetect
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | errorQueryDetect                                  ViBoolean   IN
  |   ---------------------------------------------------------------------
  |  | Boolean which enables (VI_TRUE) or disables (VI_FALSE)
  |  | automatic instrument error querying.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_errorQueryDetect( 
  ViSession vi, 
  ViBoolean errDetect)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_errorQueryDetect" );

   hpe1441_CHK_BOOLEAN( errDetect, VI_ERROR_PARAMETER2 );

   thisPtr->errQueryDetect = errDetect;

   hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_errorQueryDetect_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | pErrDetect                                        ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | Boolean indicating if automatic instrument error
  |  | querying is performed.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_errorQueryDetect_Q( 
  ViSession vi, 
  ViPBoolean pErrDetect)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_errorQueryDetect_Q" );

   *pErrDetect = thisPtr->errQueryDetect;

   hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_dcl
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_dcl( 
  ViSession vi)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   GetGlobals(vi,thisPtr)
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_dcl" );

   errStatus = viClear(vi);
   if( VI_SUCCESS > errStatus) { hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }	

   hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/* hpe1441_operEvent_Q */
ViStatus _VI_FUNC hpe1441_operEvent_Q( ViSession vi, ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1441_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
 {
  errStatus = viWrite(vi,"STAT:OPER:EVEN?",15,&count);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
                        if(thisPtr)doDelay(thisPtr->myDelay);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
 hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1441_operCond_Q */
ViStatus _VI_FUNC hpe1441_operCond_Q( ViSession vi, ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1441_globals *thisPtr;
 ViInt32 count;
 char buf[20];
 GetGlobals(vi,thisPtr)
 {
  errStatus = viWrite(vi,"STAT:OPER:COND?",15,&count);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
                        if(thisPtr)doDelay(thisPtr->myDelay);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
 hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1441_statusQuesEven_Q */
ViStatus _VI_FUNC hpe1441_quesEvent_Q(
ViSession vi,
ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1441_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
 {
  errStatus = viWrite(vi,"STAT:QUES:EVEN?",15,&count);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
                        if(thisPtr)doDelay(thisPtr->myDelay);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
 hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/* hpe1441_QuesCond_Q */
ViStatus _VI_FUNC hpe1441_quesCond_Q( ViSession vi, ViPInt32 val)
{
 ViStatus errStatus;
 struct hpe1441_globals *thisPtr;
 ViInt32 count;
 char buf[20];

 GetGlobals(vi,thisPtr)
 {
  errStatus = viWrite(vi,"STAT:QUES:COND?",15,&count);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );

  errStatus = viRead(vi,buf,20,&count);
  if(thisPtr)doDelay(thisPtr->myDelay);
  if( errStatus < VI_SUCCESS)
        hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
  buf[count]='\0';
  *val = atoi( buf);
 }
 hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}


/****************************************************************************
hpe1441_opc
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_opc (ViSession vi)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_opc" );

   errStatus = viPrintf(vi,"*OPC\n");
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_opc_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | opc                                               ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | returns a 1 when all pending instruments commands are completed

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_opc_Q (ViSession vi, ViPInt16 opc)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_opc_Q" );

   thisPtr->blockSrqIO = VI_TRUE;

   errStatus = viPrintf(vi,"*OPC?\n");
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   errStatus = viScanf(vi, "%hd%*t", opc);
  if(thisPtr)doDelay(thisPtr->myDelay);
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_wai
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_wai (ViSession vi)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   if ( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_wai" );

   errStatus = viPrintf(vi,"*WAI\n");
   if (VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

/****************************************************************************
hpe1441_trg
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_trg (ViSession vi)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;

   errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   if ( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_trg" );

   errStatus = viPrintf(vi,"*TRG\n");
   if ( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}


/****************************************************************************
hpe1441_readStatusByte_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | statusByte                                        ViPInt16    OUT
  |   ---------------------------------------------------------------------
  |  | returns the contents of the status byte

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_readStatusByte_Q(ViSession vi, ViPInt16 statusByte)
{
   ViStatus errStatus = 0;
   struct hpe1441_globals *thisPtr;
   ViUInt16 stb;

   errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   if ( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, NULL, errStatus );
   }
   hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
   hpe1441_CDE_INIT( "hpe1441_readStatusByte_Q" );

   errStatus = viReadSTB(vi,&stb);
   if( VI_SUCCESS > errStatus)
   {
      hpe1441_LOG_STATUS(vi, thisPtr, errStatus );
   }

   *statusByte = (ViInt16)stb;

   hpe1441_LOG_STATUS(vi, thisPtr, VI_SUCCESS );
}

ViInt32 hpe1441_statusHap[hpe1441_MAX_STAT_HAP] = {
hpe1441_USER_ERROR_HANDLER,


hpe1441_ESR_OPC,
hpe1441_ESR_QUERY_ERROR,
hpe1441_ESR_DEVICE_DEPENDENT_ERROR,
hpe1441_ESR_EXECUTION_ERROR,
hpe1441_ESR_COMMAND_ERROR,
hpe1441_QUES_UNLOCKED

};


/* Assumes we have driver copies of the event register.  This is needed
 * because in IEEE 488.2, the event register are cleared after they are 
 * read.  Since the event register contains several events, we need to
 * keep this information around to pass back to the user.
 */


struct hpe1441_statusAccess {
ViInt32 registerIdx; 
ViString condQry;
ViString eventQry;
ViString enableCmd;
};

const struct hpe1441_statusAccess hpe1441_accessInfo[hpe1441_MAX_STAT_REG] = {
{0, 	"",	 		"*STB?",		"*SRE"},
{400,   "STAT:QUES:COND?",      "STAT:QUES:EVEN?",      "STAT:QUES:ENAB"},
{600,	"",			"*ESR?",			"*ESE"}
};

/* this will return the index associated with the happening */
ViBoolean hpe1441_findHappeningIdx(
  ViInt32 happening,
  ViPInt32 pIdx)
{
	/* Note: this is a linear search, for faster access this
	 * could be done as a binary search since the data is arrange
	 * in order numerically.
	 */
	for( *pIdx=0; *pIdx<hpe1441_MAX_STAT_HAP; *pIdx = *pIdx + 1)
	{
		if( hpe1441_statusHap[*pIdx] == happening)
		{
			return VI_TRUE;
		}	
	}

	return VI_FALSE;
}

/* this will return the index that corresponds with regNum */
static ViBoolean hpe1441_findAccessIdx(
  ViInt32 regNum,
  ViPInt32 pIdx )
{
	for(*pIdx=0; *pIdx<hpe1441_MAX_STAT_REG; *pIdx = *pIdx + 1)
	{
		if( regNum == hpe1441_accessInfo[*pIdx].registerIdx)
		{	
			return VI_TRUE;
		}
	}
	return VI_FALSE;
}

static ViStatus hpe1441_readAllEvents( 
  ViSession vi)
{
	ViInt32 idx;
	ViStatus errStatus;
	ViInt32 eventResp;
        struct hpe1441_globals *thisPtr;

	errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
   	if( VI_SUCCESS > errStatus) 
   	{
   	   hpe1441_LOG_STATUS( vi, NULL, errStatus );
   	}
   	hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
        

	/* read all events and update driver events */
	for(idx=0; idx<hpe1441_MAX_STAT_REG; idx++)
	{
		/* if there is an event query */
		if( 0 != strlen(hpe1441_accessInfo[idx].eventQry))
		{
			/* okay - query the instrument for event */
			thisPtr->blockSrqIO = VI_TRUE;

			errStatus = viPrintf(vi,
			  "%s\n", hpe1441_accessInfo[idx].eventQry);
			if( VI_SUCCESS > errStatus)
			{
				return errStatus;
			}

			/* get the response */
			errStatus = viScanf(vi, "%ld%*t", &eventResp);
  if(thisPtr)doDelay(thisPtr->myDelay);
			if( VI_SUCCESS > errStatus )
			{
				return errStatus;
			}

			/* update the driver events copy of instr events */
			thisPtr->driverEventArray[idx] = 	
			  thisPtr->driverEventArray[idx] | eventResp;

		} /* if - an event query exists */

	} /* for - all status registers */

	return VI_SUCCESS;

} /* hpe1441_readAllEvents() */


/* this recursive routine traverse the IEEE 488.2 status structures and
 * processes events.
 */
static void hpe1441_srqTraverse(
  ViSession vi,
  ViInt32 eventReg)
{
        struct hpe1441_globals *thisPtr;
	ViInt32 accessIdx;
	ViInt32 hapIdx;
	ViUInt16 Status;
	ViInt32 eventVal;
	ViInt32 happening;
	ViInt32 bitPos;

        if(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )
	{
		/* error couldn't get attribute */
		return;  
	}

	if( VI_FALSE == hpe1441_findAccessIdx( eventReg, &accessIdx))
	{
		/* couldn't find register num 
		 *   don't know what to do so leave 
		 */
		return;
	}

	if( 0 == eventReg )
	{
		/* need to read status byte through VTL instead of SCPI */
		if( viReadSTB( vi, &Status))
		{
			/* an error occured */
			return;
		}
		eventVal = (ViInt32)Status;
	}
	else
	{
		/* find the event in the table */
		if( 0 != strlen(hpe1441_accessInfo[accessIdx].eventQry))
		{
			/* query instrument to find what event(s) occured. */
			if( viPrintf(vi,"%s\n", 
		              hpe1441_accessInfo[accessIdx].eventQry))
			{
				/* an error occured */
				return;
			}

			if( viScanf(vi,"%ld%*t\n", &eventVal))
			{
  if(thisPtr)doDelay(thisPtr->myDelay);
				/* an error occured */
				return;
			}
  if(thisPtr)doDelay(thisPtr->myDelay);
		}
		else
		{
			/* error - no events for this status register */
			return;
		}
	} /* else - not status byte */

	/* update driver events */
	thisPtr->driverEventArray[accessIdx] = 
	  thisPtr->driverEventArray[accessIdx] | eventVal;

	for( bitPos = 0; bitPos < 16; bitPos++)
	{
		/* check for an event occurance */
		if( eventVal & (1 << bitPos) )
		{
			/* find happening index */
			happening = eventReg + bitPos + 1;
			if( VI_TRUE == 
			       hpe1441_findHappeningIdx( happening, &hapIdx) )
			{
				/* does event have a handler enabled? */
				if( thisPtr->eventHandlerArray[hapIdx].eventHandler)
				{
					/* call the users handler */
			    	thisPtr->eventHandlerArray[hapIdx].eventHandler(
			    	  vi, happening, 
			    	  thisPtr->eventHandlerArray[hapIdx].userData);
				}				
			}

		} /* if - event occured */

		/* check for more status registers */
		if( VI_TRUE == hpe1441_findAccessIdx( happening*100, &accessIdx))
		{
			/* need to traverse these other registers */
			hpe1441_srqTraverse( vi, happening*100);
		}

	} /* for - all bits in the event */
}


/* this is the SRQ event handler */
ViStatus _VI_FUNCH hpe1441_srqHdlr(
  ViSession vi,
  ViEventType eventType,
  ViEvent event,
  ViAddr userData)
{
	/* the only reason we should get called is for a Service Request */	

	/* for VTL 3.0 we always need to return VI_SUCCESS, therefore
	 * we will ignore any VTL errors at this level.
	 */

        struct hpe1441_globals *thisPtr;

        if(VI_SUCCESS > 
	  viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )
	{
		/* error couldn't get attribute, 
		 * but we lie and return VI_SUCCESS */
		return VI_SUCCESS;  
	}

	/* check if it is okay to do I/O in SRQ handler */
	if( VI_TRUE == thisPtr->blockSrqIO)
	{
	    /* not okay to do I/O so just update srq count */
	    thisPtr->countSrqIO = thisPtr->countSrqIO + 1;
     	    return VI_SUCCESS;  
	}

	/* suspend event processing for now */
	viEnableEvent( vi, VI_EVENT_SERVICE_REQ, VI_SUSPEND_HNDLR, VI_NULL);

	/* recursively traverse status byte */
	hpe1441_srqTraverse( vi, 0);

	/* re-enable event processing */
        viEnableEvent( vi, VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);

	/* always return VI_SUCCESS */
	return VI_SUCCESS;
} 



/****************************************************************************
hpe1441_statCond_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | The following conditions can occur on the counter.
  |  | Conditions are transient in nature, and this function reports
  |  | the current setting of the condition. A 1 is returned if the
  |  | condition is currently true, and 0 is returned if the condition
  |  | is currently false.
  |  |
   ------------------------------------------------------------------------
  | pCondition                                        ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | VI_TRUE  = condition is currently set
  |  | VI_FALSE = condition is currently not set

*****************************************************************************/

ViStatus _VI_FUNC hpe1441_statCond_Q(
  ViSession	vi,
  ViInt32	happening,
  ViPBoolean	pCondition 
)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	ViInt32 accessIdx;
	ViInt32 cond_q;
	ViInt32 regNum;
	ViInt32 bitNum;
	char errMsg[256];

        /* initialize output parameters */
	*pCondition = VI_FALSE;

	errStatus = viGetAttribute(
                vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statCond_Q");

	/* check validity of happening */
	if( (hpe1441_USER_ERROR_HANDLER ==  happening) ||
            (VI_FALSE == hpe1441_findHappeningIdx( happening, &hapIdx) ))
	{
		sprintf(errMsg,"%hd " hpe1441_MSG_INVALID_HAPPENING, happening);
		hpe1441_CDE_MESSAGE( errMsg);
		
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	regNum = happening / 100;
	bitNum = happening % 100;

	/* get access index */
	if( VI_FALSE == hpe1441_findAccessIdx( regNum * 100, &accessIdx))
	{
		
		hpe1441_LOG_STATUS( vi, thisPtr, hpe1441_INSTR_ERROR_UNEXPECTED);
	}

	if( 0 == strlen(hpe1441_accessInfo[accessIdx].condQry))
	{
		char errMsg[256];

		/* unable to query the condition */
		sprintf(errMsg, hpe1441_MSG_CONDITION
                        "%hd" hpe1441_MSG_NOT_QUERIABLE, happening);
		hpe1441_CDE_MESSAGE(errMsg); 
		
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}
	else
	{
		/* okay - query the instrument for condition */
                thisPtr->blockSrqIO = VI_TRUE;

		errStatus = viPrintf(vi,
		  "%s\n", hpe1441_accessInfo[accessIdx].condQry);
		if( 0 > errStatus)
		{
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get the condition response  */
		errStatus = viScanf(vi, "%ld%*t", &cond_q);
  if(thisPtr)doDelay(thisPtr->myDelay);
		if( 0 > errStatus )
		{
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* check appropriate condition bit */
		if( cond_q & (1 << (bitNum-1)) )
		{
			*pCondition = VI_TRUE;
		}
		else
		{
			*pCondition = VI_FALSE;
		}
	}

	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	


/****************************************************************************
hpe1441_statEven_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens. These can refer
  |  | to conditions or events. All of the conditions listed in the
  |  | hpe1441_statCond_Q() function will be detected as events as
  |  | well.   An event may be registered when a condition changes
  |  | state from  VI_FALSE to VI_TRUE.
  |  |
   ------------------------------------------------------------------------
  | pEvent                                            ViPBoolean  OUT
  |   ---------------------------------------------------------------------
  |  | VI_TRUE  = event occured sometime between event readings
  |  | VI_FALSE = the event did not occur between event readings

*****************************************************************************/

ViStatus _VI_FUNC hpe1441_statEven_Q(
  ViSession 	vi,
  ViInt32	happening,
  ViPBoolean	pEvent
)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	ViInt32 accessIdx;
	ViInt32 event_q;
	ViInt32 regNum;
	ViInt32 bitNum;
	ViInt32 mask;
	char errMsg[256];

        /* initialize output parameters */
	*pEvent = VI_FALSE;

	errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statEven_Q");

	/* check validity of happening */
	if( (hpe1441_USER_ERROR_HANDLER == happening) ||
            (VI_FALSE == hpe1441_findHappeningIdx( happening, &hapIdx) ))
	{
		sprintf(errMsg,"%hd " hpe1441_MSG_INVALID_HAPPENING, happening);
		hpe1441_CDE_MESSAGE( errMsg);
		
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	regNum = happening / 100;
	bitNum = happening % 100;

	/* get access index */
	if( VI_FALSE == hpe1441_findAccessIdx( regNum * 100, &accessIdx))
	{
		
		hpe1441_LOG_STATUS( vi, thisPtr, hpe1441_INSTR_ERROR_UNEXPECTED);
	}

	/* see if we can query the event from the instrument */
	if( 0 == strlen(hpe1441_accessInfo[accessIdx].eventQry))
	{
		char errMsg[256];

		/* unable to query the event */
		sprintf(errMsg, hpe1441_MSG_EVENT
		  "%hd" hpe1441_MSG_NOT_QUERIABLE, happening);
		hpe1441_CDE_MESSAGE(errMsg); 
		
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}
	else
	{
		/* okay - query the instrument for event */
                thisPtr->blockSrqIO = VI_TRUE;

		errStatus = viPrintf(vi,
		  "%s\n", hpe1441_accessInfo[accessIdx].eventQry);
		if( 0 > errStatus)
		{
			
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* get the response */
		errStatus = viScanf(vi, "%ld%*t", &event_q);
  if(thisPtr)doDelay(thisPtr->myDelay);
		if( 0 > errStatus )
		{
			
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* update the driver events copy of instrument events */
		thisPtr->driverEventArray[accessIdx] = 	
		  thisPtr->driverEventArray[accessIdx] | event_q;

		/* check state of event bit */
		mask = 1 << (bitNum -1);
		if( thisPtr->driverEventArray[accessIdx] & mask )
		{
			*pEvent = VI_TRUE;

			/* clear event bit in driver events */
			thisPtr->driverEventArray[accessIdx] = 	
		 	  thisPtr->driverEventArray[accessIdx] & (~mask);
		}
		else
		{
			*pEvent = VI_FALSE;

			/* event bit in driver event array
			 * is already cleared 
			 */
		}
	}

	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	


/****************************************************************************
hpe1441_statEvenHdlr
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32     IN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens. These can refer to
  |  | conditions or events. Happenings are enumerated as ViInt32
  |  | numbers.  Only events can have happenings associated with it.
  |  |
   ------------------------------------------------------------------------
  | eventHandler                           hpe1441_InstrEventHandler  IN
  |   ---------------------------------------------------------------------
  |  | This is either NULL or a pointer to the user specified
  |  | event handler.  A NULL disables the event handler.
  |  |
  |  | An event handler has the following prototype:
  |  |
  |  | typedef void (_VI_FUNCH _VI_PTR  hpe1441_InstrEventHandler)(
  |  | ViSession vi,
  |  | ViInt32 happening,
  |  | ViAddr userData
  |  | );
   ------------------------------------------------------------------------
  | userData                                          ViAddr      IN
  |   ---------------------------------------------------------------------
  |  | This is a pointer that is passed to the handler when the
  |  | specified event occurs.  This can be used by the programmer
  |  | to pass additional information to the handler.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_statEvenHdlr(
  ViSession vi,
  ViInt32 happening,
  hpe1441_InstrEventHandler eventHandler,
  ViAddr userData
)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	ViInt32 accessIdx;
	ViInt32 propagate;
	ViInt32 regNum;
	ViInt32 bitNum;
	ViInt32 enableMask;
	char errMsg[80];

        /* initialize output parameters */

	errStatus = viGetAttribute(
                vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statEvenHdlr");

	if( hpe1441_USER_ERROR_HANDLER == happening)
	{
		/* store handler and user data */
		thisPtr->eventHandlerArray[
                   hpe1441_USER_ERROR_HANDLER_IDX].eventHandler = eventHandler;
		thisPtr->eventHandlerArray[
                   hpe1441_USER_ERROR_HANDLER_IDX].userData = userData;
		hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* check validity of happening */
	if( VI_FALSE == hpe1441_findHappeningIdx( happening, &hapIdx) )
	{

		sprintf(errMsg,"%ld " hpe1441_MSG_INVALID_HAPPENING, 
		  (long)happening);
		hpe1441_CDE_MESSAGE( errMsg);
		
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	regNum = happening / 100;
	bitNum = happening % 100;

	/* suspend SRQ events - if handler is installed */
	if( 0 > thisPtr->numEventHandlers)
	{
		errStatus = viEnableEvent( vi,VI_EVENT_SERVICE_REQ ,
		  VI_SUSPEND_HNDLR, VI_NULL);
		if( errStatus)
		{
                	
                	hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}
	}	

	/* see if we want to enable or disable the handler */
	if( NULL != eventHandler)
	{
		/* see if event handler already exists */		
		if( NULL == thisPtr->eventHandlerArray[hapIdx].eventHandler )
		{
			/* okay - no event handler */
			thisPtr->numEventHandlers =
			  thisPtr->numEventHandlers + 1;
		}
		else
		{
			/* reenable SRQ events - if handler is installed */
			if( 0 > thisPtr->numEventHandlers)
			{
				errStatus = viEnableEvent( vi,
				  VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);
				if( errStatus)
				{
               			 	
                			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
				}
			}	

			/* error - event handler already exists */
			sprintf(errMsg, 
			  hpe1441_MSG_EVENT_HDLR_INSTALLED
			  " %ld.  "
			  hpe1441_MSG_EVENT_HDLR_INST2, (long)happening);
			
			hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
			
		}

		/* store handler and user data */
		thisPtr->eventHandlerArray[hapIdx].eventHandler = eventHandler;
		thisPtr->eventHandlerArray[hapIdx].userData = userData;

		/* do we need to install the event handler? */
		if( 1 == thisPtr->numEventHandlers)
		{
			/* install the event handler */
			errStatus = viInstallHandler(vi, 
			  VI_EVENT_SERVICE_REQ, hpe1441_srqHdlr, VI_NULL);
			if( errStatus)
			{
               			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
			}

			/* enable events */
			errStatus = viEnableEvent( vi,
			  VI_EVENT_SERVICE_REQ, VI_HNDLR, VI_NULL);
			if( errStatus)
			{
                		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
			}
		}

	} /* if - install event handler */
	else
	{
		/* see if event handler already exists */		
		if( NULL != thisPtr->eventHandlerArray[hapIdx].eventHandler )
		{
			/* okay - no event handler */
			thisPtr->numEventHandlers =
			  thisPtr->numEventHandlers - 1;
		}

		/* clear handler and user data */
		thisPtr->eventHandlerArray[hapIdx].eventHandler = NULL;
		thisPtr->eventHandlerArray[hapIdx].userData = NULL;

		/* do we need to uninstall the event handler? */
		if( 0 == thisPtr->numEventHandlers)
		{
			/* disable SRQ events */
			errStatus = viDisableEvent( vi,
			  VI_EVENT_SERVICE_REQ ,VI_HNDLR);
			if( VI_SUCCESS > errStatus)
			{
               			
     		          	hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
			}

			/* flush any pending SRQ events */
			errStatus = viDiscardEvents( vi,
			  VI_EVENT_SERVICE_REQ ,VI_SUSPEND_HNDLR);
			if( VI_SUCCESS > errStatus)
			{
               			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
			}

			/* uninstall the event handler */
			errStatus = viUninstallHandler(vi, 
			  VI_EVENT_SERVICE_REQ, hpe1441_srqHdlr, VI_NULL);
			if( VI_SUCCESS > errStatus && 
			    VI_ERROR_INV_HNDLR_REF != errStatus)
			{
               			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
			}

		}

		/* remove all stale events */
		errStatus = hpe1441_readAllEvents( vi);
		if( VI_SUCCESS > errStatus)
		{
               		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

	} /* else - remove event handler */

	/* enable event bits to propagate through the IEEE 488.2 
	 * status system and eventually assert SRQ.
	 */
	propagate = happening;
	while( propagate > 0)
	{
		regNum = propagate / 100;
		bitNum = propagate % 100;

		/* get access index */
		if( VI_FALSE == hpe1441_findAccessIdx( regNum * 100, &accessIdx))
		{
			hpe1441_CDE_MESSAGE( "hpe1441_findAccessIdx failed");
			hpe1441_LOG_STATUS( vi, thisPtr, hpe1441_INSTR_ERROR_UNEXPECTED);
		}

		/* query enable value from instrument */
		thisPtr->blockSrqIO = VI_TRUE;

		errStatus = viPrintf( vi, "%s?\n",
		   hpe1441_accessInfo[accessIdx].enableCmd);
		if( VI_SUCCESS > errStatus)
		{
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}
	
		/* read back the response */
		errStatus = viScanf( vi, "%ld%*t", &enableMask);
  if(thisPtr)doDelay(thisPtr->myDelay);
		if( VI_SUCCESS > errStatus)
		{
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		/* enable/disable  the corresponding event bit */
		if( NULL != eventHandler)
		{
			/* enable event bit */
			enableMask = enableMask | (1 << (bitNum - 1));
		}	
		else
		{
			/* disable event bit */
			enableMask = enableMask & (~(1 << (bitNum - 1)));
		}

		/* write back the new enable mask */
		errStatus = viPrintf( vi, "%s %hd\n",
		   hpe1441_accessInfo[accessIdx].enableCmd,
		   enableMask);
		if( VI_SUCCESS > errStatus)
		{
			
			hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
		}

		if( NULL == eventHandler)
		{
			/* for disabling events, only disable the
			 * lowest level.
			 */
			propagate = 0;
		}
		else
		{
			propagate = propagate / 100;
		}

	} /* while - propagate */

	
	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	


/****************************************************************************
hpe1441_statEvenHdlr_Q
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()
   ------------------------------------------------------------------------
  | happening                                         ViInt32 happeningIN
  |   ---------------------------------------------------------------------
  |  | Happenings refer to something that happens.  These can refer to
  |  | conditions or events.  Happenings are enumerated as ViInt32
  |  | numbers.
  |  |
   ------------------------------------------------------------------------
  | pEventHandler                     hpe1441_InstrPEventHandler      OUT
  |   ---------------------------------------------------------------------
  |  | This is the definition of a hpe1441_InstrPEventHandler:
  |  |
  |  | typedef void (_VI_PTR _VI_PTR hpe1441_InstrPEventHandler)(
  |  | ViSession vi,
  |  | ViInt32 happening,
  |  | ViAddr userData
  |  | );
   ------------------------------------------------------------------------
  | pUserData                                         ViPAddr     OUT
  |   ---------------------------------------------------------------------
  |  | This is a pointer to the userData that was registered
  |  | with the handler.

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_statEvenHdlr_Q(
  ViSession vi,
  ViInt32 happening,
  hpe1441_InstrPEventHandler pEventHandler,
  ViPAddr pUserData)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 hapIdx;
	char errMsg[80];

        /* initialize output parameters */

	errStatus = viGetAttribute(
                vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statEvenHdlr_Q");

	/* check validity of happening */
	if( hpe1441_USER_ERROR_HANDLER == happening)
	{
		hapIdx = hpe1441_USER_ERROR_HANDLER_IDX;
	}
	else if( VI_FALSE == hpe1441_findHappeningIdx( happening, &hapIdx) )
	{
		sprintf(errMsg,"%hd " hpe1441_MSG_INVALID_HAPPENING, happening);
		hpe1441_CDE_MESSAGE( errMsg);
		hpe1441_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
	}

	*((void **)pEventHandler) = (void *)thisPtr->eventHandlerArray[hapIdx].eventHandler;
	*pUserData = thisPtr->eventHandlerArray[hapIdx].userData;

	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/****************************************************************************
hpe1441_statEvenHdlrDelAll
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_statEvenHdlrDelAll(
  ViSession vi)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;

        /* initialize output parameters */

	errStatus = viGetAttribute(
                vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statEvenHdlrDelAll");

	/* disable SRQ events */
	errStatus = viDisableEvent( vi,VI_EVENT_SERVICE_REQ ,VI_HNDLR);
	if( VI_SUCCESS > errStatus)
	{
               	hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* flush any pending SRQ events */
	errStatus = viDiscardEvents( vi,VI_EVENT_SERVICE_REQ ,VI_SUSPEND_HNDLR);
	if( VI_SUCCESS > errStatus)
	{
               	hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}
                                                                               
                                                                               
	/* uninstall the event handler */
	errStatus = viUninstallHandler(vi, 
	  VI_EVENT_SERVICE_REQ, hpe1441_srqHdlr, VI_NULL);
	if( VI_SUCCESS > errStatus && VI_ERROR_INV_HNDLR_REF != errStatus)
	{
        	hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/****************************************************************************
hpe1441_statEvenClr
*****************************************************************************
    Parameter Name                                       Type    Direction
   ------------------------------------------------------------------------
  | vi                                                ViSession   IN
  |   ---------------------------------------------------------------------
  |  | Instrument Handle returned from hpe1441_init()

*****************************************************************************/
ViStatus _VI_FUNC hpe1441_statEvenClr(
  ViSession vi
)
{
        struct hpe1441_globals *thisPtr;
        ViStatus errStatus;
	ViInt32 idx;

	errStatus = viGetAttribute(
                vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
        if( VI_SUCCESS > errStatus)
        {
                /* Errors: VI_ERROR_INV_SESSION */
                hpe1441_LOG_STATUS( vi, NULL, errStatus);
        }
        hpe1441_DEBUG_CHK_THIS( vi, thisPtr);
        hpe1441_CDE_INIT( "hpe1441_statEvenClr");

	/* clear instrument events */
	errStatus = viPrintf(vi, "*CLS\n");
	if( VI_SUCCESS > errStatus)
	{
		hpe1441_LOG_STATUS( vi, thisPtr, errStatus);
	}

	/* clear driver copy of instrument events */
	for(idx=0; idx<hpe1441_MAX_STAT_REG; idx++)
	{
		thisPtr->driverEventArray[idx] = 0;
	}

	hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
}	

/* The scpi escape funnctions */


/*	hpe1441_cmd:
	Send a scpi command, it does not look for a response
	
	input:
		vi			- session
		p1			- Scpi command string

	comment:
		assumes p1 is NULL terminated C string.
*/

ViStatus _VI_FUNC hpe1441_cmd( ViSession vi, ViString p1)
{
   ViStatus errStatus;
   struct   hpe1441_globals *thisPtr;
   GetGlobals(vi,thisPtr)

   errStatus = viPrintf(vi,"%s\n", p1);
   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
}


/* 	hpe1441_cmdString_Q
	Scpi escape function that returns a string.
	
	input:
		vi			- session
		p1			- The scpi command to be exectuted
		p2			- The size of p3
	
	output:
		p3			- the string returned by the instrument
		
*/
ViStatus _VI_FUNC hpe1441_cmdString_Q( ViSession vi,
           ViString p1,ViInt32 p2,ViChar _VI_FAR p3[] )
{
   ViStatus errStatus;
   ViInt32 mySize;
   struct hpe1441_globals *thisPtr;
   

   GetGlobals(vi,thisPtr);
   if( p2 < 2 || p2 > 32767)
      		hpe1441_LOG_STATUS(vi,thisPtr,VI_ERROR_PARAMETER2);
   {
      if( (errStatus = viPrintf(vi, "%s\n",p1 )) < VI_SUCCESS)
      		hpe1441_LOG_STATUS(vi,thisPtr,errStatus);
      
      mySize = p2-1;
      
      {
      int sz;
      sz = (int)p2;
      errStatus = viScanf(vi, "%#t", &sz,p3);
      if( thisPtr ) doDelay( thisPtr->myDelay);
      if( errStatus < VI_SUCCESS) hpe1441_LOG_STATUS(vi,thisPtr,errStatus);
      if(sz>=(int)p2) sz=(int)p2-1;
      p3[sz]='\0';
      }
   }

   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
} 

/* 	hpe1441_cmdInt
	Scpi command that takes ONE int16 or int32 command
	
	input:
		vi	- session
		p1	- NULL terminated C string containing scpi command
		p2	- integer parameter
	
*/
ViStatus _VI_FUNC hpe1441_cmdInt( ViSession vi, ViString p1, ViInt32 p2 )
{  
   ViStatus errStatus;
   struct hpe1441_globals *thisPtr;
   GetGlobals(vi,thisPtr);

   if( (errStatus = viPrintf(vi, "%s %ld\n",p1,p2 )) < VI_SUCCESS)
	hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
} 


/*	hpe1441_cmdInt16_Q:
	Scpi command that must respond with a number that can be interpreted as an int16
	
	input:
		vi	- session
		p1	- NULL terminated C string with command
	
	output:
		p2	- response converted to int 16
		
	comment:
		will return 0 if response is not a valid number
*/
 
ViStatus _VI_FUNC hpe1441_cmdInt16_Q( ViSession vi, ViString p1, ViPInt16 p2)
{
   ViStatus errStatus;
   struct hpe1441_globals *thisPtr;
   GetGlobals(vi,thisPtr)

   if( (errStatus = viPrintf(vi, "%s\n",p1)) < VI_SUCCESS)
     	hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   errStatus = viScanf(vi,"%hd%*t",p2);
   if( thisPtr ) doDelay( thisPtr->myDelay);
   if( errStatus < VI_SUCCESS) hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
} 

/* 	hpe1441_cmdReal
	Scpi command that takes ONE Real32 or Real64 param
	
	input:
		vi	- session
		p1	- NULL terminated C string containing scpi command
		p2	- Real parameter
	
*/
ViStatus _VI_FUNC hpe1441_cmdReal( ViSession vi, ViString p1, ViReal64 p2 )
{
   ViStatus errStatus;
   struct hpe1441_globals *thisPtr;

   GetGlobals(vi,thisPtr);

   if( (errStatus = viPrintf(vi, "%s %g\n",p1,p2 )) < VI_SUCCESS)
      	hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );

} 


/*	hpe1441_cmdReal64_Q
	sends scpi command that must elicit a response that can be represented as a real64 (double)
	
	input:
		vi	- session
		p1	- NULL terminated C string containing scpi command
	
	output:
		p2	- Response converted to real 64
		
	comment:
		non numeric response will yeild 0 in p2, in case of underflow underfined.
*/
	
ViStatus _VI_FUNC hpe1441_cmdReal64_Q( ViSession vi, ViString p1, ViPReal64 p2)
{   
   ViStatus errStatus;
   struct hpe1441_globals *thisPtr;
   GetGlobals(vi,thisPtr)

   if( (errStatus = viPrintf(vi, "%s\n",p1)) < VI_SUCCESS)
      	hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   errStatus = viScanf(vi,"%lf%*t",p2);
   if( thisPtr ) doDelay( thisPtr->myDelay);
   if( errStatus < VI_SUCCESS) hpe1441_LOG_STATUS(vi,thisPtr,errStatus);

   hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
} 

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_amConf
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function selects the amplitude modulation state and
 *            AM waveform source, shape, frequency and depth.
 *           
 *            In AM, the amplitude of the carrier is varied by the amplitude
 *           of the modulating waveform. The function generator will accept
 *           an external modulating signal or both an external plus internal
 *           modulating signals.
 *            The shape, and frequency are set, but not output if the source
 *           is EXT.
 *            You cannot use the noise function or dc volts as the carrier
 *           waveform.
 *            For the external modulating source, the modulation depth
 *            is controlled by signal level present on the AM Modulation
 *           terminal.
 *            Only one modulation mode can be enabled at a time.
 *            Attempting to enable AM when another modulation is active will
 *           result in a settings conflict error.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 Sour
 * IN        
 *            Selects the source of the modulating signal, as follows.
 * 
 * PARAM 3 : ViReal64 Dept
 * IN        
 *            Sets the AM modulation depth in percent.
 * 
 *      MAX = hpe1441_AMP_MOD_MAX   120.0
 *      MIN = hpe1441_AMP_MOD_MIN   0.0
 * 
 * PARAM 4 : ViInt16 Func
 * IN        
 *            Selects the shape of the modulating waveform, as follows.
 * 
 * PARAM 5 : ViReal64 Freq
 * IN        
 *            Sets the frequency of the modulating waveform.
 * 
 *      MAX = hpe1441_AMP_MOD_FREQ_MAX   20.0e3
 *      MIN = hpe1441_AMP_MOD_FREQ_MIN   10.0e-3
 * 
 * PARAM 6 : ViInt16 Stat
 * IN        
 *            Disable or enable AM.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_amConf_Sour_a[] = {"BOTH","EXT",0};
static const char * const hpe1441_amConf_Func_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER",0};
static const char * const hpe1441_amConf_Stat_a[] = {"OFF","ON",0};
ViStatus _VI_FUNC hpe1441_amConf(ViSession vi,
  ViInt16 Sour,
  ViReal64 Dept,
  ViInt16 Func,
  ViReal64 Freq,
  ViInt16 Stat)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_amConf" );

    hpe1441_CHK_ENUM(Sour,1,VI_ERROR_PARAMETER2);
    hpe1441_CHK_REAL_RANGE(Dept
                          ,hpe1441_AMP_MOD_MIN
                          ,hpe1441_AMP_MOD_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1441_CHK_ENUM(Func,5,VI_ERROR_PARAMETER4);
    hpe1441_CHK_REAL_RANGE(Freq
                          ,hpe1441_AMP_MOD_FREQ_MIN
                          ,hpe1441_AMP_MOD_FREQ_MAX
                          ,VI_ERROR_PARAMETER5);

    hpe1441_CHK_ENUM(Stat,1,VI_ERROR_PARAMETER6);
    errStatus = viPrintf(vi,"AM:SOUR %s;DEPT %g;INT:FUNC %s;FREQ %g;:AM:STAT %s\n",hpe1441_amConf_Sour_a[Sour],Dept,hpe1441_amConf_Func_a[Func],Freq,hpe1441_amConf_Stat_a[Stat]);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
    }

    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_amConf_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This queries the Amplitude Modulation parameters.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPInt16 pSour
 * OUT       
 *            The AM Source is returned in this parameter.
 * 
 * PARAM 3 : ViPReal64 pDept
 * OUT       
 *            The AM Depth is returned in this parameter.
 * 
 * PARAM 4 : ViPInt16 pFunc
 * OUT       
 *            Returns the shape of the internal modulating waveform.
 * 
 * PARAM 5 : ViPReal64 pFreq
 * OUT       
 *            The Internal Amplitude Modulation Frequency is returned in this
 *           parameter.
 * 
 * PARAM 6 : ViPInt16 pStat
 * OUT       
 *            The AM State is returned in this parameter.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_amConf_Q_pSour_a[] = {"BOTH","EXT",0};
static const char * const hpe1441_amConf_Q_pFunc_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER",0};
ViStatus _VI_FUNC hpe1441_amConf_Q(ViSession vi,
  ViPInt16 pSour,
  ViPReal64 pDept,
  ViPInt16 pFunc,
  ViPReal64 pFreq,
  ViPInt16 pStat)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pSour_str[32];
    char pFunc_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_amConf_Q" );

    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"AM:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_amConf_Q_pSour_a,
            pSour_str, pSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"AM:DEPT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pDept);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"AM:INT:FUNC?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pFunc_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_amConf_Q_pFunc_a,
            pFunc_str, pFunc);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"AM:INT:FREQ?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pFreq);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"AM:STAT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%hd%*t",pStat);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_bmConf
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This selects the burst modulation configuration parameters.
 *           
 *            You can configure the function generator to output a waveform
 *           with a specified number of cycles, called burst. You can output
 *           the burst at a rate determined by the internal rate generator or
 *           an external signal. The function generator can produce a burst
 *           using sine, square, triangle, ramp, and arbitrary waveforms.
 *           
 *            In the external (gated burst) mode,
 *            the output waveform is either "on" or "off" based on the level
 *           of the trigger selected by the trigSour parameter.  (Immediate
 *           and BUS trigger sources will produce the same result as if
 *           External had been selected.) When the external gate source is
 *           selected, the burst count, burst rate, burst phase, and burst
 *           trigger source are not used to generate the output signal (but
 *           they are set by the instrument).
 *            The burst count indicates how many cycles will be output with
 *           each trigger. The trigger source is selected by the trigSour
 *           parameter. The burst count ranges from 1 to 50000 cycles. 
 *           However more than 1 cycle may be required at higher frequencies.
 *            An infinite burst count can be used if the phase lock option is
 *           installed.  Setting the burst count parameter to 0 will send
 *           "INF" to the instrument.
 *            The burst rate defines the interval between bursts when the
 *           internal trigger source has been selected.
 *           
 *            It is possible to specify a burst rate which is too fast for
 *           the function generator to output with the specified carrier
 *           frequency and burst count. If the burst rate is too high, the
 *           function generator will internally adjust it as needed to
 *           continuously re-trigger the burst. The adjustment is handled
 *           internally by the function generator.
 *            Only one modulation mode can be enabled at a time.
 *            Attempting to enable BM when another modulation is active will
 *           result in a settings conflict error.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 Source
 * IN        
 *            Selects the burst modulation source, as follows.
 * 
 * PARAM 3 : ViInt32 Ncycle
 * IN        
 *            Sets the number of cycles to be output per burst.  Numbers from
 *            1 to 50000 are valid.  0 is also valid, and will set "INF" if
 *           the phase lock option is installed.
 * 
 *      MAX = hpe1441_BM_MOD_NCYC_MAX   50000
 *      MIN = hpe1441_BM_MOD_NCYC_MIN   0
 * 
 * PARAM 4 : ViReal64 Phase
 * IN        
 *            Sets the starting phase in degrees for the bursts.
 * 
 *      MAX = hpe1441_BM_MOD_PHAS_MAX   360
 *      MIN = hpe1441_BM_MOD_PHAS_MIN   -360
 * 
 * PARAM 5 : ViReal64 Rate
 * IN        
 *            Sets the burst rate in Hz for internally triggered  bursts.
 * 
 *      MAX = hpe1441_BM_MOD_RATE_MAX   50.0e3
 *      MIN = hpe1441_BM_MOD_RATE_MIN   10.0e-3
 * 
 * PARAM 6 : ViInt16 State
 * IN        
 *            Disable or enable burst modulation.
 * 
 * PARAM 7 : ViInt16 trigSour
 * IN        
 *            Selects the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_bmConf_Source_a[] = {"INT","EXT",0};
static const char * const hpe1441_bmConf_State_a[] = {"OFF","ON",0};
static const char * const hpe1441_bmConf_trigSour_a[] = {"IMM","EXT",
        "BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5","TTLT6",
        "TTLT7",0};
ViStatus _VI_FUNC hpe1441_bmConf(ViSession vi,
  ViInt16 Source,
  ViInt32 Ncycle,
  ViReal64 Phase,
  ViReal64 Rate,
  ViInt16 State,
  ViInt16 trigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_bmConf" );

    hpe1441_CHK_ENUM(Source,1,VI_ERROR_PARAMETER2);
    hpe1441_CHK_LONG_RANGE(Ncycle
                          ,hpe1441_BM_MOD_NCYC_MIN
                          ,hpe1441_BM_MOD_NCYC_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1441_CHK_REAL_RANGE(Phase
                          ,hpe1441_BM_MOD_PHAS_MIN
                          ,hpe1441_BM_MOD_PHAS_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1441_CHK_REAL_RANGE(Rate
                          ,hpe1441_BM_MOD_RATE_MIN
                          ,hpe1441_BM_MOD_RATE_MAX
                          ,VI_ERROR_PARAMETER5);

    hpe1441_CHK_ENUM(State,1,VI_ERROR_PARAMETER6);
    hpe1441_CHK_ENUM(trigSour,10,VI_ERROR_PARAMETER7);
    {
        if(Ncycle)
          errStatus = viPrintf(vi,"BM:SOUR %s;NCYC %ld;PHAS %g;INT:RATE %g;:TRIG:SOUR %s;:BM:STAT %s\n",hpe1441_bmConf_Source_a[Source],Ncycle,Phase,Rate,hpe1441_bmConf_trigSour_a[trigSour],hpe1441_bmConf_State_a[State]);
        else
          errStatus = viPrintf(vi,"BM:SOUR %s;NCYC INF;PHAS %g;INT:RATE %g;:TRIG:SOUR %s;:BM:STAT %s\n",hpe1441_bmConf_Source_a[Source],Phase,Rate,hpe1441_bmConf_trigSour_a[trigSour],hpe1441_bmConf_State_a[State]);
    
    
        if (errStatus < VI_SUCCESS)
        {
           hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_bmConf_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This queries the present burst modulation configuration
 *           parameters.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPInt16 pSour
 * OUT       
 *            The burst modulation source is returned in this parameter.
 * 
 * PARAM 3 : ViPInt32 pNcyc
 * OUT       
 *            The number of cycles output per trigger is returned in this
 *           parameter. A "0" will be returned to indicate an Infinite count.
 * 
 * PARAM 4 : ViPReal64 pPhas
 * OUT       
 *            The Burst Modulation Phase (in degrees) is returned in this
 *           parameter.
 * 
 * PARAM 5 : ViPReal64 pRate
 * OUT       
 *            The internal  burst modulation rate is returned in this
 *           parameter. The value is in Hertz.
 * 
 * PARAM 6 : ViPInt16 pStat
 * OUT       
 *            The Burst Modulation State is returned in this parameter.
 * 
 * PARAM 7 : ViPInt16 pTrigSour
 * OUT       
 *            Returns the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_bmConf_Q_pSour_a[] = {"INT","EXT",0};
static const char * const hpe1441_bmConf_Q_pTrigSour_a[] = {"IMM","EXT",
        "BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5","TTLT6",
        "TTLT7",0};
ViStatus _VI_FUNC hpe1441_bmConf_Q(ViSession vi,
  ViPInt16 pSour,
  ViPInt32 pNcyc,
  ViPReal64 pPhas,
  ViPReal64 pRate,
  ViPInt16 pStat,
  ViPInt16 pTrigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pSour_str[32];
    char pTrigSour_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_bmConf_Q" );

    {
        ViReal64 temp;
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"BM:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_bmConf_Q_pSour_a,
            pSour_str, pSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"BM:NCYC?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",&temp);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if(temp > 100000.0) *pNcyc = 0;
        else *pNcyc = (ViInt32)temp;
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"BM:PHAS?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pPhas);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"BM:INT:RATE?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pRate);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"BM:STAT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%hd%*t",pStat);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"TRIG:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pTrigSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_bmConf_Q_pTrigSour_a,
            pTrigSour_str, pTrigSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_conf
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  The command provides the most straightforward method to program
 *            the main waveform of the function generator.  You can select
 *           the function, frequency, amplitude, offset and output load.
 *           
 *            The amplitude is in the currently set units (which is defaulted
 *           to VPP at power on).  Although this function shows a range of
 *           amplitude values from -23.8 to 22.22, the output amplitude is
 *           limited to the range of 0.05 to 10 Volts Peak-to-Peak into a 50
 *           Ohm load.  The wider range of values given here will enable you
 *           to program the full range in other voltage units and into an
 *           open load termination.  You may program the amplitude in other
 *           units by first using the SCPI Pass Through function:
 *            hpe1441_cmd( vi, "VOLT:UNIT VPP|VRMS|DBM");
 *           
 *            (using just one of the VPP, VRMS or DBM units).
 *           
 *            The function generator has a fixed output impedence of 50 ohms
 *           on the OUTPUT terminal. You can specify whether you are
 *           terminating the output into a 50 ohm load or an open circuit. 
 *           Incorrect impedence matching between the function generator and
 *           your load will result in an amplitude or offset which does not
 *           match the specified signal level.
 *            If you specify a 50 ohm termination but are actually
 *           terminating into an open circuit, the actual output will be
 *           twice the value specified.
 *            To tweak other characteristics of the waveforms you may use the
 *           "Pass-Through" functions provided with this driver.  You will
 *           need to know the SCPI commands for the items that you wish to
 *           change.  For instance, to change the duty cycly of a square wave
 *           to 30% you would use:
 *            errStatus =  hpe1441_cmdInt( vi, "PULS:DCYC", 30);
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 func
 * IN        
 *            The desired function to be output by the function generator.
 *            The allowable settings are shown below.
 * 
 * PARAM 3 : ViReal64 freq
 * IN        
 *            The desired frequency of the function generator.  Note that the
 *            frequency range may be limited by the function that is
 *           selected.
 * 
 *      MAX = hpe1441_FREQ_MAX   15.0e6
 *      MIN = hpe1441_FREQ_MIN   100.0e-6
 * 
 * PARAM 4 : ViReal64 ampl
 * IN        
 *            The desired amplitude of the function generator.  This
 *           amplitude is in the currently set units (which is defaulted to
 *           VPP at power on).
 * 
 *      MAX = hpe1441_AMPL_MAX   22.22
 *      MIN = hpe1441_AMPL_MIN   -23.8
 * 
 * PARAM 5 : ViReal64 offs
 * IN        
 *            The desired offset voltage of the function generator.  Note
 *           that the actual range of offset voltages depends upon the
 *           amplitude setting.
 * 
 *      MAX = hpe1441_OFFS_MAX   10.0
 *      MIN = hpe1441_OFFS_MIN   -10.0
 * 
 * PARAM 6 : ViReal64 outpLoad
 * IN        
 *            Sets the output termination for output amplitude and offset
 *           voltage.
 * 
 *      MAX = hpe1441_OUTPUT_LOAD_INF   9.9E+37
 *      MIN = hpe1441_OUTPUT_LOAD_50   50
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_conf_func_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER","DC",0};
ViStatus _VI_FUNC hpe1441_conf(ViSession vi,
  ViInt16 func,
  ViReal64 freq,
  ViReal64 ampl,
  ViReal64 offs,
  ViReal64 outpLoad)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_conf" );

    hpe1441_CHK_ENUM(func,6,VI_ERROR_PARAMETER2);
    hpe1441_CHK_REAL_RANGE(freq
                          ,hpe1441_FREQ_MIN
                          ,hpe1441_FREQ_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1441_CHK_REAL_RANGE(ampl
                          ,hpe1441_AMPL_MIN
                          ,hpe1441_AMPL_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1441_CHK_REAL_RANGE(offs
                          ,hpe1441_OFFS_MIN
                          ,hpe1441_OFFS_MAX
                          ,VI_ERROR_PARAMETER5);

    hpe1441_CHK_REAL_RANGE(outpLoad
                          ,hpe1441_OUTPUT_LOAD_50
                          ,hpe1441_OUTPUT_LOAD_INF
                          ,VI_ERROR_PARAMETER6);

    errStatus = viPrintf(vi,"OUTP:LOAD %g;:APPL:%s %g,%g,%g\n",outpLoad,hpe1441_conf_func_a[func],freq,ampl,offs);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
    }

    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_conf_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Queries the function, frequency, amplitude, offset and load of
 *            the function generator.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPInt16 pShape
 * OUT       
 *            Shows the shape of the selected arbitrary waveform, as follows
 * 
 * PARAM 3 : ViPReal64 pFreq
 * OUT       
 *            Gives the output frequency.
 * 
 * PARAM 4 : ViPReal64 pVolt
 * OUT       
 *            Gives the present output amplitude in whatever units have been
 *           set by the user.
 * 
 * PARAM 5 : ViPReal64 pOffset
 * OUT       
 *            Gives the present output offset.
 * 
 * PARAM 6 : ViPReal64 pOutpLoad
 * OUT       
 *            The Output Load setting is returned in this parameter.  This
 *           will be 50 or 9.9E+37 Ohms.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_conf_Q_pShape_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER","DC",0};
ViStatus _VI_FUNC hpe1441_conf_Q(ViSession vi,
  ViPInt16 pShape,
  ViPReal64 pFreq,
  ViPReal64 pVolt,
  ViPReal64 pOffset,
  ViPReal64 pOutpLoad)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pShape_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_conf_Q" );

    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FUNC:SHAP?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pShape_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_conf_Q_pShape_a,
            pShape_str, pShape);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FREQ?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pFreq);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"VOLT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pVolt);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"VOLT:OFFS?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pOffset);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"OUTP:LOAD?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pOutpLoad);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_dataDacVolatile
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Download binary integer values between -2047 and +2047 into
 *           volatile memory You can download between 8 and 16,000 points per
 *           waveform as a list of values. The binary range of values
 *           corresponds to the values available using internal 12-bit DAC
 *           codes.
 *            The values -2047 and +2047 correspond to the peak values of the
 *           waveform.
 *            The DATA:DAC VOLATILE command overwrites the previous waveform
 *           in VOLATILE memory.
 *           
 *            There is a known defect in an old version of VISA which will
 *           generate a Format Error on this command.  Upgrading to the
 *           latest version of Agilent VISA should solve that problem.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 NoOfPoints
 * IN        
 *            Number of floating-point values to download from array[].
 * 
 *      MAX = hpe1441_POINTS_MAX   16000
 *      MIN = hpe1441_POINTS_MIN   8
 * 
 * PARAM 3 : ViInt16 _VI_FAR array[]
 * IN        
 *            Integer values between -2047 and +2047 to download into
 *           volatile memory.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1441_dataDacVolatile(ViSession vi,
  ViInt16 NoOfPoints,
  ViInt16 _VI_FAR array[])
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_dataDacVolatile" );

    hpe1441_CHK_INT_RANGE(NoOfPoints
                         ,hpe1441_POINTS_MIN
                         ,hpe1441_POINTS_MAX
                         ,VI_ERROR_PARAMETER2);

    {
    #if 0
      /* the ASCII (slow way) */
      ViInt16 i;
      errStatus = viPrintf(vi,"DATA:DAC VOLATILE");
      if (errStatus < VI_SUCCESS) { hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
      for ( i=0; i < NoOfPoints; i++ )
      {
        errStatus = viPrintf( vi, ",%hd", array[i] );
        if (errStatus < VI_SUCCESS ) {hpe1441_LOG_STATUS(vi, thisPtr, errStatus);}
      }
      errStatus = viPrintf(vi,"\n");
      if (errStatus < VI_SUCCESS) { hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    #else
      /* the Binary (fast way) */
      char cmd[128];
      /* compensate for an NI-VISA defect */
      sprintf(cmd,"DATA:DAC VOLATILE, %%%hdhb\n",NoOfPoints);
      errStatus = viPrintf(vi,cmd,array);
      /* errStatus = viPrintf(vi,"DATA:DAC VOLATILE, %*hb\n",NoOfPoints,array); */
      if (errStatus < VI_SUCCESS) { hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    #endif
    
    }
    		
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_dataVolatile
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Download floating-point values between -1 and +1 into volatile
 *           memory You can download between 8 and 16,000 points per waveform
 *           
 *            The values -1 and +1 correspond to the peak values of the
 *           waveform.
 *            Downloading the floating-point values is slower than
 *           downloading binary values but is more convenient when using the
 *           trignometric functions which returns values between -1 and +1.
 *           
 *            The DATA VOLATILE command overwrites the previous waveform in
 *           VOLATILE memory.
 *           
 *            The following statement shows how to use the DATA VOLATILE
 *           command to download nine points to volatile memory:
 *           
 *            "DATA VOLATILE, 1,.75,.5,.25,0,.25,-.5,-.75,-1"
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 NoOfPoints
 * IN        
 *            Number of floating-point values to download from array[].
 * 
 *      MAX = hpe1441_POINTS_MAX   16000
 *      MIN = hpe1441_POINTS_MIN   8
 * 
 * PARAM 3 : ViReal64 _VI_FAR array[]
 * IN        
 *            Floating-point values between -1 or +1 to download into
 *           volatile memory.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1441_dataVolatile(ViSession vi,
  ViInt16 NoOfPoints,
  ViReal64 _VI_FAR array[])
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_dataVolatile" );

    hpe1441_CHK_INT_RANGE(NoOfPoints
                         ,hpe1441_POINTS_MIN
                         ,hpe1441_POINTS_MAX
                         ,VI_ERROR_PARAMETER2);

    {
       ViInt16 i;
    	
       errStatus = viPrintf(vi,"DATA VOLATILE");
       if (errStatus < VI_SUCCESS)
        {
    	hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
        }
        
        for ( i=0; i < NoOfPoints; i++ )
        {
        	errStatus = viPrintf( vi, ",%f", array[i] );
         	if (errStatus < VI_SUCCESS )
         	{
         		hpe1441_LOG_STATUS( vi, thisPtr, errStatus );	
         	}
        }
        	
        errStatus = viPrintf(vi,"\n");
        if (errStatus < VI_SUCCESS)
        {
    	hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
        }
    }	
        
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_fmConf
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This selects the Frequency Modulation parameters and shape and
 *            frequency of the Internal Frequency Modulating waveform.
 *           
 *            In FM, the frequency of the carrier is varied by the amplitude
 *           of the modulating waveform. The function generator will accept
 *           only an internal FM modulating signal--no external source is
 *           available.
 *            You cannot use the noise function or dc volts as the carrier
 *           waveform.
 *            The carrier frequency must always be greater than or equal to
 *           the peak frequency deviation.
 *           
 *            The sum of the carrier frequency and peak frequency deviation
 *           must be less than or equal to the maximum frequency for the
 *           selected function plus 100 kHz.
 *            Only one modulation mode can be enabled at a time.
 *            Attempting to enable FM when another modulation is active will
 *           result in a settings conflict error.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViReal64 Dev
 * IN        
 *            Sets the peak frequency deviation in hertz.  This value
 *           represents the variation in frequency of the modulating waveform
 *           from the carrier frequency.
 * 
 *      MAX = hpe1441_FREQ_MOD_DEV_MAX   7.5e+6
 *      MIN = hpe1441_FREQ_MOD_DEV_MIN   10.0e-3
 * 
 * PARAM 3 : ViInt16 Func
 * IN        
 *            Selects the shape of the modulating waveform, as follows.
 * 
 * PARAM 4 : ViReal64 Freq
 * IN        
 *            Sets the frequency of the frequency modulation waveform.
 * 
 *      MAX = hpe1441_FREQ_MOD_FREQ_MAX   10.0e3
 *      MIN = hpe1441_FREQ_MOD_FREQ_MIN   10.0e-3
 * 
 * PARAM 5 : ViInt16 Stat
 * IN        
 *            Disable or enable FM.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_fmConf_Func_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER",0};
static const char * const hpe1441_fmConf_Stat_a[] = {"OFF","ON",0};
ViStatus _VI_FUNC hpe1441_fmConf(ViSession vi,
  ViReal64 Dev,
  ViInt16 Func,
  ViReal64 Freq,
  ViInt16 Stat)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_fmConf" );

    hpe1441_CHK_REAL_RANGE(Dev
                          ,hpe1441_FREQ_MOD_DEV_MIN
                          ,hpe1441_FREQ_MOD_DEV_MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1441_CHK_ENUM(Func,5,VI_ERROR_PARAMETER3);
    hpe1441_CHK_REAL_RANGE(Freq
                          ,hpe1441_FREQ_MOD_FREQ_MIN
                          ,hpe1441_FREQ_MOD_FREQ_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1441_CHK_ENUM(Stat,1,VI_ERROR_PARAMETER5);
    errStatus = viPrintf(vi,"FM:DEV %g;INT:FUNC %s;FREQ %g;:FM:STAT %s\n",Dev,hpe1441_fmConf_Func_a[Func],Freq,hpe1441_fmConf_Stat_a[Stat]);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
    }

    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_fmConf_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Queries of the Frequency Modulation the parameters.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPReal64 pDev
 * OUT       
 *            The FM Deviation is returned in this parameter.
 * 
 * PARAM 3 : ViPInt16 pFunc
 * OUT       
 *            Returns the frequency modulation source, as follows.
 * 
 * PARAM 4 : ViPReal64 pFreq
 * OUT       
 *            The frequency of the frequency modulation source is returned in
 *           this parameter.
 * 
 * PARAM 5 : ViPInt16 pStat
 * OUT       
 *            The FM State is returned in this parameter.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_fmConf_Q_pFunc_a[] = {"SIN","SQU","TRI",
        "RAMP","NOIS","USER",0};
ViStatus _VI_FUNC hpe1441_fmConf_Q(ViSession vi,
  ViPReal64 pDev,
  ViPInt16 pFunc,
  ViPReal64 pFreq,
  ViPInt16 pStat)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pFunc_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_fmConf_Q" );

    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FM:DEV?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pDev);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FM:INT:FUNC?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pFunc_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_fmConf_Q_pFunc_a,
            pFunc_str, pFunc);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FM:INT:FREQ?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pFreq);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FM:STAT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%hd%*t",pStat);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_freqSweep
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This sets the frequency sweep parameters.
 *           
 *            In the frequency sweep mode, the function generator "steps"
 *           from the start frequency to the stop frequency at a sweep rate
 *           which you specify. You can sweep up or down in frequency, and
 *           with either linear or logarithmic spacing.  To sweep down in
 *           frequency, simply set the stop frequency lower than the start
 *           frequency.  You can configure the function generator to output a
 *           single sweep by applying an external or BUS trigger as set by
 *           the trigger source (trigSour parameter).
 *            The frequencies may be limited by the currently selected
 *           function.
 *            Frequency sweep cannot be used if another modulation mode is
 *           active. Attempting to do so will result in a settings conflict
 *           error.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViReal64 swStart
 * IN        
 *            Sets the sweep start frequency.
 * 
 *      MAX = hpe1441_SWEEP_FREQ__MAX   15.0e6
 *      MIN = hpe1441_SWEEP_FREQ__MIN   10.0e-3
 * 
 * PARAM 3 : ViReal64 swStop
 * IN        
 *            Sets the sweep stop frequency.
 * 
 *      MAX = hpe1441_SWEEP_FREQ_MAX   15.0e6
 *      MIN = hpe1441_SWEEP_FREQ_MIN   10.0e-3
 * 
 * PARAM 4 : ViReal64 Time
 * IN        
 *            Sets the number of seconds for a complete frequency sweep.
 * 
 *      MAX = hpe1441_SWEEP_TIME_MAX   500
 *      MIN = hpe1441_SWEEP_TIME_MIN   1.0e-3
 * 
 * PARAM 5 : ViInt16 Space
 * IN        
 *            Selects the linear or logarithmic spacing for the sweep, as
 *           follows.
 * 
 * PARAM 6 : ViInt16 State
 * IN        
 *            Disable or enable the sweep mode.
 * 
 * PARAM 7 : ViInt16 trigSour
 * IN        
 *            Selects the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_freqSweep_Space_a[] = {"LIN","LOG",0};
static const char * const hpe1441_freqSweep_State_a[] = {"OFF","ON",0};
static const char * const hpe1441_freqSweep_trigSour_a[] = {"IMM","EXT",
        "BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5","TTLT6",
        "TTLT7",0};
ViStatus _VI_FUNC hpe1441_freqSweep(ViSession vi,
  ViReal64 swStart,
  ViReal64 swStop,
  ViReal64 Time,
  ViInt16 Space,
  ViInt16 State,
  ViInt16 trigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_freqSweep" );

    hpe1441_CHK_REAL_RANGE(swStart
                          ,hpe1441_SWEEP_FREQ__MIN
                          ,hpe1441_SWEEP_FREQ__MAX
                          ,VI_ERROR_PARAMETER2);

    hpe1441_CHK_REAL_RANGE(swStop
                          ,hpe1441_SWEEP_FREQ_MIN
                          ,hpe1441_SWEEP_FREQ_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1441_CHK_REAL_RANGE(Time
                          ,hpe1441_SWEEP_TIME_MIN
                          ,hpe1441_SWEEP_TIME_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1441_CHK_ENUM(Space,1,VI_ERROR_PARAMETER5);
    hpe1441_CHK_ENUM(State,1,VI_ERROR_PARAMETER6);
    hpe1441_CHK_ENUM(trigSour,10,VI_ERROR_PARAMETER7);
    errStatus = viPrintf(vi,"FREQ:STAR %g;STOP %g;:SWE:TIME %g;SPAC %s;:TRIG:SOUR %s;:SWE:STAT %s\n",swStart,swStop,Time,hpe1441_freqSweep_Space_a[Space],hpe1441_freqSweep_trigSour_a[trigSour],hpe1441_freqSweep_State_a[State]);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
    }

    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_freqSweep_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This queries the frequency sweep parameters.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPReal64 pStart
 * OUT       
 *            The sweep start frequency is returned in this parameter.
 * 
 * PARAM 3 : ViPReal64 pStop
 * OUT       
 *            The sweep stop frequency is returned in this parameter.
 * 
 * PARAM 4 : ViPReal64 pTime
 * OUT       
 *            The frequency sweep time is returned in this parameter.
 * 
 * PARAM 5 : ViPInt16 pSpace
 * OUT       
 *            The frequency sweep spacing is returned in this parameter.
 * 
 * PARAM 6 : ViPInt16 pState
 * OUT       
 *            The sweep state is returned in this parameter.
 * 
 * PARAM 7 : ViPInt16 pTrigSour
 * OUT       
 *            Returns the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_freqSweep_Q_pSpace_a[] = {"LIN","LOG",0};
static const char * const hpe1441_freqSweep_Q_pTrigSour_a[] = {"IMM",
        "EXT","BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5",
        "TTLT6","TTLT7",0};
ViStatus _VI_FUNC hpe1441_freqSweep_Q(ViSession vi,
  ViPReal64 pStart,
  ViPReal64 pStop,
  ViPReal64 pTime,
  ViPInt16 pSpace,
  ViPInt16 pState,
  ViPInt16 pTrigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pSpace_str[32];
    char pTrigSour_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_freqSweep_Q" );

    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FREQ:STAR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pStart);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FREQ:STOP?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pStop);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SWE:TIME?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pTime);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SWE:SPAC?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pSpace_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_freqSweep_Q_pSpace_a,
            pSpace_str, pSpace);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"SWE:STAT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%hd%*t",pState);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"TRIG:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pTrigSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_freqSweep_Q_pTrigSour_a,
            pTrigSour_str, pTrigSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_fskConf
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This selects the FSK parameters.
 *           
 *            You can configure the function generator to "shift" its output
 *           frequency between two preset values using FSK modulation. The
 *           rate at which the output shifts between the two frequencies
 *           (called the "carrier frequency" and the "hop frequency") is
 *           determined by the internal rate generator or the signal level on
 *           the selected trigger source (trigSour parameter).
 *            When the internal FSK source is selected, the rate at which the
 *            output frequency "shifts" between the carrier frequency and hop
 *            frequency is determined by the FSK rate specified.  The trigger
 *           source (trigSour) will be set but ignored by the output.
 *           
 *            When the external FSK source is selected, the output frequency
 *           is determined by the signal level of the trigger source
 *           (trigSour).  When a "low" TTL level is present, the carrier
 *           frequency is output. When a "high" TTL level is present, the hop
 *           frequency is output.
 *            The hop frequency may be limited by the currently selected
 *           waveform.
 *            Only one modulation mode can be enabled at a time.
 *            Attempting to enable FSK when another modulation is active will
 *           result in a settings conflict error.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViInt16 Source
 * IN        
 *            Selects an internal or external FSK source, as follows.
 * 
 * PARAM 3 : ViReal64 hopFreq
 * IN        
 *            Sets the FSK "hop" frequency.
 * 
 *      MAX = hpe1441_FSK_FREQ_MAX   15.0e6
 *      MIN = hpe1441_FSK_FREQ_MIN   10.0e-3
 * 
 * PARAM 4 : ViReal64 Rate
 * IN        
 *            Sets the rate at which the output frequency "shifts" when the
 *           FSK source is set to internal.
 * 
 *      MAX = hpe1441_FSK_RATE_MAX   50.0e3
 *      MIN = hpe1441_FSK_RATE_MIN   10.0e-3
 * 
 * PARAM 5 : ViInt16 State
 * IN        
 *            Disable or enable FSK modulation.
 * 
 * PARAM 6 : ViInt16 trigSour
 * IN        
 *            Selects the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_fskConf_Source_a[] = {"INT","EXT",0};
static const char * const hpe1441_fskConf_State_a[] = {"OFF","ON",0};
static const char * const hpe1441_fskConf_trigSour_a[] = {"IMM","EXT",
        "BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5","TTLT6",
        "TTLT7",0};
ViStatus _VI_FUNC hpe1441_fskConf(ViSession vi,
  ViInt16 Source,
  ViReal64 hopFreq,
  ViReal64 Rate,
  ViInt16 State,
  ViInt16 trigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_fskConf" );

    hpe1441_CHK_ENUM(Source,1,VI_ERROR_PARAMETER2);
    hpe1441_CHK_REAL_RANGE(hopFreq
                          ,hpe1441_FSK_FREQ_MIN
                          ,hpe1441_FSK_FREQ_MAX
                          ,VI_ERROR_PARAMETER3);

    hpe1441_CHK_REAL_RANGE(Rate
                          ,hpe1441_FSK_RATE_MIN
                          ,hpe1441_FSK_RATE_MAX
                          ,VI_ERROR_PARAMETER4);

    hpe1441_CHK_ENUM(State,1,VI_ERROR_PARAMETER5);
    hpe1441_CHK_ENUM(trigSour,10,VI_ERROR_PARAMETER6);
    errStatus = viPrintf(vi,"FSK:SOUR %s;FREQ %g;INT:RATE %g;:TRIG:SOUR %s;:FSK:STAT %s\n",hpe1441_fskConf_Source_a[Source],hopFreq,Rate,hpe1441_fskConf_trigSour_a[trigSour],hpe1441_fskConf_State_a[State]);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, thisPtr, errStatus );
    }

    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1441_fskConf_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This queries the FSK parameters.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            The handle to the instrument
 * 
 * PARAM 2 : ViPInt16 pSour
 * OUT       
 *            The FSK Source is returned in this parameter.
 * 
 * PARAM 3 : ViPReal64 pHopFreq
 * OUT       
 *            The FSK Hop Frequency (in Hz) is returned in this parameter.
 * 
 * PARAM 4 : ViPReal64 pRate
 * OUT       
 *            The FSK Internal Rate is returned in this parameter.
 * 
 * PARAM 5 : ViPInt16 pStat
 * OUT       
 *            The FSK State is returned in this parameter.
 * 
 * PARAM 6 : ViPInt16 pTrigSour
 * OUT       
 *            Returns the trigger source, as follows.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1441_error_message"
 * 
 *-----------------------------------------------------------------------------
 */
static const char * const hpe1441_fskConf_Q_pSour_a[] = {"INT","EXT",0};
static const char * const hpe1441_fskConf_Q_pTrigSour_a[] = {"IMM","EXT",
        "BUS","TTLT0","TTLT1","TTLT2","TTLT3","TTLT4","TTLT5","TTLT6",
        "TTLT7",0};
ViStatus _VI_FUNC hpe1441_fskConf_Q(ViSession vi,
  ViPInt16 pSour,
  ViPReal64 pHopFreq,
  ViPReal64 pRate,
  ViPInt16 pStat,
  ViPInt16 pTrigSour)
{
    ViStatus errStatus = 0;
    struct hpe1441_globals *thisPtr;
    char pSour_str[32];
    char pTrigSour_str[32];

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1441_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1441_DEBUG_CHK_THIS( vi, thisPtr );
    hpe1441_CDE_INIT( "hpe1441_fskConf_Q" );

    {
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FSK:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_fskConf_Q_pSour_a,
            pSour_str, pSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FSK:FREQ?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pHopFreq);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FSK:INT:RATE?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%lg%*t",pRate);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"FSK:STAT?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%hd%*t",pStat);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        thisPtr->blockSrqIO = VI_TRUE;
        errStatus = viPrintf(vi,"TRIG:SOUR?\n");
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = viScanf(vi,"%s%*t",pTrigSour_str);
        if(thisPtr) doDelay(thisPtr->myDelay);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    
        errStatus = hpe1441_findIndex(thisPtr,hpe1441_fskConf_Q_pTrigSour_a,
            pTrigSour_str, pTrigSour);
        if (VI_SUCCESS > errStatus){ hpe1441_LOG_STATUS( vi, thisPtr, errStatus ); }
    }
    hpe1441_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
