
/* Agilent E1459 VXI Plug and Play driver source */
/* Version A.01.05 */

#define INSTR_CALLBACKS		/* for hpe1459_Handler */

#ifdef __hpux
#define HPUX_SOURCE
#endif

#include <stdlib.h>		/* prototype for malloc() */
#include <string.h>		/* prototype for strcpy() */
#include <stdio.h>		/* prototype for sprintf() */
#include <stdarg.h>		/* variable are list for tracing */
#include "visa.h"


/* Foward declaration */
struct hpe1459_globals;

/*Just use ONE token to find out if we are on windows 32 or not */
#if defined _WIN32 || defined __WIN32__
#ifndef WIN32
#define WIN32
#endif
#endif

#ifdef WIN32
#include <windows.h>
#endif

#include "hpe1459.h"

/* define an int64 */
#ifdef WIN32
#ifdef __BORLANDC__
#define	VINT64					int64
#else
#define VINT64					__int64
#endif
#define CARD_LOCK_TYPE			long
#define HANDLER_TYPE			hpe1459_Handler
#define CRITICAL_SECTION_TYPE	CRITICAL_SECTION
#endif /* WIN32 */
#ifdef __hpux
/* ??? */
#endif

/* Global states for the instrument lock */
#define _UNLOCK_	0
#define _LOCK_		1

#define _NOT_CLEARING_	0
#define _CLEARING_	1

/* 	define interface values */

/*	if this is a message based card then uncomment the following */
/*
#define f_INTERFACE	1
*/
/* if this is a register based card uncomment the following */

#define p_INTERFACE	(thisPtr->iointerface == 0)
#define f_INTERFACE	(thisPtr->iointerface == 1)

#define WAIT_OPC {ViInt16 opc=0,i; for(i=0; i<1000 && !opc; i++) if( viQueryf(vi,"*OPC?\n","%hd",&opc) < VI_SUCCESS) break ; }


#ifndef f_INTERFACE
#error f_INTERFACE MUST be defined!
#endif

/* PROTO  */
ViStatus init_vxi(ViRsrc InstrDesc, ViBoolean id_query, ViBoolean id_reset,ViSession defRM,ViSession *vi, struct hpe1459_globals * thisPtr);
ViStatus init_1406(ViRsrc InstrDesc, ViBoolean id_query, ViBoolean id_reset,ViSession defRM,ViSession *vi, struct hpe1459_globals * thisPtr);

ViStatus util_card_lock( struct hpe1459_globals *thisPtr,
										unsigned long tmo);

ViStatus util_card_unlock( struct hpe1459_globals *thisPtr);

ViStatus util_device_clear( struct hpe1459_globals *thisPtr , long tmo);

void util_wait( long );

void close_interrupts(ViSession vi, struct hpe1459_globals * thisPtr);

ViStatus setEnablePorts( ViSession vi, ViUInt16 EnabledEvents);


#ifdef INSTR_CALLBACKS
ViStatus setup_handler(ViSession vi, struct hpe1459_globals * thisPtr);
ViStatus _VI_FUNCH visa_handler( ViSession vi,
			ViEventType eType,
			ViEvent ctx,
			ViAddr userH );

ViStatus ClearISR(ViSession vi, struct hpe1459_globals *thisPtr);
#endif

ViStatus BankSelect( ViSession, short  );

ViStatus init_regs(ViSession );

ViBoolean util_marktime( unsigned VINT64 *start,unsigned VINT64 usec );

/* selects the appropriate regester address for a given port */
#define REGISTER_ADDRESS(port,reg) ( (port&1) ? reg+16 : reg  )
#define CHAN_TO_PORT( chan ) ( (chan < 16) ? 0 : (chan < 32 ) ? 1 : (chan < 48) ? 2 : 3 )

#ifdef WIN32
#define hpe1459_DECLARE_TIME_LOCALS 	SYSTEMTIME st;
#define hpe1459_GET_TIME		GetLocalTime(&st);
#define hpe1459_TIME_FIELDS	st.wMonth, st.wDay, st.wHour, \
				st.wMinute, st.wSecond,	st.wMilliseconds
#else				/* not win32 */
#include <time.h>		/* standard time functions */
#ifdef __hpux
#define hpe1459_DECLARE_TIME_LOCALS 	struct timeval tv; \
				struct timezone tz; \
				struct tm *tmp;
#define hpe1459_GET_TIME		gettimeofday(&tv, &tz); \
				tmp = localtime((time_t*)&tv.tv_sec);
#define hpe1459_TIME_FIELDS	tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, \
				tmp->tm_min, tmp->tm_sec, tv.tv_usec/1000
#else				/* not unix, use ANSI time function */
#define hpe1459_DECLARE_TIME_LOCALS 	struct tm *tmp; time_t seconds;
#define hpe1459_GET_TIME		time(&seconds); \
				tmp = localtime(&seconds);
#define hpe1459_TIME_FIELDS	tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, \
				tmp->tm_min, tmp->tm_sec, 0
#endif  /* ifdef __unix  */
#endif  /* ifdef WIN32 */

/* Be careful using _E.  It is primarily intended for your
interface functions.  In interface functions do NOT call log_status as
the calling function will do that for you.
*/
#define _E(x) \
{\
 errStatus = (x);\
 if( errStatus < VI_SUCCESS) return errStatus;\
}

/* _LS WILL call log_status */
#define _LS( s )\
{\
errStatus = (s );\
if( errStatus < VI_SUCCESS)\
   hpe1459_LOG_STATUS( vi, thisPtr, errStatus)\
}


#define hpe1459_MODEL_CODE   340
#define hpe1459_MANF_ID      0xFFF
#define hpe1459_IDN_STRING "HEWLETT-PACKARD,E1459"

#define hpe1459_REV_CODE "A.01.04"  /* Driver Revision */

#define hpe1459_ERR_MSG_LENGTH 256  /* size of error message buffer */

/* actual win32 global data for the lock functions */
#ifdef WIN32
static BOOL performance_clock_exists;
static VINT64 clock_frequency=0;
static short performanceclock_set_up=0;
#endif
#ifdef __hpux
/* Don't know if unix needs global clock vars or not */
#endif

/*===============================================================
 *
 *  All messages are stored in this area to aid in localization
 *
 *===============================================================
 */


#define hpe1459_MSG_VI_OPEN_ERR 				\
	"vi was zero.  Was the hpe1459_init() successful?"

#define hpe1459_MSG_IN_FUNCTION					\
	"in function"
	/* hpe1459_error_message() */

#define hpe1459_MSG_INVALID_STATUS					\
  	"Parameter 2 is invalid"				\
	"in function hpe1459_error_message()."
	/* hpe1459_error_message() */

#define hpe1459_MSG_INVALID_STATUS_VALUE				\
	"is not a valid viStatus value."
	/* hpe1459_error_message() */

#define  hpe1459_MSG_INVALID_VI					\
  	"Parameter 1 is invalid"				\
	" in function hpe1459_error_message()"			\
	".  Using an inactive ViSession may cause this error."	\
	"  Was the instrument driver closed prematurely?"
	/* hpe1459_message_query() */

#define hpe1459_MSG_NO_ERRORS					\
	"No Errors"
	/* hpe1459_error_message() */

#define hpe1459_MSG_SELF_TEST_FAILED 				\
	"Self test failed."
	/* hpe1459_self_test() */

#define hpe1459_MSG_SELF_TEST_PASSED 				\
	"Self test passed."
	/* hpe1459_self_test() */

#define hpe1459_MSG_ERROR_NOT_LOCK_OWNER					\
	"Caller does not own the instrument lock."
	/* util_card_unlock */

#define hpe1459_MSG_ERROR_DAV_SET 	   \
	"Clock source cannot be INTernal when DAV interrupt enabled."
    /* hpe1459_input */


/* the following messages are used by the functions to check parameters */

#define hpe1459_MSG_BOOLEAN   "Expected 0 or 1; Got %d"
#define hpe1459_MSG_REAL   "Expected %lg to %lg; Got %lg"
#define hpe1459_MSG_INT   "Expected %hd to %hd; Got %hd"
#define hpe1459_MSG_LONG   "Expected %ld to %ld; Got %ld"

#define hpe1459_MSG_LOOKUP "Error converting string response to integer"
#define hpe1459_MSG_NO_MATCH "Could not match string %s"

/*
 * static error message
 */

#define VI_ERROR_PARAMETER1_MSG	"Parameter 1 is invalid"
#define VI_ERROR_PARAMETER2_MSG	"Parameter 2 is invalid"
#define VI_ERROR_PARAMETER3_MSG	"Parameter 3 is invalid"
#define VI_ERROR_PARAMETER4_MSG	"Parameter 4 is invalid"
#define VI_ERROR_PARAMETER5_MSG	"Parameter 5 is invalid"
#define VI_ERROR_PARAMETER6_MSG	"Parameter 6 is invalid"
#define VI_ERROR_PARAMETER7_MSG	"Parameter 7 is invalid"
#define VI_ERROR_PARAMETER8_MSG	"Parameter 8 is invalid"
#define VI_ERROR_PARAMETER9_MSG	"Parameter 9 is invalid"
#define VI_ERROR_PARAMETER10_MSG "Parameter 10 is invalid"
#define VI_ERROR_PARAMETER11_MSG "Parameter 11 is invalid"
#define VI_ERROR_PARAMETER12_MSG "Parameter 12 is invalid"
#define VI_ERROR_PARAMETER13_MSG "Parameter 13 is invalid"
#define VI_ERROR_PARAMETER14_MSG "Parameter 14 is invalid"
#define VI_ERROR_PARAMETER15_MSG "Parameter 15 is invalid"
#define VI_ERROR_PARAMETER16_MSG "Parameter 16 is invalid"
#define VI_ERROR_PARAMETER17_MSG "Parameter 17 is invalid"
#define VI_ERROR_PARAMETER18_MSG "Parameter 18 is invalid"

#define VI_ERROR_FAIL_ID_QUERY_MSG "Instrument IDN does not match."

#define INSTR_ERROR_INV_SESSION_MSG \
	"ViSession (parmeter 1) was not created by this driver"

#define INSTR_ERROR_NULL_PTR_MSG "NULL pointer detected"

#define INSTR_ERROR_RESET_FAILED_MSG "reset failed"

#define INSTR_ERROR_UNEXPECTED_MSG "An unexpected error occurred"

#define INSTR_ERROR_DETECTED_MSG \
	"Instrument Error Detected, call hpe1459_error_query()."

#define INSTR_ERROR_LOOKUP_MSG  "String not found in table"

/*================================================================*/

/* don't check the debug pointer all the time!*/
#ifdef DEBUG
#define hpe1459_DEBUG_CHK_THIS( vi, thisPtr) 			\
	/* check for NULL user data */				\
	if( 0 == thisPtr)					\
	{							\
 		hpe1459_LOG_STATUS(                             	\
		  vi, NULL, hpe1459_INSTR_ERROR_INV_SESSION );	\
	}							\
	{							\
		ViSession defRM;				\
								\
		/* This should never fail */			\
		errStatus = viGetAttribute( vi,                 \
			VI_ATTR_RM_SESSION, &defRM);		\
		if( VI_SUCCESS > errStatus )			\
		{						\
 			hpe1459_LOG_STATUS(				\
			  vi, NULL, hpe1459_INSTR_ERROR_UNEXPECTED );	\
		}						\
		if( defRM != thisPtr->defRMSession)		\
		{						\
 			hpe1459_LOG_STATUS(				\
			  vi, NULL, hpe1459_INSTR_ERROR_INV_SESSION );	\
		}						\
	}
#else
#define hpe1459_DEBUG_CHK_THIS( vi, thisPtr)
#endif

#define hpe1459_CDE_INIT( funcname)  				\
	strcpy(thisPtr->errFuncName, funcname);			\
	thisPtr->errNumber = VI_SUCCESS;			\
	thisPtr->errMessage[0] = 0;

#define hpe1459_CDE_MESSAGE( message ) 	 			\
	strcpy(thisPtr->errMessage, message)

/* Note /Zp should be 8 bytes for compilation of these drivers */
struct hpe1459_globals
{
	/* card_loc & ISR_count MUST be first to be aligned on
	32 bit boundardy */
	CARD_LOCK_TYPE		card_lock;
#ifdef INSTR_CALLBACKS
	long		ISR_count;
#endif

	ViSession			defRMSession;
	volatile char		*a24Address;
	int					deref;
	ViStatus			errNumber;
	char 				errFuncName[40];
	char				errMessage[160];
	char 				address[256];

	HANDLE				hEventSource;
	HANDLE				owner_thread;
	int	lock_count;
	ViBoolean			owner_valid;

	CRITICAL_SECTION	device_clear_cs	;
	long		lock_timeout;
	short		iointerface;
	ViBoolean	VISAdisabled;
	ViBoolean	HDWIdisabled;
	short		device_clear;

	ViInt16		traceLevel;
	ViInt16		traceDest;
	void (_VI_FUNCH _VI_PTR traceCallback)(ViAddr, const char*);
	ViAddr		traceHandle;
	ViBoolean	interrupts; /* 0 for GPIB-VXI */

#ifdef INSTR_CALLBACKS
	ViBoolean			eventArray[ NUM_EVENTS ];
	HANDLER_TYPE		eventHndlr[ NUM_EVENTS ];
	ViInt32	eventAttrib[NUM_EVENTS ];
	hpe1459_Handler	ErrorHandler;
#endif

	/* to alert visa_handler that a self test is in progress */
	ViBoolean		self_test_in_progress;
	ViUInt16		EnabledEvents;
	ViUInt32		diag_event;
};


/*-----------------------------------------------------------------------------
 * FUNC    : hpe1459_printTrace
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :   Print log to appropriate destination.
 *             Called by hpe1459trace and hpe1459_statusUpdate
 * PARAM 1 : thisPtr
 * IN
 *           state record pointer
 *
 * PARAM 2 : const char *s
 * IN
 *           Message to be printed
 *
 *-----------------------------------------------------------------------------
 */
static void _VI_FUNC hpe1459_printTrace(const struct hpe1459_globals *thisPtr,
				     const char *s)
{
#ifdef WIN32
	if (thisPtr->traceDest == hpe1459_TRAC_STDERR)
	{
		fprintf(stderr, "%s", s);
	}
	else if (thisPtr->traceDest == hpe1459_TRAC_EVENTLOG)
	{
		const char *array = s;

		ReportEvent(thisPtr->hEventSource,
			    EVENTLOG_INFORMATION_TYPE,
			    0, 0x40000001, NULL,
			    1, 0, &array, NULL);
	}
	else if (thisPtr->traceDest == hpe1459_TRAC_DEBUGSTRING)
	{
		OutputDebugString(s);
	}
	else if (thisPtr->traceDest == hpe1459_TRAC_CALLBACK)
	{
		/* call the trace callback function if it was set */
		if (thisPtr->traceCallback){
			(thisPtr->traceCallback)(thisPtr->traceHandle, s);
		}
		else
		{
			OutputDebugString("hpe1459" "No callback function defined");
		}
	}
#endif
#ifdef __hpux
	if (thisPtr->traceDest == hpe1459_TRAC_STDERR)
	{
		fputs(s, stderr);
	}
#endif
}

/*-----------------------------------------------------------------------------
 * Macro aud FUNC    : hpe1459_TRACE,  hpe1459_trace
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :   Logs function calls if tracing is enabled
 * PARAM 1 : thisPtr
 * IN
 *           state record pointer
 *
 * PARAM 2 : const char *nameViBoolean name
 * IN
 *	     name of function called
 *
 * PARAM 3 : const char *fmt
 * IN
 *           Format for printing the log message
 *
 * PARAM 4... : input parameters
 *
 *-----------------------------------------------------------------------------
 */
#define hpe1459_TRACE if (thisPtr->traceLevel==hpe1459_TRAC_ALL) hpe1459_trace

static void hpe1459_trace(const struct hpe1459_globals *thisPtr, const char *name,
		       const char *fmt, ...)
{
	hpe1459_DECLARE_TIME_LOCALS
	char *p;
	char logmsg[500];
	va_list ap;

	hpe1459_GET_TIME
	sprintf(logmsg, "%02d%02d%02d%02d%02d.%03d, %s(%s",
		hpe1459_TIME_FIELDS,
		name, thisPtr->address);
	p = logmsg + strlen(logmsg);
	va_start(ap, fmt);
	vsprintf(p, fmt, ap);
	va_end(ap);
	strcat(p, ")\n");
	hpe1459_printTrace(thisPtr, logmsg);
}

#ifdef INSTR_CALLBACKS

#define hpe1459_LOG_STATUS( vi, tptr, status ) \
{ ViStatus e;\
e = hpe1459_statusUpdate( vi, (struct hpe1459_globals *)tptr, status);\
if( tptr && allow_unlock ) util_card_unlock( (struct hpe1459_globals *)tptr ); \
if( e < VI_SUCCESS && tptr && ((struct hpe1459_globals *)tptr)->eventArray[hpe1459_EVENT_ERROR] )\
((struct hpe1459_globals *)tptr)->eventHndlr[hpe1459_EVENT_ERROR](vi,((struct hpe1459_globals *)tptr)->eventAttrib[hpe1459_EVENT_ERROR]);\
return e;\
}

#else

#define hpe1459_LOG_STATUS( vi, tptr, status ) \
{ ViStatus e;\
e = hpe1459_statusUpdate( vi, (struct hpe1459_globals *)tptr, status);\
if( tptr && allow_unlock ) util_card_unlock( (struct hpe1459_globals *)tptr ); \
return e;\
}
#endif

ViStatus hpe1459_statusUpdate(ViSession vi,struct hpe1459_globals *thisPtr,
			   ViStatus s)
{
	if (thisPtr)
	{
		thisPtr->errNumber = s;
		if (s && thisPtr->traceLevel)
		{
			char message[256];
			hpe1459_DECLARE_TIME_LOCALS
			char logmsg[500];

			hpe1459_error_message(vi, s, message);

			hpe1459_GET_TIME
			sprintf(logmsg,
				"%02d%02d%02d%02d%02d.%03d, %s(%s): %s\n",
				hpe1459_TIME_FIELDS,
				thisPtr->errFuncName,
				thisPtr->address,
				thisPtr->errMessage);
			hpe1459_printTrace(thisPtr, logmsg);
		}
	}

	return s;
}

/*
 * Error Message Structures
 */

struct instrErrStruct
{
	ViStatus errStatus;
	ViString errMessage;
};

const static struct instrErrStruct instrErrMsgTable[] =
{
        { VI_ERROR_PARAMETER1,	VI_ERROR_PARAMETER1_MSG },
        { VI_ERROR_PARAMETER2,	VI_ERROR_PARAMETER2_MSG },
        { VI_ERROR_PARAMETER3,	VI_ERROR_PARAMETER3_MSG },
        { VI_ERROR_PARAMETER4,	VI_ERROR_PARAMETER4_MSG },
        { VI_ERROR_PARAMETER5,	VI_ERROR_PARAMETER5_MSG },
        { VI_ERROR_PARAMETER6,	VI_ERROR_PARAMETER6_MSG },
        { VI_ERROR_PARAMETER7,	VI_ERROR_PARAMETER7_MSG },
        { VI_ERROR_PARAMETER8,	VI_ERROR_PARAMETER8_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER9,	VI_ERROR_PARAMETER9_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER10,	VI_ERROR_PARAMETER10_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER11,	VI_ERROR_PARAMETER11_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER12,	VI_ERROR_PARAMETER12_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER13,	VI_ERROR_PARAMETER13_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER14,	VI_ERROR_PARAMETER14_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER15,	VI_ERROR_PARAMETER15_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER16,	VI_ERROR_PARAMETER16_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER17,	VI_ERROR_PARAMETER17_MSG },
        { hpe1459_INSTR_ERROR_PARAMETER18,	VI_ERROR_PARAMETER18_MSG },
        { VI_ERROR_FAIL_ID_QUERY,	VI_ERROR_FAIL_ID_QUERY_MSG },

	{ hpe1459_INSTR_ERROR_INV_SESSION,	INSTR_ERROR_INV_SESSION_MSG },
        { hpe1459_INSTR_ERROR_NULL_PTR,	INSTR_ERROR_NULL_PTR_MSG },
        { hpe1459_INSTR_ERROR_RESET_FAILED,INSTR_ERROR_RESET_FAILED_MSG },
        { hpe1459_INSTR_ERROR_UNEXPECTED,	INSTR_ERROR_UNEXPECTED_MSG },
	{ hpe1459_INSTR_ERROR_DETECTED,	INSTR_ERROR_DETECTED_MSG },
	{ hpe1459_INSTR_ERROR_LOOKUP,	INSTR_ERROR_LOOKUP_MSG },
        { hpe1459_ERROR_NOT_LOCK_OWNER,	hpe1459_MSG_ERROR_NOT_LOCK_OWNER},
		{hpe1459_ERROR_DAV_SET ,hpe1459_MSG_ERROR_DAV_SET},
};

/* macros for testing parameters */
#define hpe1459_CHK_BOOLEAN( my_val, err ) if( hpe1459_chk_boolean( thisPtr, my_val) ) hpe1459_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1459_chk_boolean(
  struct hpe1459_globals *thisPtr,
  ViBoolean my_val)
{
   char message[hpe1459_ERR_MSG_LENGTH];
   if( (my_val != VI_TRUE) && (my_val != VI_FALSE) )
   {
      /* true = parameter is invalid */
      sprintf(message, hpe1459_MSG_BOOLEAN, my_val);
      hpe1459_CDE_MESSAGE(message);
      /* true = parameter is invalid */
      return VI_TRUE;
   }

   /* false = okay */
   return VI_FALSE;
}


#define hpe1459_CHK_REAL_RANGE( my_val, min, max, err ) if( hpe1459_chk_real_range( thisPtr, my_val, min, max) ) hpe1459_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1459_chk_real_range(
  struct hpe1459_globals *thisPtr,
  ViReal64 my_val,
  ViReal64 min,
  ViReal64 max)
{
   char message[hpe1459_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )
   {
      sprintf(message, hpe1459_MSG_REAL, min, max, my_val);
      hpe1459_CDE_MESSAGE(message);
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
}

#define hpe1459_CHK_INT_RANGE( my_val, min, max, err ) if( hpe1459_chk_int_range( thisPtr, my_val, min, max) ) hpe1459_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1459_chk_int_range(
  struct hpe1459_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 min,
  ViInt16 max)
{
   char message[hpe1459_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )
   {
      sprintf(message, hpe1459_MSG_INT, min, max, my_val);
      hpe1459_CDE_MESSAGE(message);
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
}


#define hpe1459_CHK_LONG_RANGE( my_val, min, max, err ) if( hpe1459_chk_long_range( thisPtr, my_val, min, max) ) hpe1459_LOG_STATUS( vi, thisPtr, err);

static ViBoolean hpe1459_chk_long_range(
  struct hpe1459_globals *thisPtr,
  ViInt32 my_val,
  ViInt32 min,
  ViInt32 max)
{
   char message[hpe1459_ERR_MSG_LENGTH];

   if ( ( my_val < min ) || (my_val > max) )
   {
      sprintf(message, hpe1459_MSG_LONG, min, max, my_val);
      hpe1459_CDE_MESSAGE(message);
      /* true = parameter is invalid */
      return VI_TRUE;
   }
   return VI_FALSE;
}


#define hpe1459_CHK_ENUM( my_val, limit, err ) if( hpe1459_chk_enum( thisPtr, my_val, limit) ) hpe1459_LOG_STATUS( vi, thisPtr, err);

/* utility routine which searches for a string in an array of strings. */
/* This is used by the CHK_ENUM macro */
static ViBoolean hpe1459_chk_enum (
  struct hpe1459_globals *thisPtr,
  ViInt16 my_val,
  ViInt16 limit)
{
    char message[hpe1459_ERR_MSG_LENGTH];

    if ( ( my_val < 0 ) || (my_val > limit) )
    {
        sprintf(message, hpe1459_MSG_INT, 0, limit, my_val);
        hpe1459_CDE_MESSAGE(message);
        /* true = parameter is invalid */
        return VI_TRUE;
    }

    return VI_FALSE;
}



/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_close
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :  The close function terminates the software connection to the
 *          instrument and deallocates system resources.  It is generally a
 *          good programming habit to close the instrument handle when the
 *          program is done using the instrument.
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_close(ViSession vi)
{
	struct hpe1459_globals *thisPtr;
	ViStatus errStatus;
	ViSession defRM;
	unsigned VINT64 stime=0;
	ViAttr tout=0;
	short allow_unlock = 0;

	/* free memory */
    thisPtr = VI_NULL;
	errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if( errStatus < VI_SUCCESS) thisPtr = NULL;

    hpe1459_TRACE(thisPtr, "hpe1459_close","");

	/* Try and shut down interrupts now.  No new ones will come in but
	   'old' ones can still be in the sceduler
	*/
	close_interrupts(vi,thisPtr);
	/* Try to clear the scheduler.  The assumption here is that a new time slice
	   while it may not FINISH an ISR will cause the ISRcount to be set.
	*/

	Sleep(0); /* let the ISR run */

	viGetAttribute(vi,VI_ATTR_TMO_VALUE,&tout);

	util_marktime(&stime,tout*1000);

	while( thisPtr && thisPtr->ISR_count)
	{
		if( util_card_lock(thisPtr,1000) < VI_SUCCESS)
		{
			/* if this times out we have a driver error and we are toast.
			   (driver writer should always clear accesses that last around
			   a second or more ).  We CANNOT CLOSE as the thisPtr is IN
			   USE in another thread and to close is an instant UAE.  We
			   simply return the timeout and hope the user can do someting
			   about it and then close again.
			*/
			allow_unlock=1;
			_LS(util_device_clear( thisPtr,1000 ) )
		}

		/* We have to clear the ISR.  Thats because the driver code will
		   use the thisPtr meaning the thisPtr IS in use by another thread
		   even if the user does NOT  reference the instrument.  The
		   user cannot run an ISR thread forever AND expect us to be able
		   to close the instrument.
		*/
		util_card_unlock(thisPtr); /* allows ISR to keep running */
		Sleep(0);
		/* just timeout if the ISR doesn't finish */
		if( util_marktime(&stime, tout*1000) ) hpe1459_LOG_STATUS(vi,thisPtr,VI_ERROR_TMO);
    }

	if( thisPtr ) util_card_lock(thisPtr,0);

    /* at this point we have the card locked.  Just go ahead and
           kill it.
    */
	viSetAttribute(vi,VI_ATTR_USER_DATA,0);

	/* retrieve Resource Management session */
	errStatus = viGetAttribute( vi,	VI_ATTR_RM_SESSION, &defRM);
    /* be careful do NOT reset errStatus as its used below */


	/* ISR's are already dead from above code */
#ifdef WIN32
	VirtualFree(thisPtr,0,MEM_RELEASE);
#endif
#ifdef __hpux
	free( thisPtr);
#endif
	if( errStatus >= VI_SUCCESS)
	{
		viClose( defRM);
        return errStatus;
	}
	else
	{
		return errStatus;
	}
}



/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_reset
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :  The reset function places the instrument in a default state.
 *
 *            Reset will affect the following state:
 *
 *            Overlap on
 *
 *            Open all relays
 *
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_reset(ViSession vi)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock = 0;
	int i;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1459_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
    allow_unlock=1;
    if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

    hpe1459_CDE_INIT( "hpe1459_reset" );

    hpe1459_TRACE(thisPtr, "hpe1459_reset","");

	viDiscardEvents(vi,VI_ALL_ENABLED_EVENTS,VI_ALL_MECH);
	viDisableEvent(vi,(f_INTERFACE) ? VI_EVENT_SERVICE_REQ :
				VI_EVENT_VXI_SIGP,VI_HNDLR) ;
	thisPtr->VISAdisabled = VI_TRUE;
	thisPtr->HDWIdisabled = VI_TRUE;

	util_card_unlock(thisPtr );

	_LS(ClearISR(vi,thisPtr) )

    errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
    if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

	for(i=1; i<NUM_EVENTS; i++)
	{
		thisPtr->eventArray[i] = VI_FALSE;
	}
    /* if this is a register card then the following is correct.  If not
       then remove the 'if ...' below as a Message based card will
       take the reset command as is.
    */
    if( f_INTERFACE )
	{
		errStatus = viPrintf(vi,"*RST\n") ;
		WAIT_OPC
		hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
   	}


	_LS( init_regs(vi )	)

    hpe1459_LOG_STATUS( vi,thisPtr,errStatus);
}


/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_self_test
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :  The self-test function causes the instrument to perform a
 *           self-test and returns the result of that self-test.  This is
 *           used to verify that an instrument is operating properly.  A
 *           failure may indicate a potential hardware problem.
 *
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * PARAM 2 : ViPInt16 test_result
 * OUT
 *            Numeric result from self-test operation.
 *
 *            0 = no error ( test passed)
 *            1 = bad status readback
 *            2 = bad config readback
 *            3 = bad pattern readback
 *            4 = interrupt fail
 *
 * PARAM 3 : ViChar _VI_FAR test_message[]
 * OUT
 *            Self-test status message.  This is limited to 256 characters.
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_self_test(ViSession vi,
  ViPInt16 test_result,
  ViChar _VI_FAR test_message[])
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock = 0;
	ViInt16 tmp,i;
    ViInt16 sleepytime;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1459_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
    allow_unlock=1;
    if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

    hpe1459_CDE_INIT( "hpe1459_self_test" );

    hpe1459_TRACE(thisPtr, "hpe1459_self_test","");

	_LS( viDisableEvent(vi,VI_ALL_ENABLED_EVENTS,VI_HNDLR) );

	util_card_unlock(thisPtr );

	_LS(ClearISR(vi,thisPtr) )

    errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
    if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

	for(i=1; i<NUM_EVENTS; i++)
	{
		thisPtr->eventArray[i] = VI_FALSE;
	}

    *test_result = 0;
    strcpy(test_message, hpe1459_MSG_SELF_TEST_PASSED);

	if( f_INTERFACE )
	{
		errStatus = viQueryf(vi,"*TST?\n","%hd",test_result) ;
		if( errStatus >= VI_SUCCESS && *test_result)
		   strcpy(test_message,hpe1459_MSG_SELF_TEST_FAILED );
		hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
   	}

	thisPtr->VISAdisabled=VI_TRUE;
	thisPtr->HDWIdisabled=VI_TRUE;
   	_LS( init_regs(vi) )

	_LS( viIn16(vi,VI_A16_SPACE,0,&tmp) )
	if( tmp & 0xfff != 0xfff) /* manf id wrong */
	{
	  strcpy(test_message,hpe1459_MSG_SELF_TEST_FAILED );
	  *test_result = 1;
	  hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
  	}

	_LS( viIn16(vi,VI_A16_SPACE,2,&tmp) )
	if( tmp & 0xfff != 0x154) /* model wrong */
	{
	  strcpy(test_message,hpe1459_MSG_SELF_TEST_FAILED );
	  *test_result = 2;
	  hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
  	}

	if( !thisPtr->interrupts ) /* essentially GPIB-VXI = no interrupts*/
		      hpe1459_LOG_STATUS( vi, thisPtr, errStatus );

	/* if self test done while handlers enabled interrupts can be lost */

	for(i=0; i< 4; i++ ) /* go through each port */
	{
	  ViUInt16 sr;
	  thisPtr->self_test_in_progress = VI_TRUE;
	  _LS(viEnableEvent(vi,(f_INTERFACE) ? VI_EVENT_SERVICE_REQ : VI_EVENT_VXI_SIGP,
		VI_HNDLR, VI_NULL) )
	  _LS( init_regs(vi) )

	  _LS( BankSelect(vi,(short)i) ) /* select bits in status reg */

	  _LS(viIn16(vi,VI_A16_SPACE,Status_reg, &sr ))

	  sr |= 0x40; /* enable DAV interrupts */

      _LS( viOut16(vi, VI_A16_SPACE, Status_reg,sr) )

	  /* send an internal clock interrupt.  Will send gobs of em*/
	  _LS( viOut16(vi, VI_A16_SPACE, REGISTER_ADDRESS(i,Cmd_reg),0xfffc) )

          sleepytime=0;

          while( ( thisPtr->self_test_in_progress ) && sleepytime<100) {
	    Sleep(1);
	    Sleep(0) ; /* wait to collect at least one */
            sleepytime++;
          }

	  if( thisPtr->self_test_in_progress ) /* didn't get it */
	  {
		 thisPtr->self_test_in_progress=VI_FALSE;
		 viDisableEvent(vi, VI_ALL_ENABLED_EVENTS, VI_ALL_MECH);
		 init_regs( vi );
		 *test_result = 200 + i;
	 	 strcpy(test_message,hpe1459_MSG_SELF_TEST_FAILED );
		 hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
	  }

	}

	/* after self test the card is reset and handlers are turned off */
	_LS( viDisableEvent(vi,VI_ALL_ENABLED_EVENTS,VI_HNDLR) );
	errStatus = VI_SUCCESS; /* visa warn about already disable, ignore */
	init_regs(vi);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}


/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_error_query
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :  Returns error message from instrument.  Is not
 *			  supported unless using Agilent E1406 command module
 *			  and will return warning in that case.
 *
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * PARAM 2 : ViPInt32 error_number
 * OUT
 *            Always 0
 *
 * PARAM 3 : ViChar _VI_FAR error_message[]
 * OUT
 *            Always an empty string.
 *
 * RETURN  :  Always VI_WARN_NSUP_ERROR_QUERY.
 *
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_error_query(ViSession vi,
  ViPInt32 error_number,
  ViChar _VI_FAR error_message[])
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock = 0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (errStatus < VI_SUCCESS)
    {
       hpe1459_LOG_STATUS( vi, 0, errStatus );
    }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
    allow_unlock=1;
    if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

    hpe1459_CDE_INIT( "hpe1459_error_query" );

    hpe1459_TRACE(thisPtr, "hpe1459_error_query","");

    *error_number = 0;
    *error_message = '\0';

    if( f_INTERFACE )
	{
		errStatus=viQueryf( vi, "SYST:ERR?\n","%ld,%t",
			 error_number, error_message) ;

		error_message[strlen(error_message)-1] = 0;

		hpe1459_LOG_STATUS( vi, thisPtr, errStatus);

   	}


    hpe1459_LOG_STATUS( vi, thisPtr, VI_WARN_NSUP_ERROR_QUERY );
}


/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_error_message
 *-----------------------------------------------------------------------------
 *
 * PURPOSE : The error message function translates the error return value
 *  from an instrument driver function to a user-readable string.
 *
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * PARAM 2 : ViStatus error_number
 * IN
 *            The error return value from an instrument driver function
 *
 * PARAM 3 : ViChar _VI_FAR [] message
 * OUT
 *            Error message string.  This is limited to 256 characters.
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */


ViStatus _VI_FUNC hpe1459_error_message(ViSession vi,
			     ViStatus error_number,ViChar _VI_FAR message[])
{
	struct hpe1459_globals *thisPtr;
	ViStatus errStatus;
	ViInt32 idx;
        short allow_unlock = 0;

	/* initialize output parameters */
	message[0] = 0;

	/* try to find a thisPtr */
	if (vi)
	{
		errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA,
					   (ViAddr) &thisPtr);
		if( VI_SUCCESS > errStatus)
		{
			hpe1459_LOG_STATUS( vi, NULL, errStatus);
		}
		hpe1459_DEBUG_CHK_THIS( vi, thisPtr);
	        errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
                allow_unlock= 1;
                if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);
	}
	else
		thisPtr = NULL;

	if( VI_SUCCESS == error_number)
	{
		sprintf( message, hpe1459_MSG_NO_ERRORS);
		hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* return the static error message */
	for(idx=0;
	    idx < (sizeof instrErrMsgTable / sizeof( struct instrErrStruct));
	    idx++)
	{
		/* check for a matching error number */
		if( instrErrMsgTable[idx].errStatus == error_number)
		{
			if( thisPtr && (thisPtr->errNumber == error_number))
			{
				/* context dependent error
				 * message is available.
				 */
				sprintf( message,
				  "%s " hpe1459_MSG_IN_FUNCTION " %s() %s",
		          	  instrErrMsgTable[idx].errMessage,
				  thisPtr->errFuncName,
				  thisPtr->errMessage);
			}
			else
			{
				/* No context dependent eror
				 * message available so copy
				 * the static error message
				 */
				strcpy( message,
				       instrErrMsgTable[idx].errMessage);

			}

			hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
		}
	}


	/* try VTL/VISA to see if it exists there
	 *
	 * At this point vi is either VALID or VI_NULL
	 */
	errStatus = viStatusDesc(vi, error_number, message);
	if( VI_SUCCESS == errStatus)
	{
		/* check for a context dependent error message */
		if( thisPtr && (thisPtr->errNumber == error_number))
		{
			/* context dependent error
			 * message is available.
			 */
			strcat( message, " ");
			strcat( message, hpe1459_MSG_IN_FUNCTION);
			strcat( message, " ");
			strcat( message, thisPtr->errFuncName);
			strcat( message, "() ");
			strcat( message, thisPtr->errMessage);
		}

		/* VTL found an error message, so return success */
		hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* if we have a VI_NULL, then we need to return a error message */
	if( VI_NULL == vi)
	{
		strcpy(message, hpe1459_MSG_VI_OPEN_ERR);
		hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS);
	}

	/* user passed in a invalid status */
	sprintf( message,
	  hpe1459_MSG_INVALID_STATUS
	  "  %ld"
	  hpe1459_MSG_INVALID_STATUS_VALUE
	  , (long)error_number );

	hpe1459_LOG_STATUS( vi, thisPtr, VI_ERROR_PARAMETER2);
}


/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_revision_query
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :  The revision function returns the revision of the instrument
 *             driver and the firmware of the instrument being used.
 * PARAM 1 : ViSession vi
 * IN
 *            Instrument Handle returned from hpe1459_init().
 *
 * PARAM 2 : ViChar _VI_FAR driver_rev[]
 * OUT
 *            Instrument driver revision.  This is limited to 256 characters.
 *
 * PARAM 3 : ViChar _VI_FAR instr_rev[]
 * OUT
 *            Instrument firmware revision.  This is limited to 256 characters.
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_revision_query(ViSession vi,
 ViChar _VI_FAR driver_rev[],
 ViChar _VI_FAR instr_rev[])
{
	struct hpe1459_globals *thisPtr;
	ViStatus errStatus=VI_SUCCESS;
	ViStatus cStatus = VI_SUCCESS;
	char temp_str[256];
	char * last_comma;
        short allow_unlock = 0;

	/* initialize output parameters */
        driver_rev[0] = 0;
	instr_rev[0] = 0;

	errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
	if( VI_SUCCESS > errStatus)
	{
		hpe1459_LOG_STATUS( vi, NULL, errStatus);
	}
	hpe1459_DEBUG_CHK_THIS( vi, thisPtr);

        errStatus=util_card_lock( thisPtr,thisPtr->lock_timeout);
        allow_unlock = 1;
        if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS(vi,0,errStatus);

	hpe1459_TRACE(thisPtr, "hpe1459_revision_query","");

	sprintf(driver_rev, "%s", hpe1459_REV_CODE);

	if( f_INTERFACE )
	{

		 errStatus=viQueryf( vi, "*IDN?\n","%t",temp_str);
		 if( errStatus < VI_SUCCESS)
		 {
        		driver_rev[0] = 0;
			instr_rev[0] = 0;

			hpe1459_LOG_STATUS( vi, thisPtr, errStatus);
		 }

        	last_comma = strrchr(temp_str,',');
		/* error and exit if last comma not found */
        	if (!last_comma)
        	{
            		instr_rev[0] = 0;
			hpe1459_CDE_MESSAGE("no last comma found in IDN string" );
			hpe1459_LOG_STATUS( vi, thisPtr, hpe1459_INSTR_ERROR_UNEXPECTED);
        	}

			strcpy(instr_rev,"SCPI: ");
        	strcat(instr_rev, last_comma+1);

	    	hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS);

	}


	sprintf(instr_rev, "None");

	hpe1459_LOG_STATUS( vi, thisPtr, errStatus);
}


/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_init
 *-----------------------------------------------------------------------------
 *
 * PURPOSE :   The initialize function initializes the software connection to
 *           the instrument and optionally verifies that instrument is in
 *           the system.  In addition, it may perform any necessary actions
 *           to place the instrument in its reset state.
 * PARAM 1 : ViRsrc InstrDesc
 * IN
 *           The Instrument Description.  Example: VXI::5.
 *
 * PARAM 2 : ViBoolean id_query
 * IN
 *
 *           if( VI_TRUE)  Verify model code.
 *           if(VI_FALSE)  Do not verify model code.
 *
 * PARAM 2 : ViBoolean do_reset
 * IN
 *           if( VI_TRUE) Perform Reset Operation.
 *           if(VI_FALSE) Do not perform Reset operation
 *
 * PARAM 3 : ViPSession vi
 * OUT
 *           Instrument Handle. This is VI_NULL if an error occurred
 *            during the init.
 *
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 *
 *-----------------------------------------------------------------------------
 */

ViStatus _VI_FUNC hpe1459_init(ViRsrc InstrDesc,ViBoolean id_query,ViBoolean do_reset,
ViPSession vi)
{
	struct hpe1459_globals *thisPtr=NULL;
	ViSession defRM=0;
	char gpibdesc[64];
	ViUInt16 num, primary, secondary;
	ViSession vi1406;
	char idn_buf[256];
	ViStatus errStatus = VI_SUCCESS;
	ViStatus cStatus = VI_SUCCESS ; /* keeps track of warnings */
	ViUInt16 intf;
	short allow_unlock = 0;

	*vi = VI_NULL;

   	/* Find the Default Resource Manager */
	_E(viOpenDefaultRM( &defRM))

	/* Open a session to the instrument */
	errStatus = viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi);
        if( errStatus < VI_SUCCESS)
	{
		viClose( defRM);
		*vi = VI_NULL;
		return errStatus;
	}

	/* get memory for instance specific globals.  MUST be a 32 bit address */
#ifdef WIN32
	thisPtr = (struct hpe1459_globals*) VirtualAlloc(NULL,sizeof(struct hpe1459_globals),MEM_RESERVE | MEM_COMMIT | MEM_TOP_DOWN,PAGE_READWRITE );
#endif
#ifdef __hpux
	thisPtr = (struct hpe1459_globals*) calloc(sizeof(struct hpe1459_globals),0);
#endif
	if( 0 == thisPtr)
	{
		viClose( defRM);	/* also closes vi session */
		*vi = VI_NULL;
		return VI_ERROR_ALLOC;
	}

	/* zero the whole block */
	memset(thisPtr,0,sizeof(struct hpe1459_globals));

	/* go ahead and set up some of the global area since if
	   successful 'thisPtr' will be the one assigned
	*/
    thisPtr->errNumber = VI_SUCCESS;
	strcpy(thisPtr->address, InstrDesc);

#ifdef WIN32
	if( !performanceclock_set_up )
		{
			performance_clock_exists =
				QueryPerformanceFrequency( (LARGE_INTEGER *)&clock_frequency );
			performanceclock_set_up	= 1 ;
		}
#endif
#ifdef __hpux
/* ??? */
#endif
	errStatus=viGetAttribute( *vi, VI_ATTR_INTF_TYPE, &intf);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}


	if( intf == VI_INTF_VXI)
		{

			/* embeddeds MXI etc will go here */
			/* init_vxi will re-open */
		    viClose( *vi );
		    errStatus = init_vxi(InstrDesc,id_query,do_reset,
							defRM,vi,thisPtr) ;
   		   hpe1459_LOG_STATUS(*vi, NULL, errStatus );
		}

	/* Start dowload case */

	/* check for 1406 internal driver by reading *IDN? of 0 and
		*IDN? of secondary for this address*/

	errStatus=viGetAttribute(*vi, VI_ATTR_INTF_NUM,&num);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}

	errStatus=viGetAttribute(*vi, VI_ATTR_GPIB_PRIMARY_ADDR, &primary);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}

	/* this assumes this must be a GPIB interface */
	sprintf(gpibdesc, "GPIB%d::%d::0", (int)num,(int)primary);
	errStatus=viOpen( defRM, gpibdesc, VI_NULL, VI_NULL,&vi1406);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}

	/* try sending an IDN? to 1406 */
	errStatus = viSetAttribute(vi1406, VI_ATTR_TMO_VALUE, 500);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}

        idn_buf[0]=0;
	viQueryf(vi1406, "*IDN?\n","%t",idn_buf);

	if (memcmp(idn_buf, "HEWLETT-PACKARD,E140", 20))
		{
			/* not a 1406.  This device can be opened ....
			   but its performance in peek/poke mode is
			   likely to leave something to be desired ....
			   (boy is that the under statement of the century!)
			*/
		  	viClose(vi1406);
			viClose( *vi ); /* init_vxi will re open */

		        errStatus=init_vxi(InstrDesc,id_query,do_reset, defRM,vi,thisPtr);

			 return errStatus;
		}

	viClose(vi1406);

	/* ok 1406.  Now is there a down load driver installed?? */
	errStatus=viGetAttribute(*vi, VI_ATTR_GPIB_SECONDARY_ADDR, &secondary);
	if( errStatus < VI_SUCCESS)
	{
		viClose(defRM);
		*vi = VI_NULL;
		return errStatus;
	}
	sprintf(gpibdesc, "GPIB%d::%d::%d", (int)num, (int)primary, (int)secondary);
	viClose(*vi);
	errStatus=viOpen( defRM, gpibdesc, VI_NULL, VI_NULL, vi);
	if( errStatus < VI_SUCCESS )
	{
		       /* No download driver ... */
			errStatus=init_vxi(InstrDesc,id_query,do_reset, defRM,
						 vi,thisPtr) ;

   			hpe1459_LOG_STATUS(*vi, NULL, errStatus );
	}

  /* GPIB opened.  Must be 1406 with a downloaded driver */

   viClose( *vi );  /* init_1406 will open its own session */
   errStatus = init_1406(gpibdesc,id_query,do_reset, defRM, vi,thisPtr) ;

   hpe1459_LOG_STATUS(*vi, NULL, errStatus );
}

/*
	init_vxi is the P&P init routine for emmbededs and MXI etc.
*/
ViStatus init_vxi(ViRsrc InstrDesc, ViBoolean id_query, ViBoolean do_reset,ViSession defRM,ViSession *vi, struct hpe1459_globals * thisPtr)
{
	ViUInt16 manfId;
	ViUInt16 modelCode;
	ViStatus errStatus = VI_SUCCESS;
	ViAttr itype;

	/* Open a session to the instrument */
	_E(viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi))

        thisPtr->interrupts = 0;
	thisPtr->iointerface = 0; /* number of register portable interface*/

/*	The following is a smorgasbord of general code that you might
        wish to uncomment or if uneeded simply delete.
*/
        /* clear VISA events, and be sure Queuing can be turned on.
	   we will take it that if SIGP's can be got for Queuing then
	   they can be got for handlers.
	*/

/* 	This code will clear interrupts and set the interrupt flag so
	that this test can be performed easier later in the driver.
*/

	viDiscardEvents(*vi, VI_EVENT_VXI_SIGP ,VI_ALL_MECH);

	itype = VI_INTF_GPIB_VXI;
	viGetAttribute(*vi,VI_ATTR_INTF_TYPE,&itype);
	if( itype != VI_INTF_GPIB_VXI) thisPtr->interrupts = 1;


/* 	checks if this device has a24 memory */
/*
	_E(viGetAttribute(*vi, VI_ATTR_MEM_SPACE, &attribute));

	if (attribute != VI_A24_SPACE) return VI_ERROR_INV_SPACE;
*/


	/* find the VXI manfacturer's ID needed for c_h flag*/
	_E(viGetAttribute( *vi, VI_ATTR_MANF_ID, &manfId));
	manfId &= 0xFFF;


	/* at this point the vi returned is good so associate the globals pointer*/
	_E(viGetAttribute( *vi, VI_ATTR_RM_SESSION, &thisPtr->defRMSession) );

	_E(viSetAttribute(*vi,VI_ATTR_USER_DATA,(ViAttr)thisPtr));

	if( VI_TRUE == id_query )
		{

		        /* find the instrument's model code */
				_E(viGetAttribute( *vi, VI_ATTR_MODEL_CODE,
					   (ViPAttrState)(&modelCode)) );
				modelCode &= 0xFFF;

			/* In general a new reg based driver may have
                           to put special code here to detect if this
                           is the right instrument. */

			if( manfId != hpe1459_MANF_ID ||
                            modelCode != hpe1459_MODEL_CODE )
				{
				 viClose( defRM );
				 *vi = VI_NULL;
				 return  VI_ERROR_FAIL_ID_QUERY;
  				}


		}	 /* if - id_query */

	if( VI_TRUE == do_reset )
		{

				/* call the reset function to reset the instrument */
				_E( hpe1459_reset(*vi) );

		}
	else
		{
				/* copy the state from the card */
		}


	/* uncomment if handler part of driver */
	setup_handler(*vi,thisPtr);


	viSetAttribute(*vi,VI_ATTR_TMO_VALUE,10000);
	return errStatus;
}

/*
	init_1406 is the P&P init when there is a commmand module and
 	a download driver is in that command module.
*/
ViStatus init_1406(ViRsrc InstrDesc, ViBoolean id_query, ViBoolean do_reset,ViSession defRM,ViSession *vi, struct hpe1459_globals * thisPtr)
{
	ViStatus errStatus=VI_SUCCESS;

	char idn_buf[256];

	_E(viOpen( defRM, InstrDesc, VI_NULL, VI_NULL, vi));

    thisPtr->interrupts = 1;

	/* OK to turn on read flush globally for this driver
	   (means all viScanf's and viQueryf's will flush out
	   the read buffer)
	*/

	_E(viSetAttribute(*vi,VI_ATTR_RD_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS))

	/* interface value for 1406 downloaded driver */
	thisPtr->iointerface = 1;

	/* at this point the vi returned is good so associate the globals pointer*/
	_E(viGetAttribute( *vi, VI_ATTR_RM_SESSION, &thisPtr->defRMSession) );

	_E(viSetAttribute(*vi,VI_ATTR_USER_DATA,(ViAttr)thisPtr) );

        /* uncomment if handlers are part of the driver*/
	/*thisPtr->interrupts = 1;*/

	if(id_query)
	{
	  _E(viQueryf(*vi,"*IDN?\n","%t",idn_buf));

	  if( memcmp( idn_buf,hpe1459_IDN_STRING,21 ) )
	  {
		viClose( defRM );
	 	*vi = VI_NULL;
	   	return VI_ERROR_FAIL_ID_QUERY;
	  }
	}

	if( do_reset)
	{
		_E( hpe1459_reset( *vi ) );
   	}

	setup_handler(*vi,thisPtr);

	viSetAttribute(*vi,VI_ATTR_TMO_VALUE,10000);

	return errStatus;

}

ViStatus _VI_FUNC hpe1459_timeOut(ViSession vi, ViInt32 tmo)
{
   ViStatus errStatus = VI_SUCCESS;
   short allow_unlock=0;

   struct hpe1459_globals *thisPtr=0 ;

   errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);

   if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS( vi, thisPtr, errStatus );

   hpe1459_CDE_INIT( "hpe1459_timeOut" );

   hpe1459_TRACE(thisPtr, "hpe1459_timeOut","");

   hpe1459_CHK_LONG_RANGE( tmo, 0, 2147483647, VI_ERROR_PARAMETER2 );

   errStatus = viSetAttribute(vi, VI_ATTR_TMO_VALUE, tmo);

   hpe1459_LOG_STATUS( vi, thisPtr, errStatus );


}

ViStatus _VI_FUNC hpe1459_timeOut_Q(ViSession vi, ViPInt32 tmo)
{
   ViStatus errStatus = VI_SUCCESS;
   short allow_unlock=0;
   struct hpe1459_globals *thisPtr=0 ;

   errStatus = viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);

   if( errStatus < VI_SUCCESS) hpe1459_LOG_STATUS( vi, thisPtr, errStatus );

   hpe1459_CDE_INIT( "hpe1459_timeOut_Q" );

   hpe1459_TRACE(thisPtr, "hpe1459_timeOut_Q","");

   errStatus = viGetAttribute(vi, VI_ATTR_TMO_VALUE, tmo);

   hpe1459_LOG_STATUS( vi, thisPtr, errStatus );


}


#ifdef INSTR_CALLBACKS

ViStatus setup_handler(ViSession vi, struct hpe1459_globals * thisPtr)
{
	/* We always set up the handler for interrupts
	   even if there aren't any.
	*/
	ViStatus errStatus = VI_SUCCESS;
	ViUInt32 intf;
	ViInt16 tmp,i;

	_E( viGetAttribute(vi,VI_ATTR_INTF_TYPE,&intf))

	/* VISA does not support interrupts on GPIB-VXI */
	/* from now on thisPtr->Interrupts is valid */
	if( intf == VI_INTF_GPIB_VXI || !thisPtr->interrupts )
	{
		thisPtr->interrupts = VI_FALSE;
		return errStatus;
	}

	/* go ahead and install the global handler but do
	   not enable the event, there is no need to actually
	   recieve events unless the user installs a handler.
	*/

	_E( viDisableEvent(vi,
		(f_INTERFACE) ? VI_EVENT_SERVICE_REQ
					  : VI_EVENT_VXI_SIGP,VI_HNDLR) );

	_E( viInstallHandler(vi,
		(f_INTERFACE) ? VI_EVENT_SERVICE_REQ
					  : VI_EVENT_VXI_SIGP,visa_handler,0));
	thisPtr->VISAdisabled=VI_TRUE;
	thisPtr->HDWIdisabled = VI_TRUE;
	if( p_INTERFACE)
	{
	/* set hardware state for driver function */
	/* disable DAV and EDGE detection on all ports */
		_E( viOut16(vi,VI_A16_SPACE,Status_reg,0) ) /*disable interrupts*/
		for(i=0; i<4;i++)
		{
			_E(BankSelect(vi,i) )

			_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),&tmp))
			tmp &= 2;
			_E( viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),tmp ) )
		}
		/* all that reamins is to set I/D bits in Status_reg to fire interrupts*/
 	}

	if( f_INTERFACE)
	{


		_E(viPrintf(vi,"*SRE 0\n")) /* shut off interrupts*/
		{
			ViInt16 sre,i;

			sre=1;
			for(i=0; (i<1000) && (sre != 0); i++)
				if( viQueryf(vi,"*SRE?\n","%hd",&sre) < VI_SUCCESS)break;
		}
		WAIT_OPC
		_E(viPrintf(vi,"SENS:EVEN:PORT0:DAV:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT1:DAV:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT2:DAV:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT3:DAV:ENABLE OFF\n"))
		WAIT_OPC
		_E(viPrintf(vi,"SENS:EVEN:PORT0:EDGE:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT1:EDGE:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT2:EDGE:ENABLE OFF\n"))
		_E(viPrintf(vi,"SENS:EVEN:PORT3:EDGE:ENABLE OFF\n"))
		WAIT_OPC

		/* enable psumary of all bits */
		_E(viPrintf(vi,"STAT:OPER:PSUM:ENAB %d\n",255))
		/* enable the summary bit to be passed along */
		_E(viPrintf(vi,"STAT:OPER:ENABLE 512\n"))

		/*all that remains to 'fire' interrupts is *SRE 128*/
		WAIT_OPC

	}
	return errStatus;

}

ViStatus _VI_FUNCH visa_handler( ViSession vi,
			ViEventType eType,
			ViEvent ctx,
			ViAddr userH )
{
	ViStatus errStatus = VI_SUCCESS;
	struct hpe1459_globals *thisPtr ;
	int i;
	ViUInt16 dav;
	ViUInt16 edge_event;

	_E(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr)) ;

	/* get access to the instrument */

	if( !thisPtr->interrupts )	return errStatus;

	if( thisPtr->self_test_in_progress )
	{
		 /* self test is a special case.  We don't need to lock the card
		    as the self test code won't touch the card while we are
			touching it.   If the users doing multi thread well he has
			to sync for himself.
		 */
	   	init_regs( vi ); /* reset this thing and stop interrupts */
		viDiscardEvents(vi,VI_ALL_ENABLED_EVENTS,VI_ALL_MECH);
		viDisableEvent(vi,(f_INTERFACE) ? VI_EVENT_SERVICE_REQ :
				VI_EVENT_VXI_SIGP,VI_HNDLR) ;
		thisPtr->self_test_in_progress = VI_FALSE; /* handshake */
		return VI_SUCCESS;
	}

    /* this code is to assure that a blocked ISR does not
       come in AFTER a disable interrupt occurs.  Essentailly this
       is very difficult to guarentee in VISA but what we do
       is have disable give up a time slice or two.  The win32
       scheduler SHOULD run to this point and alert the main
       thread that an ISR is running.  Be sure that code you add
	   to this routine decrements ISR_count before returning.
    */
	InterlockedIncrement(&thisPtr->ISR_count);

	util_card_lock(thisPtr, 0 ) ; /* wait indefinatly */

	if( f_INTERFACE )
	{
		/*  for download case we need to get the status reg
		   	and be sure its pattern match
		*/
		ViUInt16 event,cond;

		if(viQueryf(vi,"STAT:OPER:EVENT?\n","%hd",&event)<VI_SUCCESS)
			thisPtr->diag_event |= 0x80000000;

		/* doing this here is a 'poor mans' way of synching
		between this and _(un)installHandler.  It too will
		lock so if we own it the user will NOT be
		able to zero out thisPtr->handler UNTIL this
		run of the handler is complete.  However if the users
		call wins the race then the above lock won't happen
		until AFTER uninstallHandler exits in which case
		thisPtr->handler will be zero so the check below is
		essential.
		*/

		/* must check all the handlers for your device example:


        errStatus = viQueryf(vi,"Some SCPI command that retrieves the
			events from the hardware",&e)
        DRIVER WRITER: you must transform the event into one of the
		event enumerated values here.  The following is skeleton code
		you must flesh out.	 Num_events is the number of events found,
		remember you'v read the conditon reg so you MUST handle all
		events, any left will be lost.
		*/

		if( viQueryf(vi,"STAT:OPER:PSUM:COND?\n","%hd",&cond)<VI_SUCCESS)
			thisPtr->diag_event |= 0x40000000;

		edge_event=0;
		if(viQueryf(vi,"STAT:OPER:PSUM:EVENT?\n","%hd",&edge_event)<VI_SUCCESS)
			thisPtr->diag_event |= 0x20000000;

		thisPtr->diag_event |= edge_event;
		dav = edge_event & 0xf; /*PSUM gets all*/
		edge_event = edge_event >> 4;
		util_card_unlock( thisPtr );
		if( edge_event )
		{
			if( thisPtr->eventArray[ hpe1459_EVENT_EDGE])
			{
			/* single large edge event */


			if(thisPtr->eventHndlr[hpe1459_EVENT_EDGE] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_EDGE])
							(vi,thisPtr->eventAttrib[hpe1459_EVENT_EDGE]);
			}
			else
			for(i=0; i<4; i++)
			{
			if( (edge_event & (1<<i)) &&
				thisPtr->eventArray[ hpe1459_EVENT_EDGE_0+i])
				{
				if(thisPtr->eventHndlr[hpe1459_EVENT_EDGE_0 + i] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_EDGE_0 + i])
					(vi,thisPtr->eventAttrib[hpe1459_EVENT_EDGE_0 + i]);
		   		}


			}
		}
		if( dav )
		{
			if( thisPtr->eventArray[ hpe1459_EVENT_DAV])
			{
			/* single large edge event */

			if(thisPtr->eventHndlr[hpe1459_EVENT_DAV] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_DAV])
							(vi,thisPtr->eventAttrib[hpe1459_EVENT_DAV]);
			}
			else
			for(i=0; i<4; i++)
			{
			if( (dav & (1<<i)) &&
				thisPtr->eventArray[ hpe1459_EVENT_DAV_0+i])
				{
				if(thisPtr->eventHndlr[hpe1459_EVENT_DAV_0 + i] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_DAV_0 + i])
					(vi,thisPtr->eventAttrib[hpe1459_EVENT_DAV_0 + i]);
		   		}


			}
		}

	}

	if( p_INTERFACE )
	{

		/* shut off the hardware interrupts */
		viOut16(vi,VI_A16_SPACE,Status_reg,0xff8e);

		/* edge events */
		edge_event=0;
		viIn16(vi,VI_A16_SPACE,EdgeIntr_reg,&edge_event);
		util_card_unlock( thisPtr );
		edge_event &= 0xf;
		if( edge_event  )
		{
			if( thisPtr->eventArray[ hpe1459_EVENT_EDGE])
			{
			/* single large edge event */
			if(thisPtr->eventHndlr[hpe1459_EVENT_EDGE] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_EDGE])
							(vi,thisPtr->eventAttrib[hpe1459_EVENT_EDGE]);
			}
			else
			for(i=0; i<4; i++)
			{
			if( (edge_event & (1<<i)) &&
				thisPtr->eventArray[ hpe1459_EVENT_EDGE_0+i])
				{
				if(thisPtr->eventHndlr[hpe1459_EVENT_EDGE_0 + i] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_EDGE_0 + i])
					(vi,thisPtr->eventAttrib[hpe1459_EVENT_EDGE_0 + i]);
		   		}


			}
		}
		dav = 0;
		viIn16(vi,VI_A16_SPACE,DAVStatus_reg,&dav);
		dav &= 0xf;
		if( dav  )
		{
			if( thisPtr->eventArray[ hpe1459_EVENT_DAV])
			{
			/* single large edge event */

			if(thisPtr->eventHndlr[hpe1459_EVENT_DAV] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_DAV])
							(vi,thisPtr->eventAttrib[hpe1459_EVENT_DAV]);
			}
			else
			for(i=0; i<4; i++)
			{
			if( (dav & (1<<i)) &&
				thisPtr->eventArray[ hpe1459_EVENT_DAV_0+i])
				{
				if(thisPtr->eventHndlr[hpe1459_EVENT_DAV_0 + i] )
				  (thisPtr->eventHndlr[hpe1459_EVENT_DAV_0 + i])
					(vi,thisPtr->eventAttrib[hpe1459_EVENT_DAV_0 + i]);
		   		}


			}
		}

	}

	viClose(ctx);
    InterlockedDecrement(&thisPtr->ISR_count);
	if( p_INTERFACE )
		viOut16(vi,VI_A16_SPACE,Status_reg,0xffee); /*restore interrupts */
	return errStatus;
}

#endif /* INSTR_CALLBACKS */


void close_interrupts(ViSession vi, struct hpe1459_globals * thisPtr)
{
 	/* can be called in all cases it does no harm */
	viDisableEvent(vi,VI_ALL_ENABLED_EVENTS,VI_ALL_MECH);

	viDiscardEvents(vi,VI_ALL_ENABLED_EVENTS,VI_ALL_MECH);

	viUninstallHandler(vi,VI_ALL_ENABLED_EVENTS,VI_ANY_HNDLR,
			thisPtr);

}

#ifdef WIN32

ViStatus util_card_lock( struct hpe1459_globals *thisPtr,
										unsigned long tmo)
{

  unsigned VINT64 start_time;

  /* avoid deadlock on internal driver calls */
  if( thisPtr->owner_valid  && GetCurrentThreadId() == (DWORD)thisPtr->owner_thread )
     {
      /* this is just another use of the lock by the owner thread
         be sure to ONLY release when ALL uses have expired i.e.
         keep a count.
      */
      thisPtr->lock_count++;
      return VI_SUCCESS;
     }

  if( tmo )
     {

        if( !performance_clock_exists )
          {
           /* Warning GetTickCount is supported everywhere but it
              will roll over every 49.7 days making time look possibly
              negative .  I don't think this code can ever be
			  executed.
           */
           start_time = GetTickCount();
	      }
        else
          {
           /* in this case we know that the high frequency counter is
              supported.  That should be true in all cases and hence
              the warning above is more for completness than anything
              else.
              Note we will compute the time just in case the 8253 chip
              is changed but it is a known fact that
              it generates a time of 838 nano seconds
              per tick and is what QueryPerformanceCounter is based
			  upon.
           */

           QueryPerformanceCounter( (LARGE_INTEGER *)&start_time );

          }
     }

  while(1)
  {
    unsigned VINT64 current_time;

    if( InterlockedExchange( &thisPtr->card_lock, _LOCK_) ==
	_UNLOCK_)
		{
            if( thisPtr->device_clear == _NOT_CLEARING_)
			{
				thisPtr->owner_thread = (HANDLE)GetCurrentThreadId();
				thisPtr->lock_count=1;
				/* when this statment done other threads can
				start examining the lock data.
				*/
				thisPtr->owner_valid = VI_TRUE;
      			return VI_SUCCESS;
			}
			else
			{
				 /* if the device is being cleared don't take the
				    lock, give up our time slice so the thread that
					is trying to clear the device can run.
				 */
                 InterlockedExchange( &thisPtr->card_lock,
							_UNLOCK_);
			}

        }

    if( tmo )
    {
      if( QueryPerformanceCounter( (LARGE_INTEGER *)&current_time )	)
		{
          VINT64 xtime = (current_time - start_time);

		  /* convert to milli seconds */
          xtime = (xtime * 1000) / clock_frequency;

          if( xtime > tmo ) return VI_ERROR_TMO;
		}
      else
		{

        	if( GetTickCount() - (DWORD)start_time > tmo )
			{
			 return VI_ERROR_TMO;
			}
		}
     }

    Sleep(0); /* gives up time slice */
   } /* end of wait loop */
 return VI_SUCCESS;
}

/* Unlocks the device.  A previous call to card_lock must have been done
   or hpe1459_NOT_LOCK_OWNER will be returned.
*/
ViStatus util_card_unlock(struct hpe1459_globals *thisPtr)
{

   if( !thisPtr->owner_valid || GetCurrentThreadId() != (DWORD)thisPtr->owner_thread)
		return hpe1459_ERROR_NOT_LOCK_OWNER;

   /* if there is still a count then this was not the last use of the
      lock for this thread.
   */
   if( --thisPtr->lock_count ) return  VI_SUCCESS;

   /* P&P calls will now NOT examine the owner data.  This protects against
      the system re-use of the thread ID (which may happen when this
      call returns, for example if this call is from an ISR callback and it
      is the last call in the ISR then upon returning the thread will die
      and its ID become available for use again.  But if another thread
      were to get its ID it does not matter as owner_valid would already
      be false)
   */

   thisPtr->owner_thread = 0;
   thisPtr->owner_valid = VI_FALSE;
   InterlockedExchange( &thisPtr->card_lock, _UNLOCK_);

   return VI_SUCCESS;
}

ViStatus util_device_clear( struct hpe1459_globals *thisPtr , long tmo)
{
	VINT64 current_time,start_time;

 /* This is a real dramatic step.  The user should be cautioned against
    using this in more than one thread  (doesn't make sense)

    The device clear global can have two values:

	_NOT_CLEARING_ - i.e. not being used at this time (normal)

	_CLEARING_  - this state is 'sent' when a device clear starts.
                      Upon seeing this the lock owner should
                      unlock the device and return the proper
                      status.  This means call 'card_unlock'.
*/


   /* as expected sombody else owns the lock.  We want them to clear it
      so set the global.  Use the critical seciton below to make sure ONLY
      ONE device clear at a time can run (user shoudl be admonished not to
      start several of these in  multiple threads, but if he did nothing
      bad would happen.)
   */
    EnterCriticalSection( &thisPtr->device_clear_cs );

	if( tmo )
     {
        if( !performance_clock_exists )
          {
           /* Warning GetTickCount is supported everywhere but it
              will roll over every 49.7 days making time look possibly
              negative .
           */
           start_time = GetTickCount();

          }
        else
          {
           /* in this case we know that the high frequency counter is
              supported.  That should be true in all cases and hence
              the warning above is more for completness than anything
              else.
              Note we will compute the time just in case the 8253 chip
              is changed but it is a known fact that
              it generates a time of 838 nano seconds
              per tick and is what QueryPerformanceCounter is based
			  upon.
           */

           QueryPerformanceCounter( (LARGE_INTEGER *)&start_time );

          }
     }

    thisPtr->device_clear = _CLEARING_;
    /* Wait for the card owner to release the device */
    while( 1 )
 	{

	 if( InterlockedExchange( &thisPtr->card_lock, _LOCK_) == _UNLOCK_)
		{

				 /*
				   The device has been released for us to clear.  We
                   own the lock now and can proceed.
                 */
                 thisPtr->device_clear = _NOT_CLEARING_;
                 LeaveCriticalSection( &thisPtr->device_clear_cs);

				 /* leave with the device locked so the clear
				    can proceed */

				 return VI_SUCCESS;
		}
	 if( tmo )
	  {
      if( QueryPerformanceCounter( (LARGE_INTEGER *)&current_time )	)
		{
          VINT64 xtime = (current_time - start_time);

          xtime = (xtime * 1000) / clock_frequency;

          if( xtime > tmo )
		  {
			  thisPtr->device_clear = _NOT_CLEARING_;
              LeaveCriticalSection( &thisPtr->device_clear_cs);
		   	  return VI_ERROR_TMO;
		  }
		}
      else
		{

        	if( GetTickCount() - (DWORD)start_time > (DWORD)tmo )
			{
		     thisPtr->device_clear = _NOT_CLEARING_;
             LeaveCriticalSection( &thisPtr->device_clear_cs);
			 return VI_ERROR_TMO;
			}
		}
      }

     Sleep(0);

	}
  return VI_SUCCESS;
}
#endif
#ifdef __hpux
/* ??? for UNIX */
ViStatus util_card_lock( struct hpe1459_globals *thisPtr,
									unsigned long tmo)
{ return VI_SUCCESS; }

ViStatus util_card_unlock(struct hpe1459_globals *thisPtr)
{ return VI_SUCCESS;}

ViStatus util_device_clear( struct hpe1459_globals *thisPtr , long tmo)
{ return VI_SUCCESS;}
#endif

/*  INTERRUPT functions begin here.  Remove if you do not use them */
#ifdef INSTR_CALLBACKS

/*
	viEnableEvent:

	vi		- session
	eventType	- Type of event to enable
	mechanism	- VI_HNDLR
	context		- VI_NULL

	Enables previously diabled interrupts.  hpe1459_disableEvents MUST
 	have been called previously.
*/
ViStatus _VI_FUNC hpe1459_enableEvent(ViSession vi,ViInt32 eventType,ViInt16 mechanism,ViInt32 context)
{
		struct hpe1459_globals *thisPtr = VI_NULL;
		ViStatus errStatus=VI_SUCCESS;
        short allow_unlock = 0;

		_LS(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )

		if( eventType != hpe1459_ALL_ENABLED_EVENTS)
			hpe1459_CHK_ENUM((ViInt16)eventType, hpe1459_EVENT_MAX,VI_ERROR_PARAMETER2)

		hpe1459_TRACE(thisPtr, "hpe1459_enableEvent",",%ld,%ld,%ld",eventType,mechanism,context);

		_LS( util_card_lock(thisPtr,thisPtr->lock_timeout) )
        allow_unlock = 1;

        if( eventType != hpe1459_ALL_ENABLED_EVENTS ) /* in this case just ONE event */
        {

		   /* simply set the eventArray to true to signify event is
		      active */
           if( !thisPtr->eventHndlr[eventType] )
			      hpe1459_LOG_STATUS(vi,thisPtr,VI_ERROR_HNDLR_NINSTALLED);

		   /* ingnore error */
		   if( eventType ) thisPtr->EnabledEvents |= (1<<eventType);
           thisPtr->eventArray[eventType] = VI_TRUE;
		   if( eventType) setEnablePorts(vi,thisPtr->EnabledEvents);
        }

	    /*	ALL_ENABLED_EVENTS serves the special purpose of ASSURING the user
			that AFTER a disableEvent(vi, ALL_ENABLED_EVENTS ...) until an
			_enableEVent(vi,ALL_ENABLED_EVENTS ...) NO ISR will interrupt his
			code block.  This is the ONLY safe way the user can do
			multiple instrument accesses that rely upon or change the
			instrument state.
        */

		if( thisPtr->EnabledEvents ) /* ALL visa interrupts */
        {
		   if( f_INTERFACE ) /* DSCPI */
			{
				if( thisPtr->VISAdisabled)
					_LS( viEnableEvent(vi,VI_EVENT_SERVICE_REQ,VI_HNDLR,VI_NULL))
				if( thisPtr->HDWIdisabled )
				{
					_LS( viPrintf(vi,"*SRE 128\n") ) /* Fire off interrupts */
					{
						ViInt16 sre,i;

						sre=0;
						for(i=0; (i<1000) && (sre != 128); i++)
							if( viQueryf(vi,"*SRE?\n","%hd",&sre) < VI_SUCCESS)break;
					}
				   _LS(viPrintf(vi,"STAT:OPER:PSUM:ENAB %d\n",0xffffffff))
				   WAIT_OPC
				}
			}
			else /* REG based */
			{
				if( thisPtr->VISAdisabled)
					_LS( viEnableEvent(vi,VI_EVENT_VXI_SIGP,VI_HNDLR,VI_NULL))
				/* fire interrupts */
				if( thisPtr->HDWIdisabled )
					_LS(viOut16(vi,VI_A16_SPACE,Status_reg,0x60))
			}
			thisPtr->VISAdisabled = VI_FALSE;
			thisPtr->HDWIdisabled=VI_FALSE;
	    }

   hpe1459_LOG_STATUS(vi,thisPtr,VI_SUCCESS);

}
ViStatus _VI_FUNC hpe1459_disableEvent(ViSession vi,ViInt32 eventType,ViInt16 mechanism,ViInt32 context)
{
	struct hpe1459_globals *thisPtr = VI_NULL;
	ViStatus errStatus=VI_SUCCESS;
	short allow_unlock = 0;

	_LS(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )

	if( f_INTERFACE ) /* DSCPI */
	{
		_LS( viEnableEvent(vi,VI_EVENT_SERVICE_REQ,VI_SUSPEND_HNDLR,VI_NULL))
    }
	else
	{
		_LS( viEnableEvent(vi,VI_EVENT_VXI_SIGP,VI_SUSPEND_HNDLR,VI_NULL))
	}

	if( eventType != hpe1459_ALL_ENABLED_EVENTS)
	     hpe1459_CHK_ENUM((ViInt16)eventType, hpe1459_EVENT_MAX,VI_ERROR_PARAMETER2)

	hpe1459_TRACE(thisPtr, "hpe1459_disableEvent",",%ld,%ld,%ld",eventType,mechanism,context);

	/* wait for any scheduled ISR's to complete */

	_LS(ClearISR(vi,thisPtr) )

	_LS(util_card_lock(thisPtr,thisPtr->lock_timeout) )
	allow_unlock = 1;


    /*if only one event is being suspeneded then re start ISR's if
	  all currently installed events are being diabled just leave the
	  ISR's suspened
    */
	if( eventType == hpe1459_ALL_ENABLED_EVENTS)
		{
		 thisPtr->VISAdisabled=TRUE;
		 hpe1459_LOG_STATUS(vi,thisPtr,VI_SUCCESS);
		}

	/* signifies that this event is disabled */
	thisPtr->eventArray[ eventType ] = VI_FALSE;

	/* ignore the error event */
	if( eventType )thisPtr->EnabledEvents &= ~(1 << eventType);

	_LS(setEnablePorts(vi, thisPtr->EnabledEvents))

    /* calls to the ISR are serialized by VISA so after the following
           API ALL ISR's will be completed.
	*/

	if( !thisPtr->EnabledEvents)
	{
		thisPtr->HDWIdisabled = VI_TRUE;
		thisPtr->VISAdisabled=VI_TRUE;
		if( f_INTERFACE )
			{
			  _LS(viPrintf(vi,"*SRE 0\n"))
			  {
				ViInt16 sre,i;

				sre=1;
				for(i=0; (i<1000) && (sre != 0); i++)
					if( viQueryf(vi,"*SRE?\n","%hd",&sre) < VI_SUCCESS)break;
			  }

			}
		else
			{
			  _LS(viOut16(vi,VI_A16_SPACE,Status_reg,0))
			}
	}
	else
	{
		if( f_INTERFACE ) /* DSCPI */
		{
		_LS( viEnableEvent(vi,VI_EVENT_SERVICE_REQ,VI_HNDLR,VI_NULL))
		}
		else
		{
		_LS( viEnableEvent(vi,VI_EVENT_VXI_SIGP,VI_HNDLR,VI_NULL))
		}
    }
	hpe1459_LOG_STATUS(vi,thisPtr,VI_SUCCESS);

}

ViStatus _VI_FUNC hpe1459_installHandler( ViSession vi, ViInt32 eventType, hpe1459_Handler handler, ViInt32 userHandle)
{
	struct hpe1459_globals *thisPtr = VI_NULL;
	ViStatus errStatus=VI_SUCCESS;
	short allow_unlock = 0;

	_LS(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )

    hpe1459_CHK_ENUM((ViInt16)eventType, hpe1459_EVENT_MAX,VI_ERROR_PARAMETER2)

	hpe1459_TRACE(thisPtr, "hpe1459_installHandler",",%ld,%ld,%ld",eventType,handler,userHandle);

	if( !thisPtr->interrupts) hpe1459_LOG_STATUS(VI_NULL,VI_NULL,VI_ERROR_INV_ACC_MODE);

	_LS(util_card_lock(thisPtr,thisPtr->lock_timeout) )
	allow_unlock = 1;

	thisPtr->eventHndlr[ eventType ] = handler;
	thisPtr->eventAttrib[eventType] = userHandle;
	thisPtr->eventArray[ eventType] = VI_FALSE; /* default disabled */


	hpe1459_LOG_STATUS(vi, thisPtr, VI_SUCCESS);
}

ViStatus _VI_FUNC hpe1459_uninstallHandler( ViSession vi, ViInt32 eventType, hpe1459_Handler handler, ViInt32 userHandle)
{
	struct hpe1459_globals *thisPtr = VI_NULL;
	ViStatus errStatus=VI_SUCCESS;
	short allow_unlock = 0;

	_LS(viGetAttribute( vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr) )

    hpe1459_CHK_ENUM((ViInt16)eventType, hpe1459_EVENT_MAX,VI_ERROR_PARAMETER2)

	hpe1459_TRACE(thisPtr, "hpe1459_uninstallHandler",",%ld,%ld,%ld",eventType,handler,userHandle);

	if( f_INTERFACE ) /* DSCPI */
	{
		_LS( viEnableEvent(vi,VI_EVENT_SERVICE_REQ,VI_SUSPEND_HNDLR,VI_NULL))
    }
	else
	{
		_LS( viEnableEvent(vi,VI_EVENT_VXI_SIGP,VI_SUSPEND_HNDLR,VI_NULL))
	}

	_LS(ClearISR(vi,thisPtr) )


	errStatus = util_card_lock(thisPtr,thisPtr->lock_timeout) ;
	if( errStatus < VI_SUCCESS)
	{
	   hpe1459_LOG_STATUS(vi,thisPtr,errStatus);
	}
	allow_unlock = 1;

	thisPtr->eventHndlr[ eventType ] = VI_NULL;
	thisPtr->eventAttrib[eventType] = 0;
	thisPtr->eventArray[ eventType] = VI_FALSE; /* disable */


	/* if no handlers remain then simply disable interrupts */
	thisPtr->EnabledEvents &= ~(1<<eventType);

	if( !thisPtr->EnabledEvents )
	{
		thisPtr->VISAdisabled = VI_TRUE;
		thisPtr->HDWIdisabled = VI_TRUE;
		if( f_INTERFACE )
			{
			  _LS(viDisableEvent(vi, VI_EVENT_SERVICE_REQ,VI_ALL_MECH))
			  /* disable interrupts */
			  _LS(viPrintf(vi,"*SRE 0\n"))
			  {
				ViInt16 sre,i;

				sre=1;
				for(i=0; (i<1000) && (sre != 0); i++)
					if( viQueryf(vi,"*SRE?\n","%hd",&sre) < VI_SUCCESS)break;
			  }

			}
		else
			{
			  _LS(viDisableEvent(vi, VI_EVENT_VXI_SIGP,VI_ALL_MECH))
			  /* stop interrupts */
			  _LS(viOut16(vi,VI_A16_SPACE,Status_reg,0))
			}
	}

	_LS( setEnablePorts( vi, thisPtr->EnabledEvents))

	if( thisPtr->EnabledEvents )
		if( f_INTERFACE ) /* DSCPI */
		{
		_LS( viEnableEvent(vi,VI_EVENT_SERVICE_REQ,VI_HNDLR,VI_NULL))
		}
		else
		{
		_LS( viEnableEvent(vi,VI_EVENT_VXI_SIGP,VI_HNDLR,VI_NULL))
		}

	hpe1459_LOG_STATUS(vi, thisPtr, VI_SUCCESS);
}

/* wait for any scheduled ISR to complete before returning
   (or time out)
*/
ViStatus ClearISR(ViSession vi, struct hpe1459_globals *thisPtr)
{
	VINT64 stime = 0;
	ViAttr tmo=0;

	viGetAttribute(vi,VI_ATTR_TMO_VALUE,&tmo);

	Sleep(0);  /* give any scheduled ISR a chance to run */

	util_marktime(&stime,tmo*1000);
	while( thisPtr->ISR_count  )
	{
	  if( util_marktime(&stime,tmo*1000) ) return VI_ERROR_TMO;
	}

	return VI_SUCCESS;
}

#endif

#ifdef WIN32
void util_wait( long val ) /* wait val micro seconds */
{
  /* just busy wait.  This isn't for long waits!  < 1 milli */
  /* for longer waits just sleep */

  unsigned VINT64 s,e,ticks ;

  if( !QueryPerformanceCounter((LARGE_INTEGER *)&s) )
  { Sleep(1); return; }

  ticks = (clock_frequency*(unsigned VINT64 )val)/(unsigned VINT64)1000000;
  ticks++;

  do {
	   QueryPerformanceCounter((LARGE_INTEGER *)&e );
	 } while( e-s < ticks);

}
#endif
#ifdef __hpux
void util_wait( long val)
{
	 /* ??? */
}
#endif

ViStatus BankSelect( ViSession vi, short port )
{
  ViInt16 tmp;
  ViStatus errStatus = VI_SUCCESS;
  /* send the bank to the status reg */

  _E( viIn16(vi,VI_A16_SPACE,Status_reg,&tmp) )

  if( port > 1 ) tmp |= 16;
  else tmp &= ~16;

  _E( viOut16(vi, VI_A16_SPACE,Status_reg,tmp) )

  return errStatus;
}

ViStatus init_regs( ViSession vi )
{
	short i;
	/* basically do a reset of the hardware */

	ViStatus errStatus = VI_SUCCESS;
    /*  Interrupt disable, bank = 0,reset 0 */

	_E( viOut16(vi,VI_A16_SPACE,Status_reg,0xff8e)	)
	/* reset */
	_E( viOut16(vi,VI_A16_SPACE,Status_reg,0xff8f))

	util_wait( 100 ); /* wait hundred micros seconds */

	_E( viOut16(vi,VI_A16_SPACE,Status_reg,0xff8e)	)

	/* Enable DAV and EDGE, Internal clock for all ports */
	for(i=0; i<4; i++)
	{
		_E(BankSelect(vi,i))
		_E( viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),0))
	}
	return errStatus;
}

#ifdef WIN32
ViBoolean util_marktime( unsigned VINT64 *start,unsigned VINT64 usec ) /* mark the current time for a wait of usec */
{
	unsigned VINT64 now,ticks;

	/* Time marker that can be used in loops.  Must pass in a local 64 bit
	   integer so that multiple threads can access without messing up
	   each others times.  For the start of the loop make sure the
	   variable has a value of zero.
	*/
	if( !*start )
		if( !QueryPerformanceFrequency( (LARGE_INTEGER *)start) )
		{
			 *start = GetTickCount();
		}

	if(performance_clock_exists)
	{
		QueryPerformanceCounter( (LARGE_INTEGER *) &now );
		ticks = (clock_frequency*(unsigned VINT64 )usec)/(unsigned VINT64)1000000;
		ticks++;
		if( now - *start < ticks) return VI_FALSE;
		else return VI_TRUE;
	}

	now = GetTickCount();
	if( now - *start < usec/1000) return VI_FALSE;

	return VI_TRUE;
}
#endif
#ifdef __hpux
ViBoolean util_marktime( unsigned VINT64 *start,unsigned VINT64 usec ) /* mark the current time for a wait of usec */
{ return VI_FALSE; }
#endif

/*  match the hardware enable to 'EnabledEvents' */
ViStatus setEnablePorts( ViSession vi, ViUInt16 EnabledEvents)
{
	short doall;
	short i;
	ViStatus errStatus=VI_SUCCESS;
	ViInt16 tmp;
	struct hpe1459_globals *thisPtr;

	_E(viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr))

	if( EnabledEvents & (1 << hpe1459_EVENT_EDGE) ) doall=1;
	else doall=0 ;

	for(i=0; i<4; i++)
	{
		if( p_INTERFACE)
		{
		_E(BankSelect( vi,i))

		if(doall || (EnabledEvents & (1 << (hpe1459_EVENT_EDGE_0+i))) )
		{
			_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),&tmp))
			tmp |= 1;
			_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),tmp))
		}
		else
		{
			_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),&tmp))
			tmp &= ~1;
			_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),tmp))
		}
		}
		if( f_INTERFACE)
		{
			if(doall || (EnabledEvents & (1 << (hpe1459_EVENT_EDGE_0+i))) )
			{
				_E(viPrintf(vi,"SENS:EVENT:PORT%d:EDGE:ENABLE ON\n",i) )
				WAIT_OPC
			}
			else
			{
				_E(viPrintf(vi,"SENS:EVENT:PORT%d:EDGE:ENABLE OFF\n",i) )
				WAIT_OPC
			}
		}
	}

	if( EnabledEvents & (1 << hpe1459_EVENT_DAV) ) doall=1;
	else doall=0;

	for(i=0; i<4; i++)
	{
		if( p_INTERFACE)
		{
		_E(BankSelect( vi,i))

		if(doall || (EnabledEvents & (1 << (hpe1459_EVENT_DAV_0+i))) )
		{
			_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),&tmp))
			tmp |= 4;
			_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),tmp))
		}
		else
		{
			_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),&tmp))
			tmp &= ~4;
			_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(i,Cmd_reg),tmp))
		}
		}
		if( f_INTERFACE)
		{
			if(doall || (EnabledEvents & (1 << (hpe1459_EVENT_DAV_0+i))) )
			{
				_E(viPrintf(vi,"SENS:EVENT:PORT%d:DAV:ENABLE ON\n",i) )
				WAIT_OPC
			}
			else
			{
				_E(viPrintf(vi,"SENS:EVENT:PORT%d:DAV:ENABLE OFF\n",i) )
				WAIT_OPC
			}
		}

	}

	return VI_SUCCESS;
}

/* returns the error from the last API */
ViStatus _VI_FUNC hpe1459_getLastError(
  ViSession vi,
  ViPInt32 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1;
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_TRACE(thisPtr, "hpe1459_getLastError","");

    /* Perform Error Checking on Each Parameter */
    {
    	*data = thisPtr->errNumber;
    }
    hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_clearAllEvents
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function will read ALL data & edge registers.  It operates
 *           as a special kind of reset but unlike the reset function, it
 *           does not change the clock state or alter enabled events.  This
 *           function does clear all data in the registers.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_clearAllEvents(
  ViSession vi)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViUInt16 tmp;
    	ViUInt16 Status;
    
    	_E(viIn16(vi,VI_A16_SPACE,Status_reg,&Status))
    	_E(viOut16(vi,VI_A16_SPACE,Status_reg,0)) /* shut off interrupts */
    
    	_E(BankSelect(vi,0))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,ChData_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,ChData_reg),&tmp))
    	_E(BankSelect(vi,2))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(2,ChData_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(3,ChData_reg),&tmp))
    
    	_E(BankSelect(vi,0))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,PosEdgeDet_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,PosEdgeDet_reg),&tmp))
    	_E(BankSelect(vi,2))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(2,PosEdgeDet_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(3,PosEdgeDet_reg),&tmp))
    
    	_E(BankSelect(vi,0))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,NegEdgeDet_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,NegEdgeDet_reg),&tmp))
    	_E(BankSelect(vi,2))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(2,NegEdgeDet_reg),&tmp))
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(3,NegEdgeDet_reg),&tmp))
    
    	_E( viOut16(vi,VI_A16_SPACE,Status_reg,Status))
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_clearAllEvents
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function will read ALL data & edge registers.  It operates
 *           as a special kind of reset but unlike the reset function, it
 *           does not change the clock state or alter enabled events.  This
 *           function does clear all data in the registers.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_clearAllEvents(
  ViSession vi)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViInt32 tmp;
    
    	_E(viQueryf(vi,"MEAS:DIG:DATA0:LWORD?\n","%d",&tmp) )
    	_E(viQueryf(vi,"MEAS:DIG:DATA2:LWORD?\n","%d",&tmp) )
    
    	_E(viQueryf(vi,"SENS:EVENT:PORT0:PEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT1:PEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT2:PEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT3:PEDGE?\n","%d",&tmp) )
    
    	_E(viQueryf(vi,"SENS:EVENT:PORT0:NEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT1:NEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT2:NEDGE?\n","%d",&tmp) )
    	_E(viQueryf(vi,"SENS:EVENT:PORT3:NEDGE?\n","%d",&tmp) )
    
    	_E(viQueryf(vi,"STAT:OPER:PSUM:EVENT?\n","%d",&tmp))
    	_E(viQueryf(vi,"STAT:OPER:EVENT?\n","%d",&tmp))
    
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_clearAllEvents[ hpe1459_MAX_INTERFACES]) (
  ViSession vi) =
{
  hpe1459p_clearAllEvents, /* thisPtr->iointerface = 0 */
  hpe1459f_clearAllEvents  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_clearAllEvents
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function will read ALL data & edge registers.  It operates
 *           as a special kind of reset but unlike the reset function, it
 *           does not change the clock state or alter enabled events.  This
 *           function does clear all data in the registers.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_clearAllEvents(
  ViSession vi)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_clearAllEvents" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_clearAllEvents[ thisPtr->iointerface](
       vi);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_enablePort
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function enables data available and edge detection on the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to enable.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 data
 * IN        
 *            Enable data, either hpe1459_ENABLE_DAV or hpe1459_ENABLE_EDGE.
 *            These can be ored together to get both.
 * 
 *      Macro Name               Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_ENABLE_DAV           4  Enable DAV
 *      hpe1459_ENABLE_EDGE          1  Enable Edge
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_enablePort(
  ViSession vi,
  ViInt32 port,
  ViInt32 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViUInt16 tmp;
    
    	_E(BankSelect(vi,(short)port))
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),&tmp))
    
    	tmp &= 2; /* keep int/ext bit */
    	tmp |= data;
    
    	_E( viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),tmp))
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_enablePort
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function enables data available and edge detection on the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to enable.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 data
 * IN        
 *            Enable data, either hpe1459_ENABLE_DAV or hpe1459_ENABLE_EDGE.
 *            These can be ored together to get both.
 * 
 *      Macro Name               Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_ENABLE_DAV           4  Enable DAV
 *      hpe1459_ENABLE_EDGE          1  Enable Edge
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_enablePort(
  ViSession vi,
  ViInt32 port,
  ViInt32 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    
    	_E(viPrintf(vi,"SENS:EVENT:PORT%d:DAV:ENABLE %s\n",port,
    		(data & 4) ? "ON" : "OFF") )
    	WAIT_OPC
    
    	_E(viPrintf(vi,"SENS:EVENT:PORT%d:EDGE:ENABLE %s\n",port,
    		(data & 1) ? "ON" : "OFF"))
    	WAIT_OPC
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_enablePort[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViInt32 data) =
{
  hpe1459p_enablePort, /* thisPtr->iointerface = 0 */
  hpe1459f_enablePort  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_enablePort
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function enables data available and edge detection on the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to enable.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 data
 * IN        
 *            Enable data, either hpe1459_ENABLE_DAV or hpe1459_ENABLE_EDGE.
 *            These can be ored together to get both.
 * 
 *      Macro Name               Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_ENABLE_DAV           4  Enable DAV
 *      hpe1459_ENABLE_EDGE          1  Enable Edge
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_enablePort(
  ViSession vi,
  ViInt32 port,
  ViInt32 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_enablePort" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_enablePort[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_getAttribute
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Gets attributes to the interface.  See the "Comments" section
 *           for the attributes supported.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 attrib
 * IN        
 *            Attribute to get.
 * 
 *      Macro Name                         Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_ATTRIB_LOCK_TMO                0  Card lock timeout
 *      hpe1459_ATTRIB_TRACE_DEST              1  Trace destination
 *      hpe1459_ATTRIB_TRACE_LEVEL             2  Trace level
 *      hpe1459_ATTRIB_EVENT_ERROR_UH          3  Error user handle
 * 
 * PARAM 3 : ViPInt32 data
 * OUT       
 *            Value of attribute.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_getAttribute(
  ViSession vi,
  ViInt32 attrib,
  ViPInt32 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_getAttribute" );

    /* Perform Error Checking on Each Parameter */
    {
    	switch(attrib)
    	{
    		case hpe1459_ATTRIB_LOCK_TMO:
    			*data=thisPtr->lock_timeout ;
    			break;
    		case hpe1459_ATTRIB_TRACE_DEST:
    			*data=thisPtr->traceDest;
    			break;
    		case hpe1459_ATTRIB_TRACE_LEVEL:
    			*data=thisPtr->traceLevel;
    			break;
    		case hpe1459_ATTRIB_EVENT_ERROR_UH:
    			if( thisPtr->eventHndlr[hpe1459_EVENT_ERROR])
    				*data=thisPtr->eventAttrib[hpe1459_EVENT_ERROR];
    			else
    			errStatus = VI_ERROR_HNDLR_NINSTALLED;
    			break;
    		case 1000: //diag_event
    			*data = thisPtr->diag_event;
    			break;
    		default:
    			break;
    	}
    
    }
    
    hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_input
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the clock source (internal or external ) and the debounce
 *           time.  Ports 0 and 1 share the same debounce time ports as do
 *            ports 2/3.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to set clock source and debounce time.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 clock_src
 * IN        
 *            Clock souce internal, external, or unused.  If unused, this
 *           routine will not change the clock source.
 * 
 *      Macro Name             Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_UNUSED             0  Not used
 *      hpe1459_CLOCK_INT          1  Clock Internal
 *      hpe1459_CLOCK_EXT          2  Clock Extern
 * 
 * PARAM 4 : ViReal64 deb_time
 * IN        
 *            Debounce clock time. Values are: 18.0 us, 36 us, 144 us
 *            288us 1.13 ms, 2.26 ms, 4.6 ms,9.2 ms. 18.4 ms, 36.9 ms 73.8
 *           ms, 148.0 ms, 294.0 ms, 590 ms, 1.18 S, 2.36 S, 4.72 S, 9.43 S,
 *            18.9 S, 37.8 S, 75.0 S, 150.0 S, 300 S, 1200 S, 2400 S,
 *            4800 S and 9600 S.   Other values will be rounded to the
 *           nearest discrete value except for zero which will tell this
 *           routine to ignore setting the debounce time.
 * 
 *      MAX = hpe1459_DEBOUNCE_MAX   9600
 *      MIN = hpe1459_DEBOUNCE_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_input(
  ViSession vi,
  ViInt32 port,
  ViInt32 clock_src,
  ViReal64 deb_time)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViUInt16 tmp;
    
    	_E(BankSelect(vi,(short)port ) )
    
    	if( clock_src )
    	{
    		_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),&tmp) )
    
    		if( (tmp & 4) && (clock_src == hpe1459_CLOCK_INT) ) /* DAV enabled */
    		{
    			/* Don't allow as interrupts occur to fast */
    			return hpe1459_ERROR_DAV_SET;
    		}
    
    		tmp &= ~2;
    		tmp |= 	(clock_src == hpe1459_CLOCK_INT)? 0 : 2;
    		_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),
    			tmp  ) )
    	}
    
    	if( deb_time )
    	{
    	ViUInt16 debounce_reg_data;
    
    	if (deb_time > 4800.50)
            debounce_reg_data = 0x001F;
        else if (deb_time > 2400.5)
            debounce_reg_data = 0x001E;
        else if (deb_time > 1200.5)
            debounce_reg_data = 0x001D;
        else if (deb_time > 600.5)
            debounce_reg_data = 0x001C;
        else if (deb_time > 300.5)
            debounce_reg_data = 0x001B;
        else if (deb_time > 150.5)
            debounce_reg_data = 0x001A;
        else if (deb_time > 75.5)
            debounce_reg_data = 0x0019;
        else if (deb_time > 37.85)
            debounce_reg_data = 0x0018;
        else if (deb_time > 18.95)
            debounce_reg_data = 0x0017;
        else if (deb_time > 9.435)
            debounce_reg_data = 0x0016;
        else if (deb_time > 4.725)
            debounce_reg_data = 0x0015;
        else if (deb_time > 2.365)
            debounce_reg_data = 0x0014;
        else if (deb_time > 1.185)
            debounce_reg_data = 0x0013;
        else if (deb_time > 590.5E-3)
            debounce_reg_data = 0x0012;
        else if (deb_time > 294.5E-3)
            debounce_reg_data = 0x0011;
        else if (deb_time > 148.5E-3)
            debounce_reg_data = 0x0010;
        else if (deb_time > 73.85E-3)
            debounce_reg_data = 0x000F;
        else if (deb_time > 36.95E-3)
            debounce_reg_data = 0x000E;
        else if (deb_time > 18.45E-3)
            debounce_reg_data = 0x000D;
        else if (deb_time > 9.250E-3)
            debounce_reg_data = 0x000C;
        else if (deb_time > 4.650E-3)
            debounce_reg_data = 0x000B;
        else if (deb_time > 2.265E-3)
            debounce_reg_data = 0x000A;
        else if (deb_time > 1.135E-3)
            debounce_reg_data = 0x0009;
        else if (deb_time > 576.5E-6)
            debounce_reg_data = 0x0008;
        else if (deb_time > 288.5E-6)
            debounce_reg_data = 0x0007;
        else if (deb_time > 144.5E-6)
            debounce_reg_data = 0x0006;
        else if (deb_time > 72.50E-6)
            debounce_reg_data = 0x0005;
        else if (deb_time > 36.50E-6)
            debounce_reg_data = 0x0004;
        else if (deb_time > 18.500E-6)
            debounce_reg_data = 0x0001;
        else
            debounce_reg_data = 0x0000;
    
    	_E( viOut16(vi,VI_A16_SPACE,DebCntl_reg,debounce_reg_data))
    
    	}
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_input
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the clock source (internal or external ) and the debounce
 *           time.  Ports 0 and 1 share the same debounce time ports as do
 *            ports 2/3.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to set clock source and debounce time.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 clock_src
 * IN        
 *            Clock souce internal, external, or unused.  If unused, this
 *           routine will not change the clock source.
 * 
 *      Macro Name             Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_UNUSED             0  Not used
 *      hpe1459_CLOCK_INT          1  Clock Internal
 *      hpe1459_CLOCK_EXT          2  Clock Extern
 * 
 * PARAM 4 : ViReal64 deb_time
 * IN        
 *            Debounce clock time. Values are: 18.0 us, 36 us, 144 us
 *            288us 1.13 ms, 2.26 ms, 4.6 ms,9.2 ms. 18.4 ms, 36.9 ms 73.8
 *           ms, 148.0 ms, 294.0 ms, 590 ms, 1.18 S, 2.36 S, 4.72 S, 9.43 S,
 *            18.9 S, 37.8 S, 75.0 S, 150.0 S, 300 S, 1200 S, 2400 S,
 *            4800 S and 9600 S.   Other values will be rounded to the
 *           nearest discrete value except for zero which will tell this
 *           routine to ignore setting the debounce time.
 * 
 *      MAX = hpe1459_DEBOUNCE_MAX   9600
 *      MIN = hpe1459_DEBOUNCE_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_input(
  ViSession vi,
  ViInt32 port,
  ViInt32 clock_src,
  ViReal64 deb_time)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       if( clock_src )
       	_E(viPrintf(vi,"INP%d:CLOCK %s\n",port,
    		(clock_src==hpe1459_CLOCK_INT)?"INT":"EXT") )
       WAIT_OPC
       if( deb_time )
       	_E(viPrintf(vi,"INP%d:DEB:TIME %f\n",port,deb_time) )
       WAIT_OPC
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_input[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViInt32 clock_src,
  ViReal64 deb_time) =
{
  hpe1459p_input, /* thisPtr->iointerface = 0 */
  hpe1459f_input  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_input
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the clock source (internal or external ) and the debounce
 *           time.  Ports 0 and 1 share the same debounce time ports as do
 *            ports 2/3.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number to set clock source and debounce time.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt32 clock_src
 * IN        
 *            Clock souce internal, external, or unused.  If unused, this
 *           routine will not change the clock source.
 * 
 *      Macro Name             Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_UNUSED             0  Not used
 *      hpe1459_CLOCK_INT          1  Clock Internal
 *      hpe1459_CLOCK_EXT          2  Clock Extern
 * 
 * PARAM 4 : ViReal64 deb_time
 * IN        
 *            Debounce clock time. Values are: 18.0 us, 36 us, 144 us
 *            288us 1.13 ms, 2.26 ms, 4.6 ms,9.2 ms. 18.4 ms, 36.9 ms 73.8
 *           ms, 148.0 ms, 294.0 ms, 590 ms, 1.18 S, 2.36 S, 4.72 S, 9.43 S,
 *            18.9 S, 37.8 S, 75.0 S, 150.0 S, 300 S, 1200 S, 2400 S,
 *            4800 S and 9600 S.   Other values will be rounded to the
 *           nearest discrete value except for zero which will tell this
 *           routine to ignore setting the debounce time.
 * 
 *      MAX = hpe1459_DEBOUNCE_MAX   9600
 *      MIN = hpe1459_DEBOUNCE_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_input(
  ViSession vi,
  ViInt32 port,
  ViInt32 clock_src,
  ViReal64 deb_time)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_input" );

    /* Perform Error Checking on Each Parameter */
    hpe1459_CHK_REAL_RANGE(deb_time
                          ,hpe1459_DEBOUNCE_MIN
                          ,hpe1459_DEBOUNCE_MAX
                          ,VI_ERROR_PARAMETER4);

    /* call the interface specific function */
    errStatus = hpe1459I_input[ thisPtr->iointerface](
       vi,
       port,
       clock_src,
       deb_time);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_input_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function queries the clock source and debounce time.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number whose clock source and debounce time are queried.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 clock_src
 * OUT       
 *            Clock souce.  1 = Internal, 2 = External.
 * 
 * PARAM 4 : ViPReal64 deb_time
 * OUT       
 *            Debounce clock time (see input above for possible values).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_input_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 clock_src,
  ViPReal64 deb_time)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViUInt16 tmp,debounce_reg_data;
    
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),&tmp) )
    
        *clock_src = ((tmp>>1) & 1)+1;
    
    	_E(viIn16(vi,VI_A16_SPACE,DebCntl_reg,&debounce_reg_data) )
    
    	switch (debounce_reg_data & 0xf) {
    	  case 0x00:
    	  case 0x02:
    	    *deb_time =  18.0E-6;
    	    break;
    	  case 0x01:
    	  case 0x03:
    	    *deb_time =  36.0E-6;
    	    break;
    	  case 0x04:
    	    *deb_time =  72.0E-6;
    	    break;
    	  case 0x05:
    	    *deb_time =  144.0E-6;
    	    break;
    	  case 0x06:
    	    *deb_time =  288.0E-6;
    	    break;
    	  case 0x07:
    	    *deb_time =  576.0E-6;
    	    break;
    	  case 0x08:
    	    *deb_time =  1.13E-3;
    	    break;
    	  case 0x09:
    	    *deb_time =  2.26E-3;
    	    break;
    	  case 0x0A:
    	    *deb_time =  4.60E-3;
    	    break;
    	  case 0x0B:
    	    *deb_time =  9.20E-3;
    	    break;
    	  case 0x0C:
    	    *deb_time =  18.4E-3;
    	    break;
    	  case 0x0D:
    	    *deb_time =  36.9E-3;
    	    break;
    	  case 0x0E:
    	    *deb_time =  73.8E-3;
    	    break;
    	  case 0x0F:
    	    *deb_time =  148.0E-3;
    	    break;
    	  case 0x10:
    	    *deb_time =  294.0E-3;
    	    break;
    	  case 0x11:
    	    *deb_time =  590.0E-3;
    	    break;
    	  case 0x12:
    	    *deb_time =  1.180;
    	    break;
    	  case 0x13:
    	    *deb_time =  2.360;
    	    break;
    	  case 0x14:
    	    *deb_time =  4.720;
    	    break;
    	  case 0x15:
    	    *deb_time =  9.430;
    	    break;
    	  case 0x16:
    	    *deb_time =  18.90;
    	    break;
    	  case 0x17:
    	    *deb_time =  37.80;
    	    break;
    	  case 0x18:
    	    *deb_time =  75.00;
    	    break;
    	  case 0x19:
    	    *deb_time =  150.0;
    	    break;
    	  case 0x1A:
    	    *deb_time =  300.0;
    	    break;
    	  case 0x1B:
    	    *deb_time =  600.0;
    	    break;
    	  case 0x1C:
    	    *deb_time =  1200.0;
    	    break;
    	  case 0x1D:
    	    *deb_time =  2400.0;
    	    break;
    	  case 0x1E:
    	    *deb_time =  4800.0;
    	    break;
    	  case 0x1F:
    	    *deb_time =  9600.0;
    	    break;
          default:
    	    *deb_time = 0.0;
    	}
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_input_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function queries the clock source and debounce time.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number whose clock source and debounce time are queried.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 clock_src
 * OUT       
 *            Clock souce.  1 = Internal, 2 = External.
 * 
 * PARAM 4 : ViPReal64 deb_time
 * OUT       
 *            Debounce clock time (see input above for possible values).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_input_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 clock_src,
  ViPReal64 deb_time)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	char lc[256];
    
       _E(viQueryf(vi,"INP%d:CLOCK?\n","%s",port,lc))
    
       if( !strcmp( lc,"INT") ) *clock_src = 1;
       else *clock_src = 2;
    
       _E(viQueryf(vi,"INP%d:DEB:TIME?\n","%lf",port,deb_time) )
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_input_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 clock_src,
  ViPReal64 deb_time) =
{
  hpe1459p_input_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_input_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_input_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function queries the clock source and debounce time.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number whose clock source and debounce time are queried.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 clock_src
 * OUT       
 *            Clock souce.  1 = Internal, 2 = External.
 * 
 * PARAM 4 : ViPReal64 deb_time
 * OUT       
 *            Debounce clock time (see input above for possible values).
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_input_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 clock_src,
  ViPReal64 deb_time)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_input_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_input_Q[ thisPtr->iointerface](
       vi,
       port,
       clock_src,
       deb_time);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_measAllDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads all four 16-bit digital ports.  Each port
 *           has the following bits (channels). The MSB is on the left, the
 *           LSB is on the right.
 *           port 0: bits (channels) 15 - 0
 *           port 1: bits (channels) 31 - 16
 *           port 2: bits (channels) 47 - 32
 *           port 3: bits (channels) 63 - 48
 *           
 *            A '1' means the channel data is high, a '0' means the data is
 *           low.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from digital port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from digital port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from digital port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from digital port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_measAllDigData_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    
    	_E(BankSelect(vi,0))  /* status reg bit 4 = 0 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,ChData_reg),port0) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,ChData_reg),port1) )
    
    	_E(BankSelect(vi,2)) /* status reg bit 4 = 1 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,ChData_reg),port2) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,ChData_reg),port3) )
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_measAllDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads all four 16-bit digital ports.  Each port
 *           has the following bits (channels). The MSB is on the left, the
 *           LSB is on the right.
 *           port 0: bits (channels) 15 - 0
 *           port 1: bits (channels) 31 - 16
 *           port 2: bits (channels) 47 - 32
 *           port 3: bits (channels) 63 - 48
 *           
 *            A '1' means the channel data is high, a '0' means the data is
 *           low.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from digital port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from digital port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from digital port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from digital port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_measAllDigData_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    {
    	_E(viQueryf(vi,"MEAS:DIG:DATA0?\n","%hd",port0) )
    	_E(viQueryf(vi,"MEAS:DIG:DATA1?\n","%hd",port1) )
    	_E(viQueryf(vi,"MEAS:DIG:DATA2?\n","%hd",port2) )
    	_E(viQueryf(vi,"MEAS:DIG:DATA3?\n","%hd",port3) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_measAllDigData_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0) =
{
  hpe1459p_measAllDigData_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_measAllDigData_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_measAllDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads all four 16-bit digital ports.  Each port
 *           has the following bits (channels). The MSB is on the left, the
 *           LSB is on the right.
 *           port 0: bits (channels) 15 - 0
 *           port 1: bits (channels) 31 - 16
 *           port 2: bits (channels) 47 - 32
 *           port 3: bits (channels) 63 - 48
 *           
 *            A '1' means the channel data is high, a '0' means the data is
 *           low.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from digital port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from digital port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from digital port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from digital port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_measAllDigData_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_measAllDigData_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_measAllDigData_Q[ thisPtr->iointerface](
       vi,
       port3,
       port2,
       port1,
       port0);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_measAllNegTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for all four ports.  A
 *           '1' indicates a negative edge has occurred on that port since
 *           the last time the register was read.  A '0' means a negative
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from negative edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from negative edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from negative edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from negative edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_measAllNegTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	ViUInt16 tmp;
    
    	_E( viIn16(vi, VI_A16_SPACE,Status_reg, &tmp ))
    
        _E( viOut16(vi,VI_A16_SPACE,Status_reg,0xff8e) ) /* interrupts off*/
    
    	_E(BankSelect(vi,0))  /* status reg bit 4 = 0 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,NegEdgeDet_reg),port0) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,NegEdgeDet_reg),port1) )
    
    	_E(BankSelect(vi,2)) /* status reg bit 4 = 1 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,NegEdgeDet_reg),port2) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,NegEdgeDet_reg),port3) )
    
        _E( viOut16(vi,VI_A16_SPACE,Status_reg,tmp) ) /* restore */
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_measAllNegTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for all four ports.  A
 *           '1' indicates a negative edge has occurred on that port since
 *           the last time the register was read.  A '0' means a negative
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from negative edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from negative edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from negative edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from negative edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_measAllNegTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    {
    	_E(viQueryf(vi,"SENS:EVEN:PORT0:NEDG?\n","%hd",port0) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT1:NEDG?\n","%hd",port1) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT2:NEDG?\n","%hd",port2) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT3:NEDG?\n","%hd",port3) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_measAllNegTran_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0) =
{
  hpe1459p_measAllNegTran_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_measAllNegTran_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_measAllNegTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for all four ports.  A
 *           '1' indicates a negative edge has occurred on that port since
 *           the last time the register was read.  A '0' means a negative
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from negative edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from negative edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from negative edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from negative edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_measAllNegTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_measAllNegTran_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_measAllNegTran_Q[ thisPtr->iointerface](
       vi,
       port3,
       port2,
       port1,
       port0);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_measAllPosTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positive edge detect register for all four ports.  A
 *           '1' indicates a positive edge has occurred on that port since
 *           the last time the register was read.  A '0' means a positive
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from positive edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from positive edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from positive edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from positive edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_measAllPosTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
        ViUInt16 tmp;
    
    	_E( viIn16(vi,VI_A16_SPACE,Status_reg,&tmp) )
    
        _E( viOut16(vi,VI_A16_SPACE,Status_reg,0xff8e) ) /* interrupts off*/
    
    	_E(BankSelect(vi,0))  /* status reg bit 4 = 0 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,PosEdgeDet_reg),port0) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,PosEdgeDet_reg),port1) )
    
    	_E(BankSelect(vi,2)) /* status reg bit 4 = 1 */
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(0,PosEdgeDet_reg),port2) )
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(1,PosEdgeDet_reg),port3) )
    
        _E( viOut16(vi,VI_A16_SPACE,Status_reg,tmp) ) /* restore */
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_measAllPosTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positive edge detect register for all four ports.  A
 *           '1' indicates a positive edge has occurred on that port since
 *           the last time the register was read.  A '0' means a positive
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from positive edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from positive edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from positive edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from positive edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_measAllPosTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = VI_SUCCESS;
    {
    	_E(viQueryf(vi,"SENS:EVEN:PORT0:PEDG?\n","%hd",port0) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT1:PEDG?\n","%hd",port1) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT2:PEDG?\n","%hd",port2) )
    	_E(viQueryf(vi,"SENS:EVEN:PORT3:PEDG?\n","%hd",port3) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_measAllPosTran_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0) =
{
  hpe1459p_measAllPosTran_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_measAllPosTran_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_measAllPosTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positive edge detect register for all four ports.  A
 *           '1' indicates a positive edge has occurred on that port since
 *           the last time the register was read.  A '0' means a positive
 *           edge has not occurred.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 port3
 * OUT       
 *            Data returned from positive edge detect register for port 3.
 * 
 * PARAM 3 : ViPInt16 port2
 * OUT       
 *            Data returned from positive edge detect register for port 2.
 * 
 * PARAM 4 : ViPInt16 port1
 * OUT       
 *            Data returned from positive edge detect register for port 1.
 * 
 * PARAM 5 : ViPInt16 port0
 * OUT       
 *            Data returned from positive edge detect register for port 0.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_measAllPosTran_Q(
  ViSession vi,
  ViPInt16 port3,
  ViPInt16 port2,
  ViPInt16 port1,
  ViPInt16 port0)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_measAllPosTran_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_measAllPosTran_Q[ thisPtr->iointerface](
       vi,
       port3,
       port2,
       port1,
       port0);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_measDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads the specified channel (0-63) and returns
 *           VI_TRUE if the line is high ('1').
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from 'channel'.  VI_TRUE means the line is high.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_measDigData_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	short port = CHAN_TO_PORT( channel);
    	ViInt16 tmp;
    
    	_E(BankSelect(vi,port ) )
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,ChData_reg),&tmp) )
    
    	*data = ( tmp & ( 1 << (channel-port*16)) ) ? VI_TRUE : VI_FALSE;
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_measDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads the specified channel (0-63) and returns
 *           VI_TRUE if the line is high ('1').
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from 'channel'.  VI_TRUE means the line is high.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_measDigData_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	short port = CHAN_TO_PORT( channel);
    
    	_E(viQueryf(vi,"MEAS:DIG:DATA%d:WORD:BIT%d?\n","%hd",port,channel-port*16,data) )
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_measDigData_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data) =
{
  hpe1459p_measDigData_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_measDigData_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_measDigData_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function reads the specified channel (0-63) and returns
 *           VI_TRUE if the line is high ('1').
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from 'channel'.  VI_TRUE means the line is high.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_measDigData_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_measDigData_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1459_CHK_LONG_RANGE(channel
                          ,hpe1459_CHANNEL_MIN
                          ,hpe1459_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    /* call the interface specific function */
    errStatus = hpe1459I_measDigData_Q[ thisPtr->iointerface](
       vi,
       channel,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_measDigPort_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the data on the specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data returned from the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_measDigPort_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,ChData_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_measDigPort_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the data on the specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data returned from the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_measDigPort_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viQueryf(vi,"MEAS:DIG:DATA%d?\n","%hd",port,data) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_measDigPort_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 data) =
{
  hpe1459p_measDigPort_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_measDigPort_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_measDigPort_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the data on the specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data returned from the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_measDigPort_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_measDigPort_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_measDigPort_Q[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_negMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the negative mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect negative edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's negative Mask register to enable
 *           selected channels to detect negative edges. A '1'  unmasks the
 *           channel enabling it to detect a negative edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_negMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,NegMask_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_negMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the negative mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect negative edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's negative Mask register to enable
 *           selected channels to detect negative edges. A '1'  unmasks the
 *           channel enabling it to detect a negative edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_negMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viPrintf(vi,"SENS:EVEN:PORT%d:NEDGE:ENAB %hd\n",port,data) )
    	WAIT_OPC
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_negMask[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViInt16 data) =
{
  hpe1459p_negMask, /* thisPtr->iointerface = 0 */
  hpe1459f_negMask  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_negMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the negative mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect negative edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's negative Mask register to enable
 *           selected channels to detect negative edges. A '1'  unmasks the
 *           channel enabling it to detect a negative edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_negMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_negMask" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_negMask[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_negMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's negative mask register.
 *           Unmasked bits (channels) are able to detect negative edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Negative Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and negative
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_negMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,NegMask_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_negMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's negative mask register.
 *           Unmasked bits (channels) are able to detect negative edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Negative Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and negative
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_negMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viQueryf(vi,"SENS:EVEN:PORT%d:NEDGE:ENAB?\n", "%hd",port,data) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_negMask_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 data) =
{
  hpe1459p_negMask_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_negMask_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_negMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's negative mask register.
 *           Unmasked bits (channels) are able to detect negative edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Negative Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and negative
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_negMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_negMask_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_negMask_Q[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_posMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the positive mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect positive edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's positive Mask register to enable
 *           selected channels to detect positive edges. A '1'  unmasks the
 *           channel enabling it to detect a positive edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_posMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viOut16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,PosMask_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_posMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the positive mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect positive edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's positive Mask register to enable
 *           selected channels to detect positive edges. A '1'  unmasks the
 *           channel enabling it to detect a positive edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_posMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viPrintf(vi,"SENS:EVEN:PORT%d:PEDGE:ENAB %hd\n",port,data) )
    	WAIT_OPC
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_posMask[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViInt16 data) =
{
  hpe1459p_posMask, /* thisPtr->iointerface = 0 */
  hpe1459f_posMask  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_posMask
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets the positive mask register for the specified port. 
 *           Unmasked bits (channels) are able to detect positive edges.
 *           Setting a bit to '1' unmasks the channel. Setting a bit to '0'
 *           masks the channel.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViInt16 data
 * IN        
 *            Sets the specified port's positive Mask register to enable
 *           selected channels to detect positive edges. A '1'  unmasks the
 *           channel enabling it to detect a positive edge.
 * 
 *      MAX = hpe1459_MASK_MAX   32767
 *      MIN = hpe1459_MASK_MIN   -32768
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_posMask(
  ViSession vi,
  ViInt32 port,
  ViInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_posMask" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_posMask[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_posMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's positive mask register.
 *           Unmasked bits (channels) are able to detect positive edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Positive Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and positive
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_posMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,PosMask_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_posMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's positive mask register.
 *           Unmasked bits (channels) are able to detect positive edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Positive Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and positive
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_posMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viQueryf(vi,"SENS:EVEN:PORT%d:PEDGE:ENAB?\n", "%hd",port,data) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_posMask_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 data) =
{
  hpe1459p_posMask_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_posMask_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_posMask_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns the setting of the port's positive mask register.
 *           Unmasked bits (channels) are able to detect positive edges. Bits
 *           set to '1' correspond to unmasked channels. Bits set to '0'
 *           correspond to masked channels.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Value of the Positive Mask register.  A '1' in a bit position
 *            indicates the corresponding channel is unmasked and positive
 *           edges can be detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_posMask_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_posMask_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_posMask_Q[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_sensEvenNedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a negative edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from negative edge detect register for 'channel'.
 *            VI_TRUE means there has been a negative edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_sensEvenNedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	short port = CHAN_TO_PORT( channel);
    	ViInt16 tmp;
    
    	_E(BankSelect(vi,port ) )
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,NegEdgeDet_reg),&tmp) )
    
    	*data = ( tmp & ( 1 << (channel-port*16)) ) ? VI_TRUE : VI_FALSE;
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_sensEvenNedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a negative edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from negative edge detect register for 'channel'.
 *            VI_TRUE means there has been a negative edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_sensEvenNedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	short port = CHAN_TO_PORT( channel);
    	ViInt16 tmp;
    
    	_E(viQueryf(vi,"SENS:EVEN:PORT%d:NEDG?\n","%hd",port,&tmp) )
    
    	*data = ( tmp & ( 1 << (channel-port*16)) ) ? VI_TRUE : VI_FALSE;
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_sensEvenNedg_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data) =
{
  hpe1459p_sensEvenNedg_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_sensEvenNedg_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_sensEvenNedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the negative edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a negative edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from negative edge detect register for 'channel'.
 *            VI_TRUE means there has been a negative edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_sensEvenNedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_sensEvenNedg_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1459_CHK_LONG_RANGE(channel
                          ,hpe1459_CHANNEL_MIN
                          ,hpe1459_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    /* call the interface specific function */
    errStatus = hpe1459I_sensEvenNedg_Q[ thisPtr->iointerface](
       vi,
       channel,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_sensEvenPedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positve edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a positive edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from positive edge detect register for 'channel'.
 *            VI_TRUE means there has been a positive edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_sensEvenPedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	short port = CHAN_TO_PORT( channel);
    	ViInt16 tmp;
    
    	_E(BankSelect(vi,port ) )
    
    	_E( viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,PosEdgeDet_reg),&tmp) )
    
    	*data = ( tmp & ( 1 << (channel-port*16)) ) ? VI_TRUE : VI_FALSE;
    
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_sensEvenPedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positve edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a positive edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from positive edge detect register for 'channel'.
 *            VI_TRUE means there has been a positive edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_sensEvenPedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	short port = CHAN_TO_PORT( channel);
    	ViInt16 tmp;
    
    	_E(viQueryf(vi,"SENS:EVEN:PORT%d:PEDG?\n","%hd",port,&tmp) )
    
    	*data = ( tmp & ( 1 << (channel-port*16)) ) ? VI_TRUE : VI_FALSE;
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_sensEvenPedg_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data) =
{
  hpe1459p_sensEvenPedg_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_sensEvenPedg_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_sensEvenPedg_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Reads the positve edge detect register for the specified
 *           channel (0-63) and returns VI_TRUE if a positive edge has been
 *           detected.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 channel
 * IN        
 *            Channel line to get the data from.
 * 
 *      MAX = hpe1459_CHANNEL_MAX   63
 *      MIN = hpe1459_CHANNEL_MIN   0
 * 
 * PARAM 3 : ViPBoolean data
 * OUT       
 *            Data returned from positive edge detect register for 'channel'.
 *            VI_TRUE means there has been a positive edge detected.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_sensEvenPedg_Q(
  ViSession vi,
  ViInt32 channel,
  ViPBoolean data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_sensEvenPedg_Q" );

    /* Perform Error Checking on Each Parameter */
    hpe1459_CHK_LONG_RANGE(channel
                          ,hpe1459_CHANNEL_MIN
                          ,hpe1459_CHANNEL_MAX
                          ,VI_ERROR_PARAMETER2);

    /* call the interface specific function */
    errStatus = hpe1459I_sensEvenPedg_Q[ thisPtr->iointerface](
       vi,
       channel,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_sensNegPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the negative edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from negative edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_sensNegPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,NegEdgeDet_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_sensNegPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the negative edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from negative edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_sensNegPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viQueryf(vi,"SENS:EVEN:PORT%d:NEDGE?\n","%hd",port,data) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_sensNegPortTran_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 data) =
{
  hpe1459p_sensNegPortTran_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_sensNegPortTran_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_sensNegPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the negative edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from negative edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_sensNegPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_sensNegPortTran_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_sensNegPortTran_Q[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_sensPosPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the positive edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from positive edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_sensPosPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,PosEdgeDet_reg),data ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_sensPosPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the positive edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from positive edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_sensPosPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = VI_SUCCESS;
    {
       	_E(viQueryf(vi,"SENS:EVEN:PORT%d:PEDGE?\n","%hd",port,data) )
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_sensPosPortTran_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViInt32 port,
  ViPInt16 data) =
{
  hpe1459p_sensPosPortTran_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_sensPosPortTran_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_sensPosPortTran_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Returns data from the positive edge detect register for the
 *           specified port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 3 : ViPInt16 data
 * OUT       
 *            Data from positive edge detect register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_sensPosPortTran_Q(
  ViSession vi,
  ViInt32 port,
  ViPInt16 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_sensPosPortTran_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_sensPosPortTran_Q[ thisPtr->iointerface](
       vi,
       port,
       data);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459p_sensRegState_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function returns conditions monitored by the Status, Edge
 *           interrupt, DAV interrupt, and command registers.
 *           
 *            The lower four bits of the DAV and Edge interrupt registers can
 *           be used to determine if any edge or DAV events have occurred.
 *           Bit 3 (MSB) is port 3, Bit 0 (LSB) is port 0. A '1' in any of
 *           these bit positions indicates an event occurred on that port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 StatusReg
 * OUT       
 *            Returns the devices Status/Control register.
 * 
 * PARAM 3 : ViPInt16 EdgeReg
 * OUT       
 *            Returns Edge Interrupt Status register.
 * 
 * PARAM 4 : ViPInt16 DAVReg
 * OUT       
 *            Returns Data Available Status register.
 * 
 * PARAM 5 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 6 : ViPInt16 CmdReg
 * OUT       
 *            Returns the command register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459p_sensRegState_Q(
  ViSession vi,
  ViPInt16 StatusReg,
  ViPInt16 EdgeReg,
  ViPInt16 DAVReg,
  ViInt32 port,
  ViPInt16 CmdReg)
{
    ViStatus errStatus = VI_SUCCESS;
    struct hpe1459_globals *thisPtr;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ return errStatus; }

    {
    	_E(BankSelect(vi,(short)port ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,Status_reg,StatusReg ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,EdgeIntr_reg,EdgeReg ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,DAVStatus_reg,DAVReg ) )
    
    	_E(viIn16(vi,VI_A16_SPACE,REGISTER_ADDRESS(port,Cmd_reg),CmdReg ) )
    }
    return errStatus;
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459f_sensRegState_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function returns conditions monitored by the Status, Edge
 *           interrupt, DAV interrupt, and command registers.
 *           
 *            The lower four bits of the DAV and Edge interrupt registers can
 *           be used to determine if any edge or DAV events have occurred.
 *           Bit 3 (MSB) is port 3, Bit 0 (LSB) is port 0. A '1' in any of
 *           these bit positions indicates an event occurred on that port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 StatusReg
 * OUT       
 *            Returns the devices Status/Control register.
 * 
 * PARAM 3 : ViPInt16 EdgeReg
 * OUT       
 *            Returns Edge Interrupt Status register.
 * 
 * PARAM 4 : ViPInt16 DAVReg
 * OUT       
 *            Returns Data Available Status register.
 * 
 * PARAM 5 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 6 : ViPInt16 CmdReg
 * OUT       
 *            Returns the command register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
static ViStatus _VI_FUNC hpe1459f_sensRegState_Q(
  ViSession vi,
  ViPInt16 StatusReg,
  ViPInt16 EdgeReg,
  ViPInt16 DAVReg,
  ViInt32 port,
  ViPInt16 CmdReg)
{
    ViStatus errStatus = VI_SUCCESS;
    {
    	*StatusReg = 0;
    
    	*CmdReg = 0;
    
       	_E(viQueryf(vi,"SENS:EVEN:PSUM:EDGE?\n","%hd",EdgeReg) )
    
       	_E(viQueryf(vi,"SENS:EVEN:PSUM:DAV?\n","%hd",DAVReg) )
    
    }
    return errStatus;
}
static ViStatus ( _VI_FUNC * hpe1459I_sensRegState_Q[ hpe1459_MAX_INTERFACES]) (
  ViSession vi,
  ViPInt16 StatusReg,
  ViPInt16 EdgeReg,
  ViPInt16 DAVReg,
  ViInt32 port,
  ViPInt16 CmdReg) =
{
  hpe1459p_sensRegState_Q, /* thisPtr->iointerface = 0 */
  hpe1459f_sensRegState_Q  /* thisPtr->iointerface = 1 */
};

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_sensRegState_Q
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  This function returns conditions monitored by the Status, Edge
 *           interrupt, DAV interrupt, and command registers.
 *           
 *            The lower four bits of the DAV and Edge interrupt registers can
 *           be used to determine if any edge or DAV events have occurred.
 *           Bit 3 (MSB) is port 3, Bit 0 (LSB) is port 0. A '1' in any of
 *           these bit positions indicates an event occurred on that port.
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViPInt16 StatusReg
 * OUT       
 *            Returns the devices Status/Control register.
 * 
 * PARAM 3 : ViPInt16 EdgeReg
 * OUT       
 *            Returns Edge Interrupt Status register.
 * 
 * PARAM 4 : ViPInt16 DAVReg
 * OUT       
 *            Returns Data Available Status register.
 * 
 * PARAM 5 : ViInt32 port
 * IN        
 *            Port number.
 * 
 *      Macro Name          Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_PORT_0          0  Port zero
 *      hpe1459_PORT_1          1  Port one
 *      hpe1459_PORT_2          2  Port two
 *      hpe1459_PORT_3          3  Port three
 * 
 * PARAM 6 : ViPInt16 CmdReg
 * OUT       
 *            Returns the command register for the specified port.
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_sensRegState_Q(
  ViSession vi,
  ViPInt16 StatusReg,
  ViPInt16 EdgeReg,
  ViPInt16 DAVReg,
  ViInt32 port,
  ViPInt16 CmdReg)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_sensRegState_Q" );

    /* Perform Error Checking on Each Parameter */
    /* call the interface specific function */
    errStatus = hpe1459I_sensRegState_Q[ thisPtr->iointerface](
       vi,
       StatusReg,
       EdgeReg,
       DAVReg,
       port,
       CmdReg);

    hpe1459_LOG_STATUS( vi, thisPtr, errStatus );
}

/*-----------------------------------------------------------------------------
 * FUNC    : ViStatus _VI_FUNC hpe1459_setAttribute
 *-----------------------------------------------------------------------------
 * 
 * PURPOSE :  Sets attributes to the interface.  The following attributes are
 *            supported:
 *           
 *            hpe1459_ATTRIB_LOCK_TMO
 *            This timeout is distinct from the hpe1459_timeOut functions
 *           which set timeouts for the underlying VISA commands which
 *           directly communicate with the instrument.  The lock timeout
 *           waits for the instrument to be unlocked by other threads such as
 *           event handlers. If the lock timeout is set to 0, API will wait
 *           indefinitely for the unlock (default).  Value is in
 *           microseconds.
 *            hpe1459_ATTRIB_TRACE_DEST
 *            Trace destination, should be set before Trace level.  It can
 *            be hpe1459_TRAC_EVENTLOG	(NT only),
 *            hpe1459_TRAC_DEBUGSTRING
 *           
 *            hpe1459_ATTRIB_TRACE_LEVEL.  It can be hpe1459_TRAC_ALL or
 *            hpe1459_TRAC_NONE.
 *           
 *            hpe1459_ATTRIB_EVENT_ERROR_UH.  This will change the user
 *           handle passed into an error handler (if installed).  This allows
 *           the error handler to know specifically where an error occurred.
 *           When using this function in multiple threads you need to
 *            protect the value (i.e. the value of the handle will be the
 *           LAST call to this function).
 *           
 *            hpe1459_ATTRIB_CLEAR_CMD:  Valid only for command module
 *           systems.  It sends a IEEE *CLS command.  On other systems it has
 *           no affect.
 *           
 * 
 * PARAM 1 : ViSession vi
 * IN        
 *            Instrument Handle returned from hpe1459_init().
 * 
 * PARAM 2 : ViInt32 attrib
 * IN        
 *            Attribute to set.
 * 
 *      Macro Name                         Value  Description
 *      -----------------------------------------------------------
 *      hpe1459_ATTRIB_LOCK_TMO                0  Card lock timeout
 *      hpe1459_ATTRIB_TRACE_DEST              1  Trace destination
 *      hpe1459_ATTRIB_TRACE_LEVEL             2  Trace level
 *      hpe1459_ATTRIB_EVENT_ERROR_UH          3  Error user handle
 *      hpe1459_ATTRIB_CLEAR_CMD               4  *CLS to cmd module
 * 
 * PARAM 3 : ViInt32 data
 * IN        
 *            Value to set the attribute to.  Card lock timout is in
 *            milliseconds.
 * 
 *      MAX = hpe1459_SA_MAX   2147483647L
 *      MIN = hpe1459_SA_MIN   0
 * 
 * RETURN  :  VI_SUCCESS: No error. Non VI_SUCCESS: Indicates error
 *           condition. To determine error message, pass the return value to
 *           routine "hpe1459_error_message".
 * 
 *-----------------------------------------------------------------------------
 */
ViStatus _VI_FUNC hpe1459_setAttribute(
  ViSession vi,
  ViInt32 attrib,
  ViInt32 data)
{
    ViStatus errStatus = 0;
    struct hpe1459_globals *thisPtr;
    short allow_unlock=0;

    errStatus = viGetAttribute(vi, VI_ATTR_USER_DATA, (ViAddr) &thisPtr);
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_DEBUG_CHK_THIS( vi, thisPtr );

    errStatus=util_card_lock(thisPtr,thisPtr->lock_timeout);

    allow_unlock = 1; 
    if (VI_SUCCESS > errStatus){ hpe1459_LOG_STATUS( vi, 0, errStatus ); }

    hpe1459_CDE_INIT( "hpe1459_setAttribute" );

    /* Perform Error Checking on Each Parameter */
    hpe1459_CHK_LONG_RANGE(data
                          ,hpe1459_SA_MIN
                          ,hpe1459_SA_MAX
                          ,VI_ERROR_PARAMETER3);

    {
    	switch(attrib)
    	{
    		case hpe1459_ATTRIB_LOCK_TMO:
    			thisPtr->lock_timeout = data;
    			break;
    		case hpe1459_ATTRIB_TRACE_DEST:
    			thisPtr->traceDest=(short)data;
    			break;
    		case hpe1459_ATTRIB_TRACE_LEVEL:
    			thisPtr->traceLevel=(short)data;
    			break;
    		case hpe1459_ATTRIB_EVENT_ERROR_UH:
    			if( thisPtr->eventHndlr[hpe1459_EVENT_ERROR])
    				thisPtr->eventAttrib[hpe1459_EVENT_ERROR] = data;
    			else
    			errStatus = VI_ERROR_HNDLR_NINSTALLED;
    			break;
    		case hpe1459_ATTRIB_CLEAR_CMD:
    			if(f_INTERFACE) errStatus = viPrintf(vi,"*CLS\n");
    			break;
    		case 1000: /* diag_event */
    			thisPtr->diag_event= data;
    			break;
    		default:
    			break;
    	}
    }
    
    hpe1459_LOG_STATUS( vi, thisPtr, VI_SUCCESS );
}
