/*cncc4ds.c -  A sequence for 4D chemical shift correlation.
               Franks et al, J. Biomol NMR (2007) 39:107-131

               D. Rice  07/08/08                           */

#include "standard.h"
#include "solidstandard.h"

// Define Values for Phasetables

static int table1[4] = {1,1,1,1};           // phXpresat
static int table2[4] = {0,0,0,0};           // phH90
static int table3[4] = {0,0,0,0};           // phXhx
static int table4[4] = {3,3,3,3};           // phHhx
static int table5[4] = {0,0,0,0};           // ph1Hspnlm
static int table6[4] = {0,1,2,3};           // ph1Y180
static int table7[4] = {0,0,0,0};           // ph1Hcw
static int table8[4] = {0,0,0,0};           // phYxy
static int table9[4] = {0,2,0,2};           // phXxy
static int table10[4] = {0,0,0,0};          // ph2Hspnlm
static int table11[4] = {0,1,2,3};          // phX180
static int table12[4] = {0,0,0,0};          // ph2Hcw
static int table13[4] = {1,3,1,3};          // phXyx
static int table14[4] = {0,0,2,2};          // phYyx
static int table15[4] = {0,0,0,0};          // ph3Hspnlm
static int table16[4] = {0,1,2,3};          // ph2Y180
static int table17[4] = {0,0,0,0};          // phHdarr
static int table18[4] = {0,2,0,2};          // ph1X90
static int table19[4] = {0,1,0,1};          // ph2X90
static int table20[4] = {0,0,0,0};          // phHspnl
static int table21[4] = {0,0,0,0};          // phYpips
static int table22[4] = {1,0,3,2};          // phRec

#define phXpresat t1
#define phH90 t2
#define phXhx t3
#define phHhx t4
#define ph1Hspnlm t5
#define ph1Y180 t6
#define ph1Hcw t7
#define phYxy t8
#define phXxy t9
#define ph2Hspnlm t10
#define phX180 t11
#define ph2Hcw t12
#define phXyx t13
#define phYyx t14
#define ph3Hspnlm t15
#define ph2Y180 t16
#define phHdarr t17
#define ph1X90 t18
#define ph2X90 t19
#define phHspnl t20
#define phYpips t21
#define phRec t22

pulsesequence() {

// Define Variables and Objects and Get Parameter Values

   double duty; 

   int decmode = getval("decmode");
   int decmodem = getval("decmodem");
   int decmodey = getval("decmodey");

   double d22 = getval("d2")/2.0 - getval("pwY180")/2.0; 
   if (d2 < 0.0) d22 = 0.0; 
   double d32 = getval("d3")/2.0 - getval("pwX180")/2.0;
   if (d32 < 0.0) d32 = 0.0;
   double d42 = getval("d4")/2.0 - getval("pwY180")/2.0;
   if (d42 < 0.0) d42 = 0.0;
   double mix = getval("mix");

   CP hx = getcp("HX",0.0,0.0,0,1); 
   strncpy(hx.fr,"dec",3);
   strncpy(hx.to,"obs",3);
   putCmd("frHX='dec'\n");
   putCmd("toHX='obs'\n");

   CP xy = getcp("XY",0.0,0.0,0,1); 
   strncpy(xy.fr,"obs",3);
   strncpy(xy.to,"dec2",4);
   putCmd("frXY='obs'\n");
   putCmd("toXY='dec2'\n");

   CP yx = getcp("YX",0.0,0.0,0,1); 
   strncpy(yx.fr,"dec2",4);
   strncpy(yx.to,"obs",3);
   putCmd("frYX='dec2'\n");
   putCmd("toYX='obs'\n");

   MPSEQ spnlm;
   if (decmodem>1) { spnlm = getspnl("spnlmH",0,0.0,0.0,0,1);
      strncpy(spnlm.ch,"dec",3);
      putCmd("chHspnlm ='dec'\n");
   }

   MPSEQ spnl;
   if (decmode>1) { spnl = getspnl("spnlH",0,0.0,0.0,0,1);
      strncpy(spnl.ch,"dec",3);
      putCmd("chHspnl ='dec'\n");
   }

   MPSEQ pips; 
   if (decmodey>1) {pips = getpipsxy("pipsY",0,0.0,0.0,0,1); 
      strncpy(pips.ch,"dec2",4);
      putCmd("chHpips ='dec2'\n");
   }

// Dutycycle Protection

   duty = 4.0e-6 + getval("pwH90") + getval("tHX") + getval("ad") +
            getval("rd") + at;

   duty = duty/(duty + d1 + 4.0e-6);
   if (duty > 0.1) {
      printf("Duty cycle %.1f%% >10%%. Abort!\n", duty*100);
      psg_abort(1);
   }

// Set Phase Tables

   settable(phXpresat,4,table1);
   settable(phH90,4,table2);
   settable(phXhx,4,table3);
   settable(phHhx,4,table4);
   settable(ph1Hspnlm,4,table5);
   settable(ph1Y180,4,table6);
   settable(ph1Hcw,4,table7);
   settable(phYxy,4,table8);
   settable(phXxy,4,table9);
   settable(ph2Hspnlm,4,table10);
   settable(phX180,4,table11);
   settable(ph2Hcw,4,table12);
   settable(phXyx,4,table13);
   settable(phYyx,4,table14);
   settable(ph3Hspnlm,4,table15);
   settable(ph2Y180,4,table16);
   settable(phHdarr,4,table17);
   settable(ph1X90,4,table18);
   settable(ph2X90,4,table19);
   settable(phHspnl,4,table20);
   settable(phYpips,4,table21);
   settable(phRec,4,table22);
   setreceiver(phRec);

// Begin Sequence

   txphase(phXhx); decphase(phH90); dec2phase(phYxy); 
   obspwrf(getval("aXhx")); decpwrf(getval("aH90")); dec2pwrf(getval("aYxy"));
   obsunblank(); decunblank(); _unblank34();
   delay(d1);
   sp1on(); delay(2.0e-6); sp1off(); delay(2.0e-6);

// X Hard-Pulse Presat

   if ((getval("pwXpresat")) > 0.0) {
      obspwrf(getval("aXpresat"));
      rgpulse(getval("pwXpresat"),phXpresat,2.0e-6,2.0e-6);
      txphase(phXhx); 
      obspwrf(getval("aXhx"));
      obsunblank();
   }

// H to X Cross Polarization

   decrgpulse(getval("pwH90"),phH90,0.0,0.0);
   decphase(phHhx);
   _cp_(hx,phHhx,phXhx);
   txphase(phXxy);

// X F1 Period with a Y Refocussing Pulse

   if (d22 > 0.0) {
      if (decmodem>0){ decpwrf(getval("aHspnlm")); decon(); if (decmodem>1){ _mpseqon(spnlm,ph1Hspnlm);}}
      dec2phase(ph1Y180);
      obspwrf(getval("aXxy")); dec2pwrf(getval("aY180"));
      delay(d22);
      dec2rgpulse(getval("pwY180"),ph1Y180,0.0,0.0);
      dec2phase(phYxy);
      dec2pwrf(getval("aYxy"));
      dec2unblank();
      delay(d22);
      if (decmodem>0){ decoff(); if (decmodem>1){ _mpseqoff(spnlm);}}
   }

// X to Y Cross Polarization

   decphase(ph1Hcw);
   decpwrf(getval("aHcw"));
   decon();
   _cp_(xy, phXxy, phYxy);
   txphase(phXyx); dec2phase(phYyx);
   decoff();

// Y F2 Period with an X Refocussing Pulse

   if (d32 > 0.0) {
      if (decmodem>0){ decpwrf(getval("aHspnlm")); decon(); if (decmodem>1){ _mpseqon(spnlm,ph2Hspnlm);}}
      txphase(phX180);
      obspwrf(getval("aX180")); dec2pwrf(getval("aYyx")); 
      delay(d32);
      rgpulse(getval("pwX180"),phX180,0.0,0.0);
      txphase(phXyx);
      obspwrf(getval("aXyx"));
      obsunblank();
      delay(d32);
      if (decmodem>0){ decoff(); if (decmodem>1){ _mpseqoff(spnlm);}}
   }

// Y to X Cross Polarization

   decphase(ph2Hcw);
   decpwrf(getval("aHcw"));
   decon();
   _cp_(yx,phYyx,phXyx);
   dec2phase(phYpips);
   obspwrf(getval("aX90")); dec2pwrf(getval("aYpips"));
   decoff();

// X F3 Period with a Y Refocussing Pulse 

   if (d42 > 0.0) {      
      if (decmodem>0){ decpwrf(getval("aHspnlm")); decon(); if (decmodem>1){ _mpseqon(spnlm,ph3Hspnlm);}}
      txphase(ph1X90); dec2phase(ph2Y180);
      dec2pwrf(getval("aY180"));
      delay(d42);
      dec2rgpulse(getval("pwY180"),ph2Y180,0.0,0.0);
      dec2phase(phYpips);
      dec2pwrf(getval("aYpips"));
      dec2unblank();
      delay(d42);
      if (decmodem>0){ decoff(); if (decmodem>1){ _mpseqoff(spnlm);}}
   }

// X Mixing with DARR

   decphase(phHdarr);
   decpwrf(getval("aHdarr"));
   decon();
   rgpulse(getval("pwX90"),ph1X90,0.0,0.0);
   delay(mix);
   rgpulse(getval("pwX90"),ph2X90,0.0,0.0);
   decoff();

// Begin Acquisition with Y PIPS Decoupling

   if (decmode>0){ decpwrf(getval("aHspnl")); decon(); if (decmode>1){ _mpseqon(spnl,phHspnl);}}
   if (decmodey>0){ dec2pwrf(getval("aYpips")); dec2on(); if (decmodey>1){ _mpseqon(pips,phYpips);}}  
   obsblank(); _blank34();
   delay(getval("rd"));
   startacq(getval("ad"));
   acquire(np, 1/sw);
   endacq();
   if (decmode>0){ decoff(); if (decmode>1){ _mpseqoff(spnl);}}
   if (decmodey>0){ dec2off(); if (decmodey>1){ _mpseqoff(pips);}}
   obsunblank(); decunblank(); _unblank34();
}

