/*nccdqm3ds.c -  A sequence for 3D chemical shift correlation.
               Reinstra et al JACS (2000) 122, 10979-10990

               D. Rice  07/08/08                           */

#include "standard.h"
#include "solidstandard.h"

// Define Values for Phasetables

static int table1[4] = {1,1,1,1};           // phYpresat
static int table2[4] = {1,1,1,1};           // phH90
static int table3[4] = {0,2,0,2};           // phYhy
static int table4[4] = {0,0,0,0};           // phHhy
static int table5[4] = {0,0,0,0};           // ph1Hspnlm
static int table6[4] = {0,0,0,0};           // phX180
static int table7[4] = {0,0,0,0};           // phHcw
static int table8[16] = {0,0,0,0,0,0,0,0,
                         2,2,2,2,2,2,2,2};  // phXyx
static int table9[4] = {0,0,0,0};           // phYyx
static int table10[4] = {0,0,0,0};          // ph2Hspnlm
static int table11[4] = {0,0,0,0};          // phY180
static int table12[4] = {0,0,0,0};          // phHmix
static int table13[4] = {1,1,1,1};          // ph1X90
static int table14[4] = {0,0,0,0};          // phXspc5
static int table15[8] = {0,0,1,1,2,2,3,3};  // ph2X90
static int table16[4] = {0,0,0,0};          // phHspnl
static int table17[4] = {0,0,0,0};          // phYpips
static int table18[16] = {0,2,1,3,2,0,3,1,
                         2,0,3,1,0,2,1,3};  // phRec

#define phYpresat t1
#define phH90 t2
#define phYhy t3
#define phHhy t4
#define ph1Hspnlm t5
#define phX180 t6
#define phHcw t7
#define phXyx t8
#define phYyx t9
#define ph2Hspnlm t10
#define phY180 t11
#define phHmix t12
#define ph1X90 t13
#define phXspc5 t14
#define ph2X90 t15
#define phHspnl t16
#define phYpips t17
#define phRec t18

pulsesequence() {

// Define Variables and Objects and Get Parameter Values

   double duty;

   int decmode = getval("decmode");
   int decmodem = getval("decmodem");
   int decmodey = getval("decmodey");

   double d22 = getval("d2")/2.0 - getval("pwX180")/2.0;
   if (d2 < 0.0) d22 = 0.0;
   double d32 = getval("d3")/2.0 - getval("pwY180")/2.0;
   if (d32 < 0.0) d32 = 0.0;

   double tZF = getval("tZF");

   CP hy = getcp("HY",0.0,0.0,0,1);
   strncpy(hy.fr,"dec",3);
   strncpy(hy.to,"dec2",4);
   putCmd("frHY='dec'\n");
   putCmd("toHY='dec2'\n");

   CP yx = getcp("YX",0.0,0.0,0,1);
   strncpy(yx.fr,"dec2",4);
   strncpy(yx.to,"obs",3);
   putCmd("frYX='dec2'\n");
   putCmd("toYX='obs'\n");

   MPSEQ spnlm;
   if (decmodem>1) { spnlm = getspnl("spnlmH",0,0.0,0.0,0,1);
      strncpy(spnlm.ch,"dec",3);
      putCmd("chHspnlm ='dec'\n");
   }

   MPSEQ spnl;
   if (decmode>1) { spnl = getspnl("spnlH",0,0.0,0.0,0,1);
      strncpy(spnl.ch,"dec",3);
      putCmd("chHspnl ='dec'\n");
   }

   MPSEQ pips;
   if (decmodey>1) {pips = getpipsxy("pipsY",0,0.0,0.0,0,1);
      strncpy(pips.ch,"dec2",4);
      putCmd("chHpips ='dec2'\n");
   }

   MPSEQ spc5 = getspc5("spc5X",0,0.0,0.0,0,1);
   strncpy(spc5.ch,"obs",3);
   putCmd("chXspc5 ='obs'\n");

// Dutycycle Protection

   duty = 4.0e-6 + getval("pwH90") + getval("tHX") + getval("ad") +
            getval("rd") + at;

   duty = duty/(duty + d1 + 4.0e-6);
   if (duty > 0.1) {
      printf("Duty cycle %.1f%% >10%%. Abort!\n", duty*100);
      psg_abort(1);
   }

// Set Phase Tables

   settable(phYpresat,4,table1);
   settable(phH90,4,table2);
   settable(phYhy,4,table3);
   settable(phHhy,4,table4);
   settable(ph1Hspnlm,4,table5);
   settable(phX180,4,table6);
   settable(phHcw,4,table7);
   settable(phXyx,16,table8);
   settable(phYyx,4,table9);
   settable(ph2Hspnlm,4,table10);
   settable(phY180,4,table11);
   settable(phHmix,4,table12);
   settable(ph1X90,4,table13);
   settable(ph1X90,4,table14);
   settable(ph2X90,8,table15);
   settable(phHspnl,4,table16);
   settable(phYpips,4,table17);
   settable(phRec,16,table18);
   setreceiver(phRec);

// Begin Sequence

   txphase(phX180); decphase(phH90); dec2phase(phYhy);
   obspwrf(getval("aX180")); decpwrf(getval("aH90")); dec2pwrf(getval("aYhy"));
   obsunblank(); decunblank(); _unblank34();
   delay(d1);
   sp1on(); delay(2.0e-6); sp1off(); delay(2.0e-6);

// Y Hard-Pulse Presat

   if ((getval("pwYpresat")) > 0.0) {
      obspwrf(getval("aYpresat"));
      rgpulse(getval("pwYpresat"),phYpresat,2.0e-6,2.0e-6);
      txphase(phYhy);
      obspwrf(getval("aYhy"));
      obsunblank();
   }

// H to X Cross Polarization

   decrgpulse(getval("pwH90"),phH90,0.0,0.0);
   decphase(phHhy);
   _cp_(hy,phHhy,phYhy);
   txphase(phXyx); dec2phase(phYyx);

// X F1 Period with an X Refocussing Pulse

   if (d22 > 0.0) {
      if (decmodem>0){ decpwrf(getval("aHspnlm")); decon(); if (decmodem>1){ _mpseqon(spnlm,ph1Hspnlm);}}
      txphase(phX180);
      obspwrf(getval("aX180")); dec2pwrf(getval("aYyx"));
      delay(d22);
      rgpulse(getval("pwX180"),phX180,0.0,0.0);
      txphase(phXyx);
      obspwrf(getval("aXyx"));
      decunblank();
      delay(d22);
      if (decmodem>0){ decoff(); if (decmodem>1){ _mpseqoff(spnlm);}}
   }

// Y to X Cross Polarization

   decphase(phHcw);
   decpwrf(getval("aHcw"));
   decon();
   _cp_(yx,phYyx,phXyx);
   dec2phase(phYpips);
   obspwrf(getval("aX90")); dec2pwrf(getval("aYpips"));
   decoff();

// X F2 Period with a Y Refocussing Pulse

   if (d32 > 0.0) {
      if (decmodem>0){ decpwrf(getval("aHspnlm")); decon(); if (decmodem>1){ _mpseqon(spnlm,ph2Hspnlm);}}
      txphase(ph1X90); dec2phase(phY180);
      dec2pwrf(getval("aY180"));
      delay(d32);
      dec2rgpulse(getval("pwY180"),phY180,0.0,0.0);
      dec2phase(phYpips); 
      dec2pwrf(getval("aYpips"));
      dec2unblank();
      delay(d32);
      if (decmodem>0){ decoff(); if (decmodem>1){ _mpseqoff(spnlm);}}
   }

// X Mixing with SPC5 on X and CW on H

   decphase(phHmix);
   decpwrf(getval("aHmix"));
   rgpulse(getval("pwX90"),ph1X90,0.0,0.0);
   delay(tZF);
   txphase(phXspc5);
   obspwrf(getval("aXspc5"));
   obsunblank();
   decon();
   _mpseq(spc5,phXspc5);
   decoff();
   txphase(ph2X90);
   obspwrf(getval("aX90"));
   obsunblank();
   delay(getval("tZF"));
   rgpulse(getval("pwX90"),ph2X90,0.0,0.0);

// Begin Acquisition with Y PIPS Decoupling

   if (decmode>0){ decpwrf(getval("aHspnl")); decon(); if (decmode>1){ _mpseqon(spnl,phHspnl);}}
   if (decmodey>0){ dec2pwrf(getval("aYpips")); dec2on(); if (decmodey>1){ _mpseqon(pips,phYpips);}}
   obsblank(); _blank34();
   delay(getval("rd"));
   startacq(getval("ad"));
   acquire(np, 1/sw);
   endacq();
   if (decmode>0){ decoff(); if (decmode>1){ _mpseqoff(spnl);}}
   if (decmodey>0){ dec2off(); if (decmodey>1){ _mpseqoff(pips);}}
   obsunblank(); decunblank(); _unblank34();
}

