/*sammy2d.c -  A sequence to provide correlation between the
               chemical-shift and 1H-dipolar interaction using
               a pair of magic-sandwich spinlocks (SAMMY).

               Uses a modified SAMMY sequence based on a personal
               communication by Alex Nevzorov.  Changes from the paper
               are:
               1.  invert the X phase at the center of each MS cycle.
                   That means the X durations are 6.0*pwX90, 12.0*pwX90, 6.0*pwX90.
               2.  set each of four H spinlocks = 3.5*pwH90 and the two delays
                   = 3.0*pwH90 each. The four explicit pwH90's fill 2 MS cycles.
               3.  del1 and del2 remain the same but should be zero.
    
      Alexander A. Nevzorov and Stanley J. Opella, J. Magn. Reson.
      164 (2003) 182-186.

               D. Rice 6/23/07                                             */

#include "solidstandard.h"
#include <standard.h>

// Define Values for Phasetables

static int table1[4] = {0,2,0,2};           // phH90
static int table2[4] = {3,3,0,0};           // phXhx
static int table3[4] = {3,3,3,3};           // phHhx
static int table4[4] = {1,1,1,1};           // phHsmyd
static int table5[4] = {3,3,0,0};           // phXsmyo 
static int table6[4] = {1,3,2,0};           // phRec

#define phH90 t1
#define phXhx t2
#define phHhx t3
#define phHsmyd t4
#define phXsmyo t5
#define phRec t6

pulsesequence() {

//Define Variables and Objects and Get Parameter Values 

   double duty;

   CP hx = getcp("HX",0.0,0.0,0,1);
   strncpy(hx.fr,"dec",3);
   strncpy(hx.to,"obs",3);
   putCmd("frHX='dec'\n");
   putCmd("toHX='obs'\n");

   MPSEQ sd = getsammyd("smydH",0,0.0,0.0,0,1);
   strncpy(sd.ch,"dec",3);
   putCmd("chHsmyd='dec'\n");

   MPSEQ so = getsammyo("smyoX",0,0.0,0.0,0,1);
   strncpy(so.ch,"obs",3);
   putCmd("chXsmyo='obs'\n");

   DSEQ dec = getdseq("H");

// Dutycycle Protection

   duty = 4.0e-6 + getval("pwH90") + getval("tHX") + d2 + getval("ad") +
            getval("rd") + at;

   duty = duty/(duty + d1 + 4.0e-6);
   if (duty > 0.1) {
      printf("Duty cycle %.1f%% >10%%. Abort!\n", duty*100);
      psg_abort(1);
   }

// Set Phase Tables

   settable(phH90,4,table1);
   settable(phXhx,4,table2);
   settable(phHhx,4,table3);
   settable(phHsmyd,4,table4);
   settable(phXsmyo,4,table5);
   settable(phRec,4,table6);
   setreceiver(phRec);
    
// Begin Sequence

   txphase(phXhx); decphase(phH90);
   obspwrf(getval("aXhx")); decpwrf(getval("aH90"));
   obsunblank(); decunblank(); _unblank34();
   delay(d1);
   sp1on(); delay(2.0e-6); sp1off(); delay(2.0e-6);

// H to X Cross Polarization

   decrgpulse(getval("pwH90"),phH90,0.0,0.0);
   decphase(phHhx);
   _cp_(hx,phHhx,phXhx);

// SAMMY Spinlocks on X and H

   _mpseqon(sd,phHsmyd); _mpseqon(so,phXsmyo);
   delay(d2);
   _mpseqoff(sd); _mpseqoff(so);

// Begin Acquisition

   obsblank(); _blank34();
   _dseqon(dec);
   delay(getval("rd"));
   startacq(getval("ad"));
   acquire(np, 1/sw);
   endacq();
   _dseqoff(dec);
   obsunblank(); decunblank(); _unblank34();
}
