#  SolidsPack Version 8/14/07
#  SIMPLEXER: simplexOptions.tcl (version 4)
###########################################################################
#
# MODIFICATIONS
#   12/21/04, ceb:  Adapt for use on Inova Systems  
#   2/14/06, ceb:   Modify to allow bypass of GUI
#               - add section that copies a method into an experiment
#               - add save method button
#               - delete verbose button in order to make room
#
#   3/06, ceb:   Add ability to load/store methods from this panel
#                Check load method against curpar for valid parameters
#                Clean up abort mechanism
#
#   8/06, ceb:   Extend getAutoFile to allow for queues
#
################################################################################

#================================================================================
# Menu to enter Simplex Options
# and Generate Initial Simplex Guesses
# output results to file for use by SIMPLEXER
# 
#   FUNCTIONS
#
#    popup:   Utility Popup Message Generator
#    start:   Check the data, write file if OK
#    testRange:  Check to see that parameters are within specified limits
#    seed: Generate the initial n+1 guesses for simplex
#    writeFileGo: write the data and exit
#    abort: Write file with abort msg
#  
#    VARIABLES
#
#      Name:  Array of parameter names
#      Val:   Array of initial values, obtained from acq file
#      Type:  Data type, eg time, phase ...
#      Format: Output format string
#      Units:  acqpar file style units
#      nDim:   number of variables to optimize
#      p:      Matrix (nDim+1)x(nDim) of guesses. The simplex
#      nIter:  Maximum number of iterations allowed.
#      Range:  Array of %variation (+/-) allowed for parameters,
#      Min,Max:  Arrays containing limits on the values of the parameters as dictated by curpar
#      sMin, sMax:  Arrays holding user set limits
#      fTol:   Fractional variation of the nDim + 1 error values that must be met.
#      method:  Method of determining the figure of merit.
#      vb:     If 1, both SIMPLEXER and simplexer will dump data on each pass.
#              If 0, they will be silent.
#
#
#     The first vertex contains the parameter values contained in the
#     acquisition panel when SIMPLEXER is called
#
#     The subsequent n vertices are generated by randomly perturbing
#     a single parameter away from its initial value in each
#     vertex and using initial values for the remaining n-1
#     parameters. The code was originally written so that every parameter
#     in vertices after the first was perturbed. You get better
#     coverage this way but the current method generally gives 
#     faster convergence
#
#     INPUTS:  File containing simplex variables and acqpar file. 
#              These are specified by the calling program
#
#     OUTPUTS: communication file, name specified by caller, contains simplex data
#              tempMinMax, used by caller to check whether guess violate min/max
#              limits specified in acqpar file
#=====================================================================================  

global ssColor
#set ssColor #24c3c3
set ssColor gray75
set hdColor red
set textColor blue

set systemButtonText $textColor


tk_setPalette $ssColor

proc popup msg {
    tk_messageBox -type ok -message $msg
}

###########################################################################
#
#                   Change log
# 2/06, ceb
#    proc start
#         Check entries for errors
#         Convert time variables to correct scale 
#         Test values against limits
#         Write the tempCommun file
#
# 3/06, ceb
#
# Add capability to load/save optimization data at go time
#       - supports launching optimization w/out GUI
#       - allows load/save of all details of optimization setup 
#
#
############################################################################


#======================================================
# parIndex returns the index number of a variable
# in aqpArr or -1 if the variable is not in aqpArr
#======================================================

proc parIndex  par {
   global Name
   set n [array size Name ]
   for {set i 0 } {$i < $n} {incr i 1} {
       if  { [string equal $Name($i)  $par] } {
           return $i
      }
   }
   return -1
}

proc parseCurpar curparFile {     
    
    global Name Val Max Min parmin parmax parstep
    set n [array size Name]
    set fd [open $curparFile  r]
    
#number of parameter traits, number of entries in parameter array
    
    for {set i 0} {$i < $n } {incr i 1} {
        set found($i) 0
    }
    
    while { [gets $fd line] >= 0 } {
    	 
         if {  [llength $line]  < 1} {
                continue
           }
        #
        # check whether this is a keeper
        #
       
        set e0 [lindex $line 0];                    #name
        set idx  [parIndex  $e0 ]
        if  { $idx != -1 }  {
            set found($idx) 1
            set e1 [lindex $line 1];                 #subtype
            set e2 [lindex $line 2];                 #basic type
	    if { $e2 == 2 } {
	    	set strError $e2
		    append strError " is a string! Cannot optimize"
		    popup $strError
		    close $fd
		    return
	    }
            set e3 [lindex $line 3];                 #maxvalue
            set e4 [lindex $line 4];                 #minvalue
            set e5 [lindex $line 5];                 #stepsize
            set e8 [lindex $line 8];                 #protection word
                           
        #crack protect word
        set val 32768 
        for { set i 0 } {$i < 16 } {incr i 1} {
            set lProt($i) 0
        }
        for { set i 15 } {$i >= 0 } {incr i -1} {
            if {$e8 >= $val} {
                set lProt($i) 1
                incr e8 -$val
            } 
            set val [expr $val/2]
        }
        # check bit 13
        set k 13
        if { $lProt($k) == 1 } {
            set Max($idx)   [ lindex $parmax [ expr $e3 - 1] ]
            set Min($idx)   [ lindex $parmin [ expr $e4 - 1] ]
        }  else {
            set Max($idx) $e3
            set Min($idx) $e4
        }
        gets $fd line
        if { [lindex $line 0] !=  1} {
           set  strMsg $e0
           append strMsg "  Is an array!"
           popup strMsg
	       close $fd
	       return
        } else {
           set Val($idx) [lindex $line 1]
        }
        gets $fd line
    } else {
        if { [lindex $line 2] == 2 } {
          gets $fd line
	      set n [lindex $line 0]
          for {set i 0} {$i < $n} {incr i 1} {
             gets $fd line
          } 
	 } else {
	   gets $fd line
	   gets $fd line
	 }
   }
  }
   close $fd  
   
   # All elements in Name found in curpar?
   
   set configError 0
   set strError "Error in simplexOptions: The following elements were not found in curpar: \n"
   for {set i 0} {$i < $n } {incr i 1 } {
      if {$found($i) == 0 } {
        set configError 1
        append strError $aqpArr($i,$name)
        append strError \n
      }
    }
    if {$configError == 1} {
        popup $strMsg
        abort $expDir
    }       
}
  
proc parseConpar { } {

    set    conparFile "/vnmr/conpar"
    global parmin parmax parstep

    set fd [open $conparFile r]

    while { [gets $fd line] >= 0 } {

      if { [lindex $line 0] == "parstep" } {
         gets $fd line
       for {set i 1} {$i < [llength $line] } {incr i 1} {
             lappend parstep  [lindex $line $i]
        }
         gets $fd line
         continue
      }

        if { [lindex $line 0] == "parmax" } {
            gets $fd line
            for {set i 1} {$i < [llength $line] } {incr i 1} {
                 lappend parmax  [lindex $line $i]
            }
            gets $fd line
            continue
          }

       if { [lindex $line 0] == "parmin" } {
          gets $fd line
         for {set i 1} {$i < [llength $line] } {incr i 1} {
               lappend parmin  [lindex $line $i]
       }
         gets $fd line
        continue
      }
    }
    close $fd
}

proc start fileName {

global p fTol nDim Val Name Type Tolerance nIter method Min Max bSave
   
    if { $fTol <= 0.0 } {
        popup [list "Tolerance is Less Than/Equal  Zero"]
        return
    }

    if {$fTol >= 100.0} {
        popup [list "Tolerance is Greater Than/Equal  100"]
        return
    }
 
    if { $nIter < 10} {
        popup "Nuber of Iterations < 10"
        return
    }
    
    set iTest [testRange]
    if {$iTest == 0} {
       set iTest [seed] 
       if {$iTest == 0} {
           writeFileGo $fileName
           if {$bSave == 1 } {
            set bSave 0
            popup "File Saved"
            return
           }
           exit
       }
    }
}


############################################################
# proc testRange
#
#     See whether user specified limits for the parameters
#     are consistent with curpar parameter limits
#
############################################################

proc testRange { } {

global nDim Name Min Max Val sMin sMax

  
    for {set i 0} {$i < $nDim } {incr i 1} {
        if { $sMin($i) < $Min($i)} {
            set strError "Minimum for  "
            append strError $Name($i) " is Less Than Allowed Minimum"
            popup $strError
            set iResult 1
            return $iResult
        }
        if { $sMin($i) >  $Max($i)} {
            set strError "Minimum for  "
            append strError $Name($i) " is Greater Than Allowed Maximum"
            popup $strError
            set iResult 1
            return $iResult
       }
        if { $sMax($i) < $Min($i)} {
            set strError "Maximum  for  "
            append strError $Name($i) " is Less Than Allowed Minimum"
            popup $strError
            set iResult 1
            return $iResult
        }
        if { $sMax($i) >  $Max($i)} {
            set strError "Maximum for  "
            append strError $Name($i) " is Greater Than Allowed Maximum"
            popup $strError
            set iResult 1
            return $iResult
       }
       
       if {$Val($i) > $sMax($i) } {
          popup [list "Start value for " $Name($i) " is greater than Set Value for Maximum"]
          set iResult 1
          return $iResult 
       }
       
       if {$Val($i) < $sMin($i) } {
          popup [list "Start value for " $Name($i) " is less than Set Value fo  Minimum"]
          set iResult 1
          return $iResult 
       } 
   } 
     
     set iResult 0
     return $iResult    
}

#
# Produce nDim + 1 initial guesses
# and place them  in matrix p
#

proc seed { } {
    global p fTol nDim Name Val Range Type Min Max bSeed
    
    #
    # populate simplex p with initial values for angles
    # Either 1 or all elements from second row to end are
    # initialized to random variable depending on bSeed
    #
    
    set xx [clock clicks]
    set xx [expr $xx/100000]
    expr srand($xx)
    set initVal 0.0;
    set halfpi 1.5708    
    
     for {set i 0} {$i < [expr $nDim + 1]} { incr i 1 } {
        for {set j 0 } {$j < $nDim} {incr j 1} {
           set p($i$j) [ theta $j $Val($j) ]
        }
     }
    
    if { $bSeed == "Single" } {
        for {set i 1} {$i < [expr $nDim + 1]} { incr i 1 } {
            set j [expr $i - 1]
            set r [expr sin( [expr 31403.2* [expr rand()]] ) ]
            set p($i$j) [expr $halfpi*$r]
        }
    } else {
        for {set i 1}  {$i < [expr $nDim + 1]}  { incr i 1 } {
            for  {set j 0 } { $j < [expr $nDim + 1] } {incr j 1} {
                set r [expr sin( [expr 31403.2* [expr rand()]] ) ]
                set p($i$j) [expr $halfpi*$r]
            }
        }
   }
    set iResult 0
    return $iResult
}

#
# Determine angle for specific value of parameter
#

proc theta {idx value}  {
    
    global sMax sMin;
    set Max $sMax($idx)
    set Min $sMin($idx)
    set t1 [expr $value - $Min]
    set t2 [expr $Max - $Min]
    set t1 [expr double($t1)/double($t2)]
    set result [expr asin($t1)]
    return $result 
}

 
proc writeFileGo file {

    global method nDim nIter Min Max fTol p vb sMin sMax bSave saveFile Name Val
   
    set fd [open $file w]
    puts $fd $vb
    puts $fd $method
    puts $fd "INIT"
    puts $fd "INIT"
    puts $fd $nDim
    puts $fd $nIter
    puts $fd $fTol
    puts $fd 0
    
    #
    # y
    #
    
    for {set i 0} {$i < [expr $nDim + 1] } {incr i 1} {
        puts -nonewline $fd 0.0
        puts -nonewline $fd " "
    }
    puts  $fd \t
    
    #
    # p
    #
    
    for {set i 0} {$i < [expr $nDim +1] } {incr i 1} {
        for {set j 0} {$j < $nDim} {incr j 1} {
            puts -nonewline $fd $p($i$j)
            puts -nonewline $fd " "
        }
        puts $fd \t
    }
    
    #
    # psum and ptry, dummies
    #
    
    for {set i 0} {$i <  $nDim } {incr i 1} {
        puts -nonewline $fd 0.0
        puts -nonewline $fd " "
    }
    puts  $fd \t
    
    for {set i 0} {$i < $nDim } {incr i 1} {
        puts -nonewline $fd 0.0
        puts -nonewline $fd " "
    }
    puts  $fd \t
    
    #
    # inhi ihi ilo ytry ysave
    #
    
    puts $fd 0
    puts $fd 0
    puts $fd 0
    puts $fd 0
    puts $fd 0
    
    close $fd
    
    #
    # if bSave, copy tempCommun to Simplexer directory
    #   
    
    #
     if {$bSave == 1 } {
        set idx [string last "s" $file]
        set methodDir [string range $file 0 $idx]
        set methodFile [file join [file join $methodDir "Simplexer"] $saveFile]
        file copy -force -- $file $methodFile
     }
    
    #
    # Put user-selected limits in simplex_vars file
    #
  
    
    set idx [string last "/" $file]
    set fileMinMax [string range $file 0 $idx ]
    append fileMinMax "simplex_vars"
   
   
#    set fd [open $fileMinMax r]
#    gets $fd line
#    set strTem [lindex $line 0]
#    set iCount 0
#    while { [gets $fd line] != -1 } {
#       set n($iCount) [lindex $line 0]
#       set v($iCount) [lindex $line 1]
#       incr iCount 1
#    }
    
 #   close $fd

    
    set fd [open $fileMinMax w]
    puts $fd "simplex_vars"
    for {set i 0} {$i < $nDim} {incr i 1} {
        puts -nonewline $fd $Name($i)\t
        puts -nonewline $fd $Val($i)\t
        puts -nonewline $fd $sMin($i)\t
        puts  $fd $sMax($i) 
    }
 
    close $fd
    
    #
    # if bSave, make a copy of simplex_vars
    #
    
    if {$bSave == 1 } {
        append methodFile "_vars"
        file copy -force -- $fileMinMax $methodFile
    }  
}

#======================================================
# Read a method file
#======================================================

proc getAutoFile { expFile methodName } {

#
# copy files from Simplexer/methods directory to current exp
#
   
   set tempCommunFile [file join $expFile "tempCommun"]
   set varsFile [file join $expFile "simplex_vars"]
   
   set idx [string last "/" $expFile]
   set idx [expr $idx - 1]
   
   set tempCommunSourceFile [string range $expFile 0 $idx]
   set tempCommunSourceFile [file join [file join [file join $tempCommunSourceFile "Simplexer/methods"] $methodName ] $methodName]
   set varsSourceFile $tempCommunSourceFile
   append varsSourceFile "_vars"
   
   if { [file exists $tempCommunSourceFile] != 1 } {
      popup [list $tempCommunSourceFile " Not Found!"]
      exit
   } 
   if { [file exists $varsSourceFile] != 1 } {
      popup [list $varsSourceFile " Not Found!"]
      exit
   }   
   
   file copy -force -- $tempCommunSourceFile $tempCommunFile 
   file copy -force -- $varsSourceFile $varsFile 
   exit
}


#
# Dump directory into file browser
#

proc loadDir  { wFrame methodDir} {

   global strDir
   bind $wFrame <Double-1>  {
      set strDir [selection get] 
   }
   
   list delete 0 end
   foreach i [lsort [glob -nocomplain -directory $methodDir *]] {
      if { [file isdirectory $i] } {
          $wFrame insert end [file tail $i]
      }
   }
}

#
#  Launch a child file browser
#

proc launchChild { methodDir } {
   global  strDir ssColor
   toplevel .q
   wm title .q "Methods"
   wm geometry .q 20x20

   
   frame .q.buttonFrame
   pack .q.buttonFrame -side top -anchor w
   button .q.buttonFrame.button -text "OK" -command { destroy .q }
   pack .q.buttonFrame.button -side left

   frame .q.lf
   pack .q.lf -side top

   listbox .q.lf.lb -yscroll ".q.lf.scroll set" \
                         -width 20 -height 20 -setgrid 1
                                             
   scrollbar .q.lf.scroll  -command  ".q.lf.lb yview"
   pack  .q.lf.lb -side left -anchor w 
   pack .q.lf.scroll  -side right  -expand yes -fill y

   frame .q.r
   label .q.r.l -text File
   entry .q.r.e -textvariable strDir

   pack .q.r -side top
   pack .q.r.l .q.r.e -side left
   loadDir .q.lf.lb $methodDir
}


proc store { communfile } {
    global strDir bSave saveFile
    set idx [string last "/" $communfile]
    incr idx -1
    set tFile [string range $communfile 0 $idx]
    set idx [string last "/" $tFile]
    incr idx -1
    set tFile [string range $tFile 0 $idx]

    set initDir [file join $tFile "Simplexer/methods"]
    launchChild $initDir
    tkwait window .q
    if { $strDir eq "" } {
      popup "No File Selected "
      return
    }
    set strDir [file join $initDir $strDir]
    if { [file exists $strDir] != 1 } {
      file mkdir $strDir
    }
    set idx [string last "/" $strDir]
    incr idx 1
    set tFile [string range $strDir $idx end]
    set saveFile [ file join $strDir $tFile]
    set bSave 1
    start $communfile
}

proc load { file } {

    global saveFile method nIter fTol Max Min strDir
 
# build path to method file
   
    set idx [string last "/" $file]
    incr idx -1
    set tFile [string range $file 0 $idx]
    set idx [string last "/" $tFile]
    incr idx -1
    set tFile [string range $tFile 0 $idx]
    set initDir [file join $tFile "Simplexer/methods"]
    launchChild $initDir
    tkwait window .q
    set methodFile $strDir
    if { $methodFile eq "" } {
      popup "No File Selected"
      return
    }
    set methodFile [file join $initDir [file join $methodFile $methodFile]]

#build path to curpar
    
    set idx [string last "/" $file]
    set curparFile [string range $file 0 $idx]
    append curparFile "curpar"
    
    #
    #get what we need from tempCommun equivalent
    #
    
    set fd [open $methodFile r]
    gets $fd line
    gets $fd line
    set method [lindex $line 0]
    gets $fd line
    gets $fd line
    gets $fd line
    gets $fd line
    set nIter [lindex $line 0]
    gets $fd line
    set fTol [lindex $line 0]
    close $fd
    
    
    #
    # now read simplex_vars
    #
    append methodFile "_vars"
    readVars $methodFile 1
    
    #
    # check for name validity and find Min/Max
    #
    
    parseConpar
    parseCurpar $curparFile
    displayVars 1
    
}

#======================================================
#   Read the simplex_vars file or the equivalent 
#   stored file.
#
#  sMin, sMax may or may not be present in this file
#=====================================================

proc readVars { file update } {

    global communFileName
    global Name Val Min Max sMin sMax Units nDim
    set nCurrent $nDim
    set fileType "simplex_vars"
    set fd [open $file r]
    gets $fd line

    if { $line != $fileType } {
        popup "File Type Error"
        close $fd
        abort $communFileName
    }
    
    
    set n [array size Name]
    if  { $n > 0 } {
       unset Name
       unset Val
       unset sMin
       unset sMax
    } 
    
    set iCount 0
    while { [gets $fd line] >= 0 } {
        set Name($iCount)  [lindex $line 0]
        set Val($iCount)  [lindex $line 1]
        if { $update == 0 } {
           set Min($iCount)   [lindex $line 2]
           set Max($iCount)   [lindex $line 3]
        } else {
           set sMin($iCount)   [lindex $line 2]
           set sMax($iCount)   [lindex $line 3]
        }
        if { [llength line] > 4 } {
            set Units($iCount) [lindex $line 4]
        } else {
            set Units($iCount) ""
        }
        incr iCount 1   
    }

    close $fd
    set nDim $iCount
    displayVars $update
}

proc abort { file } {
   set STATUS 99
   set fd [open $file w]
   puts $fd $STATUS
   close $fd
   exit
} 

proc displayVars { update } {

global vb
global Name Val Min Max sMin sMax Units nDim fTol nIter
global ssColor textColor hdColor

   if { $update == 1} {
        pack forget .text
        destroy .text
        pack forget .spar
        destroy .spar
    }

    frame .text -bg $ssColor -borderwidth 2 -width 28 
    label .text.title -text "To Optimize" -fg $hdColor 
    pack .text.title -side top                                     
    pack .text -side left -fill x -fill y -anchor n
    frame .text.pars -bg $ssColor -borderwidth 1  -width 28
    pack .text.pars -side top
    label .text.pars.1 -text "Parameter" -width 12 -anchor w -justify left -fg $hdColor -bg $ssColor
    label .text.pars.2 -text "Value"  -width 10 -anchor w -justify left -fg $hdColor -bg $ssColor
    label .text.pars.3 -text "Minimum" -width 8 -anchor w -justify left -fg $hdColor -bg $ssColor
    frame .text.pars.spacer -width 4
    label .text.pars.4 -text "Maximum" -width 8 -anchor w -justify left -fg $hdColor -bg $ssColor
    pack .text.pars.1 .text.pars.2 .text.pars.3  .text.pars.spacer  .text.pars.4 -side left


   for {set f 0} {$f < $nDim} {incr f 1} {
     frame .text.f$f
    .text.f$f configure -height 10 -bg $ssColor
     label .text.f$f.l1 -text  $Name($f) -width 12 -anchor w -justify left  -fg $textColor -bg $ssColor 
     label .text.f$f.l2 -text   [format %3.2f $Val($f)] -width 10 -anchor w -justify left  -fg $textColor -bg $ssColor
     entry .text.f$f.e1 -textvariable sMin($f) -width 8 -bg $ssColor 
     entry .text.f$f.e2 -textvariable sMax($f) -width 8 -bg $ssColor 
     frame .text.f$f.spacer -width 4
     pack .text.f$f -side top -anchor w
     pack .text.f$f.l1 .text.f$f.l2  .text.f$f.e1 .text.f$f.spacer  .text.f$f.e2 -side left  
   }
   
   
   
   frame .spar  -borderwidth 1 -bg $ssColor 
   label .spar.title -text "Simplex Settings" -fg $hdColor
   pack .spar.title -side top 

   frame .spar.f1 -bg $ssColor -borderwidth 1 -width 28
   label .spar.f1.l -text Tolerance -width 12 -anchor w -justify left  -fg $textColor -bg $ssColor
   entry .spar.f1.e -textvariable fTol -width 10  -bg $ssColor
   pack  .spar.f1 -side top
   pack .spar.f1.l .spar.f1.e -side left

   frame .spar.f2 -bg $ssColor -borderwidth 1 -width 28
   label .spar.f2.l -text Iterations -width 12 -anchor w -justify left -fg $textColor -bg $ssColor
   entry .spar.f2.e -textvariable nIter -width 10  -bg $ssColor
   pack .spar.f2 -side top
   pack .spar.f2.l .spar.f2.e -side left

   frame .spar.f3 -bg $ssColor -width 28
   label .spar.f3.title -text "Seed Options" -fg $hdColor
   pack  .spar.f3.title -side top
   radiobutton .spar.f3.b1  -text  "Single" -fg $textColor -variable bSeed -value Single
   radiobutton .spar.f3.b2   -text  "All"  -fg $textColor -variable bSeed -value All
   pack .spar.f3.b1 .spar.f3.b2 -side left
   pack .spar.f3 -side top -anchor w

   .spar.f3.b1 select

#============================================================
# hard wire verbosity = 0
#============================================================

   set vb 0


   frame .spar.opt -bg $ssColor  
   label .spar.opt.title -fg $hdColor -text "Optimization Measure"
   pack .spar.opt.title -side top
   radiobutton .spar.opt.b1 -text "Shim Sum" -fg $textColor -bg $ssColor  -variable method -value shimSum
   radiobutton .spar.opt.b2 -text "Integral" -fg $textColor -bg $ssColor -variable method -value integral
   #radiobutton .spar.opt.b3 -text "Ratio" -fg $textColor -bg $ssColor -state disabled -variable method -value ratio
   radiobutton .spar.opt.b4 -text "Peak Height" -fg $textColor -bg $ssColor  -variable method -value peakHeight

   .spar.opt.b1 select
   pack .spar.opt.b1 .spar.opt.b2 .spar.opt.b4 -anchor w -padx 2 -pady 2

   #pack .spar.opt.b1 .spar.opt.b2 .spar.opt.b3 .spar.opt.b4 -anchor w -padx 2 -pady 2
   pack .spar.opt -side bottom  -fill x 
   pack .spar -side left -fill x

}

#=======================================================================================
#=======================================================================================
#
#                            PROGRAM ENTRY POINT
#
#=======================================================================================
#=======================================================================================


global Name Val Min Max Units nDim nCurrent

#
# Initialize some parameters
#

set argCnt $argc
set bSave 0
set nDim 0
set bIsParsed 0

#
# Check whether this is an auto or queued run
#

if {$argc == 2 } {
    set expName [lindex $argv 0]
    set methodName [lindex $argv 1]
    getAutoFile   $expName $methodName 
}

#
# not an auto run/queue if we get to this point
#

#
#  Construct Paths
#

set fileName [lindex $argv 0]
set communFileName $fileName
set fileName [file join $fileName "simplex_vars"]
set communFileName [file join $communFileName "tempCommun"]


if { [file exists $fileName] != 1 } {
    popup [list $fileName " Not Found!"]
    exit
} 

#================================
#    Set Up window
#================================

set wMain .
wm geometry . +100+200
wm title $wMain "Simplex Initialization"

$wMain configure -bg $ssColor

#
#create frame to hold a menu bar
#

frame .mb -bg $ssColor -borderwidth 1 
pack .mb -fill x

#
#Command Buttons
#

button .mb.1 -text "Start"  -fg $textColor -bg $ssColor -command { start $communFileName }
button .mb.2 -text "Abort"  -fg $textColor -bg $ssColor -command { abort $communFileName }
button .mb.3 -text "Save"   -fg $textColor -bg $ssColor -command { store $communFileName }
button .mb.4 -text "Load"   -fg $textColor -bg $ssColor -command { load  $communFileName }

pack .mb.1 .mb.2 -side right 
pack .mb.3 .mb.4 -side left

#=============================
# User Options
#=============================

##############################  
#  List Variables
##############################


#
# display the current set of variables
#

readVars $fileName 0



