/*make_waveform1.h - an include file to be added to all programs that
                     calculate waveforms.   D. Rice 10/30/00 vnmr61c
                     and modified 03/01/03 for solidslibvj11a.
                     Modified 06/11/03 to add round_offset.

                     see the documentation in this file below

                     modified for NPSG 03/10/05 - used for the
                     360.0/8192 step of the NPSG 0of VNMRS

                     added to SolidsLibVJ2.1B 05/17/06               */

typedef struct
{
  int     steps,cycles;
  double  offset,stepsize,length;
} offsettype;

static double lastamp,
              lastphase,
              lastgate,
              duration,
              lastamp2,
              lastphase2,
              duration2,
              tduration,
              ampsum,
              phasesum,
              count,
              delaycount,
              dmffactor;

static DECpattern waveform[32768];
static RFpattern waveform1[32768];

static int    wstep;

void initialize_waveform()
{
   lastamp=0.0;
   lastphase=0.0;
   lastgate=0.0;
   duration=0.0; 
   lastamp2=0.0; 
   lastphase2=0.0;
   duration2=0.0;
   tduration=0.0;
   ampsum=0.0;
   phasesum=0.0;
   count=0.0;
   wstep=0;
   delaycount=0,
   dmffactor=1.0,
   ok=0;
}

/*add_scale_step is used to add a 0 duration step of 1023 amplitude to a .DEC
waveform.  The controller automatically scales the maximum step of a waveform
to the current modulator value. This function should not be used with a .RF
waveform.  RF wavefroms have a built-in scale step already. */

void add_scale_step(lwstep)
int lwstep;
{
   waveform[lwstep].tip = 0.0;
   waveform[lwstep].phase = 0.0;
   waveform[lwstep].amp = 1023.0;
   waveform[lwstep].gate = 0.0;
}

void add_scale_step1(lwstep)
int lwstep;
{
   waveform1[lwstep].time = 0.0;
   waveform1[lwstep].phase = 0.0;
   waveform1[lwstep].amp = 1023.0;
}

double round_amp(lamp)
   double lamp;
{
   double amp1,amp;
   amp = lamp;
   if (amp < 0.0) amp = 0.0;
   if (amp > 1023.0) amp = 1023.0;
   amp1 = (double) ((int) amp);
   if ((amp - amp1) >= 0.5)
      amp = amp1 + 1.0;
   else
      amp = amp1;
   return(amp);
}

/*round_phase() rounds steps to 360.0/8192 = .0439 degrees, the natural phase
step size in NPSG. Phases are also rounded in the controller so one need carry
only enough significant figures to select the correct value.  Rounding of
waveforms is used to avoid the calculation of identical steps*/

double round_phase(lphase)
   double lphase;
{
   double phase1, phase, pstep;

   pstep =360.0/8192;
   phase = lphase;
   while (phase < 0.0) phase = phase + 360.0;
   while (phase >= 360.0) phase = phase - 360.0;
   phase1 = ((double) ((int) (phase/pstep)))*pstep;
   if ((phase - phase1) >= pstep/2.0)
      phase = phase1 + pstep;
   else
      phase = phase1;
   while (phase >= 360.0) phase = phase - 360.0;
   return(phase);
}

double round_step(lstep,lres)
   double lstep,lres;
{
   double step1,step;
   step = lstep;
   if (step < 0.0) step = 0.0;
   step1 = (((double) ((int) (step/lres)))*lres);
   if ((step - step1) >= lres/2)
      step = step1 + lres;
   else
      step = step1;
   return(step);
}

double set_delayflag(ltime,ldelay1,lstep)
   double ltime,ldelay1,lstep;
{
   double ldelayflag;
   ldelayflag=0.0;
   if ((ltime >= ldelay1) && (ltime < ldelay1 + lstep)) ldelayflag = 1.0;
   return(ldelayflag);
}

offsettype round_offset(loffset,lminsteps,lmaxcycles)
   double loffset;
   int lminsteps,lmaxcycles;
{
   double currentdstep,currentstep,currentcycles,currentindex,cycletime,
          teststep,rteststep,dteststep,sign,absoffset; 

   int index,index1;

   offsettype newoffset;

   currentdstep=0.05e-6;
   currentstep = 0.2e-6;
   currentcycles = 1.0;
   currentindex=0.0;
   cycletime = lminsteps*currentstep;

   absoffset = loffset;
   sign=1.0;
   if (absoffset < 0.0)
   {
      absoffset=-absoffset;
      sign = -1.0;
   }
   cycletime = 1/absoffset;

   if (absoffset>0.0)
   {
      index1=1;
      index=0;
      for(index1=1; index1<=lmaxcycles; index1=index1+1)
      {
         for(index=0; index<=(index1*lminsteps-1); index=index+1)
         {
            teststep = index1*cycletime/(lminsteps*index1 + index);
            if (teststep >= 2.0e-7)
            {
               rteststep = round_step(teststep,0.05e-6);
               dteststep = rteststep - teststep;
               if (dteststep<0.0) dteststep = - dteststep;
               if (dteststep < currentdstep)
               {
                  currentdstep = dteststep;
                  currentstep = rteststep;
                  currentcycles = index1;
                  currentindex = index;
               }
            }
         }
      }
      newoffset.cycles = currentcycles;
      newoffset.steps =  (int) (lminsteps*currentcycles + currentindex);
      newoffset.stepsize = currentstep;
      newoffset.length = (lminsteps*currentcycles + currentindex)*currentstep;
      newoffset.offset = sign*currentcycles/newoffset.length;
      return(newoffset);
   }
   else
   {
      newoffset.cycles = currentcycles;
      newoffset.steps =  (int) (lminsteps*currentcycles + currentindex);
      newoffset.stepsize = currentstep;
      newoffset.length = (lminsteps*currentcycles + currentindex)*currentstep;
      newoffset.offset = 0.0;
   }
   return(newoffset);
}

/*COMMENTS AND DOCUMENTATION*/

/* The software described here provides the user with the means to
   calculate waveforms directly in the pulse sequence. This software
   is an alternative to Pbox and can be used if the the desired wavefrom
   is not available in Pbox.

   The advantage of this software is that the user has control over all
   the source code used to make the waveform.  The user must supply
   mathematical formulae for amplitude and phase as a function of time.
   All code is contained in "include" files (.h) in the sequence.
   A disadvantage is that that the user must have some facility with C
   programming.

   If the desired waveform is available in Pbox it is usually better to
   use Pbox instead of this software. Pbox can be invoked from a pulse
   sequence with the "shell" command.  */

/* make_waveform() takes a set of values of amplitude, phase, gate
   and delayflag during the calculation of a waveform and determines 
   whether a new step of the waveform should be written. Use
   make_waveform in a loop that calculates waveform steps.  The output
   of make_waveform is one element of an array of structures with type 
   DECpattern. This array (called "waveform") is used as input for the
   command init_decpattern(). init_decpattern transforms the array into
   a .DEC file in shapelib that can be run with obsprgon(), decprgon(),
   dec2prgon(), dec3prgon().

   The array, waveform, will have a minimum number of steps with variable
   duration.  Call make_waveform() in each step of the loop.  If the 
   values of phase or amplitude have changed from the last step, a new
   element of waveform will be written.  If they have not changed the
   duration of the current step will be increased.  To take advantage
   of this condensation, round phases and amplitudes to their hardware
   resolution before input into make_waveform(). The functions round_phase()
   and round_amp() can be used for this purpose.

   Piece-wise waveforms can be constructed using "delayflag" and windowed 
   (xmtrgate off) waveforms can be produced by manipulating the "gate"
   entry.  A change in the "gate" entry or "delayflag=1.0" will force a
   new step immediately, to begin a new pulse or delay, whether or not amp
   or phase have changed.  "delayflag=1.0" does not appear in the .DEC file.
   It simply forces a new entry into the waveform array.

   To use make_waveform() in a pulse sequence:

   1. one must first place the instruction "#include <make_wavefrom.h>"
      in the .c file before the beginning of the pulsesequence() function,
      This file is placed in the pulse sequence code.  make_wavefrom.h
      contains the function make_waveform() and creates static variables
      that are automatically accessible to make_waveform().  It also
      contains several other functions that are used in creating waveforms.

   2. Add the code to calculate the waveform to the pulse sequence, before
      the beginning of active statements. It is best to contain the
      waveform calculation in a second include file.  The calculation
      is placed in a function to be called in the pulse sequence.
      For example, the include file "tppm.h" contains a function called
      make_tppm_dec() that contains the calculation of the tppm waveform.
      Add the statement "#include <make_tppm.h>" after "#include
      <make_wavefrom.h>" and call make_tppm_dec() in the pulse sequence.

      Use one of the existing waveform functions as a template for new
      waveforms. make_waveform.h and individual .h files are found in
      the user "psg" directory.

   3. in the waveform calculation, invoke the function "initialize_waveform()"
      before the loop that is used to calculate waveform steps.
      This initializes all the static variables used in make_wavefrom().

   3. set up a "for" loop that calculates amp, phase, gate and delayflag
      for each step of the waveform.  The time step (c.f. "lstep") of
      the loop must be equal to the minimum step length of the waveform
      generator (for INOVA: 200 ns or greater in 50 ns increments).
      This time step must also be set as the second argument of decprgon(),
      etc (often 1/dmf is used to set the time step for decoupler waveforms).

      It is important to round the values of amp and phase before invoking
      make_waveform().  Use the functions round_amp() and round_phase(). Also
      round_step() can be used to round arbitrary intervals to a designated
      step length. set_delayflag() will set the value of delayflag=1.0 in
      particular time step.

   4  after the loop, invoke make_waveform() once more with delayflag=1.0,
      to force the last step. Read the value of "wstep" to determine how
      many waveform steps have been created. Read the value of "duration"
      to determine the total length of a cycle. The cycle time is
      "duration*lstep/ldres" where lstep is the time step and ldres is
      usually 90.0, representing the minimum time step.

   5. optionally invoke the function compress_steps() to find a common factor
      "dmffactor" in all the durations of the waveform.  All durations are
      scaled by dmffactor and (static) dmffactor is used in the pulse sequence
      to scale the waveform pulse width (center argument). This scaling lets
      the wavegen or processor run with a lower speed. Use compress_steps1()
      with .RF waveforms.

   6. optionally invoke the function dupup_waveform() to expand the number
      of cycles that are contained in the waveform file.  Use of dupup_waveform
      optimizes waveforms for speed in NPSG with the new controller (and where
      memory is in excess). This function is not needed and should *NOT* be
      used with files for the UNITY/INOVA wavegen, where wavegen memory is a
      premium.  Also dupup_waveform() should never be used with .RF waveforms().

   7. optionally invoke the function add_scale_step() in order to add a
      0 duration step with amplitude 1023.  At run time the waveform is
      scaled so that the maximum amplitude in the waveform has the current
      modulator setting (tpwrf or tpwrm). The scale step insures that, that
      value is the maximum, 1023.  Do not include the scale step if you
      want tpwrm or tpwrf to be the maximum amplitude of the waveform. Do
      not use add_scale_step with .RF waveforms, they have their own scale
      step.

   8. invoke init_DECpattern(lpattern,waveform,wstep+1) where "lpattern"
      is a string with the waveform filename. The variable "waveform" is
      an array with the waveform steps. "wstep + 1" is the total number
      of steps, when the scale step is included. Use "wstep" if the scale
      step was not written. Use init_RFpattern (lpattern,waveform,wstep+1) for
      .RF waveformss

   PROPERTIES of make_waveform():

   A new waveform step is written if:

   1.  the amplitude value has changed
   2.  the phase value has changed
   3.  the gate value has changed
   4.  delayflag=1.0

   and:

   5. for amplitude and phase, a new waveform step is written only if
   the duration since the last waveform step is greater than the
   minimum step, "liminstep". If it is not, the duration is increased
   and the values of amplitude and phase are averaged with those values
   calculated since the last waveform step.

   6. for the gate and the delayflag, a new waveform step is forced
   immediately. If the duration since the last step is less than
   the duration of the minumum step, "lminstep", the values of "amp" and
   "phase" are averaged with the previous waveform step and that step
   is rewritten.

   NOTE:  You must set he interval between gate changes or steps with
   delayflag=1.0 greater than "lminstep".  If you do not, the second
   "gate" value or "delayflag" instruction will be ignored. The old
   step will be extended and the new value of gate will incorrectly replace
   the old value (i.e this is another way of saying that delays in a
   waveform can not be less than "lminstep").

   The arguments of make_waveform are:

   amp:  the current amplitude rounded to steps of 1.0.
   phase: the current phase rounded to steps of 0.25 degrees.
   gate: 1.0 to turn the transmitter on during the waveform
         independently of the pulse programmer transmitter
         gate. 0.0 to turn the transmitter off during the waveform
         unless the pulse programmer transmitter gate is on (the
         pulse programmer gate and the waveform gate are "or'd"
         at run time.
   delayflag:  if delayflag=1.0, a new step is forced in order
         to begin a new delay (even if the gate, amp or phase
         has not changed.
   ldres: the value listed for the duration in the waveform file
         of the fine step of the waveform. Usually ldres is
         set to "90.0" irrespective of the actual length of the
         step.  See the NOTE below.  .RF files ldres should be 1.0.
   lminstep: the value listed for the duration in the waveform file
         for the length of the minimum step of wavefrom. For
         INOVA wavegens the fine step and the minimum step have
         the same length and  lminstep = ldres = 90.  One might set
         a larger minimum step (lminstep=n*ldres) to run the wavegen
         with a longer minimum time step in order to conserve steps.

         NOTE:  The units of "90" for the fine step size are
         historical.  They date from the days when one step of
         the waveform was usually a hard 90 degree pulse. 90 is
         still used because the second argument of the decprgon(), etc,
         the "step length" is actually a "90 degree pulse length"
         decprgon() will assign the value of the "90 degree pulse"
         to an accumulated duration of 90 degrees, irregardless of
         the number of steps.

         If you designate a step time as "90" in the wavefrom,
         then its length will be set correctly by the middle
         argument of obsprgon().  If not, 90 units of duration will
         be set by this parameter. This latter use is legitimate but
         usually leads to confusion, so it is best to use "90".  */

/* make_waveform is used with the structure "waveform" and creates a .DEC file*/

void make_waveform(amp,phase,gate,delayflag,ldres,lminstep)
   double amp,phase,gate,delayflag,ldres,lminstep;
{
   double mindelaycount = (double) (int) (lminstep/ldres);
   double amp1 = amp;
   amp = round_amp(amp);
   double phase1 = phase;
   phase = round_phase(phase);
   if (((gate != lastgate) || (delayflag == 1.0)) && (delaycount >= mindelaycount))
   {
      if (duration < lminstep)
      {
         lastphase = (duration*lastphase + duration2*lastphase2)/(duration + duration2);
         lastamp = (duration*lastamp + duration2*lastamp2)/(duration + duration2);      
         tduration = tduration - duration2;
         wstep = wstep - 1;
         duration = duration + duration2;
      }
      waveform[wstep].tip = duration;
      waveform[wstep].phase = round_phase(lastphase);
      lastphase2 = lastphase;
      waveform[wstep].amp = round_amp(lastamp);
      lastamp2=lastamp;
      waveform[wstep].gate = lastgate;
      tduration = tduration + duration;
      duration2 = duration;
      duration = ldres;
      wstep = wstep + 1;
      if (wstep > 32768)
      {
         fprintf(stdout,"Maximum Steps Exceeded\n");
         psg_abort(1);
      }
      delaycount = 1.0;
   }
   else
   {
      if (((phase != lastphase) || (amp != lastamp)))
      {
         if (duration >= lminstep)
         {
            waveform[wstep].tip = duration;
            waveform[wstep].phase = round_phase(lastphase);
            lastphase2 = lastphase;
            waveform[wstep].amp = round_amp(lastamp);
            lastamp2=lastamp;
            waveform[wstep].gate = lastgate;
            tduration = tduration + duration;
            duration2 = duration;
            duration = ldres;
            wstep = wstep + 1;
            if (wstep > 35768)
            {
               fprintf(stdout,"Maximum Steps Exceeded\n");
               psg_abort(1);
            }
            count = 1;
            ampsum = amp1;
            phasesum = phase1;
         }
         else
         {
            count = count + 1;
            ampsum = ampsum + amp1;
            phasesum = phasesum + phase1;
            amp = ampsum/count;
            phase = phasesum/count;
            duration = duration + ldres;
         }
      }
      else
      {
         duration = duration + ldres;
      }
      delaycount = delaycount + 1.0;
   }
   lastphase = round_phase(phase);
   lastamp = round_amp(amp);
   lastgate = gate;
}

/* make_waveform1 is used with the structure "waveform1" and creates a .RF file*/

void make_waveform1(amp,phase,gate,delayflag,ldres,lminstep)
   double amp,phase,gate,delayflag,ldres,lminstep;
{
   double mindelaycount = (double) (int) (lminstep/ldres);
   double amp1 = amp;
   amp = round_amp(amp);
   double phase1 = phase;
   phase = round_phase(phase);

   if (((gate != lastgate) || (delayflag == 1.0)) && (delaycount >= mindelaycount))
   {
      if (duration < lminstep)
      {
         lastphase = (duration*lastphase + duration2*lastphase2)/(duration + duration2);
         lastamp = (duration*lastamp + duration2*lastamp2)/(duration + duration2);      
         tduration = tduration - duration2;
         wstep = wstep - 1;
         duration = duration + duration2;
      }
      waveform1[wstep].time = duration;
      waveform1[wstep].phase = round_phase(lastphase);
      lastphase2 = lastphase;
      waveform1[wstep].amp = round_amp(lastamp);
      lastamp2=lastamp;
      tduration = tduration + duration;
      duration2 = duration;
      duration = ldres;

      wstep = wstep + 1;
      if (wstep > 32768)
      {
         fprintf(stdout,"Maximum Steps Exceeded\n");
         psg_abort(1);
      }
      delaycount = 1.0;
   }
   else
   {
      if (((phase != lastphase) || (amp != lastamp)) || (duration >= 255.0*ldres))
      {
         if (duration >= lminstep)
         {
            waveform1[wstep].time = duration;
            waveform1[wstep].phase = round_phase(lastphase);
            lastphase2 = lastphase;
            waveform1[wstep].amp = round_amp(lastamp);
            lastamp2=lastamp;
            tduration = tduration + duration;
            duration2 = duration;
            duration = ldres;
            wstep = wstep + 1;
            if (wstep > 35768)
            {
               fprintf(stdout,"Maximum Steps Exceeded\n");
               psg_abort(1);
            }
            count = 1;
            ampsum = amp1;
            phasesum = phase1;
         }
         else
         {
            count = count + 1;
            ampsum = ampsum + amp1;
            phasesum = phasesum + phase1;
            amp = ampsum/count;
            phase = phasesum/count; 
            duration = duration + ldres;
         }
      }
      else
      {
         duration = duration + ldres;
      }
   delaycount = delaycount + 1.0;
   }
   lastphase = round_phase(phase);
   lastamp = round_amp(amp);
   lastgate = gate;
}

/*compress_steps() (.DEC) and compress_steps1() (.RF) find all the common
factors in the durations of a waveform and compress the step length
to 1-2 steps. The static variable dmffactor is calculated and used in the
pulse sequence to correspondingly increase the waveform step size. This
calculation spares the processor in the controller from needlessly
calculating identical steps. (uses factor_steps() (.DEC) or factor steps1()
(.RF). */

void factor_steps(lfac,ldres)
   double lfac,ldres;
{
   double temp = 0.0, temp1 = 0.0;
   int index, ok = 0, limit = 0;

   while ((ok==0) && (limit<10)) {
      index = 0;
      while ((ok==0) && (index<wstep)) {
         temp = waveform[index].tip/(ldres*dmffactor*lfac);
         temp1 = (double) (int) (waveform[index].tip/(ldres*dmffactor*lfac));
         if (temp != temp1) ok = 1;
         if (temp1 < 1.0) ok = 1;
         index = index + 1;
       /*fprintf(stdout,"temp = %f temp1 = %f index = %d\n",temp,temp1,index);*/}

      if (ok==0) dmffactor = dmffactor*lfac;
    /*fprintf(stdout,"dmffactor = %f lfac = %f limit = %d ok = %d\n",dmffactor,lfac,limit,ok);*/      
      limit = limit + 1;}
}

void factor_steps1(lfac,ldres)
   double lfac,ldres;
{
   double temp = 0.0, temp1 = 0.0;
   int index, ok = 0, limit = 0;

   while ((ok==0) && (limit<10)) {
      index = 0;
      while ((ok==0) && (index<wstep)) {
         temp = waveform1[index].time/(ldres*dmffactor*lfac);
         temp1 = (double) (int) (waveform1[index].time/(ldres*dmffactor*lfac));
         if (temp != temp1) ok = 1;
         if (temp1 < 1.0) ok =1;
         index = index + 1;
       /*fprintf(stdout,"temp = %f temp1 = %f index = %d\n",temp,temp1,index);*/}

      if (ok==0) dmffactor = dmffactor*lfac;
    /*fprintf(stdout,"dmffactor = %f lfac = %f limit = %d ok = %d\n",dmffactor,lfac,limit,ok)*/;
      limit = limit + 1;}
}

void compress_steps(ldres)
   double ldres;
{
   double temp = 0.0;
   int index;
   factor_steps(2.0,ldres);
   factor_steps(3.0,ldres);
   factor_steps(5.0,ldres);
   factor_steps(7.0,ldres);
   factor_steps(11.0,ldres);
   factor_steps(13.0,ldres);
   factor_steps(17.0,ldres);
   factor_steps(19.0,ldres);
   factor_steps(23.0,ldres);
   factor_steps(29.0,ldres);
   factor_steps(37.0,ldres);
   factor_steps(41.0,ldres);
   factor_steps(43.0,ldres);
   factor_steps(47.0,ldres);
   factor_steps(51.0,ldres);
   factor_steps(53.0,ldres);
   factor_steps(57.0,ldres);
   factor_steps(59.0,ldres);
   factor_steps(61.0,ldres);
   factor_steps(67.0,ldres);
   factor_steps(71.0,ldres);
   factor_steps(73.0,ldres);
   factor_steps(79.0,ldres);
   factor_steps(83.0,ldres);
   factor_steps(87.0,ldres);
   factor_steps(89.0,ldres);
   factor_steps(91.0,ldres);
   factor_steps(97.0,ldres);
   factor_steps(101.0,ldres);
   for(index=0; index<wstep; index=index+1) {
      temp = waveform[index].tip/dmffactor;
      waveform[index].tip = temp;}
}

void compress_steps1(ldres)
   double ldres;
{
   double temp = 0.0;
   int index;
   factor_steps1(2.0,ldres);
   factor_steps1(3.0,ldres);
   factor_steps1(5.0,ldres);
   factor_steps1(7.0,ldres);
   factor_steps1(11.0,ldres);
   factor_steps1(13.0,ldres);
   factor_steps1(17.0,ldres);
   factor_steps1(19.0,ldres);
   factor_steps1(23.0,ldres);
   factor_steps1(29.0,ldres);
   factor_steps1(37.0,ldres);
   factor_steps1(41.0,ldres);
   factor_steps1(43.0,ldres);
   factor_steps1(47.0,ldres);
   factor_steps1(51.0,ldres);
   factor_steps1(53.0,ldres);
   factor_steps1(57.0,ldres);
   factor_steps1(59.0,ldres);
   factor_steps1(61.0,ldres);
   factor_steps1(67.0,ldres);
   factor_steps1(71.0,ldres);
   factor_steps1(73.0,ldres);
   factor_steps1(79.0,ldres);
   factor_steps1(83.0,ldres);
   factor_steps1(87.0,ldres);
   factor_steps1(89.0,ldres);
   factor_steps1(91.0,ldres);
   factor_steps1(97.0,ldres);
   factor_steps1(101.0,ldres);
   for(index=0; index<wstep; index=index+1) {
      temp = waveform1[index].time/dmffactor;
      waveform1[index].time = temp;}
}

/*dupup_waveform() expands the number of cycles in a .DEC waveform
to 2500 to 5000 in order to increase the efficiency of controller
DMA access. dupup_waveform returns the new number of steps "nwstep".
Invoke "wstep = dupup_waveform()" to update wsteps in init_DECpattern.
dupup_waveform() should not be used with .RF waveforms.*/

int dupup_waveform()
{
   int index, index1, nwstep, nwstepfactor;

   fprintf(stdout,"wstep = %d\n", wstep);

   if (wstep <= 2499) {
      nwstepfactor = 5000/(wstep);
      nwstep = nwstepfactor*wstep;}
   else
      nwstep = wstep;

   if (nwstep > wstep) {
      for(index=0; index <= nwstep; index=index+1) {
         index1 = (index%wstep);
         waveform[index] = waveform[index1];}}
   fprintf(stdout,"nwstep = %d\n", nwstep);
   return(nwstep);
}


