/*stmasspltse2d1.c - A split echo sequence to generate a two pulse STMAS 2D spectrum
                with a third selective echo pulse.

                See: Ashbrooke, S.E., Wimperis, S.E. Progress in NMR Spectroscopy 45 (2004)
                     53-108.

                D. Rice 05/10/06                                                           */

#include "standard.h"
#include "solidstandard.h"

// Define Values for Phasetables

static int table1[4] = {0,0,0,0};       // ph1Xstmas
static int table2[4] = {0,1,2,3};       // ph2Xstmas
static int table3[16] = {0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3};   // phXechsel
static int table4[8] = {0,0,0,0,2,2,2,2};   // phRec

#define ph1Xstmas t1
#define ph2Xstmas t2
#define phXechsel t3
#define phRec t4

pulsesequence() {

// Define Variables and Objects and Get Parameter Values

   double duty;
                                             
   double ival = getval("ival");                 //Obtain the K-values from ival and mval
   double mval = getval("mval");                 //Table 3 of Ashbrook & Wimperis
   double k0 = 1.0;                             
   double k1 = 0.0;
   double k2 = 0.0;

   if ((ival == 1.5) && (mval == 1.5)) {
      k0 = 9.0/17.0;
      k1 = 8.0/17.0;
   } 
   if (ival == 2.5) {
      if (mval == 1.5) {
         k0 = 24.0/31.0;
         k2 = 7.0/31.0;
      }
      else {
         k0 = 9.0/17.0;
         k1 = 8.0/17.0;
      }
   }
   if (ival == 3.5) {
      if (mval == 1.5) {
         k0 = 45.0/73.0;
         k2 = 28.0/73.0;
      }
      else if (mval == 2.5) {
         k0 = 45.0/68.0;
         k1 = 23.0/68.0;
      }
      else {
         k0 = 5.0/17.0;
         k1 = 12.0/17.0;
      }
   }
   if (ival == 4.5) {
      if (mval == 1.5) {
         k0 = 72.0/127.0;
         k2 = 55.0/127.0;
      }
      else if (mval == 2.5) {
         k0 = 18.0/19.0;
         k2 = 1.0/19.0;
      }
      else if (mval == 3.5) {
         k0 = 8.0/17.0;
         k1 = 9.0/17.0;
      }
      else {
         k0 = 9.0/34.0;
         k1 = 25.0/34.0;
      }
   }

   double d2init = getval("d2");            // Define the Split d2 in the Pulse Sequence
   double d20 = k0*d2init;
   double d21 = k1*d2init;
   double d22 = k2*d2init;

   double pw1Xstmas = getval("pw1Xstmas");  // Get STMAS and echo pulsewidths for synchronization 
   double pw2Xstmas = getval("pw2Xstmas");  // of the F1 delay

   double tXechselinit = getval("tXechsel");// Adjust the selective echo delay for the
   double tXechsel = tXechselinit - 3.0e-6; // attenuator switch time.
   if (tXechsel < 0.0) tXechsel = 0.0;

                                            // Adjust the F1 delay for the pw1 and pw2 pulses
   d20 = d20 - pw1Xstmas/2.0 - pw2Xstmas/2.0;
   if (d20 < 0.0) d20 = 0.0;

   DSEQ dec = getdseq("H");

// Dutycycle Protection

   duty = 4.0e-6 + getval("pw1Xstmas") + d2 + getval("pw2Xstmas") + tXechselinit +
                   getval("pwXechsel") + getval("rd") + getval("ad") + at;
   duty = duty/(duty + d1 + 4.0e-6);
   if (duty > 0.1) {
      printf("Duty cycle %.1f%% >10%%. Abort!\n", duty*100);
      psg_abort(1);
   }

// Set Phase Tables

   settable(ph1Xstmas,4,table1);
   settable(ph2Xstmas,4,table2);
   settable(phXechsel,16,table3);
   settable(phRec,8,table4);

   if (phase1 == 2) {
      tsadd(ph1Xstmas,1,4);
   }
   setreceiver(phRec);

// Begin Sequence

   txphase(ph1Xstmas); decphase(zero);
   obspower(getval("tpwr"));
   obspwrf(getval("aXstmas"));
   obsunblank(); decunblank(); _unblank34();
   delay(d1);
   sp1on(); delay(2.0e-6); sp1off(); delay(2.0e-6);

// H Decoupler on Before STMAS

   _dseqon(dec);
   
// Two-Pulse STMAS

   rgpulse(getval("pw1Xstmas"),ph1Xstmas,0.0,0.0);
   txphase(ph2Xstmas);
   delay(d20);
   rgpulse(getval("pw2Xstmas"),ph2Xstmas,0.0,0.0);

// Selective Echo Pulse

   txphase(phXechsel);
   obsblank();
   obspower(getval("dbXechsel"));
   obspwrf(getval("aXechsel"));
   delay(3.0e-6);
   obsunblank();
   delay(d21 + tXechsel);
   rgpulse(getval("pwXechsel"),phXechsel,0.0,0.0);
   delay(d22);

// Begin Acquisition

   obsblank(); _blank34();
   delay(getval("rd"));
   startacq(getval("ad"));
   acquire(np, 1/sw);
   endacq();
   _dseqoff(dec);
   obsunblank(); decunblank(); _unblank34();
}

