/***************************************************************************

  osdepend.c

  OS dependant stuff (display handling, keyboard scan...)
  This is the only file which should me modified in order to port the
  emulator to a different system.

***************************************************************************/

#include <pc.h>
#include <sys/farptr.h>
#include <stdio.h>
#include <go32.h>
#include <allegro.h>
#include "audio.h"
#include "gfxdecode.h"
#include "osdepend.h"
#include "machine.h"

unsigned        IOport;
int             osd_joy_up,
                osd_joy_down,
                osd_joy_left,
                osd_joy_right;
int             osd_joy_b1,
                osd_joy_b2,
                osd_joy_b3,
                osd_joy_b4;

/* put here anything you need to do when the program is started. Return 0 if */
/* initialization was successful, nonzero otherwise. */
int             osd_init(void)
{
    allegro_init();
    osd_sound_init();
    install_keyboard();         /* Allegro keyboard handler */
    if (initialise_joystick())
        Joystick = 0;           /* joystick not found */

    return 0;
}

void            osd_create_display(int graphicsmode)
{
    int             count1,
                    count2;

    if (graphicsmode == VESA)
        set_gfx_mode(GFX_VESA1, 640, 480, 0, 0);    // Allegro routine to
    // start video
    else if (graphicsmode == MODEX)
        set_gfx_mode(GFX_MODEX, 256, 256, 0, 0);    // Allegro routine to
    // start video

    piccie = create_bitmap(256, 256);   // Allocate memory for temp screen
    // (in memory)

    for (count1 = 0; count1 < 256; count1++)
        for (count2 = 0; count2 < 256; count2++)
            piccie->line[count1][count2] = 0;   // Clear the temp screen

}

/* shut up the display */
void            osd_close_display(void)
{
    destroy_bitmap(piccie);
    set_gfx_mode(GFX_TEXT, 80, 25, 0, 0);
}

/* Update the display. */
void            osd_update_display(void)
{
    PALETTE         pal;
    char            name[13];
    FILE           *f;
    static int      snapno;

    if (graphicsmode == VESA)
    {
        if ((gameemulated == GALAXIAN) || (gameemulated == GALNAMCO) ||
            (gameemulated == SUPGALAX) || (gameemulated == GALAPX) ||
            (gameemulated == GALTURBO) || (gameemulated == AMIDAR) ||
            (gameemulated == THEEND) || (gameemulated == GALAP1) ||
            (gameemulated == GALAP4))
            blit(piccie, screen, 0, 0, 196, 112, 248, 256);
        else if ((gameemulated == PHOENIX) || (gameemulated == PLEIADS))
            blit(piccie, screen, 48, 0, 216, 112, 208, 256);
        else if (gameemulated == BAGMAN)
            blit(piccie, screen, 32, 0, 208, 112, 224, 256);
        else if (gameemulated == FROGGER)
            blit(piccie, screen, 32, 0, 208, 112, 224, 256);
        else
            blit(piccie, screen, 0, 0, 192, 112, 256, 256);
    }
    else if (graphicsmode == MODEX)
    {
        if ((gameemulated == GALAXIAN) || (gameemulated == GALNAMCO) ||
            (gameemulated == SUPGALAX) || (gameemulated == GALAPX) ||
            (gameemulated == GALTURBO) || (gameemulated == AMIDAR) ||
            (gameemulated == THEEND) || (gameemulated == GALAP1) ||
            (gameemulated == GALAP4))
            blit(piccie, screen, 0, 0, 4, 0, 248, 256);
        else if ((gameemulated == PHOENIX) || (gameemulated == PLEIADS))
            blit(piccie, screen, 48, 0, 24, 0, 208, 256);
        else if (gameemulated == BAGMAN)
            blit(piccie, screen, 16, 0, 8, 0, 224, 256);
        else if (gameemulated == FROGGER)
            blit(piccie, screen, 16, 0, 8, 0, 224, 256);
        else
            blit(piccie, screen, 0, 0, 0, 0, 256, 256);

    }

    /* if the user pressed F6, save a snapshot of the screen. */
    if (osd_key_pressed(OSD_KEY_F6))
    {

        do
        {
            sprintf(name, "snap%04d.pcx", snapno);
            /* avoid overwriting of existing files */
            if ((f = fopen(name, "rb")) != 0)
            {
                fclose(f);
                snapno++;
            }
        } while (f != 0);

        get_palette(pal);
        save_pcx(name, piccie, pal);
        snapno++;

        /* wait for the user to release F6 */
        while (osd_key_pressed(OSD_KEY_F6));
    }


}

/* put here cleanup routines to be executed when the program is terminated. */
void            osd_exit(void)
{
    osd_sound_stop();
}

/* Set the graphics palette */
void            osd_set_palette(void)
{
    int             count;
    outportb(0x3c8, 0x00);
    for (count = 0; count < 768; count++)
        outportb(0x3c9, pal[count]);
}


/* check if a key is pressed. The keycode is the standard PC keyboard code, as */
/* defined in osdepend.h. Return 0 if the key is not pressed, nonzero otherwise. */
int             osd_key_pressed(int keycode)
{
    return key[keycode];
}

/* Initialize sound system */
void            osd_sound_init()
{
    int             retval;
    unsigned        dummy;
    if (soundsource == SB)
    {
        retval = ReadBlasterEnv(&IOport, &dummy, &dummy, &dummy);
        if (retval != 0)
            soundsource = SPEAKER;
        else
        {
            fm(1, 0);           /* must initialize this to zero */

            fm(0xC0, 1);        /* parallel connection */

            /***************************************
             * Set parameters for the carrier cell *
             ***************************************/

            fm(0x23, 0x21);     /* no amplitude modulation (D7=0), no vibrato
                                 * (D6=0), sustained envelope type (D5=1),
                                 * KSR=0 (D4=0), frequency multiplier=1
                                 * (D4-D0=1) */

            fm(0x43, 0x0);      /* no volume decrease with pitch (D7-D6=0),
                                 * no attenuation (D5-D0=0) */

            fm(0x63, 0xff);     /* fast attack (D7-D4=0xF) and decay
                                 * (D3-D0=0xF) */
            fm(0x83, 0x05);     /* high sustain level (D7-D4=0), slow release
                                 * rate (D3-D0=5) */


            /*****************************************
             * Set parameters for the modulator cell *
             *****************************************/

            fm(0x20, 0x20);     /* sustained envelope type, frequency
                                 * multiplier=0    */
            fm(0x40, 0x3f);     /* maximum attenuation, no volume decrease
                                 * with pitch */

            /* Since the modulator signal is attenuated as much as possible,
             * these next two values shouldn't have any effect. */
            fm(0x60, 0x44);     /* slow attack and decay */
            fm(0x80, 0x05);     /* high sustain level, slow release rate */
        }
    }

}

/* Play a sound at the desired frequency */
void            osd_sound(int frequency)
{

    int             block,
                    b,
                    m,
                    f,
                    fn;
    static int      KEYON = 0x20;

    if (Sound)
    {

        if (soundsource == SPEAKER)
        {
            sound(frequency * 8);   // used for speaker
        }
        else if (soundsource == SB)
        {
            block = 5;          /* block is the octive number */
            m = 1;              /* m is the frequency multiple number */
            f = frequency * 8;  /* frequency in Hz */
            b = 1 << block;

            /* This is the equation to calculate frequency number from
             * frequency. */

            fn = (long) f  *1048576 / b / m / 50000L;

            fm(0x23, 0x20 | (m & 0xF)); /* 0x20 sets sustained envelope, low
                                         * nibble is multiple number */
            fm(0xA0, (fn & 0xFF));
            fm(0xB0, ((fn >> 8) & 0x3) + ((block << 2) | KEYON));
        }
    }
}

/* Disable sound */
void            osd_sound_stop()
{

    int             block,
                    b,
                    m,
                    f,
                    fn;
    static int      KEYON = 0x20;

    if (soundsource == SPEAKER)
    {
        sound(0);               // used for speaker

    }
    else if (soundsource == SB)
    {
        block = 5;              /* block is the octive number */
        m = 1;                  /* m is the frequency multiple number */
        f = 0;                  /* frequency in Hz */
        b = 1 << block;

        /* This is the equation to calculate frequency number from frequency. */

        fn = (long) f  *1048576 / b / m / 50000L;

        fm(0x23, 0x20 | (m & 0xF)); /* 0x20 sets sustained envelope, low
                                     * nibble is multiple number */
        fm(0xA0, (fn & 0xFF));
        fm(0xB0, ((fn >> 8) & 0x3) + ((block << 2) | KEYON));

    }
}

/* Poll the joystick */
void            osd_poll_joystick(void)
{
    unsigned char   joystatus;

    if (Joystick == 1)
    {
        poll_joystick();
        joystatus = inportb(0x201);
        osd_joy_up = joy_up;
        osd_joy_down = joy_down;
        osd_joy_left = joy_left;
        osd_joy_right = joy_right;
        osd_joy_b1 = joy_b1;
        osd_joy_b2 = joy_b2;
        osd_joy_b3 = ((joystatus & 0x40) == 0);
        osd_joy_b4 = ((joystatus & 0x80) == 0);
    }
}
