#ifndef __PrintSD__
#define __PrintSD__
#include <ntsecapi.h>

#define PSD_SID_BYTES (8192*sizeof(TCHAR))
#define PSD_ACL_BYTES (32768*sizeof(TCHAR))
#define PSD_SDS_BYTES (65536*sizeof(TCHAR))
#define PSD_TKS_BYTES (65536*sizeof(TCHAR))

LPTSTR WINAPI GetSidInfo(PSID pSID, LPCTSTR *SystemNames) {
  LPTSTR SidString, CurSidString;
  if((SidString = (LPTSTR)LocalAlloc(LMEM_FIXED, PSD_SID_BYTES)) == NULL) {
    return(NULL);
  } 
  CurSidString = SidString;

  #define PREDEF_SID_IAs 6
  static LPWSTR PredefSidIA[PREDEF_SID_IAs] = {
    L"0",
    L"1",
    L"2",
    L"3",
    L"4",
    L"5"
  };

  static LPTSTR PredefSidIAName[PREDEF_SID_IAs] = {
    _T("NULL_SID_AUTHORITY"),
    _T("WORLD_SID_AUTHORITY"),
    _T("LOCAL_SID_AUTHORITY"),
    _T("CREATOR_SID_AUTHORITY"),
    _T("NON_UNIQUE_AUTHORITY"),
    _T("NT_AUTHORITY")
  };

  static LPTSTR SidNameUse[] = {
    _T(""),
    _T("User"), 
    _T("Group"), 
    _T("Domain"), 
    _T("LocalGroup"), //("Alias")
    _T("WellKnownGroup"), 
    _T("DeletedAccount"), 
    _T("Invalid"), 
    _T("Unknown"),
    _T("Computer") 
  };
  
  typedef LONG (WINAPI *TRtlConvertSidToUnicodeString)(PUNICODE_STRING, PSID, BOOL);
  static TRtlConvertSidToUnicodeString pRtlConvertSidToUnicodeString = NULL;
  HINSTANCE hntdll;
  if(pRtlConvertSidToUnicodeString == NULL)
    if(hntdll = GetModuleHandle(_T("ntdll.dll")))
      pRtlConvertSidToUnicodeString = (TRtlConvertSidToUnicodeString)GetProcAddress(hntdll, "RtlConvertSidToUnicodeString");

  if(!IsValidSid(pSID)) {
    CurSidString += _stprintf(CurSidString, _T("Security identifier is not valid! (%u)\n"), GetLastError());
    return(SidString);
  }
  
  CurSidString += _stprintf(CurSidString, _T("Security identifier length = %u bytes\n"),
                    GetLengthSid(pSID));
  int i;
  WCHAR SidStringW[64];
  UNICODE_STRING UnicodeSid = {0, sizeof(SidStringW), SidStringW};
  if(pRtlConvertSidToUnicodeString && (pRtlConvertSidToUnicodeString(&UnicodeSid, pSID, FALSE) >= 0)) {  
    SidStringW[UnicodeSid.Length/sizeof(WCHAR)] = L'\0';
    CurSidString += _stprintf(CurSidString, _T("Security identifier = %ws\n"), SidStringW);

    int PredefIdx = -1, CmpLen = wcschr(SidStringW+4, '-')-(SidStringW+4);
    for(i=0; i<PREDEF_SID_IAs; i++) {
      if(wcsncmp(PredefSidIA[i], SidStringW+4, CmpLen) == 0) {
        PredefIdx = i;
        break;
      }
    }
    CurSidString += _stprintf(CurSidString, _T("Security identifier authority = %s\n"), (PredefIdx == -1) ? _T("<Not predefined>") : PredefSidIAName[i]);
  }

  CurSidString += _stprintf(CurSidString, _T("Security identifier subauthorities = %u\n"),
                    *GetSidSubAuthorityCount(pSID));

  DWORD LastError;
  LPTSTR AccountName;
  DWORD cbAccountName;
  LPTSTR ReferencedDomainName;
  DWORD cbReferencedDomainName;
  SID_NAME_USE eUse;
  BOOL SidFound = FALSE;
  LPCTSTR rSystemNames[2] =  {NULL, LPCTSTR(-1)};

  if(SystemNames == NULL) {
    SystemNames = rSystemNames;
  }
 
  for(i=0; SystemNames[i] != LPCTSTR(-1); i++) {
    AccountName = NULL;
    cbAccountName = 0;
    ReferencedDomainName = NULL;
    cbReferencedDomainName = 0;

    LookupAccountSid(SystemNames[i], pSID,
      AccountName, &cbAccountName,
      ReferencedDomainName, &cbReferencedDomainName,
      &eUse);

    LastError = GetLastError();
    if(LastError != ERROR_INSUFFICIENT_BUFFER) {
      continue;
    }
    if(!(AccountName = (LPTSTR)LocalAlloc(LMEM_FIXED, (cbAccountName)*sizeof(TCHAR)))) {
      CurSidString += _stprintf(CurSidString, _T("Heap allocation failed! (%u)\n"), GetLastError());
      return(SidString);
    }
    if(!(ReferencedDomainName = (LPTSTR)LocalAlloc(LMEM_FIXED, (cbReferencedDomainName)*sizeof(TCHAR)))) {
      CurSidString += _stprintf(CurSidString, _T("Heap allocation failed! (%u)\n"), GetLastError());
      LocalFree(AccountName);
      return(SidString);
    }
    if(LookupAccountSid(SystemNames[i], pSID,
         AccountName, &cbAccountName,
         ReferencedDomainName, &cbReferencedDomainName,
         &eUse)) {
      SidFound = TRUE;
      break;
    }
    else {
      LocalFree(AccountName);
      LocalFree(ReferencedDomainName);
    }
  }

  if(SidFound) {
    CurSidString += _stprintf(CurSidString, _T("Security identifier domain  name = %s\n"), ReferencedDomainName);
    CurSidString += _stprintf(CurSidString, _T("Security identifier account name = %s\n"), AccountName);
    CurSidString += _stprintf(CurSidString, _T("Security identifier type = %s\n"), SidNameUse[eUse]);
    LocalFree(AccountName);
    LocalFree(ReferencedDomainName);
  }
  else {
    CurSidString += _stprintf(CurSidString, _T("Can't get account information! (%u)\n"), GetLastError());
  }
  return(SidString);
}


LPTSTR WINAPI GetAclInfo(PACL pAcl, LPCTSTR *SystemNames) {
  DWORD i;
  LPTSTR AclString, CurAclString;
  if((AclString = (LPTSTR)LocalAlloc(LMEM_FIXED, PSD_ACL_BYTES)) == NULL) {
    return(NULL);
  }
  CurAclString = AclString;
  ACL_REVISION_INFORMATION AclRevision;
  ACL_SIZE_INFORMATION AclSizeInfo;
  if(!GetAclInformation(pAcl, &AclRevision, sizeof(AclRevision), AclRevisionInformation)) {
    CurAclString += _stprintf(CurAclString, _T("Can't get ACL revision information! (%u)\n"), GetLastError());
  }
  else {        
    CurAclString += _stprintf(CurAclString, _T("ACL revision = (%u) %s\n"), AclRevision.AclRevision,
                      (AclRevision.AclRevision == ACL_REVISION) ? _T("ACL_REVISION") : _T(""));
  }
  if(!GetAclInformation(pAcl, &AclSizeInfo, sizeof(AclSizeInfo), AclSizeInformation)) {
    CurAclString += _stprintf(CurAclString, _T("Can't get ACL size information! (%u)\n"), GetLastError());
  }
  else {        
    CurAclString += _stprintf(CurAclString, _T("ACL size :  AceCount = %u, AclBytesInUse = %u, AclBytesFree = %u"),
                      AclSizeInfo.AceCount, AclSizeInfo.AclBytesInUse, AclSizeInfo.AclBytesFree);
  }

  CurAclString += _stprintf(CurAclString, _T("\n"));
  PACCESS_ALLOWED_ACE pAce;
  for(i=0; (i<AclSizeInfo.AceCount); i++) {
    if(GetAce(pAcl, i, (LPVOID*)&pAce)) {
      LPCTSTR AceTypeStr;
      switch(pAce->Header.AceType) {
        case          ACCESS_ALLOWED_ACE_TYPE : AceTypeStr = _T(         "ACCESS_ALLOWED_ACE"); break;
        case           ACCESS_DENIED_ACE_TYPE : AceTypeStr = _T(          "ACCESS_DENIED_ACE"); break;
        case            SYSTEM_AUDIT_ACE_TYPE : AceTypeStr = _T(           "SYSTEM_AUDIT_ACE"); break;
        case            SYSTEM_ALARM_ACE_TYPE : AceTypeStr = _T(           "SYSTEM_ALARM_ACE"); break;
        case ACCESS_ALLOWED_COMPOUND_ACE_TYPE : AceTypeStr = _T("ACCESS_ALLOWED_COMPOUND_ACE"); break;
        case   ACCESS_ALLOWED_OBJECT_ACE_TYPE : AceTypeStr = _T(  "ACCESS_ALLOWED_OBJECT_ACE"); break;
        case    ACCESS_DENIED_OBJECT_ACE_TYPE : AceTypeStr = _T(   "ACCESS_DENIED_OBJECT_ACE"); break; 
        case     SYSTEM_AUDIT_OBJECT_ACE_TYPE : AceTypeStr = _T(    "SYSTEM_AUDIT_OBJECT_ACE"); break;
        case     SYSTEM_ALARM_OBJECT_ACE_TYPE : AceTypeStr = _T(    "SYSTEM_ALARM_OBJECT_ACE"); break;
        default                               : AceTypeStr = _T(                "UNKNOWN_ACE"); break;
      }
      CurAclString += _stprintf(CurAclString, _T("\n***ACE %2u :  AceSize = %u, AceType = %s\nAceFlags = (0x%X)"),
                        i, pAce->Header.AceSize, AceTypeStr, pAce->Header.AceFlags);
      if(pAce->Header.AceFlags & CONTAINER_INHERIT_ACE)
        CurAclString += _stprintf(CurAclString, _T(" CONTAINER_INHERIT_ACE"));
      if(pAce->Header.AceFlags & INHERIT_ONLY_ACE)
        CurAclString += _stprintf(CurAclString, _T(" INHERIT_ONLY_ACE"));
      if(pAce->Header.AceFlags & NO_PROPAGATE_INHERIT_ACE)
        CurAclString += _stprintf(CurAclString, _T(" NO_PROPAGATE_INHERIT_ACE"));
      if(pAce->Header.AceFlags & OBJECT_INHERIT_ACE)
        CurAclString += _stprintf(CurAclString, _T(" OBJECT_INHERIT_ACE"));
      if(pAce->Header.AceFlags & FAILED_ACCESS_ACE_FLAG)
        CurAclString += _stprintf(CurAclString, _T(" FAILED_ACCESS_ACE"));
      if(pAce->Header.AceFlags & SUCCESSFUL_ACCESS_ACE_FLAG)
        CurAclString += _stprintf(CurAclString, _T(" SUCCESSFUL_ACCESS_ACE"));

      CurAclString += _stprintf(CurAclString, _T("\nMask = (0x%.8X)"), pAce->Mask);

      if(pAce->Mask & GENERIC_READ)
        CurAclString += _stprintf(CurAclString, _T(" GENERIC_READ"));
      if(pAce->Mask & GENERIC_WRITE)
        CurAclString += _stprintf(CurAclString, _T(" GENERIC_WRITE"));
      if(pAce->Mask & GENERIC_EXECUTE)
        CurAclString += _stprintf(CurAclString, _T(" GENERIC_EXECUTE"));
      if(pAce->Mask & GENERIC_ALL)
        CurAclString += _stprintf(CurAclString, _T(" GENERIC_ALL"));

      if(pAce->Mask & ACCESS_SYSTEM_SECURITY)
        CurAclString += _stprintf(CurAclString, _T(" ACCESS_SYSTEM_SECURITY"));

      if((pAce->Mask & STANDARD_RIGHTS_ALL) == STANDARD_RIGHTS_ALL) {
        CurAclString += _stprintf(CurAclString, _T(" STANDARD_RIGHTS_ALL"));
      }
      else {
        if((pAce->Mask & STANDARD_RIGHTS_REQUIRED) == STANDARD_RIGHTS_REQUIRED) {
          CurAclString += _stprintf(CurAclString, _T(" STANDARD_RIGHTS_REQUIRED"));
        }
        else {
          if(pAce->Mask & SYNCHRONIZE)
            CurAclString += _stprintf(CurAclString, _T(" SYNCHRONIZE"));
          if(pAce->Mask & WRITE_OWNER)
            CurAclString += _stprintf(CurAclString, _T(" WRITE_OWNER"));
          if(pAce->Mask & WRITE_DAC)
            CurAclString += _stprintf(CurAclString, _T(" WRITE_DAC"));
          if(pAce->Mask & READ_CONTROL)
            CurAclString += _stprintf(CurAclString, _T(" READ_CONTROL"));
          if(pAce->Mask & DELETE)
            CurAclString += _stprintf(CurAclString, _T(" DELETE"));
        }
      }

      pAce->Mask &= SPECIFIC_RIGHTS_ALL;
      if(pAce->Mask == SPECIFIC_RIGHTS_ALL) {
        CurAclString += _stprintf(CurAclString, _T(" SPECIFIC_RIGHTS_ALL"));
      }
      else {
        if(pAce->Mask)
          CurAclString += _stprintf(CurAclString, _T(" 0x%.4X"), pAce->Mask);
      }
   
      PACCESS_ALLOWED_OBJECT_ACE pAceO = NULL;
      if(pAce->Header.AceType >= ACCESS_MIN_MS_OBJECT_ACE_TYPE) {
        pAceO = (PACCESS_ALLOWED_OBJECT_ACE)pAce;
        CurAclString += _stprintf(CurAclString, _T("\nFlags = (%u)\n"), pAceO->Flags);
        #define sGUID _T("{%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X}")
        if(pAceO->Flags & ACE_OBJECT_TYPE_PRESENT) {
          CurAclString += _stprintf(CurAclString, _T(" ACE_OBJECT_TYPE_PRESENT\n ObjectType = "));         
          CurAclString += _stprintf(CurAclString, sGUID,
                           pAceO->ObjectType.Data1, pAceO->ObjectType.Data2, pAceO->ObjectType.Data3,
                           pAceO->ObjectType.Data4[0], pAceO->ObjectType.Data4[1],
                           pAceO->ObjectType.Data4[2], pAceO->ObjectType.Data4[3],
                           pAceO->ObjectType.Data4[4], pAceO->ObjectType.Data4[5],
                           pAceO->ObjectType.Data4[6], pAceO->ObjectType.Data4[7]);
        }
        if(pAceO->Flags & ACE_INHERITED_OBJECT_TYPE_PRESENT) {
          CurAclString += _stprintf(CurAclString, _T(" ACE_INHERITED_OBJECT_TYPE_PRESENT\n InheritedObjectType = "));
          CurAclString += _stprintf(CurAclString, sGUID,
                           pAceO->InheritedObjectType.Data1, pAceO->InheritedObjectType.Data2, pAceO->InheritedObjectType.Data3,
                           pAceO->InheritedObjectType.Data4[0], pAceO->InheritedObjectType.Data4[1],
                           pAceO->InheritedObjectType.Data4[2], pAceO->InheritedObjectType.Data4[3],
                           pAceO->InheritedObjectType.Data4[4], pAceO->InheritedObjectType.Data4[5],
                           pAceO->InheritedObjectType.Data4[6], pAceO->InheritedObjectType.Data4[7]);
        }
      }
        
      LPTSTR SidString;
      if(SidString = GetSidInfo(pAceO ? (PSID)&pAceO->SidStart : (PSID)&pAce->SidStart, SystemNames)) { 
        CurAclString += _stprintf(CurAclString, _T("\n%s"), SidString);
        LocalFree(SidString);
      }
      else {  
        CurAclString += _stprintf(CurAclString, _T("Can't get ACE's %u SID! (%u)\n"), i, GetLastError());
      }

    }
    else {
      CurAclString += _stprintf(CurAclString, _T("Can't get ACE %u!"), i);
    }
  }    
  return(AclString);
}


LPTSTR WINAPI PrintSD(HANDLE hObject, SECURITY_INFORMATION ReqSecInfo, LPCTSTR *SystemNames) {
  BOOL bIsDACL, bIsSACL;
  DWORD LastError, ReqLen;
  PSECURITY_DESCRIPTOR pSD = NULL;

  LPTSTR InfoString, MainString;
  if((MainString = (LPTSTR)LocalAlloc(LMEM_FIXED, PSD_SDS_BYTES)) == NULL) {
    return(NULL);
  }
  InfoString = MainString;

  InfoString += _stprintf(InfoString, _T("***SECURITY DESCRIPTOR*****************************************\n"));

  GetKernelObjectSecurity(hObject, ReqSecInfo, NULL, 0, &ReqLen);
  do { 
    LastError = GetLastError();
    if(pSD) {
      LocalFree(pSD);
    }
    if(LastError != ERROR_INSUFFICIENT_BUFFER) {
      _stprintf(InfoString, _T("Can't get object security! (%u)\n"), LastError);
      return(MainString);
    }
    if(!(pSD = LocalAlloc(LMEM_FIXED, ReqLen))) {
      _stprintf(InfoString, _T("Heap allocation failed! (%u)\n"), GetLastError());
      return(MainString);
    }
  } while(!GetKernelObjectSecurity(hObject, ReqSecInfo, pSD, ReqLen, &ReqLen));
  if(!IsValidSecurityDescriptor(pSD)) {
    _stprintf(InfoString, _T("Security descriptor is not valid! (%u)\n"), GetLastError());
    LocalFree(pSD);
    return(MainString);
  }

  InfoString += _stprintf(InfoString, _T("Security descriptor length = %u bytes\n"),
                  GetSecurityDescriptorLength(pSD)); 
   
  InfoString += _stprintf(InfoString, _T("\n***Security descriptor - CONTROL*******************************\n"));
  SECURITY_DESCRIPTOR_CONTROL SDControl;
  DWORD dwRevision;
  BOOL IsDACL, IsSACL;
  if(!GetSecurityDescriptorControl(pSD, &SDControl, &dwRevision)) {
    InfoString += _stprintf(InfoString, _T("Can't get control of security descriptor! (%u)"), GetLastError());
  }
  else {
    InfoString += _stprintf(InfoString, _T("Revision = %u\nSecurity descriptor control ="), dwRevision);
    if(SDControl & SE_OWNER_DEFAULTED)
      InfoString += _stprintf(InfoString, _T(" OWNER_DEFAULTED"));
    if(SDControl & SE_GROUP_DEFAULTED)
      InfoString += _stprintf(InfoString, _T(" GROUP_DEFAULTED"));
    if(bIsDACL = (SDControl & SE_DACL_PRESENT))
      InfoString += _stprintf(InfoString, _T(" DACL_PRESENT"));
    if(SDControl & SE_DACL_DEFAULTED)
      InfoString += _stprintf(InfoString, _T(" DACL_DEFAULTED"));
    if(bIsSACL = (SDControl & SE_SACL_PRESENT))
      InfoString += _stprintf(InfoString, _T(" SACL_PRESENT"));
    if(SDControl & SE_SACL_DEFAULTED)
      InfoString += _stprintf(InfoString, _T(" SACL_DEFAULTED"));
    if(SDControl & SE_DACL_AUTO_INHERIT_REQ)
      InfoString += _stprintf(InfoString, _T(" DACL_AUTO_INHERIT_REQ"));
    if(SDControl & SE_SACL_AUTO_INHERIT_REQ)
      InfoString += _stprintf(InfoString, _T(" SACL_AUTO_INHERIT_REQ"));
    if(SDControl & SE_DACL_AUTO_INHERITED)
      InfoString += _stprintf(InfoString, _T(" DACL_AUTO_INHERITED"));
    if(SDControl & SE_SACL_AUTO_INHERITED)
      InfoString += _stprintf(InfoString, _T(" SACL_AUTO_INHERITED"));
    if(SDControl & SE_DACL_PROTECTED)
      InfoString += _stprintf(InfoString, _T(" DACL_PROTECTED"));
    if(SDControl & SE_SACL_PROTECTED)
      InfoString += _stprintf(InfoString, _T(" SACL_PROTECTED"));
    if(SDControl & SE_SELF_RELATIVE)
      InfoString += _stprintf(InfoString, _T(" SELF_RELATIVE"));
  }

  if(ReqSecInfo & OWNER_SECURITY_INFORMATION) {
  InfoString += _stprintf(InfoString, _T("\n\n***Security descriptor - OWNER*********************************\n"));
  PSID pOwner;
  BOOL bOwnerDefaulted;
  if(!GetSecurityDescriptorOwner(pSD, &pOwner, &bOwnerDefaulted)) {
    InfoString += _stprintf(InfoString, _T("Can't get owner of security descriptor! (%u)\n"), GetLastError());
  }
  else {
    if(pOwner == NULL) {
      InfoString += _stprintf(InfoString, _T("Security descriptor doesn't contain owner.\n"));
    }
    else {
      LPTSTR OwnerString;
      if(OwnerString = GetSidInfo(pOwner, SystemNames)) { 
        InfoString += _stprintf(InfoString, _T("%s"), OwnerString);
        LocalFree(OwnerString);
      }
      else {  
        InfoString += _stprintf(InfoString, _T("Can't get owner SID! (%u)\n"), GetLastError());
      }
    }
  }
  }

  if(ReqSecInfo & GROUP_SECURITY_INFORMATION) {
  InfoString += _stprintf(InfoString, _T("\n***Security descriptor - GROUP*********************************\n"));
  PSID pGroup;
  BOOL bGroupDefaulted;
  if(!GetSecurityDescriptorGroup(pSD, &pGroup, &bGroupDefaulted)) {
    InfoString += _stprintf(InfoString, _T("Can't get security descriptor's primary group! (%u)\n"), GetLastError());
  }
  else {
    if(pGroup == NULL) {
      InfoString += _stprintf(InfoString, _T("Security descriptor doesn't contain primary group.\n"));
    }
    else {
      LPTSTR GroupString;
      if(GroupString = GetSidInfo(pGroup, SystemNames)) { 
        InfoString += _stprintf(InfoString, _T("%s"), GroupString);
        LocalFree(GroupString);
      }
      else {  
        InfoString += _stprintf(InfoString, _T("Can't get group SID! (%u)\n"), GetLastError());
      }
    }
  }
  }

  if(ReqSecInfo & DACL_SECURITY_INFORMATION) {
  InfoString += _stprintf(InfoString, _T("\n***Security descriptor - DACL**********************************\n"));
  if(bIsDACL) {
    PACL pDacl;
    BOOL bDaclDefaulted, bDaclPresent;
    if(!GetSecurityDescriptorDacl(pSD, &bDaclPresent, &pDacl, &bDaclDefaulted)) {
      InfoString += _stprintf(InfoString, _T("Can't get security descriptor's DACL! (%u)\n"), GetLastError());
    }
    else {
      if(pDacl == NULL) {
        InfoString += _stprintf(InfoString, _T("NULL DACL -> all access allowed.\n"));
      }
      else {
        LPTSTR DaclString;
        if(DaclString = GetAclInfo(pDacl, SystemNames)) { 
          InfoString += _stprintf(InfoString, _T("%s"), DaclString);
          LocalFree(DaclString);
        } 
        else {  
          InfoString += _stprintf(InfoString, _T("Can't get DACL information! (%u)\n"), GetLastError());
        }
      }
    }
  }
  else {
    InfoString += _stprintf(InfoString, _T("Security descriptor doesn't contain DACL.\n"));
  }
  }

  if(ReqSecInfo & SACL_SECURITY_INFORMATION) {
  InfoString += _stprintf(InfoString, _T("\n***Security descriptor - SACL**********************************\n"));
  if(bIsSACL) {
    PACL pSacl;
    BOOL bSaclDefaulted, bSaclPresent;
    if(!GetSecurityDescriptorSacl(pSD, &bSaclPresent, &pSacl, &bSaclDefaulted)) {
      InfoString += _stprintf(InfoString, _T("Can't get security descriptor's SACL! (%u)\n"), GetLastError());
    }
    else {
      if(pSacl == NULL) {
        InfoString += _stprintf(InfoString, _T("NULL SACL.\n"));
      }
      else {
        LPTSTR SaclString;
        if(SaclString = GetAclInfo(pSacl, SystemNames)) { 
          InfoString += _stprintf(InfoString, _T("%s"), SaclString);
          LocalFree(SaclString);
        } 
        else {  
          InfoString += _stprintf(InfoString, _T("Can't get SACL information! (%u)\n"), GetLastError());
        }
      }
    }
  }
  else {
    InfoString += _stprintf(InfoString, _T("Security descriptor doesn't contain SACL.\n"));
  }
  }

  LocalFree(pSD);
  return(MainString);
}



LPTSTR WINAPI GetTokenType(TOKEN_TYPE tt) {
  if(tt == TokenPrimary)
    return(_T("Primary"));
  else 
    if(tt == TokenImpersonation)
      return( _T("Impersonation"));
    else
      return(_T("<UNKNOWN>"));
}

LPTSTR WINAPI GetImpersonationLevel(SECURITY_IMPERSONATION_LEVEL il) {
  if(il == SecurityAnonymous)
    return( _T("Anonymous"));
  else 
    if(il == SecurityIdentification)
      return(_T("Identification"));
    else
      if(il == SecurityImpersonation)
        return(_T("Impersonation"));
      else
        if(il == SecurityDelegation)
          return(_T("Delegation"));
        else
          return(_T("<UNKNOWN>"));
}

LPTSTR WINAPI GetLuidInfo(PLUID lpLuid, LPCTSTR *SystemNames) {
  DWORD i, ReqLen =1024;
  TCHAR LuidString[1024];
  LPCTSTR rSystemNames[2] =  {NULL, LPCTSTR(-1)};
  if(SystemNames == NULL) {
    SystemNames = rSystemNames;
  }
  BOOL Success = FALSE;
  for(i=0; SystemNames[i] != LPCTSTR(-1); i++) {
    if(Success = LookupPrivilegeName(SystemNames[i], lpLuid, LuidString, &ReqLen))
      break;
  }
  return(Success ? LuidString : NULL);
}

LPTSTR WINAPI PrintToken(HANDLE hToken, LPCTSTR *SystemNames) {
  BOOL bIsDACL, bIsSACL;
  DWORD LastError, ReqLen, i;

  LPTSTR InfoString, MainString;
  if((MainString = (LPTSTR)LocalAlloc(LMEM_FIXED, PSD_TKS_BYTES)) == NULL) {
    return(NULL);
  }
  InfoString = MainString;

  InfoString += _stprintf(InfoString, _T("***TOKEN*******************************************************"));

//user
  InfoString += _stprintf(InfoString, _T("\n\n***Token - USER************************************************\n"));
  PTOKEN_USER ptu = NULL;
  GetTokenInformation(hToken, TokenUser, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token user! (%u)"), LastError);
  }
  else {
    ptu = (PTOKEN_USER)LocalAlloc(LPTR, ReqLen);
    if(ptu && GetTokenInformation(hToken, TokenUser, ptu, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("Attributes = (0x%X)\n%s"), ptu->User.Attributes, GetSidInfo(ptu->User.Sid, SystemNames));
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token user! (%u)"), GetLastError());
    }
  }
  if(ptu)
    LocalFree(ptu);


//groups
  InfoString += _stprintf(InfoString, _T("\n\n***Token - GROUPS**********************************************\n"));
  PTOKEN_GROUPS ptg = NULL;
  GetTokenInformation(hToken, TokenGroups, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token groups! (%u)"), LastError);
  }
  else {
    ptg = (PTOKEN_GROUPS)LocalAlloc(LPTR, ReqLen);
    if(ptg && GetTokenInformation(hToken, TokenGroups, ptg, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("Count = %u\n"), ptg->GroupCount);
      for(i=0; i < ptg->GroupCount; i++) {
        InfoString += _stprintf(InfoString, _T("\n***Group %2u\nAttributes = (0x%X) ~ "), i, ptg->Groups[i].Attributes);
        if(ptg->Groups[i].Attributes & SE_GROUP_MANDATORY)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_MANDATORY "));
        if(ptg->Groups[i].Attributes & SE_GROUP_ENABLED_BY_DEFAULT)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_ENABLED_BY_DEFAULT "));
        if(ptg->Groups[i].Attributes & SE_GROUP_ENABLED)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_ENABLED "));
        if(ptg->Groups[i].Attributes & SE_GROUP_OWNER)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_OWNER "));
#ifndef SE_GROUP_USE_FOR_DENY_ONLY
#define SE_GROUP_USE_FOR_DENY_ONLY      (0x00000010L)
#endif
        if(ptg->Groups[i].Attributes & SE_GROUP_USE_FOR_DENY_ONLY)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_USE_FOR_DENY_ONLY "));
        if((ptg->Groups[i].Attributes & SE_GROUP_LOGON_ID) == SE_GROUP_LOGON_ID)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_LOGON_ID "));
#ifndef SE_GROUP_RESOURCE
#define SE_GROUP_RESOURCE               (0x20000000L)
#endif
        if(ptg->Groups[i].Attributes & SE_GROUP_RESOURCE)
          InfoString += _stprintf(InfoString, _T("SE_GROUP_RESOURCE"));
        InfoString += _stprintf(InfoString, _T("\n%s"), GetSidInfo(ptg->Groups[i].Sid, SystemNames));
      }
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token groups! (%u)"), GetLastError());
    }
  }
  if(ptg)
    LocalFree(ptg);


//privileges
  InfoString += _stprintf(InfoString, _T("\n\n***Token - PRIVILEGES******************************************\n"));
  PTOKEN_PRIVILEGES ptv = NULL;
  GetTokenInformation(hToken, TokenPrivileges, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token privileges! (%u)"), LastError);
  }
  else {
    ptv = (PTOKEN_PRIVILEGES)LocalAlloc(LPTR, ReqLen);
    if(ptv && GetTokenInformation(hToken, TokenPrivileges, ptv, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("Count = %u\n"), ptv->PrivilegeCount);
      for(i=0; i < ptv->PrivilegeCount; i++) {
        InfoString += _stprintf(InfoString, _T("\n***Privilege %2u\nLUID = 0x%.8X%.8X ~ %s\nAttributes = (0x%X) "), i, ptv->Privileges[i].Luid.HighPart, ptv->Privileges[i].Luid.LowPart, GetLuidInfo(&ptv->Privileges[i].Luid, SystemNames), ptv->Privileges[i].Attributes);
        if(ptv->Privileges[i].Attributes == 0)
          InfoString += _stprintf(InfoString, _T("disabled"));
        else {
          if(ptv->Privileges[i].Attributes & SE_PRIVILEGE_ENABLED_BY_DEFAULT)
            InfoString += _stprintf(InfoString, _T("SE_PRIVILEGE_ENABLED_BY_DEFAULT "));
          if(ptv->Privileges[i].Attributes & SE_PRIVILEGE_ENABLED)
            InfoString += _stprintf(InfoString, _T("SE_PRIVILEGE_ENABLED "));
          if(ptv->Privileges[i].Attributes & SE_PRIVILEGE_USED_FOR_ACCESS)
            InfoString += _stprintf(InfoString, _T("SE_PRIVILEGE_USED_FOR_ACCESS"));
        }
        InfoString += _stprintf(InfoString, _T("\n"));
      }
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token privileges! (%u)"), GetLastError());
    }
  }
  if(ptv)
    LocalFree(ptv);


//owner
  InfoString += _stprintf(InfoString, _T("\n\n***Token - OWNER***********************************************\n"));
  PTOKEN_OWNER pto = NULL;
  GetTokenInformation(hToken, TokenOwner, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token owner! (%u)"), LastError);
  }
  else {
    pto = (PTOKEN_OWNER)LocalAlloc(LPTR, ReqLen);
    if(pto && GetTokenInformation(hToken, TokenOwner, pto, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("%s"), GetSidInfo(pto->Owner, SystemNames));
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token owner! (%u)"), GetLastError());
    }
  }
  if(pto)
    LocalFree(pto);


//primary group
  InfoString += _stprintf(InfoString, _T("\n\n***Token - PRIMARY GROUP***************************************\n"));
  PTOKEN_PRIMARY_GROUP ptp = NULL;
  GetTokenInformation(hToken, TokenPrimaryGroup, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token primary group! (%u)"), LastError);
  }
  else {
    ptp = (PTOKEN_PRIMARY_GROUP)LocalAlloc(LPTR, ReqLen);
    if(ptp && GetTokenInformation(hToken, TokenPrimaryGroup, ptp, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("%s"), GetSidInfo(ptp->PrimaryGroup, SystemNames));
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token primary group! (%u)"), GetLastError());
    }
  }
  if(ptg)
    LocalFree(ptp);

//default dacl
  InfoString += _stprintf(InfoString, _T("\n\n***Token - DEFAULT DACL****************************************\n"));
  PTOKEN_DEFAULT_DACL ptd = NULL;
  GetTokenInformation(hToken, TokenDefaultDacl, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token default dacl! (%u)"), LastError);
  }
  else {
    ptd = (PTOKEN_DEFAULT_DACL)LocalAlloc(LPTR, ReqLen);
    if(ptd && GetTokenInformation(hToken, TokenDefaultDacl, ptd, ReqLen, &ReqLen)) {
      if(ReqLen)  
        InfoString += _stprintf(InfoString, _T("%s"), GetAclInfo(ptd->DefaultDacl, SystemNames));
      else
        InfoString += _stprintf(InfoString, _T("No default dacl."));
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token default dacl! (%u)"), GetLastError());
    }
  }
  if(ptd)
    LocalFree(ptd);


//source
  InfoString += _stprintf(InfoString, _T("\n\n***Token - SOURCE**********************************************\n"));
  PTOKEN_SOURCE pts = NULL;
  GetTokenInformation(hToken, TokenSource, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token source! (%u)\n"), LastError);
  }
  else {
    pts = (PTOKEN_SOURCE)LocalAlloc(LPTR, ReqLen);
    if(pts && GetTokenInformation(hToken, TokenSource, pts, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("Name = %.8s\nLUID = 0x%.8X%.8X"), pts->SourceName, pts->SourceIdentifier.HighPart, pts->SourceIdentifier.LowPart);
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token source! (%u)"), GetLastError());
    }
  }
  if(pts)
    LocalFree(pts);



//type
  InfoString += _stprintf(InfoString, _T("\n\n***Token - TYPE************************************************\n"));
  PTOKEN_TYPE ptt = NULL;
  GetTokenInformation(hToken, TokenType, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token type! (%u)\n"), LastError);
  }
  else {
    ptt = (PTOKEN_TYPE)LocalAlloc(LPTR, ReqLen);
    if(ptt && GetTokenInformation(hToken, TokenType, ptt, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, GetTokenType(*ptt));
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token type! (%u)"), GetLastError());
    }
  }
  if(ptt)
    LocalFree(ptt);


//imp. level
  InfoString += _stprintf(InfoString, _T("\n\n***Token - IMPERSONATION LEVEL*********************************\n"));
  PSECURITY_IMPERSONATION_LEVEL pil = NULL;
  GetTokenInformation(hToken, TokenImpersonationLevel, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("No impersonation! (%u)\n"), LastError);
  }
  else {
    pil = (PSECURITY_IMPERSONATION_LEVEL)LocalAlloc(LPTR, ReqLen);
    if(pil && GetTokenInformation(hToken, TokenImpersonationLevel, pil, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, GetImpersonationLevel(*pil));
    }
    else {
      InfoString += _stprintf(InfoString, _T("No impersonation! (%u)\n"), GetLastError());
    }
  } 
  if(pil)
    LocalFree(pil);


//stat
  InfoString += _stprintf(InfoString, _T("\n\n***Token - STATISTICS******************************************\n"));
  PTOKEN_STATISTICS ptc = NULL;
  GetTokenInformation(hToken, TokenStatistics, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token statistics! (%u)\n"), LastError);
  }
  else {
    ptc = (PTOKEN_STATISTICS)LocalAlloc(LPTR, ReqLen);
    if(ptc && GetTokenInformation(hToken, TokenStatistics, ptc, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("TokenId = 0x%.8X%.8X\n")
                                          _T("AuthenticationId = 0x%.8X%.8X\n")
                                          _T("ExpirationTime = 0x%.8X%.8X\n")
                                          _T("TokenType = %s\n")
                                          _T("ImpersonationLevel = %s\n")
                                          _T("DynamicCharged = %u\n")
                                          _T("DynamicAvailable = %u\n")
                                          _T("GroupCount = %u\n")
                                          _T("PrivilegeCount = %u\n")
                                          _T("ModifiedId = 0x%.8X%.8X\n"),
                                          ptc->TokenId.HighPart, ptc->TokenId.LowPart,
                                          ptc->AuthenticationId.HighPart, ptc->AuthenticationId.LowPart,
                                          ptc->ExpirationTime.HighPart, ptc->ExpirationTime.LowPart,
                                          GetTokenType(ptc->TokenType),
                                          (ptc->TokenType == TokenImpersonation) ? GetImpersonationLevel(ptc->ImpersonationLevel) : _T("no impersonation"),
                                          ptc->DynamicCharged,
                                          ptc->DynamicAvailable,
                                          ptc->GroupCount,
                                          ptc->PrivilegeCount,
                                          ptc->ModifiedId.HighPart, ptc->ModifiedId.LowPart);
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token statistics! (%u)"), GetLastError());
    }
  }
  if(ptc)
    LocalFree(ptc);


//restricted sids
  InfoString += _stprintf(InfoString, _T("\n\n***Token - RESTRICTED SIDS*************************************\n"));
  ptg = NULL;
  GetTokenInformation(hToken, TokenRestrictedSids, NULL, 0, &ReqLen);
  LastError = GetLastError();
  if(LastError != ERROR_INSUFFICIENT_BUFFER) {
    InfoString += _stprintf(InfoString, _T("Can't get token restricted sids! (%u)"), LastError);
  }
  else {
    ptg = (PTOKEN_GROUPS)LocalAlloc(LPTR, ReqLen);
    if(ptg && GetTokenInformation(hToken, TokenRestrictedSids, ptg, ReqLen, &ReqLen)) {
      InfoString += _stprintf(InfoString, _T("Count = %u\n"), ptg->GroupCount);
      for(i=0; i < ptg->GroupCount; i++) {
        InfoString += _stprintf(InfoString, _T("\n***Restricted SID %2u\nAttributes = (0x%X)"), i, ptg->Groups[i].Attributes);
        InfoString += _stprintf(InfoString, _T("\n%s"), GetSidInfo(ptg->Groups[i].Sid, SystemNames));
      }
    }
    else {
      InfoString += _stprintf(InfoString, _T("Can't get token restricted sids! (%u)"), LastError);
    }
  }
  if(ptg)
    LocalFree(ptg);


//session id
  InfoString += _stprintf(InfoString, _T("\n\n***Token - SESSION ID******************************************\n"));
  DWORD SessionId;
  if(GetTokenInformation(hToken, TokenSessionId, &SessionId, sizeof(SessionId), &ReqLen)) {
    InfoString += _stprintf(InfoString, _T("%u"), SessionId);
  }
  else {
    InfoString += _stprintf(InfoString, _T("Can't get token session id! (%u)"), GetLastError());
  }

  return(MainString);
}
#endif