TITLE	clockwise.asm, version 1.0
comment #

	Programmed by G. Adam Stanislav
	Whiz Kid Technomagic
	Rhinelander, WI 54501, USA

	whizkid@bigfoot.com
	http://www.bigfoot.com/~whizkid

	March 26-27, 1998

	Copyright  1998 G. Adam Stanislav
	All rights reserved

	This is a Windows 95/NT DLL which determines whether a polygon has been
	drawn in the clockwise or counter-clockwise direction.

	Why bother? It is important when designing or interpreting Postscript
	fonts: If a path is drawn counter-clockwise, it is to be filled; if it
	is clockwise, it represents a hole inside another path.

	TrueType fonts, too, need to determine whether a path is clockwise
	or counter-clockwise, except in this case a clockwise path is to be
	filled, while a counter-clockwise path represents a hole.

	Special thanks to Paul Bourke of Brain Dynamics Research Unit,
	Mental Health Research Institute, Victoria, Australia. He developed
	the algorithm used here. My only contribution was bugging him with
	questions until he came up with a very elegant solution of how to
	determine whether a path is clockwise or counter-clockwise.

	He even created a web page for the algorithm at:

		http://www.mhri.edu.au/~pdb/geometry/clockwise/

	The following C code is his:

	/*
	Return the clockwise status of a curve, clockwise or counterclockwise
	n vertices making up curve p return 0 for incomputables eg: colinear points

		CLOCKWISE == 1
		COUNTERCLOCKWISE == -1
		
	It is assumed that
		- the polygon is closed
		- the last point is not repeated.
		- the polygon is simple (does not intersect itself or have holes)
	*/

	int ClockWise(XY *p,int n)
	{
		int i,j,k;
		int count = 0;
		double z;
		
		if (n < 3)
			return(0);
			
		for (i=0;i<n;i++) {
			j = (i + 1) % n;
			k = (i + 2) % n;
			z  = (p[j].x - p[i].x) * (p[k].y - p[j].y);
			z -= (p[j].y - p[i].y) * (p[k].x - p[j].x);
			if (z < 0)
				count--;
			else if (z > 0)
				count++;
		}
		if (count > 0)
			return(COUNTERCLOCKWISE);
		else if (count < 0)
			return(CLOCKWISE);
		else
		return(0);
	}

	This DLL is based on Paul's code, with the main difference of using a 64-bit
	integer for z instead of double. The reason is that this DLL is Windows 95
	specific, and uses Windows 95 POINT structure which declares x and y as
	long (32-bit signed values). And since they need to be multiplied, a 64-bit
	signed integer is required to store the result without the need for floating
	point math.

	Another difference is that this DLL's for loop starts with i set to n,
	going down to 1. This is simply to take advantage of the ECX counter of
	Intel processors. The result is the same thanks to the commutative law.

	Thirdly, I have added a coordinate system check. Paul's code works correctly
	when the polygon is in a standard Cartesian coordinate system, i.e., the x
	axis increases to the right, the y axis to the top. This is also how fonts
	work. However, in Windows text mapping mode the y axis increases downwards,
	and in the isotropic and anisotropic mode either axis can increase either way.

	Since this is a Windows 95 DLL, it seems necessary to add a third parameter
	to ClockWise(), namely DWORD coord. The value of this parameter should be
	set as follow:

		x grows right, y grows upwards	= 0
		x grows left, y grows upwards	= 1
		x grows right, y grows downward	= 0
		x grows left, y grows downward	= 1

	However, any non-zero value may be used where 1 is listed above.

	To find the latest version of this DLL, go to http://www.bigfoot.com/~whizkid
	and follow the link to the Windows assembly language page.

	Note: The code in this DLL is thread-safe. That means any number of threads
	may call it at the same time.

	The legal stuff:
	================

	You may use clockwise.dll with commercial or non-commercial applications
	free of charge. However, if you use this DLL commercially, you are required
	to place my copyright notice for the DLL in your documentation and in the
	"About" display of your program. Note that an icon (clockwise.ico) is
	included in the resource section of this DLL. You may use it in the "About"
	display of my copyright. It was declared in the resource file as resource
	number 1 (i.e. 1 ICON DISCARDABLE "clockwise.ico").

	The key word above is APPLICATIONS: You may not distribute this DLL as
	with an operating system (for example, Microsoft may not include this DLL
	with any version of Windows without an express permission from the author).

	If you distribute this DLL with your applications, install it in Windows
	system directory (folder). In doing so, do not overwrite any newer version
	of the DLL. You must include this source code whenever you are distributing
	the DLL.

	You may not modify this DLL in any way. That implies that you may not remove
	the resources linked with this DLL (file version, copyright and other
	information, the icon).

	Exception: Paul Bourke may use this code in any way he sees fit. After all,
	this DLL would not be written without his help.

#
.386
.model flat

.code

align DWORD
_dll@12	proc

	mov	eax, 1
	ret	12

_dll@12	endp

_p$ = 8
_n$ = 12
_coord$ = 16
_z$ = -8
align DWORD
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; #include <windows.h>
;;
;; int __stdcall ClockWise(POINT *p, unsigned int n, unsigned int coord);
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ClockWise@12 proc	public

	push	ebp
	mov	ebp, esp
	sub	esp, 8
	push	ebx
	push	ecx
	push	edx
	push	esi
	push	edi

	sub	eax, eax
	mov	ecx, DWORD PTR _n$[ebp]
	cmp	ecx, 3
	jb	$Done

	sub	edi, edi	; EDI = count = 0
	mov	ebx, DWORD PTR _p$[ebp]		; EBX = p
	mov	esi, ecx		; ESI = n

$Loop:
	; ECX = i + 1
	push	esi		; Save n
	lea	edx, [ecx+1]	; EDX = i + 2
	cmp	edx, esi
	jb	@F
	sub	edx, esi

	; EDX = k = (i + 2) % n

@@:
	push	DWORD PTR [ebx][edx*8+4]	; p[k].y
	push	DWORD PTR [ebx][edx*8]	; p[k].x

	dec	edx
	jns	@F
	add	edx, esi

	; EDX = j = (i + 1) % n

@@:
	mov	esi, DWORD PTR [ebx][edx*8+4]	; p[j].y
	sub	esi, DWORD PTR [ebx][ecx*8-4]	; - p[i].y
	pop	eax			; p[k].x
	sub	eax, DWORD PTR [ebx][edx*8]	; - p[j].x
	push	edx
	imul	esi
	mov	DWORD PTR _z$[ebp], eax
	mov	DWORD PTR _z$[ebp+4], edx
	pop	edx

	mov	esi, DWORD PTR [ebx][edx*8]	; p[j].x
	sub	esi, DWORD PTR [ebx][ecx*8-8]	; - p[i].x
	pop	eax		; p[k].y
	sub	eax, DWORD PTR [ebx][edx*8+4]	; - p[j].y
	imul	esi
	pop	esi		; Restore n
	sub	eax, _z$[ebp]
	sbb	edx, _z$[ebp+4]
	js	$decedi
	jnz	$incedi
	or	eax, eax
	jz	$LoopIt

$incedi:
	inc	edi
	loop	$Loop
	jmp	@F

$decedi:
	dec	edi

$LoopIt:
	loop	$Loop

@@:
	sub	eax, eax	; EAX = direction unknown
	cmp	edi, eax
	je	$Done

	dec	eax			; EAX = COUNTERCLOCKWISE
	cmp	edi, eax
	jg	$Done

	add	eax, 2		; EAX = CLOCKWISE

$Done:
	; Check the coordinate system
	mov	ecx, _coord$[ebp]
	jecxz	@F
	neg	eax

@@:
	pop	edi
	pop	esi
	pop	edx
	pop	ecx
	pop	ebx
	leave
	ret	12

_ClockWise@12 endp

end	_dll@12
