///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                     Labelmaster Plugin for OllyDbg                        //
//                                                                           //
//                     Copyright (C) 2004 Joe Stewart                        //
//                                                              	     //
// This plugin allows you to manipulate labels in batch. Its basic functions //
// are (1) Copy user-defined label to comment (2) save user-defined labels   //
// or comments to a file and (3) read user-defined labels or comments from a //
// file.								     //
//                                                                           //
// This code is distributed "as is", without warranty of any kind, expressed //
// or implied, including, but not limited to warranty of fitness for any     //
// particular purpose. In no event will Joe Stewart be liable to you for any //
// special, incidental, indirect, consequential or any other damages caused  //
// by the use, misuse, or the inability to use of this code, including any   //
// lost profits or lost savings, even if Joe Stewart has been advised of the //
// possibility of such damages. Or, translated into English: use at your own //
// risk!                                                                     //
//                                                                           //
// This code is free. You can modify this code, include parts of it in your  //
// own programs and redistribute modified code provided that you remove all  //
// copyright messages or, if changes are significant enough, substitute them //
// with your own copyright.                                                  //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

// VERY IMPORTANT NOTICE: COMPILE THIS DLL WITH BYTE ALIGNMENT OF STRUCTURES
// AND UNSIGNED CHAR!

#include <windows.h>
#include <stdio.h>
#include <string.h>

#include "plugin.h"

HINSTANCE        hinst;
HWND             hwmain;

void CopyLabelsToComment(void);
void ImportLabelsFromFile(void);
void ImportCommentsFromFile(void);
void ExportLabelsToFile(void);
void ExportCommentsToFile(void);
int ReadLine(FILE *fd, char *ptr, int maxlen);

BOOL WINAPI DllEntryPoint(HINSTANCE hi,DWORD reason,LPVOID reserved) {
  if (reason==DLL_PROCESS_ATTACH)
    hinst=hi;
  return 1;
};

extc int _export cdecl ODBG_Plugindata(char shortname[32]) {
  strcpy(shortname,"Labelmaster");
  return PLUGIN_VERSION;
};

extc int _export cdecl ODBG_Plugininit( int ollydbgversion,HWND hw,ulong *features) {
  if (ollydbgversion<PLUGIN_VERSION)
    return -1;
  hwmain=hw;
  Addtolist(0,0,"Labelmaster plugin v0.1");
  Addtolist(0,-1,"  Copyright (C) 2004 Joe Stewart");
  return 0;
};

extc int _export cdecl ODBG_Pluginmenu(int origin,char data[4096],void *item) {
  switch (origin) {
    case PM_MAIN:
      strcpy(data,"0 &Copy Labels To Comments|"
                  "1 &Import Labels From File,"
                  "2 &Import Comments From File|"
		  "3 &Export Labels To File,"
		  "4 &Export Comments To File|"
                  "5 &Help,"
                  "6 &About");
      return 1;
    default: break;
  };
  return 0;
};

extc void _export cdecl ODBG_Pluginaction(int origin,int action,void *item) {
  if (origin==PM_MAIN) {
    switch (action) {
      case 0:
	CopyLabelsToComment();
        break;
      case 1:
	ImportLabelsFromFile();
        break;
      case 2:
	ImportCommentsFromFile();
        break;
      case 3:
	ExportLabelsToFile();
        break;
      case 4:
	ExportCommentsToFile();
        break;
      case 5:
        MessageBox(hwmain,
          "The Labelmaster plugin allows you to insert user-defined\n"
          "labels and comments in batch from a tab-delimited textfile,\n"
          "one entry per line.\n\n"
          "The file format is:\n"
	  "<hex address><tab><comment or label>\n\n"
          "An example:\n\n"
          "004011cb	EntryPoint\n"
          "00401Fcf	ExitPoint\n\n"
          "Labels and comments should be defined in separate files.\n\n"
	  "You can export labels and comments from your current window\n"
	  "to a file in this same format for import at a later time.\n\n"
          "Another useful feature is the ability to copy all\n"
          "user-defined labels to comments, to provide additional\n"
          "disassembly context\n\n",
          "Labelmaster Plugin Help",MB_OK|MB_ICONINFORMATION);
        break;
      case 6:
        MessageBox(hwmain,
          "Labelmaster plugin v0.1\n"
          "Import/export labels in batch\n"
          "Copyright (C) 2004 Joe Stewart",
          "Labelmaster Plugin",MB_OK|MB_ICONINFORMATION);
        break;
      default: break;
    };
  };
};

void CopyLabelsToComment(void) {
  ulong i, start, stop;
  char name[256];
  char result[256];
  struct t_dump *currentmodule;
  int count = 0;

  currentmodule = (t_dump *)Plugingetvalue(VAL_CPUDASM);
  start = currentmodule->base;
  stop = start + currentmodule->size;

  for (i=start; i < stop; i++) {
    if (Findname(i, NM_LABEL, name) != 0) {
      if (Quickinsertname(i, NM_COMMENT, name) != -1)
        count++;
    }
  }
  Mergequicknames();
  Redrawdisassembler();
  sprintf(result, "Copied %d user-defined labels from 0x%x-0x%x to user-defined comments\n",
		   count, start, stop-1);
  MessageBox(hwmain, result,
          "Labelmaster Copy Result",MB_OK|MB_ICONINFORMATION);
}

void ImportLabelsFromFile(void) {
  FILE *fd;
  char label[512];
  char result[256];
  ulong addr = 0; 
  int count = 0;
  char buf[550];
  int bufsize = 550;
  char filename[2048];

  filename[0] = '\0';
  if (Browsefilename("Select File for Label Import", filename, ".txt",0) == 0)
    return;

  fd = fopen(filename, "r");

  if (fd == NULL) {
    MessageBox(hwmain, "Failed to open file!",
            "Open File Failed",MB_OK|MB_ICONINFORMATION);
    return;
  }

  while(ReadLine(fd,buf,bufsize) > 0) {
    sscanf(buf, "%x %[\x20-\x7e]", &addr, label);
    if (addr != 0) {
      if (Quickinsertname(addr, NM_LABEL, label) != -1)
        count++;
      addr = 0;
    }
  }
  fclose(fd);
  Mergequicknames();
  Redrawdisassembler();
  sprintf(result, "Copied %d user-defined labels from file\n", count);
  MessageBox(hwmain, result,
          "Labelmaster Import Result",MB_OK|MB_ICONINFORMATION);
}

void ImportCommentsFromFile(void) {
  FILE *fd;
  char label[512];
  char result[256];
  ulong addr = 0; 
  int count = 0;
  char buf[550];
  int bufsize = 550;
  char filename[2048];

  filename[0] = '\0';
  if (Browsefilename("Select File for Comment Import", filename, ".txt",0) == 0)
    return;

  fd = fopen(filename, "r");

  if (fd == NULL) {
    MessageBox(hwmain, "Failed to open file!",
            "Open File Failed",MB_OK|MB_ICONINFORMATION);
    return;
  }

  while(ReadLine(fd,buf,bufsize) > 0) {
    sscanf(buf, "%x %[\x20-\x7e]", &addr, label);
    if (addr != 0) {
      if (Quickinsertname(addr, NM_COMMENT, label) != -1)
        count++;
      addr = 0;
    }
  }
  fclose(fd);
  Mergequicknames();
  Redrawdisassembler();
  sprintf(result, "Copied %d user-defined comments from file\n", count);
  MessageBox(hwmain, result,
          "Labelmaster Import Result",MB_OK|MB_ICONINFORMATION);
};

void ExportLabelsToFile(void) {
  FILE *fd;
  char result[256];
  int count = 0;
  ulong i, start, stop;
  char name[256];
  struct t_dump *currentmodule;
  char filename[2048];

  filename[0] = '\0';
  if (Browsefilename("Select File for Label Export", filename, ".txt",0) == 0)
    return;

  fd = fopen(filename, "w");

  if (fd == NULL) {
    MessageBox(hwmain, "Failed to open export file!",
            "Open File Failed",MB_OK|MB_ICONINFORMATION);
    return;
  }

  currentmodule = (t_dump *)Plugingetvalue(VAL_CPUDASM);
  start = currentmodule->base;
  stop = start + currentmodule->size;

  for (i=start; i < stop; i++) {
    if (Findname(i, NM_LABEL, name) != 0) {
      if (fprintf(fd, "%x\t%s\n", i, name) > 0)
        count++;
    }
  }
  fclose(fd);
  sprintf(result, "Exported %d user-defined labels from 0x%x-0x%x to file\n",
		   count, start, stop-1);
  MessageBox(hwmain, result,
          "Labelmaster Export Result",MB_OK|MB_ICONINFORMATION);
}

void ExportCommentsToFile(void) {
  FILE *fd;
  char result[256];
  int count = 0;
  ulong i, start, stop;
  char name[256];
  struct t_dump *currentmodule;
  char filename[2048];

  filename[0] = '\0';
  if (Browsefilename("Select File for Comment Export", filename, ".txt",0) == 0)
    return;

  fd = fopen(filename, "w");

  if (fd == NULL) {
    MessageBox(hwmain, "Failed to open export file!",
            "Open File Failed",MB_OK|MB_ICONINFORMATION);
    return;
  }

  currentmodule = (t_dump *)Plugingetvalue(VAL_CPUDASM);
  start = currentmodule->base;
  stop = start + currentmodule->size;

  for (i=start; i < stop; i++) {
    if (Findname(i, NM_COMMENT, name) != 0) {
      if (fprintf(fd, "%x\t%s\n", i, name) > 0)
        count++;
    }
  }
  fclose(fd);
  sprintf(result, "Exported %d user-defined comments from 0x%x-0x%x to file\n",
		   count, start, stop-1);
  MessageBox(hwmain, result,
          "Labelmaster Export Result",MB_OK|MB_ICONINFORMATION);
}


int ReadLine(FILE *fd, char *ptr, int maxlen) { 
  int n,rc; 
  char c,g; 

  for (n=1; n < maxlen; n++) 
  { 
    rc = fread(&c,1,1,fd);
    if (rc == 1) 
    { 
      if (c == '\n') 
      { 		// don't include LF
        *ptr = 0; 
        return n;
      }
      if (c == '\r') 
      {		// don't include CRLF
        fread(&g,1,1,fd);
        *ptr = 0; 
        return n;
      }
      *ptr++ = c; 
    } 
    else 
    { 
      *ptr = 0;
      return 0;
    }
  }
  *ptr = 0;
  return n;
} 

extc void _export cdecl ODBG_Pluginreset(void) {
};

extc int _export cdecl ODBG_Pluginclose(void) {
  return 0;
};

extc void _export cdecl ODBG_Plugindestroy(void) {
};
