/*
msr_ring3 - last updated: 17th April 2004

Module to give ring3 access to RDMSR/WRMSR from ring3, working under both 9x and NT.

NT: Interacts with a Kernel Mode Driver, f0dd_msr.sys - source in f0dd_msr.asm .
This requires administrator privileges. Driver hasn't been widely tested, but should
work on NT4 and upwards?

9x: uses a wellknown IDT hack to gain ring0 - Source is in w9x_ring0.asm.
The code should work with all 9x versions - it's been there since the start,
has never been patched, and MS has dropped 9x support, so it should remain.
Easier and faster than coding a VxD :)

See the included msr_license.txt for, well, the license. Basically: no GPL.
Copyright by f0dder - f0dder(at)flork(dot)dk - http://f0dder.has.it
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include "msr_ring3.h"

/*
Load the following APIs dynamically from advapi32.dll, so that we can work on both
9x and NT without depending on linker delay-load (yes, this will in fact mean manual
delay-loading).

DeleteService
ControlService
CloseServiceHandle
CreateServiceA
OpenSCManagerA
StartServiceA
*/

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// defines and such
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// DeviceIoControl codes used by f0dd_msr.sys
//
#define IOCTL_MSR_RDMSR CTL_CODE(FILE_DEVICE_UNKNOWN, 0x800, METHOD_BUFFERED, FILE_READ_ACCESS|FILE_WRITE_ACCESS)
#define IOCTL_MSR_WRMSR CTL_CODE(FILE_DEVICE_UNKNOWN, 0x900, METHOD_BUFFERED, FILE_READ_ACCESS|FILE_WRITE_ACCESS)

//
// ripped from devioctl.h - included so people without the DDK can 
// play around with the loader code.
//
#define CTL_CODE( DeviceType, Function, Method, Access ) (                 \
    ((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method) \
)
#define FILE_DEVICE_UNKNOWN			0x00000022
#define METHOD_BUFFERED				0
#define FILE_READ_ACCESS			( 0x0001 )    // file & pipe
#define FILE_WRITE_ACCESS			( 0x0002 )    // file & pipe



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// prototypes
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// imports from other modules
extern "C" {
void __stdcall w9x_ring0hack_rdmsr(uint msr_index, uint *msr_reg_high, uint *msr_reg_low);
void __stdcall w9x_ring0hack_wrmsr(uint msr_index, uint msr_reg_high, uint msr_reg_low);
}

// private prototypes from this module
static void __stdcall NT_cleanup(void);
static bool __stdcall NT_connect_driver(void);
static bool __stdcall NT_disconnect_driver(void);



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// global static variables
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static bool			g_bIsNT = false;	// OS type; true=NT, false=9x
static bool			g_bIsInitialized = false;
static DWORD		g_lastError = ERROR_SUCCESS;

// the following variables are required if we're running on NT
static char			*szKMDname = "\\f0dd_msr.sys";			// file name of KMD
static char			*szDriverID = "\\\\.\\f0dd_msr";	// symbolic name of KMD
static SC_HANDLE	g_hSvcMgr = 0;		// Service Manager Handle
static SC_HANDLE	g_hService = 0;		// Service/Driver handle
static HANDLE		g_hDriver = 0;		// file handle to driver



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// public code interface
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
bool msr_initialize(void)
{
	// start by detecting NT vs. 9x . High-order of GetVersion return value is
	// set for 9x windows versions, but left clear for NT versions.
	//BUGBUG: use GetVersionEx instead?
	g_bIsNT = !(GetVersion() & 0x80000000);

	if(g_bIsNT)
	{
		g_bIsInitialized = NT_connect_driver();
	} else 
	{
		// nothing much to do for w9x
		g_bIsInitialized = true;
	}

	return g_bIsInitialized;
}


bool msr_deinitialize(void)
{
	if(g_bIsNT)
	{
		bool success;

		g_bIsInitialized = false;
		success = NT_disconnect_driver();
		NT_cleanup();
		return success;
	} else
	{
		// nothing much to do for w9x
		g_bIsInitialized = false;
		return true;
	}
}


DWORD msr_GetLastError(void)
{
	return g_lastError;
}


bool msr_rdmsr(uint msr_index, uint *msr_reg_high, uint *msr_reg_low)
{
	if(g_bIsNT)
	{
		DWORD	bytes_ret;	// bytes returned from driver
		uint	buffer[2];	// msr device input/output buffer

		buffer[0] = msr_index;
		if(FALSE == DeviceIoControl(g_hDriver, IOCTL_MSR_RDMSR, buffer, 4, buffer, 8, &bytes_ret, NULL))
		{
			g_lastError = GetLastError();
			return false;
		}

		*msr_reg_high = buffer[0];
		*msr_reg_low = buffer[1];

		return true;
	} else {
		w9x_ring0hack_rdmsr(msr_index, msr_reg_high, msr_reg_low);
		return true;
	}
}


bool msr_wrmsr(uint msr_index, uint msr_reg_high, uint msr_reg_low)
{
	if(g_bIsNT)
	{
		DWORD	bytes_ret;	// bytes returned from driver
		uint	buffer[3];	// msr device input buffer

		buffer[0] = msr_index;
		buffer[1] = msr_reg_high;
		buffer[2] = msr_reg_low;
		if(FALSE == DeviceIoControl(g_hDriver, IOCTL_MSR_WRMSR, buffer, 12, NULL, 0, &bytes_ret, NULL))
		{
			g_lastError = GetLastError();
			return false;
		}

		return true;
	} else {
		w9x_ring0hack_wrmsr(msr_index, msr_reg_high, msr_reg_low);
		return true;
	}
}



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// module-private NT code
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void NT_cleanup(void)
{
	if(g_hSvcMgr)
	{
		DeleteService(g_hSvcMgr);
		CloseServiceHandle(g_hSvcMgr);
		g_hSvcMgr = NULL;
	}

	if(g_hService)
	{
		DeleteService(g_hService);
		CloseServiceHandle(g_hService);
		g_hSvcMgr = NULL;
	}
}


static bool NT_connect_driver(void)
{
	uint	dirlen;
	char	kmd_path[MAX_PATH + 1];

	// first, open the service manager
	g_hSvcMgr = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS);
	if(g_hSvcMgr == NULL) {
		//msgbox("error with OpenSCManager");
		g_lastError = GetLastError();
		return false;
	}

	// "create the service" - register our driver with the service manager.
	// This requires the full path to the driver, so append the drivername
	// to the current directory.
	dirlen = GetCurrentDirectory(lengthof(kmd_path), kmd_path);
	lstrcat(kmd_path, szKMDname);

	g_hService = CreateService(g_hSvcMgr, &szKMDname[1], "ring3 RDMSR/WRMSR support", SERVICE_ALL_ACCESS,
		SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START, SERVICE_ERROR_IGNORE,
		kmd_path, NULL, NULL, NULL, NULL, NULL);
	if(NULL == g_hService)
	{
		g_lastError = GetLastError();
		NT_cleanup();
		//msgbox("error with CreateService");
		return false;
	}

	// "start the service" - load the driver, I assume
	if(FALSE == StartService(g_hService, 0, NULL))
	{
		g_lastError = GetLastError();
		NT_cleanup();
		//msgbox("error with StartService");
		return false;
	}

	// "open file" - connect ring3 app to the driver
	g_hDriver = CreateFile(szDriverID, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE,
		NULL, OPEN_EXISTING, 0, NULL);
	if(INVALID_HANDLE_VALUE == g_hDriver)
	{
		g_lastError = GetLastError();
		NT_cleanup();
		//msgbox("error with CreateFile");
		return false;
	}

	// device ready for use
	return true;
}


bool NT_disconnect_driver(void)
{
	SERVICE_STATUS	svcStatus;
	bool			success;

	CloseHandle(g_hDriver);

	success = (FALSE != ControlService(g_hService, SERVICE_CONTROL_STOP, &svcStatus));
	NT_cleanup();
	g_lastError = GetLastError();

	return success;
}
