; Standard Library - string manipulation
; by Gaz
;
%ifndef _MYSTRING
%define _MYSTRING
[BITS 32]
[section .bss]
_mystr_spare	resd 1

%ifdef _MYARGS_INIT_MACRO
_myargs_c	resd 1
_myargs_v	resd 1
_myargs_psp	resd 1
%endif

[section .data]

%ifdef _MYSTR_INTTOHEX_MACRO
_mystr_hexdigits_str	db '0123456789ABCDEF'
%else
%ifdef _MYSTR_INTTOHEX_FULL_MACRO
_mystr_hexdigits_str	db '0123456789ABCDEF'
%endif
%endif

[section .text]
;
;------------------------------------------------------------------------------------
;
; Argc/argv/psp routines
;
;------------------------------------------------------------------------------------
;
; Call to initialise the argument routines. Ensure that you call this at the start of your
; source so the routine can get the right register values.
;
%ifdef _MYARGS_INIT_MACRO
_myargs_init_2:
	mov	[_myargs_c],esi			; esi holds the number of arguments
	mov	[_myargs_v],edi			; edi points to array of pointers to each argument
	mov	[_myargs_psp],ebp		; ebp points to array of environment variables
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Returns the number of parameters.
;
%ifdef _MYARGS_ARGC_MACRO
_myargs_argc_2:
	push	eax
	mov	eax,[_myargs_c]
	mov	[_mystr_spare],eax
	pop	eax
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Returns the address of the parameter number requested, or returns
; zero if the parameter was out of range. Parameters are numbered from 1 upwards
;
%ifdef _MYARGS_GET_PARAMETER_MACRO
_myargs_get_parameter_2:
	push	eax
	push	ebx
	mov	eax,[esp + 12]
	dec	eax
	cmp	eax,[_myargs_c]			; parameter above max parameter number?
	jae	_myargs_get_parameter_error	; yes
	shl	eax,2				; *4 (4 bytes/address)
	mov	ebx,[_myargs_v]			; address of array pointers
	mov	eax,[ebx + eax]			; get address
	mov	[_mystr_spare],eax
	pop	ebx				; restore registers
	pop	eax
	clc					; flag success
	ret
_myargs_get_parameter_error:
	mov	[_mystr_spare],dword 0
	pop	ebx				; restore registers
	pop	eax
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Returns the environment variable asked for.
;
%ifdef _MYARGS_GET_VARIABLE_MACRO
_myargs_get_variable_2:
	push	eax				; save registers
	push	ecx
	push	esi
	push	edi
	mov	esi,[esp + 20]			; esi -> name of environment variable
	mov	eax,[_myargs_psp]		; eax -> address of PSP
_myargs_get_variable_loop:
	cmp	[eax],dword 0			; end of the environment variable list?
	je	_myargs_get_variable_error	; yes, error
	PosText	[eax],esi,ecx			; no, does string contain variable we're looking for?
	jnc	_myargs_get_variable_check	; yes, check it's at the start
_myargs_get_variable_next:
	add	eax,4				; no, onto the next variable pointer
	jmp	_myargs_get_variable_loop	; and keep looking
_myargs_get_variable_check:
	cmp	ecx,0				; is string at the beginning?
	jnz	_myargs_get_variable_next	; no, not the variable we want
	PosChar	[eax],'=',ecx			; yes, find the first equals sign
	mov	eax,[eax]			; and return the string
	add	eax,ecx
	inc	eax
	mov	[_mystr_spare],eax
	pop	edi				; yes, restore registers
	pop	esi
	pop	ecx
	pop	eax
	clc					; flag success
	ret
_myargs_get_variable_error:
	mov	[_mystr_spare],dword 0
	pop	edi				; restore registers
	pop	esi
	pop	ecx
	pop	eax
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Dynamically allocated string routines
;
;------------------------------------------------------------------------------------
;
; Uppercases a string
;
%ifdef _MYSTR_DUPPERCASE_MACRO
_mystr_duppercase_2:
	push	esi				; save registers
	push	edi
	push	ecx
	mov	esi,[esp + 16]			; the string to uppercase
	Length	esi,ecx				; find it's length
	Malloc	ecx,edi				; and allocate space for it
	jc	_mystr_duppercase_error		; couldn't allocate it!
	mov	[_mystr_spare],edi		; save allocated address
	push	edi				; save string location
	push	ecx				; save length
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore original length of string
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	pop	esi				; restore string location
_mystr_duppercase_loop:
	cmp	[esi],byte 0			; is the byte zero?
	je	_mystr_duppercase_end		; yes, end
	cmp	[esi],byte 'a'			; no, is the character below 'a'?
	jb	_mystr_duppercase_no		; yes, don't uppercase it
	cmp	[esi],byte 'z'			; no, is it above 'z'?
	ja	_mystr_duppercase_no		; yes, don't uppercase it
	sub	[esi],byte 32			; no, uppercase it
_mystr_duppercase_no:
	inc	esi				; onto next character
	jmp	_mystr_duppercase_loop		; keep going until all characters done
_mystr_duppercase_end:
	pop	ecx
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_duppercase_error:
	pop	ecx
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Lowercases a string
;
%ifdef _MYSTR_DLOWERCASE_MACRO
_mystr_dlowercase_2:
	push	esi				; save registers
	push	edi
	push	ecx
	mov	esi,[esp + 16]			; the string to uppercase
	Length	esi,ecx				; find it's length
	Malloc	ecx,edi				; and allocate space for it
	jc	_mystr_dlowercase_error		; couldn't allocate it!
	mov	[_mystr_spare],edi		; save allocated address
	push	edi				; save string location
	push	ecx				; save length
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore original length of string
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	pop	esi				; restore string location
_mystr_dlowercase_loop:
	cmp	[esi],byte 0			; is the byte zero?
	je	_mystr_dlowercase_end		; yes, end
	cmp	[esi],byte 'A'			; no, is the character below 'A'?
	jb	_mystr_dlowercase_no		; yes, don't lowercase it
	cmp	[esi],byte 'Z'			; no, is it above 'Z'?
	ja	_mystr_dlowercase_no		; yes, don't lowercase it
	add	[esi],byte 32			; no, lowercase it
_mystr_dlowercase_no:
	inc	esi				; onto next character
	jmp	_mystr_dlowercase_loop		; keep going until all characters done
_mystr_dlowercase_end:
	pop	ecx
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_dlowercase_error:
	pop	ecx
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Three trimming routines - trim, trimleft, trimright
;
%ifdef _MYSTR_DTRIM_MACRO
_mystr_dtrim_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ecx
	mov	esi,[esp + 20]
	mov	eax,-1				; setup eax for loop
_mystr_dtrim_loop:
	inc	eax				; next character on
	cmp	[esi + eax],byte 0		; end of the string?
	je	_mystr_dtrim_end		; yes, end
	cmp	[esi + eax],byte ' '		; no, is it <= ' '
	jbe	_mystr_dtrim_loop		; yes, keep going
	mov	ecx,eax				; no, save where we're up to
_mystr_dtrim_loop2:
	inc	ecx				; onto next character
	cmp	[esi + ecx],byte ' '		; is it above ' '?
	ja	_mystr_dtrim_loop2		; yes, keep looping
	sub	ecx,eax				; no, get number of characters to copy
	Malloc	ecx,edi				; allocate space for the string
	jc	_mystr_dtrim_error		; couldn't allocate space!
	CopyPart esi,edi,eax,ecx		; copy part of string
	mov	[_mystr_spare],edi
	pop	ecx				; restore registers
	pop	eax
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_dtrim_end:
	mov	esi,[esp + 20]			; get string address
	dCopy	esi,edi				; dynamically duplicate it
	jc	_mystr_dtrim_error		; couldn't copy it!
	mov	[_mystr_spare],edi
	pop	ecx
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
_mystr_dtrim_error:
	mov	[_mystr_spare],dword 0
	pop	ecx				; restore registers
	pop	eax
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
%ifdef _MYSTR_DTRIMLEFT_MACRO
_mystr_dtrimleft_2:
	push	esi
	push	edi
	push	eax
	mov	esi,[esp + 16]
	mov	eax,-1				; setup eax for loop
_mystr_dtrimleft_loop:
	inc	eax				; next character on
	cmp	[esi + eax],byte 0		; end of the string?
	je	_mystr_trimleft_end		; yes, end
	cmp	[esi + eax],byte ' '		; no, is it <= ' '
	jbe	_mystr_trimleft_loop		; yes, keep going
	Length	esi,ecx
	Malloc	ecx,edi
	jc	_mystr_dtrimleft_error
	CopyPart esi,edi,eax,ecx		; no, copy part of string
	mov	[mystr_spare],edi
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
_mystr_dtrimleft_end:
	mov	esi,[esp + 16]
	dCopy	esi,edi
	jc	_mystr_dtrimleft_error
	mov	[_mystr_spare],edi
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
_mystr_dtrimleft_error:
	mov	[_mystr_spare],dword 0
	pop	eax
	pop	edi
	pop	esi
	stc
	ret
%endif
;
%ifdef _MYSTR_DTRIMRIGHT_MACRO
_mystr_dtrimright_2:
	push	esi
	push	edi
	push	ecx
	mov	esi,[esp + 16]
	Length	esi,ecx				; get string length
_mystr_dtrimright_loop:
	dec	ecx				; next character back
	jz	_mystr_dtrimright_end		; end if at start of string
	cmp	[esi + ecx],byte ' '		; is it <= ' '
	jbe	_mystr_dtrimright_loop		; yes, keep going
	inc	ecx
	Malloc	ecx,edi
	jc	_mystr_dtrimright_error
	CopyPart esi,edi,0,ecx			; copy part of string
	mov	[_mystr_spare],edi
	pop	ecx
	pop	edi
	pop	esi
	clc
	ret
_mystr_dtrimright_end:
	mov	esi,[esp + 16]
	dCopy	esi,edi
	jc	_mystr_dtrimright_error
	mov	[_mystr_spare],edi
	pop	ecx
	pop	edi
	pop	esi
	clc
	ret
_mystr_dtrimright_error:
	pop	ecx
	pop	edi
	pop	esi
	clc
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Reverses a string
;
%ifdef _MYSTR_DREVERSE_MACRO
_mystr_dreverse_2:
	push	eax
	push	ebx
	push	ecx
	push	esi
	push	edi
	mov	esi,[esp + 24]
	dCopy	esi,esi
	jc	_mystr_dreverse_error
	mov	[_mystr_spare],esi
	mov	edi,esi				; edi = esi = address of string
	Length	esi,ecx				; get string length
	dec	ecx				; leave off terminator
	add	edi,ecx				; edi points to last character in string
	dec	edi				; one less because strings start from 0
	shr	ecx,1				; only do half the string length
_mystr_dreverse_loop:
	mov	al,[edi]			; get right-hand character
	mov	bl,[esi]			; get left-hand character
	mov	[edi],bl			; swap them
	mov	[esi],al
	inc	esi				; next character on
	dec	edi				; next character back
	dec	ecx				; loop until all characters done
	jnz	_mystr_dreverse_loop
	pop	edi
	pop	esi
	pop	ecx
	pop	ebx
	pop	eax
	clc
	ret
_mystr_dreverse_error:
	mov	[_mystr_spare],dword 0
	pop	edi
	pop	esi
	pop	ecx
	pop	ebx
	pop	eax
	stc
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Duplicates a string
;
%ifdef _MYSTR_DCOPY_MACRO
_mystr_dcopy_2:
	push	esi
	push	edi
	push	ecx
	mov	esi,[esp + 16]
	Length	esi,ecx				; find length of string
	Malloc	ecx,edi				; allocate space
	jc	_mystr_dcopy_error		; couldn't allocate space!
	mov	[_mystr_spare],edi		; save allocated address
	push	ecx				; save length
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore original length of string
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	pop	ecx
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_dcopy_error:
	pop	ecx
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Duplicates part of a string
;
%ifdef _MYSTR_DCOPYPART_MACRO
_mystr_dcopypart_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ebx
	push	ecx
	mov	esi,[esp + 32]
	mov	eax,[esp + 28]
	mov	ecx,[esp + 24]
	mov	ebx,ecx				; save number of characters to copy
	Length	esi,ecx				; get string length (sets direction flag)
	cmp	eax,ecx				; start index >= string length
	jae	_mystr_dcopypart_null		; yes, return a null string
	add	esi,eax				; no, get to start character in string
	dec	ecx				; leave off null terminator
	push	eax				; save index
	add	eax,ebx				; work out index of final character to copy
	cmp	eax,ecx				; final index > string length?
	jbe	_mystr_dcopypart_go		; no
	mov	ebx,ecx				; yes, make final index = string length
	sub	ebx,[esp]			; subtract start index to get number of chars to copy
_mystr_dcopypart_go:
	pop	eax				; restore index
	Malloc	ebx,edi				; allocate space for the string
	jc	_mystr_dcopypart_error		; couldn't allocate space!
	mov	[_mystr_spare],edi		; save address
	mov	ecx,ebx				; ecx is the count of characters to copy
	push	ecx				; save it
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore count
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	mov	[edi],byte 0			; append the null terminator
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
_mystr_dcopypart_null:
	Malloc	4,edi
	jc	_mystr_dcopypart_error
	mov	[_mystr_spare],edi
	mov	[edi],byte 0			; store a null string
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
_mystr_dcopypart_error:
	mov	[_mystr_spare],dword 0
	pop	ecx
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	clc
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Standard string routines
;
;------------------------------------------------------------------------------------
;
; Three trimming routines - trim, trimleft, trimright
;
%ifdef _MYSTR_TRIM_MACRO
_mystr_trim_2:
	push	esi
	push	eax
	push	ecx
	mov	esi,[esp + 16]
	mov	eax,-1				; setup eax for loop
_mystr_trim_loop:
	inc	eax				; next character on
	cmp	[esi + eax],byte 0		; end of the string?
	je	_mystr_trim_end			; yes, end
	cmp	[esi + eax],byte ' '		; no, is it <= ' '
	jbe	_mystr_trim_loop		; yes, keep going
	mov	ecx,eax				; no, save where we're up to
_mystr_trim_loop2:
	inc	ecx				; onto next character
	cmp	[esi + ecx],byte ' '		; is it above ' '?
	ja	_mystr_trim_loop2		; yes, keep looping
	sub	ecx,eax				; no, get number of characters to copy
	CopyPart esi,esi,eax,ecx		; copy part of string
_mystr_trim_end:
	pop	ecx
	pop	eax
	pop	esi
	ret
%endif
;
%ifdef _MYSTR_TRIMLEFT_MACRO
_mystr_trimleft_2:
	push	esi
	push	eax
	mov	esi,[esp + 12]
	mov	eax,-1				; setup eax for loop
_mystr_trimleft_loop:
	inc	eax				; next character on
	cmp	[esi + eax],byte 0		; end of the string?
	je	_mystr_trimleft_end		; yes, end
	cmp	[esi + eax],byte ' '		; no, is it <= ' '
	jbe	_mystr_trimleft_loop		; yes, keep going
	CopyPart esi,esi,eax,$0ffffffff		; no, copy part of string
_mystr_trimleft_end:
	pop	eax
	pop	esi
	ret
%endif
;
%ifdef _MYSTR_TRIMRIGHT_MACRO
_mystr_trimright_2:
	push	esi
	push	ecx
	mov	esi,[esp + 12]
	Length	esi,ecx				; get string length
_mystr_trimright_loop:
	dec	ecx				; next character back
	jz	_mystr_trimright_end		; end if at start of string
	cmp	[esi + ecx],byte ' '		; is it <= ' '
	jbe	_mystr_trimright_loop		; yes, keep going
	inc	ecx
	CopyPart esi,esi,0,ecx			; copy part of string
_mystr_trimright_end:
	pop	ecx
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Reverses a string
;
%ifdef _MYSTR_REVERSE_MACRO
_mystr_reverse_2:
	push	eax
	push	ebx
	push	ecx
	push	esi
	push	edi
	mov	esi,[esp + 24]
	mov	edi,esi				; edi = esi = address of string
	Length	esi,ecx				; get string length
	dec	ecx				; leave off terminator
	add	edi,ecx				; edi points to last character in string
	dec	edi				; one less because strings start from 0
	shr	ecx,1				; only do half the string length
_mystr_reverse_loop:
	mov	al,[edi]			; get right-hand character
	mov	bl,[esi]			; get left-hand character
	mov	[edi],bl			; swap them
	mov	[esi],al
	inc	esi				; next character on
	dec	edi				; next character back
	dec	ecx				; loop until all characters done
	jnz	_mystr_reverse_loop
	pop	edi
	pop	esi
	pop	ecx
	pop	ebx
	pop	eax
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Copies a string
;
%ifdef _MYSTR_COPY_MACRO
_mystr_copy_2:
	push	esi				; save registers
	push	edi
	push	ecx
	mov	esi,[esp + 20]
	mov	edi,[esp + 16]
	Length	esi,ecx				; get string length (sets direction flag)
	push	ecx				; save it
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore original length of string
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	pop	ecx				; restore registers
	pop	edi
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Copy part of a string
;
%ifdef _MYSTR_COPYPART_MACRO
_mystr_copypart_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ebx
	push	ecx
	mov	esi,[esp + 36]
	mov	edi,[esp + 32]
	mov	eax,[esp + 28]
	mov	ecx,[esp + 24]
	mov	ebx,ecx				; save number of characters to copy
	Length	esi,ecx				; get string length (sets direction flag)
	cmp	eax,ecx				; start index >= string length
	jae	_mystr_copypart_null		; yes, return a null string
	add	esi,eax				; no, get to start character in string
	dec	ecx				; leave off null terminator
	push	eax				; save index
	add	eax,ebx				; work out index of final character to copy
	cmp	eax,ecx				; final index > string length?
	jbe	_mystr_copypart_go		; no
	mov	ebx,ecx				; yes, make final index = string length
	sub	ebx,[esp]			; subtract start index to get number of chars to copy
_mystr_copypart_go:
	pop	eax				; restore index
	mov	ecx,ebx				; ecx is the count of characters to copy
	push	ecx				; save it
	shr	ecx,2				; /4 so we copy dwords
	rep	movsd				; copy dwords of string
	pop	ecx				; restore count
	and	ecx,3				; mask off all but lowest two bits
	rep	movsb				; copy the last few bytes (if any)
	mov	[edi],byte 0			; append the null terminator
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	ret
_mystr_copypart_null:
	mov	[edi],byte 0			; store a null string
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Compares two strings byte for byte.
;
%ifdef _MYSTR_COMPARESTRINGS_MACRO
_mystr_comparestrings_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ecx
	mov	esi,[esp + 24]
	mov	edi,[esp + 20]
	Length	esi,eax				; get length of string pointed to by esi
	Length	edi,ecx				; get length of string pointed to by edi
	cmp	eax,ecx				; are they the same length?
	jne	_mystr_comparestrings_error	; no
	cmp	ecx,4				; yes, is it a short string?
	jbe	_mystr_comparestrings_short	; yes
	shr	ecx,2				; no, divide by 4 so we compare dwords
	repe	cmpsd				; compare dwords of string
	jne	_mystr_comparestrings_error	; carry set if they're not the same
_mystr_comparestrings_short:
	mov	ecx,eax				; get original length of string
	and	ecx,3				; mask off all but lowest two bits
	repe	cmpsb				; compare the last few bytes (if any)
	jne	_mystr_comparestrings_error	; carry set if they're not the same
	pop	ecx				; restore registers
	pop	eax
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_comparestrings_error:
	pop	ecx				; restore registers
	pop	eax
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Compares two strings case-insensitively
;
%ifdef _MYSTR_COMPARETEXT_MACRO
_mystr_comparetext_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ebx
	push	ecx
	mov	esi,[esp + 28]
	mov	edi,[esp + 24]
	Length	esi,eax				; get lengths of the two strings
	Length	edi,ecx
	cmp	eax,ecx				; are they the same length?
	jne	_mystr_comparetext_error	; no, can't be the same
	dec	ecx				; ignore the null terminator
_mystr_comparetext_loop:
	mov	al,[esi]			; get characters to compare
	mov	bl,[edi]
	cmp	al,'a'				; is first character below 'a'?
	jb	_mystr_comparetext_loop_b	; yes, don't try and uppercase it
	cmp	al,'z'				; no, is it above 'z'?
	ja	_mystr_comparetext_loop_b	; yes, don't try and uppercase it
	sub	al,32				; no, uppercase it
_mystr_comparetext_loop_b:
	cmp	bl,'a'				; is second character below 'a'?
	jb	_mystr_comparetext_loop_c	; yes, don't try and uppercase it
	cmp	bl,'z'				; no, is it above 'z'?
	ja	_mystr_comparetext_loop_c	; yes, don't try and uppercase it
	sub	bl,32				; no, uppercase it
_mystr_comparetext_loop_c:
	cmp	al,bl				; are the characters the same?
	jne	_mystr_comparetext_error	; no, error
	inc	esi				; yes, keep checking
	inc	edi
	dec	ecx				; do all characters
	jnz	_mystr_comparetext_loop
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	clc					; flag success
	ret
_mystr_comparetext_error:
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Looks for a byte in a string
;
%ifdef _MYSTR_POSCHAR_MACRO
_mystr_poschar_2:
	push	esi
	mov	esi,[esp + 12]
	dec	esi				; correct for loop
_mystr_poschar_loop:
	inc	esi				; next character on
	cmp	[esi],byte 0			; end of string?
	je	_mystr_poschar_error		; yes, error
	cmp	[esi],al			; no, character we're looking for?
	jne	_mystr_poschar_loop		; no, keep looking
	mov	[_mystr_spare],esi		; yes, save address of character
	mov	esi,[esp + 12]
	sub	[_mystr_spare],esi		; and get to correct position
	pop	esi
	clc
	ret
_mystr_poschar_error:
	mov	[_mystr_spare],dword 0
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Looks for a byte (case-insensitively) in a string
;
%ifdef _MYSTR_POSLETTER_MACRO
_mystr_posletter_2:
	push	esi				; save registers
	push	eax
	push	ebx
	mov	esi,[esp + 20]
	cmp	al,'a'				; see if the character needs uppercasing
	jb	_mystr_posletter_go
	cmp	al,'z'
	ja	_mystr_posletter_go
	sub	al,32				; uppercase it
_mystr_posletter_go:
	dec	esi				; correct for loop
_mystr_posletter_loop:
	inc	esi				; next character on
	cmp	[esi],byte 0			; end of string?
	je	_mystr_posletter_error		; yes, error
	mov	bl,[esi]			; no, get character
	cmp	bl,'a'				; see if the character needs uppercasing
	jb	_mystr_posletter_check
	cmp	bl,'z'
	ja	_mystr_posletter_check
	sub	bl,32				; uppercase it
_mystr_posletter_check:
	cmp	al,bl				; found the character?
	jne	_mystr_posletter_loop		; no, keep looking
	mov	[_mystr_spare],esi		; yes save address of character
	mov	esi,[esp + 20]
	sub	[_mystr_spare],esi		; and get to correct position
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	clc					; flag success
	ret
_mystr_posletter_error:
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	mov	[_mystr_spare],dword 0
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Searches for a substring in a string.
;
%ifdef _MYSTR_POSSTR_MACRO
_mystr_posstr_2:
	push	esi				; save registers
	push	edi
	push	ebx
	push	ecx
	mov	esi,[esp + 24]
	mov	edi,[esp + 20]
	Length	esi,ecx				; get length of string to search
	Length	edi,ebx				; get length of the string to search for
	dec	ebx				; ignore the null terminator
	cmp	ebx,ecx				; is substring longer than the string?
	ja	_mystr_posstr_error		; yes, error
	sub	ecx,ebx				; no, get maximum number of characters to search
_mystr_posstr_loop:
	push	ecx				; save count of characters to do
	mov	ecx,ebx				; ecx holds the length of the string we're searching for
	push	esi				; save where we're up to
	push	edi
	repe	cmpsb				; have we found the string we're looking for?
	je	_mystr_posstr_end		; yes, finish
	pop	edi				; no, restore where we're up to
	pop	esi
	pop	ecx				; restore character count
	inc	esi				; next character on
	dec	ecx				; another character done
	jnz	_mystr_posstr_loop		; keep looping until all characters done
_mystr_posstr_error:
	pop	ecx				; restore registers
	pop	ebx
	pop	edi
	pop	esi
	xor	ecx,ecx
	stc					; flag error
	ret
_mystr_posstr_end:
	pop	edi				; restore registers from loop
	pop	esi
	pop	ecx
	mov	[_mystr_spare],esi		; save address where we were up to in the string
	mov	esi,[esp + 24]
	sub	[_mystr_spare],esi		; and correct into an index
	pop	ecx				; restore registers
	pop	ebx
	pop	edi
	pop	esi
	clc					; flag success
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Searches for a substring in a string (case-insensitively)
;
%ifdef _MYSTR_POSTEXT_MACRO
_mystr_postext_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ebx
	push	ecx
	push	edx
	mov	esi,[esp + 32]
	mov	edi,[esp + 28]
	Length	esi,ecx				; get length of string to search
	Length	edi,ebx				; get length of the string to search for
	dec	ebx				; ignore the null terminator
	cmp	ebx,ecx				; is substring longer than the string?
	ja	_mystr_postext_error		; yes, error
	sub	ecx,ebx				; no, get maximum number of characters to search
_mystr_postext_loop:
	push	ecx				; save character count
	mov	ecx,ebx				; ecx holds the length of the string we're searching for
	push	esi				; save where we're up to
	push	edi				; this next bit sees if we've found the string we're looking for (ignores case)
_mystr_postext_loop2:
	mov	al,[esi]			; get characters to compare
	mov	dl,[edi]
	cmp	al,'a'				; is first character below 'a'?
	jb	_mystr_postext_loop_b		; yes, don't try and uppercase it
	cmp	al,'z'				; no, is it above 'z'?
	ja	_mystr_postext_loop_b		; yes, don't try and uppercase it
	sub	al,32				; no, uppercase it
_mystr_postext_loop_b:
	cmp	dl,'a'				; is second character below 'a'?
	jb	_mystr_postext_loop_c		; yes, don't try and uppercase it
	cmp	dl,'z'				; no, is it above 'z'?
	ja	_mystr_postext_loop_c		; yes, don't try and uppercase it
	sub	dl,32				; no, uppercase it
_mystr_postext_loop_c:
	cmp	al,dl				; are the characters the same?
	je	_mystr_postext_loop_ok		; yes, continue the loop
_mystr_postext_loop_error:
	pop	edi				; no, restore where we're up to
	pop	esi
	pop	ecx				; restore character count
	inc	esi				; next character on
	dec	ecx				; another character done
	jnz	_mystr_postext_loop		; keep looping until all characters done
_mystr_postext_error:
	pop	edx				; restore registers
	pop	ecx
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	mov	[_mystr_spare],dword 0
	stc					; flag error
	ret
_mystr_postext_loop_ok:
	inc	esi				; these two characters are the same so keep checking
	inc	edi
	dec	ecx				; do all characters
	jnz	_mystr_postext_loop2
	pop	edi				; restore registers from loop
	pop	esi
	mov	[_mystr_spare],esi		; save where we're up to in the string
	pop	ecx				; restore character count
	mov	ecx,[esp + 32]
	sub	[_mystr_spare],ecx		; and make into an index
	pop	edx				; restore registers
	pop	ecx
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	clc					; flag success
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Displays a null-terminated string via int $21.
;
%ifdef _MYSTR_PRINT_MACRO
_mystr_print_2:
	push	esi				; save registers
	push	eax
	push	ebx
	push	ecx
	push	edx
	mov	esi,[esp + 24]
	Length	esi,ecx				; get the string's length
_mystr_print_loop:
	cmp	ecx,$03fff			; is it more than 16383 (16K - 1) bytes?
	jbe	_mystr_print_lastbit		; no, no need to worry
	mov	bl,[esi + $03fff]		; yes, save the end character
	mov	[esi + $03fff],byte '$'		; insert a terminator
	mov	edx,esi				; display this part of the string
	mov	ah,9
	int	$21
	mov	[esi + $03fff],bl		; restore the end character
	sub	ecx,$03fff			; done 16383 bytes
	add	esi,$03fff
	jmp	_mystr_print_loop		; keep going
_mystr_print_lastbit:
	cmp	ecx,0				; any more of the string to do?
	jz	_mystr_print_end		; no, end
	dec	ecx
	mov	[esi + ecx],byte '$'		; insert a terminator
	mov	edx,esi				; display last bit of string
	mov	ah,9
	int	$21
	mov	[esi + ecx],byte 0		; restore the null-terminator
_mystr_print_end:
	pop	edx				; restore registers
	pop	ecx
	pop	ebx
	pop	eax
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Works the same way as _mystr_print, but prints a CR/LF pair at the end.
;
%ifdef _MYSTR_PRINTLINE_MACRO
_mystr_printline_2:
	push	esi				; save registers
	push	eax
	push	ebx
	push	ecx
	push	edx
	mov	esi,[esp + 24]
	Length	esi,ecx				; get the string's length
_mystr_printline_loop:
	cmp	ecx,$03fff			; is it more than 16383 (16K - 1) bytes?
	jbe	_mystr_printline_lastbit	; no, no need to worry
	mov	bl,[esi + $03fff]		; yes, save the end character
	mov	[esi + $03fff],byte '$'		; insert a terminator
	mov	edx,esi				; display this part of the string
	mov	ah,9
	int	$21
	mov	[esi + $03fff],bl		; restore the end character
	sub	ecx,$03fff			; done 16383 bytes
	add	esi,$03fff
	jmp	_mystr_printline_loop		; keep going
_mystr_printline_lastbit:
	cmp	ecx,0				; any more of the string to do?
	jz	_mystr_printline_end		; no, end
	dec	ecx
	mov	[esi + ecx],byte '$'		; insert a terminator
	mov	edx,esi				; display last bit of string
	mov	ah,9
	int	$21
	mov	[esi + ecx],byte 0		; restore the null-terminator
_mystr_printline_end:
	push	dword $240a0d			; make up a temporary CR/LF string
	mov	edx,esp				; display this
	mov	ah,9
	int	$21
	pop	eax				; remove the temporary string
	pop	edx				; restore registers
	pop	ecx
	pop	ebx
	pop	eax
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Returns a string's length
;
%ifdef _MYSTR_LENGTH_MACRO
_mystr_length_2:
	push	eax				; save registers
	push	ecx
	push	edi
	mov	edi,[esp + 16]
	cld					; set direction flag
	xor	eax,eax				; zero byte is what we're looking for
	mov	ecx,$0ffffffff			; maximum count
	repne	scasb				; look for it
	mov	eax,$0ffffffff			; original count value
	sub	eax,ecx				; - result count value
	mov	[_mystr_spare],eax		; = length of string
	pop	edi				; restore registers
	pop	ecx
	pop	eax
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Uppercases a string
;
%ifdef _MYSTR_UPPERCASE_MACRO
_mystr_uppercase_2:
	push	esi
	mov	esi,[esp + 8]
_mystr_uppercase_loop:
	cmp	[esi],byte 0			; is the byte zero?
	je	_mystr_uppercase_end		; yes, end
	cmp	[esi],byte 'a'			; no, is the character below 'a'?
	jb	_mystr_uppercase_no		; yes, don't uppercase it
	cmp	[esi],byte 'z'			; no, is it above 'z'?
	ja	_mystr_uppercase_no		; yes, don't uppercase it
	sub	[esi],byte 32			; no, uppercase it
_mystr_uppercase_no:
	inc	esi				; onto next character
	jmp	_mystr_uppercase_loop		; keep going until all characters done
_mystr_uppercase_end:
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Lowercases a string
;
%ifdef _MYSTR_LOWERCASE_MACRO
_mystr_lowercase_2:
	push	esi
	mov	esi,[esp + 8]
_mystr_lowercase_loop:
	cmp	[esi],byte 0			; is the byte zero?
	je	_mystr_lowercase_end		; yes, end
	cmp	[esi],byte 'A'			; no, is the character below 'A'?
	jb	_mystr_lowercase_no		; yes, don't lowercase it
	cmp	[esi],byte 'Z'			; no, is it above 'Z'?
	ja	_mystr_lowercase_no		; yes, don't lowercase it
	add	[esi],byte 32			; no, lowercase it
_mystr_lowercase_no:
	inc	esi				; onto next character
	jmp	_mystr_lowercase_loop		; keep going until all characters done
_mystr_lowercase_end:
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Conversion routines
;
;------------------------------------------------------------------------------------
;
; Converts a signed decimal string to a 32-bit integer.
;
%ifdef _MYSTR_STRTOINT_MACRO
_mystr_strtoint_2:
	push	esi				; save registers
	push	eax
	push	ebx
	mov	esi,[esp + 16]
	xor	eax,eax				; result is zero to start with
	xor	ebx,ebx				; clear ebx so we can work with 32 bits
	cmp	[esi],byte '0'			; is first character '0' or more?
	jae	_mystr_strtoint_loop		; yes, all's well
	inc	esi				; no, bypass the character
_mystr_strtoint_loop:
	mov	bl,[esi]			; get character
	or	ebx,ebx				; end of string?
	jz	_mystr_strtoint_end		; yes
	inc	esi				; no, onto next character
	cmp	ebx,byte '0'			; character below '0'?
	jb	_mystr_strtoint_error		; yes, error
	cmp	ebx,byte '9'			; no, character above '9'?
	ja	_mystr_strtoint_error		; yes, error
	cmp	eax,429496728			; no, is number too large?
	ja	_mystr_strtoint_error		; yes, error
	sub	ebx,byte '0'			; no, make value in ebx actual number
	lea	eax,[eax + eax*4]		; current result = current result * 10
	shl	eax,1				; (ie shift left by one decimal place)
	add	eax,ebx				; add next digit
	jmp	_mystr_strtoint_loop		; keep going
_mystr_strtoint_end:
	mov	esi,[esp + 16]
	cmp	[esi],byte '-'			; was the string negative?
	jne	_mystr_strtoint_end2		; no
	neg	eax				; yes, negate it
_mystr_strtoint_end2:
	mov	[_mystr_spare],eax		; save result
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	clc					; flag success
	ret
_mystr_strtoint_error:
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	mov	[_mystr_spare],dword 0
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts an unsigned decimal string to a 32-bit integer.
;
%ifdef _MYSTR_STRTOINTU_MACRO
_mystr_strtointu_2:
	push	esi				; save registers
	push	eax
	push	ebx
	mov	esi,[esp + 16]
	xor	eax,eax				; result is zero to start with
	xor	ebx,ebx				; clear ebx so we can work with 32 bits
_mystr_strtointu_loop:
	mov	bl,[esi]			; get character
	or	ebx,ebx				; end of string?
	jz	_mystr_strtointu_end		; yes
	inc	esi				; no, onto next character
	cmp	ebx,byte '0'			; character below '0'?
	jb	_mystr_strtointu_error		; yes, error
	cmp	ebx,byte '9'			; no, character above '9'?
	ja	_mystr_strtointu_error		; yes, error
	cmp	eax,429496728			; no, is number too large?
	ja	_mystr_strtointu_error		; yes, error
	sub	ebx,byte '0'			; no, make value in ebx actual number
	lea	eax,[eax + eax*4]		; current result = current result * 10
	shl	eax,1				; (ie shift left by one decimal place)
	add	eax,ebx				; add next digit
	jmp	_mystr_strtointu_loop		; keep going
_mystr_strtointu_end:
	mov	[_mystr_spare],eax		; save result
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	clc					; flag success
	ret
_mystr_strtointu_error:
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	mov	[_mystr_spare],dword 0
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts a hexadecimal string to a 32-bit integer.
;
%ifdef _MYSTR_HEXTOINT_MACRO
_mystr_hextoint_2:
	push	esi				; save registers
	push	eax
	push	ebx
	mov	esi,[esp + 16]
	xor	eax,eax				; result is zero to start with
	xor	ebx,ebx				; clear ebx so we can work with 32 bits
_mystr_hextoint_loop:
	mov	bl,[esi]			; get character
	or	ebx,ebx				; end of string?
	jz	_mystr_hextoint_end		; yes
	inc	esi				; no, onto next character
	cmp	ebx,byte '0'			; character below '0'?
	jb	_mystr_hextoint_error		; yes, error
	cmp	ebx,byte '9'			; no, character above '9'?
	ja	_mystr_hextoint_notnum		; yes, possibly a hex digit
	cmp	eax,268435455			; number too large?
	ja	_mystr_hextoint_error		; yes, error
	sub	ebx,byte '0'			; no, make character into number
	shl	eax,4				; make space for the next digit
	add	eax,ebx				; add it in
	jmp	_mystr_hextoint_loop		; keep going
_mystr_hextoint_notnum:
	cmp	ebx,byte 'A'			; character below 'A'?
	jb	_mystr_hextoint_error		; yes, error
	cmp	ebx,byte 'F'			; above 'F'?
	ja	_mystr_hextoint_maybehex	; yes, possibly a hex digit
	cmp	eax,268435455			; number too large?
	ja	_mystr_hextoint_error		; yes, error
	sub	ebx,byte ('A' - 10)		; no, make character into number
	shl	eax,4				; make space for the next digit
	add	eax,ebx				; add it in
	jmp	_mystr_hextoint_loop		; keep going
_mystr_hextoint_maybehex:
	cmp	ebx,byte 'a'			; character below 'a'?
	jb	_mystr_hextoint_error		; yes, error
	cmp	ebx,byte 'f'			; above 'F'?
	ja	_mystr_hextoint_error		; yes, error
	cmp	eax,268435455			; number too large?
	ja	_mystr_hextoint_error		; yes
	sub	ebx,byte ('a' - 10)		; no, make character into number
	shl	eax,4				; make space for the next digit
	add	eax,ebx				; add it in
	jmp	_mystr_hextoint_loop		; keep going
_mystr_hextoint_end:
	mov	[_mystr_spare],eax		; save result
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	clc					; flag success
	ret
_mystr_hextoint_error:
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	mov	[_mystr_spare],dword 0
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts a binary string to a 32-bit integer.
;
%ifdef _MYSTR_BINTOINT_MACRO
_mystr_bintoint_2:
	push	esi				; save registers
	push	eax
	push	ebx
	mov	esi,[esp + 16]
	xor	eax,eax				; result is zero to start with
	xor	ebx,ebx				; clear ebx so we can work with 32 bits
_mystr_bintoint_loop:
	mov	bl,[esi]			; get character
	or	ebx,ebx				; end of string?
	jz	_mystr_bintoint_end		; yes
	inc	esi				; no, onto next character
	sub	ebx,byte '0'			; make character a number
	cmp	ebx,1				; is it above 1?
	ja	_mystr_bintoint_error		; yes, error
	shl	eax,1				; no, make space for the bit
	or	eax,ebx				; and put the bit in
	jmp	_mystr_bintoint_loop		; and keep going
_mystr_bintoint_end:
	mov	[_mystr_spare],eax		; save result
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	clc					; flag success
	ret
_mystr_bintoint_error:
	pop	ebx				; restore registers
	pop	eax
	pop	esi
	stc					; flag error
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts a signed 32-bit integer to a decimal string.
;
%ifdef _MYSTR_INTTOSTR_MACRO
_mystr_inttostr_2:
	push	eax
	push	ecx
	push	edx
	push	esi
	push	edi
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	or	eax,eax				; is the number zero?
	jz	_mystr_intotostr_zero		; yes
	mov	ecx,eax
	and	ecx,$080000000			; get top bit
	jz	_mystr_inttostr_ok		; zero?
	neg	eax				; no, must be a negative
_mystr_inttostr_ok:
	mov	ecx,10				; want to divide by 10
	mov	[edi + 11],byte 0		; put in null-terminator
	add	edi,10				; start at right side of string
_mystr_inttostr_loop:
	xor	edx,edx				; top half of divisor is zero
	div	ecx				; get last digit
	add	edx,byte '0'			; make digit into a character
	mov	[edi],dl			; add to string
	dec	edi				; onto next character
	or	eax,eax				; done all the digits?
	jnz	_mystr_inttostr_loop		; no, keep going
	mov	eax,[esp + 28]
	and	eax,$080000000			; get top bit of original value
	jz	_mystr_inttostr_end		; zero?
	mov	[edi],byte '-'			; no, put in minus sign
	dec	edi				; and correct address
_mystr_inttostr_end:
	inc	edi				; point edi to the start of the string
	mov	ecx,[esp + 24]
	sub	ecx,edi
	add	ecx,12
	mov	eax,edi
	mov	edi,[esp + 24]
	mov	esi,eax
	rep	movsb
	pop	edi				; restore registers
	pop	esi
	pop	edx
	pop	ecx
	pop	eax
	ret
_mystr_intotostr_zero:
	mov	[edi],dword $030		; put in a '0' null-terminated string
	pop	edi				; restore registers
	pop	esi
	pop	edx
	pop	ecx
	pop	eax
	ret
%endif
%ifdef _MYSTR_INTTOSTR_FULL_MACRO
_mystr_inttostr_full_2:
	push	eax				; save registers
	push	ebx
	push	ecx
	push	edx
	push	edi
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	mov	ecx,eax
	and	ecx,$080000000			; get top bit
	jz	_mystr_inttostr_full_ok		; zero?
	neg	eax				; no, must be a negative
_mystr_inttostr_full_ok:
	mov	ecx,10				; want to divide by 10
	mov	[edi + 11],byte 0		; put in null-terminator
	add	edi,10				; start at right side of string
	mov	ebx,10				; 10 digits to do
_mystr_inttostr_full_loop:
	xor	edx,edx				; top half of divisor is zero
	div	ecx				; get last digit
	add	edx,byte '0'			; make digit into a character
	mov	[edi],dl			; add to string
	dec	edi				; onto next character
	dec	ebx				; do all digits
	jnz	_mystr_inttostr_full_loop
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	mov	[edi],byte '0'			; assume it's zero
	and	eax,$080000000			; get top bit of original value
	jz	_mystr_inttostr_full_end	; zero?
	mov	[edi],byte '-'			; no, put in minus sign
_mystr_inttostr_full_end:
	pop	edi
	pop	edx
	pop	ecx
	pop	ebx
	pop	eax
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts an unsigned 32-bit integer to a decimal string.
;
%ifdef _MYSTR_INTTOSTRU_MACRO
_mystr_inttostru_2:
	push	eax
	push	ecx
	push	edx
	push	esi
	push	edi
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	or	eax,eax				; is the number zero?
	jz	_mystr_intotostru_zero		; yes
	mov	ecx,10				; want to divide by 10
	mov	[edi + 10],byte 0		; put in null-terminator
	add	edi,9				; start at right side of string
_mystr_inttostru_loop:
	xor	edx,edx				; top half of divisor is zero
	div	ecx				; get last digit
	add	edx,byte '0'			; make digit into a character
	mov	[edi],dl			; add to string
	dec	edi				; onto next character
	or	eax,eax				; done all the digits?
	jnz	_mystr_inttostru_loop		; no, keep going
	inc	edi				; yes, point edi to the start of the string
	mov	ecx,[esp + 24]
	sub	ecx,edi
	add	ecx,11
	mov	eax,edi
	mov	edi,[esp + 24]
	mov	esi,eax
	rep	movsb
	pop	edi				; restore registers
	pop	esi
	pop	edx
	pop	ecx
	pop	eax
	ret
_mystr_intotostru_zero:
	mov	[edi],dword $030		; put in a '0' null-terminated string
	pop	edi				; restore registers
	pop	esi
	pop	edx
	pop	ecx
	pop	eax
	ret
%endif
%ifdef _MYSTR_INTTOSTRU_FULL_MACRO
_mystr_inttostru_full_2:
	push	eax				; save registers
	push	ebx
	push	ecx
	push	edx
	push	edi
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	mov	ecx,10				; want to divide by 10
	mov	[edi + 10],byte 0		; put in null-terminator
	add	edi,9				; start at right side of string
	mov	ebx,10				; 10 digits to do
_mystr_inttostru_full_loop:
	xor	edx,edx				; top half of divisor is zero
	div	ecx				; get last digit
	add	edx,byte '0'			; make digit into a character
	mov	[edi],dl			; add to string
	dec	edi				; onto next character
	dec	ebx				; do all digits
	jnz	_mystr_inttostru_full_loop
	pop	edi
	pop	edx
	pop	ecx
	pop	ebx
	pop	eax
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts a 32-bit integer to a hex string
;
%ifdef _MYSTR_INTTOHEX_MACRO
_mystr_inttohex_2:
	push	esi
	push	edi
	push	eax
	push	ebx
	push	ecx
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	or	eax,eax				; is the number zero?
	jz	_mystr_inttohex_zero		; yes
	mov	esi,_mystr_hexdigits_str	; esi points to string of hex digits
	mov	ecx,8				; 8 digits to do
_mystr_inttohex_loop:
	rol	eax,4				; get next digit into right-most 4 bits
	mov	ebx,eax
	and	ebx,$0f				; mask off bits
	mov	bl,[esi + ebx]			; get character that represents this number
	mov	[edi],bl			; add it to the string
	inc	edi				; next character on
	dec	ecx				; do all digits
	jnz	_mystr_inttohex_loop
	mov	[edi],byte 0			; put in null-terminator
	mov	edi,[esp + 24]
	dec	edi				; correct for loop
_mystr_inttohex_loop2:
	inc	edi				; next character on
	cmp	[edi],byte '0'			; is it a '0'?
	je	_mystr_inttohex_loop2		; yes, keep looping
	mov	ecx,[esp + 24]
	sub	ecx,edi
	add	ecx,9
	mov	eax,edi
	mov	edi,[esp + 24]
	mov	esi,eax
	rep	movsb
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	ret
_mystr_inttohex_zero:
	mov	[edi],word $030			; special case where number's zero
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	ret
%endif
%ifdef _MYSTR_INTTOHEX_FULL_MACRO
_mystr_inttohex_full_2:
	push	esi				; save registers
	push	edi
	push	eax
	push	ebx
	push	ecx
	mov	eax,[esp + 28]
	mov	edi,[esp + 24]
	mov	esi,_mystr_hexdigits_str	; esi points to string of hex digits
	mov	ecx,8				; 8 digits to do
_mystr_inttohex_full_loop:
	rol	eax,4				; get next digit into right-most 4 bits
	mov	ebx,eax
	and	ebx,$0f				; mask off bits
	mov	bl,[esi + ebx]			; get character that represents this number
	mov	[edi],bl			; add it to the string
	inc	edi				; next character on
	dec	ecx				; do all digits
	jnz	_mystr_inttohex_full_loop
	mov	[edi],byte 0			; put in null-terminator
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	pop	edi
	pop	esi
	ret
%endif
;
;------------------------------------------------------------------------------------
;
; Converts a 32-bit integer to a binary string
;
%ifdef _MYSTR_INTTOBIN_MACRO
_mystr_inttobin_2:
	push	eax				; save registers
	push	ecx
	push	esi
	push	edi
	mov	edi,[esp + 20]
	or	eax,eax				; is the number zero?
	jz	_mystr_inttobin_zero		; yes
	mov	ecx,32				; 32 digits to do
_mystr_inttobin_loop:
	mov	[edi],byte '0'			; assume the digit's a 0
	shl	eax,1				; get the bit
	jnc	_mystr_inttobin_notset		; don't worry if it's not set
	mov	[edi],byte '1'
_mystr_inttobin_notset:
	inc	edi				; onto next character
	dec	ecx				; do all digits
	jnz	_mystr_inttobin_loop
	mov	[edi],byte 0			; put in a null-terminator
	mov	edi,[esp + 20]
	dec	edi				; correct for loop
_mystr_inttobin_loop2:
	inc	edi				; next character on
	cmp	[edi],byte '0'			; is it a '0'?
	je	_mystr_inttobin_loop2		; yes, keep looping
	mov	ecx,[esp + 20]
	sub	ecx,edi
	add	ecx,33
	mov	eax,edi
	mov	edi,[esp + 20]
	mov	esi,eax
	rep	movsb
	pop	edi				; restore registers
	pop	esi
	pop	ecx
	pop	eax
	ret
_mystr_inttobin_zero:
	mov	[edi],word $030			; special case where number's zero
	pop	edi				; restore registers
	pop	esi
	pop	ecx
	pop	eax
	ret
%endif
%ifdef _MYSTR_INTTOBIN_FULL_MACRO
_mystr_inttobin_full_2:
	push	eax				; save registers
	push	ecx
	push	edi
	mov	edi,[esp + 16]
	mov	ecx,32				; 32 digits to do
_mystr_inttobin_full_loop:
	mov	[edi],byte '0'			; assume the digit's a 0
	shl	eax,1				; get the bit
	jnc	_mystr_inttobin_full_notset	; don't worry if it's not set
	mov	[edi],byte '1'
_mystr_inttobin_full_notset:
	inc	edi				; onto next character
	dec	ecx				; do all digits
	jnz	_mystr_inttobin_full_loop
	mov	[edi],byte 0			; put in a null-terminator
	pop	edi				; restore registers
	pop	ecx
	pop	eax
	ret
%endif

%endif
