/*
    Software Authorization Access System (For UNIX)               { Ver 2.50 }
    
    Modified from Version 2.00 for use with DS1410E and DS1405E
    using the DS1481.
          
    Copyright (C) 1996 Dallas Semiconductor Corporation. 
    All rights Reserved. Printed in U.S.A.
    This software is protected by copyright laws of
    the United States and of foreign countries.
    This material may also be protected by patent laws of the United States 
    and of foreign countries.
    This software is furnished under a license agreement and/or a
    nondisclosure agreement and may only be used or copied in accordance
    with the terms of those agreements.
    The mere transfer of this software does not imply any licenses
    of trade secrets, proprietary technology, copyrights, patents,
    trademarks, maskwork rights, or any other form of intellectual
    property whatsoever. Dallas Semiconductor retains all ownership rights.
*/

#include <fcntl.h>  
#include <signal.h> 
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <time.h>
#include <stdio.h>

#define TRUE  1
#define FALSE 0

#define DO_RESET         0
#define DO_BIT           1 
#define DO_BYTE          2
#define DO_TOGGLEOD      3
#define DO_TOGGLEPT      4
#define DO_BRICKDETECT   5

typedef unsigned char uchar;
typedef unsigned long ulong;

typedef struct _DOWComm
{
   uchar Command;
   uchar Xfer;
}
DOWComm, *PDOWComm;

static uchar   SetupOk = FALSE; 
static uchar   FailNext;
static uchar   AccessGood;
static uchar   l0 = 0;
static uchar   RomDta[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
static int     FileDes;
static uchar   AuthFlag = FALSE;
static uchar   sem_init = FALSE;        
static key_t   DS1410_key;                                        
static int     sem_id;          
static struct  sembuf sops;
static DOWComm DS1481Comm;

#define DS1410_KEY 0x130506C4; /* Just some number for a key */
#define OUR_SEM_PERMS 0x1b6;   /* We want any process to be allowed sem access */

struct timeval tp;
struct timezone tzp;

usleep (microseconds)
unsigned long microseconds;
{
  int i;
  long total = 0;
  long initial;
  
  /* Get current time (just the usec) */
  gettimeofday(&tp,&tzp);
  initial = tp.tv_usec;
  
  /* While elapsed time is less than requested time */
  while (total < microseconds)
  {
    /* Give up the CPU for a while */
    for (i = 0;i < 10;i++)
      sleep(0);
  
    /* Get new time */    
    gettimeofday(&tp,&tzp);
    
    /* Total up the time spent */
    /* Protect against rollovers that give us large positive
       and negative numbers */
    if ( ((tp.tv_usec - initial) > (microseconds >> 2)) ||
         ((tp.tv_usec - initial) < 0) ) 
      total += 0;
    else
      total += tp.tv_usec - initial;
    initial = tp.tv_usec;
  }
/*  printf("total: %d\n",total);*/
}

/* 
   Process went wild with keyopen 
*/
static runawaysem()                       
{
   /* Set semval back to 1 */
   semctl(sem_id, 0, SETVAL, 1);                    
}

/* Enter overdrive mode */
uchar enterOD()
{
  int result;
  
  if (AuthFlag)
  {
    /* Reset the DOW bus */
    DOWReset();
    /* Send an overdrive skip ROM command before
       we change the mode of the 1481 */
    DOWByte(0x3C);
    
    /* Switch the 1481 into OD mode */
    /* Check to see if we accidentally turned overdrive
       off and turn it back on again if we did. */
    result = DOWToggleOD();
    if (result == 0)
      result = DOWToggleOD();

    /* Return 1 for OD on, 0 for OD off */    
    return result;
  }
  
  return FALSE;
}

/* Exit overdrive mode */
uchar exitOD()
{
  int result;
  
  if (AuthFlag)
  {
    /* Switch the 1481 out of OD mode */
    /* Check to see if we accidentally turned overdrive
       on and turn it back off again if we did. */
    result = DOWToggleOD();
    if (result != 0)
      result = DOWToggleOD();

    /* Reset the DOW bus to kick OD parts down. */
    DOWReset();
    /* All parts should now talk at normal speeds. */
    
    /* Return 1 for OD on, 0 for OD off */    
    return result;
  }
  
  return FALSE;
}

/* Enter EPP passthrough mode */
uchar enterEPPmode()
{
  if (DOWBrickDetect())
  {
    DOWTogglePT();
    /* No wait required (well, maybe a little one) */
    usleep(20000);
  }
  
  return TRUE;
}

/* Exit EPP passthrough mode */
uchar exitEPPmode()
{
  if (!DOWBrickDetect())
  {
    DOWTogglePT();
    /* Wait approx. 10ms (this may need to change) */
    usleep(20000);
  }
  
  return FALSE;
}

uchar keyopen()
{
   int semflag;
   key_t DS1410_key;

   if (!sem_init)
   {
      DS1410_key = DS1410_KEY;
      semflag = IPC_CREAT | IPC_EXCL | OUR_SEM_PERMS; 
      sem_id = semget(DS1410_key, 1, semflag); 

      /* Already created */
      if (sem_id == -1)                                  
      { 
         semflag = IPC_EXCL | OUR_SEM_PERMS; 
         sem_id = semget(DS1410_key, 1, semflag); 
      }
      else
         semctl(sem_id, 0, SETVAL, 1);  /* Initialize semval */

      sem_init = TRUE;
   }

   if (!AuthFlag)
   {
      sops.sem_num = 0;
      sops.sem_op  = -1;
      sops.sem_flg = SEM_UNDO;

      /* If we don't get key in 9 sec. grab it */
      signal(SIGALRM, runawaysem); 
      alarm(9);

      semop(sem_id, &sops, 1);

      alarm(0);
      signal(SIGALRM, SIG_IGN);

      FileDes = open("/dev/dow0", O_RDWR, 1);
      AuthFlag = TRUE;
   } 
  
   /* Exit EPP passthough mode */  
   exitEPPmode();
   /* Make sure we're out of overdrive */
   exitOD();
   return AuthFlag;
}

uchar keyclose()
{
   if (AuthFlag)
   {
      /* Enter EPP passthrough mode */
      enterEPPmode();
      close(FileDes);
      /* Set semval back to 1 */
      semctl(sem_id, 0, SETVAL, 1);                 

      sleep(0);
      sleep(0);

      AuthFlag = FALSE;
   } 
   return TRUE;
}

/*
   See if device driver has been installed
*/
uchar dowcheck()                 
{
   FileDes = open("/dev/dow0", O_RDWR, 1);
   close(FileDes);

   return (FileDes == -1) ? FALSE : TRUE;
}      

/*
    This function is pretty useless on this platform. 
*/
uchar setup(pn)             

uchar pn;

{
   /* Initialize global flags */
   SetupOk = FailNext = AccessGood = TRUE;       

   return SetupOk;                       
}

/*
   Find next DOW part
*/
uchar next()                                          
{
   uchar tr, i;

   if (SetupOk)
   {
      /* See if last search found last button */
      if (FailNext)                 
      {
         FailNext = FALSE; 
         /* Reset next function  */
         l0 = 0;                                    
      } 
      else while ((tr = RomSearch()) != 0)  /* Do that ROM search thang */
      {
         /* See if we should force failure */
         if (tr == 2) 
            FailNext = 1;                 
        
         /* !!! Removed filter !!! */
         return TRUE;
      }
   }

   return FALSE;                                                 
}

/*
   Find first DOW part on specified port
*/
uchar first()                     
{
   FailNext = FALSE; 
   l0       = 0;     /* Point Rom Search algorithm to the top */

   /* Go look for the first DOW part on the bus  */
   return next();             
}

/*
   "Strong" access
*/
uchar access()                                            
{
   uchar i, j;                                                 

   /* Assume failure  */
   AccessGood = FALSE;                                   

   /* Send reset pulse */
   if (DOWReset())                                      
   {
      /* ROM search command byte */
      DOWByte(0xF0);                             
      for (i = 0; i < 8; i++)                   
      {
         for (j = 0; j < 8; j++)               
         {
            if (((DOWBit(TRUE) << 1) | DOWBit(TRUE)) == 3)                
               return FALSE;    /* Part not there man  */

            /* Send write time slot */
            DOWBit((uchar) ((RomDta[i] >> j) & 1)); 
         }      
      }

      /* Success if we made it through all the bits */
      AccessGood = TRUE;      
   }

   return AccessGood;  
}

uchar databyte(br)           
uchar br;
{
   return (SetupOk && AccessGood) ? DOWByte(br) : br;
}

uchar gndtest()
{
   if (SetupOk)
   {
      /* Put parts on bus in a reset state */
      DOWReset();                        

      /* Return result of write 1 time slot */
      return DOWBit(TRUE);            
   }
   return FALSE;
}

uchar *romdata()
{
   /* Return pointer to ROM Data buffer  */
   return RomDta;                     
}

/*
     This function performs a ROM search and finds one part on the DOW bus
 per call.

      Return values : 0 => No parts on bus or bus error
                      1 => A part was found and more are out there
                      2 => A part was found and it was last one on the bus
*/
int RomSearch() 
{
   uchar i = 0, x = 0, ld = l0;
   uchar RomBit, Mask;

   /* Reset DOW bus */
   if (DOWReset())    
      DOWByte(0xF0);  /* Send search command */

   /* While not done and bus error */
   while ((i++ < 64) && (x < 3))            
   {
      /* Get bit mask */
      Mask = 1 << ((i - 1) % 8);                            

      RomBit = RomDta[(i - 1) >> 3] & Mask ? TRUE : FALSE; 

      x = DOWBit(TRUE) << 1; /* Send first read time slot */
      x |= DOWBit(TRUE);     /* Send second read time slot */

      /* Is there a disagreement in this bit position */
      if (!x)               
      {
         /* Stay on old path or pick a new one ?  */
         if (i >= ld)              
            RomBit = (i == ld); /* Send write 1 if at position of ld  */

         /* Save this value as temp last disagreement  */
         if (!RomBit)         
            l0 = i;
      } 
      else 
         RomBit = (x & 1) ^ 1;  /* Get lsb of x and flip it  */

      if (RomBit)              
         RomDta[(i - 1) >> 3] |= Mask;          /* Set bit in Rom Data byte */
      else
         RomDta[(i - 1) >> 3] &= (Mask ^ 0xFF); /*Clear bit in Rom Data byte*/

      /* Send write time slot */
      DOWBit(RomBit);                               
   }

   return (x == 3) ? 0 : 1 + (ld == l0);    
}

static int DOWReset()
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_RESET;                 
      DS1481Comm.Xfer = 0;

      /* Send packet to driver */
      write(FileDes, &DS1481Comm.Command, 1);      
      /* Get result of write */
      read(FileDes, &DS1481Comm.Xfer, 1);            

      /* Return result of reset operation */
      return DS1481Comm.Xfer;           
   }

   return FALSE;     
}

static int DOWBit(BitVal)  
uchar BitVal;
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_BIT;                   
      DS1481Comm.Xfer = BitVal;               

      write(FileDes, &DS1481Comm.Command, 2);
      read(FileDes, &DS1481Comm.Xfer, 1);   

      return DS1481Comm.Xfer; 
   } 

   return BitVal;
}

static int DOWByte(ByteVal)                
uchar ByteVal;
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_BYTE;       
      /* Byte to send to 1-wire bus */ 
      DS1481Comm.Xfer = ByteVal;              

      /* Send packet to driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result of write */
      read(FileDes, &DS1481Comm.Xfer, 1);            

      return DS1481Comm.Xfer;  
   } 

   return ByteVal;
}

static int DOWToggleOD()
{
  if (AuthFlag)
  {
    /* Assign command byte */
    DS1481Comm.Command = DO_TOGGLEOD;
    DS1481Comm.Xfer = 0;
    
    /* Send packet to driver */
    write(FileDes,&DS1481Comm.Command,1);
    /* Get result of write */
    read(FileDes,&DS1481Comm.Xfer,1);
    
    /* Return status of overdrive (1 means 1481 is in overdrive) */
    return DS1481Comm.Xfer;
  }
  
  return FALSE;
}

/* Toggle Pass Through */
static int DOWTogglePT()
{
  if (AuthFlag)
  {
    /* Assign command byte */
    DS1481Comm.Command = DO_TOGGLEPT;
    DS1481Comm.Xfer = 0;
    
    /* Send packet to driver */
    write(FileDes,&DS1481Comm.Command,1);
    /* Get result of write */
    read(FileDes,&DS1481Comm.Xfer,1);
    
    /* Return status of passthrough */
    return DS1481Comm.Xfer;
  }
  
  return FALSE;
}

int DOWBrickDetect()
{
  if (AuthFlag)
  {
    /* Assign command byte */
    DS1481Comm.Command = DO_BRICKDETECT;
    DS1481Comm.Xfer = 0;
    
    /* Send packet to driver */
    write(FileDes,&DS1481Comm.Command,1);
    /* Get result of write */
    read(FileDes,&DS1481Comm.Xfer,1);
    
    /* Return status of passthrough */
    return DS1481Comm.Xfer;
  }
  
  return FALSE;
}
