/*
    Software Authorization Access System (For UNIX)           { Ver 4.00 }

          
    Copyright (C) 1994, 1996 Dallas Semiconductor Corporation. 
    All rights Reserved. Printed in U.S.A.
    This software is protected by copyright laws of
    the United States and of foreign countries.
    This material may also be protected by patent laws of the United States 
    and of foreign countries.
    This software is furnished under a license agreement and/or a
    nondisclosure agreement and may only be used or copied in accordance
    with the terms of those agreements.
    The mere transfer of this software does not imply any licenses
    of trade secrets, proprietary technology, copyrights, patents,
    trademarks, maskwork rights, or any other form of intellectual
    property whatsoever. Dallas Semiconductor retains all ownership rights.
*/

#include <fcntl.h>  
#include <signal.h> 
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <stdio.h>

#define TRUE  1
#define FALSE 0

#define DO_RESET         0
#define DO_BIT           1 
#define DO_BYTE          2 
#define TOGGLE_OVERDRIVE 3 
#define TOGGLE_PASSTHRU  4 
#define CHECK_BRICK      5 

typedef unsigned char uchar;

typedef struct _DOWComm
{
   uchar Command;
   uchar Xfer;
}
DOWComm, *PDOWComm;

static uchar   SetupOk = FALSE; 
static uchar   FailNext;
static uchar   AccessGood;
static uchar   Passthru = TRUE;
static uchar   l0 = 0;
static uchar   RomDta[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
static int     FileDes;
static uchar   AuthFlag = FALSE;
static uchar   sem_init = FALSE;        
static key_t   DS1410_key;                                        
static int     sem_id;          
static struct  sembuf sops;
static DOWComm DS1481Comm;
static uchar   PortNum;

#define DS1410_KEY    0x130506C4; /* Just some number for a key */
#define OUR_SEM_PERMS 0x1B6;      /* Any process is allowed sem access */

/*
   DS1481 mode control functions
*/
uchar EnterPassthru();
uchar ExitPassthru();
void  TogglePassthru();
uchar OverdriveOn();
void  OverdriveOff();

static runawaysem()                       
{
   /* Set semval bact to 1 */
   semctl(sem_id, 0, SETVAL, 1);         
}

uchar keyopen()
{
   key_t DS1410_key;
   int   semflag;

   if (!sem_init)
   {
      DS1410_key = DS1410_KEY;
      semflag    = IPC_CREAT | IPC_EXCL | OUR_SEM_PERMS; 
      sem_id     = semget(DS1410_key, 1, semflag); 

      /* Already created ? */
      if (sem_id == -1) 
      { 
         semflag = IPC_EXCL | OUR_SEM_PERMS; 
         sem_id  = semget(DS1410_key, 1, semflag); 
      }
      else
         semctl(sem_id, 0, SETVAL, 1);  /* Initialize semval */

      sem_init = TRUE;
   }

   if (!AuthFlag)
   {
      sops.sem_num = 0;
      sops.sem_op  = -1;
      sops.sem_flg = SEM_UNDO;

      /* If we don't get key in 9 sec. grab it */
      signal(SIGALRM, runawaysem); 
      alarm(9);

      semop(sem_id, &sops, 1);

      alarm(0);
      signal(SIGALRM, SIG_IGN);

      FileDes = open("/dev/dow0", O_RDWR);

      AuthFlag = TRUE;

      ExitPassthru();
   } 

   return AuthFlag;
}

uchar keyclose()
{
   if (AuthFlag)
   {
      EnterPassthru();
      close(FileDes);

      /* Set semval back to 1 */
      semctl(sem_id, 0, SETVAL, 1);                 

      nap(32);
      AuthFlag = FALSE;
   } 
   return TRUE;
}

/* 
   See if device driver has been installed 
*/
uchar dowcheck()                 
{
   FileDes = open("/dev/dow0", O_RDWR);
   close(FileDes);

   return (FileDes == -1) ? FALSE : TRUE;
}      

uchar EnterPassthru()
{
   TogglePassthru();
   Passthru = TRUE;

   return TRUE;
}

uchar ExitPassthru()
{
   int i = 0;

   Passthru = TRUE;

   while (Passthru && (i++ < 3))
   {
      if (Passthru)
         TogglePassthru();

      Passthru = !DS1481Present();
   }

   OverdriveOff(); 

   return !Passthru; 
}

void TogglePassthru()
{
   Send4();
   /* Wait for mode to change */
   nap(16);
}

uchar OverdriveOn()
{
   if (!DOWReset())
      return FALSE;
 
   /* Put all devices (that support overdrive) in overdrive mode */
   DOWByte(0x3C);

   return ToggleOverdrive() ? TRUE : ToggleOverdrive();
}

void OverdriveOff()
{
   if (ToggleOverdrive())
      ToggleOverdrive();

   return;
}

uchar setup(pn)             

uchar pn;

{
   SetupOk = FALSE;

   if ((pn > 0) && (pn < 4))
   {
      PortNum = pn;

      /* Initialize global flags */
      SetupOk = FailNext = AccessGood = TRUE;       
   }
   else
      PortNum = 1;
 
   return SetupOk;                       
}

/*
   Find next part on DOW bus 
*/
uchar next()                                          
{
   uchar tr, i;

   if (SetupOk)
   {
      /* See if last search found last button */
      if (FailNext)                 
      {
         FailNext = FALSE; 
         /* Reset next function  */
         l0 = 0;                                    
      } 
      else if ((tr = RomSearch()) != 0) 
      {
         /* See if we should force failure */
         if (tr == 2) 
            FailNext = TRUE;                 
        
         return TRUE;
      }
   }

   return FALSE;                                
}

/* 
   Find first DOW part on specified port  
*/
uchar first()                     
{
   /* Don't force a failure here  */
   FailNext = FALSE;              
   /* Point Rom Search algorithm to the top */
   l0 = 0;                         

   /* Go look for the first DOW part on the bus */
   return next();             
}

/*
   "Strong" access
*/
uchar access()               
{
   uchar i, j;              

   /* Assume failure  */
   AccessGood = FALSE;                                   

   /* Send reset pulse */
   if (DOWReset())                                      
   {
      /* ROM search command byte */
      DOWByte(0xF0);                             

      /* Byte loop */
      for (i = 0; i < 8; i++)          
      {
         /* Bit loop */
         for (j = 0; j < 8; j++)      
         {
            /* Send two read time slots */
            if (((DOWBit(TRUE) << 1) | DOWBit(TRUE)) == 3)                
               return FALSE;         

            /* Send write time slot */
            DOWBit((uchar) ((RomDta[i] >> j) & 1)); 
         }      
      }

      /* Success if we made it throug all bits */
      AccessGood = TRUE;      
   }

   return AccessGood;  
}

/* 
   Xmit data to buttons on the port 
*/
uchar databyte(br)                      

uchar br;

{
   return (SetupOk && AccessGood) ? DOWByte(br) : br;
}

uchar gndtest()
{
   if (SetupOk)
   {
      /* Put parts on bus in known state */
      DOWReset();                        

      /* Return result of write 1 time slot */
      return DOWBit(TRUE);            
   }

   return FALSE;
}

uchar *romdata()
{
   /* Return pointer to ROM Data buffer  */
   return RomDta;                     
}

/*
     This function performs a ROM search and finds one part on the DOW bus
 per call.

      Return values : 0 => No parts on bus or bus error
                      1 => A part was found and more are out there
                      2 => A part was found and it was last one on the bus
*/
int RomSearch() 
{
   uchar i = 0, x = 0, ld = l0;
   uchar RomBit, Mask;

   /* Reset DOW bus */
   if (DOWReset())     
      DOWByte(0xF0);   /* Send search command */
   else
      return FALSE;

   /* While not done and bus error */
   while ((i++ < 64) && (x < 3))            
   {
      /* Get bit mask */
      Mask = 1 << ((i - 1) % 8);           

      RomBit = RomDta[(i - 1) >> 3] & Mask ? TRUE : FALSE; 

      /* Send first read time slot */
      x = DOWBit(TRUE) << 1; 
      /* Send second read time slot */
      x |= DOWBit(TRUE);      

      /* Is there a disagreement in this bit position */
      if (!x)               
      {
         /* Stay on old path or pick a new one ?  */
         if (i >= ld)              
            RomBit = (i == ld);   /* Send write 1 if at position of ld  */

         /* Save this value as temp last disagreement  */
         if (!RomBit)         
            l0 = i;
      } 
      else 
         RomBit = (x & 1) ^ 1;    /* Get lsb of x and flip it  */

      if (RomBit)              
         RomDta[(i - 1) >> 3] |= Mask;          /* Set bit in Rom Data byte */
      else
         RomDta[(i - 1) >> 3] &= (Mask ^ 0xFF); /*Clear bit in Rom Data byte*/

      /* Send write time slot */
      DOWBit(RomBit);                               
   }

   return (x == 3) ? 0 : 1 + (ld == l0);    
}

static int DOWReset()
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_RESET;                 
      /* Set port number */
      DS1481Comm.Xfer = PortNum;

      /* Send info to DOW driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result */
      read(FileDes, &DS1481Comm.Xfer, 1);         

      /* Return result of reset operation */
      return DS1481Comm.Xfer;           
   }

   /* Return FALSE if process did not have control of bus */
   return FALSE;     
}

/*
   Send bit to 1-wire bus
*/
static int DOWBit(BitVal)                         

uchar BitVal;

{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_BIT;               
      /* Byte to send to 1-wire bus */ 
      DS1481Comm.Xfer = BitVal;               
      /* Send packet to driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result of write */
      read(FileDes, &DS1481Comm.Xfer, 1);            

      return DS1481Comm.Xfer; 
   } 

   return BitVal;
}

/*
   Send byte to 1-wire bus 
*/   
static int DOWByte(ByteVal)                      

uchar ByteVal;

{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = DO_BYTE;             
      /* Byte to send to 1-wire bus */
      DS1481Comm.Xfer = ByteVal;              
      /* Send info to DOW driver */
      write(FileDes, &DS1481Comm.Command, 2);  
      /* Get result of write */
      read(FileDes, &DS1481Comm.Xfer, 1);     

      return DS1481Comm.Xfer;  
   } 

   return ByteVal;
}

static int ToggleOverdrive()
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = TOGGLE_OVERDRIVE;                 
      DS1481Comm.Xfer    = 0;

      /* Send info to DOW driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result */
      read(FileDes, &DS1481Comm.Xfer, 1);         

      /* Return result of reset operation */
      return DS1481Comm.Xfer;           
   }

   return FALSE;     
}

static int Send4()
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = TOGGLE_PASSTHRU;                 
      DS1481Comm.Xfer    = 0;

      /* Send info to DOW driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result */
      read(FileDes, &DS1481Comm.Xfer, 1);         

      /* Return result of reset operation */
      return DS1481Comm.Xfer;           
   }

   return FALSE;     
}

static int DS1481Present()
{
   if (AuthFlag)
   {
      /* Assign command byte */
      DS1481Comm.Command = CHECK_BRICK;                 
      DS1481Comm.Xfer    = 0;

      /* Send info to DOW driver */
      write(FileDes, &DS1481Comm.Command, 2);      
      /* Get result */
      read(FileDes, &DS1481Comm.Xfer, 1);         

      /* Return result of reset operation */
      return DS1481Comm.Xfer;           
   }

   return FALSE;     
}
