//------------ Dallas Semiconductor Corporation Copyright 1994-1996---------------
//  Copyright (C) 1994, 1995, 1996 Dallas Semiconductor Corporation. 
//  All rights Reserved. Printed in U.S.A.
//  This software is protected by copyright laws of
//  the United States and of foreign countries.
//  This material may also be protected by patent laws of the United States 
//  and of foreign countries.
//  This software is furnished under a license agreement and/or a
//  nondisclosure agreement and may only be used or copied in accordance
//  with the terms of those agreements.
//  The mere transfer of this software does not imply any licenses
//  of trade secrets, proprietary technology, copyrights, patents,
//  trademarks, maskwork rights, or any other form of intellectual
//  property whatsoever. Dallas Semiconductor retains all ownership rights.
//
//  SatLink - A Microsoft Windows 3.1 DLL that calls on the software
//             authorization DLL SACWD400.DLL to read and set a DS1427/DS1994
//             secure button.
//
//  Version 4.00                                                      
//  Version 4.04 Modified to support DS1994/DS1427
//

// Includes
#include <windows.h>
#include <time.h>
#include <stdio.h>
#include "SACWD400.H"

// Typedefs
typedef unsigned char uchar;
typedef unsigned int  uint;
typedef unsigned long ulong;

typedef struct          // type to hold the time/date
{
     uint  frsecond;
     uint  second;
     uint  minute;
     uint  hour;
     uint  day;
     uint  month;
     uint  year;
     uchar raw[6];
} timedate;

// external function declarations from SACWD200.DLL
extern uint  far pascal setup(uint,sa_struct far *);
extern uint  far pascal next(sa_struct far *);
extern uint  far pascal first(sa_struct far *);
extern uint  far pascal access(sa_struct far *);
extern uint  far pascal databyte(uint,sa_struct far *);
extern uint  far pascal gndtest(sa_struct far *);
extern uchar far *far pascal romdata(sa_struct far *);          
extern uchar far pascal keyopen(void);
extern void  far pascal keyclose(void);

// local function declarations
static uint crc16(int);
static int Copy_Scratchpad(int, int);
static int Write_Scratchpad(uchar far *, int, int);
static void Str2Time(timedate far *, uchar far *);
static ulong uchar_to_bin(int, uchar far *);

// local function declarations to be exported
int _export far pascal SmartPortSearch(void);
int _export far pascal ReadID(uchar far *);
int _export far pascal ReadData(uchar far *);
int _export far pascal WriteData(uchar far *, int);
int _export far pascal TimerSet(int);
int _export far pascal CheckTimer(void);
int _export far pascal ReadTimer(uchar far *, uchar far *);
int _export far pascal WriteProtectT(void);
int _export far pascal NextDS1427(void);

// global variables
static sa_struct gb;                 // state structure for each LPT
static int LPT = 0;                  // current LPT number
static int GotOne = 0;               // flag to indicate a DS1427/DS1994 is found
static int GotPort = 0;              // flag to indicate a DS1410E is found
static uint CRC16;                   // CRC storage for writing data


//---------------------------------------------------------------------------
// Search the the ports to find the port with a DS1410E.
// Returns the port number 1-3 or 0 if no port found.
//
int far pascal SmartPortSearch(void)
{
   int portnum=1,rt=0;
   
   // check each port until the DS1410E is found or out of ports
   do 
   {   
      // attempt to setup the port
      if (setup(portnum,&gb))
      {
         if (keyopen())   
            if(gndtest(&gb))
               if (first(&gb))
                  rt = portnum;
         keyclose();
      } 
   }
   while ((++portnum <= 3) && (rt == 0));
   
   // if found a port then keep it as the LPT
   LPT = rt;
   
   return rt;
}                                             


//--------------------------------------------------------------------------
// Read the DS1427/DS1994 button ID and return it in a provided 8 byte buffer.
//
// Returns:  1 the DS1427's/DS1994's Rom code is in the buffer
//           0 there is not current DS1427/DS1994
//
int far pascal ReadID(uchar far *buf)
{
   int i;
   uchar far *ROM;
   
   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
      
   // get a pointer to the rom buffer        
   ROM = romdata(&gb);           
   
   // loop to copy the current DS1427/DS1994 ID into the buffer     
   for (i = 0; i < 8; i++) 
      buf[i] = ROM[i];   
   
   // return success   
   return 1;
}


//--------------------------------------------------------------------------
// Read a default data structure string from the DS1427/DS1994 nv ram space and
// return it in the provided buffer. The buffer 'buf' must be at least
// 512 bytes long.
//
// Returns:  >=1 success, number of bytes in the buffer
//           0   failed to read a valid data packet on the DS1427/DS1994
//
int far pascal ReadData(uchar far *buf)
{
   uint i,len,cnt=0,rt=0;

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // open the port            
   if (keyopen())  
   {                         
      // select the device
      if (access(&gb))       
      {
         databyte(0xF0,&gb);     // read memory command
         databyte(0,&gb);        // write the target address
         databyte(0,&gb);
         CRC16 = 0;                      // reset crc
         
         len = databyte(0xFF,&gb);    // read the first length byte
         crc16(len);                     // compute CRC16 on first byte
         
         // if = 255 then read second length byte
         if (len == 255)
         {
             len = len + databyte(0xFF,&gb);
             crc16(len-255);            // compute crc16 on second byte
         }
         
         if (len <= 508)                // abort read if length to large
         {
            for (i = 0; i < len+2; ++i)     // loop to read record
            {
                buf[cnt] = databyte(0xFF,&gb);
                crc16(buf[cnt++]);
            }
            
            buf[cnt-2] = NULL;              // zero end the string over the CRC
            
            if (CRC16 == 0xB001)            // check to see if CRC is correct
              rt = cnt-2;                   // return number of byte in record
         }  
      }
       // release the port
      keyclose();
   }                                   
  
   return rt;
}


//--------------------------------------------------------------------------
// Write a default data structure to the nv ram space of a DS1427/DS1994.  Use
// the data in the provided buffer 'buf' of length 'len'.
//
// Returns: 1 success
//          0 failure
//
int far pascal WriteData(uchar far *buf, int len)
{
   uchar wrbuf[32];
   int   i,pcnt=0,cnt=0,mx,flg=0,ret=0,page=0;

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // check to see if data too long to fit on device
   if (len > 508)
     return 0;

   // open the port            
   if (keyopen())  
   { 
      CRC16 = 0; // reset crc
      
      // set length bytes
      if (len < 0xFF)
      {
         wrbuf[pcnt++] = (uchar)(len);
         crc16(len);
      }
      else
      {
         wrbuf[pcnt++] = 0xFF;
         crc16(0xFF);
         wrbuf[pcnt++] = (uchar)(len - 0xFF);
         crc16(len - 0xFF);
      }
      
      // loop to write each page
      do
      {
         // fill rest of page to write
         mx = ((len-cnt) > 32) ? 32 - pcnt : (len-cnt);
               
         for (i = 0; i < mx; i++)
         {
           crc16(buf[cnt]);
           wrbuf[pcnt++] = buf[cnt++];
         }  
                         
         // check to see if time for CRC
         if (cnt >= len)
         {
            if ((pcnt < 32) && (flg == 0))
            {
               flg = 1;  // flag indicates added 1st crc byte
               wrbuf[pcnt++] = (uchar)(~(CRC16 & 0xFF));
            }
               
            if ((pcnt < 32) && (flg == 1))
            {
               flg = 2;  // flag indicates added 2nd crc byte
               wrbuf[pcnt++] = (uchar)(~((CRC16 & 0xFF00) >> 8));
            }
         }
               
         // write the page
         ret = Write_Scratchpad(wrbuf,page,pcnt);
         if (ret != 1)  // check on success
           break;
               
         ret = Copy_Scratchpad(page,pcnt);
         if (ret != 1)   // check on success
           break;
               
         page++;     // increment page number
         pcnt = 0;   // reset internal byte count for write page
      
      } 
      while (flg != 2);  // do until flag indicates 2nd crc writen
      
      // release the port
      keyclose();
   }
     
   // if last copy scratch pad was good then success
   return (ret == 1) ? 1 : 0;
}


//--------------------------------------------------------------------------
// Set the real time clock of the DS1427/DS1994 to the current PC time and set
// the real time clock alarm register to the current time plus the number
// of days offset. The number of days can be from 0 to 7500.
//
// Returns: 1 success, real time clock and timer set
//          0 fail, no DS1427/DS1994 or the DS1427/DS1994 is write protected
//
int far pascal TimerSet(int days)
{
   ulong c,t;
   int i,rt=0;
   uchar tt[5],ct[5],wr[32];

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // check number of days
   if ((days < 0) || (days > 7500))
      return 0;

   // get the current time in seconds from dos
   c = (ulong)time(NULL) - _timezone;  // clock time
   t = c + 86400L * (ulong)days;      // clock alarm time
   
   // convert the ulong to a uchar string
   tt[0] = 0; // fraction of second
   ct[0] = 0; // fraction of second
   for (i = 1; i <= 4; i++)
   {
      ct[i] = (uchar)(c & 0xFF);
      c /= 0x100;
      tt[i] = (uchar)(t & 0xFF);
      t /= 0x100;
   }
   
   // construct a string to write to the timekeeping registers
   wr[0] = 0x30;  // status register, enable real-time alarm
   wr[1] = 0x50;  // control register, stop manual interval, osc enable
   for (i = 0; i < 5; i++)
   {
      wr[i+2]  = ct[i];
      wr[i+7]  = 0;
      wr[i+12] = 0;
      wr[i+16] = tt[i];
      wr[i+22] = 0;
      wr[i+27] = 0;
   }
 
   // open the port            
   if (keyopen())  
   { 
      // read the status register to clear any alarm flags
      // and read the write protect flags
      if (access(&gb))               // if Access is unsuccessful
      {
         databyte(0xF0,&gb);             // read memory command
         databyte(0x00,&gb);             // write the target address
         databyte(0x02,&gb);
            
         databyte(0xFF,&gb);             // clear status
         if ((databyte(0xFF,&gb) & 0x0F) == 0)  // check write protect
            // write the timekeeping registers
            if (Write_Scratchpad(wr,16,32) == 1)
               if (Copy_Scratchpad(16,32) == 1)   
                  rt = 1;
      }
      // release the port
      keyclose();
   }

   return rt; 
}


//--------------------------------------------------------------------------
// Check the real time clock of the DS1427/DS1994 and the alarm register time to
// see if the time has expired.
//
// Returns: 1 time has not expired
//          0 time expired or DS1427/DS1994 is missing
//
int far pascal CheckTimer(void)
{
   int change,i,cnt=0,rt=0;
   uchar bf[26],ch;
   ulong c,t;

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // open the port            
   if (keyopen())  
   { 
      // read the real time clock and alarm registers
      do
      {         
         change = 1;
         if (access(&gb))            // select the device
         {
            databyte(0xF0,&gb);          // read memory command
            databyte(0x00,&gb);          // write the target address
            databyte(0x02,&gb);
                      
            change = 0;          
            for (i = 0; i < 0x15; i++)
            {
               ch = databyte(0xFF,&gb);     // get a character
               if (ch != bf[i])
               {
                    if (i != 2)   // don't count a change in fraction sec
                       change = 1;
                    bf[i] = ch;
               }
            }  
         }
      }
      while (change && (++cnt < 5));
      
      // release the port
      keyclose();
      
      // check to see if could read
      if (cnt < 5)  
      {
         // check osc enable and interval timer stop
         // and only real-time clock alarm on
         if (((bf[1] & 0xF0) != 0x50) || ((bf[0] & 0x38) != 0x30))
           return 0;
         
         // compare the real time with the real time alarm registers
         c = uchar_to_bin(4,&bf[3]);
         t = uchar_to_bin(4,&bf[17]);
         if (c < t)
           return 1;  // not elapsed
         else
           return 0;  // elapsed  
      }  
   }

   return 0;  // failed keyopen or cnt >= 5      
}


//--------------------------------------------------------------------------
// WriteProtect the DS1427/DS1994 so that the real-time clock and alarm register
// can not be changed.  After the alarm time has elapsed then the nv ram
// can be read but not changed.
//
// Returns: 1 DS1427/DS1994 is now write protected or already write protected
//          0 could not write protect or no DS1427/DS1994 found
//
int far pascal WriteProtectT(void)
{
   uchar bf[3];
   int i,rt=0;

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // check to make sure the Timer has been set and still active
   if (!CheckTimer())
      return 0;

   // open the port            
   if (keyopen())  
   { 
      // construct a string to write to status/control registers
      bf[0] = 0x30;
      bf[1] = 0x59;
      
      // write the data to the scratchpad
      if (Write_Scratchpad(bf,16,2) == 1)
      {
         // copy the scratchpad 3 times to write protect
         for (i = 0; i < 3; i++)
         {
            if (!access(&gb))            // if Access is unsuccessful
             break;
               
            databyte(0x55,&gb);          // sent the copy command
            databyte(0x00,&gb);          // write the target address 1
            databyte(0x02,&gb);          // write the target address 2
            databyte((i)?0x81:0x01,&gb); // send the offset byte
         }
   
         // read back the control register to verify write protected
         if (access(&gb))    
         {
            databyte(0xF0,&gb);               // read memory command
            databyte(0x01,&gb);               // write the target address
            databyte(0x02,&gb);
                  
            if (databyte(0xFF,&gb) == 0x59)   // check write protect   
              rt = 1; // success
         } 
      }
      // release the port
      keyclose();
  }  
  
  return rt;
}


//--------------------------------------------------------------------------
// ReadTimer reads the real time register and real time alarm register,
// formats them and returns them as two strings.  The strings location
// must be at least 24 bytes long each.
//
// Returns: 1 registers read and returned
//          0 failure to find the registers
//
int far pascal ReadTimer(uchar far *tm, uchar far *tma)
{
   int change,i,cnt=0,rt=0;
   uchar bf[26],ch;
   timedate c,t;  
   char temp[120];

   // check to see if there is a current DS1427/DS1994
   if (!GotOne)
      return 0;
  
   // open the port            
   if (keyopen())  
   { 
      // read the real time and real time alarm registers
      do
      {
         change = 1;
         if (access(&gb))            // select the device
         {
            databyte(0xF0,&gb);          // read memory command
            databyte(0x01,&gb);          // write the target address
            databyte(0x02,&gb);
                     
            change = 0;
            for (i = 1; i < 0x15; i++)
            {
               ch = databyte(0xFF,&gb);     // get a character
               if (ch != bf[i])
               {
                  if (i != 2)   // don't count a change in fraction sec
                    change = 1;
                  bf[i] = ch;
               }
            }
         }
      }
      while (change && (++cnt < 5));

      // check to see if could read
      // check osc enable and interval timer stop
      if ((cnt < 5) && ((bf[1] & 0xF0) == 0x50))
      {
         // convert the 2 times to timedate values.
         Str2Time(&c,&bf[2]);
         Str2Time(&t,&bf[16]);
            
         // print the values to the provided buffers
         cnt = sprintf(temp," %02d/%02d/%02d  %2d:%02d:%02d.%02d ",
                c.month,c.day,c.year,c.hour,c.minute,c.second,c.frsecond); 
         for (i = 0; i <= cnt; i++)
            tm[i] = temp[i];
         sprintf(temp," %02d/%02d/%02d  %2d:%02d:%02d.%02d ",
                t.month,t.day,t.year,t.hour,t.minute,t.second,t.frsecond);
         for (i = 0; i <= cnt; i++)
            tma[i] = temp[i];
         rt = 1;
      }
      // release the port
      keyclose();
  }  

  return rt; 
}


//--------------------------------------------------------------------------
// Search the LPT ports to find a brick with a DS1427/DS1994
//
// Returns:  1 found a DS1427/DS1994
//           0 failed to find a DS1427/DS1994
//
int far pascal NextDS1427(void)
{
   int flag,didfirst=0;
   uchar far *ROM;
   
   // check if got a port   
   if (!GotPort)   
   {  
      // make use of the knowledge the SmartPortSearch does a first
      flag = SmartPortSearch();
      if (flag != 0)    
      {
         GotPort = 1; 
         didfirst = 1;
      }
      else
         return 0;
   }

   // assume fail to find a DS1427/DS1994
   GotOne = 0;
 
   // open the port            
   if (keyopen())  
   {    
      // get a pointer to the rom buffer        
      ROM = romdata(&gb);           
 
      do
      {   
         // do a next if did not just do a SPS or loop
         if (!didfirst)
            flag = next(&gb);  
         didfirst = 0;
             
         if (flag > 0)
         {
            // check for the DS1427/DS1994's family code
            if ((ROM[0] & 0x7F) == 0x04)
            {
               GotOne = 1;
               break;  // success
            }
         }
      }
      while (flag);
      
      // release the port
      keyclose();
  }  

  return GotOne; 
}


//--------------------------------------------------------------------------
// Write the scratchpad of the DS1427/DS1994 and verify its contents. The data
// to write is in 'buf' with a length of 'len' starting on page 'pg'.
//
// Returns: 1 success
//          0 failure
// 
// ###### Assume KeyOpen has already been called ######
//
int Write_Scratchpad(uchar far *buf, int pg, int len)
{
     int i;

     if (!access(&gb))               // if Access is unsuccessful
        return 0;

     databyte(0x0F,&gb);             // write scratchpad command
     databyte((pg << 5) & 0xFF,&gb); // write the target address 1
     databyte(pg >> 3,&gb);          // write the target address 2

     // write num bytes scratchpad contents
     for (i = 0; i < len; i++)
         databyte(buf[i],&gb);

     // now attempt to read back to check
     if (!access(&gb))               // if Access is unsuccessful
        return 0;

     databyte(0xAA,&gb);             // read scratchpad command
     // read the target address byte 1
     if (databyte(0xFF,&gb) != (uint)((pg << 5) & 0xFF))
        return 0;
     // read the target address byte 2
     if (databyte(0xFF,&gb) != (uint)(pg >> 3))
        return 0;
     // read the offset     
     if (databyte(0xFF,&gb) != (uint)(len-1))
        return 0;

     // read num bytes scratchpad contents
     for (i = 0; i < len; i++)
         if (databyte(0xFF,&gb) != buf[i])
            return 0;

     return 1;  // success
}


//--------------------------------------------------------------------------
// Copy the contents of the scratchpad to its intended nv ram page.  The
// page and length of the data is needed to build the authorization bytes
// to copy.
//
// Returns: 1 success
//          0 failure
//
// ###### Assume KeyOpen has already been called ######
//
int Copy_Scratchpad(int pg, int len)
{
     int flag;

     if (!access(&gb))                // if Access is unsuccessful
        return 0;

     databyte(0x55,&gb);              // sent the copy command
     databyte((pg << 5) & 0xFF,&gb);  // write the target address 1
     databyte(pg >> 3,&gb);           // write the target address 2
     databyte(len-1,&gb);             // send the offset byte

     // get result of the copy
     flag = databyte(0xFF,&gb);

     // check 4 msb to see if device sending 0's
     if (flag & 0xF0)
        return 0;     // failure to receive 0's
     else
        return 1;     // success
}


//--------------------------------------------------------------------------
// Take a 5 byte long string and convert it into a timedata structure.
//
static int dm[] = { 0,0,31,59,90,120,151,181,212,243,273,304,334,365 };
//
void Str2Time(timedate far *td, uchar far *ptr)
{
     int tmp,i,j;
     ulong x,y;

     // get value of number of seconds
     x = uchar_to_bin(4,&ptr[1]);

     // get the hundredths of a second
     tmp = (99*(ptr[0]+2))/255;

     td->frsecond = tmp;

     // check to make sure date is not over 2020
     if (x > 0x5FDD4280L)
          x = 0;

     y = x/60;  td->second = (uint)(x-60*y);
     x = y/60;  td->minute = (uint)(y-60*x);
     y = x/24;  td->hour   = (uint)(x-24*y);
     x = 4*(y+731);  td->year = (uint)(x/1461);
     i = (int)((x-1461*(ulong)(td->year))/4);  td->month = 13;

     do
     {
          td->month -= 1;
          tmp = (td->month > 2) && ((td->year & 3)==0) ? 1 : 0;
          j = dm[td->month]+tmp;

     } while (i < j);

     td->day = i-j+1;

     // slight adjustment to algorithm
     if (td->day == 0) td->day = 1;

     td->year = (td->year < 32)  ? td->year + 68 : td->year - 32;
}


//--------------------------------------------------------------------------
// UCHAR_TO_BIN converts a binary uchar string 'str' into a ulong return
// number.  'num' indicates the length of the binary uchar string.
//
ulong uchar_to_bin(int num, uchar far *str)
{
     int i;
     ulong l = 0;

     for (i = (num-1); i >= 0; i--)
          l = (l << 8) | (int)(str[i]);

     return l;
}


//--------------------------------------------------------------------------
// Calculate a new CRC16 from the input data integer.  Return the current
// CRC16 and also update the global variable CRC16.
//
// Returns the current CRC16
//
static int oddparity[16] = { 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0 };

uint crc16(int data)
{
   data = (data ^ (CRC16 & 0xff)) & 0xff;
   CRC16 >>= 8;

   if (oddparity[data & 0xf] ^ oddparity[data >> 4])
      CRC16 ^= 0xc001;

   data <<= 6;
   CRC16   ^= data;
   data <<= 1;
   CRC16   ^= data;

   return CRC16;
}

/* Removed because not needed in stand alone version
//--------------------------------------------------------------------------
// LibMain is called when the DLL is loaded
//
int FAR PASCAL LibMain(HANDLE hInstance, WORD wDataSeg, WORD cbHeapSize, LPSTR lpCmdLine)
{
     return 1;
}


//--------------------------------------------------------------------------
// Function that is called on DLL unload.
//
int FAR PASCAL WEP(int ExtType)
{

     return 1;
}
*/